package nxm.ice.lib;

import nxm.sys.lib.Convert;

import java.net.MulticastSocket;
import java.net.DatagramPacket;
import java.net.InetSocketAddress;
import java.net.InetAddress;

/**
  A class to handle communications with an ICE network stream.

  @author Jeff Schoen
  @version %I%, %G%

*/
public class NetIO extends Object {

  public static int RXBUFSIZE=8388608, TXBUFSIZE=262080;
  public static String flagsList = "Input,Output,InOut,OutIn,RMIF,HTTP,RTable,UDP,Thread,ThrSlv";
  public static int INPUT=0x01, OUTPUT=0x02, INOUT=0x04, OUTIN=0x08, RMIF=0x10, HTTP=0x20, RTAB=0x40, UDP=0x80, THREAD=0x100, THRSLV=0x200;
  public static int MSG=0, SET=1, GET=2, RET=3, ACK=4, MSGR=5;

  protected int sock;
  protected MulticastSocket ms;
  protected DatagramPacket dp;
  protected boolean useNative;
  protected InetSocketAddress lisa,risa;
  protected InetAddress lia,ria;

  protected static boolean canUseNative = load();

  // default do nothing constructor
  public NetIO (boolean useNative) {
    this.useNative = useNative && canUseNative;
  }

  // load the native libraries
  public static boolean load() {
    try { return nxm.sys.lib.Shell.loadLibrary("ice","lib","NetIO"); }
    catch (Exception e) { throw new RuntimeException (e); }
  }

  public int open (String url, int chans, int flags) {
    if (useNative) { sock = nioOpen (url,chans,flags); return sock; }
    int i0=url.indexOf(":",4), i1=url.indexOf("/",4), i2=url.indexOf(":",i1), i3=url.length();
    if (i0<0 || i0>i1) i0=i1; if (i2<0) i2=i3;
    String lhost = url.substring(4,i0); 
    int lport = (i1>i0)? Convert.s2l(url.substring(i0+1,i1)) : 0x7337;
    String rhost = url.substring(i1+1,i2);
    int rport = (i3>i2)? Convert.s2l(url.substring(i2+1,i3)) : 0x7337;
    lisa = new InetSocketAddress(lhost,lport);  lia = lisa.getAddress();
    risa = new InetSocketAddress(rhost,rport);  ria = risa.getAddress();
    String slisa = lisa.toString().substring(1);
    String srisa = risa.toString().substring(1);
    System.out.println("Open local="+slisa+" remote="+srisa+" native="+useNative);
    try {
      ms = new MulticastSocket(lport);
      ms.setInterface(lisa.getAddress());
      ms.setReuseAddress(true);
      ms.setLoopbackMode(false);
      if (flags>0) {	// output
        ms.setSendBufferSize(TXBUFSIZE);
      } else {		// input
        ms.setSoTimeout(1); // in msec
        ms.setReceiveBufferSize(RXBUFSIZE);
        if (ria.isMulticastAddress()) ms.joinGroup(ria);
        //System.out.println("Join "+ria+" "+ria.isMulticastAddress());
      }
      return 1;
    }
    catch (Exception e) { System.out.println("openErr: "+e); return -1; }

  }
  public int send (int chan, byte[] buffer, int off, int bytes, int flags) {
    if (useNative) return nioSendB (sock,chan,buffer,off,bytes,flags);
    if (dp!=null) dp.setData(buffer,off,bytes);
    else dp = new DatagramPacket(buffer,off,bytes);
    try { ms.send(dp); return dp.getLength(); }
    catch (Exception e) {  System.out.println("sendErr: "+e); return -1; }
  }
  public int recv (int chan, byte[] buffer, int off, int bytes, int flags) {
    if (useNative) return nioRecvB (sock,chan,buffer,off,bytes,flags);
    if (dp!=null) dp.setData(buffer,off,bytes);
    else dp = new DatagramPacket(buffer,off,bytes);
    try { ms.receive(dp); return dp.getLength(); }
    catch (java.net.SocketTimeoutException e) { return 0; }
    catch (Exception e) { System.out.println("recvErr: "+e); return -1; }
  }
  public int send (int chan, long vaddr, int bytes, int flags) {
    if (useNative) return nioSend (sock,chan,vaddr,bytes,flags);
    return 0;
  }
  public int recv (int chan, long vaddr, int bytes, int flags) {
    if (useNative) return nioRecv (sock,chan,vaddr,bytes,flags);
    return 0;
  }
  public int msg (int chan, int func, byte[] buffer, int bytes, int flags) {
    if (useNative) return nioMsg (sock,chan,func,buffer,bytes,flags);
    return 0;
  }
  public int file (String fname, String aname, int flags) {
    if (useNative) return nioFile (sock,fname,aname,flags);
    return 0;
  }
  public int close () {
    if (useNative) { if (sock>0) nioClose(sock); return sock; }
    if (ms!=null) ms.close();
    return 1;
  }

  public native int nioOpen (String url, int chans, int flags);
  public native int nioClose (int sock);
  public native long nioAlloc (int bytes);
  public native int nioFree (long vaddr);
  public native int nioSendBank (int sock, int chns, int ncpa, long haddr, int hlen, int dlen, long vaddr, int bytes, int flags);
  public native int nioSend (int sock, int chan, long vaddr, int bytes, int flags);
  public native int nioRecv (int sock, int chan, long vaddr, int bytes, int flags);
  public native int nioSendB (int sock, int chan, byte[] array, int off, int bytes, int flags);
  public native int nioRecvB (int sock, int chan, byte[] array, int off, int bytes, int flags);
  public native int nioMsg (int sock, int chan, int func, byte[] array, int bytes, int flags);
  public native int nioFile (int sock, String fname, String aname, int flags);

  public native static int swap2     (long ptr1, long ptr2, int bytes);
  public native static int cvt12to16 (long ptr1, long ptr2, int bytes);
  public native static int cvt16to12 (long ptr1, long ptr2, int bytes);

}
