package nxm.ice.lib;

import nxm.sys.inc.Constants;
import nxm.sys.inc.DataTypes;
import nxm.sys.lib.MidasException;
import nxm.sys.lib.Convert;
import nxm.sys.lib.Time;

/**
  Common routines for a Code-Once-Run-Everywhere library function

  @author Jeff Schoen
  @version $Id: CoreCommon.java,v 1.2 2011/05/24 13:20:30 jgs Exp $
*/
public class CoreCommon extends Object implements Constants,DataTypes {

  public final static int FUNC=0, FORM=1, FLOW=2, IMPL=3, FLGS=4, SCID=5, MCID=6, VBPR=7;
  public final static int ARRAY=0, STREAM=1, BUFFER=2, BUFSTR=3, PACKET=4;

  // class data structures
  public static class TypeDef {
  }

  // scalar data structures
  public static class SxShort {
    public short x;
    public static SxShort stk() { return new SxShort(); }
  }
  public SxShort SxShort_stk() { return new SxShort(); }

  public static class SxFloat {
    public float x;
    public static SxFloat stk() { return new SxFloat(); }
  }
  public SxFloat SxFloat_stk() { return new SxFloat(); }

  public static class SxDouble {
    public double x;
    public static SxDouble stk() { return new SxDouble(); }
  }
  public SxDouble SxDouble_stk() { return new SxDouble(); }

  // complex data structures

  public static class CxByte {
    public byte x,y;
    public static CxByte stk() { return new CxByte(); }
  }
  public CxByte CxByte() { return new CxByte(); }

  public static class CxShort {
    public short x,y;
    public static CxShort stk() { return new CxShort(); }
  }
  public CxShort CxShort_stk() { return new CxShort(); }

  public static class CxFloat {
    public float x,y;
    public static CxFloat stk() { return new CxFloat(); }
    public void set(CxFloat to) { x=to.x; y=to.y; }
    public void setxy (float _x, float _y) { x=_x; y=_y; }
    public static CxFloat[] newArray (int n) { 
      CxFloat[] cfa = new CxFloat[n];
      for (int i=0; i<n; i++) cfa[i] = new CxFloat();
      return cfa;
    }
  }
  public CxFloat CxFloat_stk() { return new CxFloat(); }

  public static class CxDouble {
    public double x,y;
    public static CxDouble stk() { return new CxDouble(); }
  }
  public CxDouble CxDouble_stk() { return new CxDouble(); }

  // vector data structures
  public static class VxShort {
    public short x,y,z;
    public static VxShort stk() { return new VxShort(); }
  }
  public VxShort VxShort_stk() { return new VxShort(); }

  public static class VxFloat {
    public float x,y,z;
    public static VxFloat stk() { return new VxFloat(); }
  }
  public VxFloat VxFloat_stk() { return new VxFloat(); }

  public static class VxDouble {
    public double x,y,z;
    public static VxDouble stk() { return new VxDouble(); }
  }
  public VxDouble VxDouble_stk() { return new VxDouble(); }


  // bit manipulators for non-verilog implementations
  public static boolean getBit (int val, int i) {
    return ((val>>i)&1) != 0;
  }
  public static boolean getBit (long val, int i) {
    return ((val>>i)&1) != 0;
  }
  public static int getBits (int val, int i2, int i1) {
    int mask = (2<<(i2-i1))-1;
    return (val>>i1) & mask;
  }
  public static int getBits (long val, int i2, int i1) {
    int mask = (2<<(i2-i1))-1;
    return (int)(val>>i1) & mask;
  }
  public static int getPBS (int cur, int val, int i2, int i1) {
    int len = i2-i1+1;
    int mask = (1<<len)-1;
    return (cur<<len) | ((val>>i1)&mask);
  }
  public static long getPBS (long cur, long val, int i2, int i1) {
    int len = i2-i1+1;
    long mask = (1<<len)-1;
    return (cur<<len) | ((val>>i1)&mask);
  }
  public static int getPABS (int cur, int val, int len) {
    int mask = (1<<len)-1;
    return (cur<<len) | (val&mask);
  }
  public static long getPABS (long cur, long val, int len) {
    long mask = (1<<len)-1;
    return (cur<<len) | (val&mask);
  }
  public static int getPBBS (int cur, boolean val) {
    return (cur<<1) | (val?1:0);
  }
  public static int gatherBits (int val, int i0, int inc, int n) {
    int i, mask=0;
    for (i=0; i<n; i++,i0+=inc) mask |= (((val>>i0)&1)<<i);
    return mask;
  }
  public static int setBit (int val, int i, boolean valb) {
    int mask = 1<<i;
    return valb? (val | mask) : (val & ~mask);
  }
  public static long setBit (long val, int i, boolean valb) {
    long mask = 1L<<i;
    return valb? (val | mask) : (val & ~mask);
  }
  public static byte setBits (byte val, int i2, int i1, int valb) {
    int mask = (2<<(i2-i1))-1;
    return (byte)((val & ~(mask<<i1)) | ((valb&mask)<<i1));
  }
  public static short setBits (short val, int i2, int i1, int valb) {
    int mask = (2<<(i2-i1))-1;
    return (short)((val & ~(mask<<i1)) | ((valb&mask)<<i1));
  }
  public static int setBits (int val, int i2, int i1, int valb) {
    int mask = (2<<(i2-i1))-1;
    return (val & ~(mask<<i1)) | ((valb&mask)<<i1);
  }
  public static long setBits (long val, int i2, int i1, long valb) {
    long mask = (2L<<(i2-i1))-1;
    return (val & ~(mask<<i1)) | ((valb&mask)<<i1);
  }

  // time operators
  public static double cpuTime() { return 0.001 * System.currentTimeMillis(); }
  public static String cpuDate() { return Time.currentTime().toString(); }
  public static void   cpuWait (double sec) { try { Thread.sleep((long)(sec*1000)); } catch (InterruptedException e) { } }

  // cast operators
  public static int    s2i (short s)  { return (int)s; }
  public static long   s2l (short s)  { return (long)s; }
  public static short  i2s (int i)    { return (short)i; }
  public static float  i2f (int i)    { return (float)i; }
  public static float  d2f (double d) { return (float)d; }
  public static int    d2i (double d) { return (int)d; }
  public static long   d2l (double d) { return (long)d; }
  public static int    f2i (float f)  { return (int)f; }
  public static int    l2i (long l)   { return (int)l; }
  public static short  l2s (long l)   { return (short)l; }
  public static double f2d (float f) { return (double)f; }
  public static double i2d (int i)   { return (double)i; }
  public static double l2d (long l)   { return (double)l; }
  public static long   ii2l (int ui, int li)  { return ((long)ui<<32) | (li&0xFFFFFFFFL); }

  public static int   f2x (float f)  { return (int)(f*65536.0F); }
  public static long  d2x (double f)  { return (long)(f*4294967296.0); }
  public static int   d2fi (double f)  { return (int)(d2x(f)&0xFFFFFFFF); }
  public static int   d2wi (double f)  { return (int)(d2x(f)>>32); }
  public static double fi2d (int f)  { return ((double)f)/4294967296.0; }

  public static float i2fp (int i)    { return ((float)i)/65536.0F; }
  public static int   fp2i (float f)  { return (int)(f*65536.0F); }

  // cast to HW formats
  public static int    f2fptx (float f) { return (int)(f*(1<<16)); }
  public static float  fptx2f (int x) { return ((float)x)/(1<<16); }
  public static long   d2dptx (double d) { return (long)(d*(1L<<32)); }
  public static double dptx2d (long x) { return ((double)x)/(1L<<32); }

  // cast operators to Midas formats with scaling 
  public static short F2I (float f)  { return (short)(f*32768.0F); }
  public static float I2F (short s)  { return (s/32768.0F); }
  public static float L2F (int i)  { return (i/2147483648.0F); }
  public static float K2F (int i)  { return (i/65536.0F); }
  public static int   F2K (float f)  { return (int)(f*65536); }

  public static int CI2L (short ix, short iy)  { return (iy<<16) | (ix&0xFFFF); }
  public static int CF2L (float fx, float fy)  { return CI2L(F2I(fx),F2I(fy)); }

  public static void  CF2CI (CxFloat cf, CxShort ci) { ci.x=(short)(cf.x*32768.0F); ci.y=(short)(cf.y*32768.0F); }
  public static void  CI2CF (CxShort ci, CxFloat cf) { cf.x=ci.x/32768.0F; cf.y=ci.y/32768.0F; }

  public static float CI2Fx (int i) { i<<=16;        return i/2147483648.0F; }
  public static float CI2Fy (int i) { i&=0xFFFF0000; return i/2147483648.0F; }

  public static int IN2P (int i) { return ((i>>3 )&0x01) | ((i>>6 )&0x02) | ((i>>9 )&0x04) | ((i>>12)&0x08); }
  public static int LN2P (int i) { return ((i>>3 )&0x01) | ((i>>6 )&0x02) | ((i>>9 )&0x04) | ((i>>12)&0x08) | 
					  ((i>>15)&0x10) | ((i>>18)&0x20) | ((i>>21)&0x40) | ((i>>24)&0x80); }
  public static int IO2P (int i) { return ((i>>2 )&0x01) | ((i>>4 )&0x02) | ((i>>6 )&0x04) | ((i>>8)&0x08) | ((i>>10)&0x10); }
  public static int LO2P (int i) { return ((i>>2 )&0x01) | ((i>>4 )&0x02) | ((i>>6 )&0x04) | ((i>>8)&0x08) | 
	((i>>10)&0x10) | ((i>>12)&0x20) | ((i>>14)&0x40) | ((i>>16)&0x80) | ((i>>18)&0x100) | ((i>>20)&0x200); }

  // cast to Byte Buffer formats
  public static byte[] cvbb = new byte[8];
  public static byte[] s2bb (short v)  { Convert.packI(cvbb,0,v); return cvbb; }
  public static byte[] i2bb (int   v)  { Convert.packL(cvbb,0,v); return cvbb; }
  public static byte[] f2bb (float v)  { Convert.packF(cvbb,0,v); return cvbb; }
  public static byte[] CF2bb (CxFloat v) { Convert.packF(cvbb,0,v.x); Convert.packF(cvbb,4,v.y); return cvbb; }

  // convenient access to C math functions
  public static int    imin (int x, int y) { return (x<y)?x:y; }
  public static int    imax (int x, int y) { return (x>y)?x:y; }
  public static float  fminf (float x, float y) { return (x<y)?x:y; }
  public static float  fmaxf (float x, float y) { return (x>y)?x:y; }
  public static double fmin (double x, double y) { return (x<y)?x:y; }
  public static double fmax (double x, double y) { return (x>y)?x:y; }

  public static int abs (int x) { return (x<0)?-x:x; }
  public static int iabs (int x) { return (x<0)?-x:x; }
  public static long labs (long x) { return (x<0)?-x:x; }
  public static float fabsf (float x) { return (x<0)?-x:x; }
  public static double fabs (double x) { return (x<0)?-x:x; }

  public static float sinf (double x) { return (float)Math.sin(x); }	// in radians
  public static float cosf (double x) { return (float)Math.cos(x); }	// in radians
  public static float tanf (double x) { return (float)Math.tan(x); }	// in radians
  public static float atan2f (float x, float y) { return (float)Math.atan2(x,y); }	// in radians
  public static float sincf (double x) { return (float)Math.sin(x*Constants.TWOPI); }	// in cycles
  public static float coscf (double x) { return (float)Math.cos(x*Constants.TWOPI); }	// in cycles
  public static float tancf (double x) { return (float)Math.tan(x*Constants.TWOPI); }	// in cycles
  public static float atan2cf (float x, float y) { return (float)(Math.atan2(x,y)/Constants.TWOPI); }	// in cycles

  public static float invf (float x)  { return 1.0F/x; }
  public static float invfi (int x)  { return 1.0F/x; }
  public static float sqrtf (float x) { return (float)Math.sqrt(x); }
  public static float log10f (float x) { return (float)Math.log10(x); }
  public static float roundf (float x) { return (float)Math.round(x); }
  public static float floorf (float x) { return (float)Math.floor(x); }
  public static float fracf (float x) { return x-(float)Math.floor(x); }
  public static float wrapf (float x) { return fracf(x+0.5F)-0.5F; }
  public static float ceilf (float x) { return (float)Math.ceil(x); }
  public static float fmodf (float x, float y) { return x%y; }
  public static float powf (float x, float y) { return (float)Math.pow(x,y); }
  public static int   fintf (float x) { return (int)x; }

  public static double sin (double x) { return Math.sin(x); }	// in radians
  public static double cos (double x) { return Math.cos(x); }	// in radians
  public static double sinc (double x) { return Math.sin(x*Constants.TWOPI); }	// in cycles
  public static double cosc (double x) { return Math.cos(x*Constants.TWOPI); }	// in cycles
  public static double tan (double x) { return Math.tan(x); }	// in radians
  public static double atan2 (double x, double y) { return Math.atan2(x,y); }	// in radians

  public static double inv (double x) { return 1.0/x; }
  public static double sqrt (double x) { return Math.sqrt(x); }
  public static double log10 (double x) { return Math.log10(x); }
  public static double log2 (double x) { return Math.log(x)*1.44269504088896; }
  public static double round (double x) { return Math.round(x); }
  public static double floor (double x) { return Math.floor(x); }
  public static double ceil (double x) { return Math.ceil(x); }
  public static double frac (double x) { return x-Math.floor(x); }
  public static double wrap (double x) { return frac(x+0.5)-0.5; }
  public static double fmod (double x, double y) { return x%y; }
  public static double pow (double x, double y) { return Math.pow(x,y); }

  public static double divii (int x, int y) { return ((double)x)/y; }
  public static double divdi (double x, int y) { return x/y; }
  public static double divp2 (double x, int y) { return x/(1<<y); }
  public static double mulp2 (double x, int y) { return x*(1<<y); }
  public static float  divf  (float x, float y) { return x/y; }
  public static float  divfi (float x, int y) { return x/y; }
  public static float  divfp2 (float x, int y) { return x/(1<<y); }
  public static float  mulfp2 (float x, int y) { return x*(1<<y); }
  public static double muldf (double x, float y) { return x*y; }
  public static double muldi (double x, int y) { return x*y; }
  public static float  mulfi (float x, int y) { return x*y; }
  public static float  mulff (float x, float y) { return x*y; }
  public static int    mulif (int x, float y) { return (int)(x*y); }
  public static int    fint  (double x) { return (int)x; }

  public static void   muxf (float x, float y, CxFloat cx) { cx.x=x; cx.y=y; }

  public static float  phaseditherf () { return 0.0F; }

  public static double clip (double x, double min, double max) { return (x<min)? min : (x>max)? max : x; }	// clip number
  public static float  clipf (float x, float min, float max) { return (x<min)? min : (x>max)? max : x; }
  public static int    clipl (int x, int min, int max) { return (x<min)? min : (x>max)? max : x; }

  public static double sclip (double x, double mm) { return (x< -mm)? -mm : (x> mm)? mm : x; }		// symmetric clip number
  public static float  sclipf (float x, float mm) { return (x< -mm)? -mm : (x> mm)? mm : x; }
  public static int    sclipl (int x, int mm) { return (x< -mm)? -mm : (x> mm)? mm : x; }

  public static short  bswap2 (int x) { return (short)(((x>>8)&0xFF)|(x<<8)); }
  public static short  bswap2 (short x) { return (short)(((x>>8)&0xFF)|(x<<8)); }
  public static int    bswap2s (int x) { return ((x>>8)&0xFF)|((x<<8)&0xFF00)|((x>>8)&0xFF0000)|((x<<8)&0xFF000000); }
  public static int    bswap4 (int x) { return ((x>>24)&0xFF)|((x>>8)&0xFF00)|((x<<8)&0xFF0000)|(x<<24); }

  public static int brev4 (int val) {
    int rval=0; 
    if (val!=0) for (int i=0; i<32; i++) if (((val>>i)&1)!=0) rval |= (1<<(31-i));
    return rval;
  }

  public static int    iLRS (int lrs) { return (lrs<<1) | ( ((lrs>>0)&1) ^ ((lrs>>1)&1) ^ ((lrs>>5)&1) ^ ((lrs>>25)&1) ); }
  public static int    iLRSP (int lrs, int lrsp) { return (lrs<<1) | ((ones(lrs&lrsp,32)+1)&1); }

  public static int ones (int x, int n) {
    int ones=0;
    for (int i=0; i<n; i++) { ones+=(x&1); x>>=1; }
    return ones;
  }

  public static int qwrap2z (CxFloat a) {
    float absx = (a.x<0)? -a.x : a.x;
    return (a.y>absx)? 3 : (a.y<-absx)? 1 : (a.x<0)? 2 : 0;
  }
  public static void qrotcc (CxFloat a, int quad, CxFloat b) {
    switch (quad&3) {
    case 0: b.x =  a.x; b.y =  a.y; break;
    case 1: b.x = -a.y; b.y =  a.x; break;
    case 2: b.x = -a.x; b.y = -a.y; break;
    case 3: b.x =  a.y; b.y = -a.x; break;
    }
  }

  public static void dupcc (CxFloat a, CxFloat b) {
    b.x = a.x;
    b.y = a.y;
  }
  public static void sincoscf (float pf, CxFloat sc) {
    double p = pf*Constants.TWOPI;
    sc.x = (float)Math.cos(p);
    sc.y = (float)Math.sin(p);
  }
  public static void pol2recf (float amp, float pf, CxFloat sc) {
    double p = pf*Constants.TWOPI;
    sc.x = amp*(float)Math.cos(p);
    sc.y = amp*(float)Math.sin(p);
  }
  public static void addcc (CxFloat a, CxFloat b, CxFloat c) {
    c.x = a.x+b.x;
    c.y = a.y+b.y;
  }
  public static void mulfc (float a, CxFloat b, CxFloat c) {
    c.x = a*b.x;
    c.y = a*b.y;
  }
  public static void mulcf (CxFloat a, float b, CxFloat c) {
    c.x = a.x*b;
    c.y = a.y*b;
  }
  public static void mulcc (CxFloat a, CxFloat b, CxFloat c) {
    c.x = a.x*b.x - a.y*b.y;
    c.y = a.y*b.x + a.x*b.y;
  }
  public static void mulccj (CxFloat a, CxFloat b, CxFloat c) {
    c.x = a.x*b.x + a.y*b.y;
    c.y = a.y*b.x - a.x*b.y;
  }
  public static float mulffrnc (float x, float y) { 
    float z = x*y;
    if (z>1.0F) z=1.0F;
    else if (z<-1.0F) z=-1.0F;
    return z;
  }
  public static void mulfcrnc (float a, CxFloat b, CxFloat c) {
    c.x = mulffrnc(a,b.x);
    c.y = mulffrnc(a,b.y);
  }
  public static void mulcfrnc (CxFloat a, float b, CxFloat c) {
    c.x = mulffrnc(a.x,b);
    c.y = mulffrnc(a.y,b);
  }
  public static float mag2f (CxFloat a) {
    return a.x*a.x + a.y*a.y;
  }
  public static float log10cf (CxFloat v) { 
    return (float)Math.log10(v.x*v.x+v.y*v.y); 
  }

  public static void mulccA (CxFloat a, CxFloat[] b, CxFloat[] c, int n) { for (int i=0; i<n; i++) mulcc(a,b[i],c[i]); }
  public static void mulcAc (CxFloat[] a, CxFloat b, CxFloat[] c, int n) { for (int i=0; i<n; i++) mulcc(a[i],b,c[i]); }
  public static void mulcAcA (CxFloat[] a, CxFloat[] b, CxFloat[] c, int n) { for (int i=0; i<n; i++) mulcc(a[i],b[i],c[i]); }

  public static int pow2ge (int num) {
    int pow; for (pow=0; (1<<pow) < num; pow++); return pow;
  }
  public static int power2ge (int num) {
    int pow; for (pow=0; (1<<pow) < num; pow++); return (1<<pow);
  }

  public static void msumf (float[] a, float[] b, SxFloat c, int n) { 
    c.x=0; 
    for (int i=0; i<n; i++) c.x += a[i]*b[i]; 
  }
  public static void msumcsf (float[] ax, float[] ay, float[] b, CxFloat c, int n) { 
    c.x=0; c.y=0;
    for (int i=0; i<n; i++) {
      c.x += ax[i]*b[i];
      c.y += ay[i]*b[i];
    }
  }
  public static void msumccf (float[] ax, float[] ay, float[] bx, float[] by, CxFloat c, int n) { 
    c.x=0; c.y=0;
    for (int i=0; i<n; i++) {
      c.x += ax[i]*bx[i] - ay[i]*by[i];
      c.y += ax[i]*by[i] + ay[i]*bx[i];
    }
  }

  public static int getListBase (String sa) {
    if (sa.startsWith("NONE,") || sa.startsWith("OFF,")) return 0;
    if (sa.startsWith("AUTO,") || sa.startsWith("ALL,")) return -1;
    return 1;
  }

  public static int findListItem (String sa, String sb, int offset) {
    int i, j=0, k=1, la=sa.length(), lb=sb.length();
    sa=sa.toUpperCase(); sb=sb.toUpperCase();
    k = (offset==-2)? getListBase(sa) : offset;
    for (i=0;i<=la;i++) {
      if (i==la || sa.charAt(i)==',') {
        if (i>=j+lb && sb.equals(sa.substring(j,j+lb))) return k;
        j=i+1; k++;
      }
    }
    return -2;
  }
  public static String getListItem (String sa, int index, int offset) {
    int i, j=0, k=1, la=sa.length();
    sa=sa.toUpperCase();
    k = (offset==-2)? getListBase(sa) : offset;
    for (i=0;i<=la;i++) {
      if (i==la || sa.charAt(i)==',' || sa.charAt(i)=='|') {
        if (k==index) return sa.substring(j,i);
        j=i+1; k++;
      }
    }
    return "Undefined";
  }

}

