//************
//  Description: Driver functions for the ICE-PIC DSP card
//
//  Author:	Jeff Schoen  2/2013
//
//  $Revision:	$
//  $Modtime:	$
//
//************
#include <stdio.h>
#include <primitive.h>
#include <fintrinsics.h>
#define _XMIDAS
#include "icelib.h"
#include "packet.h"
#include "sourcepic.h"
#include "qmessages.h"

char *s2c (string s) { return (char*)s.c_str(); }
char *l2c (int i) { static char cstr[8]; sprintf(cstr,"%d",i); return cstr; }

void mainroutine ()
{
  sourcepic().run();
}

sourcepic::sourcepic() { p=&ps; }

void sourcepic::run() 
{
  int_4  tskip,ipktps,feed,pktmod,mlost,tccnt;
  int_4  tcstatuslast,swap,tinc,tcm,adone,nchan,nodma;
  real_4 wait,pause,elapse,timeout,fstat;
  bool_4 boot,sss,xts,stats,syncdata,flush,tuner,rtfile;
  real_8 maxout, tcps, dval;
  int_1  *buf,*out;

  // form extra flags- for reconnect
  str = "";
  if (m_get_uswitch("FLAGS",tmpstr) > 0) {
    str = tmpstr+","+str;
  }
  i = m_get_uswitch("TC",tcmode);
  if (i>=0) {
    if (i==0) tcmode = "SDN4";
    tcoff = m_get_dswitch("TCOFF");
    if (tcoff==-1) {
      m_now (time1,time2);
      i = m_times2tod (time1,time2,tmpstr);
      tmpstr = tmpstr.substr(0,5)+":01:01::00:00:00";
      i = m_tod2times (tmpstr,tcoff,time2);
      if (tcmode.substr(0,3)=="CPU") tcoff = 0;
    }
    tctolr = m_get_dswitch("TCTOLR");
    str += ",TC="+tcmode;
  } else {
    tcmode = "OFF";
  }

  i = m_get_uswitch("ALG",tmpstr);
  if (i>0) str = "ALG="+tmpstr+","+str;

  i = m_get_switch("VERBOSE");
  if (i>=0) { str+=",VERBOSE="; str+=l2c(i); }

  if (m_get_uswitch("PORT",tmpstr) > 0) {
    str += ",PORT="+tmpstr;
  }
  flgstr = str+",";

  // form device alias
  m_hwf_open (HWF);
  m_hwf_alias (HWF, m_apick(3), str);
  m_hwf_close (HWF);

  str = flgstr+str;

  rtfile = m_get_sswitch("RTFILE");
  info  = !m_get_sswitch("QUIET");

  status = pic_open (p,s2c(str),(int_4*)&Mc->break_,flags);
  if (status <= 0) m_error ("Opening port: "+str);

  boot = m_get_sswitch("REBOOT");
  flags = 0;
  if (boot) flags |= FLG_BOOT;

  ratio = m_get_dswitch("RATIO");
  if (ratio>0) pic_setkeyd (p, 0, KEY_RATIO, ratio);

  status = pic_reset (p,flags);
  if (status <= 0) m_error ("Resetting port: "+str);

  type  = pic_getkeyl(p,0,KEY_PTYPE);
  port  = pic_getkeyl(p,0,KEY_PINDEX);
  tuner = type==IOPT_TUNER || type==IOPT_TBANK;

  // prep for algorithms
  feed=-1;
  nargs=-1;
  alg = pic_getkeyl(p,0,KEY_ALG);
  if (alg>0) {
    nargs = m_get_switch_def("NARG",0);
    for (n=1; n<=nargs; n++) args[n] = m_get_switch_def("ARG"+n,0);
    feed = m_get_switch("FEED");
  }

  // open and check input file
  m_init (HIN, m_apick(1),"1000,2000","SP,SB,SI,SL,CB,CI,CL",0);
  m_open (HIN, rtfile? HCBF_INPUT+HCBF_OUTPUT+HCBF_PRESERVE : HCBF_INPUT+HCBF_PRESERVE);

  // get reduction 
  ired = m_lpick(4);
  if (ired==0) ired = 1;
  skiponcard = m_get_switch("SKIPONCARD");

  // check for TUNER options
  if (tuner) {
    i = ired;
    ired = pic_tuner_dec (p,ired,-1,0);
    if (ired!=i) m_warning ("Tuner dec changed to "+ired);
    host = m_get_sswitch_def("HOST",0);
    if (host) m_error ("Host TUNER mode not supported yet");
  } else if (type==IOPT_CORE) {
    host = m_get_sswitch_def("HOST",0);
  } else {
    host = m_get_sswitch_def("HOST",(skiponcard>0)?0:1);
  }

  // fixup deltas
  idelta = abs(ired);

  // get tuner frequency / gain
  tfreq = m_dpick(5);
  tgain = m_lpick(6);

  // get the transfer length switch
  tl = m_get_switch("TL");
  if (tl<=0) tl = max(1, NINT(L_bbuf/HIN.dbpe));
  tskip = m_get_switch("SKIP");
  swap = m_get_switch("SWAP");
  wait = m_get_dswitch("WAIT");
  if (wait<0) wait = m_get_dswitch("PAUSE");
  pause = m_get_dswitch_def("POLL",0.025);
  maxout = m_get_dswitch("MAXOUT");
  timeout = m_get_dswitch_def("TIMEOUT",1.0);
  dtfreq = m_get_dswitch_def("DFREQ",0.0);
  stats = m_get_pswitch("STATS");
  syncdata = m_get_pswitch("SYNCDATA");
  flush = m_get_pswitch("FLUSH");

  // can we allow blocking this file
  noblock = HIN.ape>1 || HIN.spa>2;

  // enable autorestart feature ?
  autors = m_get_switch_def("AUTORS",0);

  // enable multi-channel mode ?
  multi = m_get_switch_def("MULTI",1);
  autoss = m_get_sswitch("AUTOSS");
  renum = m_get_switch("RENUM");

  // fill out header and open output file
  m_propagate (HIN,HOUT);
  HOUT.file_name = m_apick(2);
  if (host) {
    if (HIN.class_ == 2) HOUT.ydelta = HIN.ydelta * idelta;
    else                HOUT.xdelta = HIN.xdelta * idelta;
    HOUT.size  = DINT( (HIN.size-1.0)/idelta ) + 1.0;
    rate  = 1.0 / max(1.0e-10,HIN.xdelta);
  } else if (type==IOPT_CORE) {
    rate  = ired / max(1.0e-10,HIN.xdelta);
  } else if (type!=IOPT_MODULE) {
    rate  = HIN.spa * ired / max(1.0e-10,HIN.xdelta);
  } else {
    rate  = 1.0 / max(1.0e-10,HIN.xdelta);
  }
  if (ratio > 0.0) rate = HIN.spa * ired / max(1.0e-10,HIN.xdelta*ratio);
  m_open (HOUT, HCBF_OUTPUT+HCBF_OPTIONAL);

  // get archive parameters 
  archid   = 0;
  archdur  = -1;
  archtime = 0;
  archmode = ARCH_OFF;
  i = m_get_uswitch("ARCH",archname);
  if (i>0) archmode = ARCH_WAIT;
  archtl = m_get_switch_def("ARCHTL",128*1024);
  archsf = m_get_switch("ARCHSF");

  // tuner bank specific startup
  if (type == IOPT_TBANK || type == IOPT_CORE) {
    nchan = pic_getkeyl (p,port,KEY_CHNS);
    if (type == IOPT_CORE) nchan = 1;
    nchan = m_get_switch_def("NCHN",nchan);
    pic_setkeyl(p,port,KEY_CHNS,nchan);
    i = m_get_switch_def("PKTLEN",tl*HIN.bpe);
    pic_setkeyl(p,port,KEY_PKTLEN,i);
    pic_setkeyd(p,port,KEY_DFREQ,2*dtfreq/rate);
    if (renum<=0) renum = 1;
  } else {
    nchan = 1;
  }

  // get the timecode sampling rate
  tcpp = m_get_switch("TCPP");
  tcps = m_get_dswitch("TCPS");
  if (tcps>0 && tcpp<0) tcpp = max(1.0,1.0/(tcps*HIN.xdelta*HIN.ape*tl*max(1,tskip))+.5);

  // get the packet mode and initialize
  pktmod = m_get_switch("PKTMODE");
  packet = m_get_uswitch("PACKET",HPACK.file_name);
  apacket = m_get_uswitch("APACKET",afpext);
  if (packet==0) {
    ipkt = ICE_HDR_SZ / HOUT.dbpe;
    if (ipkt<=0) m_error ("Int Packet Mode Format Incompatibility");
    if (tl<=0) m_error ("Packet mode must specify transfer length /TL=n");
  }
  if (apacket==0) {
    ipkt = ICE_HDR_SZ / HIN.bpa;
    if (ipkt<=0) m_error ("Int Packet Mode Format Incompatibility");
    if (archtl<=0) m_error ("Packet mode must specify transfer length /TL=n");
  }
  if (packet>0 || apacket>0) {
    m_init (HPACK,HPACK.file_name,"3000","NH",0);
    m_addsubr (HPACK,"KEYS","4B");
    m_addsubr (HPACK,"CNT ","SL");
    m_addsubr (HPACK,"ELEM","SL");
    m_addsubr (HPACK,"USER","SI");
    m_addsubr (HPACK,"CHAN","SI");
    m_addsubr (HPACK,"SID ","2B");
    m_addsubr (HPACK,"REP ","SB");
    m_addsubr (HPACK,"BPA ","SB");
    m_addsubr (HPACK,"MODE","SB");
    m_addsubr (HPACK,"TYPE","SB");
    m_addsubr (HPACK,"TCM ","SB");
    m_addsubr (HPACK,"TCS ","SB");
    m_addsubr (HPACK,"TCO ","SD");
    m_addsubr (HPACK,"TCWS","SD");
    m_addsubr (HPACK,"TCFS","SD");
   if (pktmod==1) {
    m_addsubr (HPACK,"XDEL","SD");
    m_addsubr (HPACK,"XSTA","SD");
   } else if (pktmod==2 || pktmod==3) {
    m_addsubr (HPACK,"RATE","SD");
    m_addsubr (HPACK,"FREQ","SD");
   } else {
    m_addsubr (HPACK,"RBST","SL");
    m_addsubr (HPACK,"RBSZ","SL");
    m_addsubr (HPACK,"RBDO","SL");
    m_addsubr (HPACK,"RBDS","SL");
   }
    HAPACK.hcb = HPACK.hcb;
  }
  if (packet>0) m_open (HPACK, HCBF_OUTPUT);

  // get startup modes
  mpp = 0;
  msync = 0;
  msyncid = m_get_switch("MASTER");
  if (msyncid>0) msync = 1;
  i = m_get_uswitch("SLAVE",tmpstr);
  if (i>=0) msync = 2;
  sss = (i==2 && tmpstr=="SS"); // same side slave
  xts = (i==2 && tmpstr=="XT"); // external trigger slave
  mgo = -1;			// non-piped default
  if (Mc->mode!=0) mgo = 2;	// piped default
  mgo = m_get_switch_def("REPLAY",mgo);
  if (msync>=2) mgo = 0;	// slave wait

  // get results labels
  if (m_get_uswitch ("SYNC", synclab)<=0) synclab = "";
  if (m_get_uswitch ("LOST", lostlab)<=0) lostlab = "";
  if (m_get_uswitch ("TCSTAT", tcslab)<=0) tcslab = "";
  if (m_get_uswitch ("SEQERR", seqerr)<=0) seqerr = "";
  if (m_get_uswitch ("SEQFILL", seqfill)<=0) seqfill = "";
  if (m_get_uswitch ("PFULL", statlab)<=0) statlab = "";
  if (m_get_uswitch ("NTPOFF", ntplab)<=0) ntplab = "";

  // init the PIC device parameters
  dir   = -1;
  ape   = HOUT.ape;
  spa   = HOUT.spa;
  epb   = HOUT.size / max(nchan,multi);
  bps   = HIN.bps;
  dbpe  = HIN.dbpe;
  bits  = HIN.dbpe / HIN.ape / HIN.spa * 8;
  bits  = m_get_switch_def ("BITS",bits);
  if (spa == 2) bits = -bits;
  rate  = m_get_dswitch_def ("SRATE",rate);
  if (tfreq==-1) tfreq = rate * 0.25;
  if (!host) idelta = 1;
  delta = HOUT.xdelta;
  tfreq = (rate/2) * pic_tuner_freq (p,2*tfreq/rate,0);
  nblock = max(0,tl) * HIN.dbpe;
  if (tskip>1 || idelta>1) nblock = -1;

  // set up real time controls
  idm = m_mwinit("Replay Mode",mgo,"REPLAY;Play One,Continuous,Stop Top,Stop Now,Spin,Archive,ReStart,Abort,NewDevice");
  ids = m_mwinit("Sync Mode",msync,"Master,Slave Wait,Slave Run");
  idr = m_dwinit("Sample Rate", rate, 1.0, 100.e6, 1.e3);
  if (tuner) idd = m_lwinit("Tuner Dec", ired, 1, 32768, 2);
  else       idd = m_lwinit("Reduction", ired, 0, 2048, 1);
  idf = m_dwinit("Tuner Freq", tfreq, -rate*.5, rate*.5,rate*.005);
  idg = m_lwinit("Tuner Gain", tgain, -100, 100, 1);
  idc = m_lwinit("Curr Cycle", 0, 1, -1, 1);
  idlh = m_lwinit("Lost HostBuf", 0, 1, -1, 1);
  idlc = m_lwinit("Lost CardBuf", 0, 1, -1, 1);
  if (ratio>0.0) idrat = m_dwinit("Resampler", ratio, .5, 2.0,.001);
  if (alg>0) ida = m_mwinit("Algorithm", alg, "Noop,User,Swap,LUT,AM,FM,PM,PSK,QPSK");
  for (n=1; n<=nargs; n++) m_lwinit("Arg"+n, args[n], 1, -1, 1);

  // realtime verbose mode
  idv = 0;
  verbose = m_get_switch("VERBOSE");
  if (verbose>=0) idv = m_lwinit("Verbose", verbose, 0, 5, 1);

  // init the PIC device for IO
  fmode = m_get_switch_def("FMODE",1);
  pic_setkeyd (p, 0, KEY_RATIO, ratio);
  LABEL_111:
  status = pic_mapfile (p, &map, &HIN.hcb, fmode);
  if (status<=0) m_error ("Unacceptable DMA input file");
  tinc = pic_getkeyl (p,0,KEY_TINC);
  mnbytes = map.bytes/multi;
  m_move(&map,&mapi,sizeof(map));
  mapi.bytes = mnbytes;
  for (n=1; n<=multi; n++) {
    if (n==1 && msync<=1) {
      flags = 0;
    } else if (n==1 && msync>1) {
      flags = FLG_SGO;
      if (sss) flags = FLG_RGO;
      if (xts) flags = FLG_XTGO;
    } else if (n==2 && !autoss) {
      flags = FLG_SGO;
    } else {
      flags = FLG_RGO;
    }
    dmac = pic_ioport (p, type, port, feed, dir, bits, NINT(rate), 2*(tfreq+(n-1)*dtfreq)/rate, ired, tgain, flags);
    if (dmac<=0) m_error ("Bad I/O Port parameters");
    mapi.offset = (n-1)*mnbytes;
    status = pic_dmasetup (p, dmac, dir, &mapi, nblock, 0);
    if (status<=0) m_error ("Unacceptable channel for DMA");
    if (swap>0) pic_setkeyl(p,dmac,KEY_SWAP,swap);
    mdmac[n] = dmac;
    mport[n] = port;
    if (autoss)           port += tinc;
    else if (n%2==1) port += 1;
    else                  port += tinc-1;
  }
  dmac = mdmac[1];
  port = mport[1];
  for (n=1; n<=nargs; n++) pic_setkeyl(p,dmac,KEY_ARGS+n-1,args[n]);
  

  // report the DMA channel
  if (m_get_uswitch("DMAC",tmpstr)>0) m_lrslt(tmpstr,dmac);
  nodma = pic_getkeyl (p,dmac,KEY_NODMA);

  // setup framed decimation
  if (skiponcard>0) {
      pic_setkeyl (p, dmac, KEY_FRAME, HIN.subsize);
      pic_setkeyl (p, dmac, KEY_FRAMEDEC, tskip);
      tskip = 1;
  }
  else if (type==IOPT_MODULE && HIN.class_==2 && !host) {
    pic_setkeyl (p, dmac, KEY_FRAME, HIN.subsize);
    pic_setkeyl (p, dmac, KEY_DEC, ired);
  }

  // setup tuner resampler
  if (ratio>0.0) {
    pic_setkey (p, dmac, KEY_RATIO, &ratio, 8);
    pic_getkey (p, dmac, KEY_RATIO, &ratio, 8);
    m_dwput(idrat,ratio);
  }

  // init the packet parameters
  tcm = pic_getkeyl (p, dmac, KEY_TCMODE);
  pkt.keys[0] = 101;
  pkt.keys[1] = 102;
  pkt.keys[2] = 103;
  pkt.keys[3] = 0;
  pkt.channel = port;
  pkt.rep = HOUT.data_rep[0];
  pkt.bpa = HOUT.bpa;
  pkt.mode = HOUT.format[0];
  pkt.type = HOUT.format[1];
  pkt.tcmode = tcm;
  pkt.tcstatus = 0;
  pkt.tcwsec = 0;
  pkt.tcfsec = 0;
  pkt.tcoff = 0;
  pkt.elem = tl*HOUT.ape;
  pkt.u.R.ramphys = map.paddr;
  pkt.u.R.ramsize = map.bytes;
  tcstatuslast = 0;

  // check packet size / buffer ratio
  if (apacket>=0) {
    iper =  mnbytes / nchan / (archtl*HOUT.bpa);
    if ( iper*nchan*(archtl*HOUT.bpa) != mnbytes) m_error ("Host buffer must be multiple of archive packet size");
    apkt = pkt;
  } else {
    archtl = 1;
  }
  if (packet>=0) {
    iper =  mnbytes / nchan / (tl*HOUT.bpe);
    if ( iper*nchan*(tl*HOUT.bpe) != mnbytes) m_error ("Host buffer must be multiple of packet size");
    if (tcpp<0) tcpp = iper / max(1,tskip);
  }

  // initialize parameters
  nbuf = 0;
  npass = 0;
  ncycle = 1;
  ramsync (P_SYNCINIT);
  if (mgo>=9 && mgo<=11) {
    mgo = 0; // indicate ready
    if (autors==-3) mgo = 2;
    m_mwput(idm,mgo);
    if (msyncid>0) m_mwput(msyncid-1,mgo);
    goto LABEL_1;
  } else {
    outoff = 0;
    lastout = 0;
    nlost = 0;
    mlost = 0;
  }

  // signal start of processing block
  m_sync();

  // release resources
  if (mgo==0) {
    for (n=1; n<=multi; n++) {
      status = pic_dmafunc (p,mdmac[n],DMA_RESET);
    }
  }

  // initial slave sync up
  if (msyncid>0) m_mwput(msyncid-1,mgo);
  LABEL_101:
  if (wait>0) m_pause(wait);

  // top of throttle loop
  LABEL_1:
  while (mgo==0 || mgo==3 || mgo==4 || mgo==7 || mgo==99) {
    update(0);
    isok = m_mwget(idm,mgo);
    if (mgo==7) { // restart ?
      if (autors>=2) goto LABEL_99;
      mgo = 2; // already stopped - just start
      m_mwput(idm,mgo);
    } else if (mgo==3 || mgo==4) {
      mgo = 0;
      m_mwput(idm,mgo);
    }
    // validate tuner decimation 
    if (mgo!=0 && tuner && mgo!=99) {
      ired = pic_tuner_dec (p,ired,-1,0);
      m_lwput (idd, ired);
    }
    if (msyncid>0) m_mwput(msyncid-1,mgo);
    if (mgo>=8 && mgo<=11) goto LABEL_999;
    if (Mc->break_) goto LABEL_999;
    m_pause(pause);
  }

  // get stride
  // reduction widget was polled in update
  if (ired==0) ired = 1;
  if (!host) {
    idelta = 1;	// thinning is done in card 
  } else {
    if (ired<-1 && noblock) {
      m_warning("Cannot perform blocking on input data type");
      ired = 1;
    }
    idelta = abs(ired);
  }

  // calculate copy and stride lengths in bytes
  iskip = 1;
  icopy = max( 1.0, HIN.dbpe);
  mdbpe = max( HIN.dbpe, HOUT.dbpe );
  if (!host || icopy*(ired-1)<512) {	// small stride
    iper = tl;
    if (iper<=0) iper = max(1, NINT(L_bbuf/mdbpe));
    if (tskip > 1) {
      iper = 1;
      iskip = tskip;
    }
  } else {				// large stride
    iper = 1;
    iskip = idelta;
    if (tskip>1) iskip = iskip*tskip;
  }
  ibuf = iper*mdbpe;
  if (ibuf>nbuf) {
    if (nbuf>0) free (buf);
    nbuf = ibuf;
    buf = (int_1*)malloc (nbuf);
  }
  HOUT.buf_type = HIN.buf_type;
  tcdelta = -1;				// reset tctolr check

  // make sure the slaved PIC device is running
  if (msyncid>0) {	
    isok = m_mwget (msyncid,ssync);
    while (ssync==2 && !Mc->break_) {	// wait for slave
      m_pause (pause);
      isok = m_mwget (msyncid,ssync);
    }
  }

  // make sure the PIC device is running
  if (mpp!=1) {
    for (n=multi; n>=1; n-=1) {
      if (nodma>0) {
        mgo = 5;
        m_mwput(idm,mgo);
        status = pic_dmafunc (p,mdmac[n],DMA_ENABLE);
      } else if ((mgo==2 || mgo==5 || mgo<-1) && autors!=-2) {
        status = pic_dmafunc (p,mdmac[n],DMA_CONTINUOUS);
      } else {
        status = pic_dmafunc (p,mdmac[n],DMA_ONESHOT);
      }
    }
    if (status<0) {
      m_warning("Problem starting port: "+devlab);
      mgo = 99;
      m_mwput(idm,mgo);
      goto LABEL_1;
    }
    mpp = 1;
    ramsync (P_SYNCSET);
    if (msync>1) m_mwput (ids,3);	// report slave running
    pkt.count = 0;
    pkt_count = 0;
    tccnt = 0;
  } 

  // spin
  while (mgo==5) {
    m_pause(pause);
    update(0);
    isok = m_mwget(idm,mgo);
    if (Mc->break_) goto LABEL_999;
  }

  // archive 
  if (mgo==6) {
    mpp = 2;
    status = pic_dmafunc (p,dmac,DMA_STOP);
    nstop = status/(HIN.dbpe*idelta);		// logical stop offset
    nstop = max((int_8)0,min((int_8)HIN.size,nstop));
    if (msyncid>0) m_mwput (msyncid-1,mgo);	// trigger slave 
    npass = ncycle-1-nlost;
  }

  // update the pass through RAM buffer count
  LABEL_2:
  npass = npass + 1;
  m_lwput (idc,npass);

  // set xfer parameters
  LABEL_3:
  dnext = 0;
  adone = 0;
  ntogo = HIN.size / multi;

  // debug statistics output
  if (stats || statlab!="") {
    status = pic_dmafuncx (p,dmac,DMA_STATUS);
    fstat = (double)status*100/mnbytes;
    if (stats) printf("Sourcepic top pass=%d cyc=%d favg=%f\n",npass,ncycle,fstat);
    if (statlab!="") m_frslt (statlab,fstat);
  }

  // adjust start/stop pointers for split buffer archive mode
  if (mgo==6) {
    if (nstop>=0) {
      dnext = nstop;
      ntogo = HIN.size/multi - nstop;
      nstop = -nstop;
    } else {
      ntogo = -nstop;
      nstop = 0;
      npass = npass - 1;
    }

  // warn if we are falling way behind
  } else if (ncycle>npass+nlost && mgo!=1) {
    i = ncycle-(npass+nlost); tmpstr=""+i;
    m_warning ("Fell behind "+tmpstr+" RAM buffers");
    nlost = (ncycle-npass);
    m_lrslt (lostlab,nlost);
  }

  // wait for current data pointer
  elapse = 0;
  while (npass+nlost>ncycle && mgo<7) {
    ramsync (P_SYNCSTAT);
    update(1);
    m_pause (pause);
    elapse = elapse + pause;
    if (autors>=2 && elapse>timeout) goto LABEL_99;
    if (Mc->break_) goto LABEL_999;
  }

  // loop through buffers of data  (data buffer loop)
  while (ntogo>0 && !Mc->break_) {
    nget = min((int_8)iper,ntogo);
    ramsync (P_SYNCSTAT);

    // get new real time controls
    update(1);

    // check for falling behind
    if (npass+nlost==ncycle-1) {
      status = pic_dmafuncx (p,dmac,DMA_STATUS);
      nnext = dnext*HIN.dbpe;
      if (status > nnext) {
        m_warning ("Falling behind one RAM buffer");
        nlost = nlost + 1;
        m_lrslt (lostlab,nlost);
      }
    }

    // add timecode to packet header (top of buffer or per packet)
    if (tcmode!="OFF") {
      if (pkt_count==0) {
        // wait for the input clock to be active
        elapse = 0;
        while (pic_dmafuncx (p,dmac,DMA_STATUS) == 0 && (elapse<timeout || autors==0)) {
          update(1);
          m_mwget(idm,i);
          if (i==4 || i>=7 || Mc->break_) goto LABEL_97;
          m_pause(pause);
          elapse = elapse+pause;
        }
        if (HOUT.pipe==0) { dval=0.0;
          status = pic_tc (p, dmac, &dval, &delta, &pkt.tcwsec, &pkt.tcfsec, FLG_TCINTERP);
          m_put_epoch (HOUT, tcoff+pkt.tcwsec, pkt.tcfsec, false);
          m_update_header (HOUT);
        }
      }
      tccnt = tccnt + 1;
      if ( (pkt_count==0) || (tcpp>0 && tccnt>=tcpp) ) {
        tccnt = max(0,tccnt-tcpp);
        pkt.tcstatus = pic_tc (p, dmac, &pkt.tcoff, &delta, &pkt.tcwsec, &pkt.tcfsec, 0);
        if (pkt.tcstatus > 0) {
  	  dtime = pkt.tcwsec - tcwlast;
          if (pkt.tcwsec < 100.0) {	// 1st 100 sec of year
            j = NINT(-dtime/86400);
            if (j==365 || j==366) {	// assume ray day
              tcoff = tcoff + j*86400;
              dtime = dtime + j*86400;
            }
          }
          dtime = dtime + pkt.tcfsec - tcflast;
          dsamp = pkt.tcoff-tcofflast;
          tcerr = dtime - dsamp*tcdelta;
          if (pkt_count>1 && tctolr>0 &&  tcdelta>0 && fabs(tcerr)>tctolr) {
            if (info) printf("Time code slip=%f > tolr=%f cnt=%d dmac=%d\n",tcerr,tctolr,pkt.count,dmac);
            if (autors>0) { m_mwput (idm,7); goto LABEL_97; }
          }
  	  if (pkt_count>0 && dsamp>1) {
  	    tcdelta = dtime/dsamp;
  	    if (tcdelta<=0) tcdelta = delta;
          }
          tcwlast = pkt.tcwsec;
          tcflast = pkt.tcfsec;
          tcofflast = pkt.tcoff;
          // now adjust to be relative to the packet start
          pkt.tcwsec = pkt.tcwsec + tcoff;
          if (apacket >= 0) {
            apkt.tcwsec = pkt.tcwsec;
            apkt.tcfsec = pkt.tcfsec;
            apkt.tcstatus = pkt.tcstatus;
            apkt.tcoff = pkt.tcoff - (double)(nlost*epb*ape) - apkt_count*(double)(apkt.elem);
          }
          pkt.tcoff = pkt.tcoff - (double)(nlost*epb*ape) - pkt_count*(double)(pkt.elem*iskip);
          if (tcslab!="") m_drslt (tcslab,pkt.tcwsec+pkt.tcfsec);
        }
        else {
          if (tcslab!="") m_drslt (tcslab,(double)(pkt.tcstatus));
          if ((autors<=0 || pkt.tcstatus!=tcstatuslast) && info) printf("*** Bad TC status=%d. See PIC HELP TC.\n",pkt.tcstatus);
          if (autors>0) { m_mwput (idm,7); goto LABEL_97; }
        }
        tcstatuslast = pkt.tcstatus;
      }
    }

    // check for max output branch
    if (maxout>0.0 && HOUT.offset+nget>=maxout) {
      nget = maxout - HOUT.offset;
      if (nget<=0) { m_mwput(idm,8); goto LABEL_97; }
    }

    // top of multi-channel data loop
    for (n=1; n<=max(nchan,multi); n++) {

    // wait until data is available
    if (npass+nlost>=ncycle) {
      if (archmode==ARCH_RT) {
        nnext = (dnext+nget*iskip)*HIN.dbpe;
      } else if (nchan>1) {
        nnext = (dnext+n*nget)*HIN.dbpe;
      } else {
        nnext = (dnext+nget)*HIN.dbpe;
      }
      if (multi>1) {
        status = pic_dmafuncx (p,mdmac[n],DMA_STATUS);
      } else {
        status = pic_dmafuncx (p,dmac,DMA_STATUS);
      }
      elapse = 0;
      while (status<nnext && npass+nlost>=ncycle && mgo!=7 && !Mc->break_) {
        update(1);
        m_pause(pause);
        ramsync (P_SYNCSTAT);
        elapse = elapse+pause;
        if (autors>0 && elapse>timeout) {
          m_mwput(idm,7);
  	  m_warning("Detected DMA stall - auto restarting");
  	  if (n==1) goto LABEL_97;
        }
        j = m_mwget(idm,i);
        if (n==1 && (i==4 || i==7 || i==8)) goto LABEL_97;
        if (multi>1) {
          status = pic_dmafuncx (p,mdmac[n],DMA_STATUS);
        } else {
          status = pic_dmafuncx (p,dmac,DMA_STATUS);
        }
      }
    }

    // get the data - its already in memory - go for speed
    out = (int_1*)(map.vaddr + (int_8)(dnext*HIN.dbpe));
    if (multi>1) out = out + (n-1)*mnbytes;
    if (nchan>1) out = out + (n-1)*nget*NINT(HIN.dbpe);
    ngot = nget;

    // large or no skip
    if (iper==1 || idelta==1) {	
      npot = ngot;

    // regular thinning/blocking operation
    } else {				
      // move is smart enough not to move onto itself
      m_move (out,buf,ngot*HIN.bpe);
      out = buf;	// FILAD now from buf
      npot = (ngot+idelta-1)/idelta;
      if (ired>=0) {				// thinning
        m_vmovn (buf,idelta,buf,1,npot,icopy);
      } else {					// blocking
        HOUT.buf_type = HOUT.format[1];
        m_reformat (buf,HIN.format, buf,HOUT.buf_type, npot*idelta);
        m_vtype (HOUT.buf_type);
        if (spa == 2) m_cvblk (buf,buf,idelta,ngot);
        else          m_vblk (buf,buf,idelta,ngot);
      }
    }

    // output packet headers and data to pipes/files
    i = npot * HOUT.dbpe;
    if (packet>=0) {
      if (multi>1) pkt.channel = mport[n];
      if (renum>0) pkt.channel = renum-1+n;
      pkt.elem = npot * HOUT.ape;
      pkt.u.R.dataoffset = (int_4)(out-(int_1*)map.vaddr);
      pkt.u.R.datasize = i;
      if (pktmod==1) {
        pkt.u.X.delta = delta;
        pkt.u.X.start = pkt.tcwsec+pkt.tcfsec-pkt.tcoff*delta;
        pkt.u.X.start = pkt.u.X.start - pkt_count*pkt.elem*delta;
      } else if (pktmod==2) {
        pkt.u.X.delta = rate;
        pkt.u.X.start = tfreq;
      } else if (pktmod==3) {
        pkt.u.X.delta = rate;
        pkt.u.X.start = pic_getkeyd(p,dmac,KEY_RFFREQ);
      }
      if (packet==0) m_filad (HOUT,&pkt,ipkt);
      if (packet>0) m_filad (HPACK,&pkt,1);
    }
    if (HOUT.open) m_filad (HOUT,out,npot);
    lastout = HOUT.offset;

    } // bottom of multi-channel data loop

    // write out real time archive data
    if (archmode==ARCH_RT) {
      nnext = min(HIN.size,dnext+nget*iskip) * HIN.ape;
      while (nnext-adone >= archtl) {
        for (n=1; n<=max(nchan,multi); n++) {
          out = (int_1*)(map.vaddr + adone*HIN.bpa);
          if (multi>1) out = out + (n-1)*mnbytes;
          if (nchan>1) out = out + (n-1)*ngot*HIN.bpa;
          i = nnext-adone;
          if (apacket>=0) {
            i = min(archtl,i);
    	    apkt.elem = i;
  	    if (multi>1) apkt.channel = mport[n];
  	    if (renum>0) apkt.channel = renum-1+n;
  	    if (apacket==0) m_filad (HARCH,&apkt,ipkt);
  	    if (apacket>0) m_filad (HAPACK,&apkt,1);
          }
          m_filad (HARCH,out,i);
        }
        apkt.count = apkt.count+1;
        apkt_count = apkt_count+1;
        apkt.tcoff = apkt.tcoff - apkt.elem;
        adone = adone + i*nchan;
      }
      if (archdur>=0 && HARCH.offset>=HARCH.size) {
        m_close(HARCH);
        if (apacket>0) m_close(HAPACK);
        archmode = ARCH_OFF;
        if (archid!=0) m_send_msgl ("ARCHDONE",0,archid,0,0,0,"",0,0);
      }
    }

    // update pointers
    pkt.count = pkt.count + 1;
    pkt_count = pkt_count + 1;
    if (packet>=0) pkt.tcoff = pkt.tcoff - pkt.elem*iskip;
    i = nchan*ngot*iskip;
    ntogo = ntogo - i;
    dnext = dnext + i;

    // update realtime file fields
    if (rtfile) {
      HIN.in_byte = (dnext+(npass-1)*HIN.size)*HIN.dbpe;
      m_update_header (HIN);
    }

    // check for large misalignment of slave
    if (ntogo==0 && autors>0 && syncdata) {
      j = m_mwget (msyncid+5,i);
      if ( abs(i-npass) >= 2) {
        m_warning ("Large Master/Slave misalignment - restarting");
        m_mwput (idm,7);
      }
    }

    // check throttle control widget
    LABEL_97:
    isok = m_mwget (idm,mgo);
    if (msyncid>0 && isok>0) {
      isok = m_mwget (msyncid+5,i);
      while (syncdata && i<npass && !Mc->break_) {
        m_pause(pause);
        isok = m_mwget (msyncid+5,i);
      }
      m_mwput (msyncid-1,mgo);
    }
    if (mgo==4 || mgo==5) ntogo = 0;
    if (mgo==7 || mgo==8) ntogo = 0;

  }

  // update the status widgets
  LABEL_99:
  m_lwput(idlh,nlost);
  i = pic_dmafunc(p,dmac,DMA_LOST);
  m_lwput(idlc,i);
  if (i>mlost) m_warning("Lost one or more Card Buffers");
  mlost = i;
  if (seqfill!="") m_lrslt(seqfill, pic_getkeyl(p,dmac,KEY_SEQFILL));
  if (seqerr!="") m_lrslt(seqerr, pic_getkeyl(p,dmac,KEY_SEQERR));
  if (ntplab!="") m_drslt(ntplab, pic_getkeyd(p,dmac,KEY_NTPOFF));

  // flush packet header
  if (HPACK.open && HPACK.pipe==0) {
    m_hcbfunc(HPACK,HCBF_FLUSH,0);
  }
  if (archmode==ARCH_RT && flush) {
    m_hcbfunc(HARCH,HCBF_FLUSH,0);
    if (apacket>0) m_hcbfunc(HAPACK,HCBF_FLUSH,0);
  }

  // stop PIC if necessary
  if (mgo==1 || mgo==-1 || mgo==3 || mgo==4 || mgo>=7 || autors<0) {
    if (mpp==1 && msyncid>0) {
      isok = m_mwget (msyncid+5,i);
      if (syncdata && i>npass) goto LABEL_2;	// do one more
      isok = m_mwget (msyncid,ssync);
      while (ssync==3 && !Mc->break_) {	// wait for stop
        m_pause (pause);
        isok = m_mwget (msyncid,ssync);
      }
      isok = m_mwget (msyncid+5,i);
      if (syncdata && i>npass) goto LABEL_2;	// do one more
    }
    if (mgo==7 && autors>=2) {
      m_pause(0.1);
      // make sure the other channels will at least DMA stall
      status = pic_dmafunc(p,dmac,DMA_KILL);
    }
    for (n=1; n<=multi; n++) {
      if (mpp==1) status = pic_dmafunc (p,mdmac[n],DMA_STOP);
      if (mpp==1) status = pic_dmafunc (p,mdmac[n],DMA_RESET);
    }
    mpp = 0;
    npass = 0;
    nlost = 0;
    mlost = 0;
    ncycle = 1;
    outoff = lastout;
    pkt.count = 0;
    pkt_count = 0;
    tcstatuslast = 0;
    if (msync>1) {
      if (mgo==7) {
        mgo = 0; // slave reports stopped
        m_mwput (idm,mgo);
      }
      m_mwput (ids,2);	// report slave wait
    }
    if (autors==-3) mgo = 11;	// reopen
    elapse = -1;	// extra second to wait for other channels
    while (mgo==7 && autors>=2 && elapse<timeout && pic_dmafunc(p,dmac,DMA_ACTIVE)>0) {
      m_pause(0.1);
      elapse = elapse + 0.1;
      if (elapse>=timeout) {
        m_warning ("All DMA channels on this input did not shut down");
        if (autors>=3) {
  	  status = pic_dmafunc(p,dmac,DMA_BURY);
  	  m_warning ("Buried the non-responsive channels");
        }
      }
    }
    if (autors>=2) m_pause(0.5);
  } else if (mgo==6) {
    if (nstop!=0) goto LABEL_3;
    mgo = 5;
    m_mwput (idm,mgo);
  }

  if (archmode==ARCH_RT && archsf>0) {
    m_close(HARCH);
    if (apacket>0) m_close(HAPACK);
    archmode = ARCH_WAIT;
  }

  // go back for replay
  if (mgo>=0) {
    if (mgo==1||mgo==3||mgo==4) {
      mgo = 0;
      m_mwput (idm,mgo);
    }
    if (mgo==2 && !Mc->break_ && autors!=-2) goto LABEL_2;
    if (mgo==5 && !Mc->break_) goto LABEL_1;
    if (mgo==7) {
      mgo = 2;
      if (autors>=4) mgo = autors;
      m_mwput (idm,mgo);
      if (msyncid>0) m_mwput(msyncid-1,mgo);
    }
    if (mgo<8 && !Mc->break_) goto LABEL_101;
  } else {
    mgo = mgo + 1; // replicate mode
    m_mwput (idm,mgo);
    if (mgo<0 && !Mc->break_) goto LABEL_1;
  }

  // close PIC device
  LABEL_999:
  for (n=1; n<=multi; n++) {
    status = pic_dmafunc (p,mdmac[n],DMA_CANCEL);
  }
  status = pic_mapfile (p, &map, &HIN.hcb, -fmode);
  if (mgo!=11) status = pic_close (p);

  // restart with new device ?
  if (mgo>=9 && mgo<=11 && !Mc->break_) {
    if (mgo==9 && m_rfind(devlab,devstr,dummy)!='A') m_error ("Device argument not a proper result name: " +devlab);
    if (mgo!=11) {
      i  = m_hwf_open (HWF);
      ls = m_hwf_alias (HWF, devstr, str);
      i  = m_hwf_close (HWF);
      str = flgstr+str;;
      status = pic_open (p,s2c(str),(int_4*)&Mc->break_,flags);
      if (status <= 0) m_error ("ReOpening port: "+str);
    }
    flags = 0;
    if (boot) flags = flags || FLG_BOOT;
    if (p->ptype!=type) m_error ("Not allowed to dynamically change port to different type");
    if (tuner) {
      ired = pic_tuner_dec (p,ired,-1,0);
      m_lwput (idd, ired);
    }
    port = p->pindex;
    dmac = -1;
    goto LABEL_111;
  }

  // close input/output files
  m_close (HIN);
  m_close (HOUT);
  if (packet>0) m_close (HPACK);
  if (archmode!=ARCH_OFF) {
    m_close (HARCH);
    if (apacket>0) m_close(HAPACK);
  }
}


void sourcepic::ramsync (int_4 mode)
{
  real_4 time;

  if (mode==P_SYNCINIT) {		// to initialize
    statuslast = 0;
    return;
  }
  status = pic_dmafuncx (p,dmac,DMA_STATUS);
  if (status==-1) m_error ("ICE-PIC is down");
  status = max((int_8)0,min(mnbytes,status));
  if (status<statuslast || mode==P_SYNCSET) { 	// next buffer
    m_secnds(0.0,time);
    time = time - status/(rate*dbpe/ape);
    m_frslt (synclab, time);
    ncycle = pic_dmafunc (p,dmac,DMA_CYCLE) + 1;	// 0->1 based
  } else if (status==0 && mgo==1) {
    // check for small buffer fast completion case
    ncycle = pic_dmafunc (p,dmac,DMA_CYCLE) + 1;	// 0->1 based
  }
  statuslast = status;
}


void sourcepic::update (int_4 mode)
{
  real_8 time, trigger, soy, fsoy, sample, timet, timed;
  int_4  ratex, iredx, freqx, gainx, algx, argx, ratx;
  int_4  nelem, nm, lostoff, stat, cycle, index;
  bool_4 stopping = false;
  real_8 ttfreq;

  int_4 larch; int_1 *parch;

  MQH.info = 0;	// set for multi-mode channel queue
  // look for messages
  if (Mu->id <= 0) goto LABEL_100;
  if (stopping) {
    if (mgo!=0) goto LABEL_100;
    trigger = 0;
    MQH.info = 0;
    stopping = false;
    goto LABEL_1;
  }
  if (m_get_msg(MQH,MQD,0.0,0)<=0) goto LABEL_100;
  LABEL_1:
  if (MQH.name == "MGO") {
    if (MQH.ndata>0 && MQD.dbuf[0]>0) {
      trigger = MQD.dbuf[0];
    } else {
      trigger = 0;
    }
    LABEL_2:
    m_now (time1,time2);
    time = time1 + time2;
    if (MQH.info==0 && mgo!=0) {
      if (mgo==2 || mgo>4) m_mwput (idm,4);
      stopping = true;	// now we are
      goto LABEL_100;
    } else if (trigger==0) {
    } else if (trigger-time<=0) {
      // take it now
    } else if (trigger-time>600.0) {
      m_warning ("Huge trigger offset in message");
    } else {
      m_pause (Mc->pause);
      goto LABEL_2;
    }
    m_mwput (idm, MQH.info);
    m_send_msgl ("=MGO",0,MQH.sender,MQH.info,0,1,"D",&time,0);
  }
  else if (MQH.name == "DEVICE") {
    if (mgo!=0) m_error ("DEVICE message can only be processed on a stopped port");
    if (MQH.info > 0) {
      p->pindex = MQH.info;
      mgo = 11;
    } else {
      i = m_length(string(MQD.sbuf,40));
      devstr = string(MQD.sbuf,i);
      mgo = 10;
    }
    m_mwput (idm, mgo);
  }
  else if (MQH.name == "RFDG") {
    m_lwput (idr, NINT(MQD.dbuf[0]) );
    m_dwput (idf, MQD.dbuf[1] );
    m_lwput (idd, NINT(MQD.dbuf[2]) );
    m_lwput (idg, NINT(MQD.dbuf[3]) );
    if (multi>1 || MQH.info==1) rate = -1;
    if (multi>1 || MQH.info==1) tfreq = -1;
    if (multi>1 || MQH.info==1) ired = -1;
    if (multi>1 || MQH.info==1) tgain = -1;
  }
  else if (MQH.name == "RATE") {
    m_dwput (idr, MQD.dbuf[0] );
    if (multi>1 || MQH.info==1) rate = -1;
  }
  else if (MQH.name == "FREQ") {
    m_dwput (idf, MQD.dbuf[0] );
    if (multi>1 || MQH.info==1) tfreq = -1;
    if (type==IOPT_TBANK) pic_setkeyl(p, dmac, KEY_CHAN, MQH.info);
  }
  else if (MQH.name == "DEC") {
    m_lwput (idd, NINT(MQD.dbuf[0]) );
    if (multi>1 || MQH.info==1) ired = -1;
  }
  else if (MQH.name == "GAIN") {
    m_lwput (idg, NINT(MQD.dbuf[0]) );
    if (multi>1 || MQH.info==1) tgain = -1;
  }
  else if (MQH.name == "RATIO") {
    m_dwput (idrat, MQD.dbuf[0] );
  }
  else if (MQH.name == "NFREQ") {
    ttfreq = tfreq*2/rate;
    i = pic_setkey (p, dmac, KEY_CHAN, &MQH.info, 4);
    i = pic_getkey (p, dmac, KEY_NFREQ, &ttfreq, 8);
    MQD.dbuf[0] = ttfreq * (rate/2);
    m_send_msgl ("=NFREQ",0,MQH.sender,MQH.info, 0,1,"D",&MQD.dbuf[0],0);
  }
  else if (MQH.name == "STATUS") {
    MQD.dbuf[0] = mgo;
    MQD.dbuf[1] = msync;
    MQD.dbuf[2] = rate;
    MQD.dbuf[3] = ired;
    MQD.dbuf[4] = tfreq;
    MQD.dbuf[5] = tgain;
    MQD.dbuf[6] = ncycle;
    if (MQH.info>0 && MQH.info<=7) {
      m_send_msgl ("=STATUS",0,MQH.sender,MQH.info, 0,1,"D",&MQD.dbuf[MQH.info-1],0);
    } else {
      m_send_msgl ("=STATUS",0,MQH.sender,MQH.info, 0,7,"D",&MQD,0);
    }
  }
  else if (MQH.name == "TC") {
    trigger = MQD.dbuf[0];
    if (trigger == -3) {
      status = pic_tc (p, dmac, &sample, &delta, &soy, &fsoy, 0);
      soy = soy + tcoff;
    } else if (trigger == -2) {
      trigger = outoff;
      fsoy = pkt.tcfsec + pkt.tcoff*delta;
      soy  = pkt.tcwsec + DINT(fsoy);
      fsoy = fsoy - DINT(fsoy);
      status = pkt.tcstatus;
    } else {
      if (trigger==-1) trigger = lastout;
      lostoff = nlost * epb;
      sample = (trigger-outoff+lostoff) * ape;
      if (ired>1 && type==IOPT_MODULE) sample=sample*ired;
      status = pic_tc (p,dmac, &sample, &delta,  &soy, &fsoy, FLG_TCINTERP);
      soy = soy + tcoff;
    }
    MQD.dbuf[0] = trigger;
    MQD.dbuf[1] = soy;
    MQD.dbuf[2] = fsoy;
    m_send_msgl ("=TC",0,MQH.sender,status,0,3,"D",&MQD,0);
  }
  else if (MQH.name == "JOIN") {
    if (MQH.info>0) pic_setkeyl (p,dmac,KEY_IPVLAN,MQH.info);
    i = pic_str2ip (p,(char*)string(MQD.sbuf,16).c_str());
    pic_setkeyl (p,dmac,KEY_IPCONN,i);
  }
  else if (MQH.name == "LEAVE") {
    if (MQH.info>0) pic_setkeyl (p,dmac,KEY_IPVLAN,MQH.info);
    i = pic_str2ip (p,(char*)string(MQD.sbuf,16).c_str());
    pic_setkeyl (p,dmac,KEY_IPDISC,i);
  }
  else if (MQH.name == "ARCHIVE") {
    if (archmode != ARCH_OFF) {
      m_warning ("Previous ARCHIVE not completed yet");
      m_send_msgl ("=ARCHIVE",0,archid,-1,0,0,"",&MQD,0);
    } else {
      archname = string(MQD.sbuf,80);
      archtime = MQD.dbuf[10];
      archdur  = MQD.dbuf[11];
      archid   = MQH.sender;
      archmode = ARCH_WAIT;
      if (archsf>1) archsf = 1;
    }
  }
  else if (MQH.name == "ARCHDONE" || MQH.name == "ARCHSTOP") {
    if (archmode==ARCH_RT) {
      m_close(HARCH);
      if (apacket>0) m_close(HAPACK);
      archmode = ARCH_OFF;
      m_send_msgl ("ARCHDONE",0,archid,0,0,0,"",&MQD,0);
    } else {
      m_warning ("No RT ARCHIVE active for ARCHDONE");
    }
  }
  else if (MQH.name == "RFFREQ") {
    pic_setkeyd (p, dmac, KEY_RFFREQ, MQD.dbuf[0]);
  }
  else if (MQH.name == "RFGAIN") {
    pic_setkeyl (p, dmac, KEY_RFGAIN, MQD.ibuf[0]);
  }
  else if (MQH.name == "REPLAY") {
    m_mwput (idm,MQH.info);
  }
  else if (MQH.name == "EXIT") {
    m_mwput (idm,8);
  }
  else {
    m_warning ("Unexpected message: "+MQH.name);
  }

  LABEL_100:
  // look for widgets
  ratex = m_dwget(idr,rate);
  iredx = m_lwget(idd,ired);
  freqx = m_dwget(idf,tfreq);
  gainx = m_lwget(idg,tgain);
  algx  = 0;
  argx  = 0;
  ratx  = 0;
  if (nargs>=0) {
    algx = m_mwget(ida,alg);
    for (j=1; j<=nargs; j++) {
      if (m_lwget(ida+j,args[j])>0) argx |= (0x1<<j);
    }
  }
  if (ratio > 0.0) ratx  = m_dwget(idrat,ratio);
  nm = MQH.info;
  if (nm<1 || nm>multi) nm = 0;
  if (freqx>0) {
    tfreq = (rate/2) * pic_tuner_freq (p,2*tfreq/rate,0);
    m_dwput(idf,tfreq);
  }
  if (ratex>0 || iredx>0) {
    delta = (1/rate);
    if (type!=IOPT_MODULE) delta = ired*delta*spa;
    tcdelta = -1;
    if (tcmode!="OFF" && mgo!=0) {
      if (autors>0) m_lwput (idm,7);
      if (autors==0 && ratex>0) m_warning ("Sample rate change while acquiring may invalidate timecode");
    }
  }
  if  (iredx>0) {
    i = ired;
    if (type==IOPT_TUNER || type==IOPT_TBANK) {
      ired = pic_tuner_dec (p,ired,-1,0);
    }
    if (ired!=i) {
      m_lwput (idd, ired);
      if (info) printf("Tuner decimation limited to %d\n",ired);
    }
    if (tcmode!="OFF" && mgo!=0) {
      if (autors>0) m_lwput (idm,7);
      if (autors==0) m_warning ("Tuner decimation change while acquiring invalidates timecode");
    }
  }
  for (n=1; n<=multi; n++) {	// loop through multi-channel structure
   if (nm==0 || n==nm) {
    if (ratex>0) {
      pic_setkeyl (p, mdmac[n], KEY_RATE, NINT(rate));
    }
    if  (iredx>0) {
      pic_setkeyl (p, mdmac[n], KEY_DEC, ired);
    }
    if (ratex>0 || iredx>0 || freqx>0) {
      ttfreq = tfreq;
      if (nm==0) ttfreq = tfreq + (n-1)*dtfreq;
      pic_setkeyd (p, mdmac[n], KEY_FREQ, 2*ttfreq/rate);
    }
    if  (gainx>0) {
      pic_setkeyl (p, mdmac[n], KEY_GAIN, tgain);
    }
    if  (algx>0) {
      pic_setkeyl (p, mdmac[n], KEY_ALG, alg);
    }
    if  (argx!=0) {
     for (j=1; j<=nargs; j++) {
       if (argx&(0x1<<j)) pic_setkeyl (p, mdmac[n], KEY_ARGS+j-1, args[j]);
     }
    }
    if  (ratx>0) {
      pic_setkey(p, mdmac[n], KEY_RATIO, &ratio, 8);
      pic_getkey(p, mdmac[n], KEY_RATIO, &ratio, 8);
      m_dwput(idrat,ratio);
    }
   }
  }

  if (archmode == ARCH_WAIT) {
    stat = pic_dmastat (p,mdmac[1],&index,&cycle);
    m_now (time1,time2);
    time = time1 + time2;  // current time
    timet = HIN.size * ape * delta;
    time1 = archtime;
    time2 = archdur;
    status = 1;
    if (archdur<=0.0) time2 = timet;
    if (time1<=0.0) time1 = time-time1-.001;
    if (archtime == 0 || archdur < 0) {
      sample = ((npass-1)*HIN.size+dnext)*HIN.ape;
      if (mpp==1) archmode = ARCH_RT;
    } else if (timet < time2) {
      m_warning ("ARCHIVE length longer than buffer");
      archmode = ARCH_OFF;
      status = -2;
    } else if (time1 < time-timet) {
      m_warning ("ARCHIVE length not in current buffer");
      archmode = ARCH_OFF;
      status = -3;
    } else if (time1 > time+timet) {
      m_warning ("ARCHIVE too far in future");
      archmode = ARCH_OFF;
      status = -4;
    } else if (time1+time2>time) {
      // wait for buffer to be available
    } else { 
      archbuf = 0;
      larch = (time2/delta) * HIN.bpa;
      archbuf = (int_1*)malloc(larch);
      index = index/HIN.bpa - NINT((time-time1)*rate);
      sample = cycle*HIN.size*HIN.ape + index;
      archoff = index*HIN.bpa;
      // capture buffer into temp memory
      parch = (int_1*)map.vaddr;
      if (archoff < 0) {	// from tail of last pass + some
        nelem = min(-archoff,larch);
        m_move (parch+map.bytes+archoff,archbuf,nelem);
        m_move (parch,archbuf+nelem,larch-nelem);
      } else {
        m_move (parch+archoff,archbuf,larch);
      }
      archmode = ARCH_SNAP;
    }
    if (archmode==ARCH_RT || archmode==ARCH_SNAP) {
      m_propagate (HIN,HARCH);
      if (archsf>0) {
        i = m_search(archname,"(");
        j = m_length(archname);
	tmpstr = "_"+archsf;
        if (i<=0) HARCH.file_name = archname+tmpstr;
        if (i>0)  HARCH.file_name = archname.substr(0,i-1)+tmpstr+archname.substr(i);
        archsf = archsf+1;
      } else {
        HARCH.file_name = archname;
      }
      if (HIN.class_==2) HARCH.type = 1000;
      HARCH.xdelta = delta;
      HARCH.size = NINT(time2/delta);
      if (tcmode!="OFF") {
        stat = pic_tc (p, dmac, &trigger, &delta, &time1, &time2, 0);
        MQD.dbuf[0] = trigger-sample;
        MQD.dbuf[1] = tcoff+time1;
        MQD.dbuf[2] = time2;
        m_put_epoch (HARCH, tcoff+time1, time2+(sample-trigger)*delta, false);
      }
      m_open(HARCH,HCBF_OUTPUT+HCBF_NOABORT);
      if (! HARCH.open) {
        archmode = ARCH_OFF;
        status = -4;
      }
      if (apacket>0) {
        i = m_searchb(HARCH.file_name,".")-1;
        if (i<=0) i = m_length(HARCH.file_name);
        HAPACK.file_name = HARCH.file_name.substr(0,i)+afpext;
        m_open(HAPACK,HCBF_OUTPUT);
        apkt.count = 0;
        apkt_count = 0;
      }
    }
    if (archmode != ARCH_WAIT && archid!=0) m_send_msgl ("=ARCHIVE",0,archid,status,0,3,"D",&MQD,0);
  }

  // write next archive buffer
  if (archmode == ARCH_SNAP) {
    larch = HARCH.offset * HARCH.dbpe;
    nelem = min( (NINT(HARCH.size-HARCH.offset)), archtl);
    m_filad (HARCH,archbuf+larch,nelem);
    if (HARCH.offset >= HARCH.size) {
      m_close(HARCH);
      free(archbuf);
      archmode = ARCH_OFF;
      m_send_msgl ("ARCHDONE",0,archid,0,0,0,"",&MQD,0);
    }
  }

  if (idv>0 && m_lwget(idv,verbose)>0) {
    pic_setkeyl (p, 0, KEY_VERBOSE, verbose);
  }

}
