/****************************************************************/
/*  ICE-PIC Device Driver for DEC-OSF                           */
/*                                                              */
/*  Author:  Jeff Schoen - Innovative Computer Engineering      */
/*                                                              */
/****************************************************************/

#define ICE_DRIVER_VERSION 338

#define _OSF51B_PLUS 0

/*****************/
/* Include Files */
/*****************/
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/ioctl.h>
#include <sys/tty.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/map.h>
#include <sys/buf.h>
#include <sys/vm.h>
#include <sys/file.h>
#include <sys/uio.h>
#include <sys/types.h>
#include <sys/errno.h>
#include <sys/conf.h>
#include <sys/kernel.h>
#include <sys/devio.h>
#include <hal/cpuconf.h>
#include <sys/exec.h>
#include <sys/sysconfig.h>
#include <machine/cpu.h>
#include <sys/malloc.h>
#include <io/common/devdriver.h>
#include <io/dec/pci/pci.h>

/***********************************/
/* Device Driver Macro Definitions */
/***********************************/
#define RAISE_IPL(a) ( a=splsched() )
#define LOWER_IPL(a) ( splx(a) )

/*************************************/
/* Device Driver Bus Access Routines */
/*************************************/
void READ_IO(int a,caddr_t b,int c,int d,void *e) {
  long f; int *pie = (int *)e;
  f = read_io_port( (io_handle_t)((char*)b+c), d, 0);
  mb();	/* memory barrier to force IO */
  *pie = (int)(f & 0xFFFFFFFF);
}
void WRITE_IO(int a,caddr_t b,int c,int d,void *e) {
  long f; int *pie = (int *)e;
  f = (long)(*pie & 0xFFFFFFFF);
  write_io_port( (io_handle_t)((char*)b+c), d, 0, f);
  mb();	/* memory barrier to force IO */
}

/***********************/
/* General Definitions */
/***********************/
#define vprintf if (icepic_debug>=1) printf
#define wprintf if (icepic_debug>=2) printf

/*******************************/
/* Driver Specific Definitions */
/*******************************/

#define _UNIX 1
#define _OSF  1
#define _EEEI 1

#include "../../inc/iceioctl.h"
#include "../../inc/iceflex.h"

/* 
MAXPIC is then number of cards in the system
MAXMINOR is the number of minor devices per card 
MAXIOBUF is the number of DMA buffers per minor device
*/

#define MAXPIC 12
#define MAXMINOR 40
#define MAXIOBUF 8
#define MAXBCNT (1024*1024*1024)

typedef struct {
  dma_handle_t	handle;
  pid_t 	pid;
  vm_offset_t	usradr;
  int		bytes;
  int		pciadr;
} DMACHANNEL;

typedef struct {
  int		attached;
  int		opened;
  caddr_t	membase;
  DMACHANNEL 	dma[MAXIOBUF];
  struct controller *ctlr;
} UCB;

UCB ucbs[MAXPIC * MAXMINOR];

/************************************/
/* Prototypes for External Routines */
/************************************/

int icepic_configure(), icepic_probe(), icepic_attach(), icepic_unattach(); 
int icepic_open(), icepic_close(), icepic_read(), icepic_write(), icepic_ioctl();

/***********************************************/
/* Loadable Driver Configuration Definitions   */
/***********************************************/

int icepic_preconfig();
int icepic_postconfig();
void icepic_preconfig_callback(int point, int order,
                           ulong argument, ulong event_arg);
void icepic_postconfig_callback(int point, int order,
                            ulong argument, ulong event_arg);
int callback_return_status = ESUCCESS;

extern int nodev(), nulldev();

#define DRIVER_NUM DRIVER_WILDNUM
#define DRIVER_NAME "pci"

#define ICEPIC_NAME "icepic"
#define MAJOR_INSTANCE 1
#define NO_DEV -1

int icepic_config = FALSE;
int icepic_devno = NO_DEV;
int icepic_num_installed=1;
int icepic_num_probed=0;
int icepic_dev_number = NO_DEV;
int icepic_is_dynamic;

static int majnum = NO_DEV;
static int icepic_debug = 0;
static char icepic_version[] = "1.00";
static char mcfgname[] = ICEPIC_NAME;
static unsigned char unused[500] = "";
static unsigned char icepic_desc[100] = "";
static unsigned char pci_optiondata[300] = "";

struct controller *icepic_info[MAXPIC];

struct driver icepic_driver = {
	icepic_probe,
	0,
	icepic_attach, 
	0,
	0,
	0,
	0,
	0,
	ICEPIC_NAME,
	icepic_info,
	0,
	0,
	0,
	0,
	0,
	icepic_unattach,
	0
};

struct dsent icepic_devsw_entry = {
	icepic_open,
	icepic_close,
#if _OSF51B_PLUS
	nodev_strategy,
#else
	nodev,
#endif
	icepic_read,
	icepic_write,
	icepic_ioctl,
	nodev,
	nodev,
	nulldev,
	nulldev,
	nulldev,
	0,
	0,
	NULL,
	DEV_FUNNEL,
	0,
	0
};

cfg_subsys_attr_t icepic_attributes[] = {

{"Subsystem_Description",CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE | CFG_OP_QUERY,
                        (caddr_t)icepic_desc,2,300,0},
{"Module_Config_Name",  CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE | CFG_OP_QUERY,
                        (caddr_t)mcfgname,2,CFG_ATTR_NAME_SZ,0},
{"majnum",              CFG_ATTR_INTTYPE, CFG_OP_QUERY,
                        (caddr_t)&majnum,0,512,0},
{"ICEPIC_Debug",        CFG_ATTR_INTTYPE, CFG_OP_QUERY | CFG_OP_CONFIGURE | CFG_OP_RECONFIGURE,
                        (caddr_t)&icepic_debug,0,9,0},
{"Num_Installed",       CFG_ATTR_INTTYPE, CFG_OP_QUERY | CFG_OP_CONFIGURE,
                        (caddr_t)&icepic_num_installed,1,MAXPIC,0},
{"Num_Probed",          CFG_ATTR_INTTYPE, CFG_OP_QUERY,
                        (caddr_t)&icepic_num_probed,0,MAXPIC,0},
{"Module_Type",         CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
                        (caddr_t)unused,2,CFG_ATTR_NAME_SZ,0},
{"Device_Major_Req",    CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
                        (caddr_t)unused,0,300,0},
{"Device_Char_Major",   CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
                        (caddr_t)unused,0,300,0},
{"Device_Char_Minor",   CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
                        (caddr_t)unused,0,300,0},
{"Device_Char_Files",   CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
                        (caddr_t)unused,0,500,0},
{"Device_User",         CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
                        (caddr_t)unused,0,300,0},
{"Device_Group",        CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
                        (caddr_t)unused,0,300,0},
{"Device_Mode",         CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
                        (caddr_t)unused,0,300,0},
{"Device_Subdir",       CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE,
                        (caddr_t)unused,0,300,0},
{"PCI_Option",          CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE | CFG_OP_QUERY,
                        (caddr_t)pci_optiondata,0,300,0},
{"version",             CFG_ATTR_STRTYPE, CFG_OP_QUERY,
                        (caddr_t)icepic_version,0,5,0},
{0,0,0,0,0,0,0}
};



/************************************/
/* pic configure routine            */
/************************************/
icepic_configure ( cfg_op_t op, cfg_attr_t *indata, size_t indatalen, 
				cfg_attr_t *outdata, size_t outdatalen)
{
  int i, cfg_state, retval;

  vprintf ("pic_configure - op %d name [%s]\n", op, mcfgname);

  switch (op) {

  case CFG_OP_CONFIGURE:

    /* already configured ? */
    if (icepic_config == TRUE) {
      vprintf ("pic_configure - already configured\n");
      return (EINVAL);
    }

    /* get the current config state */
    if (cfgmgr_get_state(mcfgname, &cfg_state) != ESUCCESS) {
      vprintf ("pic_configure - cfgmgr_get_state failed\n");
      return (EINVAL);
    }

    /* handle the static config state registrations */
    if (cfg_state == SUBSYSTEM_STATICALLY_CONFIGURED) {
      icepic_is_dynamic = 0;
      callback_return_status = ESUCCESS;
      register_callback (icepic_preconfig_callback,
		CFG_PT_PRECONFIG, CFG_ORD_NOMINAL, (long) 0L );
      register_callback (icepic_postconfig_callback,
		CFG_PT_POSTCONFIG, CFG_ORD_NOMINAL, (long) 0L );
    }
    /* handle the dynamic config state registrations */
    else {
      icepic_is_dynamic = 1;
      if ( (retval=icepic_preconfig()) != ESUCCESS) return (retval);
      if ( (retval=configure_driver(mcfgname, DRIVER_NUM, DRIVER_NAME,
	&icepic_driver)) != ESUCCESS) return (retval);
      if ( (retval=icepic_postconfig()) != ESUCCESS) return (retval);
    }
    icepic_config = TRUE;
    break;

  case CFG_OP_UNCONFIGURE:

    if (icepic_is_dynamic == 0) return (ESUCCESS);
    for (i=0; i<icepic_num_probed * MAXMINOR; i++) if (ucbs[i].opened != 0) return (EBUSY);
    if (devsw_del(mcfgname, MAJOR_INSTANCE) == NODEV) {
      vprintf ("devsw_del failed\n");
      return (ESRCH);
    }
    if (unconfigure_driver(DRIVER_NAME, DRIVER_NUM, &icepic_driver, 
		ICEPIC_NAME, DRIVER_NUM) != 0) {
      vprintf ("unconfigure_driver failed\n");
      return (ESRCH);
    }
    majnum = NO_DEV;
    icepic_devno = majnum;
    icepic_is_dynamic = 0;
    icepic_num_probed = 0;
    icepic_config = FALSE;
    break;

  case CFG_OP_RECONFIGURE:
    break;

  case CFG_OP_QUERY:
    break;

  default:
    return (ENOTSUP);
    break;
  }
  return (ESUCCESS);
}

int icepic_preconfig()
{
  int i,status;
  struct controller_config ctlr_register;

  ctlr_register.revision = CTLR_CONFIG_REVISION;
  strcpy(ctlr_register.subsystem_name,mcfgname);
  strcpy(ctlr_register.bus_name,DRIVER_NAME);
  ctlr_register.devdriver = &icepic_driver;

  for (i=0; i<icepic_num_installed; i++) {
    if ((status=create_controller_struct(&ctlr_register)) != ESUCCESS) {
        vprintf ("create_controller_struct failed\n");
	return (status);
    }
  }

  return (ESUCCESS);
}

int icepic_postconfig()
{

  majnum = devsw_add (mcfgname, MAJOR_INSTANCE, majnum, &icepic_devsw_entry);
  if (majnum == ENODEV) return (ENODEV);

  icepic_devno = majnum;

  return (ESUCCESS);
}

void icepic_preconfig_callback (int point, int order, 
				ulong argument, ulong event_arg)
{
  int status;
  if (callback_return_status != ESUCCESS) return;
  if ( (status=icepic_preconfig()) != ESUCCESS) {
    cfgmgr_set_status(mcfgname);
    callback_return_status = status;
    return;
  }
}

void icepic_postconfig_callback (int point, int order, 
				ulong argument, ulong event_arg)
{
  int status;
  if (callback_return_status != ESUCCESS) return;
  if ( (status=icepic_postconfig()) != ESUCCESS) {
    cfgmgr_set_status(mcfgname);
    callback_return_status = status;
    return;
  }
}



/************************************/
/* pic probe routine                */
/************************************/
icepic_probe (io_handle_t bus_io_handle, struct controller *ctlr)
{
  struct pci_config_hdr *pci_cfg_hdr = 
	(struct pci_config_hdr *)bus_io_handle;
  UCB *ucb;	/* unit control block */
  int i;

  vprintf ("pic_probe - unit %d\n",ctlr->ctlr_num);

  if (ctlr->bus_hd->bus_type != BUS_PCI) {
    vprintf("pic_probe - unsupported bus type\n");
    return(0);  /* bad probe */
  }
  if (ctlr->ctlr_num >= MAXPIC) {
    vprintf("pic_probe - device unit too large\n");
    return(0);  /* bad probe */
  }

  ctlr->private[0] = (void *)pci_cfg_hdr;  /* save PCI config header handle */

  /* Each minor device associated with this controller needs to point back 
     to the correct controller & bus address */
  for (i = 0; i < MAXMINOR; i++) {
    ucb = &ucbs[(ctlr->ctlr_num * MAXMINOR) + i];
    ucb->ctlr = ctlr;
    ucb->attached = 0;
    ucb->opened   = 0;
    ucb->membase  = (caddr_t)pci_cfg_hdr->bar0;
  }
  icepic_num_probed++;
  return (1);   /* good probe */
}

/************************************/
/* pic attach routine              */
/************************************/
icepic_attach (struct controller *ctlr)
{
  int i, j;
  UCB *ucb;

  vprintf ("pic_attach - unit %d\n",ctlr->ctlr_num);

  /* Each minor device associated with this controller needs to have its
     dma resources initialized */

  for (i = 0; i < MAXMINOR; i++) {
    ucb = &ucbs[(ctlr->ctlr_num * MAXMINOR) + i];
    ucb->attached = 1;
    for (j=0; j<MAXIOBUF; j++) {
      ucb->dma[j].handle = NULL;
      ucb->dma[j].pciadr = 0;
      ucb->dma[j].bytes = 0;
      ucb->dma[j].pid = 0;
    }
  }
  return (1);   /* good probe */
}



/************************************/
/* pic unattach routine            */
/************************************/
icepic_unattach (struct bus *bus, struct controller *ctlr)
{
  return (ESUCCESS);   /* good probe */
}



/************************************/
/* pic open routine                 */
/************************************/
icepic_open (dev_t dev, int flag, int format)
{
  UCB *ucb;	/* unit control block */

  vprintf ("pic_open - dev %#x, flag %#x, format %#x\n",dev,flag,format);

  ucb = &ucbs[minor(dev)];
  if (ucb->attached == 0) {
    vprintf("pic_open - device unit not attached\n");
    return(ENXIO);
  }
  /* Fail is this minor number is already opened - icelib will select
     the next minor number for them to try again */
  if (ucb->opened) {
    vprintf("pic_open - device busy\n");
    return(EWOULDBLOCK);
  }
  ucb->opened++;
  return(ESUCCESS); 
}
    


/************************************/
/* pic close routine                */
/************************************/
icepic_close (dev_t dev, int flag, int format)
{
  UCB *ucb;	/* unit control block */

  vprintf ("pic_close - dev %#x, flag %#x, format %#x\n",dev,flag,format);

  ucb = &ucbs[minor(dev)];
  if (ucb->attached == 0) {
    vprintf("pic_close - device unit not attached\n");
    return(EINVAL);
  }
  /* make sure all DMA's have been terminated */
  icepic_cancel (dev, flag, format, 1);
  ucb->opened--;
  return(ESUCCESS); 
}
    


/************************************/
/* pic read stub                    */
/************************************/
icepic_read (dev_t dev, struct uio *uio, int flag)
{
  vprintf ("pic_read - dev %#x, flag %#x\n",dev,flag);
}



/************************************/
/* pic write stub                   */
/************************************/
icepic_write (dev_t dev, struct uio *uio, int flag)
{
  vprintf ("pic_write - dev %#x, flag %#x\n",dev,flag);
}



/************************************/
/* pic ioctl routine                */
/************************************/
#define PIC_ADPADDR	0		/* user buffer addr */
#define PIC_IOHANDLE	ucb->membase	/* PCI controller base address */
#define PIC_BUFADDR	p->bufaddr	/* user buffer addr */
#define PIC_BYTES	p->bytes	/* bytes to transfer */
#define PIC_FUNCTION	p->function	/* user function code */
#define PIC_OFFSET	p->offset	/* register offset */
#define PIC_STATUS	p->status	/* return status */

icepic_ioctl (dev_t dev, u_int cmd, caddr_t data, int flag)
{
  UCB 		*ucb;		/* unit control block */
  int		i,j,k;		/* Local counters */
  int		status,stat;	/* Local function return status */
  int		cur_pri;	/* Holds the current process IPL */
  int		chan;		/* Current dma channel */
  PICIOCTL	*p;		/* pic ioctl handle structure */
  struct task 	*tsk;		/* current task pointer */
  int		usermap;	/* whether to map user space */
  int		ipl;		/* ipl level */

  unsigned char *bdata;		/* byte data reference */
  int		worda,wordb;	/* load IOC temps */

  wprintf("pic_ioctl - dev %#x, cmd %#x, data %#lx, flag %#x\n",
              				  dev,cmd,data,flag);

  ucb = &ucbs[minor(dev)];		/* ucb for this controller */

  p = (PICIOCTL *)data;			/* ioctl data = the PICIOCTL */

  i = PIC_BYTES;			/* normal|default completion status */
  status = ESUCCESS; 			/* assume success */

  if (PIC_FUNCTION==IOCTL_MAP || PIC_FUNCTION==IOCTL_UNMAP 
	|| PIC_FUNCTION==IOCTL_QMAP || PIC_BYTES==0) usermap = 0; else usermap = 1;

  if (PIC_FUNCTION<0 || PIC_FUNCTION>IOCTL_MAXVALID) {
    vprintf ("pic_ioctl - invalid function %d\n", PIC_FUNCTION);
    return (EINVAL);
  }

  if (usermap) {			/* Map user buffer */
    /* If the buffer address is NULL don't panic the system by
       calling vm_map_pageable().  It'd be nice to do a check of the
       address to make sure it's valid before trying to map it. */
    if (!PIC_BUFADDR) {
      vprintf ("pic_ioctl - trying to map bad buffer 0x%lx\n", PIC_BUFADDR);
      return (EINVAL);
    }
    tsk = current_task();	
    if (vm_map_pageable (tsk->map, (vm_offset_t)PIC_BUFADDR, 
	(vm_offset_t)PIC_BUFADDR+PIC_BYTES, VM_PROT_READ|VM_PROT_WRITE)) {
      vprintf("Unable to lock user buffer in memory\n");
      status = EINVAL; p->status = i; return (status);
    }
    RAISE_IPL(ipl); 
  }


  switch (PIC_FUNCTION) {

  case IOCTL_READ:			/* read a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=0; i<PIC_BYTES; ) {
        READ_IO (PIC_ADPADDR, PIC_IOHANDLE, REG_MCSR, 4, (&stat));
             if ((stat&MCSR_IFIFO_FULL) && (PIC_BYTES-i>=24)) j=6;
        else if ((stat&MCSR_IFIFO_AFULL) && (PIC_BYTES-i>=16)) j=4;
        else if ((stat&MCSR_IFIFO_HALF) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_IFIFO_EMPTY)) j=1;
        else break;
        while (j>0) { 
	  READ_IO (PIC_ADPADDR, PIC_IOHANDLE, REG_FIFO, 4, (char *)PIC_BUFADDR+i );  
    	  i+=4; j--;
        }
      }
    }
    else if ( (PIC_OFFSET&0XFFFFF000) != 0) 	/* invalid register ? */
      status = EINVAL;
    else 
      READ_IO (PIC_ADPADDR, PIC_IOHANDLE, PIC_OFFSET, PIC_BYTES, PIC_BUFADDR );		
    break;

  case IOCTL_WRITE:			/* write a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=0; i<PIC_BYTES; ) {
        READ_IO (PIC_ADPADDR, PIC_IOHANDLE, REG_MCSR, 4, (&stat));
             if ((stat&MCSR_OFIFO_EMPTY) && (PIC_BYTES-i>=24)) j=6;
        else if (!(stat&MCSR_OFIFO_HALF) && (PIC_BYTES-i>=16)) j=4;
        else if (!(stat&MCSR_OFIFO_AFULL) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_OFIFO_FULL)) j=1;
        else break;
        while (j>0) {
	  WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, REG_FIFO, 4, (char *)PIC_BUFADDR+i );
	  i+=4; j--;
        }
      }
    }
    else if ( (PIC_OFFSET&0XFFFFF000) != 0) 	/* invalid register ? */
      status = EINVAL;
    else 
      WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, PIC_OFFSET, PIC_BYTES, PIC_BUFADDR );		
    break;

  case IOCTL_ACMD: 			/* command to SHARC processor */

    for (j=16; j>=0; j-=4) 
      WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, REG_MBX+j, 4, (char *)PIC_BUFADDR+j );
    for (i=0,j=0; j<32; j++) {    	/* cant wait forever in kernel mode */
      READ_IO (PIC_ADPADDR, PIC_IOHANDLE, REG_MCSR, 4, (&stat) );
      if ( (stat&MCSR_IMB1F) != 0x0 ) {
        READ_IO (PIC_ADPADDR, PIC_IOHANDLE, REG_IMB1, 4, PIC_BUFADDR );
	i=4; break;
      }
    }
    break;

  case IOCTL_MAP:			/* map user buffer to PCI */

    /* find an open dma handler slot */
    for (chan=0; chan<MAXIOBUF && ucb->dma[chan].pciadr!=0; chan++);
    if (chan>=MAXIOBUF) { status = EINVAL; break; }

    /* Validate user buffer */
    if ( ((long)PIC_BUFADDR % 4) != 0 ) status = EINVAL;
    if ( (PIC_BYTES <= 0)  || (PIC_BYTES > MAXBCNT) ||
        ((PIC_BYTES % 4) != 0) ) status = EINVAL;
    if (status!=ESUCCESS) break;

    /* map user buffer into PCI bus space */
    ucb->dma[chan].usradr = (vm_offset_t)PIC_BUFADDR;
    ucb->dma[chan].bytes  = PIC_BYTES;
    status = icepic_mapio (ucb, IOCTL_MAP, chan);
    i = ucb->dma[chan].pciadr;
    break;
    
  case IOCTL_UNMAP: 			/* unmap user buffer from PCI */

    /* find the matching dma handler slot */
    for (chan=0; chan<MAXIOBUF && ucb->dma[chan].pciadr!=PIC_OFFSET; chan++);
    if (chan>=MAXIOBUF) { status = EINVAL; break; }

    /* unmap user buffer from PCI bus space */
    status = icepic_mapio (ucb, IOCTL_UNMAP, chan);
    break;

  case IOCTL_QMAP: 			/* query user buffer map */

    for (i=0,chan=0; chan<MAXIOBUF && i==0; chan++) {
      if (PIC_OFFSET >= (ucb->dma[chan].pciadr) &&
          PIC_OFFSET+PIC_BYTES <= (ucb->dma[chan].pciadr+ucb->dma[chan].bytes))
                i = ucb->dma[chan].bytes;
    }
    break;
 
  case IOCTL_LIOC:                      /* load IOC words */

    bdata = (unsigned char *)PIC_BUFADDR;
    for (k=0; k<PIC_BYTES; k++) {
      worda = (int)bdata[k];
      for (j=0; j<8; j++) {
        wordb = worda>>j;
        WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, REG_IOC, 4, (&wordb) );
      }
    }
    break;

  case IOCTL_VERSION:			/* read the driver version */
    *(int *)PIC_BUFADDR = ICE_DRIVER_VERSION;
    break;

  default: 				/* handle unknown function codes */
    status = EINVAL;
    break;
  }

  if (usermap) {			/* Unmap user buffer */
    LOWER_IPL(ipl);
    if (vm_map_pageable (tsk->map, (vm_offset_t)PIC_BUFADDR, 
	(vm_offset_t)PIC_BUFADDR+PIC_BYTES, VM_PROT_NONE)) {
      vprintf("Unable to unlock user buffer in memory\n");
    }
  }

  /* fill in IO status block and signal completion */
  p->status = i;
  return(status);
}



/************************************/
/* pic cancel routine               */
/************************************/
icepic_cancel (dev_t dev, int flag, int format, int reason)
{
  int reset,status,chan;
  UCB *ucb;
  struct task *tsk;
  struct proc *prc;

  vprintf ("pic_cancel - dev %#x, flag %#x, format %#x\n",dev,flag,format);

  ucb = &ucbs[minor(dev)];
  tsk = current_task();
  prc = task_to_proc(tsk);
  reset = MCSR_RESET;

  if (reason == 1) {	/* close */

    for (chan=0; chan<MAXIOBUF; chan++) {

      if (ucb->dma[chan].pid == prc->p_pid && ucb->dma[chan].pciadr != 0) {

        vprintf ("pic_cancel - stray DMA - resetting card");

        /* Do we really want to reset the card?  There may be other 
           processes that have an open connection to the card */

        /* unexpected termination - reset card to avoid stray DMAs */
	WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, REG_MCSR, 4, (&reset) );

	/* deallocate/unmap stray buffer */
	status = icepic_mapio (ucb, IOCTL_UNMAP, chan);

      }
    }
  }
  return(ESUCCESS); 
}



/************************************/
/* pic mapio routine                */
/************************************/
icepic_mapio (UCB *ucb, int function, int chan)
{
  int status,bytes;
  DMACHANNEL *dma;
  struct task *tsk;
  struct proc *prc;
  sg_entry_t sg;

  tsk = current_task();
  prc = task_to_proc(tsk);
  dma = ucb->dma + chan;

  status = ESUCCESS;	/* assume success */

  switch (function) {

  case IOCTL_MAP:

    /* Lock user buffer in memory */
    if (vm_map_pageable (tsk->map, dma->usradr, dma->usradr+dma->bytes, 
					VM_PROT_READ|VM_PROT_WRITE)) {
      vprintf("Unable to lock user pages in memory\n");
      status = EINVAL; 
      goto BADUSERMAP;
    }
    /* Allocate and load PCI bus mapping registers */
    bytes = dma_map_load (dma->bytes, dma->usradr, 
	prc, ucb->ctlr, &(dma->handle), 0, (DMA_ALL|DMA_CONTIG) );
    if (bytes < dma->bytes) {
      vprintf("pic_mapio: dma_map_load failure %d of %d\n",bytes,dma->bytes);
      status = EINVAL; 
      goto BADMAPLOAD; 
    }
    /* Check to make sure allocation is contiguous */
    sg = dma_get_curr_sgentry(dma->handle);
    vprintf("DMA map bc=%d ba=%x bytes=%d\n",sg->bc,sg->ba,dma->bytes);
    if (sg->bc < dma->bytes) {
      vprintf("Non-contiguous DMA map load entry %d != %d\n",sg->bc,dma->bytes);
      status = EINVAL; 
      goto BADMAPLOAD; 
    }
    dma->pid = prc->p_pid;
    dma->pciadr = (int)((unsigned int)sg->ba & 0x0FFFFFFFFL);
    vprintf("PCI address = %x\n",dma->pciadr);
    break;

  case IOCTL_UNMAP:

    vprintf("DMA unmap bytes=%d\n",dma->bytes);
    BADMAPLOAD:
    if (dma->handle != NULL) dma_map_unload (DMA_DEALLOC, dma->handle);

    BADUSERMAP:
    if (dma->bytes > 0) vm_map_pageable (tsk->map, dma->usradr, 
			dma->usradr+dma->bytes, VM_PROT_NONE);

    dma->handle = NULL;
    dma->pciadr = 0;
    dma->bytes = 0;
    dma->pid = 0;
    break;

  default:
    status = EINVAL;
  }

  return (status);
}
