/****************************************************************/
/*  ICE-PIC Device Driver for LINUX 2.4+ kernels new PCI        */
/*  Author:  Jeff Schoen - Innovative Computer Engineering      */
/****************************************************************/
/*
  Build defines:

    ICECD - is the character device driver for the ICE RAM buffer
    ICEBD - is the block device driver for the ICE RAM buffer and FPGA attached drives
    ICEFS - is the ICE file system driver for direct DMA to block devices
    ICEKAM - is the Kernel Allocated Memory driver for the ICE RAM buffer
    ICEQOS - is the Quality Of Services interface for CPU latency tuning

   Bill Reece Mods:  
   membase must be unsigned so it's not sign extended when it's cast
    to an unsigned long for the call to ioremap().
   vfree to iounmap in detach
*/

#define ICE_DRIVER_VERSION 399

#pragma GCC diagnostic ignored "-Wmissing-prototypes"

#include "picdrv.h"

#define MAXPIC 12
#define MAXNODE 4
#define MAXNUMA 1
#define MAXIOBUF 512
#define MAXTEMP 70

#ifndef _ALLOBF
#define _ALLOBF 0
#endif

#ifdef _ALL
#define ICEFS  1
#define ICEBD  1
#define ICECD  1
#define ICEKAM 1

#elif _ALLOBF
#define ICEFS  2
#define ICEBD  2
#define ICECD  1
#define ICEKAM 2

#else
#define ICEFS  0
#define ICEBD  0
#define ICECD  1
#define ICEKAM _KAM
#endif

/*****************/
/* Include Files */
/*****************/
#include <linux/version.h> 
#include <linux/module.h> 
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/uaccess.h>
#include <linux/vmalloc.h>
#include <linux/delay.h>
#include <linux/blkdev.h>
#include <asm/io.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,0,0)
 #undef CONFIG_COMPAT
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,22)
 #include <linux/ioctl.h>
#else
 #ifdef CONFIG_COMPAT
  #include <linux/ioctl32.h>
 #endif
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,18,0)
 #define ICEQOS 3
 #include <linux/pm_qos.h>
 static struct pm_qos_request icepic_qos_handle;
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(3,2,0)
 #define ICEQOS 2
 #include <linux/pm_qos.h>
 static struct pm_qos_request icepic_qos_handle;
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,32)
 #define ICEQOS 1
 #include <linux/pm_qos_params.h>
#else
 #define ICEQOS 0
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(3,8,0)
 #define PDE_DATA(A) PDE(A)->data
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
 #define FMINOR(A) MINOR(A->f_path.dentry->d_inode->i_rdev)
#else
 #define FMINOR(A) MINOR(A->f_dentry->d_inode->i_rdev)
#endif
 #define IMINOR(A) MINOR(A->i_rdev)

#if ICECD
 static unsigned int iceram_major=0;
#endif

/**********************/
/* Module Definitions */
/**********************/
/* RedHat has internal inconsistencies for these 3 Macros comment out for now */
MODULE_AUTHOR("Jeff Schoen");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Driver for DSP Cards from Innovative Computer Engineering");

/*******************************/
/* Module Parameters           */
/*******************************/
static int ice_ra = -1;
module_param(ice_ra, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_ra, "DMA RAM allocation method (0=Manual,1=LowKernel,2=HighKernel)");

static int ice_rn = 1;
module_param(ice_rn, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_rn, "DMA RAM nodes to create for NUMA system memory subsystem (1=Default)");

static int ice_nn = 1;
module_param(ice_nn, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_nn, "Number of NUMA system memory subsystem nodes (1=Default)");

static int ice_rs = 0;
module_param(ice_rs, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_rs, "DMA start address in MB (1024*1024 B)");

static int ice_rd = 0;
module_param(ice_rd, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_rd, "DMA ramdisk size in MB (1024*1024 B)");

static int ice_rm = 0;
module_param(ice_rm, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_rm, "DMA mapable RAM size in MB (1024*1024 B)");

static int ice_rb = -1;
module_param(ice_rb, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_rb, "DMA block dev size in MB (1024*1024 B)");

static int ice_mc = MAXPIC;
module_param(ice_mc, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_mc, "Maximum devices to enumerate");

static int ice_lat = 10;
module_param(ice_lat, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_lat, "QoS Max tolerable CPU latency in uSec");

static int ice_nvm = 0;
module_param(ice_nvm, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_nvm, "driver FPGA attached NVME drives 0=none 1=PM1 2=PM2 3=Both");

static int ice_vb = 1;
module_param(ice_vb, int, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
MODULE_PARM_DESC(ice_vb, "driver verbosity 0=quiet 1=normal 2=debug");

/****************************/
/* Debug/Status Definitions */
/****************************/
#define ESUCCESS 0
#define	SUCCESS(a)	(((a) & 1) == 1)
#define	FAILURE(a)	(((a) & 1) == 0)

#define vprintf if (ice_vb>0) printk
#define wprintf if (ice_vb&2) printk
#define uprintf if (ice_vb&4) printk

/***************************************/
/* IceKAM Kernel Allocated Memory Code */
/***************************************/
char *iceram_v2p (char *uvaddr, size_t len);
char *iceram_v2p_temp (size_t len);
#if ICEBD > 0
 #include "icebd.h"
#endif
#if ICEFS > 0
 #undef __KERNEL__
 #include <linux/hdreg.h> 
 #define __KERNEL__
 #include "icefs.h"
#endif

/***********************************/
/* Device Driver Macro Definitions */
/***********************************/
/* reread if timeout occured and not the FIFO register */
#define _RDX(A,B) {B=readl(ucb->loc+A); if (B==-1 && A!=REG_FIFO) B=readl(ucb->loc+A);}
#define _RD(A,B)  {B=readl(ucb->loc+A);}
#define _WR(A,B)  {writel(B,(int *)(ucb->loc+A));}

/************************************/
/* Linux Special Access Enforcement */
/************************************/
#define is_rd() ((filp->f_flags&0x3)==0)
#define is_wr() ((filp->f_flags&0x3)==1)
#define is_su() ((filp->f_flags&0x3)==2)
#define vm_rd() ((vma->vm_flags&0xFF)==0xfa)
#define vm_wr() ((vma->vm_flags&0xFF)==0xd1)
#define vm_su() ((vma->vm_flags&0xFF)==0xfb)

/*********************************/
/* Generate info for /proc entry */
/********************************/
static int proc_icepic (char *buf, char **a, off_t b, int c);

/* 3.10 kernel limit */
#define MAX_INFO_BUF  2040

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,26)
#include <linux/seq_file.h>
static int icepic_info_seq_show (struct seq_file *seq, void *offset) {
  char p[MAX_INFO_BUF];
  proc_icepic(p,0,0,0);
  seq_printf(seq,"%s",p);
  return 0;
}
static int icepic_info_open_fs (struct inode *inode, struct file *file) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,14,0)
  return single_open (file,icepic_info_seq_show,inode);
#else
  return single_open (file,icepic_info_seq_show,PDE_DATA(inode));
#endif
}
#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,5,0)
static const struct proc_ops icepic_info_fops = {
  .proc_open = icepic_info_open_fs,
  .proc_read = seq_read,
  .proc_lseek = seq_lseek,
  .proc_release = single_release,
};
#else
static const struct file_operations icepic_info_fops = {
  .owner = THIS_MODULE,
  .open = icepic_info_open_fs,
  .read = seq_read,
  .llseek = seq_lseek,
  .release = single_release,
};
#endif
void create_proc_entry_withInfo (const char *name, mode_t mode, 
				 struct proc_dir_entry *base, void *get_info) {
  struct proc_dir_entry *res = proc_create_data (name,S_IRUGO,base,&icepic_info_fops,NULL);
  if (!res) vprintf("icepic: unable to register /proc entry\n");
}
#else
void create_proc_entry_withInfo (const char *name, mode_t mode, 
				 struct proc_dir_entry *base, get_info_t *get_info) {
  struct proc_dir_entry *res = create_proc_entry (name,mode,base);
  if (res) res->get_info = get_info;
}
#endif
/*
 Ioctl support for 32 bit apps in 64 bit Linux kernels is implemented either
 with an entry in the 'fops' table (if supported) or with registering a
 function directly with the kernel. This next macro, if defined, implies
 that support exists in the 'fops' table. It probably should be set as part
 of 'picdrv.h' definition. 
*/

#if defined(CONFIG_COMPAT) && !defined(HAVE_COMPAT_IOCTL)
#define ICEIOCCMD32   (_IOWR('f',0x1, PICIOCTL32))
#endif

/*******************************/
/* Driver Specific Definitions */
/*******************************/

typedef int64_t int_8;
typedef uint64_t int_u8;
typedef int32_t int_4;
typedef uint32_t int_u4;

#include "../../inc/iceioctl.h"
#include "../../inc/iceflex.h"
#include "../../inc/iceconv.h"

#define B1G   0x0040000000LL	// 1G
#define B4G   0x0100000000LL	// 4G
#define B16G  0x0400000000LL	// 16G
#define B64G  0x1000000000LL	// 64G
#define B256G 0x4000000000LL	// 256G

#define MAXWCNT B256G	

#define ADON_DMAX 0x1A		// card DMA for read and write routines
#define PMSK 0xFFFFFFC0		// 64 by packet boundary mask

#define VENDOR_ID 0x1172
#define DEVICE_ID 0x7777
#define PICDRV_NAME "icepic"
#define RAMDRV_NAME "iceram"

typedef struct {
  int_u8	paddr;
  int_u8	psize;
  int		device;
  int		node;
  pid_t 	pid,gid;
} DMACHANNEL;

typedef struct {
  int_u8 vstr,vend;
  int_u8 paddr;
} MAPCHANNEL;

typedef struct {
  int	attached;
  int	opened;
  unsigned int membase;
  int   lastoffset;
  void	*loc;
  int   csr,lat,rev,rid; 
  struct pci_dev *pdev;
  int time, temp, temps[5], pktime, pktemp, active, flags, pvid;
} UCB;

typedef struct {
  int_u8 offset_min, offset_max;
  int_4 ra, rs, rd, rm;
} RAMNODE;

static UCB ucbs[MAXPIC];
static RAMNODE rnd[MAXNODE];
static DMACHANNEL dma[MAXIOBUF];
static MAPCHANNEL map[MAXIOBUF];
static int_u8 offset_min, offset_max;
static int_u8 rstart, rsized, rsizem;
static unsigned int icepic_major=0;
static unsigned int icepic_minor=0;
static int ndma=0, nmap=0, nram=0, icepic_count=0;
static int cstat;
static struct semaphore icepic_mutex;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6,0,0)
static DEFINE_SEMAPHORE (icepic_mutex,1);
#elif defined(RHEL_RELEASE_CODE) && RHEL_RELEASE_CODE >= 2310 // RHEL 9.6
#define RHRC_GT_2310 1
static DEFINE_SEMAPHORE (icepic_mutex,1);
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,37)
static DEFINE_SEMAPHORE (icepic_mutex);
#else
static DECLARE_MUTEX (icepic_mutex);
#endif

#define ICEPIC_LOCK down(&icepic_mutex)
#define ICEPIC_UNLOCK up(&icepic_mutex)

#if _NEW
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,24)
#define mygid task_pgrp_nr(current)
//#define mygid current_gid()
#define myuid current_uid()
#else
#define mygid process_group(current)
//#define mygid current->fsgid
#define myuid current->fsuid
#endif
#define mypid current->pid
#else
#define mygid current->pgrp
#define mypid current->pid
#define myuid current->uid
#endif

/*************************************/
/* Internal Function Prototypes      */
/*************************************/
static int icepic_attach (struct pci_dev *pdev, const struct pci_device_id *ent);
static void icepic_detach (struct pci_dev *pdev);

void dma_alloc (int_4 node, int_4 *ice_ra, int_4 *ice_rs, int_4 *ice_rd, int_4 *ice_rm);
void dma_free (int_4 node);
int icemapknr (struct vm_area_struct *vma, int dir);

/************************************/
/* Prototypes for External Routines */
/************************************/
#ifdef __cplusplus
#define EXTERN_CDEFINE extern "C"
#else
#define EXTERN_CDEFINE extern
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,36)
EXTERN_CDEFINE long icepic_ioctl (struct file *filp, unsigned int cmd, unsigned long arg);
#else
EXTERN_CDEFINE int icepic_ioctl (struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg);
#endif
EXTERN_CDEFINE long icepic_compat_ioctl (struct file *filp, unsigned int cmd, unsigned long arg);
EXTERN_CDEFINE int icepic_ioctl32 (unsigned int fd, unsigned int cmd, unsigned long arg, struct file *filp);
EXTERN_CDEFINE int iceram_mmap (struct file *filp, struct vm_area_struct *vma);

EXTERN_CDEFINE int iceram_open (struct inode *inode, struct file *filp);
EXTERN_CDEFINE ssize_t iceram_write (struct file *filp, const char *buf, size_t len, loff_t *off);
EXTERN_CDEFINE ssize_t iceram_read (struct file *filp, char *buf, size_t len, loff_t *off);
EXTERN_CDEFINE int iceram_close (struct inode *inode, struct file *filp);

struct file_operations iceram_fops = {
  open:		iceram_open, 
  read:		iceram_read, 
  write:	iceram_write, 
  release:	iceram_close,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,36)
  unlocked_ioctl: icepic_ioctl,
#else
  ioctl:	icepic_ioctl, 
#endif
#if defined(CONFIG_COMPAT) && defined(HAVE_COMPAT_IOCTL)
  compat_ioctl:	icepic_compat_ioctl, 
#endif
  mmap:		iceram_mmap
};

EXTERN_CDEFINE int icepic_open (struct inode *inode, struct file *filp);
EXTERN_CDEFINE ssize_t icepic_write (struct file *filp, const char *buf, size_t len, loff_t *off);
EXTERN_CDEFINE ssize_t icepic_read (struct file *filp, char *buf, size_t len, loff_t *off);
EXTERN_CDEFINE int icepic_close (struct inode *inode, struct file *filp);

int icepic_cancel (struct inode *inode, struct file *filp, int reason);
int icepic_mapio (int index, int function, int *channel, int_u8 psize);
int icepic_reattach (int index);
int icepic_setram (int_4 node, int_4 start, int_4 sized, int_4 sizem, int_4 verbose);
void reverse (void* pBuf, int pLen);

struct file_operations icepic_fops = {
  open:		icepic_open, 
  read:		icepic_read, 
  write:	icepic_write, 
  release:	icepic_close,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,36)
  unlocked_ioctl: icepic_ioctl,
#else
  ioctl:	icepic_ioctl, 
#endif
#if defined(CONFIG_COMPAT) && defined(HAVE_COMPAT_IOCTL)
  compat_ioctl:	icepic_compat_ioctl, 
#endif
  mmap:		iceram_mmap
};

static struct pci_device_id icepic_pci_tbl[] = {
  {
    vendor:VENDOR_ID,
    device:DEVICE_ID,
    subvendor:PCI_ANY_ID,
    subdevice:PCI_ANY_ID,
    class: 0, class_mask:0,
  },
  {}
};
MODULE_DEVICE_TABLE(pci, icepic_pci_tbl);

static struct pci_driver icepic_driver = {
  name:       PICDRV_NAME,
  id_table:   icepic_pci_tbl,
  probe:      icepic_attach,
  remove:     icepic_detach,
  suspend:    NULL,
  resume:     NULL
};

/************************************/
/* picdrv init/exit routine s       */
/************************************/
static int __init icepic_init (void) {
  int i,j; 
  if (ice_ra<0) return -1; // called by udev - not icepic script
  if (ice_rb<0) ice_rb = 0;
  for (i=0; i<MAXPIC; i++) ucbs[i].attached=0;
  icepic_major = register_chrdev (0, PICDRV_NAME, &icepic_fops);
  icepic_count = pci_register_driver (&icepic_driver); /* probe and init */
  for (i=j=0; i<MAXPIC; i++) { if (ucbs[i].attached > 0) j++; }
  if (j>icepic_count) icepic_count=j; /* pci_register_driver count is unreliable */
  vprintf("icepic: init major=%d count=%d\n",icepic_major,icepic_count);
#if defined(CONFIG_COMPAT) && !defined(HAVE_COMPAT_IOCTL)
  if (register_ioctl32_conversion(ICEIOCCMD32, icepic_ioctl32) != 0)
    vprintf("icepic: unable to register ioctl32_conversion entry\n");
#endif
#if ICEQOS == 3
  cpu_latency_qos_add_request(&icepic_qos_handle, ice_lat);
  printk(KERN_INFO "icepic: QOS_CPU_DMA_LATENCY-r3 set=%d uS\n",ice_lat);
#elif ICEQOS == 2
  pm_qos_add_request(&icepic_qos_handle, PM_QOS_CPU_DMA_LATENCY, ice_lat);
  printk(KERN_INFO "icepic: QOS_CPU_DMA_LATENCY-r2 set=%d get=%d uS\n",ice_lat,pm_qos_request(PM_QOS_CPU_DMA_LATENCY));
#elif ICEQOS == 1
  pm_qos_add_requirement(PM_QOS_CPU_DMA_LATENCY, PICDRV_NAME, ice_lat);
  printk(KERN_INFO "icepic: QOS_CPU_DMA_LATENCY-r1 set=%d get=%d uS\n",ice_lat,pm_qos_requirement(PM_QOS_CPU_DMA_LATENCY));
#endif
  for (i=0; i<ice_rn; i++) { rnd[i].rs=ice_rs; rnd[i].rd=ice_rd; rnd[i].rm=ice_rm; }
#if ICECD
  iceram_major = register_chrdev (0, RAMDRV_NAME, &iceram_fops);
  vprintf("iceram: init major=%d count=%d\n",iceram_major,ice_rn);
#endif
#if ICEKAM
  if (ice_ra>0) for (i=0; i<ice_rn; i++) dma_alloc(i,&ice_ra,&rnd[i].rs,&rnd[i].rd,&rnd[i].rm);
#else
  if (ice_ra>0) printk(KERN_INFO "icekam: DMA_ALLOC_KERNEL - kernel allocation requested - not supported in this build\n");
  ice_ra = 0;
#endif
  for (i=0; i<ice_rn; i++) icepic_setram(i, rnd[i].rs,rnd[i].rd,rnd[i].rm, 1);
  create_proc_entry_withInfo ("driver/icepic", 0, NULL, proc_icepic);
#if ICEBD
  icebd_init();
#endif
#if ICEFS
  icefs_init();
#endif
  return 0; /* keep loaded for hot plug even if no devices found */
}

static void __exit icepic_cleanup (void) {
#if ICEKAM
  int_4 i;
#endif
#if ICEFS
  icefs_exit();
#endif
#if ICEBD
  icebd_exit();
#endif
  remove_proc_entry("driver/icepic",NULL);
  vprintf("icepic: unloading major=%d count=%d\n",icepic_major,icepic_count);
#if defined(CONFIG_COMPAT) && !defined(HAVE_COMPAT_IOCTL)
  unregister_ioctl32_conversion(ICEIOCCMD32);
#endif
#if ICECD
  unregister_chrdev (iceram_major, RAMDRV_NAME);
  iceram_major=0;
#endif
  unregister_chrdev (icepic_major, PICDRV_NAME);
  pci_unregister_driver (&icepic_driver);
  icepic_major=icepic_minor=icepic_count=0;
#if ICEKAM
  if (ice_ra>0) for (i=0; i<ice_rn; i++) dma_free(i);
#endif
#if ICEQOS == 3
  cpu_latency_qos_remove_request(&icepic_qos_handle);
#elif ICEQOS == 2
  pm_qos_remove_request(&icepic_qos_handle);
#elif ICEQOS == 1
  pm_qos_remove_requirement(PM_QOS_CPU_DMA_LATENCY, PICDRV_NAME);
#endif
}

module_init(icepic_init);
module_exit(icepic_cleanup);

/************************************/
/* pic attach/detach routines       */
/************************************/
static int icepic_attach (struct pci_dev *pdev, const struct pci_device_id *ent)
{
  int i,status,index=icepic_minor++;
  UCB *ucb = &ucbs[index];
  if (index>=ice_mc) return -1;
  ucb->attached = 0;
  pci_set_master(pdev);
  status = pci_read_config_dword(pdev,0x10,&ucb->membase); ucb->membase &= 0xFFFFFFF0;
  status = pci_read_config_dword(pdev,0x04,&ucb->csr);
  status = pci_read_config_dword(pdev,0x08,&ucb->rev);
  status = pci_read_config_dword(pdev,0x0C,&ucb->lat);
  vprintf ("icepic: attach - unit %d base %x csr %x lat %d\n",index,ucb->membase,ucb->csr,(ucb->lat>>8)&0xFF);
  ucb->loc = (void *)ioremap( (unsigned long)ucb->membase, 0x10000);
  ucb->pdev = pdev;
  ucb->attached = 1;
  ucb->lastoffset = 0;
  ucb->active = 0;
  ucb->flags = 0;
  ucb->time = ucb->pktime = 0;
  ucb->temp = ucb->pktemp = -2;
  for (i=0; i<5; i++) ucb->temps[i]=-2;
  status = pci_enable_device(pdev);
  ucb->pvid = readl(ucb->loc);
  return 0; 
}

static void icepic_detach (struct pci_dev *pdev)
{
  UCB *ucb; int index; 
  for (index=0; index<MAXPIC && ucbs[index].pdev!=pdev; index++);
  if (index==MAXPIC) return;
  icepic_minor--;
  ucb = &ucbs[index];
  if (ucb->attached==0) return;
  vprintf ("icepic: detach - unit %d baseaddr %x\n",index,ucb->membase);
  iounmap (ucb->loc);
  ucb->attached = 0;
  return;  
}

/************************************/
/* pic open routine                 */
/************************************/
int icepic_open (struct inode *inode, struct file *filp)
{
  UCB *ucb; int index = IMINOR(inode); 
  if (filp->f_flags&O_NONBLOCK) return (ESUCCESS); /* mmap */
  wprintf ("icepic: open - dev %#x fp.flags %#x\n",index,filp->f_flags);
  ucb = &ucbs[index];
  if (ucb->attached == 0) {
    wprintf("icepic: open - device unit not attached\n"); /* normal detection case - dont spam kernel log */
    return(-ENXIO);
  }
  ucb->opened++;
  return(ESUCCESS); 
}

/************************************/
/* pic close routine                */
/************************************/
int icepic_close (struct inode *inode, struct file *filp)
{
  UCB *ucb; int index = IMINOR(inode);
  if (filp->f_flags&O_NONBLOCK) return (ESUCCESS); /* mmap */
  ucb = &ucbs[index];;
  wprintf ("icepic: close - dev %#x fp.flags %#x\n",index,filp->f_flags);
  if (ucb->attached == 0) {
    vprintf("icepic: close - device unit=%d not attached\n",index);
  }
  /* make sure all DMA's from this PID have been terminated */
  icepic_cancel (inode, filp, ucb->opened);
  ucb->opened--;
  return 0;
}
    
int icepic_acmdw (int index, volatile int *wtmp, int timeout)
{
  int stat,j,k;
  UCB *ucb = &ucbs[index];
  for (k=4; k>=0; k--) _WR(REG_MBX+k*4,wtmp[k]);
  for (j=0; j<timeout; j++) {  /* cant wait forever in kernel mode */
    _RD (REG_MCSR,stat);
    if ( (stat&MCSR_IMB1F) != 0x0 ) {
      _RD (REG_IMB1, wtmp[0]);		
      return 4;
    }
    if      (j>40) udelay(1000);
    else if (j>12) udelay(100);
  }
  return 0;
}

int icepic_acmd (int index, int cmd1, int cmd2, int cmd3, int cmd4)
{
  volatile int wtmp[6];	/* kernel space temp data */
  int stat,retry=100;
  do {
    wtmp[0]=cmd1;
    wtmp[1]=cmd2;
    wtmp[2]=cmd3;
    wtmp[3]=cmd4;
    wtmp[4]=cmd1^cmd2^cmd3^cmd4;
    stat = icepic_acmdw(index,wtmp,40);
  }
  while (stat==0 && --retry>0);
  return wtmp[0];
}

void *icepic_user_to_phys (const void *buf)
{
//  return (void*)buf;
  return (void*)0;
}

/************************************/
/* ram access routines              */
/************************************/
int iceram_open (struct inode *inode, struct file *filp)
{
  int index = IMINOR(inode);
  wprintf ("iceram: open - dev %#x fp.flags %#x\n",index,filp->f_flags);
  return (ESUCCESS);
}
int iceram_close (struct inode *inode, struct file *filp)
{
  int index = IMINOR(inode);
  wprintf ("iceram: close - dev %#x fp.flags %#x\n",index,filp->f_flags);
  return (ESUCCESS);
}

/************************************/
/* ram write routine                */
/************************************/
ssize_t iceram_write (struct file *filp, const char *buf, size_t len, loff_t *off)
{
  int status, index=FMINOR(filp);
  void *puse = icepic_user_to_phys((const void*)buf);
  void *padr = (void*)phys_to_virt(rnd[index].offset_min+*off);
  wprintf("iceram: write node=%d len=%zx off=%llx phys=%p buf=%p\n",index,len,*off,padr,buf);
  if (puse==0) {
    status = copy_from_user(padr, (void *)buf, len);
  } else {
    memcpy(padr, puse, (size_t)len);
  }
  *off += len;
  return len;
}

/************************************/
/* ram read routine                 */
/************************************/
ssize_t iceram_read (struct file *filp, char *buf, size_t len, loff_t *off)
{
  int status, index=FMINOR(filp);
  void *puse = icepic_user_to_phys((const void*)buf);
  void *padr = (void*)phys_to_virt(rnd[index].offset_min+*off);
  wprintf("iceram: read node=%d len=%zx off=%llx phys=%p buf=%p\n",index,len,*off,padr,buf);
  if (puse==0) {
    status = copy_to_user((void *)buf, padr, len);
  } else {
    memcpy(puse, padr, (size_t)len);
  }
  *off += len;
  return len;
}

/************************************/
/* pic write routine                */
/************************************/
ssize_t icepic_write (struct file *filp, const char *buf, size_t len, loff_t *off)
{
  int status, chan, index=FMINOR(filp);
  int poff=(*off)&PMSK, pend=(*off+len+63)&PMSK, poffb=(*off)&0x3F, plen=pend-poff;
  void *puse = icepic_user_to_phys((const void*)buf);
  wprintf("icepic: write node=%d len=%zx off=%llx plen=%d phys=%p\n",index,len,*off,plen,puse);
  if (puse==0) {
    status = icepic_mapio (index, IOCTL_MAP, &chan, plen); 
    status = copy_from_user((void *)phys_to_virt(dma[chan].paddr+poff+poffb), (void *)buf, len);
    status = icepic_acmd (index, ADON_DMAX, dma[chan].paddr>>6, poff, plen);
    status = icepic_mapio (index, IOCTL_UNMAP, &chan, 0);
  } else {
    status = icepic_acmd (index, ADON_DMAX, (int)((size_t)puse>>6), poff, plen);
  }
  *off += len;
  return len;
}

/************************************/
/* pic read routine                 */
/************************************/
ssize_t icepic_read (struct file *filp, char *buf, size_t len, loff_t *off)
{
  int status, chan, index=FMINOR(filp);
  int poff=(*off)&PMSK, pend=(*off+len+63)&PMSK, poffb=(*off)&0x3F, plen=pend-poff;
  void *puse = icepic_user_to_phys((const void*)buf);
  wprintf("icepic: read node=%d len=%zx off=%llx plen=%d phys=%p\n",index,len,*off,plen,puse);
  if (puse==0) {
    status = icepic_mapio (index, IOCTL_MAP, &chan, plen); 
    status = icepic_acmd (index, ADON_DMAX, dma[chan].paddr>>6, poff, -plen);
    status = copy_to_user((void *)buf, (void *)phys_to_virt(dma[chan].paddr+poff+poffb), len);
    status = icepic_mapio (index, IOCTL_UNMAP, &chan, 0);
  } else {
    status = icepic_acmd (index, ADON_DMAX, (int)((size_t)puse>>6), poff, -plen);
  }
  *off += len;
  return len;
}

/************************************/
/* pic ioctl routine                */
/************************************/
#define PIC_FUNCTION	p->function	/* user function code */
#define PIC_OFFSET	p->offset	/* register offset */
#define PIC_BUFADDR	p->bufaddr	/* user buffer addr */
#define PIC_BYTES	p->bytes	/* bytes to transfer */
#define PIC_STATUS	p->status	/* return status */

int icepic_ioctl_work (int index, PICIOCTL *p)
{
  UCB 		*ucb  = &ucbs[index];

  int		i,j,k;		/* Local counters */
  int		status;		/* Local function return status */
  volatile int	stat;		/* bus read return status */
  int		chan;		/* Current dma channel */
  volatile int	wtmp[8];	/* kernel space temp data */
  int		worda,wordb;	/* load IOC temps */
  int_u8	msize,maddr;	/* word temps */
  int_8		addr;
  int_4		*data;

  if (PIC_FUNCTION==IOCTL_MAP||PIC_FUNCTION==IOCTL_UNMAP) ICEPIC_LOCK;

  i = PIC_BYTES;			/* normal|default completion status */
  status = ESUCCESS; 			/* assume success */

  switch (PIC_FUNCTION) {

  case IOCTL_RCFG:			/* read a configuration register */

    status = pci_read_config_dword(ucb->pdev,PIC_OFFSET,(int*)wtmp);
    cstat=copy_to_user((void *)PIC_BUFADDR, (void *)wtmp, 4);
    break;

  case IOCTL_READ:			/* read a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=0; i<PIC_BYTES; ) {
        _RD (REG_MCSR,stat);
             if ((stat&MCSR_IFIFO_FULL) && (PIC_BYTES-i>=24)) j=6;
        else if ((stat&MCSR_IFIFO_AFULL) && (PIC_BYTES-i>=16)) j=4;
        else if ((stat&MCSR_IFIFO_HALF) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_IFIFO_EMPTY)) j=1;
        else break;
        for (k=0; k<j; k++) _RD(REG_FIFO, wtmp[k]);  
	cstat=copy_to_user(((void *)PIC_BUFADDR)+i, (void *)wtmp, j*4);
	i += (j*4);
      }
    }
    else if ( (PIC_OFFSET&0XFFFF0000) != 0) 	/* invalid register ? */
      status = EINVAL;
    else {
      _RD (PIC_OFFSET, wtmp[0]);		
      cstat=copy_to_user((void *)PIC_BUFADDR, (void *)wtmp, 4);
    }
    break;

  case IOCTL_WRITE:			/* write a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=0; i<PIC_BYTES; ) {
        _RD (REG_MCSR,stat);
             if ((stat&MCSR_OFIFO_EMPTY) && (PIC_BYTES-i>=24)) j=6;
        else if (!(stat&MCSR_OFIFO_HALF) && (PIC_BYTES-i>=16)) j=4;
        else if (!(stat&MCSR_OFIFO_AFULL) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_OFIFO_FULL)) j=1;
        else break;
	cstat=copy_from_user((void *)wtmp, ((void *)PIC_BUFADDR)+i, j*4);
        for (k=0; k<j; k++) _WR(REG_FIFO, wtmp[k]);  
	i += (j*4);
      }
    }
    else if ( (PIC_OFFSET&0XFFFF0000) != 0) 	/* invalid register ? */
      status = EINVAL;
    else {
      cstat=copy_from_user((void *)wtmp, (void *)PIC_BUFADDR, 4);
      if (PIC_OFFSET==ucb->lastoffset+4) _WR(REG_PBAR+8,0);
      _WR(PIC_OFFSET,wtmp[0]);
      ucb->lastoffset=PIC_OFFSET;
    }
    break;

  case IOCTL_ACMD: 			/* command to CARD processor */

    cstat=copy_from_user((void *)wtmp, (void *)PIC_BUFADDR, 20);
    i = icepic_acmdw (index,wtmp,40);
    if (i>0) cstat=copy_to_user((void *)PIC_BUFADDR, (void *)wtmp, i);
    break;

  case IOCTL_MAP:			/* map user buffer to PCI */

    i=0;	/* assume failure */
    /* find an open dma handler slot */
    if (ndma==0 || ndma+1>=MAXIOBUF) status=EINVAL; 

    msize = phys2long(PIC_BYTES);

    /* Validate user buffer */
    if ( (msize == 0)  || (msize > MAXWCNT) ) status = EINVAL;
    if (status!=ESUCCESS) break;

    /* map buffer into PCI bus space */
    status = icepic_mapio (index, IOCTL_MAP, &chan, msize);
    if (status==ESUCCESS) i=long2phys(dma[chan].paddr); 
    break;
    
  case IOCTL_UNMAP: 			/* unmap user buffer from PCI */

    msize = phys2long(PIC_BYTES);
    maddr = phys2long(PIC_OFFSET);

    /* find the matching dma handler slot */
    for (chan=1; chan<ndma; chan++) {
      if (maddr >= dma[chan].paddr && 
	  maddr+msize <= dma[chan+1].paddr) break;
    }
    if (chan>=ndma) { status = EINVAL; break; }

    /* unmap user buffer from PCI bus space */
    status = icepic_mapio (index, IOCTL_UNMAP, &chan, 0);
    break;

  case IOCTL_QMAP: 			/* query user buffer map */

    msize = phys2long(PIC_BYTES);
    maddr = phys2long(PIC_OFFSET);

    /* is it in the table of driver mapped PCI memory */
    for (chan=0; chan<ndma; chan++) {
      if (maddr >= dma[chan].paddr && 
	  maddr+msize <= dma[chan].paddr+dma[chan].psize) break;
    }
    if (chan==ndma) i=0;
    else if (PIC_BYTES!=0) i=PIC_BYTES;
    else i=long2phys(dma[chan].paddr+dma[chan].psize-maddr);  
    break;

  case IOCTL_RALLOC: 			/* allocate contiguous RAM */

    status = icepic_setram ( 0, (int_4)PIC_OFFSET, (int_4)(long)PIC_BUFADDR, (int_4)PIC_BYTES, 1);

    break;

  case IOCTL_QALLOC: 			/* report allocated RAM */

    i = long2phys(rstart);
    break;

  case IOCTL_LIOC:			/* load IOC words */

    for (i=0; i<PIC_BYTES; i+=4) {
      k = PIC_BYTES-i; if (k>4) k=4;
      cstat=copy_from_user((void *)(&worda), ((void *)PIC_BUFADDR)+i, k);
      reverse(&worda,4);
      for (j=0; j<(k*8); j++) {
        wordb = (worda>>j)&0x1;
        _WR(REG_IOC,wordb);
      }
    }
    break;

  case IOCTL_REATTACH:			/* reattach test device */

    i = icepic_reattach (index);
    break;

  case IOCTL_VERSION:			/* return version */

    wtmp[0] = ICE_DRIVER_VERSION;
    wtmp[1] = icepic_count;
    wtmp[2] = ice_ra;
    wtmp[3] = ice_rs;
    wtmp[4] = ice_rd;
    wtmp[5] = ice_rm;
    cstat=copy_to_user((void *)PIC_BUFADDR, (void *)wtmp, PIC_BYTES);
    break;

  case IOCTL_LOG:			/* log status */

    cstat=copy_from_user((void *)wtmp, (void *)PIC_BUFADDR, PIC_BYTES);
    if (PIC_BYTES==4) { ucb->active = wtmp[0]; break; }
    ucb->time = wtmp[0];
    ucb->temps[0] = wtmp[1];
    ucb->temps[1] = (PIC_BYTES>8)?  wtmp[2]:-1;
    ucb->temps[2] = (PIC_BYTES>12)? wtmp[3]:-1;
    ucb->temps[3] = (PIC_BYTES>16)? wtmp[4]:-1;
    ucb->temps[4] = (PIC_BYTES>20)? wtmp[5]:-1;
    for (k=-2,i=0; i<5; i++) if (ucb->temps[i] > k) k = ucb->temps[i];
    ucb->temp = k;
    if (k > ucb->pktemp) { ucb->pktemp = k;  ucb->pktime = ucb->time; }
    if (k>MAXTEMP) printk("ICEPIC-%d Overtemp nodes=(%d,%d,%d,%d,%d)\n",
	index,ucb->temps[0],ucb->temps[1],ucb->temps[2],ucb->temps[3],ucb->temps[4]);
    if (PIC_BYTES>24) ucb->flags    = wtmp[6];
    if (PIC_BYTES>28) ucb->active   = wtmp[7];
    break;

  case IOCTL_QLOG:			/* log status */

    wtmp[0] = ucb->time;
    wtmp[1] = ucb->temp;
    wtmp[2] = ucb->pktime;
    wtmp[3] = ucb->pktemp;
    cstat=copy_to_user((void *)PIC_BUFADDR, (void *)wtmp, 16);
    break;

  case IOCTL_PMDUMP: 			/* allocate contiguous RAM */
    addr = (int_8)PIC_BUFADDR;
    if (addr<0) data = (int_4*)addr;	/* assume already virtual */
    else data = phys_to_virt(addr);	/* convert to virtual address */
    printk("iceram: dump of %d bytes at phys=%llx virt=%llx\n",PIC_BYTES,(int_8)PIC_BUFADDR,(int_8)data);
    for (i=0; i<(PIC_BYTES>>2); i+=8) {
      for (j=0;j<8;j++) wtmp[j]=data[i+j];
      printk(" i=%04x  %08x %08x %08x %08x %08x %08x %08x %08x\n",i,wtmp[0],wtmp[1],wtmp[2],wtmp[3],wtmp[4],wtmp[5],wtmp[6],wtmp[7]);
    }
    break;

  case IOCTL_RCD:			/* register card disk */
#if ICEBD
    cstat=copy_from_user((void *)wtmp, (void *)PIC_BUFADDR, PIC_BYTES);
    icebd_register_card_disk(PIC_OFFSET,wtmp[0]);
#else
    status = EINVAL;
#endif
    break;

  default: 				/* handle unknown function codes */
    status = EINVAL;
    break;
  }

  if (PIC_FUNCTION==IOCTL_MAP||PIC_FUNCTION==IOCTL_UNMAP) ICEPIC_UNLOCK;

  p->status = i;
  return(status);
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,36)
long icepic_ioctl (struct file *filp, unsigned int cmd, unsigned long arg)
{
  long		status;
  int		index = FMINOR(filp);
#else
int icepic_ioctl (struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
  int		status;
  int 		index = IMINOR(inode);
#endif
  int		psize = _IOC_SIZE(cmd);
  PICIOCTL	ploc;	/* pic ioctl handle structure */
  uprintf("icepic: ioctl - dev %#x, cmd %#x, arg %#lx\n", index,cmd,arg);
  cstat=copy_from_user ((void *)(&ploc), (void *)arg, psize);
  status = icepic_ioctl_work (index,&ploc);
  cstat=copy_to_user((void *)&((PICIOCTL *)arg)->status, (void *)&ploc.status, 4);
  return status;
}

long icepic_compat_ioctl (struct file *filp, unsigned int cmd, unsigned long arg)
{
  int 		index = FMINOR(filp);
  int		psize = _IOC_SIZE(cmd);
  int		status;
  PICIOCTL32	ploc32;	/* pic ioctl handle structure */
  PICIOCTL	ploc;	/* pic ioctl handle structure */
  uprintf("icepic: compat_ioctl - dev %#x, cmd %#x, arg %#lx\n", index,cmd,arg);
  cstat=copy_from_user ((void *)(&ploc32), (void *)arg, psize);
  ploc.function = ploc32.function;
  ploc.offset   = ploc32.offset;
  ploc.bufaddr  = (void *)(long)ploc32.bufaddr;
  ploc.bytes    = ploc32.bytes;
  ploc.status   = ploc32.status;
  status = icepic_ioctl_work (index,&ploc);
  cstat=copy_to_user((void *)&((PICIOCTL32 *)arg)->status, (void *)&ploc.status, 4);
  return (long)status;
}

int icepic_ioctl32 (unsigned int fd, unsigned int cmd, unsigned long arg, struct file *filp)
{
  return (int)icepic_compat_ioctl (filp, cmd, arg);
}

int icepic_setram (int_4 node, int_4 start, int_4 sized, int_4 sizem, int_4 verbose) {
  nram=node+1;
  
  if (start!=-1) rstart = (int_u8)start<<20; 	/* RAM start in bytes */
  if (sized!=-1) rsized = (int_u8)sized<<20; 	/* RAMDISK size in bytes */
  if (sizem!=-1) rsizem = (int_u8)sizem<<20; 	/* DYNRAM size in bytes */
  offset_min = rstart;
  offset_max = rstart+rsized+rsizem;

  rnd[node].rs = rstart>>20;
  rnd[node].rd = rsized>>20;
  rnd[node].rm = rsizem>>20;
  rnd[node].offset_min = offset_min;
  rnd[node].offset_max = offset_max;

  dma[node].paddr=rstart;
  dma[node].psize=rsized;
  dma[node].pid=0;
  dma[nram].paddr=rstart+rsized+rsizem;
  dma[nram].psize=0;
  dma[nram].pid=0;
  ndma=nram;

  if (verbose) printk("icepic: setram node=%d start=%dM sized=%dM sizem=%dM  off_min=%llx off_max=%llx\n",
					node,start,sized,sizem,offset_min,offset_max);
  return 0;
}

/************************************/
/* pic cancel routine               */
/************************************/
int icepic_cancel (struct inode *inode, struct file *filp, int reason)
{
  int reset,status,chan;
  int index = IMINOR(inode);
  UCB *ucb = &ucbs[index];
  reset = MCSR_RESET;

  wprintf ("icepic: cancel - dev %#x reason %x flags %x\n",index,reason,filp->f_flags);

  if (reason==1) {	/* close */
    ICEPIC_LOCK;
    for (chan=ndma-1; chan>=1; chan--) {
      if (dma[chan].device==index && dma[chan].gid==mygid && dma[chan].paddr!=0) {
	vprintf ("icepic: cancel - stray DMA - resetting card\n");
	/* unexpected termination - reset card to avoid stray DMAs */
	_WR (REG_MCSR, reset);
	/* deallocate/unmap stray buffer */
	status = icepic_mapio (index, IOCTL_UNMAP, &chan, 0L);
      }
    }
    ICEPIC_UNLOCK;
  }
  return(ESUCCESS); 
}



/************************************/
/* pic mapio routine                */
/************************************/
int icepic_mapio (int index, int function, int *channel, int_u8 psize)
{
  int i,n,status,chan;

  status = ESUCCESS;	/* assume success */

  switch (function) {

  case IOCTL_MAP:

    for (i=nram; i<=ndma; i++) if (dma[i].paddr-
	(dma[i-1].paddr+dma[i-1].psize) >= psize) goto ALLOCATE;

    return ENOMEM; /* out of room */

    ALLOCATE:
    n=i; ndma++;
    for (i=ndma; i>n; i--) dma[i] = dma[i-1];
    dma[n].paddr = dma[n-1].paddr+dma[n-1].psize;
    dma[n].psize = psize;
    dma[n].pid = mypid;
    dma[n].gid = mygid;
    dma[n].device = index;
    wprintf("icepic: DMA map psize=%llx n=%d paddr=%llx ndma=%d\n",psize,n,dma[n].paddr,ndma);
    *channel = n;
    break;

  case IOCTL_UNMAP:

    chan = *channel;
    if (chan>=0) psize = dma[chan].psize;
    wprintf("icepic: DMA unmap psize=%llx %d\n",psize,ndma);
    for (i=chan; i<ndma; i++) dma[i] = dma[i+1];
    ndma--;
    break;

  default:
    status = EINVAL;
  }
#ifdef DEBUG_MAPIO
  for (i=0; i<=ndma; i++) wprintf("icepic: DMA table i=%d paddr=%llx psize=%llx\n",i,dma[i].paddr,dma[i].psize);
#endif
  return (status);
}

/************************************/
/* pic reattach routine             */
/************************************/
int icepic_reattach (int index)
{
  UCB *ucb = &ucbs[index];
  int status;
  ucb->attached = 0;
  if (!ucb->pdev) return 0; 
  status = pci_write_config_dword(ucb->pdev,0x10,ucb->membase);
  status = pci_write_config_dword(ucb->pdev,0x04,ucb->csr);
  status = pci_write_config_dword(ucb->pdev,0x0C,ucb->lat);
  vprintf ("icepic: reattach - unit %d base %x csr %x\n",index,ucb->membase,ucb->csr);
  ucb->attached = 1;
  return (1); 
}

void reverse (void* pBuf, int pLen) 
{
#if defined(BYTE_ORDER) && BYTE_ORDER == BIG_ENDIAN
  int i,j;
  char temp, *tPtr;
  tPtr = (char *)pBuf;
  j=pLen; pLen >>= 1;
  for (i=0; i<pLen; i++) {
    temp = tPtr[--j]; tPtr[j]=tPtr[i]; tPtr[i]=temp;
  }
#endif
}

char *iceram_v2p (char *uvaddr, size_t len) 
{
  int i;
  int_u8 vaddr = (int_u8)uvaddr;
  for (i=0; i<nmap; i++) {
    if (vaddr>=map[i].vstr && vaddr+len<=map[i].vend) {
      wprintf("iceram: v2p %d vaddr=%llx len=%ld vstr=%llx vend=%llx\n",i,vaddr,len,map[i].vstr,map[i].vend);
      return (char*)phys_to_virt(map[i].paddr + (vaddr-map[i].vstr));
    }
  }
  return NULL;
}

char *iceram_v2p_temp (size_t len) 
{
   return (char*)phys_to_virt(rstart+rsized);
}

void iceram_vm_close (struct vm_area_struct *vma) 
{
  int i;
  for (i=0; i<nmap; i++) {
    if (map[i].vstr==vma->vm_start && map[i].vend==vma->vm_end) {
      for (--nmap; i<nmap; i++) map[i]=map[i+1];
      return;
    }
  }
}

static const struct vm_operations_struct iceram_vm_ops = {
  .close = iceram_vm_close,
};

/************************************/
/* pic mmap routine                 */
/************************************/
int iceram_mmap (struct file *filp, struct vm_area_struct *vma)
{
  int    index  = FMINOR(filp);
  int_u8 pfnoff = (int_u8)vma->vm_pgoff;
  int_u8 offset = pfnoff << PAGE_SHIFT;
  int_8  length = vma->vm_end - vma->vm_start;
  int_8  i;
#if ICEKAM
  int_8  pgcnt;
  void*  lpg;
#endif

  for (i=0; i<nram; i++) {
    wprintf("iceram: mmap devno=%d off=%llx end=%llx off_min=%llx, off_max=%llx vaddr=%lx flags=%x prot=%x\n",
	index,offset,offset+length,rnd[i].offset_min,rnd[i].offset_max,vma->vm_start,filp->f_flags,(int_u4)vma->vm_flags);
    if (offset>=rnd[i].offset_min && offset+length<=rnd[i].offset_max) goto GOTIT;
  }
  return -EAGAIN;
  GOTIT:
#if LINUX_VERSION_CODE < KERNEL_VERSION(6,0,0)
  #ifdef RHRC_GT_2310
    // do nothing, because kernel 5.14.0-570 doesn't allow this
  #else
    vma->vm_flags |= VM_IO;
  #endif
#endif
  if (nmap<MAXIOBUF) {
    vma->vm_ops     = &iceram_vm_ops;
    map[nmap].vstr  = vma->vm_start;
    map[nmap].vend  = vma->vm_end;
    map[nmap].paddr = offset;
    nmap++;
  }

#if ICEKAM
  if (ice_ra>0) { /* special mode for kernel allocated memory */
    pgcnt = (length > 0) ? (length >> PAGE_SHIFT) : (0);
    for(i=0; i < pgcnt; i++) {
      if ((lpg=pfn_to_page(pfnoff+i)) == (void *)NULL) return -EAGAIN;
      if (vm_insert_page (vma, vma->vm_start + i*PAGE_SIZE, lpg)) return -EAGAIN;
    }
    return 0;
  }
#endif

#if _VMA == 0
  if (remap_page_range(vma->vm_start, (vma->vm_pgoff<<PAGE_SHIFT), 
	vma->vm_end-vma->vm_start, vma->vm_page_prot)) return -EAGAIN;
#elif _VMA == 2
  if (remap_pfn_range(vma, vma->vm_start, vma->vm_pgoff, 
	vma->vm_end-vma->vm_start, vma->vm_page_prot)) return -EAGAIN;
#else
  if (remap_page_range(vma, vma->vm_start, (vma->vm_pgoff<<PAGE_SHIFT), 
	vma->vm_end-vma->vm_start, vma->vm_page_prot)) return -EAGAIN;
#endif
  return 0;
}

#define MAX_INFO_BUFC MAX_INFO_BUF-200
#define MAX_INFO_BUFS 32

static int proc_icepic (char *buf, char **a, off_t b, int c)
{
    char*   p = buf;
    int     i;

    p += sprintf (p, "RamStart  : %d Mby\n", rnd[0].rs);
    p += sprintf (p, "RamDisk   : %d Mby\n", rnd[0].rd);
    p += sprintf (p, "RamMapped : %d Mby\n", rnd[0].rm);
    p += sprintf (p, "RamPage   : %d Kby\n", (int)(PAGE_SIZE>>10));
    p += sprintf (p, "RamAlloc  : %d\n", ice_ra);
    p += sprintf (p, "Nodes     : %d\n", ice_rn);
    for (i=0; i<ice_rn; i++) {
      p += sprintf (p, "Node#%d    : Start=0x%010llx End=0x%010llx    RS=%d RD=%d RM=%d Mby\n", i,rnd[i].offset_min,rnd[i].offset_max,rnd[i].rs,rnd[i].rd,rnd[i].rm);
    }
    p += sprintf (p, "Buffers   : %d\n",ndma);
    for (i=0; i<ndma && i<MAX_INFO_BUFS; i++) {
      if ((p-buf)<MAX_INFO_BUFC)
      p += sprintf (p, "Buffer#%d  : Addr=0x%010llx Size=0x%010llx\n", i, dma[i].paddr, dma[i].psize);
    }
    p += sprintf (p, "Devices   : %d\n",icepic_count);
    for (i=0; i<MAXPIC; i++) if (ucbs[i].attached > 0) {
      if ((p-buf)<MAX_INFO_BUFC)
      p += sprintf (p, "Device#%d  : Revision=0x%x MemBase=0x%x CardTemp=%d MJS=%d PeakTemp=%d PMJS=%d CompTemp=(%d,%d,%d,%d,%d) Flags=0x%08x Active=%d\n",
		i,ucbs[i].rev&0xFF,ucbs[i].membase,ucbs[i].temp,ucbs[i].time,ucbs[i].pktemp,ucbs[i].pktime,
		ucbs[i].temps[0],ucbs[i].temps[1],ucbs[i].temps[2],ucbs[i].temps[3],ucbs[i].temps[4],ucbs[i].flags,ucbs[i].active);
    }
    p += sprintf (p, "Maps      : %d\n",nmap);
    p += sprintf (p, "Version   : %d\n", ICE_DRIVER_VERSION);
    return  (p - buf);
}

#if _ALLOBF
#include "icekbfs.cx"
#else

/***********************************/
/* IceKAM Block Device Driver Code */
/***********************************/
#if ICEKAM == 1
#include "icekam.c"
#elif ICEKAM == 2
#include "icekam.cx"
#endif

/**********************************/
/* IceBD Block Device Driver Code */
/**********************************/
#if ICEBD == 1
#include "icebd.c"
#endif

/*********************************/
/* IceFS File System Driver Code */
/*********************************/
#if ICEFS == 1
#include "icefs.c"
#endif

#endif
