/****************************************************************/
/*  ICE-PIC Device Driver for LINUX  2.4 kernels new PCI        */
/*                                                              */
/*  Author:  Jeff Schoen - Innovative Computer Engineering      */
/*                                                              */
/****************************************************************/
/*
   Bill Reese Mods:  
   membase must be unsigned so it's not sign extended when it's cast
    to an unsigned long for the call to ioremap().
   vfree to iounmap in detach

   For IA64:
     The usual method of reserving a block of memory at the top of available
   physical memory won't work on the IA64 platforms.  The issue is that
   physical memory is discontiguous in that it has large gaps in the memory
   map.  For example, an HP RX2600 with 2Gb of memory has one bank from
   0x00000000.00000000 to 0x00000000.3FFFFFFF, and the second bank is
   from 0x00000040.00000000 to 0x00000040.3FFFFFFF.  Memory reserved at the
   top of physical memory needs more address bits than the ICE can generate,
   and the IA64 platforms don't necessarily have IO remapping capabilities.
     To write to a physical address at the upper end of memory the ICE would
   have to generate Dual Address Cycles on the PCI bus.  The Linux DMA
   interface is set up to handle that by using bounce buffers.  That is, if
   you tell the kernel you want to DMA to upper physical memory, it will
   allocate a contiguous region in lower memory for you and copy memory
   back and forth for you when you do a pci_dma_sync_single() call.
     This would work, albeit with a performance penalty, if we knew
   a priori the direction of DMA transfers; pci_dma_sync_single doesn't like
   bidirectional DMA setups because it can't tell which way it should copy
   the bounce buffer.
     The second problem with using bounce buffers is that they're small, and
   the dma routines won't tell you if you ask for more memory than they have.
   In this case, you don't know that when you initiate your transfer you'll
   be writing past the end of the bounce buffer and overwriting bits of the
   operating system.  That doesn't work very well.
     The way I'm getting around this for now is to dynamically allocate
   memory when icelib sets up a mapping using the __get_free_pages()
   routine.  This allocates 2^N physically contiguous pages, and I can
   specify that I want the memory DMA-able.  On IA32 platforms this
   severely restricts the range of acceptable memory, but on IA64 platforms
   it just ensures the allocated memory is less than 0x00000001.00000000.
     The restriction left is that you can only allocate 2^MAX_ORDER
   contiguous pages using __get_free_pages.  This defaults to 11; however,
   on Red Hat Enterprise Linux 3.0 for IA64 this is set to 15 using
   CONFIG_FORCE_MAX_ZONEORDER in the kernel's .config file.  Also in
   the IA64 kernel's .config file CONFIG_IA64_PAGE_SIZE is set to 16k,
   so the maximum contiguous physical memory we can allocate on the IA64
   is 512Mb.  Of course, this requires there be 512Mb of contiguous physical
   memory to allocate, so you'd better ask early.
*/

#define ICE_DRIVER_VERSION 338

/*****************/
/* Include Files */
/*****************/
#include <linux/config.h>
#include <linux/module.h> 
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/isdnif.h>
#include <linux/vmalloc.h>
#include <linux/delay.h>
#include <linux/mm.h>
#include <asm/io.h>
#include <linux/mm_inline.h>

/**********************/
/* Module Definitions */
/**********************/
/* RedHat has internal inconsistencies for these 3 Macros comment out for now */
MODULE_AUTHOR("Jeff Schoen");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Driver for DSP Cards from Innovative Computer Engineering");

/****************************/
/* Debug/Status Definitions */
/****************************/
#define ESUCCESS 0
#define	SUCCESS(a)	(((a) & 1) == 1)
#define	FAILURE(a)	(((a) & 1) == 0)

#define vprintf printk
#define wprintf if (1!=0) printk

/***********************************/
/* Device Driver Macro Definitions */
/***********************************/

#define _RD(A,B)  {B = readl(ucb->loc+A);}
#define _WR(A,B)  {writel(B,(int *)(ucb->loc+A));}

/*******************************/
/* Driver Specific Definitions */
/*******************************/

typedef long long int_8;
typedef int int_4;

#include "iceioctl.h"
#include "iceflex.h"
#include "iceconv.h"

#define MAXPIC 12
#define MAXIOBUF 512
#define MAXWCNT 0x40000000

#define VENDOR_ID 0x1172
#define DEVICE_ID 0x7777
#define DRV_NAME "icepic"

typedef struct {
  int		device;
  pid_t 	pid,gid;
  int_4		psize;
  int_4		paddr;   /* "Encoded" physical address - doesn't work for  */
                         /* machines with physical addresses higher than   */
                         /* 16Gb                                           */
#if _USENEWDMA
  unsigned long pgoff;   /* Page offset for the physical address           */
                         /* Don't need offset in page because we'll        */
                         /* require page alignment                         */
  unsigned long kvaddr;  /* Kernel Virtual address                         */
  dma_addr_t    busaddr; /* Mapped bus address for DMA                     */
  int order;             /* power of 2 number of pages allocated           */
#endif
} DMACHANNEL;

typedef struct {
  int	attached;
  int	opened;
  unsigned int membase;
  void	*loc;
  int   csr,lat; 
  struct pci_dev *pdev;
} UCB;

static UCB ucbs[MAXPIC];
static DMACHANNEL dma[MAXIOBUF];
static unsigned int icepic_major=0;
static unsigned int icepic_minor=0;
static int ndma=0, icepic_count=0;
static unsigned long offset_min=0, offset_max=0;
static int_4 rstart, rsized, rsizem;
static struct semaphore icepic_mutex;
static DECLARE_MUTEX (icepic_mutex);
#define ICEPIC_LOCK down(&icepic_mutex)
#define ICEPIC_UNLOCK up(&icepic_mutex)

/************************************/
/* Prototypes for External Routines */
/************************************/

static int icepic_attach();
static void icepic_detach();

int icepic_open(), icepic_ioctl(), icepic_close(), icepic_mmap();
int icepic_cancel(), icepic_mapio(), icepic_reattach(); 
void icepic_setram(), reverse();

struct file_operations icepic_fops = {
  ioctl:	icepic_ioctl, 
  open:		icepic_open, 
  release:	icepic_close,
  mmap:		icepic_mmap
};

static struct pci_device_id icepic_pci_tbl[] = {
  {
    vendor:VENDOR_ID,
    device:DEVICE_ID,
    subvendor:PCI_ANY_ID,
    subdevice:PCI_ANY_ID,
    class: 0, class_mask:0,
  },
  {}
};
MODULE_DEVICE_TABLE(pci, icepic_pci_tbl);

static struct pci_driver icepic_driver = {
  name:       DRV_NAME,
  id_table:   icepic_pci_tbl,
  probe:      icepic_attach,
  remove:     icepic_detach,
  suspend:    NULL,
  resume:     NULL
};

static int __init icepic_init (void) {
  icepic_major = register_chrdev (0, DRV_NAME, &icepic_fops);
  icepic_count = pci_register_driver (&icepic_driver); /* probe and init */
  vprintf("icepic init major=%d count=%d\n",icepic_major,icepic_count);
  icepic_setram (RAM_START,RAM_SIZED,RAM_SIZEM); /* initial settings only */
  return 0; /* keep loaded for hot plug even if no devices found */
}

static void __exit icepic_cleanup (void) {
  vprintf("icepic exit major=%d count=%d\n",icepic_major,icepic_count);
  unregister_chrdev (icepic_major, DRV_NAME);
  pci_unregister_driver (&icepic_driver);
  icepic_major=icepic_minor=icepic_count=0;
}

module_init(icepic_init);
module_exit(icepic_cleanup);

/************************************/
/* pic attach/detach routines       */
/************************************/
static int __devinit icepic_attach (struct pci_dev *pdev, 
			const struct pci_device_id *ent)
{
  int status, index=icepic_minor++;
  UCB *ucb = &ucbs[index];
  ucb->attached = 0;
  pci_set_master(pdev);
  status = pci_read_config_dword(pdev,0x10,&ucb->membase);
  status = pci_read_config_dword(pdev,0x04,&ucb->csr);
  status = pci_read_config_dword(pdev,0x0C,&ucb->lat);
  vprintf ("icepic attach - unit %d base %x csr %x lat %d\n",
	index,ucb->membase,ucb->csr,(ucb->lat>>8)&0xFF);
  ucb->loc = (void *)ioremap( (unsigned long)ucb->membase, 4096);
  ucb->pdev = pdev;
  ucb->attached = 1;
  status = pci_enable_device(pdev);
  /*
   * Make sure this platform will do 32 bit DMA addresses
   */
  if (pci_set_dma_mask(pdev, 0xffffffff)) {
    wprintf ("icepic: No suitable DMA support\n");
  }
  return 0; 
}

static void __devexit icepic_detach (struct pci_dev *pdev)
{
  UCB *ucb; int index; 
  for (index=0; index<MAXPIC && ucbs[index].pdev!=pdev; index++);
  if (index==MAXPIC) return;
  icepic_minor--;
  ucb = &ucbs[index];
  if (ucb->attached==0) return;
  vprintf ("icepic detach - unit %d baseaddr %x\n",index,ucb->membase);
  iounmap (ucb->loc);
  ucb->attached = 0;
  return;  
}

/************************************/
/* pic open routine                 */
/************************************/
int icepic_open (struct inode *inode, struct file *filp)
{
  UCB *ucb; int index = MINOR(inode->i_rdev); 
  wprintf ("pic_open - dev %#x fp.flags %#x\n",index,filp->f_flags);
  ucb = &ucbs[index];
  if (ucb->attached == 0) {
    vprintf("pic_open - device unit not attached\n");
    return(-ENXIO);
  }
  if ((filp->f_flags&O_NONBLOCK)==0) /* not mmap */
  ucb->opened++;
  return(ESUCCESS); 
}
 


/************************************/
/* pic close routine                */
/************************************/
int icepic_close (struct inode *inode, struct file *filp)
{
  int index = MINOR(inode->i_rdev);
  UCB *ucb = &ucbs[index];;
  wprintf ("pic_close - dev %#x fp.flags %#x\n",index,filp->f_flags);
  if (ucb->attached == 0) {
    vprintf("pic_close - device unit not attached\n");
  }
  /* make sure all DMA's from this PID have been terminated */
  icepic_cancel (inode, filp, ucb->opened);
  if ((filp->f_flags&O_NONBLOCK)==0) /* not mmap */
  ucb->opened--;
  return 0;
}
    
/***************************************************
 * Walk through the Page Global Directory, the Page
 * Middle Directory then the Page Table Entries to find
 * the physical address associated with a particular
 * virtual address in this process's address space.
 ***************************************************/
void icepic_printva (unsigned long vaddr) {
  wprintf ("va 0x%016lx ", vaddr);
  pgd_t *pgd;
  if (rgn_index (vaddr) == RGN_KERNEL)
    pgd = pgd_offset_k (vaddr);
  else
    pgd = pgd_offset (current->mm, vaddr);
  if (pgd_present (*pgd)) {
    pmd_t *pmd = pmd_offset (pgd, vaddr);
    if (pmd_present (*pmd)) {
      pte_t *pte = pte_offset_map (pmd, vaddr);
      if (pte_present (*pte)) {
        wprintf ("pa 0x%016lx\n", (unsigned long) page_address (pte_page (*pte)));
      }
      else {
        wprintf ("pte not present\n");
      }
    }
    else {
      wprintf ("pmd not present\n");
    }
  }
  else {
    wprintf ("pgd not present\n");
  }
}

/************************************/
/* pic ioctl routine                */
/************************************/
#define PIC_BUFADDR	p->bufaddr	/* user buffer addr */
#define PIC_BYTES	p->bytes	/* bytes to transfer */
#define PIC_FUNCTION	p->function	/* user function code */
#define PIC_OFFSET	p->offset	/* register offset */
#define PIC_STATUS	p->status	/* return status */
#if _USENEWDMA
#define PIC_PGOFF       p->pgoff        /* Page offset of physical address */
#define PIC_BUSADDR     p->busaddr      /* Bus address for DMA             */
#endif
static int lastoffset=0;

int icepic_ioctl (struct inode *inode, struct file *filp, 
	unsigned int cmd, unsigned long arg)
{
  int 		index = MINOR(inode->i_rdev);
  UCB 		*ucb = &ucbs[index];

  int		i,j,k;		/* Local counters */
  int		status;		/* Local function return status */
  volatile int	stat;		/* bus read return status */
  int		chan;		/* Current dma channel */
  PICIOCTL	*p,ploc,*parg;	/* pic ioctl handle structure */
  volatile int	wtmp[6];	/* kernel space temp data */
  int		worda,wordb;	/* load IOC temps */
  int_4		msize;		/* word temp */
#if _USENEWDMA
  unsigned long page;           /* Page offset */
#else
  int_4		maddr;		/* word temp */
#endif
/*
  wprintf("pic_ioctl - dev %#x, cmd %#x, arg %#lx\n", index,cmd,arg);
*/

  p = &ploc;	/* ioctl data = the PICIOCTL */
  copy_from_user ((void *)p, (void *)arg, sizeof(PICIOCTL));
  parg = (PICIOCTL *) arg;

  if (PIC_FUNCTION==IOCTL_MAP||PIC_FUNCTION==IOCTL_MAP) ICEPIC_LOCK;

  i = PIC_BYTES;			/* normal|default completion status */
  status = ESUCCESS; 			/* assume success */

  switch (PIC_FUNCTION) {

  case IOCTL_READ:			/* read a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=0; i<PIC_BYTES; ) {
        _RD (REG_MCSR,stat);
             if ((stat&MCSR_IFIFO_FULL) && (PIC_BYTES-i>=24)) j=6;
        else if ((stat&MCSR_IFIFO_AFULL) && (PIC_BYTES-i>=16)) j=4;
        else if ((stat&MCSR_IFIFO_HALF) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_IFIFO_EMPTY)) j=1;
        else break;
        for (k=0; k<j; k++) _RD(REG_FIFO, wtmp[k]);  
	copy_to_user(((void *)PIC_BUFADDR)+i, (void *)wtmp, j*4);
	i += (j*4);
      }
    }
    else if ( (PIC_OFFSET&0XFFFFF000) != 0) 	/* invalid register ? */
      status = EINVAL;
    else {
      _RD (PIC_OFFSET, wtmp[0]);		
      copy_to_user((void *)PIC_BUFADDR, (void *)wtmp, 4);
    }
    break;

  case IOCTL_WRITE:			/* write a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=0; i<PIC_BYTES; ) {
        _RD (REG_MCSR,stat);
             if ((stat&MCSR_OFIFO_EMPTY) && (PIC_BYTES-i>=24)) j=6;
        else if (!(stat&MCSR_OFIFO_HALF) && (PIC_BYTES-i>=16)) j=4;
        else if (!(stat&MCSR_OFIFO_AFULL) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_OFIFO_FULL)) j=1;
        else break;
	copy_from_user((void *)wtmp, ((void *)PIC_BUFADDR)+i, j*4);
        for (k=0; k<j; k++) _WR(REG_FIFO, wtmp[k]);  
	i += (j*4);
      }
    }
    else if ( (PIC_OFFSET&0XFFFFF000) != 0) 	/* invalid register ? */
      status = EINVAL;
    else {
      copy_from_user((void *)wtmp, (void *)PIC_BUFADDR, 4);
      if (PIC_OFFSET==lastoffset+4) _WR(REG_PBAR+8,0);
      _WR(PIC_OFFSET,wtmp[0]);
      lastoffset=PIC_OFFSET;
    }
    break;

  case IOCTL_ACMD: 			/* command to SHARC processor */

    copy_from_user((void *)wtmp, (void *)PIC_BUFADDR, 20);
    for (k=4; k>=0; k--) _WR(REG_MBX+k*4,wtmp[k]);
    for (i=0,j=0; j<32; j++) {  /* cant wait forever in kernel mode */
      _RD (REG_MCSR,stat);
      if ( (stat&MCSR_IMB1F) != 0x0 ) {
        _RD (REG_IMB1, wtmp[0]);		
        copy_to_user((void *)PIC_BUFADDR, (void *)wtmp, 4);
	i=4; break;
      }
      if (j>4) udelay(100);
    }
    break;

  case IOCTL_MAP:			/* map user buffer to PCI */
    i=0;	/* assume failure */
    /* find an open dma handler slot */
    if (ndma==0 || ndma+1>=MAXIOBUF) status=EINVAL; 

    msize = phys2word(PIC_BYTES);
    /* Validate user buffer */
    if ( (msize <= 0)  || (msize > MAXWCNT) ) status = EINVAL;
    if (status!=ESUCCESS) break;
    /* map buffer into PCI bus space */
    status = icepic_mapio (ucb, IOCTL_MAP, &chan, msize);

    if (status == ESUCCESS) {
#if _USENEWDMA
      /*
       * Mark success (anything non-zero)
       */
      i = 1;
      /*
       * Copy the page frame number and bus addresses to user space
       */
      put_user (dma[chan].pgoff, &parg->pgoff);
      put_user (dma[chan].busaddr, &parg->busaddr);
#else
      i=word2phys(dma[chan].paddr); 
#endif
      dma[chan].device=index; 
    }
    break;
    
  case IOCTL_UNMAP: 			/* unmap user buffer from PCI */

#if _USENEWDMA

    page = PIC_PGOFF;
    msize = PIC_BYTES;

    /*
     * Turn msize into a number of pages
     */
    msize >>= PAGE_SHIFT;
    /* find the matching dma handler slot */
    for (chan=0; chan<MAXIOBUF; chan++) {
      if (page >= dma[chan].pgoff && 
	  page + msize <=
	  dma[chan].pgoff + (word2long(dma[chan].psize) >> PAGE_SHIFT)) break;
    }

    if (chan>=MAXIOBUF) { status = EINVAL; break; }

#else

    msize = phys2word(PIC_BYTES);
    maddr = phys2word(PIC_OFFSET);

    /* find the matching dma handler slot */
    for (chan=1; chan<ndma; chan++) {
      if (maddr >= dma[chan].paddr && 
	  maddr+msize <= dma[chan+1].paddr) break;
    }

    if (chan>=ndma) { status = EINVAL; break; }

#endif

    /* unmap user buffer from PCI bus space */
    status = icepic_mapio (ucb, IOCTL_UNMAP, &chan, 0);
    break;

  case IOCTL_QMAP: 			/* query user buffer map */

#if _USENEWDMA

    page = PIC_PGOFF;
    msize = PIC_BYTES;

    /*
     * Turn msize into a number of pages
     */
    msize >>= PAGE_SHIFT;

    /* is it in the table of driver mapped PCI memory */
    for (chan=0; chan<MAXIOBUF; chan++) {
      if (page >= dma[chan].pgoff && 
	  page + msize <=
	  dma[chan].pgoff + (word2long(dma[chan].psize) >> PAGE_SHIFT)) break;
    }
    if (chan==MAXIOBUF) i=0;
    else if (PIC_BYTES!=0) i=PIC_BYTES;
    else {
      /*
       * Mark success
       */
      i = 1;
      /*
       * Return the correct page number.
       */
      put_user (dma[chan].pgoff + (dma[chan].psize >> PAGE_WORD_SHIFT) - page,
		&parg->pgoff);
    }
    
#else
    msize = phys2word(PIC_BYTES);
    maddr = phys2word(PIC_OFFSET);

    /* is it in the table of driver mapped PCI memory */
    for (chan=0; chan<ndma; chan++) {
      if (maddr >= dma[chan].paddr && 
	  maddr+msize <= dma[chan].paddr+dma[chan].psize) break;
    }
    if (chan==ndma) i=0;
    else if (PIC_BYTES!=0) i=PIC_BYTES;
    else i=word2phys(dma[chan].paddr+dma[chan].psize-maddr);  
#endif
    break;

#if _USENEWDMA
  case IOCTL_GETDMA:
    page = PIC_PGOFF;
    msize = PIC_BYTES;

    /*
     * Turn msize into a number of pages
     */
    msize >>= PAGE_SHIFT;

    /* find the matching dma handler slot */
    for (chan=0; chan<MAXIOBUF; chan++) {
      if (page >= dma[chan].pgoff && 
	  page + msize <=
	  dma[chan].pgoff + (word2long(dma[chan].psize) >> PAGE_SHIFT)) break;
    }

    if (chan == MAXIOBUF)
      i = 0;
    else {
      /*
       * Mark success
       */
      i = 1;
      /*
       * Put bus address for DMA in bus address field
       */
      put_user (dma[chan].busaddr, &parg->busaddr);
    }
    break;

  case IOCTL_SYNCDMA:
    page = PIC_PGOFF;
    msize = PIC_BYTES;

    /* find the matching dma handler slot */
    for (chan=0; chan<MAXIOBUF; chan++) {
      if (page >= dma[chan].pgoff && 
	  page + msize <=
	  dma[chan].pgoff + (word2long(dma[chan].psize) >> PAGE_SHIFT)) break;
    }

    if (chan == MAXIOBUF)
      i = 0;
    else {
      /*
       * Mark success
       */
      i = 1;
      /*
       * Sync the DMA buffer
       */
      /* - in which kernel versions does this exist? */
      pci_dma_sync_single (ucb->pdev, dma[chan].busaddr,
		           msize, PCI_DMA_BIDIRECTIONAL);
/*		       */
    }
    break;

  case IOCTL_SHOWMAP:
    printk("ioctl showmap: ");
    icepic_printva (PIC_PGOFF);
    break;
    
#endif

  case IOCTL_RALLOC: 			/* allocate contiguous RAM */

/*
 * Has no effect using dynamic memory allocation.
 */
#if !_USENEWDMA
    icepic_setram ( (int_4)PIC_OFFSET, (int_4)(int)PIC_BUFADDR, (int_4)PIC_BYTES);
#endif
    break;

  case IOCTL_LIOC:                      /* load IOC words */

    for (i=0; i<PIC_BYTES; i+=4) {
      k = PIC_BYTES-i; if (k>4) k=4;
      copy_from_user((void *)(&worda), ((void *)PIC_BUFADDR)+i, k);
      reverse(&worda,4);
      for (j=0; j<(k*8); j++) {
        wordb = (worda>>j)&0x1;
        _WR(REG_IOC,wordb);
      }
    }
    break;

  case IOCTL_REATTACH:                      /* reattach test device */

    i = icepic_reattach (index);
    break;

  case IOCTL_VERSION:                      /* return version */

    *(int*)PIC_BUFADDR = ICE_DRIVER_VERSION;
    break;

  default: 				/* handle unknown function codes */
    status = EINVAL;
    break;
  }

  if (PIC_FUNCTION==IOCTL_MAP||PIC_FUNCTION==IOCTL_MAP) ICEPIC_UNLOCK;

  /* copy internal status to user space */
  copy_to_user((void *)&((PICIOCTL *)arg)->status, (void *)&i, 4);

  return(status);
}

void icepic_setram (int_4 start, int_4 sized, int_4 sizem) {
  /*
   * RAM start in pages
   */
  if (start!=-1) rstart = start * 0x40000; 	/* RAM start in words */
  if (sized!=-1) rsized = sized * 0x40000; 	/* RAMDISK size in words */
  if (sizem!=-1) rsizem = sizem * 0x40000; 	/* DYNRAM size in words */
  dma[0].paddr=rstart;
  dma[0].psize=rsized;
  dma[0].pid=0;

  dma[1].paddr=dma[0].paddr+rsized+rsizem;
  dma[1].psize=0;
  dma[1].pid=0;
  ndma=1;
#if _USENEWDMA
  /*
   * Not used
   */
  rstart = 0;
  rsized = 0;
  rsizem = 0;
#endif
  offset_min = word2long(rstart);
  offset_max = word2long(rstart+rsized+rsizem);
  wprintf ("icepic_setram: start at %ld, sized %d sizem %d\n",
	   offset_min, rsized, rsizem);
}


/************************************/
/* pic cancel routine               */
/************************************/
int icepic_cancel (struct inode *inode, struct file *filp, int reason)
{
  int reset,status,chan;
  int index = MINOR(inode->i_rdev);
  UCB *ucb = &ucbs[index];
  reset = MCSR_RESET;

  wprintf ("pic_cancel - dev %#x reason %x flags %x\n",index,reason,filp->f_flags);

  /* the mmap opens from icelib.c will have the O_NONBLOCK flag */
  /* these closes should not perform cleanup tasks */
  if (reason==1 && (filp->f_flags&O_NONBLOCK)==0) {	/* close */
    ICEPIC_LOCK;
#if _USENEWDMA
    for (chan=MAXIOBUF-1; chan>=0; chan--) {
#else
    for (chan=ndma-1; chan>=1; chan--) {
#endif
      if (dma[chan].gid == current->pgrp 
/*       && dma[chan].pid == current->pid */
       && dma[chan].device == index
       && dma[chan].paddr != 0) {
        vprintf ("pic_cancel - stray DMA - resetting card\n");
        /* unexpected termination - reset card to avoid stray DMAs */
	_WR (REG_MCSR, reset);
	/* deallocate/unmap stray buffer */
	status = icepic_mapio (ucb, IOCTL_UNMAP, &chan, 0);
      }
    }
    ICEPIC_UNLOCK;
  }
  return(ESUCCESS); 
}

/************************************/
/* pic mapio routine                */
/************************************/
int icepic_mapio (UCB *ucb, int function, int *channel, int_4 psize)
{
  int i,n,status,chan;
#if _USENEWDMA
  struct page *pg;
#endif

  status = ESUCCESS;	/* assume success */

  switch (function) {

  case IOCTL_MAP:
#if _USENEWDMA
    for (i = 0; i < MAXIOBUF; i++)
      if (!dma[i].kvaddr)
        goto ALLOCATE;
#else
    for (i=1; i<=ndma; i++) if (dma[i].paddr-
	(dma[i-1].paddr+dma[i-1].psize) >= psize) goto ALLOCATE;
#endif
    return ENOMEM; /* out of room */

    ALLOCATE:
    n=i; ndma++;
#if !_USENEWDMA
    for (i=ndma; i>n; i--) dma[i] = dma[i-1];
#endif

    dma[n].psize = psize;
#if _USENEWDMA
    /*
     * .psize is the size of the request in words.  Convert this to
     * an order size for the call to alloc_pages, so that we request
     * enough pages to cover the requested size.
     * Don't bother checking for MAX_ORDER because we have to handle the
     * allocation failing in either case.
     */
    int pages = dma[n].psize >> PAGE_WORD_SHIFT;
    dma[n].order = 0;
    while ((1UL << dma[n].order) < pages)
      dma[n].order++;
    /*
     * Specifying __GFP_DMA ensures the memory will be low enough to
     * DMA without using a bounce buffer since the ICE can only do
     * 32 bit addresses on the PCI bus.  On the IA32 platforms this
     * severely restricts the location of the buffer, so don't specify
     * this flag for IA32, but on IA64 platforms it just ensures the
     * memory can be accessed with a 32 bit address.
     */
#if __ia64__
    dma[n].kvaddr = __get_free_pages (GFP_USER | __GFP_DMA, dma[n].order);
#else
    dma[n].kvaddr = __get_free_pages (GFP_USER, dma[n].order);
#endif
    dma[n].pgoff  = __pa(dma[n].kvaddr) >> PAGE_SHIFT;
    /*
     * Need to mark the pages reserved so the remap_page_range
     * later will work.  If the pages aren't reserved, remap_page_range
     * in the mmap() call will leave the PTEs pointing to NULL pages, so
     * the first time you access the pages from user space you get a page
     * fault and the kernel gives you a random page somewhere rather than
     * the physical page we've requested.  Also, we need to fiddle with
     * the reference counts of the pages so things will end up right after
     * this memory is released.
     */
    /*
     * NOTE: Do things really get cleaned up afterwards so the memory
     * can be re-used?  Still need to check that.
     */
    pg = virt_to_page(__va(dma[n].pgoff << PAGE_SHIFT));
    for (i = (1UL << dma[n].order); i > 0; i--, pg++) {
      SetPageReserved (pg);
      set_page_count(pg, page_count(pg)+1);
    }

    if (!dma[n].kvaddr) {
      wprintf ("icepic: __get_free_pages failed\n");
      return ENOMEM;
    }
    /*
     * Don't know which direction this mapping will be used for,
     * so make it bidirectional.  If this requires bounce buffers
     * DMA on this system will be slow!
     */
    dma[n].busaddr = pci_map_single (ucb->pdev, (caddr_t) dma[n].kvaddr,
         		             word2long(dma[n].psize),
				     PCI_DMA_BIDIRECTIONAL);
#else
    dma[n].paddr = dma[n-1].paddr+dma[n-1].psize;
#endif
    dma[n].pid = current->pid;
    dma[n].gid = current->pgrp;
#if _USENEWDMA
    wprintf ("DMA map psize=%08x n=%d pgoff=0x%016lx ndma=%d busaddr=0x%016lx kvaddr=0x%016lx\n",
	     psize, n, dma[n].pgoff, ndma, dma[n].busaddr, dma[n].kvaddr);
#else
    wprintf("DMA map psize=%08x n=%d paddr=%08x ndma=%d\n",psize,n,dma[n].paddr,ndma);
#endif
    *channel = n;
    break;

  case IOCTL_UNMAP:
    chan = *channel;
    if (chan>=0) psize = dma[chan].psize;
    wprintf("DMA unmap psize=%08x %d\n",psize,ndma);
#if _USENEWDMA
    pci_unmap_single (ucb->pdev, dma[chan].busaddr, word2long(psize),
		        PCI_DMA_BIDIRECTIONAL);
    pg = virt_to_page(__va(dma[chan].pgoff << PAGE_SHIFT));
    for (i = (1UL << dma[chan].order); i > 0; i--, pg++) {
      ClearPageReserved (pg);
    }
    free_pages (dma[chan].kvaddr, dma[chan].order);
    dma[chan].busaddr = 0;
    dma[chan].order   = 0;
    dma[chan].kvaddr  = 0;
#else
    for (i=chan; i<ndma; i++) dma[i] = dma[i+1];
#endif
    ndma--;
    break;

  default:
    status = EINVAL;
  }
#if DEBUG
  for (i=0; i<=ndma; i++)
  wprintf("DMA table i=%d paddr=%08x psize=%08x\n",i,dma[i].paddr,dma[i].psize);
#endif
  return (status);
}

/************************************/
/* pic reattach routine             */
/************************************/
int icepic_reattach (int index)
{
  UCB *ucb = &ucbs[index];
  int status;
  ucb->attached = 0;
  if (!ucb->pdev) return 0; 
  status = pci_write_config_dword(ucb->pdev,0x10,ucb->membase);
  status = pci_write_config_dword(ucb->pdev,0x04,ucb->csr);
  status = pci_write_config_dword(ucb->pdev,0x0C,ucb->lat);
  vprintf ("pic_reattach - unit %d base %x csr %x\n",index,ucb->membase,ucb->csr);
  ucb->attached = 1;
  return (1); 
}

void reverse (void* pBuf, int pLen) {
#if _IEEE
  int i,j;
  char temp, *tPtr;
  tPtr = (char *)pBuf;
  j=pLen; pLen >>= 1;
  for (i=0; i<pLen; i++) {
    temp = tPtr[--j]; tPtr[j]=tPtr[i]; tPtr[i]=temp;
  }
#endif
}

/************************************/
/* pic mmap routine                 */
/************************************/
int icepic_mmap (struct file *filp, struct vm_area_struct *vma)
{
  unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
  unsigned long length = vma->vm_end - vma->vm_start;
#if _USENEWDMA
  /*
   * Iterate through the allocated regions to ensure the requested memory
   * is indeed ours; otherwise, this gives the ability to map any memory
   * on the system.
   */
#else
  if (offset<offset_min || offset+length>offset_max) return -EAGAIN;
#endif
  vma->vm_flags |= VM_RESERVED;
#if OLDVMA
  if (remap_page_range(vma->vm_start, offset, length, 
    vma->vm_page_prot)) return -EAGAIN;
#else
  if (remap_page_range(vma, vma->vm_start, offset, length,
    vma->vm_page_prot)) return -EAGAIN;
#endif
  return 0;
}
