/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  ICEPIC-8+ Processor Module Controller

  Joe Burns 
  Innovative Computer Engineering, Inc.
  1/15/2020

  Packet Contents

*/
`include "../lib/ramdefs.h"

module smodule (
  sclk,rst, scs,saddr, swr,swrbus, srd,srdbus, sack,
  ioclk, istat,iena,ibus,isel, ostat,oena,obus,osel,
  streams, test);			

  parameter PORT=1;
  parameter IBW=64;
  parameter NC=4;

  localparam DBW=64;
  localparam DBV=DBW/8;
  localparam IBV=IBW/8;

  localparam CBW=38;		// channel feed width {xena,xbus}
  localparam SBW=NC*CBW;	// stream feed width-1 {xclk,{4{xena,xbus}}}
  localparam NR=4;		// number of RAMS in output fifo

  // system bus
  input sclk,rst,scs,swr,srd;
  input [31:0] saddr,swrbus;
  output [31:0] srdbus;
  output sack;

  // DMA bus
  input ioclk,iena,oena;
  input [IBW-1:0] ibus;
  input [7:0] isel;
  output [IBW-1:0] obus;
  output [7:0] osel;
  output istat,ostat;

  // Stream Interface
  input [SBW:0] streams;

  // test port
  output [7:0] test;

  // global signals
  wire L=0, H=1;
  reg [31:0] pm,sysdbg,chnena;
  wire [31:0] sysch [NC-1:0]; 
  assign sack=H;

  wire good = H;
  wire ena  = pm[0];
  wire [3:0] ichn = pm[11:8];
  wire [3:0] ochn = pm[15:12];
  wire orst = pm[16];   // test reset
  wire irst = pm[17];   // test reset
  wire gmii = pm[28]; 
  wire mode = pm[29]; 
  wire cmode = pm[30]; 
  wire raw = pm[31];   // raw data collect

  reg enb;
  wire [NC-1:0] swrch;
  wire swriom = (scs && swr && saddr[23:20]==0);
  wire swrena = (scs && swr && saddr[23:20]==9);
  wire swrdbg = (scs && swr && saddr[23:20]==10);
  always @(posedge sclk) begin
    if (swriom) pm <= swrbus;
    if (swrdbg) sysdbg <= swrbus;
    if (swrena) chnena <= swrbus[NC-1:0];
    enb     <= ena && !rst;
  end
  wire frst = !enb;

  wire fclk = streams[SBW];
  wire wrdy;
  wire [NC-1:0] wstat_;
  wire [NC-1:0] wstat = wstat_ & chnena;
  reg  [NC-1:0] wena;
  wire [IBW-1:0] wbus [NC-1:0];
  wire [3:0]  wchn [NC-1:0];
  wire [31:0] xbus [NC-1:0];
  wire [NC-1:0] xval;
  genvar i;
  generate
  for (i=0; i<NC; i=i+1) begin	// each QSFP stream
    assign  swrch[i] = (scs && swr && saddr[23:20]==i+1);
    wire vrdy;
    wire vclk = streams[i*CBW+37];
    wire [3:0] vgmii = streams[i*CBW+35:i*CBW+32];
    wire vval = (vgmii!=15) | raw; //streams[i*CBW+36];
    wire [31:0] vbus = (gmii)? {vgmii,streams[i*CBW+27:i*CBW]} : (cmode)? {i[3:0],12'h0,i[3:0],12'h0} : streams[i*CBW+31:i*CBW];
    wire [3:0] vchn = i;
    packet_proc pkt_proc (sclk,rst, swrch[i],saddr[17:2],swrbus,sysch[i], vclk,frst,{vval,vgmii,vbus},{xval[i],xbus[i]});
    fifoNxM #(32,IBW) fx (vclk,frst, vrdy,xval[i],xbus[i], ioclk,frst, wstat_[i],wena[i],wbus[i]);
  end
  endgenerate

  reg [3:0] chst;
  reg [3:0] chst_ [1:0];
  reg [5:0] pkt_cnt;
  wire [IBW-1:0] wbusx;
  wire [3:0] wchnx;
  reg wvalx_;
  wire wvalx;
  always@(posedge ioclk) begin
    if(frst) begin
      chst <= 0;
      chst_[0] <= 0;
      chst_[1] <= 0;
      pkt_cnt <= 0;
    end
    else begin
      chst_[0] <= chst;
      chst_[1] <= chst_[0];
      wena <= (chst==1)? 8'h01 : (chst==2)? 8'h02 : (chst==3)? 8'h04 :
              (chst==4)? 8'h08 : (chst==5)? 8'h10 : (chst==6)? 8'h20 :
              (chst==7)? 8'h40 : (chst==8)? 8'h80 : 8'h00;
  
      wvalx_ <= |wena;
    
      case (chst) 
        4'd0: begin
          pkt_cnt <= 0;
          if(wstat[0]) chst <= 1; 
          else if(wstat[1]) chst <= 2; 
          else if(wstat[2]) chst <= 3; 
          else if(wstat[3]) chst <= 4; 
          else if(wstat[4]) chst <= 5; 
          else if(wstat[5]) chst <= 6; 
          else if(wstat[6]) chst <= 7; 
          else if(wstat[7]) chst <= 8; 
        end
        4'd1: begin
           pkt_cnt <= pkt_cnt + 1;
           if(pkt_cnt==63) begin
             if(wstat[1]) chst <= 2; 
             else if(wstat[2]) chst <= 3; 
             else if(wstat[3]) chst <= 4; 
             else if(wstat[4]) chst <= 5; 
             else if(wstat[5]) chst <= 6; 
             else if(wstat[6]) chst <= 7; 
             else if(wstat[7]) chst <= 8; 
             else chst <= 0;
           end
        end
        4'd2: begin
           pkt_cnt <= pkt_cnt + 1;
           if(pkt_cnt==63) begin
             if(wstat[2]) chst <= 3; 
             else if(wstat[3]) chst <= 4; 
             else if(wstat[4]) chst <= 5; 
             else if(wstat[5]) chst <= 6; 
             else if(wstat[6]) chst <= 7; 
             else if(wstat[7]) chst <= 8; 
             else if(wstat[0]) chst <= 1; 
             else chst <= 0;
           end
        end
        4'd3: begin
           pkt_cnt <= pkt_cnt + 1;
           if(pkt_cnt==63) begin
             if(wstat[3]) chst <= 4; 
             else if(wstat[4]) chst <= 5; 
             else if(wstat[5]) chst <= 6; 
             else if(wstat[6]) chst <= 7; 
             else if(wstat[7]) chst <= 8; 
             else if(wstat[0]) chst <= 1; 
             else if(wstat[1]) chst <= 2; 
             else chst <= 0;
           end
        end
        4'd4: begin
           pkt_cnt <= pkt_cnt + 1;
           if(pkt_cnt==63) begin
             if(wstat[4]) chst <= 5; 
             else if(wstat[5]) chst <= 6; 
             else if(wstat[6]) chst <= 7; 
             else if(wstat[7]) chst <= 8; 
             else if(wstat[0]) chst <= 1; 
             else if(wstat[1]) chst <= 2; 
             else if(wstat[2]) chst <= 3; 
             else chst <= 0;
           end
        end
        4'd5: begin
           pkt_cnt <= pkt_cnt + 1;
           if(pkt_cnt==63) begin
             if(wstat[5]) chst <= 6; 
             else if(wstat[6]) chst <= 7; 
             else if(wstat[7]) chst <= 8; 
             else if(wstat[0]) chst <= 1; 
             else if(wstat[1]) chst <= 2; 
             else if(wstat[2]) chst <= 3; 
             else if(wstat[3]) chst <= 4; 
             else chst <= 0;
           end
        end
        4'd6: begin
           pkt_cnt <= pkt_cnt + 1;
           if(pkt_cnt==63) begin
             if(wstat[6]) chst <= 7; 
             else if(wstat[7]) chst <= 8; 
             else if(wstat[0]) chst <= 1; 
             else if(wstat[1]) chst <= 2; 
             else if(wstat[2]) chst <= 3; 
             else if(wstat[3]) chst <= 4; 
             else if(wstat[4]) chst <= 5; 
             else chst <= 0;
           end
        end
        4'd7: begin
           pkt_cnt <= pkt_cnt + 1;
           if(pkt_cnt==63) begin
             if(wstat[7]) chst <= 8; 
             else if(wstat[0]) chst <= 1; 
             else if(wstat[1]) chst <= 2; 
             else if(wstat[2]) chst <= 3; 
             else if(wstat[3]) chst <= 4; 
             else if(wstat[4]) chst <= 5; 
             else if(wstat[5]) chst <= 6; 
             else chst <= 0;
           end
        end
        4'd8: begin
           pkt_cnt <= pkt_cnt + 1;
           if(pkt_cnt==63) begin
             if(wstat[0]) chst <= 1; 
             else if(wstat[1]) chst <= 2; 
             else if(wstat[2]) chst <= 3; 
             else if(wstat[3]) chst <= 4; 
             else if(wstat[4]) chst <= 5; 
             else if(wstat[5]) chst <= 6; 
             else if(wstat[6]) chst <= 7; 
             else chst <= 0;
           end
        end
      endcase
    end
  end

  assign wvalx = wvalx_; 
  assign wbusx = (mode)? wbus[ichn] : (chst_[1]==0)? 0 : wbus[chst_[1]-1];
generate
  if(PORT==1)  assign wchnx = (mode)? ochn : chst_[1][3:1];
  else if(PORT==2)  assign wchnx = (mode)? ochn : chst_[1][3:1]-1;
endgenerate
  assign srdbus = (saddr[23:20]==0)? pm :
                  (saddr[23:20]==1)? sysch[0] :
                  (saddr[23:20]==2)? sysch[1] :
                  (saddr[23:20]==3)? sysch[2] :
                  (saddr[23:20]==4)? sysch[3] :
                  (saddr[23:20]==5)? sysch[4] :
                  (saddr[23:20]==6)? sysch[5] :
                  (saddr[23:20]==7)? sysch[6] :
                  (saddr[23:20]==8)? sysch[7] :
                  (saddr[23:20]==9)? chnena :
                  (saddr[23:20]==10)? sysdbg : 32'hDEADBEEF ;

  fifoNxM #(IBW+IBV,IBW+IBV,-1,4,`BPAR,`BPAR) fo (ioclk,frst, wrdy,wvalx,{wchnx,wbusx}, ioclk,frst, ostat,oena,{osel,obus});

  reg ccc; always @(posedge fclk) ccc <= !ccc;
  assign test = {ccc,enb};

endmodule

module compareN(dina,dinb,eq);
  parameter N = 32;

  localparam M = N/4;

  input [N-1:0] dina,dinb;
  output eq;

  wire [M-1:0] cmp;
  genvar i;

  generate
   for(i=0;i<M;i=i+1) begin: cmploop
     assign cmp[i] = (dina[(i+1)*4-1:i*4]==dinb[(i+1)*4-1:i*4])? 1 : 0;
   end
  endgenerate

  assign eq = &cmp;

endmodule


module compare32(dina,dinb,eq);
  
  input [31:0] dina,dinb;
  output eq;

  wire [7:0] cmp;
  genvar i;

  generate
   for(i=0;i<8;i=i+1) begin: cmploop
     assign cmp[i] = (dina[(i+1)*4-1:i*4]==dinb[(i+1)*4-1:i*4])? 1 : 0;
   end
  endgenerate

  assign eq = &cmp;

endmodule

module packet_proc (sclk,rst,swr,saddr,swrbus,srdbus, fclk,srst,istream,ostream);
  parameter ND = 36;
  parameter MD = 32;

  localparam NR = 8; //Number of rams in circle buf (8*2k/32 = 512) enough to hold packet
  localparam HOLD = 32'hFFFF0000;
 
  //input [31:0] sysreg,usroff,usrsz,usrhdr;
  input sclk,rst,swr;
  input [3:0] saddr;
  input [31:0] swrbus;
  output [31:0] srdbus;

  input fclk,srst;
  input [ND:0] istream;
  output reg [MD:0] ostream;

  reg [3:0] st;
  reg [31:0] pkt_gcnt;
  reg [11:0] buf_adra,buf_adrb;
  reg [11:0] pkt_adr;
  reg [15:0] pkt_len;
  wire prdy,prdy1;
  reg pena;
  wire xstat;
  reg xena;
  wire [11:0] xadr;
  wire [15:0] xlen;
  wire [7:0] xprtcl;
  wire [31:0] xdstadr;
  reg sval;
  reg [3:0] scode;
  reg [31:0] sdat;
  reg [31:0] sdat_;
  reg [3:0] scode_;
  reg [1:0] sdat_sel;
  wire [31:0] sdat_b [3:0];
  reg [15:0] eth_lentype;
  reg [7:0] ip_prtcl;
  reg [31:0] ip_dstadr;
  reg [1:0] ip_offset;
  wire [31:0] cdat;
  reg cval;
  wire prtcleq,dsteq;

  reg [31:0] sysreg,usroff,usrsz,usrhdr,usrfu,usrdstadr,holdoff;
  reg [7:0] usrprtcl;
  always @(posedge sclk) begin
    if (swr & (saddr==0)) sysreg <= swrbus;
    if (swr & (saddr==1)) usroff <= swrbus;
    if (swr & (saddr==2)) usrsz  <= swrbus;
    if (swr & (saddr==3)) usrhdr <= swrbus;
    if (swr & (saddr==5)) usrprtcl <= swrbus[7:0];
    if (swr & (saddr==6)) usrdstadr <= swrbus;
    if (swr & (saddr==7)) holdoff <= swrbus;
    /* holdoff <= 32'h0fff0000; */
  end
  wire frst = srst|sysreg[0];
  wire byp = sysreg[1];
  wire sndhdr = (usrhdr!=0); //sysreg[2];
  wire endconv = sysreg[3];
  wire prtcl_en = (usrprtcl!=0); //sysreg[4];
  wire dstadr_en = (usrdstadr!=0); //sysreg[5];
  wire [1:0] dbg = sysreg[31:30];
  assign srdbus = (saddr==0)? sysreg : (saddr==1)? usroff : (saddr==2)? usrsz : (saddr==3)? usrhdr : 
                  (saddr==5)? usrprtcl : (saddr==6)? usrdstadr : 32'hDEADBEEF;

  assign sdat_b[0] = sdat_;
  assign sdat_b[1] = {sdat[7:0],sdat_[31:8]};
  assign sdat_b[2] = {sdat[15:0],sdat_[31:16]};
  assign sdat_b[3] = {sdat[23:0],sdat_[31:24]};
  always@(posedge fclk) begin
    if(frst) begin
      st <= 0;
      pkt_gcnt <= 0;
      buf_adra <= 12'd230;
      pkt_adr <= 0;
      pkt_len <= 0;
      pena <= 0;
      sval <= 0;
      sdat <= 0;
      scode <= 0;
      sdat_ <= 0;
      scode_ <= 0;
      sdat_sel <= 0;
      eth_lentype <= 0; 
      ip_prtcl <= 0;
      ip_offset <= 0;
      ip_dstadr <= 0;
    end
    else begin
      sval = istream[ND];
      scode <= istream[ND-1:ND-4];
      sdat <= istream[ND-5:0];
      sdat_ <= sdat;
      scode_ <= scode;
      case(st)
        4'd0: begin
          pena <= 0;
          pkt_len <= 0;

          if (pkt_gcnt < holdoff) pkt_gcnt <= pkt_gcnt+1;

          st <= ( (scode_==4'hF) && (pkt_gcnt==holdoff) )? 4'd4 : 4'd0;  //Make sure to catch start of packet...
          sdat_sel <= usroff[1:0];
        end
        4'd1: begin
          if(scode_==4'h0) begin
            //sdat is a valid packet word to be sent to the circle buffer
            pkt_adr <= buf_adra;
            buf_adra <= buf_adra+1;
            pkt_len <= pkt_len+4;
            st <= 4'd2;
          end
        end
        4'd2: begin
          if(scode_!=4'h0) begin
            st <= 4'd3;
            pkt_len <= (scode_==4'h8)? pkt_len-5 : (scode_==4'hC)? pkt_len-6 : (scode_==4'h7)? pkt_len-7 : pkt_len-8;
            buf_adra <= buf_adra+1;
          end
          else begin
            buf_adra <= buf_adra+1;
            pkt_len <= pkt_len+4;
          end

          //Process to identify packet type, first determine length/type field in the Ehternet header
          if(pkt_len == 16'h0010) begin
            //Look for type field >= 1536 which implies next level encapsulation
            eth_lentype <= ({sdat_[7:0],sdat_[15:8]} >= 16'd1536)? {sdat_[7:0],sdat_[15:8]}:0;
            ip_offset <= 0;
          end
          if(pkt_len == 16'h0014 && eth_lentype==16'h8100) begin
            //If type field is 0x8100, there is a 4Byte 802.1Q header
            eth_lentype <= {sdat_[7:0],sdat_[15:8]};
            ip_offset <= 1;
          end
          if(pkt_len == 16'h0018 && eth_lentype==16'h88A8) begin
            //If type field is 0x88A8, there is a 8Byte 802.1Q header
            eth_lentype <= {sdat_[7:0],sdat_[15:8]};
            ip_offset <= 2;
          end

         //Process to identify IP packet type, if the type field implies an IPv4 packet
         if(eth_lentype == 16'h0800) begin
           if(ip_offset==0) begin
             if(pkt_len==16'h0018) ip_prtcl <= sdat_[31:24];
             if(pkt_len==16'h0020) ip_dstadr <= {sdat_[23:16],sdat_[31:24],sdat[7:0],sdat[15:8]};
           end
           else if(ip_offset==1) begin
             if(pkt_len==16'h001C) ip_prtcl <= sdat_[31:24];
             if(pkt_len==16'h0024) ip_dstadr <= {sdat_[23:16],sdat_[31:24],sdat[7:0],sdat[15:8]};
           end
           else if(ip_offset==1) begin
             if(pkt_len==16'h0020) ip_prtcl <= sdat_[31:24];
             if(pkt_len==16'h0028) ip_dstadr <= {sdat_[23:16],sdat_[31:24],sdat[7:0],sdat[15:8]};
           end
         end

        end
        4'd3: begin
          //Write Packet Data to process FIFO...
          pkt_len <= pkt_len - usroff[1:0];
          pena <= (prtcl_en&dstadr_en)? prtcleq&dsteq : (prtcl_en)? prtcleq : (dstadr_en)? dsteq : 1;
          st <= 4'd0;
        end
        4'd4: begin
          //Verify valid Ethernet SOP so random junk doesn't go through...
          if((scode_==4'h1)&&(sdat_==32'h555555FB)) st <= 4'd1;
        end
      endcase
    end
  end

  compare32 cmpdstadr (usrdstadr,ip_dstadr,dsteq);
  compareN #(8) cmpprtcl (usrprtcl,ip_prtcl,prtcleq);

  ramNxN #(32,4096) cir_buf (fclk,1'b1,(scode_!=4'hF),buf_adra,sdat_b[sdat_sel],, fclk,1'b1,1'b0,buf_adrb,0,cdat);
  fifoNxM #(32,32,1,0,`NEMPTY,`NEMPTY) xp  (fclk,frst, prdy,pena,{pkt_len,pkt_adr}, fclk,frst, xstat,xena,{xlen,xadr});

  reg [3:0] xt;
  reg [15:0] pkt_xlen;
  reg [1:0] pkt_left;
  reg [13:0] pkt_xcnt;
  reg cval_;
  reg [31:0] cdat_;
  reg [2:0] cdat_sel,cdat_sel_;
  wire [31:0] cdat_b [7:0];
  reg [1:0] hdr_sel,hdr_sel_;
  reg [MD-1:0] hdr_endconv;
  reg [MD-1:0] dat_endconv;
  wire [31:0] usrhdr_ = (hdr_sel_==1)? usrhdr : usrfu;
  assign cdat_b[0] = cdat;
  assign cdat_b[1] = {cdat[23:0],cdat_[7:0]};
  assign cdat_b[2] = {cdat[15:0],cdat_[15:0]};
  assign cdat_b[3] = {cdat[7:0],cdat_[23:0]};
  assign cdat_b[4] = cdat;
  assign cdat_b[5] = {cdat[23:0],cdat_[31:24]};
  assign cdat_b[6] = {cdat[15:0],cdat_[31:16]};
  assign cdat_b[7] = {cdat[7:0],cdat_[31:8]};

  always@(posedge fclk) begin
    if(frst) begin
      xt <= 0;
      xena <= 0;
      pkt_xlen <= 0;
      pkt_left <= 0;
      pkt_xcnt <= 0;
      buf_adrb <= 0;
      cval <= 0;
      cval_ <= 0;
      cdat_ <= 0;
      cdat_sel <= 0;
      cdat_sel_ <= 0;
      hdr_sel <= 0;
      hdr_sel_ <= 0;
      hdr_endconv <= 0;
      dat_endconv <= 0;
    end
    else begin
      cval_ <= cval;
      cdat_ <= cdat;
      cdat_sel_ <= cdat_sel;
      hdr_sel_ <= hdr_sel;
      hdr_endconv <= (endconv)? {usrhdr_[23:16],usrhdr_[31:24],usrhdr_[7:0],usrhdr_[15:8]} : usrhdr_;
      dat_endconv <= (endconv)? {cdat_b[cdat_sel_][23:16],cdat_b[cdat_sel_][31:24],cdat_b[cdat_sel_][7:0],cdat_b[cdat_sel_][15:8]} : cdat_b[cdat_sel_];
      ostream <= (byp)? {sval,sdat} : {cval_,(hdr_sel_)? hdr_endconv : dat_endconv };
/*
      ostream <= (byp)? {sval,sdat} : {cval_,(hdr_sel_)? (endconv)? {usrhdr_[23:16],usrhdr_[31:24],usrhdr_[7:0],usrhdr_[15:8]} : usrhdr_ : 
                                                         (endconv)? {cdat_b[cdat_sel_][23:16],cdat_b[cdat_sel_][31:24],cdat_b[cdat_sel_][7:0],cdat_b[cdat_sel_][15:8]} : cdat_b[cdat_sel_]};
*/
      case(xt) 
        4'd0: begin
          pkt_xcnt <= 0;
          cval <= 0;
          if(xstat) begin
            xena <= 1;
            xt <= 4'd1;
          end
        end
        4'd1: begin
          xena <= 0;
          pkt_xlen <= (usrsz>0)? usrsz+((sndhdr)?4:0) : xlen-(usroff&16'hFFFC)+((sndhdr)?4:0)+pkt_left;
          buf_adrb <= xadr+usroff[15:2]+((sndhdr==0)?1:0);
          hdr_sel <= sndhdr;
          xt <= (usrsz<=xlen)? (sndhdr)? 4'd4 : 4'd2 : 4'd3;
        end
        4'd2: begin
          hdr_sel <= 0;
          if(pkt_xcnt < pkt_xlen[15:2]) begin
            cval <= 1;
            cdat_sel[2] <= 1;
            buf_adrb <= buf_adrb+1;
            pkt_xcnt <= pkt_xcnt+1;
          end
          else if(pkt_xlen[1:0] == 2'd0) begin 
            //Packet was 32 bit aligned, no leftover
            pkt_left <= 0;
            cval <= 0;
            cdat_sel <= 3'd0;
            xt <= 0;
          end
          else if(pkt_xlen[1:0] == 2'd1) begin
            pkt_left <= 1;
            cval <= 0;
            cdat_sel <= 3'd1;
            xt <= 0;
          end
          else if(pkt_xlen[1:0] == 2'd2) begin
            pkt_left <= 2;
            cval <= 0;
            cdat_sel <= 3'd2;
            xt <= 0;
          end
          else if(pkt_xlen[1:0] == 2'd3) begin
            pkt_left <= 3;
            cval <= 0;
            cdat_sel <= 3'd3;
            xt <= 0;
          end
        end
        4'd3: begin
          if(pkt_xcnt < 5) pkt_xcnt <= pkt_xcnt+1;
          else xt <= 4'd0;
        end
        4'd4: begin
          cval <= 1;
          usrfu[15:0] <= pkt_xlen+4;
          hdr_sel <= 2'd2;
          xt <= 4'd2;
        end
      endcase  
    end
  end
endmodule

module ramNxN (clka,ena,wea,addra,dia,doa, clkb,enb,web,addrb,dib,dob);

  parameter WIDTH      = 8;
  parameter SIZE       = 256;

  function integer log2;
    input integer value;
    reg [31:0] shifted;
    integer res;
  begin
    if (value < 2)
      log2 = value;
    else begin
      shifted = value - 1;
      for (res=0; shifted>0; res=res+1)
        shifted = shifted >> 1;
      log2 = res;
    end
  end
  endfunction
 
  localparam ADDRWIDTH = log2(SIZE);
 
  input clka, clkb;
  input ena,wea, enb,web;
  input [ADDRWIDTH-1:0] addra;
  input [ADDRWIDTH-1:0] addrb;
  input [WIDTH-1:0] dia;
  input [WIDTH-1:0] dib;
  output reg [WIDTH-1:0] doa;
  output reg [WIDTH-1:0] dob;

  reg [WIDTH-1:0] RAM [0:SIZE-1];

  always@(posedge clka) begin
    if(ena) begin
      if(wea) RAM[addra] <= dia;
      doa <= RAM[addra];
    end
  end

  always@(posedge clkb) begin
    if(enb) begin
      if(web) RAM[addrb] <= dib;
      dob <= RAM[addrb];
    end
  end

endmodule

module pipedelay (clk,rst, in,out);
  parameter DLY=1;
  parameter N=1;
  
  input clk,rst;
  input [N-1:0] in;
  output [N-1:0] out;
  
  reg [N-1:0] x [DLY-1:0];
  
  genvar i;
  generate
  if(DLY==0) begin
    assign out = in;
  end
  else begin
    for(i=0;i<DLY;i=i+1) begin: genloop
      always@(posedge clk) begin
        if(rst) x[i] <= 0;
        else if(i==0) x[i] <= in;
        else x[i] <= x[i-1];
      end
    end
    assign out = x[DLY-1];
  end 
  endgenerate
    
endmodule

`ifdef BOGUS
module icescope (sclk,rst,wr,adr,wdat,rdat, mclk,scope_en,scope_dat);
  parameter MW = 64;
  parameter MD = 512;

  input sclk,rst,wr;
  input [15:0] adr;
  input [31:0] wdat;
  output [31:0] rdat;

  input mclk;
  input scope_en;
  input [MW-1:0] scope_dat;

  reg [31:0] sysreg;
  reg go_,trig,trig_;
  reg [16:0] maddr;
  wire [MW-1:0] memrdat;
  wire [MW-1:0] scopedat_;

  function integer log2;
    input integer value;
    reg [31:0] shifted;
    integer res;
  begin
    if (value < 2)
      log2 = value;
    else begin
      shifted = value - 1;
      for (res=0; shifted>0; res=res+1)
        shifted = shifted >> 1;
      log2 = res;
    end
  end
  endfunction
  
  wire go     = sysreg[0];
  wire memrst = sysreg[1];
  wire trigt  = sysreg[2];
  wire start  = go & ~go_;
  wire scope_trig = (trigt)? trig_ : trig;

  always@(posedge sclk) begin
    if(rst) sysreg <= 0;
    else if(wr) sysreg <= wdat;
    go_ <= go;
    trig <= scope_en;
    trig_ <= (scope_en & !trig) | (trig_ & !memrst);
  end

  always@(posedge mclk) begin
    if(rst|memrst) maddr <= 0;
    else if(scope_trig & maddr<MD-1) maddr <= maddr+1;
  end

  pipedelay #(32,MW) dlyp (mclk,rst|rstmem,scope_dat,scopedat_);

generate
  if(MW>32) begin
    ramNxN #(MW,MD) ram_inst (mclk,1'b1,scope_trig,maddr,scopedat_,, sclk,1'b1,1'b0,adr[15:1],0,memrdat);
    //ramNxNo #(MW,MD,log2(MD)) ram_inst (mclk,1'b1,scope_trig,maddr,scopedat_,, sclk,1'b1,1'b0,adr[15:1],0,memrdat);
    assign rdat  = (adr[0])? memrdat[MD-1:32] : memrdat[31:0] ;
  end
  else begin
    ramNxN #(MW,MD) ram_inst (mclk,1'b1,scope_trig,maddr,scopedat_,, sclk,1'b1,1'b0,adr[15:0],0,memrdat);
    //ramNxNo #(MW,MD,log2(MD)) ram_inst (mclk,1'b1,scope_trig,maddr,scopedat_,, sclk,1'b1,1'b0,adr[15:0],0,memrdat);
    assign rdat  = (adr[16])? maddr : memrdat[31:0] ;
  end
endgenerate

endmodule
`endif

