/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  ICE HyperLink Input/Output Controllers

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  1/15/2003

  IO=SS

  Input: 210Mby each channel
   iclkp = input clock
   iclkn = A/B valid
   idatp = A/B  7:0
   idatn = A/B	15:8

  Output:	100Mby each channel
   oclkp = A valid
   oclkn = B valid
   odatp = A  7:0
   odatn = B  7:0

  IO=HH

  Input: 800Mby each channel
   iclkp = input clock
   iclkn = input clock
   idatp = A/B  7:0
   idatn = A/B	15:8

  Output: 800Mby each channel
   oclkp = output clock
   oclkn = output clock
   odatp = A/B  7:0
   odatn = B/B  15:8

  IO=RR

  Input: 4Gby each 4-lane channel
  Output: 4Gby each 4-lane channel
  ICEt transport guarantees 8*N byte align

  Packet Header: 0xccssmmA5
    A5 = start symbol
    mm = command flags [3:0]=ohold
    ss = size in 64by chunks
    cc = PM channel number or IOM timecode

  Test Pattern: 

    FF:00 EE:11 DD:22 CC:33   BB:44 AA:55 99:66 88:77   77:88 66:99 55:AA 44:BB   33:CC 22:DD 11:EE 00:FF

    Bit#0 0110011001100110 0110011001100110
    Bit#2 0101101001011010 0101101001011010
    Bit#3 0101010110101010 0101010110101010
    Bit#4 0101010101010101 1010101010101010

*/
`include "mdefs.h"
`include "dmadefs.h"
`include "ramdefs.h"
`include "hypdefs.h"

// we swapped P/N on bit 2 for layout reasons
`ifdef V6M
`define IFLP 8'h00 
`define OFLP 8'h04
`else
`define IFLP 8'h00 
`define OFLP 8'h00 
`endif

`ifdef S6M
`define HYP_PIPE 1
`define HYP_IDLY 1
`else
`define HYP_PIPE 0
`define HYP_IDLY 0
`endif

`ifndef HYP_PKT
`define HYP_PKT `DMA_PKT_LEN
`endif

module hyperlink ();
endmodule

module hypin (
`ifdef HYP_RIO
  iclk,irst,iena,ibus,
  cena,cbus,
`else
  iclk,iclks, idatp,idatn,
`endif
  ioclk,rio, ostatha,oenaha,oselha,obusha,  ostathb,oenahb,oselhb,obushb, ostathc,oenahc,
  mode,calib, rios,rdys,hold, test);

  parameter IBW=32;
  parameter MODE=5;
  parameter TUNERS=1;
  parameter FLGA=0;
  parameter FLGB=(MODE==4)? `CINV:0;

  localparam DBW=(MODE>7)? 128:32;
  localparam NR=(MODE>7)? 16:2;
  localparam ND=(MODE>7)? 3:5;
  localparam IDLY=`HYP_IDLY;
  localparam FLGO=(`HYP_PIPE!=0)? `PACKET|`OREG : `PACKET;
  localparam IBV=IBW/8;
  localparam DBV=DBW/8;
  localparam NPP=`HYP_PKT;

`ifdef HYP_RIO
  input iclk, irst, iena;
  input [DBW-1:0] ibus;
  output cena; // command bus
  output [35:0] cbus;
`else
  input iclk;
  input [3:0] iclks;
  input [7:0] idatp,idatn;
`endif

  input ioclk,rio,oenaha,oenahb,oenahc;
  output ostatha,ostathb,ostathc;
  output [7:0] oselha,oselhb;
  output [IBW-1:0] obusha,obushb;

  input mode;
  output [31:0] calib;
  input  [2:0] rios;
  output [2:0] rdys;
  output [3:0] hold;
  output [7:0] test;

  wire L=0, H=1;

  wire rha = rios[1];
  wire rhb = rios[2];

  wire hival,hcsm,hcab;
  wire [DBW-1:0] hidat; 
  wire [DBV-1:0] hisel; 
  wire [7:0] vtest;

`ifdef HYP_RIO
  hypinr #(DBW,DBW,NPP,`HYP_ONOF) qhi (iclk,H, iena,ibus, hival,hisel,hidat, cena,cbus, mode,hold,hcsm,hcab,calib,vtest);
  wire [2:0] ivals = !hival? 0 : hcsm? 1 : hcab? 4 : 2;
  mcfifoNxM #(DBW+DBV,IBW+IBV,NR,ND,`BPAR,`PACKET) fab (iclk,rios,rdys,ivals,{hisel,hidat}, 
							ioclk,rios,{ostathb,ostatha,ostathc},{oenahb,oenaha,oenahc},{oselha,obusha});
  assign oselhb=oselha; assign obushb=obusha;
`else
  wire [31:0] vdat;
  wire vclk=iclk, pclk=iclks[0], prst=iclks[1], ss=iclks[2];
  diqdrN #(8,1,`IFLP,IDLY) qdat (idatp,idatn, vclk,pclk,prst,ss, vdat[7:0],vdat[15:8],vdat[23:16],vdat[31:24]);
  hypinq #(DBW,DBW,1,`HYP_ONOF) qhi (vclk,H,vdat, hival,hisel,hidat, mode,ohold,hcsm,hcab,calib,vtest);
  fifoNxM #(DBW+DBV,IBW+IBV,NR,ND,`BPAR|FLGA,FLGO) fa (iclk,rha,rdys[1],hival&!hcab,{hisel,hidat},  ioclk,rha,ostatha,oenaha,{oselha,obusha});
  fifoNxM #(DBW+DBV,IBW+IBV,NR,ND,`BPAR|FLGB,FLGO) fb (iclk,rhb,rdys[2],hival& hcab,{hisel,hidat},  ioclk,rhb,ostathb,oenahb,{oselhb,obushb});
`endif
  assign test = {hold,rdys,hcab,hival};

endmodule

module hypout (
`ifdef HYP_RIO
  oclk,orst,ogood, oena,obus,
  crdy,cena,cbus,
`else
  oclk,oclks, odatp,odatn,
`endif
  ioclk,rio, istatha,ienaha,iselha,ibusha, istathb,ienahb,iselhb,ibushb, istat1,istat2, iselhc, 
  speed, ihold_,ohold_, test);

  parameter IBW=32;		// IO bus width in bits
  parameter MODE=5;

  localparam DBW=(MODE>7)? 128:32;
  localparam NR=(MODE>7)? 16:4;
  localparam ND=(MODE>7)? 4:3;
  localparam IBV=IBW/8;
  localparam DBV=DBW/8;
  localparam NW=(IBW/DBW);	// IO bus width in words
  localparam NPP=`HYP_PKT;

  localparam NKR=256;		// Corner Turn RAM size in KBy

`ifdef HYP_RIO
  input oclk,orst,ogood;
  output oena;
  output [DBW-1:0] obus;
  input  crdy;
  output cena;
  input [35:0] cbus;
`else
  input oclk;
  input [3:0] oclks;
  output [7:0] odatp,odatn;
`endif
  input ioclk,rio,ienaha,ienahb;
  output istatha,istathb;
  output istat1,istat2;
  input [15:0] iselhc;
  input [7:0] iselha,iselhb;
  input [IBW-1:0] ibusha,ibushb;

  input [7:0] speed;		// speed control 200/(N+2) link port
  input [3:0] ihold_,ohold_;
  output [7:0] test;

  wire L=0, H=1;

  reg ivalhx; always @(posedge ioclk) ivalhx <= (ienaha || ienahb);
  reg iselx;  always @(posedge ioclk) iselx  <=  !iselx && ivalhx;

`ifdef HYP_RIO
  wire [IBV-1:0] iselhx = {2{iselhc}};
`else
  wire [IBV-1:0] iselhx = (IBW>32)? {(IBW/64){iselha[7:0]}} : iselx? iselha[IBV+IBV-1:IBV] : iselha[IBV-1:0];
`endif
  wire [DBV-1:0] hosel;
  wire [2:0] hordys,istats;
  wire [7:0] vtest;
  wire hoena,hordy;
  wire [DBW-1:0] hodat,vdat; // note that ibusha/iselha are the same as ibushb/iselhb

  reg [3:0] ohold,oholdz; always @(posedge oclk) begin ohold <= oholdz; oholdz <= ohold_; end
  reg [3:0] ihold,iholdz; always @(posedge oclk) begin ihold <= iholdz; iholdz <= ihold_; end

`ifdef S6M
  assign hordy = (hordys[0] && !ohold[0]);	// no multichannel fifo since there are no COREs, only MCORES on S6M
  fifoNxM #(IBW+IBV,DBW+DBV,2,3,`BPAR,`PACKET) fab (ioclk,rio,istats[0],ivalhx,{iselhx,ibusha}, oclk,rio,hordys[0],hoena,{hosel,hodat});
  assign istatha=istats[0], istathb=istats[0], istat1=istats[1], istat2=istats[2]; 
`else

`ifdef MCF_CTBE
  // Corner Turn Back End
  wire [7:0] opt = speed;	// in from MCORE test outputs to control pkt8be
  wire ctena = (opt[7:6]==3) || (opt[3:2]==3);
  wire ctdual = (opt[7:6]==3) && (opt[3:2]==3);
  wire [1:0] ctsize = (opt[7:6]==3)? opt[5:4] : opt[1:0];
  wire [IBW-1:0] obusc; wire [IBV-1:0] oselc;
  wire ostatc;
  wire ovalc = ctena && ostatc && istats[0] && !(ienaha && !iselha[7]) && !(ienahb && !iselha[7]);
  reg ivalc; always @(posedge ioclk) ivalc <= ctena && (ienaha || ienahb) && iselha[7];
  fifoNxMpkt8be #(IBW+IBV,IBW+IBV,NKR/2,ND,`BPAR,0) vwf (ioclk,rio,istatc,ivalc,{iselhx,ibusha}, ioclk,rio,ostatc,ovalc,{oselc,obusc}, ctena,ctdual,ctsize);
  reg [2:0] ivals; always @(posedge ioclk) ivals <= {ienahb && !iselha[7], ienaha && !iselha[7], ctena? ovalc : (ienaha || ienahb) && iselha[7]};
  assign istatha=ctena?istatc:istats[0], istathb=ctena?istatc:istats[0], istat1=istats[1], istat2=istats[2]; 
  reg ctbe; always @(posedge ioclk) ctbe <= ovalc;
`else
  reg [2:0] ivals; always @(posedge ioclk) ivals <= {ienahb && !iselha[7], ienaha && !iselha[7], (ienaha || ienahb) && iselha[7]};
  assign istatha=istats[0], istathb=istats[0], istat1=istats[1], istat2=istats[2]; 
  wire ctbe=L; wire [7:0] oselc=0; wire [31:0] obusc=0;
`endif

  reg [1:0] hochn; always @(posedge oclk) hochn <= (hordys[2]&&!ohold[2])? 2 : (hordys[1]&&!ohold[1])? 1 : (hordys[0]&&!ohold[0])? 0 : 3;
  reg [1:0] hocho; always @(posedge oclk) if (!hoena && hochn!=3) hocho <= hochn;
  wire [2:0] hoenas = { hoena&&(hocho==2), hoena&&(hocho==1), hoena&&(hocho==0) };
  wire [2:0] rios = {rio,rio,rio};
  assign hordy = (hochn!=3);
  mcfifoNxM #(IBW+IBV,DBW+DBV,NR,ND,`BPAR,(NPP>`DMA_PKT_LEN)?0:`PACKET) fab (ioclk,rios,istats,ivals,ctbe?{oselc,obusc}:{iselhx,ibusha}, oclk,rios,hordys,hoenas,{hosel,hodat});
`endif

`ifdef HYP_RIO
  wire aohold = (ohold!=0);	// any hold
  hypoutr #(DBW,DBW,NPP,`HYP_ONOF) qho (oclk,ogood, oena,obus, hordy,hoena,hosel,hodat, crdy,cena,cbus, {aohold,ihold},vtest);
  assign test = {hordys,ohold[0],istats[0],hoenas[0],ivals[0]};
`else
  wire vclk=oclk, pclk=oclks[0], prst=oclks[1], ss=oclks[2];
  hypoutq #(DBW,DBW,NPP,`HYP_ONOF) qho (vclk,H,vdat, hordy,hoena,hosel,hodat, ihold,vtest);
  doqdrN #(8,1,`OFLP) qdat (odatp,odatn, vclk,pclk,prst,ss,L, vdat[7:0], vdat[15:8], vdat[23:16], vdat[31:24]);
  assign test = {hordy,hoena,ohold[0]};
`endif

endmodule



/*
                   idat jdat kdat
                   A5
  isyna            00   A5
  isync            01   00   A5
4 icycle           02   01   00
3 icycle           03   02   01
2 icycle           A5   03   02
1 icycle   idone   00   A5   03
  isync            01        A5

  iena - idat valid - qualify isyna,isynb
  jena - jdat valid - qualify isync,icycle
  kena - kdat valid - qualify output

  modes  0=calib 1=fsync 2=seqf
*/
module hypinq (iclk,iena,idat, hval,hsel,hdat, modes,hold,csm,cab,calib,test);
  parameter IBW=32;
  parameter OBW=32;
  parameter NPKT=1;
  parameter FLG=0;

  localparam OBV=OBW/8;
  localparam IBV=IBW/8;

  input iclk,iena;
  input [IBW-1:0] idat;
  output hval;
  output [OBV-1:0] hsel;
  output [OBW-1:0] hdat;
  input  [2:0] modes;
  output [3:0] hold;
  output csm,cab;
  output [7:0] calib;
  output [7:0] test;

  wire L=0, H=1;
  wire mode  = modes[0];
  wire fsync = modes[1];
  wire seqf  = modes[2] && !modes[0];

  // packet logic
  reg syt0,syt1,syt0d,syt1d,oop,active;
  reg isyna,isynb,isync,icycle,idone,ipha;
  reg [31:0] jdat,kdat;
  reg [7:0] icnt,ichan;
  reg [3:0] hold,iseq,nseq,dseq;
  reg sen,ten,chm,chb;
  wire [3:0] ipkt = kdat[19:16];	// test valid input widths - else default
  wire [3:0] npkt = seqf? NPKT : (NPKT==0||ipkt==1||ipkt==2||ipkt==4)? ipkt : NPKT;
  wire [7:0] vchan = kdat[31:24];
  wire vchb;  vote3 v1 (vchb, vchan[4],vchan[2],vchan[0]);
  wire vsen;  vote3 v2 (vsen, vchan[5],vchan[3],vchan[1]);
  wire vten;  vote3 v3 (vten, vchan[7],vchan[6],1'b0);
  wire vchm;  vote3 v4 (vchm, vchan[7],vchan[6],1'b1);
  wire voteh; vote3 vh (voteh, vchan[7],vchan[6],vchan[5]);
  wire votel; vote3 vl (votel, vchan[2],vchan[1],vchan[0]);
  wire ihdr = isync;
  wire icyc = isync && (kdat[18:16]!=0);
  always @(posedge iclk) begin
    syt0 <= (jdat[ 7: 0]==8'hAA); syt0d <= syt0 && !(isync || active);
    syt1 <= (jdat[23:16]==8'hAA); syt1d <= syt1 && !(isync || active);
    oop  <= syt1d? H : syt0d? L : oop;
    jdat <= idat;
    kdat <= oop? {idat[15:0],jdat[31:16]} : jdat;
    isyna  <= (idat[7:0]==8'hA5);
    isynb  <= (idat[23:16]==8'hA5);
    isync  <= !isync && !(icycle && !idone) && (oop? isynb : isyna);
    icycle <= (iena && icyc) || (icycle && !idone) || mode;
    idone  <= icycle && (icnt[5:0]==2);
    if (isync) icnt <= {npkt,4'h0}; else if (icycle) icnt <= icnt-1; // 16x4by pkt count for NPKT 
    if (isync) ipha <= 0; else ipha <= !ipha;
    if (mode) chb <= 0; else if (ihdr) chb <= vchb;
    if (mode) chm <= 0; else if (ihdr) chm <= vchm;
    if (mode) sen <= 0; else if (ihdr) sen <= vsen;
    if (mode) ten <= 0; else if (ihdr) ten <= vten;
    if (mode) hold <= 0; else if (ihdr) hold <= kdat[11:8];
    if (mode) ichan <= 0; else if (ihdr) ichan <= vchm? {H,L,L,vchan[4:0]} : {L,{6{L}},votel};
    if (!iena) iseq <= 0; else if (icyc) iseq <= kdat[15:12];
    if (!iena) nseq <= 0; else if (idone) nseq <= nseq+1;
    if (!iena) dseq <= 0; else if (idone) dseq <= iseq-nseq;
  end

`ifdef ICE_MB
  wire [OBV-1:0] ksel = ipha? ichan[7:4] : ichan[3:0];	// need full input channel address
  assign cab  = ichan[0];
`else
  wire [OBV-1:0] ksel = {4{chm,L,L,L,ten,sen,ten,sen}};	// only A/B channel number and startup synchronization signals are important
  assign cab  = chb;
`endif

  reg [OBV-1:0] xdat;
  wire jcycle = isync || (icycle && !idone);
 generate
 if (FLG&`HYP_TC) begin:tc
  reg xsync,tcval,tcbit,tcbar,tcuse;
  reg [7:0] tcoff;
  always @(posedge iclk) begin
    if (isync) tcoff <= kdat[31:24];
    xsync <= (isync && kdat[11:8]==4'h1);
    tcval <= xsync || (tcval && (icnt[3:0]!=0));
    tcbit <= tcoff[ ~icnt[2:0] ];
    if (!tcval) tcbar <= 0; else tcbar <= ~icnt[2];	// 4 cyc start pulse
    if (!tcval) tcuse <= 0; else tcuse <= ~icnt[3];	// 8 cyc binary offset count
    xdat[0] <= tcuse? tcbit:tcbar;
    if (jcycle) xdat[1] <= fsync;
  end
 end
 else begin:ntc
  always @(posedge iclk) xdat <= 0;
 end
 endgenerate

  assign hval = icycle;
  assign hsel = (FLG&`HYP_SEQ)? iseq : ksel;
  assign hdat = {xdat,kdat};
  assign csm  = chm;

  // calibration check
  reg err,estr,eend,npkterr;
  reg [3:0] tst,errc;
  reg [7:0] tcnt,errs,terrs;
  wire [3:0] tst0 = {tst[3:1],L};
  wire [3:0] tst1 = {tst[3:1],H};
  always @(posedge iclk) begin
    if (isync|icycle) tcnt <= 0; else tcnt <= tcnt+1;
    estr <= (tcnt==5);
    eend <= (tcnt==255);
    if (estr) tst <= {jdat[19:17],L}; else tst <= tst+2;
    if (estr) errs <= 0; else if (err) errs <= errs+1;
    // err <= {~tst1,~tst1,tst1,tst1,~tst0,~tst0,tst0,tst0} != jdat;
    errc[0] <= kdat[ 7: 0] != { tst0, tst0};
    errc[1] <= kdat[15: 8] != {~tst0,~tst0};
    errc[2] <= kdat[23:16] != { tst1, tst1};
    errc[3] <= kdat[31:24] != {~tst1,~tst1};
    active  <= isync || icycle || (active && !tcnt[3]);
    if (active) err <= 0; else err <= (errc!=0);
    if (eend) terrs <= errs;
    if (ihdr) npkterr <= (kdat[19:16] != `HYP_PKT);
  end
  assign calib = seqf? dseq : terrs;

//  assign test = {ten,sen,chm,cab, err,icycle,npkterr,iena};
  assign test = {iseq, err,icycle,npkterr,iena};

endmodule

/*
  ostart
  ostartd  ocyclep  hena
4 ocycle   ocyclep  hsel              hdat 
3 ocycle   ocyclep  hseld  oheader    tctick
2 ocycle   ocyclep  odone
1 ocycle            ostart

*/

module hypoutq (oclk,oenas,odat, hrdy,hena,hsel,hdat, hold,test);
  parameter IBW=32;
  parameter OBW=32;
  parameter NPKT=1;
  parameter FLG=0;	// 1=OnOffProtocol 2=TimeCode

  localparam IBV=IBW/8;
  localparam OBV=OBW/8;

  input oclk;
  input [2:0] oenas;
  output [OBW-1:0] odat;

  input hrdy;
  output hena;
  input [IBV-1:0] hsel;
  input [IBW-1:0] hdat;

  input [3:0] hold;
  output [7:0] test;

  wire L=0, H=1;

  wire oena = oenas[0];		// ok to output
  wire zfill = oenas[1];	// zero fill comma
  wire noseq = oenas[2] || !oena;	// disable sequence 

  // output
  reg ostart,ocyclep,ocycle,ocycled,odone,ordyd,oenad,ahold,tctick,tcval;
  reg [7:0] ocnt,tcoff;
  reg [3:0] tst,hseld,ohold,lhold,jhold,tdatl,oseq;
  reg [OBW-1:0] obusa,obusb,obusc;
  wire [3:0] tst0 = {tst[3:1],L};
  wire [3:0] tst1 = {tst[3:1],zfill?L:H};
  wire [31:0] tsto = {~tst1,~tst1, tst1,tst1, ~tst0,~tst0, tst0,tst0};
  wire [3:0] npkt = NPKT;
  wire [3:0] tdat = hdat[IBW-1:IBW-4];
  wire oheader; delaypipe #(3,1) dp (oclk,ostart,oheader);
  wire [7:0] hf1 = (FLG&`HYP_TC)? {oseq,3'd0,tcval} : {oseq,jhold};
  wire [7:0] hf3 = (FLG&`HYP_TC)? tcoff             : {hsel,hseld};
  wire uahold    = (FLG&`HYP_ONOF)? ahold : L;
  wire ordy      = hrdy;
  always @(posedge oclk) begin
    ordyd   <= ordy;
    ostart  <= oena && !ostart && !(ocyclep && !odone) && (ordy || uahold);
    if (noseq) oseq <= 0; else if (odone) oseq <= oseq+1;	// only count data packets
    ocyclep <= (ostart && ordyd) || (ocyclep && !odone);
    ocycle  <= ocyclep;
    ocycled <= ocycle;
    if (!ocycle) ocnt <= {npkt,4'h0}; else ocnt <= ocnt-1; // 16x4by pkt count for NPKT 
    odone   <= (ocnt[5:0]==3);	// bits good up through 256 by packet
    if (zfill) tst <= 0; else tst <= tst+2;
    if (ocycle) obusa <= hdat;
    obusb   <= ocycled? obusa : tsto;
    if (ocycle) tdatl <= tdat;
    tctick  <= ocycle && (tdat!=0) && (tdatl==0); // OPPS rising edge
    tcval   <= tctick || (tcval && !oheader);
    if (tctick) tcoff <= {~ocnt[5:0], tdatl[0]? 2'd0 : tdatl[1]? 2'd1 : tdat[2]? 2'd2 : 2'd3 };
    obusc   <= oheader? {hf3, 4'h0,ocycled?npkt:4'h0, hf1, 8'hA5} : obusb;
    hseld   <= hsel;
    ohold   <= hold;
    jhold   <= ohold;
    lhold   <= oheader? jhold : lhold;
    ahold   <= oena && (jhold!=0 || lhold!=0 || !oenad); // if current or last sent is to hold off or initializing output
    oenad   <= oena;
  end
  assign hena = ocyclep;
  assign odat = obusc;

  reg fff; always @(posedge oclk) fff <= !fff;
  assign test = {hold, fff,ordy,ocycle,ostart};
//  assign test = {hseld,hold[0],ordy,ocycle,ostart};

endmodule


/*
 Hypertransport over RocketIO port code
 Error checking and synchronization happens in the RIO layer
 note: iena is not continuous

  iena idat
  jena jdat isyncs
  kena kdat isync
            cycle

*/
module hypinr (iclk,ena, iena,idat, hval,hsel,hdat, cval,cdat, modes,hold,csm,cab,calib,test);
  parameter IBW=64;
  parameter OBW=64;
  parameter NPKT=1;
  parameter FLG=0;

  localparam WIDE=(IBW==256)? 3 : (IBW==128)? 2 : (IBW==64)? 1 : 0;
  localparam NQW=1<<WIDE;
  localparam OBV=OBW/8;
  localparam IBV=IBW/8;
  localparam CMDF=1;

  // raw RIO bus
  input iclk,ena,iena;
  input [IBW-1:0] idat;
  // data bus
  output hval;
  output [OBV-1:0] hsel;
  output [OBW-1:0] hdat;
  // cmd bus
  output cval;
  output [35:0] cdat;
  // auxiliary signals
  input  [1:0] modes;
  output [3:0] hold;
  output csm,cab;
  output [7:0] calib;
  output [7:0] test;

  wire L=0, H=1;
  wire mode = modes[0];

  // packet logic
  reg isync,icycle,idone,jena,kena;
  reg sen,ten,chm,chb;
  reg [3:0] hold,iseq;
  reg [7:0] icnt,ichan;
  reg [NQW-1:0] isyncs;
  reg [IBW-1:0] jdat,kdat;
  reg [OBV-1:0] kchn;	
  wire [7:0] npkt = (NPKT>0)? NPKT : kdat[19:16];
  wire [7:0] ncnt = npkt<<(4-WIDE);
  wire [7:0] vchan = kdat[31:24];
  wire vchb = vchan[0];
  wire vsen = vchan[1];
  wire vten = vchan[2];
  wire vchm = vchan[7];
  wire ihdr = jena && isync;
  wire icyc = jena && isync && (kdat[18:16]!=0);
  always @(posedge iclk) begin
    jena <= iena; 
    kena <= jena;
    if (iena) jdat <= idat;
    if (jena) kdat <= jdat;
    if (jena) isync <= (isyncs!=0) && !(icycle && !idone) && !isync;
    if (jena) icycle <= (ena && isync && kdat[18:16]!=0) || (icycle && !idone) || mode;
    if (jena) idone <= icycle && (icnt[5:0]==2);
    if (ihdr) icnt <= ncnt; else if (icycle&jena) icnt <= icnt-1; // 16x4by pkt count for NPKT 
    if (mode) chb  <= 0; else if (ihdr) chb <= vchb;
    if (mode) chm  <= 0; else if (ihdr) chm <= vchm;
    if (mode) sen  <= 0; else if (ihdr) sen <= vsen;
    if (mode) ten  <= 0; else if (ihdr) ten <= vten;
    if (mode) hold <= 0; else if (ihdr) hold <= kdat[11:8];
    if (mode) ichan<= 0; else if (ihdr) ichan <= vchan;
    if (mode) kchn <= 0; else if (ihdr) kchn <= {kdat[119:116],kdat[87:84],kdat[31:24]};	// extended channel
    if (mode) iseq <=15; else if (icyc) iseq <= kdat[15:12];
  end

  reg [7:0] errs;
  reg [NQW-1:0] erri;
  genvar i;
  generate
  for (i=0; i<NQW; i=i+1) begin:blk
    always @(posedge iclk) erri[i] <= (jdat[31:0] != jdat[31+i*32:i*32]);
    always @(posedge iclk) isyncs[i] <= (idat[7+i*32:i*32]==8'hA5);
  end
  endgenerate
  wire err = (erri!=0) && !icycle;	// all lanes should be equal (erri==0) for syncs and header
  always @(posedge iclk) if (!ena) errs <= 0; else if (err) errs <= errs+1;

`ifdef ICE_MB
  wire [OBV-1:0] ksel = kchn;		// special expanded channel for NVME block #
  assign cab  = ichan[0];
`else
  wire [OBV-1:0] ksel = {4{chm,L,L,L,ten,sen,ten,sen}};	// only A/B channel number and startup synchronization signals are important
  assign cab  = chb;
`endif

  assign cval = jena && isync && kdat[19+96] && kdat[19+64];				// cena is in both slots - parity starts at bit=20
  assign cdat = {kdat[17+96:16+96],kdat[17+64:16+64],kdat[15+96:96],kdat[15+64:64]};	// cdat is split across upper 2 lanes

  assign hval = kena && icycle;
  assign hsel = ksel;
  assign hdat = kdat;
  assign csm  = chm;
  assign calib = errs;

  reg fff; always @(posedge iclk) if (!iena) fff <= !fff;
  assign test = {ten,sen,chm,cab, err,icycle,fff,iena};

endmodule

/*
  ostart
           ocyclep  hena
4 ocycle   ocyclep  hsel   oheader    hdat 		
3 ocycle   ocyclep  hseld  ocycled    obusa	oena-h
2 ocycle   ocyclep  odone  	      obusc	oena-d
1 ocycle            ostart

4 ocycle

*/

module hypoutr (oclk,ena, oena,odat, hrdy,hena,hsel,hdat, crdy,cena,cbus, hold,test);
  parameter IBW=64;
  parameter OBW=64;
  parameter NPKT=1;
  parameter FLG=0;	// 1=OnOffProtocol 2=TimeCode 3=CmdFifo
  localparam WIDE=(IBW==256)? 3 : (IBW==128)? 2 : (IBW==64)? 1 : 0;
  localparam NQW=1<<WIDE;
  localparam IBV=IBW/8;
  localparam OBV=OBW/8;
  localparam ONOF=(FLG&`HYP_ONOF)? 1:0;
  localparam CMDF=1;
  // raw RIO output bus
  input oclk,ena;
  output reg oena;
  output [OBW-1:0] odat;
  // data bus
  input hrdy;
  output hena;
  input [15:0] hsel;
  input [IBW-1:0] hdat;
  // command bus
  input crdy;
  output cena;
  input [35:0] cbus;
  // auxiliary signals
  input [4:0] hold;
  output [7:0] test;

  wire L=0, H=1;

  // output
  reg ostart,ocyclep,ocycle,odone,ordy,ordyd,enad,ahold,ostartd,ocycled;
  reg [7:0] ocnt;
  reg [3:0] ohold,lhold,jhold,oseq,tst,acnt;
  reg [OBW-1:0] obusa,obusc;
  wire [OBW-1:0] headers;
  wire [3:0] npkt = NPKT;
  wire [7:0] ncnt = NPKT<<(4-WIDE);
  wire [15:0] chksum = 0;
  wire [31:0] header = {hsel[7:0], 4'h0,ocycle?npkt:4'h0, oseq,jhold, 8'hA5};
  wire [127:0] qheader = CMDF? { chksum[15:8],hsel[15:12],cena,L,cbus[35:34],cbus[31:16], chksum[7:0],hsel[11:8],cena,L,cbus[33:32],cbus[15:0], header, header} : {NQW{header}} ;
  wire oheader; delaypipe #(2,1) dph (oclk,ostart,oheader);
  wire aohold = hold[4];
  wire [3:0] tst0 = {tst[3:1],L};
  wire [3:0] tst1 = {tst[3:1],H};
  wire [31:0] tsto = {~tst1,~tst1, tst1,tst1, ~tst0,~tst0, tst0,tst0};
  always @(posedge oclk) begin
    tst     <= tst+2;
    if (!ena) oseq <= 0; else if (odone) oseq <= oseq+1;	// only count data packets
    ordy    <= hrdy && ena;
    ordyd   <= ordy;
    ostart  <= !ostart && !(ocyclep && !odone) && (ordy || ahold || crdy);
    ostartd <= ostart;
    ocyclep <= (ostart && ordyd) || (ocyclep && !odone);
    ocycle  <= ocyclep;
    ocycled <= ocycle;
    if (!ocycle) ocnt <= ncnt; else ocnt <= ocnt-1;	// 16x4by pkt count for NPKT 
    odone   <= (ocnt[5:0]==3);				// bits good up through 256 by packet
    obusa   <= ocycle? hdat : {NQW{tsto}};
    obusc   <= oheader? qheader : obusa;
    ohold   <= hold;
    jhold   <= ohold;
    lhold   <= oheader? jhold : lhold;
    ahold   <= ONOF && ena && (jhold!=0 || lhold!=0 || acnt==1);// if current or last sent is to hold off or initializing output
    if (!ena||ocycle) acnt <= 0; else acnt <= acnt+1;
    enad    <= ena;
    oena    <= oheader || ocycled;
  end
  assign cena = oheader && crdy;
  assign hena = ocyclep;
  assign odat = obusc;
  assign test = {hold, oena,ordy,ocycle,ena};

endmodule
