/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/** 
  Conversion routines for different bit lengths

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  12/15/2004

*/

/*
 12b/16b pack and unpack routines

  12b  256by = 64x4by = 84x3by + 4by = 21 loops of ABC or 63 lines then extra read of 1 line
  16b  336by =          84x4by + 0by

  A 76543210  
  B 76543210
  C 76543210

    543_210_
    321_076_ 
    107_654_
    765_432_

   go vena   cnt
   go1 wbus
   go2 q     sel
   go3 wbus

*/

module convert_12to16 (clk,rst, vrdy,vena,vbus, wrdy,wena,wbus);
  parameter BW=32;
  input clk,rst;
  input vrdy,wrdy;
  output vena,wena;
  input [BW-1:0] vbus;
  output [BW-1:0] wbus;

  reg [1:0] count;
  reg [1:0] sel,seld;
  reg go,go1,go2,go3,cap;
  assign vena = go && (count[1:0]!=3);
  assign wena = go3;
  always @(posedge clk) begin
    go <= vrdy && wrdy; 
    go1 <= go; go2 <= go1; go3 <= go2;
    cap <= go && (count[1:0]!=3);
    if (rst) count <= 0; else if (go) count <= count+1;
    sel <= count[1:0];
    seld <= sel;
  end

genvar i;
generate
for (i=0; i<BW; i=i+32) begin
  reg [63:0] q;
  always @(posedge clk) if (cap) q <= {vbus[31+i:i],q[63:32]};
  wire [31:0] q0 = {q[55:44],4'h0,q[43:32],4'h0};	// c=0
  wire [31:0] q1 = {q[47:36],4'h0,q[35:24],4'h0};	// c=1
  wire [31:0] q2 = {q[39:28],4'h0,q[27:16],4'h0};	// c=2
  wire [31:0] q3 = {q[63:52],4'h0,q[51:40],4'h0};	// c=3
  mux4xNp #(32) mx (wbus[31+i:i], q0,q1,q2,q3, seld,clk);
end
endgenerate

endmodule

/*
  cnt	sel	seld	vrdy+wrdy
  0	0	0	go	vena  
  1	0	0	go1  		vbus
  2 	1	0	go2  		qhi
  3	2	1	go3  		qlo	wbusx
  4	3	2	go4

*/
module convert_12to16i (clk,rst, bit0, vrdy,vena,vbus, wrdy,wena,wbus);
  parameter BW=64;
  input clk,rst,bit0;
  input vrdy,wrdy;
  output vena,wena;
  input [BW-1:0] vbus;
  output [BW-1:0] wbus;

  reg [2:0] count;
  reg [1:0] sel,seld;
  reg go,go1,go2,go3,cap;
  assign vena = go && (count[1:0]!=3);
  assign wena = go3;
  always @(posedge clk) begin
    go  <= vrdy && wrdy; 
    go1 <= go; go2<=go1; go3<=go2;
    if (rst) count <= 0; else if (go) count <= count+1;
    cap <= vena;
    sel <= count[1:0]; seld <= sel;
  end
  reg [3:0] lsbs;
  reg [2*BW-1:0] q;
  always @(posedge clk) if (cap) q <= {vbus,q[2*BW-1:BW]};
  always @(posedge clk) if (cap) lsbs <= {3'b0,bit0};
  wire [BW-1:0] q0 = {q[11+100:100],lsbs,  q[11+88:88],  lsbs,  q[11+76:76],lsbs,  q[11+64:64],lsbs};	// c=0
  wire [BW-1:0] q1 = {q[11+84:84],  lsbs,  q[11+72:72],  lsbs,  q[11+60:60],lsbs,  q[11+48:48],lsbs};	// c=1
  wire [BW-1:0] q2 = {q[11+68:68],  lsbs,  q[11+56:56],  lsbs,  q[11+44:44],lsbs,  q[11+32:32],lsbs};	// c=2
  wire [BW-1:0] q3 = {q[11+116:116],lsbs,  q[11+104:104],lsbs,  q[11+92:92],lsbs,  q[11+80:80],lsbs};	// c=3
  mux4xNp #(BW) mx (wbus, q0,q1,q2,q3, seld,clk);

endmodule

module cvt16to12 (clk,rst, vena,vbus, wena,wbus);
  parameter BW=32;
  input clk,rst;
  input vena;
  output reg wena;
  input [BW-1:0] vbus;
  output [BW-1:0] wbus;

  reg [1:0] wsel,count;
  reg [23:0] v24r;
  wire [23:0] v24 = {vbus[31:20],vbus[15:4]};
  always @(posedge clk) begin
    if (rst) count <= 0; else if (vena) count <= count+1;
    if (vena) v24r <= v24;
    if (vena) wsel <= count;
    wena <= vena && (count!=3);
  end
  mux3xN #(32) mm (wbus, {v24[7:0],v24r}, {v24[15:0],v24r[23:8]}, {v24,v24r[23:16]}, wsel);
endmodule

module cvt12to16 (clk,rst, vena,vbus, wena,wbus);
  parameter BW=32;
  input clk,rst;
  input vena;
  output reg wena;
  input [BW-1:0] vbus;
  output [BW-1:0] wbus;

  reg [1:0] wsel,count;
  reg [23:0] v24r;
  wire [23:0] v24 = {vbus[31:20],vbus[15:4]};
  always @(posedge clk) begin
    if (rst) count <= 0; else if (vena) count <= count+1;
    if (vena) v24r <= v24;
    if (vena) wsel <= count;
    wena <= vena && (count!=3);
  end
  mux3xN #(32) mm (wbus, {v24[7:0],v24r}, {v24[15:0],v24r[23:8]}, {v24,v24r[23:16]}, wsel);
endmodule

module cvt12leto12be (clk,rst,ena, vena,vbus, wena,wbus);
  parameter BW=32;
  input clk,rst,ena;
  input vena;
  input [BW-1:0] vbus;
  output reg wena;
  output reg [BW-1:0] wbus;

  reg [1:0] count,prime;
  reg [95:0] data;
  wire [95:0] d = {vbus,data[95:32]};
  wire zero = rst || !ena;
  always @(posedge clk) begin
    if (rst) prime <= 0; else if (vena && !prime[1]) prime <= prime+1;
    if (zero) count <= 0; else if (vena) count <= (count==3)? 1 : count+1;
    if (vena) data <= d;
    wena <= vena && prime[1];
    if (vena) wbus <=	(count==3)? 	{ d[43:32],d[57:44],d[67:60] } :
	    		(count==1)?	{ d[27:24],d[47:36],d[59:48],d[79:76] } :
			(count==2)?	{ d[35:32],d[31:28],d[51:40],d[63:52] } :
					{ d[63:32] } ;
  end
endmodule

module cvtEndian (clk,rst,swaps, vena,vbus, wena,wbus);
  parameter BW=32;
  input clk,rst;
  input [7:0] swaps;
  input vena;
  input [BW-1:0] vbus;
  output reg wena;
  output reg [BW-1:0] wbus;

  reg [1:0] count,prime;
  reg [31:0] xbus,ybus;
  wire zero = rst || !swaps[2];
  always @(posedge clk) begin
    if (rst) prime <= 0; else if (vena && !prime[1]) prime <= prime+1;
    if (zero) count <= 0; else if (vena) count <= (count==3)? 1 : count+1;
    if (vena) xbus <= vbus;
    if (vena) ybus <= xbus;
    wena <= vena && prime[1];
    if (vena) wbus <=	(count==3)? 	{vbus[3:0],xbus[31:28], xbus[19:12], xbus[3:0],xbus[23:20], xbus[11:4]} : 
	    		(count==1)?	{xbus[19:16],vbus[7:4], xbus[27:20], xbus[11:4], ybus[27:24],xbus[15:12]} :
			(count==2)?	{xbus[27:20], xbus[11:8],xbus[31:28], xbus[19:12], xbus[3:0],ybus[31:28]} :
			swaps[0]?	{xbus[23:16], xbus[31:24], xbus[7:0], xbus[15:8] } :
			swaps[1]?	{xbus[7:0], xbus[15:8], xbus[23:16], xbus[31:24] } :
					xbus;
  end
                     
endmodule

