/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  Boundary Scan Interface Controller

  This routine maps boundary scan registers USER1 and USER2 to 
  Caddr and Cdata registers for communicating with the module's host.

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  1/15/2003

  word bit functions
  11  ainc - auto increment address
  10  read - read/write data cycle
  9   addr - byte address cycle
  8   page - page address cycle
  7:0 data

  count
  0						!spi_enb
  1				spi_gow	spi_gow
  2				
  3
  4
  5
  6				
  7					!spi_gor
  8
  9				!spi_gow !spi_gow
c 10
c 11  rdy				
c 12  cyc,adc,pgc	pgc	adc	
c 13  wrc,rdc,aic	spi_enb		rdc,wrc	
c 14  wr,rd 				spi_enr
c 15				spi_gor

  BSCAN_SPI - with SPI interface
  BSCAN_TS  - with Temperature Sensor interface

  new mode

  count
  0	read
  1	addr
  2	page/ainc4
  3	ainc
  4-11	data
  12-15 pad

*/
`include "mdefs.h"

`ifdef BSCAN_INTERNAL
module bscanif (clk, cs,rd,wr, page,addr, data,datai, tck,tms,tdi,tdo, spi_clk,spi_fcs,spi_dat);
output spi_clk; inout spi_fcs; inout [3:0] spi_dat; 

`elsif BSCAN_SPI_FLASH
module bscanif (clk, cs,rd,wr, page,addr, data,datai, spi_clk,spi_fcs,spi_dat);
output spi_clk; inout spi_fcs; inout [3:0] spi_dat;

`elsif BSCAN_SPI
module bscanif (clk, cs,rd,wr, page,addr, data,datai, spi_clkf,spi_clkr,spi_ena,spi_dati,spi_dato);
output spi_clkf,spi_clkr,spi_ena,spi_dati;
input spi_dato;

`elsif BSCAN_TS
module bscanif (clk, cs,rd,wr, page,addr, data,datai, tsena,tsalert);
output tsena;
input tsalert;

`elsif BSCAN_HOLD
module bscanif (clk, cs,rd,wr, page,addr, data,datai, testo);
output [3:0] testo;

`else
module bscanif (clk, cs,rd,wr, page,addr, data,datai);
`endif

output clk,cs,wr;
output [1:0] rd;

output [7:0] page,addr,data;
input [7:0] datai;

// boundary scan pin interface
wire capture, drck1, drck2, reset, sel1, sel2, shift, update, tdo1, tdo2, runtest;
`ifdef BSCAN_INTERNAL
reg tdo0; always @(negedge tck) if (reset) tdo0 <= 0; else tdo0 <= !tdo0;
input tck,tms,tdi; output reg tdo;  always @(negedge tck) tdo <= sel2? tdo2 : sel1? tdo1 : tdo0;
`else
wire tdi,tck,tms;
`endif

`ifdef BSCAN_INTERNAL
BSCAN_INT #(.JTAG_CHAIN(2)) bs (capture, drck2, reset, runtest, sel2, shift, tck, tdi, tms, update);

`elsif SPARTAN3
BSCAN_SPARTAN3 bs (capture, drck1, drck2, reset, sel1, sel2, shift, tdi, update, tdo1, tdo2);
assign tdo1 = 0;

`elsif VIRTEX2
BSCAN_VIRTEX2 bs (capture, drck1, drck2, reset, sel1, sel2, shift, tdi, update, tdo1, tdo2);
assign tdo1 = 0;

`elsif VIRTEX4
BSCAN_VIRTEX4 bs (capture, drck2, reset, sel2, shift, tdi, update, tdo2);

`elsif VIRTEX5
BSCAN_VIRTEX5 #(2) bs (capture, drck2, reset, sel2, shift, tdi, update, tdo2);

`elsif SPARTAN6
BSCAN_SPARTAN6 #(2) bs (capture, drck1, reset, runtest, sel2, shift, tck, tdi, tms, update, tdo2);
assign drck2 = tck; // needed for temp sensor timing

`elsif ARTIX7
BSCAN_SPARTAN6 #(2) bs (capture, drck1, reset, runtest, sel2, shift, tck, tdi, tms, update, tdo2);
//BSCAN_ARTIX7 #(2) bs (capture, drck1, reset, runtest, sel2, shift, tck, tdi, tms, update, tdo2);
assign drck2 = tck; // needed for temp sensor timing

`elsif VIRTEX6
BSCAN_VIRTEX6 #("FALSE",2) bs (capture, drck2, reset, runtest, sel2, shift, tck, tdi, tms, update, tdo2);

`elsif VIRTEX7
BSCANE2 #(.JTAG_CHAIN(2)) bs (capture, drck2, reset, runtest, sel2, shift, tck, tdi, tms, update, tdo2);

`elsif VIRTEX8
BSCANE2 #(.JTAG_CHAIN(2)) bs (capture, drck2, reset, runtest, sel2, shift, tck, tdi, tms, update, tdo2);

`elsif VIRTEX8P
BSCANE2 #(.JTAG_CHAIN(2)) bs (capture, drck2, reset, runtest, sel2, shift, tck, tdi, tms, update, tdo2);

`elsif ALTERA
reg [12:0] areg;
wire cir, uir, sdr;
wire [6:0] ir;
reg sel2_;
wire icir,isir,iuir,icdr,isdr,iudr;
vjtag bs (.tdi(tdi), .tdo(tdo2), .ir_out(ir), .ir_in(ir), .virtual_state_sdr(dshift), .tck(drck2), .tms(tms),
          .jtag_state_sdr(sdr), .jtag_state_sir(ishift), .virtual_state_cir(cir), .virtual_state_uir(uir) ); //drck2) );
jtag_sm icejtag (clk, tms, icir, isir, iuir, icdr, isdr, iudr);
always @(posedge clk) begin
  areg <= {tdi,areg[12:1]};
  if (icdr) sel2_ <= (areg[9:0]==10'h0C);
end
assign sel2 = sel2_;
assign shift = isdr;
`endif

BUFG _clk (clk,drck2);

wire do2 = sel2 && shift;

// data register
reg rdy, cyc, adc, pgc, aic, ainc, binc, wrc, wrc1, wrc2, rdc, rdc0, rdc1, rdc2, act;
reg [7:0] cpage,caddr,cdata;
reg [3:0] count,cmd;
reg [11:0] sreg;

always @(posedge clk) begin

`ifdef BSCAN_GEN2
  if (do2) count <= count+1; else count <= 0;

  if (rdc2) sreg[7:0] <= datai; else if (do2) sreg[7:0] <= {tdi,sreg[7:1]};  

  rdy  <= (count==3);			// latch command
  if (rdy) cmd <= sreg[7:4];
  rdc0 <= do2 && (count==0) && tdi;	// pre-read
  rdc1 <= cmd[0] && (count==5);
  rdc2 <= rdc1;
  cyc  <= (count==11);			// latch data 
  ainc <= cmd[3] && (count==(cmd[0]?6:13));
  binc <= cmd[3] && (count==(cmd[0]?6:13)) && (caddr[1:0]==3) && !cmd[2];
  wrc1 <= cyc && cmd[1:0]==0;
  if (cyc && cmd[1:0]==0) cdata <= sreg[7:0];
  if (cyc && cmd[2:0]==6) cpage <= sreg[7:0];
`ifdef OLDWAY
  if (cyc && cmd[2:0]==2) caddr[7:0] <= sreg[7:0]; else if (ainc) caddr[7:0] <= caddr[7:0]+1;
`else
  if (cyc && cmd[2:0]==2) caddr[1:0] <= sreg[1:0]; else if (ainc) caddr[1:0] <= caddr[1:0]+1;
  if (cyc && cmd[2:0]==2) caddr[7:2] <= sreg[7:2]; else if (binc) caddr[7:2] <= caddr[7:2]+1;
`endif
  act <= (do2 && count<15);		// window for bus rd/wr activity

`else

  if (rdc2) sreg[7:0] <= datai; else if (sel2 && shift) sreg[11:0] <= {tdi,sreg[11:1]};  
  if (sel2 && shift) count <= count+1; else count <= 0;

  rdy <= (count==10);
  cyc <= rdy;				 // latch data early
  adc <= (rdy && sreg[10] && !sreg[9]);  // latch addr early - actually sreg[9:8]
  pgc <= (rdy && sreg[10] &&  sreg[9]);	 // latch page early - actually sreg[9:8]
  wrc <= (cyc && !sreg[10] && !sreg[9]);
  rdc <= (cyc && sreg[10] && sreg[9:8]!=1);	// possible addr w/ pre read but not page
  wrc1 <= wrc; 
  wrc2 <= wrc1; 
  rdc0 <= rdc;
  rdc1 <= rdc; 
  rdc2 <= rdc1;
  if (cyc) aic <= sreg[11];
  ainc <= aic && (wrc2 || rdc2);

  if (pgc) cpage <= sreg[7:0];
  if (adc) caddr <= sreg[7:0]; else if (ainc) caddr <= caddr+1;
  if (cyc) cdata <= sreg[7:0];

  act <= (count>8 && count<15);	// window for bus rd/wr activity

`endif

end

`ifdef BSCAN_SPI
reg spi_enb,spi_enr,spi_cen,spi_cen1,spi_cen2;
wire spi_gow = (count>0 && count<=8);
wire spi_gor = (count>14 || count<=6);
wire spi_go  = spi_enr? spi_gor : spi_gow;
always @(posedge clk or negedge shift) begin
  if (!shift) spi_enb <= 0; else spi_enb  <= (pgc && sreg[7]) || spi_enb;
end
always @(posedge clk) begin
  spi_enr  <= rdc || (spi_enr && !cyc);
  spi_cen  <= spi_cen ^ (spi_enb && spi_go);
  spi_cen2 <= spi_cen1;
end
always @(negedge clk) begin
  spi_cen1 <= spi_cen;
end
assign spi_ena  = spi_enb;
assign spi_clkf = !(spi_cen2 ^ spi_cen1);
assign spi_clkr = !(spi_cen1 ^ spi_cen);
assign spi_dati = sreg[10];
assign tdo2 = spi_enb? spi_dato : sreg[0];
assign cs = act && !cpage[7];

`else
assign tdo2 = sreg[0];
assign cs = act;
`endif

`ifdef BSCAN_TS
reg tsstr,tsena;
reg [7:0] tscnt;
always @(posedge clk) begin
  tsstr <= wrc1 && (page==8'hA5);
  if (tsstr) tscnt <= cdata; else if (tscnt!=0) tscnt <= tscnt-1;
  tsena <= (tscnt>0) && (tscnt<=16);
end
`endif

`ifdef BSCAN_SPI_FLASH
wire capture1,  reset1, runtest1, shift1, tck1, tdi1, tms1, update1;
`ifdef ALTERA
sld_virtual_jtag #(.sld_auto_instance_index("NO"), .sld_instance_index(0), .sld_ir_width(1)) bs1 (.virtual_state_sdr(sel1), .tck(drck1), .tdi(tdi1), .tdo(tdo1)); 
`elsif BSCAN_INTERNAL
BSCAN_INT #(.JTAG_CHAIN(1)) bs1 (capture1, drck1, reset1, runtest1, sel1, shift1, tck, tdi, tms, update1);
assign tdi1 = tdi;
`else
BSCANE2 #(.JTAG_CHAIN(1)) bs1 (capture1, drck1, reset1, runtest1, sel1, shift1, tck1, tdi1, tms1, update1, tdo1);
`endif
reg sel1d,fcs,dati,dato;
always @(negedge sel1 or negedge spi_clk) begin
  if (!sel1) sel1d <= 0; else sel1d <= sel1;
  if (!sel1) fcs   <= 0; else fcs   <= sel1d;
end
`ifdef BSCAN_GSPICLK
BUFG _sclk (spi_clk,drck1);
`else
assign spi_clk = drck1;
`endif
assign spi_fcs = !fcs;
assign spi_dat[0] = tdi1;
assign tdo1 = spi_dat[1];
assign spi_dat[2] = 1'b1;
assign spi_dat[3] = 1'b1;
`endif

// external interface
assign rd = {rdc0,rdc1};
assign wr = wrc1;
assign addr = caddr;
assign data = cdata;
assign page = cpage;

`ifdef BSCAN_HOLD
assign testo = {dshift,isdr,tdi,tms};
`endif

endmodule


module BSCAN_INT (capture, drck, reset, runtest, sel, shift, tck, tdi, tms, update);

parameter JTAG_CHAIN=1;
input tck,tms,tdi; 
output capture, drck, reset, runtest, sel, shift, update;

localparam RESET=0, IDLE=1, SELDR=2, SELIR=3, CAPT=4, SHFT=5, EXIT=6, UPDT=7;
localparam INST = (JTAG_CHAIN==1)? 2 : (JTAG_CHAIN==2)? 3 : (JTAG_CHAIN==3)? 10 : 11;

reg [2:0] state;
reg ir,valid,valid1,valid2,shfdr;
reg [5:0] inst;

wire match = (inst==INST);

always @(posedge tck) begin
  case (state)
    RESET:	state <= tms? RESET : IDLE;
    IDLE:	state <= tms? SELDR : IDLE;
    SELDR:	state <= tms? SELIR : CAPT;
    SELIR:	state <= tms? RESET : CAPT;
    CAPT:	state <= tms? EXIT  : SHFT;	// DR or IR path
    SHFT:	state <= tms? EXIT  : SHFT;	// DR or IR path
    EXIT:	state <= tms? UPDT  : UPDT;	// DR or IR path
    UPDT:	state <= tms? SELDR : IDLE;	// DR or IR path
  endcase
  if (state==SELDR) ir<=0; else if (state==SELIR) ir<=1; 
  if (state==SHFT && ir) inst <= {tdi,inst[5:1]};
  if (state==RESET) valid1 <= 0; else if (state==UPDT && ir) valid1 <= match;
  if (state==RESET) valid2 <= 0; else valid2 <= valid1;
end
always @(negedge tck) begin
  if (state==RESET || state==EXIT) valid  <= 0; else if (state==UPDT && ir) valid <= match; 
  shfdr <= (state==SHFT) && !ir;
end
assign capture = (state==CAPT);
assign drck = tck || !valid2;
assign reset = (state==RESET);
assign runtest = (state==IDLE);
assign sel = valid; 
assign shift = shfdr;
assign update = (state==UPDT);
wire shiftir = (state==SHFT) && ir;

endmodule

module bs2drp (
  bsclk,bscs,bsrd,bswr,bspage,bsaddr,bsdata,bsdatao, 			// BS Interface
  drp_clk,drp_en,drp_wex,drp_addr,drp_di,drp_rdyx,drp_dox,dmon_dox);	// DRP Interface

  parameter PORTS=8;	// total ports
  parameter PORTX=0;	// special extra port
  parameter PORTOFF=0;

  // BS ports
  input bsclk,bscs,bsrd,bswr;
  input [7:0] bspage,bsaddr,bsdata;
  output [7:0] bsdatao;

  // DRP ports
  input drp_clk;
  output reg drp_en;
  output reg [PORTS-1:0] drp_wex;
  output reg [9:0] drp_addr;
  output reg [15:0] drp_di;
  input [PORTS-1:0] drp_rdyx;
  input [PORTS*16-1:0] drp_dox;
  input [PORTS*17-1:0] dmon_dox;

  reg  [15:0] drp_dor;
  wire        drp_rdyxi;
  wire [15:0] drp_doxi;
  wire [15:0] dmon_doxi;

  wire L=0;
  wire bsrw = bscs && (bswr || bsrd);
  reg drpen,drpwe,drpad;
  reg [12:0] drpaddr;
  always @(posedge bsclk) begin
    if (bsrw) drpaddr <= {bspage[5:0],bsaddr[7:1]};
    if (bsrw) drp_di <= {bsdata,drp_di[15:8]};
    if (bsrw) drpad  <= bsaddr[0];
    if (bsrw) drpwe  <= (bswr && bsaddr[0]);
    if (!bscs) drpen <= 0; else drpen <= (bswr && bsaddr[0]) || (bsrd && !bsaddr[0]);
  end
  wire drpx = drpaddr[12];
  assign bsdatao = drpad? drp_dor[15:8] : drp_dor[7:0];

  reg drpgo_,drpgo;
  wire drp_go = drpgo && !drpgo_;
  genvar i;
  generate
  for (i=0; i<PORTS; i=i+1) begin:pi
    always @(posedge drp_clk) drp_wex[i] <= drp_go && drpwe && ( (i+1==PORTX)? drpaddr[12:9]==9 : drpaddr[12:9]==8 || drpaddr[12:9]==i+PORTOFF);
  end
  endgenerate

  always @(posedge drp_clk) begin
    drpgo   <= drpen;
    drpgo_  <= drpgo;
    drp_en  <= drp_go;
`ifdef VIRTEX8P
    drp_addr <= {drpaddr[8],L,drpaddr[7:0]};
`else
    drp_addr <= {L,drpaddr[8:0]};
`endif
    if (drpx) drp_dor <= dmon_doxi;
    else if (drp_en) drp_dor <= 16'hDEAD;
    else if (drp_rdyxi) drp_dor <= drp_doxi;
  end
  wire [3:0] xaddr = drpaddr[11:9];
  wire [3:0] raddr = (PORTX!=0 && drpaddr[12:9]==9)? PORTX-1 : xaddr;
  muxMxN #(PORTS,1) drpmr (drp_rdyxi, drp_rdyx, raddr);
  muxMxN #(PORTS,16) drpmd (drp_doxi, drp_dox, raddr);
  muxMxN #(PORTS,17) dmonmd (dmon_doxi, dmon_dox, xaddr);

endmodule
