!******************************************************************************!
! Visualization tool for diagnosing issues with uSec or TC high precision time codes
!
! Arguments:
! OUTSPECIFIED    - 5000 output file
! CSVOUTSPECIFIED - CSV output file
! REPORTSPECIFIED - report file describing issues with Timeline
!
! @since    NextMidas 4.0.2
! @author:  Jeanette McCall
!******************************************************************************!
startmacro s:outSpecified s:csvoutSpecified s:reportSpecified
  switch INTERACTIVE interAct GET DEF=TRUE  SDEF=TRUE  ! for normal interactive test(s)
  switch BASELINE    baseline GET DEF=FALSE SDEF=TRUE  ! for defects or more obscure test case(s)
  switch KEEP        keep     GET DEF=FALSE SDEF=TRUE  ! Keep created files - do not erase them
  switch DEBUG       debug    GET DEF=FALSE SDEF=TRUE  ! Show debug prints

  lookandfeel Gear4
  set checktime_testcase_good "checktime_tl_good.tmp"
  set checktime_testcase_back_in_time "checktime_tl_back_in_time.tmp"
  set checktime_testcase_wrap_normal "checktime_tl_wrap_normal.tmp"
  set checktime_testcase_wrap_drop "checktime_tl_wrap_drop.tmp"
  set checktime_testcase_wrap_order  "checktime_tl_wrap_order.tmp"
  res example_files_ready "FALSE"

  if interAct then
    res labelsOn "FALSE"
    res labelsOn
    res t:actual_color_tbl {}

    call set2DPlotInstructions
    !call set3DPlotInstructions

    invoke columnnames nxm.sys.intr.Timelineisvalid.getCSVColumnNames()
    call refreshPlotQualifiers

    set instructions "This macro can be used to visually check or see issues with timeline keywords\n\n&
                      Select either 'user specified' file or example file\n&
                      - For 'user specified', use the file chooser to select an\n  &
                        input file, or use the prompt to manually enter the path\n  &
                        to the file\n&
                      - GREEN indicates the Timeline PASSED the validation test\n&
                      - RED indicates the Timeline FAILED the validation test\n\n&
                      ********** Left Side contains 2D Plots **********\n\n&
                      ********** Right Side contains 3D Plot **********\n\n&
                      Drop Down 2D and 3D Plot information to see details&
                      "

    pipe on
      panel/msgid=MAIN/setup/controls=gc
      gcontrol label macro "Test MACRO"
      gcontrol menu  main "Exit with Status =" "PASS,FAIL" TABLE={ItemColors={PASS="GREEN",FAIL="RED"}} /tleft

      gcontrol text  idtext "Instructions" instructions 24 15 /title
      gcontrol button labels ""             "Toggle Time Labels on TL and RIGHT plots"
      gcontrol label filesel "File Selection" open
        gcontrol choice  inputfile "select input file" &
            "user specified,example TL good,example TL wrap normal,example TL wrap out-of-order,example TL wrap those before wrap dropped" &
            ""
        gcontrol FILE filechooser "FileChooser:" world.prm "*.tmp|prm" "DAT|CWD|1" 60 10
        set gc.filechooser.enabled false
        gcontrol prompt fileInputPrompt "File Prompt Input:" ""
        set gc.fileinputprompt.enabled false
      gcontrol label 2dPlotsLabel "2D Plot information" open
        gcontrol text 2dPlotsInstr "2D Plot Instructions" instructions2D 1 20
      gcontrol label 3dPlotLabel "3D Plot information" closed
        gcontrol text 3dPlotInstr "3D Plot Instructions" instructions3D 1 18
      plot/id=3dplot view=3d xlabel=TIME_(X) ylabel=ELEMENT_(Y) zlabel=SAMPLE_(Z)
      plot/id=eltimeplot  /READOUT=+ONAXIS xlabel=ELEMENT ylabel=TIME X1=1
      plot/id=entOffPlot  /READOUT=+ONAXIS xlabel=ELEMENT ylabel=ENTRY  X1=1
      plot/id=tDeltaPlot  /READOUT=+ONAXIS xlabel=ELEMENT ylabel=TDELTA X1=1
      plot/id=timePlot  /READOUT=+ONAXIS xlabel=ELEMENT ylabel=TIME X1=1
    pipe off

    call cleanupTimeCodeVisualFiles
    env set IOOptions origIOOptions
  endif
endmacro

!******************************************************************************!
! Main process message for this macro. Should be right after the ENDMACRO.
!******************************************************************************!
procedure processMessage m:msg
  if msg.name eqs "MAIN" then
    assert/text="^reg.main.name: User indicated tests" msg.data eqss "PASS"
    pipe stop
  elseif msg.name eqs "LABELS" then
    if labelsOn eqs "TRUE" then
      say "Disabling Plot Labels"
      res labelsOn "FALSE"
      res labelsOn
      set reg.3dplot.layers.3dplot.enable -LABELS
      set reg.eltimeplot.layers.elemtime.enable -LABELS
    else
      say "Enabling Plot Labels"
      set labelsOn "TRUE"
      res labelsOn
      set reg.3dplot.layers.3dplot.enable +LABELS
      set reg.eltimeplot.layers.elemtime.enable +LABELS
    endif
  elseif msg.name eqs "INPUTFILE" then
    if msg.data eqs "USER SPECIFIED" then
      set gc.filechooser.enabled true
      set gc.fileinputprompt.enabled true
    else
      set gc.filechooser.enabled false
      set gc.fileinputprompt.enabled false
      if example_files_ready EQS "FALSE" THEN
        call createExampleFiles
        res example_files_ready "TRUE"
      endif
      ! Set the input file to be analyzed
      if msg.data eqs "EXAMPLE TL GOOD" then
        call validateTimeline checktime_testcase_good
      elseif msg.data eqs "EXAMPLE TL WRAP NORMAL" then
        call validateTimeline checktime_testcase_wrap_normal
      elseif msg.data eqs "EXAMPLE TL WRAP OUT-OF-ORDER" then
        call validateTimeline checktime_testcase_wrap_order
      elseif msg.data eqs "EXAMPLE TL WRAP THOSE BEFORE WRAP DROPPED" then
        call validateTimeline  checktime_testcase_wrap_drop
      endif
      call set3DPlotInstructions
      call resetPlotsNewTimeline
      pause 0.1
      res labelsOn "FALSE"
      set reg.3dplot.layers.3dplot.enable -LABELS
      set reg.eltimeplot.layers.elemtime.enable -LABELS
    endif
  elseif msg.name eqs "FILECHOOSER" or msg.name eqs "FILEINPUTPROMPT" then
    ! Validate the new timeline, reset the instructions, and reset the plots
    call validateTimeline msg.data
    call set3DPlotInstructions
    call resetPlotsNewTimeline
  else
    say "Unhandled message name: ^msg.name from fid: ^msg.fid data: ^msg.data"
  endif
return

!******************************************************************************!
! If an unexpected exception is caught, mark test as failed, unless it is expected
!******************************************************************************!
procedure processException m:emsg
  if expectedException REXISTS then
    assert/text="^reg.main.name: Expected ERROR starts with: ^{expectedException}" expectedException EQSS emsg.data.message
    remove expectedException
  else
    assert/text="^reg.main.name: Caught unexpected exception ^{emsg.data}" "FALSE" isTrue
    if "Trace" SUBS ENV.DEBUG then invoke ,, emsg.data.printStackTrace()
  endif
return

!******************************************************************************!
! Generate timelines for each test case:
!   1.  Example good - Buffer with size to hold all timeline elements, so no wrapping
!   N/A Back-in-time - Good Timeline like #1, except with a back in time shift - current code autocorrects
!   2. Wrapped Timeline Normal - Takes last n elements of the good Timeline and
!                                copies them into a buffered Timeline in order
!                                (i.e. starting with buffer pointer)
!   3. Wrapped Timeline Drop   - Takes all the elements in the Order Timeline from
!                                the beginning of the buffer up until the buffer pointer
!                                and drops all the other elements after the buffer pointer
!                                and behind in time
!   N/A Wrapped Timeline Order  - Takes last n elements of the good Timeline and
!                                copies them into a buffered Timeline out of order
!                                (i.e. starting with the beginning of the buffer
!                                regardless of where the buffer pointer is) - current code autocorrects
!******************************************************************************!
procedure createExampleFiles
  env get IOOptions origIOOptions
  env set IOOptions +SuppressInvalidTimeLineEntryErrors

  ! Applies to all Timeline test cases
  set s:epochTL "2022:01:01::00:00:00.000000" ! used to set TimeLine
  new nxm.sys.lib.Time(^epochTL) startTime
  set timeInc 5E-2
  ! Applies to all wrapped Timeline test cases
  set  l:tl_wrap 128                                  ! Buffer size for wrapped Timelines
  calc l:off_with_time_shift tl_wrap 10 -
  calc l:half_wrap tl_wrap 2 /
  calc l:numEntries tl_wrap half_wrap +               ! Number of time hack samples
  set l:numElemBetweenEntries 128                     ! Number of elements between time hack entries
  calc l:numElements numEntries numElemBetweenEntries * ! Get the total number of elements in the Timeline
  invoke timeWhole startTime.getWSec()                ! Whole-number portion of start time
  set timeFrac 0.0                                    ! Fraction portion of start time

  ! Applies to Order and Dropped wrapped Timeline test cases
  set  l:numEntriesToDrop half_wrap                  ! Number of entries to drop when reading out of order
  calc l:numEntriesToKeep tl_wrap numEntriesToDrop - ! Number of entries to keep when reading out of order
  calc l:numKeysToDrop numEntriesToDrop 3 *          ! Number of keys to drop when reading out of order

  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  ! Populate time hacks for Good Timeline (Test Case #1) !
  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  waveform out=checktime_testcase_good elem=numElements start=startTime delta=timeInc
  do l:goodFileEntryNum 1 numEntries 1
    ! ((offset_# - 1) * numElemBetweenEntries) + 1
    calc elemOffset goodFileEntryNum 1 - numElemBetweenEntries * 1 +
    calc timeFrac timeFrac timeInc +
    while timeFrac >= 1
      calc timeWhole timeWhole 1 +
      calc timeFrac timeFrac 1 -
    endwhile
    keyword checktime_testcase_good put TCSAMPLE_^{goodFileEntryNum} elemOffset &
                                        TC_WHOLE_^{goodFileEntryNum} timeWhole &
                                        TC_FRAC_^{goodFileEntryNum}  timeFrac
  enddo

  keyword checktime_testcase_good tget normalTbl

  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  ! Populate time hacks for Back-In-Time Timeline (current code autocorrects -would need override) !
  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!  env get IOOptions origIOOptions
  !env set IOOptions +SameTimelineKeywords
!  info "GENERATING EXAMPLE #2 DATA"
!  set l:lastEntry numEntries
!  noop checktime_testcase_good checktime_testcase_back_in_time{TLL=-1} /TL=1
!  calc timeShift timeInc 20 *
!  calc l:wTimeShift timeShift
!  calc fTimeShift timeShift wTimeShift -
!  res timeShift
!  res normalTbl /all
!  do l:entry off_with_time_shift numEntries 1
!    res goodWhole normalTbl.TC_WHOLE_^{entry}
!    res goodWhole
!    res goodFrac normalTbl.TC_FRAC_^{entry}
!    res goodFrac
!    calc badWhole goodWhole wTimeShift -
!    calc badFrac goodFrac timeShift -
!    if badFrac LT 0 THEN
!      calc badFrac badFrac 1 +
!      calc badWhole badWhole 1 -
!    endif
!    keyword checktime_testcase_back_in_time put TC_WHOLE_^{entry} badWhole &
!                                                TC_FRAC_^{entry}  badFrac
!    say "after keyword"
!    !key checktime_testcase_back_in_time GET TC_WHOLE_^{entry} whole
    !key checktime_testcase_back_in_time GET TC_FRAC_^{entry} frac
!    say "entry ^entry goodWhole: ^goodwhole whole: ^whole goodfrac ^goodfrac frac: ^frac"
!  enddo
!  env set IOOptions origIOOptions

  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  ! Populate time hacks for Wrapped Normal Timeline (Test Case #2) !
  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  info "GENERATING EXAMPLE #3 DATA"
  noop checktime_testcase_good checktime_testcase_wrap_normal{TLL=^tl_wrap} /TL=1
  keyword checktime_testcase_wrap_normal tget wrapTbl

  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  ! Populate time hacks for Wrapped Timeline with time hacks before wrap dropped(Test Case #3) !
  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  info "GENERATING EXAMPLE #5 DATA"
  ! simulate this by using modulo
  calc l:numBeforePtr numEntries tl_wrap MOD
  noop checktime_testcase_good checktime_testcase_wrap_drop{TLL=^numBeforePtr} /TL=1
  !status/key checktime_testcase_wrap_drop

  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  ! Populate time hacks for Wrapped Out-Of-Order Timeline         !
  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  env get IOOptions origIOOptionsBeforeOutOfOrder
  env set IOOptions +SameTimelineKeywords
  info "GENERATING EXAMPLE #4 DATA"
 ! Simulate this by adding on the earlier keywords after the wrapped ones
  noop checktime_testcase_wrap_drop checktime_testcase_wrap_order{TLL=^numBeforePtr} /TL=1
  calc l:numAfterPtr tl_wrap numBeforePtr -
  calc l:bad_tc_num numBeforePtr 1 +
  do good_tc_num 1 numAfterPtr
    res badOffsetName TCSAMPLE_^{bad_tc_num}
    res badWholeName  TC_WHOLE_^{bad_tc_num}
    res badFracName   TC_FRAC_^{bad_tc_num}
    res goodOffsetName TCSAMPLE_^{good_tc_num}
    res goodWholeName  TC_WHOLE_^{good_tc_num}
    res goodFracName   TC_FRAC_^{good_tc_num}
    table wrapTbl GETKEY  ^goodOffsetName goodOffsetVal
    table wrapTbl GETKEY ^goodWholeName goodWholeVal
    table wrapTbl GETKEY ^goodFracName goodFracVal
    key checktime_testcase_wrap_order PUT ^badOffsetName ^goodOffsetVal ^badWholeName ^goodWholeVal ^badFracName ^goodFracVal
    calc good_tc_num good_tc_num 1 +
    calc bad_tc_num  bad_tc_num 1 +
  enddo
  env set IOOptions origIOOptionsBeforeOutOfOrder
   !status/key checktime_testcase_wrap_order

  env set IOOptions origIOOptions
return


!******************************************************************************!
! Validate the Timeline and output the results to a report file
!******************************************************************************!
procedure validateTimeline s:file
  info "checking timeline of ^file"
  status file XDELTA=file_xdelta YDELTA=file_ydelta TIMECODE=file_timecode TYPE=file_type 
  key file GET TC_PREC file_tcprec
  call getStartTimeFromInFile file

  set inBasename "^file"
  invoke l:lastIndex file.lastIndexOf(".")
  if lastIndex neq -1 then invoke inBasename file.substring(0,^lastIndex)

  set out    "^outSpecified"
  set csvout "^csvoutSpecified"
  set report "^reportSpecified"
  if out    eqs "" then set out    "^{inBasename}_out5000.tmp"
  if csvout eqs "" then set csvout "^{inBasename}_csv.csv"
  if report eqs "" then set report "^{inBasename}_report.txt"
  say "output file names"
  res out
  res csvout
  res report

  if out    fexists then erase out
  if csvout fexists then erase csvout
  if report fexists then erase report

  nxm.sys.intr.Timelineisvalid IN=^file OUT=^out CSVOUT=^csvout REPORT=^report RESULT="TL_VALID"
  res TL_VALID
  IF TL_VALID THEN
    res actual_color GREEN
    info "Your file ^file PASSED the Timelineisvalid check"
  ELSE
    res actual_color RED
    warn "Your file ^file FAILED the Timelineisvalid check"
  ENDIF
    set actual_color_tbl.^inBasename actual_color

return

!******************************************************************************!
! Get start time
!******************************************************************************!
procedure getStartTimeFromInFile s:file
  key ^file GET TIMELINE value_timeline TCSAMPLE_1 element_tc TC_WHOLE_1 tcwhole TC_FRAC_1 tcfrac

  if element_tc NISNULL THEN
    if element_tc EQ 1 THEN
      new nxm.sys.lib.Time(^tcwhole,^tcfrac) time_starttime
    else
      new nxm.sys.lib.Time(0) time_starttime
    endif
  elseif value_timeline NISNULL THEN
    if value_timeline.2 EQ 0 THEN
      res time_from_timeline value_timeline.3
      new nxm.sys.lib.Time(time_from_timeline) time_starttime
    else
      new nxm.sys.lib.Time(0) time_starttime
    endif
  else
    status ^in TIMECODE=file_timecode
    new nxm.sys.lib.Time(file_timecode) time_starttime
  endif
  invoke str_starttime time_starttime.toString(9)

return

!******************************************************************************!
! Set the instructions for the 2D plots with the new actual-color values
!******************************************************************************!
procedure set2DPlotInstructions
  set instructions2D "All plots should start with X=element=1 (far left)\n&
                      If not, the timeline has likely wrapped\n\n&
                      All plots should have their Y values increase as element increases\n\n&
                      1) TIME hack value vs Data ELEMENT plot\n\n&
                      2) Time hack ENTRY vs Data ELEMENT plot\n\n&
                      3) TDELTA vs Data ELEMENT\n   &
                         BLUE points - File tdelta\n   &
                         GREEN/RED points - effective tdelta since last hack\n\n&
                      4) TIME vs Data ELEMENT plot\n   &
                         BLUE points - anticipated time, based on last hack\n                 &
                                       and file tdelta\n   &
                         GREEN/RED points - time hack times\n   &
                         Note: A high level of zooming is normally required\n   &
                         to distinguish actual from expected."
  if reg.panel.controls.2dPlotsInstr rexists then set gc.2dPlotsInstr.value instructions2D
return

!******************************************************************************!
! Set the instructions for the 3D plot with the new calculated timeline values
!******************************************************************************!
procedure set3DPlotInstructions
  set instructions3D "This plot should also start on the far left and have all values increasing\n\n&
                      XYZ graph of BLUE TC TIMELINE VALUES\n\n&
                      Z axis - TC SAMPLE #\n&
                      Y axis - ELEMENT #\n&
                      X axis - TIME\n\n&
                      ZOOM in AND Cntrl-LMB on lowest sample\n\n&
                      A valid Timeline would have Y of 1 for Z of 1\n\n&
                      File specs:\n&
                      XDELTA     - ^{file_xdelta}\n&
                      YDELTA     - ^{file_ydelta}\n&
                      Start Time - ^{str_starttime}\n&
                      TC_PREC    - ^{file_tcprec}"
  if reg.panel.controls.3dPlotInstr rexists then set gc.3dplotinstr.value instructions3D
return

!******************************************************************************!
! Reset the plot qualifiers based on the actual color corresponding to the current Timeline
!******************************************************************************!
procedure refreshPlotQualifiers
  res elem_vs_time_quals {HEADERROWS=L:1,^columnnames,TAG=ELEMTIME,LAYER={NAME="ELEMTIME",LT="T5",DATATYPE="XYZ",Y="TIME",X="ELEM",Z="ENTRY",LABEL="TIMESTR",&
                          METADATAFIELDS="EXP_TIMESTR|ENTRY",AutoselectEnable=CLICK,AutoSelectMode=PIXEL,LINE={COLOR=^actual_color},SELECTCOLOR={COLOR="red",TYPE="symbol"}}}
  res ent_vs_off_quals {HEADERROWS=L:1,^columnnames,TAG=ENTOFF,LAYER={NAME="ENTOFF",LT="T5",DATATYPE="XYZ",Y="ENTRY",X="ELEM",Z="TIME",LABEL="TIMESTR",&
                          METADATAFIELDS="TIMESTR|EXP_TIMESTR",AutoselectEnable=CLICK,AutoSelectMode=PIXEL,LINE={COLOR=^actual_color},SELECTCOLOR={COLOR="red",TYPE="symbol"}}}
  res actTDelta_quals {HEADERROWS=L:1,^columnnames,TAG=ACTTDELTA,LAYER={NAME="ACTTDELTA",LT="T5",DATATYPE="XYZ",Y="ACT_TDELTA",X="ELEM",Z="TIME",LABEL="ACT_TDELTA",&
                          METADATAFIELDS="TIMESTR|ENTRY",AutoselectEnable=CLICK,AutoSelectMode=PIXEL,LINE={COLOR=^actual_color},SELECTCOLOR={COLOR="red",TYPE="symbol"}}}
  res fileTDelta_quals {HEADERROWS=L:1,^columnnames,TAG=FILETDELTA,LAYER={NAME="FILETDELTA",LT="T5",DATATYPE="XYZ",Y="FILE_TDELTA",X="ELEM",Z="TIME",LABEL="FILE_TDELTA",&
                          METADATAFIELDS="EXP_TIMESTR|ENTRY",AutoselectEnable=CLICK,AutoSelectMode=PIXEL,LINE={COLOR=BLUE},SELECTCOLOR={COLOR="blue",TYPE="symbol"}}}
  res expTime_quals {HEADERROWS=L:1,^columnnames,TAG=EXPTIME,LAYER={NAME=EXPTIME,LT="T5",DATATYPE="XYZ",Y="EXP_TIME",X="ELEM",Z="ENTRY",&
                          METADATAFIELDS="TOLR|ENTRY",AutoselectEnable=CLICK,AutoSelectMode=PIXEL,LINE={COLOR=BLUE},SELECTCOLOR={COLOR="red",TYPE="symbol"}}}
  res actTime_quals {HEADERROWS=L:1,^columnnames,TAG=ACTTIME,LAYER={NAME=ACTTIME,LT="T5",DATATYPE="XYZ",Y="TIME",X="ELEM",Z="ENTRY",&
                          METADATAFIELDS="EXP_TIMESTR|ENTRY",AutoselectEnable=CLICK,AutoSelectMode=PIXEL,LINE={COLOR=^actual_color},SELECTCOLOR={COLOR="blue",TYPE="symbol"}}}
return

!******************************************************************************!
! Reset all plots for the new timeline
!******************************************************************************!
procedure resetPlotsNewTimeline
  call refreshPlotQualifiers
  ! Redraw all plots
  message func=send id=3dplot     NAME=openfile info=-1 data="^out{TAG=3DPLOT,Layer={NAME="3DPLOT",LT=T5,DATATYPE="XYZ",METADATAFIELDS="EXP_TIMESTR|ELEM|ENTRY",AutoselectEnable=CLICK,AutoSelectMode=PIXEL,LINE={COLOR=^actual_color},SELECTCOLOR=AQUA,AutoSelectMode=PIXEL,SELECTCOLOR=AQUA}}"
  message func=send id=eltimeplot NAME=openfile info=-1 data="^csvout^elem_vs_time_quals"
  message func=send id=entOffPlot NAME=openfile info=-1 data="^csvout^ent_vs_off_quals"
  message func=send id=tDeltaPlot NAME=openfile info=-1 data="^csvout^fileTDelta_quals|^csvout^actTDelta_quals"
  message func=send id=timePlot   NAME=openfile info=-1 data="^csvout^expTime_quals|^csvout^actTime_quals"
return

!******************************************************************************!
! Cleanup files generated, unless we are keeping them for reference
!******************************************************************************!
procedure cleanupTimeCodeVisualFiles

    ! cleanup test files (if user did not added switch to keep test files)
    if keep isFalse then
      IF checktime_testcase_good         FEXISTS THEN ERASE checktime_testcase_good
      IF checktime_testcase_back_in_time FEXISTS THEN ERASE checktime_testcase_back_in_time
      IF checktime_testcase_wrap_normal  FEXISTS THEN ERASE checktime_testcase_wrap_normal
      IF checktime_testcase_wrap_order   FEXISTS THEN ERASE checktime_testcase_wrap_order
      IF checktime_testcase_wrap_drop    FEXISTS THEN ERASE checktime_testcase_wrap_drop
    else
      info "Not erasing these files: ^{checktime_testcase_good},^{checktime_testcase_back_in_time},&
                                     ^{checktime_testcase_wrap_normal},^{checktime_testcase_wrap_order},&
                                     ^{checktime_testcase_wrap_drop}"
    endif
return
