/*
   Example non-Midas threaded acquisition service for ICE cards 

   Author: Jeff Schoen

*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <signal.h>
#include <unistd.h>

#include "icedefs.h"
#include "icelib.h"
#include "niolib.h"
#include "icehalo.h"

#define MAXCARD 4
#define MAXDMA  16
#define MAXLINE 256

void *dmaRun (void *vp);
int_4 dmaSetup (PICSTRUCT *p);
int_4 dmaStart (PICSTRUCT *p);
int_4 dmaStop (PICSTRUCT *p);
int_4 coreSetup (PICSTRUCT *p, char *config);
int_4 runCore (PICSTRUCT *p, char *config);
int_4 quiet,remote,ctrlc;
void cleanup_srvice();

PICSTRUCT pc[MAXCARD],pd[MAXDMA];

int main (int argc, char *argv[])
{
  char *fname,*flags,*raddr,*s,*cmd,line[MAXLINE],stmp[MAXLINE],config[MAXLINE];
  char type[8],kname[80],*kvalue,ktype,c,*av[16];
  int_4 i,j,n,ls,lv,ic,na,func,key,index,cindex,status=0,ncard=0,nport=0;
  int_4 sock=0,Mcbreak=0;
  PICSTRUCT *p;
  double dv;
  FILE *fp=NULL;

  if (argc <= 1) {
    printf("Syntax: srvice <cfgFile> <flags> \n");
    printf("        <cfgFile> = config.cfg \n");
    printf("        <flags>   = VERBOSE|QDRX|QUIET... or ON \n");
    printf("        <remote>  = localif:port/remote:port \n");
    printf(" case sensitive. See ../hlp/srvice.hlp. \n");
    exit(0);
  }

  fname = argv[1];
  flags = argv[2];
  raddr = argv[3];

  quiet  = (flags!=NULL) && (strstr(flags,"QUIET")!=NULL);
  remote = (flags!=NULL) && (strcasecmp(flags,"ON")==0);

  if (strstr(fname,"udp:")!=NULL) {
    sock = nioOpen (fname,0,NIO_RMIF);
    if (sock<=0) { printf("Err opening socket=%s status=%d\n",fname,sock); goto NOFILE; }
  } else {
    fp = fopen(fname, "r");
    if (fp==NULL) { printf("Err opening config file: %s\n",fname); goto NOFILE; }
  }
  if (remote) {
    sock = nioOpen (raddr,0,NIO_RMIF);
    if (sock<=0) { printf("Err opening socket=%s status=%d\n",raddr,sock); goto NOFILE; }
  }

  for (i=0; i<MAXCARD; i++) pc[i].state=-1;
  for (i=0; i<MAXDMA; i++)  pd[i].state=-1;

  ctrlc=0;
  signal (SIGINT,cleanup_srvice);

  for (n=1; !ctrlc; n++) {

    if (fp!=NULL) {							// from file
      if (fgets(line,MAXLINE,fp)==NULL) break;				// read next line
      ls = strlen(line)-1;						// minus the LF
    }
    else if (sock>0) {							// from network client
      TRYAGAIN:
      ls = nioMsgRecv(sock,&func,line,MAXLINE,NIO_RMIF);
      if (ls<0) break; if (ls==0) { usleep(10000); goto TRYAGAIN; }	// read next packet
      if (line[0]=='{' && line[ls-1]=='}') { ls-=2; memmove(line,line+1,ls); line[ls]=0; }
    }
    line[ls]=0;								// terminate string
    strcpy(stmp,line);
    for (i=ic=na=0; i<ls; i++) { c=stmp[i];				// parse na args
      if (ic==0 && c>' ') { av[na++]=stmp+i; ic=1; }			// start of next arg
      if (ic==1 && c<=' ') { stmp[i]=0; ic=0; }				// end of arg
    }
    if (na==0) continue;						// blank line
    cmd = av[0];							// command is 1st arg
    if (cmd[0]=='!' || cmd[0]=='/' || cmd[0]=='#') continue;		// skip comment
    index = (na>1)? atol(av[1]) : -1;					// mostly common index arg

    if (!quiet) printf("[%d] Serving: %s\n",n,line);

    if (remote) {							// send to remote srvice
      status = nioMsg(sock,0,NIO_SET,line,ls,NIO_RMIF);			// send to remote srvice
    }
    else if (strcmp(cmd,"CARD")==0) {					// define a card
      strcpy(config,av[2]);
      if (flags!=NULL) { strcat(config,","); strcat(config,flags); }	// add flags
      status = pic_open (&pc[index], config, &Mcbreak, 0);
    }
    else if (strcmp(cmd,"DMA")==0) {					// define a DMA channel
      strcpy(config,av[2]);
      i=sscanf(config,"CARD=%d,",&cindex);				// if CARD=# in string, 
      if (i==1) sprintf(config,"%s,%s",pc[cindex].config,av[2]);	// sub into config string
      status = pic_open (&pd[index], config, &Mcbreak, 0);
      pd[index].dmaci = dmaSetup(&pd[index]);
    }
    else if (strcmp(cmd,"CORE")==0) {					// run a CORE setup
      strcpy(config,av[2]);
      i=sscanf(config,"CARD=%d,",&cindex);				// if CARD=# in string, 
      if (i==1) sprintf(config,"%s,%s",pc[cindex].config,av[2]);	// sub into config string
      status = pic_open (&pd[index], config, &Mcbreak, 0);
      pd[index].dmaci = coreSetup(&pd[index],config);
    }
    else if (strcmp(cmd,"RESET")==0) {					// reset cards
      for (i=1; i<na; i++) { 
	index=atol(av[i]);
	pic_reset(&pc[index],0);
      }
    }
    else if (strcmp(cmd,"SET")==0) {					// set a runtime parameter
      p = pd+index;
      for (i=2; i<na; i++) {
	strcpy (kname,av[i]);
	kvalue = strchr(kname,'=');
        if (kvalue==NULL) { printf("Error: No value for SET of %s\n",kname); continue; }
        kvalue[0]=0; kvalue++; 
	key=pic_name2key(p,kname);					// also turns "CORE+0xABCD" to KEY_CORE+off
	ktype=pic_getkeytype(p,key);
	if (ktype=='D') pic_setkeyd(p,p->dmaci,key,atof(kvalue)); 
	if (ktype=='L') pic_setkeyl(p,p->dmaci,key,atol(kvalue)); 
	if (ktype=='A') pic_setkey(p,p->dmaci,key,kvalue,strlen(kvalue));
      }
    }
    else if (strcmp(cmd,"GET")==0) {					// get a runtime parameter
      p = pd+index;
      for (i=2; i<na; i++) {
	strcpy (kname,av[i]);
	kvalue = strchr(kname,'=');
	if (kvalue!=NULL) { printf("Error: Has value for GET of %s\n",kname); continue; }
	key=pic_name2key(p,kname);					// also turns "CORE+0xABCD" to KEY_CORE+off
	ktype=pic_getkeytype(p,key);
	if (ktype=='D') { dv=pic_getkeyd(p,p->dmaci,key); sprintf(kvalue,"%f",dv); }
	if (ktype=='L') { lv=pic_getkeyl(p,p->dmaci,key); sprintf(kvalue,"%d",lv); }
	if (ktype=='A') { ls=pic_getkey(p,p->dmaci,key,kvalue,40); }
	sprintf(line,"%s=%s",kname,kvalue);
      }
    }
    else if (strcmp(cmd,"START")==0) {					// start DMA channels
      for (i=1; i<na; i++) { 
	index=atol(av[i]);
	usleep(10000);
	dmaStart(pd+index);
      }
    }
    else if (strcmp(cmd,"PAUSE")==0) {					// pause for N seconds
      i = (int)(atof(av[1])*1000000);
      usleep(i);
    }
    else if (strcmp(cmd,"WAIT")==0) {					// wait for user interrupt
      if (index<0) {
	printf("Hit return to continue script\n");
	getchar();
      } 
      else for (i=1; i<na; i++) { 
	index=atol(av[i]);
	while (pd[index].todo!=0) usleep(10000);
      }
    }
    else if (strcmp(cmd,"STOP")==0) {					// stop a DMA channel
      for (i=1; i<na; i++) { 
	index=atol(av[i]);
	dmaStop(pd+index);
      }
    }
    else if (strcmp(cmd,"LOG")==0) {					//
    }
    else if (strcmp(cmd,"ECHO")==0) {					//
      printf("%s\n",line+5);
    }
    else if (strcmp(cmd,"NOOP")==0) {					//
    }
    else if (strcmp(cmd,"EXIT")==0) {					//
    }
    else printf("** Unrecognized srvice command at line=%d **\n",n);

    if (fp==NULL && sock>0) {						// send ACK to client
      ls = nioMsgSend(sock,NIO_ACK,line,ls,NIO_RMIF);
    }
    if (strcmp(cmd,"EXIT")==0) 	break;
  }

  DONE:
  if (fp!=NULL) fclose(fp);
  if (sock>0) nioClose(sock);
  for (i=0; i<MAXDMA; i++)  if (pd[i].state>=0) pic_close(pd+i);
  for (i=0; i<MAXCARD; i++) if (pc[i].state>=0) pic_close(pc+i);
  NOFILE:
  exit (status);
}

void cleanup_srvice() {
  int_4 i;
  printf("Aborting DMAs\n");
  for (i=0; i<MAXDMA; i++) if (pd[i].state>=0) dmaStop(pd+i); 
  for (i=0; i<MAXCARD; i++) if (pc[i].state>=0) pic_close(pc+i);
  ctrlc=1;
}

int_4 dmaSetup (PICSTRUCT *p) {
  double rate=10.0,freq=0.0;
  int_4 bits=-16,dec=1,dir=-1,gain=0,flag=0; 
  int_4 dmac,status,bpe;
  int_8 nbytes;
  char form[4];

  if (pic_getstrflag(p,"FORM",form,1)==2) {
    bits = (form[1]=='N')? 4 : (form[1]=='B')? 8 : (form[1]=='J')? 12 : (form[1]=='I')? 16 : 32;
    bits = (form[0]=='C')? -bits : bits;
  }

  bits = pic_getintflagdef(p,"BITS",bits);
  dec  = pic_getintflagdef(p,"DEC",dec);
  dir  = pic_getintflagdef(p,"PLAY",-1);
  dir  = pic_getintflagdef(p,"DIR",dir);
  gain = pic_getintflagdef(p,"GAIN",gain);
  flag = pic_getintflagdef(p,"FLAG",flag);
  rate = pic_getdblflagdef(p,"RATE",rate);
  freq = pic_getdblflagdef(p,"FREQ",freq);
  flag = pic_getintflagdef(p,"FLAG",flag);

  dmac = pic_ioport (p,-1,-1,-1, dir, bits, (int)(rate*1e6), freq/rate, dec, gain, flag);
  if (dmac<0) { printf("Error opening ioport\n"); return -1; }

  return dmac;
}

int_4 dmaStart (PICSTRUCT *p) {
  int_4 nbytes, status;
  int_4 dmac = p->dmaci;
  int_4 rate = p->dma[dmac-1].rate;
  int_4 dec  = p->dma[dmac-1].dec;
  int_4 dir  = p->dma[dmac-1].dir;
  int_4 bits = p->dma[dmac-1].bits;
  int_4 bpe = (bits<0)? -bits/4 : bits/8;
  int_4 xfer = pic_getintflagdef(p,"XFER",0x10000);
  double len = pic_getdblflagdef(p,"LENGTH",1.0);

  if (p->ptype!=IOPT_MODULE) { rate /= dec; if (bits<0) rate /= 2; } // handle complex decimation 
  nbytes = (int_8)(rate*bpe*len);
  nbytes = (nbytes-xfer-1)/xfer*xfer;		// round to XFER length
  p->mapi.bytes = nbytes;

  status = pic_mapmem (p, &p->mapi, 0, 1);
  if (status<0) { printf("Error mapping memory of %d bytes\n",nbytes); return -1; }
  status = pic_dmasetup (p, dmac, dir, &p->mapi, -1, 0);
  if (status<0) { printf("Error setting up DMA\n"); return -1; }

  status = pthread_create(&p->threadid2,NULL,dmaRun,(void*)p);
  while (p->todo==0) usleep(10000);			// wait for thread to startup

  return dmac;
}

int_4 dmaStop (PICSTRUCT *p) {
  int_4 status;
  p->todo=DMA_STOP;
  status = pthread_join(p->threadid2,NULL);
  status = pic_mapmem (p, &p->mapi, 0, -1);
  if (status<0) { printf("Error unmapping memory of %d bytes\n",p->mapi.bytes); return -1; }
  return 0;
}

int_4 coreSetup (PICSTRUCT *p, char *config) {
  int_4 ls,dmac,dir=-1,bits=-16,rate=-1,dec=1,gain=0,flag=0;
  char cname[20],cimpl[8],ccfg[40],*s;
  double freq=0;
  CORE *core;
  if (pic_getstrflag(p,"NAME",cname,0)<0) printf("Missing NAME parameter in %s\n",config);
  if (pic_getstrflag(p,"IMPL",cimpl,0)<0) strcpy(cimpl,"CPU");
  printf("Ran Core=%s using config=%s\n",cname,ccfg,config);
  dmac = pic_ioport (p,-1,-1,-1, dir, bits, rate, freq, dec, gain, flag);
  if (dmac<0) { printf("Error opening ioport for %s\n",config); return -1; }
  core = (CORE*)core_alloc(cname,cimpl,config);
  ls=pic_getstrflag(p,"CFG",ccfg,0);
  if (ls>0) core_setkeytable(core,ccfg);
  core_open(core);
  core_close(core);
  printf("Ran Core=%s Cfg=%s using config=%s\n",cname,ccfg,config);
  return dmac;
}

void *dmaRun (void *vp) {
  PICSTRUCT *p = (PICSTRUCT*)vp;
  FILE *fp=NULL;
  char fn[128],func[16];
  int_4 cycle=0,floop=0,irate,dir,lfn,lfu,fstat,status,xfer=0x10000,ok2r=0;
  int_8 index=0,ind;
  double time1,time2;
  float drate;
  void *data = (void*)p->mapi.vaddr;
  int_8 nbytes = (int) p->mapi.bytes;

  dir  = pic_getintflagdef(p,"PLAY",-1);
  dir  = pic_getintflagdef(p,"DIR",dir);
  xfer = pic_getintflagdef(p,"XFER",xfer);
  lfn  = pic_getstrflag(p,"FILE",fn,-1); if (lfn<=0) { strcpy(fn,"NONE"); }
  lfu  = pic_getstrflag(p,"FUNC",func,0); if (lfu<=0) { strcpy(func,"CONT"); lfu=4; }

  if (lfn>0) fp = fopen(fn, (dir>0)? "r":"w" );
  if (lfn>0 && fp==NULL) printf("Err: Opening file %s\n",fn);

  if (strncmp(func,"CONTINUOUS",lfu)==0) p->todo=DMA_CONTINUOUS;
  else if (strncmp(func,"ONESHOT",lfu)==0) p->todo=DMA_ONESHOT;
  else if (strncmp(func,"ONDEMAND",lfu)==0) p->todo=DMA_ONDEMAND;
  else p->todo = atoi(func);

  if (dir>0 && fp!=NULL) status = fread(data,1,nbytes,fp);	// preread output buffer

  if (!quiet) printf("Starting DMA devno=%d ptype=%d pindex=%d file=%s func=%s todo=%d\n",p->devno,p->ptype,p->pindex,fn,func,p->todo);
  pic_dmafunc(p,p->dmaci,p->todo);
  time1 = pic_time();
  while (p->todo!=0) {
    ind = pic_dmafuncx(p,p->dmaci,DMA_INDEX);
    if (ind<index) {
      cycle = pic_dmafunc(p,p->dmaci,DMA_CYCLE);
      time2 = pic_time();
      drate = ( 1.0e-6 * p->mapi.bytes / (time2-time1) );
      irate = (int)(drate*1.008);
      time1 = time2;
      if (!quiet) printf("Cycle=%d DMA devno=%d ptype=%d pindex=%d rate=%d Mby/s\n",cycle,p->devno,p->ptype,p->pindex,irate);
    }
    index = ind;
    if (fp==NULL) {
      usleep(50000);
      if (pic_dmafunc(p,p->dmaci,DMA_POLL)==0) p->todo=DMA_STOP;
    } else {
      if (dir>0) ok2r = (p->todo==DMA_CONTINUOUS) || (p->todo==DMA_CONTINUOUS) || (p->todo>1);
      status = pic_dmaxptr (p, p->dmaci, &data, xfer, FLG_NOWRAP);	// blocks until xfer bytes ready, returns status or length of contiguous data ptr available
      if (status<0) {	fstat = status; p->todo=0; }		// transfer complete
      else if (dir<0)	fstat = fwrite(data,1,status,fp);	// flush next full region to disk
      else if (ok2r)	fstat = fread(data,1,status,fp);	// load newly opened region
      else fstat = status;
      if (ok2r && fstat>=0 && fstat<status) { fseek(fp,0L,SEEK_SET); floop++; fstat+=fread(data+fstat,1,status-fstat,fp); } // loop on input file
      if (fstat!=status) printf("Err: Read|Write status=%d of %d bytes from %s\n",fstat,status,fn);
    }
  }
  if (!quiet) printf("Stopping DMA devno=%d ptype=%d pindex=%d fileloops=%d\n",p->devno,p->ptype,p->pindex,floop);
  pic_dmafunc(p,p->dmaci,DMA_STOP);
  pic_dmafunc(p,p->dmaci,DMA_RESET);
  if (fp!=NULL) fclose(fp);
}

