/*
   Non-Midas acquisition routine for ICE-PIC card

   Author: Jeff Schoen

*/

#define DEVN "%s,ICEPIC,DEVNO=2,IOM1=LB2D,IOM2=LB2D,PM1=K8M,PM2=K8M,MUXCLK=P,%s,"
//#define DEVN "%s,ICEPIC,DEVNO=2,IOM1=LB2D,IOM2=A2DR13,PM1=DTDM,PM2=DTDM,MUXCLK=P,%s,"
//#define DEVN "%s,ICEPIC,DEVNO=2,IOM1=LB2D,IOM2=A2DR13,PM1=DTDM,PM2=V6M,MUXCLK=P,%s,"
//#define DEVN "%s,ICEPIC,DEVNO=0,IOM1=DR2D,IOM2=DR2D,PM1=DTDMX,PM2=DTDMX,MUXCLK=P,IOC=IIRW,%s,"
//#define DEVN "%s,ICEPIC,DEVNO=0,IOM1=LB2D,IOM2=A2DR13,PM1=V6M,PM2=ZPPM,MUXCLK=P,%s,"
//#define DEVN "%s,ICEPIC,DEVNO=0,IOM=A2DR13,PM=V6M,MUXCLK=P,%s,"

#define SRATE 100000000
#define TFREQ 10.7e6
#define BUFWB 0x04000000	/* 64M */
#define BUFNB 0x00400000	/* 16M */
#define DEC 32

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "icedefs.h"
#include "icelib.h"

static int mpsetup (PICSTRUCT *p, int_4 ports, int_4 ptype, int_4 pindex, int_4 dir, int_4 bits, int_4 rate, int_4 dec, double freq, int_4 gain);
static int mptransfer (PICSTRUCT *p, int_4 ports, int_4 ptype, int_4 pindex, int_4 dir, int_4 bits, int_4 rate, int_4 dec, double freq, int_4 gain, int_4 size, int_4 nbuf, char *fnam);

int main (int argc, char *argv[])
{
  PICSTRUCT  p,pa,pb,pc,pd;
  char *flags,*func,devname[128],noflags[1];
  int_4 Mcbreak = 0;
  int_4 status,nbuf,nmod,ntun,ncor,mcor;

  strcpy(noflags,"");
  pa.state = pb.state = pc.state = pd.state = 0;

  if (argc <= 2 || argc > 4) {
    BAIL:
    printf("Syntax: example <func> <nbuf> <flags>\n");
    printf("          <func>    = WB|NB|1|3|4|5|6|7 (case sensitive) \n");
    printf("          <nbuf>    = number of buffers for streaming to/from file (- to bypass file)\n");
    printf("          <flags>   = extra flags like VERBOSE|IOC=IIR|INP=1 etc\n\n");
    printf("    func WB = ModX -> Host\n");
    printf("    func NB = TunerY -> Host\n");
    printf("    func 1  = ModX -> Pm1TunerY -> Pm1CoreX -> Pm2MCoreY -> Host\n");
    printf("    func 2  = ModX -> Pm1TunerY -> Host , Host -> Pm2MCoreY -> Host\n");
    printf("    func 3  = ModX -> Pm1CoreX -> Host , ModX -> Pm2MCoreX -> Host\n");
    printf("    func 4  = ModX -> Pm1CoreX -> Pm2MCoreY -> Host\n");
    printf("    func 5  = ModX -> Pm1TunerY -> Pm1CoreX -> Host\n");
    printf("    func 6  = ModX -> Pm1TunerY -> Pm1CoreX -> Pm2CoreX -> Host\n");
    printf("    func 7  = ModX -> Pm1TunerY -> Pm1CoreX -> Pm2MCoreX -> Pm2CoreX -> Host\n");
    printf("    func 8  = ModX -> Pm2TunerY -> Pm2CoreX -> Host\n");
    printf("\n");
    exit(0);
  }

  func = argv[1]; 
  nbuf = atoi(argv[2]); 
  flags = (argc==4)? argv[3] : noflags;

  sprintf(devname,DEVN,flags,"");
  status = pic_open (&p, devname, &Mcbreak, 0);
  if (status<=0) { printf ("Error opening device - aborted\n"); goto DONE; }

  printf("Resetting config=%s\n",devname);
  status = pic_reset (&p, FLG_BOOT); /* reset the entire card */
  if (status<=0) { printf ("Error rebooting device - aborted\n"); goto DONE; }

  nmod = 2;
  ntun = pic_getkeyl (&p, 0, KEY_CHNS);
  ncor = 2;
  mcor = 16;

  if (strcmp(func,"NB")!=0 && ntun>32) ntun=32;

  printf("Initializing example=%s\n",func);
  if (strcmp(func,"WB")==0) {
    status = mptransfer (&p,nmod,IOPT_MODULE,1,-1,16,SRATE,1,TFREQ,0, BUFWB,nbuf,"outmod"); 
  }    
  else if (strcmp(func,"NB")==0) {
    status = mptransfer (&p,ntun,IOPT_TUNER,1,-1,-16,SRATE,DEC,TFREQ,0, BUFNB,nbuf,"outtun"); 
  }
  else if (strcmp(func,"1")==0) {
    sprintf(devname,DEVN,flags,"OPORT=NONE");
    status = pic_open (&pa, devname, &Mcbreak, 0);
    status = mpsetup (&pa,ntun,IOPT_TUNER,1,-1,-16,SRATE,DEC,TFREQ,0); 
    sprintf(devname,DEVN,flags,"OPORT=NONE,IPORT=TUNERX");
    status = pic_open (&pb, devname, &Mcbreak, 0);
    status = mpsetup (&pb,ncor,IOPT_CORE,11,-1,-16,SRATE,1,TFREQ,0); 
    sprintf(devname,DEVN,flags,"IPORT=CORE1X");
    status = pic_open (&pc, devname, &Mcbreak, 0);
    status = mptransfer (&pc,ncor,IOPT_MCORE,33,-1,16,SRATE,1,TFREQ,0, BUFNB*ntun,nbuf,"outcor"); 
  }
  else if (strcmp(func,"2")==0) {
    sprintf(devname,DEVN,flags,"OPORT=NONE,PMI=0"); // route through PM0COREX in noop mode
    status = pic_open (&pa, devname, &Mcbreak, 0);
    status = mptransfer (&pa,ncor,IOPT_CORE,1, 1,16,SRATE,1,TFREQ,0, BUFWB,0,"incor"); 
    sprintf(devname,DEVN,flags,"IPORT=PM0COREX");
    status = pic_open (&pb, devname, &Mcbreak, 0);
    status = mptransfer (&pb,ncor,IOPT_CORE,11,-1,16,SRATE,1,TFREQ,0, BUFWB,nbuf,"outcor"); 
  }
  else if (strcmp(func,"3")==0) {
    sprintf(devname,DEVN,flags,"");
    status = pic_open (&pa, devname, &Mcbreak, 0);
    status = mpsetup (&pa,ncor,IOPT_CORE,11,-1,16,SRATE,1,TFREQ,0); // no file IO on these
    sprintf(devname,DEVN,flags,"");
    status = pic_open (&pb, devname, &Mcbreak, 0);
    status = mptransfer (&pb,ncor,IOPT_MCORE,33,-1,16,SRATE,1,TFREQ,0, BUFWB,nbuf,"outmcor"); 
  }
  else if (strcmp(func,"4")==0) {
    sprintf(devname,DEVN,flags,"OPORT=NONE");
    status = pic_open (&pa, devname, &Mcbreak, 0);
    status = mpsetup (&pa,ncor,IOPT_CORE,11,-1,16,SRATE,1,TFREQ,0); 
    sprintf(devname,DEVN,flags,"IPORT=CORE1X");
    status = pic_open (&pb, devname, &Mcbreak, 0);
    status = mptransfer (&pb,mcor,IOPT_MCORE,33,-1,16,SRATE,DEC,TFREQ,0, BUFNB,nbuf,"outmcor"); 
  }
  else if (strcmp(func,"5")==0) {
    sprintf(devname,DEVN,flags,"OPORT=NONE");
    status = pic_open (&pa, devname, &Mcbreak, 0);
    status = mpsetup (&pa,ntun,IOPT_TUNER,1,-1,-16,SRATE,DEC,TFREQ,0); 
    sprintf(devname,DEVN,flags,"IPORT=TUNERX");
    status = pic_open (&pb, devname, &Mcbreak, 0);
    status = mptransfer (&pb,ncor,IOPT_CORE,1,-1,16,SRATE,1,TFREQ,0, BUFNB*ntun,nbuf,"outcor"); 
  }
  else if (strcmp(func,"6")==0) {
    sprintf(devname,DEVN,flags,"OPORT=NONE");
    status = pic_open (&pa, devname, &Mcbreak, 0);
    status = mpsetup (&pa,ntun,IOPT_TUNER,1,-1,-16,SRATE,DEC,TFREQ,0); 
    sprintf(devname,DEVN,flags,"OPORT=NONE,IPORT=TUNERX");
    status = pic_open (&pb, devname, &Mcbreak, 0);
    status = mpsetup (&pb,ncor,IOPT_CORE,11,-1,-16,SRATE,1,TFREQ,0); 
    sprintf(devname,DEVN,flags,"IPORT=CORE1X");
    status = pic_open (&pc, devname, &Mcbreak, 0);
    status = mptransfer (&pc,ncor,IOPT_CORE,21,-1,16,SRATE,1,TFREQ,0, BUFNB*ntun,nbuf,"outcor"); 
  }
  else if (strcmp(func,"7")==0) {
    sprintf(devname,DEVN,flags,"OPORT=NONE");
    status = pic_open (&pa, devname, &Mcbreak, 0);
    status = mpsetup (&pa,ntun,IOPT_TUNER,1,-1,-16,SRATE,DEC,TFREQ,0); 
    sprintf(devname,DEVN,flags,"OPORT=NONE,IPORT=TUNERX");
    status = pic_open (&pb, devname, &Mcbreak, 0);
    status = mpsetup (&pb,ncor,IOPT_CORE,11,-1,-16,SRATE,1,TFREQ,0); 
    sprintf(devname,DEVN,flags,"OPORT=NONE,IPORT=PM1COREX");
    status = pic_open (&pc, devname, &Mcbreak, 0);
    status = mpsetup (&pc,ncor,IOPT_MCORE,33,-1,16,SRATE,1,TFREQ,0); 
    sprintf(devname,DEVN,flags,"IPORT=MCOREX");
    status = pic_open (&pd, devname, &Mcbreak, 0);
    status = mptransfer (&pd,ncor,IOPT_CORE,21,-1,16,SRATE,1,TFREQ,0, BUFNB*ntun,nbuf,"outcor"); 
  }
  else if (strcmp(func,"8")==0) {
    sprintf(devname,DEVN,flags,"PMI=2,OPORT=NONE");
    status = pic_open (&pa, devname, &Mcbreak, 0);
    status = mpsetup (&pa,ntun,IOPT_TUNER,1,-1,-16,SRATE,DEC,TFREQ,0); 
    sprintf(devname,DEVN,flags,"PMI=2,IPORT=PM2TUNERX");
    status = pic_open (&pb, devname, &Mcbreak, 0);
    status = mptransfer (&pb,ncor,IOPT_CORE,1,-1,16,SRATE,1,TFREQ,0, BUFNB*ntun,nbuf,"outcor"); 
  }
  else { 
    printf("Function=%s not coded yet\n",func); 
    goto DONE;
  }

  if (status == -1) printf("Err setting up ioport\n");
  if (status == -2) printf("Err allocating memory\n");
  if (status == -3) printf("Err setting up dma\n");

  DONE:
  pic_close (&p);
  if (pa.state!=0) pic_close(&pa);
  if (pb.state!=0) pic_close(&pb);
  if (pc.state!=0) pic_close(&pc);
  if (pd.state!=0) pic_close(&pd);
  exit (status);
}


/* kick off variables for mpsetup channels */
#define MAXDMA 80
PICSTRUCT *ko_p[MAXDMA];
int_4  ko_dmac[MAXDMA];
int_4  ko_func[MAXDMA];
DMAMAP ko_map[MAXDMA];
int_4  ko_dmacs=0;

/*
  multi-port setup only
  internal port route does not transfer to/from host
*/
int mpsetup (PICSTRUCT *p, int_4 ports, int_4 ptype, int_4 pindex, int_4 dir, int_4 bits, int_4 rate,
				int_4 dec, double freq, int_4 gain) {
    return mptransfer (p, ports, ptype, pindex, dir, bits, rate, dec, freq, gain, 0, 0, NULL);
}

/*
  multi-port transfer
  assume all ports running at same rate 
  waits for all dma channels ready before transfer to collapse loop logic
*/
int mptransfer (PICSTRUCT *p, int_4 ports, int_4 ptype, int_4 pindex, int_4 dir, int_4 bits, int_4 rate,
				int_4 dec, double freq, int_4 gain, int_4 size, int_4 nbuf, char *fnam) {

  int_4 i,j,inp,inputs,ufio,flag,nbytes,nxfer,status;
  char fname[ports][80];
  int_4 dmac[ports];
  FILE *fp[ports];
  DMAMAP map[ports];
  double tdb,t1,t2,ratio;
  int_2 *idata;
  void *dptr;

  nbytes = size*bits/8;			/* size converted to bytes */
  if (nbytes<0) nbytes = -2*nbytes;	/* -bits means complex */
  nxfer = nbytes / 4;			/* quad buffering */
  inp = pic_getintflag(p,"INP");	/* non-default input module */
  inputs = (inp>0)? 1 : 2;		/* number of input modules feeding both sides */
  ratio = pic_getdblflag(p,"RATIO");	/* resample ratio */

  if (nbuf>0) ufio=1 ; else { ufio=0; nbuf=-nbuf; }

  /* present normalized freq */
  freq /= (rate/2);

  for (i=0; i<ports; i++) {

    /* set the synchronization flag for each port index: SGO=slave RGO=slaveSameSide */
    flag = (ko_dmacs>2 || i>=inputs)? FLG_RGO : (i==1)? FLG_SGO : 0;

    /* setup I/O ports in forward order */
    dmac[i] = pic_ioport (p, ptype, pindex+i, -1, dir, bits, rate, freq, dec, gain, flag);
    if (dmac[i]<=0) return -1;

    /* set keys for resampling */
    if (ratio>0) {
      pic_setkeyd (p,dmac[i],KEY_FREQ,((double)i)/ports);
      pic_setkeyl (p,dmac[i],KEY_GAIN,3);
      pic_setkeyd (p,dmac[i],KEY_RATIO,ratio);
    }

    if (nbytes>0) {

      /* map the buffer to physical memory */
      status = pic_mapmem (p, &map[i], nbytes, 1);
      if (status<0) return -2;

      /* prep the data buffer (output=ramp) */
      dptr  = (void*)map[i].vaddr;
      idata = (int_2*)dptr;
      if (dir>0) for (j=0; j<nbytes/2; j++) idata[j] = j;

      /* setup the DMA handler */
      status = pic_dmasetup (p, dmac[i], dir, &map[i], -1, 0);
      if (status<0) return -3;
    }

    /* save Kick Off dma channel info for master transfer startups */
    if (nbuf == 0) {
      ko_p[ko_dmacs]    = p;
      ko_dmac[ko_dmacs] = dmac[i];
      ko_func[ko_dmacs] = (size==0)? DMA_ENABLE : DMA_CONTINUOUS;
      ko_map[ko_dmacs]  = map[i];
      ko_dmacs++;
      continue;
    }

    /* open the input/output file */
    sprintf(fname[i],"%s_%d.dat",fnam,i+1);
    if (ufio) fp[i] = fopen(fname[i], (dir>0)? "r":"w" );

    /* preload output buffer from file */
    if (dir>0 && ufio) {
      status = fread(dptr,1,nbytes,fp[i]);
      if (status!=nbytes) printf("Read status=%d of %d bytes from %s\n",status,nbytes,fname[i]);
    }
  }

  printf("Starting transfer nbuf=%d nports=%d type=%d bits=%d rate=%d dec=%d ko=%d\n",
			nbuf,ports,ptype,bits,rate,dec,ko_dmacs);
  if (nbuf==0) return 1;

  for (i=ports-1; i>=0; i--) {
    /* start transfer for nbuf passes (1==DMA_ONESHOT) in reverse order */
    status = pic_dmafunc (p, dmac[i], nbuf); 
  }

  for (i=ko_dmacs-1; i>=0; i--) {
    /* start transfer for slave passes in reverse order */
    status = pic_dmafunc (ko_p[i], ko_dmac[i], ko_func[i]); 
  }

  t1 = pic_time();

  do {
    for (i=0; i<ports; i++) {
      status = pic_dmaxptr (p, dmac[i], &dptr, nxfer, 0);/* blocks until nxfer bytes ready, returns status and dptr */
      if (status>0 && ufio) {	                         /* use File IO */
        if (dir>0) status = fread(dptr,1,nxfer,fp[i]);   /* load newly opened region */
        else       status = fwrite(dptr,1,nxfer,fp[i]);  /* flush next full region */
        if (status!=nxfer) printf("Read|Write status=%d of %d bytes from|to %s\n",status,nxfer,fname[i]);
      }
    }
  } while (status>0);  /* status of last DMA channel */

  t2 = pic_time();

  for (i=0; i<ports; i++) {

    /* wait for the last buffer to finish */
    status = pic_dmafunc (p, dmac[i], DMA_WAIT); 

    /* cancel any remaining transfer */
    status = pic_dmafunc (p, dmac[i], DMA_CANCEL); 

    /* close the input/output file */
    if (ufio) fclose(fp[i]);

    /* unmap the buffer from physical memory */
    status = pic_mapmem (p, &map[i], nbytes, -1);
  }

  for (i=ko_dmacs-1; i>=0; i--) {
    /* stop transfer for NODMA passes in reverse order */
    status = pic_dmafunc (ko_p[i], ko_dmac[i], DMA_CANCEL); 
    if (ko_func[i]==DMA_CONTINUOUS) status = pic_mapmem (ko_p[i], &ko_map[i], -1, -1);
  }

  tdb = 1.e-6*nbytes*ports*nbuf; t2-=t1;
  printf("Finished transfer size=%lfMby elapse=%lfs rate=%lfMby/s\n",tdb,t2,tdb/t2);

  return 1;
}
