package nxm.ice.prim;

import nxm.sys.inc.*;
import nxm.sys.lib.*;

import nxm.ice.lib.ICEPacket;
import nxm.ice.lib.MDevIce;
import nxm.ice.lib.RamDiskResource;
import nxm.ice.lib.DevIce.DmaMap;
import nxm.ice.lib.DevIce.TimeCode;
import nxm.ice.lib.ArchSFN;
import nxm.ice.inc.PicSlave;

/**
  Synchronize a PIC playback from a Midas file/pipe.

  @author Jeff Schoen
  @version %I%, %G%
*/
public final class sinkpic extends Primitive implements PicSlave,Keyable {

  /** List of supported replay modes */
  public static String replayList="-NShot,File,Stopped,OneShot,Continuous,"
        +"StopTop,StopNow,Spin,Archive,Restart,Abort,Finish,NewDevice,Reopen,Reconnect,Release";
  public static int rFILE=-1,rSTOPPED=0,rONESHOT=1,rCONTINUOUS=2, // replay modes
        rSTOPTOP=3,rSTOPNOW=4,rSPIN=5,rARCHIVE=6,rRESTART=7,rABORT=8,rFINISH=9, 
        rNEWDEVICE=10, rREOPEN=11, rRECONNECT=12, rRELEASE=13, rRESTART_WAIT=99;
  private static int replayListOffset=-3;

  /** List of supported master/slave modes */
  public static String syncList="Master,SlaveWait,SlaveRun";
  private static int sSTOP=1,sWAIT=2,sRUN=3;

  /** List of supported throttle modes */
  public static String throttleList="Off,OneShots,Continuous,Throttled,OnDemand,Bypass";
  private static int tOFF=0,tONESHOTS=1,tCONTINUOUS=2,tTHROTTLED=3,tONDEMAND=4,tBYPASS=5;

  /** List of supported asynchronous monitor modes */
  public static String monitorList="Off,Async,Info,Full";
  private static int mOFF=0,mASYNC=1,mINFO=2,mFULL=3; 

  private long ondx,indx,lndx,rndx,bdelay,bstart,ramsz;
  private int ocyc,icyc,lcyc,rcyc,status,frame,bytes,bpa,flags,dmamode,throttle,monitor;
  private int dmac,port,bits,gain,rate,trate,xfer,dec,skip,oskip,replay,lost,lostc,block;
  private int packet,pktmod,ipkt,ptype,mnbytes,tinc,tcmode,tcpp,archtl,archsf,nchn,rtfile;
  private int stats,pmin,pmax,pavg,ptot;
  private boolean sgo,sss,xts,autors,reset,inmem,ramoncard,wrap,doneReading,isSlave,isCX,isPipe;
  private boolean nodma,async,w2e,skiponcard,lbdma,nopref,tuner,core;
  private double freq,dfreq,fscale,mtodo,delta,ratio,drate,tcoff,poll,wait,delay,prefill,sfactor,timer,timetop,tctolr=10e-9;
  private String archfn,alias,slave;
  private TimeCode tc = new TimeCode();
  private Time tcs, time = new Time();
  private DataFile hr,ho,hi;
  private MDevIce pic;
  private DmaMap map,omap;
  private ICEPacket pkt;
  private PicSlave picSlave;
  private Table keys = new Table();
  private float mbps=0;
  private ArchSFN asfn;
  
  @Override
  public int open() {

    // open the PIC device
    alias  = MA.getS("DEV");
    pic = new MDevIce (MA,alias);
    if (pic.open()<=0) M.error("Problem opening device: "+alias); 

    // get operational switches
    stats  = MA.getL("/STATS");
    autors = MA.getState("/AUTORS");
    wrap   = MA.getState("/WRAP");
    rtfile = MA.getL("/RTFILE");
    throttle = MA.getSelectionIndex("/THROTTLE",throttleList,0,-1);
    w2e    = MA.getState("/WARN2ERR");
    lbdma  = pic.getIntFlagDef("LBDMA",0)>0;
    nopref = pic.getIntFlagDef("NOPREFILL",0)>0;

    // get PIC parameters
    ptype = pic.getKeyL(0,pic.KEY_PTYPE);
    port  = pic.getKeyL(0,pic.KEY_PINDEX);
    tuner = (ptype == pic.IOPT_TUNER) || (ptype == pic.IOPT_TBANK);
    core  = (ptype == pic.IOPT_CORE) || (ptype == pic.IOPT_MCORE);
    reset = !(ptype==0 || port==0);
    reset = MA.getState("/RESET",reset);

    // get Archive or Input File
    archfn = MA.getCS ("IN");
    archsf = MA.getL("/ARCHSF", 0);
    archtl = MA.getL("/ARCHTL",B1M);
    if (archsf!=0) { 
      if (archsf==1) archsf=-1;      // default to automatic
      asfn = new ArchSFN(this,archfn,archsf); 
      asfn.setTL(archtl);
      asfn.setTop(MA.getD("/ARCHTOP"));
      asfn.setDur(MA.getD("/ARCHDUR"));
      openNextFile(true);
    } else {
      hi = MA.getDataFile ("IN");
      hi.open(hi.NATIVE);
    }
    tcs = hi.getTime();

    // get RAM template
    hr = MA.getDataFile ("OUT");
    hr.open(hr.INOUT);
    hr.setOutput(false);
    frame = (hr.typeClass==2)? hr.getSubSize() : 1;
    isCX = hr.getSPA()==2;
    bpa = hr.getBPA();

    // get output reduction 
    dec  = MA.getL("DEC");
    skip = Math.max(1,MA.getL("/SKIP")); 
    skiponcard = MA.getState("/SKIPONCARD");
    if (ptype==pic.IOPT_MODULE && dec>1 && skip==1) { skip=dec; dec=1; } // handle old syntax
    oskip = skiponcard? 1 : skip;
    delta = hr.getXDelta();
    ratio = MA.getD("/RATIO");
    nchn = MA.getL("/NCHN",1);

    // get the monitor output pipe
    ho = MA.getDataFile ("/MON",hr,0);
    ho.setYDelta(delta*frame*skip);
    ho.size = (hr.size-1)/oskip + 1;
    ho.open(ho.OPTIONAL);
    isPipe = ho.isPipe();

    // force type 1000 handling
    hi.setDFS(0);
    hr.setFS(0); hr.update();

    // get IO port parameters
    if (hr.bps>0) bits=hr.bps*8; else bits = -hr.bps;
    if (hr.spa==2) bits = -bits;
    bits = MA.getL("/BITS",bits);
    drate = MA.getD("/SRATE");
    if (dec<=0) dec=Math.max(1,(int)Math.round(delta*drate));
    drate = Math.round(dec/delta);
    dfreq = MA.getD("/DFREQ",drate);
    if (ratio>0) drate *= ratio;
    drate = MA.getD("/SRATE",drate);
    if (ratio<0) ratio = drate/(dec/delta);
    delay = MA.getD("/DELAY",0.2);
    prefill = MA.getD("/PREFILL", nopref? 0.0 : -1.0);
    nopref |= (prefill==0);
    if (prefill>0) delay = prefill;
    freq = MA.getD("FREQ",0.0*drate);
    gain = MA.getL("GAIN");
    rate = (int)Math.round(drate);
    trate = MA.getL("/TRATE",Math.max(10000,rate/8));
    fscale = isCX? 1/drate : 2/drate;
    if (ratio>0 && ratio!=1) pic.setKeyD(0,pic.KEY_RATIO,ratio);

    // inverse tuner bank parameters
    if (ptype==pic.IOPT_TBANK) {
      ratio = drate*delta;
      pic.setKeyD(0,pic.KEY_DFREQ,fscale*dfreq);
      pic.setKeyD(0,pic.KEY_RATIO,ratio);
      pic.setKeyL(0,pic.KEY_CHNS,nchn);
    }

    // get transfer parameters
    xfer = frame * MA.getL("/TL", (hr.typeClass==2)? 1:4096 );
    ramsz = (long)hr.size;
    if ((ramsz%xfer!=0 || ramsz==xfer) && ho.isOpen) warning("Ram size="+ramsz+" not a multiple>1 of /TL="+xfer+". Allow for frame offsets.");
    bytes = (int)(xfer*hr.dbpe);
    ramsz = (long)(hr.size*hr.dbpe);
    if (bytes*oskip>ramsz/2) warning("/TL * DEC should be less than 1/2 ram buffer");
    block = MA.getL("/BLOCK",-1);
    if (M.pipeMode==Midas.PINIT) replay=rCONTINUOUS; else replay=rFILE;
    replay = MA.getSelectionIndex("/REPLAY",replayList,replay,replayListOffset);

    // get timecode parameters
    tcmode = pic.getKeyL(0,pic.KEY_TCMODE);
    if (!tcs.isZero()) time.fromTime(tcs);
    else time.fromCurrent();
    if (tcmode==pic.TCM_CPU) tcoff=0;	// CPU includes year 
    else tcoff = time.getYiS();		// add current year
    tcoff  = MA.getD("/TCOFF",tcoff);
    tctolr = MA.getD("/TCTOLR",tctolr);
    tcpp   = MA.getL("/TCPP",Math.max(1,(int)(hr.size/xfer/oskip)));

    // setup packet output
    packet = MA.getLength("/PACKET");
    pktmod = MA.getL("/PKTMODE",-1);
    pkt = new ICEPacket(hi.getFormat(),xfer,pktmod);

    // master/slave modes
    slave = MA.getS("/SLAVE","NONE");
    isSlave = !slave.equals("NONE");
    sss = slave.equals("SS"); 
    xts = slave.equals("XT");
    sgo = isSlave && !(sss||xts||slave.equals("NS"));
    slave = MA.getS("/MASTER",null);

    // initialize port
    if (reset) status = pic.reset(0); else status=1;
    if (status<0) M.error("Problem resetting device"); 

    // allocate and map memory buffer
    map = pic.mapFile(hr);
    if (map==null) M.error("Problem mapping DMA memory"); 
    if (!MA.getState("/PAGED",true)) map.setPaged(false);
    if (ptype == pic.IOPT_STREAM) {
      map.setPaged(false);
      map.getVirtualAddress(0,(int)map.bytes);                  // map this space
    }
    omap = map.getSubMap(0,1);

    flags = 0;
    if (sgo) flags = pic.FLG_SGO;
    if (sss) flags = pic.FLG_RGO;
    if (xts) flags = (pic.FLG_XGO|pic.FLG_TGO);
    if (throttle==tONESHOTS||throttle==tONDEMAND) flags |= pic.FLG_NCCLK;

    openPorts();

    status = pic.dmaSetup(dmac,1,map,block,0);
    if (status<0) M.error("Problem setting up DMA channel "+status); 

    // report DMA Channel number
    MR.put(MA.getU("/DMAC"),dmac);
    nodma = pic.getKeyL(dmac,pic.KEY_NODMA)>0;

    // setup framed decimation in IOC chip
    if (skiponcard) {
      pic.setKeyL(dmac,pic.KEY_FRAME,frame);
      pic.setKeyL(dmac,pic.KEY_FRAMEDEC,skip);
    }

    // prepare for in memory mode
    inmem = hi.getURL().equals(hr.getURL());
    inmem = MA.getState("/INMEM",inmem);
    ramoncard = MA.getState("/RAMONCARD");
    wait  = MA.getD("/WAIT");

    // initialize modes
    if (replay==rSTOPPED && MA.getState("/RELEASE")) replay=rSTOPNOW; // to deallocate
    if (isSlave) replay=rSTOPPED; // special slave startup
    if (!MA.find("/POLL")) setPollTime(0.025);
    if (!tcs.isZero()) pic.setKeyD(dmac,pic.KEY_TCOFF,tcs.getWSec(),tcs.getFSec());
    dmamode = pic.DMA_STOP;
    todo = 0; 
    mtodo = map.bytes;
    ocyc = icyc = lcyc = rcyc = 0; 
    ondx = indx = lndx = rndx = 0;
    if (inmem && rtfile!=1) doneReading = loadRamBuffer();
    sfactor = 100.0/map.bytes;
    if (ramoncard) {
      long p2; for (p2=4096; p2<map.bytes; p2=p2*2);
      if (p2!=map.bytes) M.warning("RamOnCard file size="+map.bytes+" must be a power of 2");
    }
    int csize = pic.getKeyL(dmac,pic.KEY_CBUFSZ);
    bdelay = Math.round(delay/delta * hi.dbpe);
    bstart = Math.min( (long)(0.95*map.bytes), bdelay);
    if (bstart>0) bstart = Math.min( map.bytes, Math.max( bstart, 2*csize) );
    if (throttle==tONESHOTS) bstart = map.bytes;
    if (throttle==tONDEMAND && prefill<0) bstart = 0;	// implement delay not prefill
    if (nodma|lbdma|nopref) bstart = 0;
    if (lbdma) pic.setKeyD(dmac,pic.KEY_MTOFF,delay*1e9);

    // prep DeArchiver
    async = MA.getState("/ASYNC");
    if (async && !doneReading) {
      DeArchiver dearch = new DeArchiver();
      MidasThread thread = new MidasThread(M,dearch);
      thread.start();
    }

    // copy params to Keywords
    String sport = pic.getPortKey();
    hr.keywords.put("PORT",sport);
    if (archfn!=null) hr.keywords.put("ARCHIVE",getArchFN());
    double rffreq = pic.getKeyD(dmac,pic.KEY_RFFREQ);
    if (rffreq>0) hr.keywords.put("D:RFFREQ",rffreq);
    hr.update();

    // get monitor throttling mode
    monitor = MA.getSelectionIndex("/MONITOR",monitorList,mFULL,-1);

    // get extra query list entries
    setKeys(MA.getTable("/PICKEYS"));

    return (NORMAL);
  }
  
  @Override
  public synchronized int process() {

    if (replay==rABORT) return(FINISH);
    int noop = NOOP;

    // monitor DeArchiver data in
    if (replay!=rSTOPPED) {
      if (!async && !doneReading) { if (readBuffer()) noop=NORMAL; } // we did work here, dont pause if nothing to monitor
      if (throttle==tONDEMAND && indx!=lndx) pic.setKeyL(dmac,pic.KEY_INBYTE_W64,(int)(indx>>6));
      if (throttle==tONESHOTS && icyc!=lcyc && icyc>1) pic.dmaFunc(dmac,pic.DMA_RESHOT);
      if (throttle==tCONTINUOUS || throttle==tTHROTTLED) {	// check throttle modes
	long index = pic.dmaFuncX(dmac,pic.DMA_STATUS);
	double full = doneReading? 1.0 : ((double)(indx-index)/map.bytes);
	if (full<=0) full += 1; 				// handle buffer wrap around
	if (throttle==tCONTINUOUS && full<0.25) { pic.setKeyL(dmac,pic.KEY_RATE,trate); throttle=tTHROTTLED; }
	if (throttle==tTHROTTLED && full>0.75) { pic.setKeyL(dmac,pic.KEY_RATE,rate); throttle=tCONTINUOUS; }
      }
      lndx=indx; lcyc=icyc;
    }

    // start a playback
    if (todo==0) {
      if (replay==rSTOPTOP||replay==rSTOPNOW) {
	pic.dmaFunc(dmac,pic.DMA_RESET);
	replay=rSTOPPED;
      }
      if (replay==rSTOPPED) return(noop);
      ocyc=rcyc=lost=lostc=0; ondx=rndx=0; pkt.setCount(0);
      if (icyc==0 && indx<bstart && !doneReading) return (noop);// wait for input data delay
      if (rtfile>1) updateRTF();
      int newmode = nodma? pic.DMA_ENABLE : ramoncard? pic.DMA_LOADNSPIN : lbdma? pic.DMA_LOOPNSPIN : pic.DMA_CONTINUOUS;
      if (replay==rSPIN) newmode = pic.DMA_SPIN;
      if (replay==rONESHOT) newmode = pic.DMA_ONESHOT;
      if (throttle==tONESHOTS) newmode = pic.DMA_RESHOT;
      if (throttle==tONDEMAND) newmode = pic.DMA_ONDEMAND;
      if (slave!=null) setSlaveReplay(replay,newmode);		// wait for slave startup
      if (wait>0) Time.sleep(wait);
      if (stats>0) M.info("Sinkpic-"+dmac+" START");
      pic.dmaFunc(dmac,newmode);
      dmamode=newmode;					// dont change dmamode until dmaFunc completes
      todo=mtodo; 
      timer = timetop = Time.current();
    }

    if (replay==rSPIN) return(noop);
    long index = pic.dmaFuncX(dmac,pic.DMA_STATUS);
    if (index==-1 && pic.getKeyL(0,pic.KEY_STATE)<0) replay=rSTOPNOW;
    if (replay==rSTOPNOW || replay==rFINISH) todo=0; 

    // transfer data out
    if (todo>0) {
      if (index<rndx) rcyc = pic.dmaFunc(dmac,pic.DMA_CYCLE);		// low overhead cycle/index counter
      rndx = index;
      if (inmem && rtfile!=1) { indx=rndx; icyc=rcyc+1; }
      if (rcyc<ocyc+lost) return (noop);				// data not ready
      if (doneReading && !inmem && (ocyc>icyc || ondx>=indx))		// finish playback
	 { replay=rFINISH; return(NORMAL); }
      int behind = rcyc - (icyc+lost);					// check for falling behind
      if (index>indx) behind++;
      if (behind>0) { 
        if (!inmem) warning("Falling behind "+behind+" buffers"); 
        //warning("Falling behind "+behind+" buffers"+" (rcyc="+rcyc+" rndx="+rndx+" icyc="+icyc+" indx="+indx+")"); 
        lost += behind; 
      }
      int tbytes = bytes;						// total bytes
      if (ondx+tbytes>map.bytes) tbytes=(int)(map.bytes-ondx);		// trim bytes
      if (index<ondx+tbytes && rcyc<=ocyc+lost) return (noop);		// not ready
      int count = pkt.getCount();
      if (tcmode!=pic.TCM_OFF && count%tcpp==0) {			// handle timecode every tcpp packets
        int tcstat = pic.tc (dmac,-1.0,delta,tc,0); tc.wsec+=tcoff;	// get current timecode
        pkt.setTC (tcmode,tcstat,tc.offset,tc.wsec,tc.fsec);		// set in packet header
        if (count==0) ho.setTime(tc.wsec,tc.fsec-tc.offset*delta); 	// record 1st TC to output header
      } 
      if (rtfile>1) updateRTF();

      if (ho.isOpen && tbytes>0) {
	long vaddr;							// map this space
        if (ondx+bytes>map.bytes) {
	  vaddr = omap.getVirtualAddress(ondx-bytes,bytes);		// map this space
        } else {
	  vaddr = omap.getVirtualAddress(ondx,bytes);			// map this space
        }
	ho_write(vaddr,0,bytes);					// output data
      }

      if (tbytes<=0) todo=0;
      else { ondx+=bytes*oskip; todo=mtodo-ondx; pkt.upCount(); }	// advance pointers

      if (tcmode==pic.TCM_CPU) time.fromCurrent();
      else if (tcmode!=pic.TCM_OFF) {
	double dtc = (ocyc*map.bytes + ondx)/hr.dbpe;
	if (wrap && dtc>hi.size) dtc = dtc % hi.size;
	time.fromJ1950(hi.getTimeAt(dtc));
      }
    }

    // stop a playback 
    if (todo<=0) {
      ocyc++; todo=0; ondx-=map.bytes;
      long gap = (rndx-indx) + (rcyc-icyc-lost+1)*map.bytes;
      int pf = (doneReading)? 0 : (int)Math.round(sfactor*gap);
      timer = Time.current();
      boolean partial = (replay==rSTOPNOW || replay==rFINISH);
      if (stats>0 && !partial) {
        if (pf<pmin) pmin=pf; if (pf>pmax) pmax=pf; ptot+=pf;
        if (ocyc%stats==0) { pavg = ptot/stats;
          mbps = (float)(1e-6*map.bytes*stats/Math.max(0.01,timer-timetop));
	  String smm = (stats>1)? ",FMIN="+pmin+",FMAX="+pmax : "";
          M.info("Sinkpic-"+dmac+" STATS={CYC="+ocyc+",LOST="+lost+",FAVG="+pavg+smm+",MBPS="+mbps+"}");
          pmin=100; pmax=0; ptot=0;
          timetop = timer;
      } }
      int lostd = pic.dmaFunc(dmac,pic.DMA_LOST);
      if (lostd>lostc) { warning("Lost "+(lostd-lostc)+" card buffers"); lostc=lostd; }
      if (rtfile>1) updateRTF();
      if (replay==rCONTINUOUS) { todo=mtodo; return(noop); }
      if (replay==rONESHOT) { pic.dmaFunc(dmac,pic.DMA_WAIT); rcyc=1; }
      if (stats>0) M.info("Sinkpic-"+dmac+" STOP");
      pic.dmaFunc(dmac,pic.DMA_STOP);
      pic.dmaFunc(dmac,pic.DMA_RESET);
      dmamode=pic.DMA_STOP;
      if (slave!=null) setSlaveReplay(replay,dmamode);                  // wait for slave shutdown
      if (replay==rFILE || replay==rFINISH) return(FINISH);
      if (replay==rRESTART) replay=rCONTINUOUS;
      else replay=rSTOPPED;
    }

    return (NORMAL);
  }

  @Override
  public int close() {
    while (async) { doneReading = true;	Time.sleep(0.1); } // stop DeArchiver thread
    if (dmac>0) pic.dmaFunc(dmac,pic.DMA_CANCEL);
    if (dmamode!=pic.DMA_STOP && stats>0) M.info("Sinkpic-"+dmac+" CANCEL");
    if (hr!=null) hr.close();
    if (ho!=null) ho.close();
    if (hi!=null) hi.close();
    if (omap!=null) omap.close(DmaMap.NOCHK);
    if (map!=null) map.close();
    if (pic!=null) pic.close();
    replay = rSTOPPED;
    dmamode = pic.DMA_STOP;
    return (NORMAL);
  }

  // pipe write that cannot block
  private void ho_write (long vaddr, int off, int bytes) {
    if (monitor==mOFF) return;
    if (isPipe) {
      long avail = ho.io.avail();
      if (avail<bytes && bytes<=((PipeResource)ho.io).pipeSize) {       // wait for it ?
	for (int i=0; i<20 && avail<bytes; i++) {
	  Time.sleep(0.010);
	  avail = ho.io.avail();
	}
	if (avail>bytes);
	else if (monitor==mINFO) M.info("Dropping monitor frame");
	else return;    // dropped one
      }
    }
    ho.write(vaddr,off,bytes);
  }


  private void warning (String text) {
    if (w2e) M.error(text);
    else M.warning(text);
  }

  private void openPorts() {
    dmac = pic.ioPort(ptype,port,-1,1,bits,rate,fscale*freq,dec,gain,flags);
    if (dmac<=0) M.error("Problem setting up IO port"); 
  }

  private boolean loadRamBuffer() {
    if (!(hi.io instanceof RamDiskResource)) {
      int tbytes = (int)(hr.size*hr.dbpe);
      long vaddr = map.getVirtualAddress(0L,tbytes);	// map this space
      hi.read(vaddr,0,tbytes);
    }
    icyc=1; indx=0;
    return true;
  }  
  
  private boolean readBuffer () {
    boolean isStream = (rtfile==1) || hi.isStream();
    long maxavail = (rtfile==1)? availRTF() : isStream? hi.io.avail() : (long)((hi.size-hi.seek())*hi.dbpe);
    int avail = (int)Math.min(B1G,maxavail);
    int abytes = (int)Math.min(archtl*bpa,map.bytes-indx);
    if (avail>0) abytes = Math.min(abytes,avail);		// trim to whats available
    else if (isStream);						// piped wait for it
    else if (archsf!=0 && openNextFile(false));			// found next file - continue
    else if (wrap) hi.seek(0.0);				// go back to beginning
    else doneReading = true;					// mark finished
    if (avail>0 && avail<abytes) abytes = avail; 		// end of file | all in pipe
    if ( (icyc>(rcyc-lost) && (indx+abytes)>rndx) 		// wait for RAM pointer
      || (icyc>ocyc && (indx+abytes)>ondx) ) avail=0;		// wait for output pointer
    if (packet>=0 && avail>pkt.headerLength) {
      hi.read (pkt.buf,0,pkt.headerLength);
      avail = abytes = pkt.getBytes();
    }
    if (avail>=abytes) {
      if (!inmem) {
        long vaddr = map.getVirtualAddress(indx,abytes);	// map this space
        abytes = hi.read(vaddr,0,abytes);
      }
      if (abytes>0) { 
        indx += abytes;
        if (indx>=map.bytes) { icyc++; indx=0; }
        if (wrap && !isStream && abytes==maxavail && archsf==0) hi.seek(0.0); // prevent pause on short wrapped files
      }
      return true;
    }
    return false;
  }

  private boolean openNextFile (boolean reload) {
    if (hi!=null && hi.isOpen()) hi.close();
    int index = reload? asfn.getIndex() : asfn.nextIndex();
    String sfn = asfn.getSFN(index);
    if (sfn==null || sfn.equals("NULL") || sfn.equals("NONE")) return false; // name triggered completion
    if (archsf<0 && hi!=null) { hi.setName(sfn); if (!hi.find(-1)) { index=asfn.resetIndex(); sfn=asfn.getSFN(index); }}
    if (stats!=0 && (archsf>0 || asfn.isLegit())) M.info ("Sinkpic-"+dmac+" ArchSF index="+index+" fn="+sfn);
    hi = new DataFile (this,sfn,"","",0);
    hi.open(hi.NATIVE);
    if (!hi.isOpen) return false;   // problem finding file
    tcs = hi.getTime(); // update timecode
    if (tcmode!=pic.TCM_OFF && tcmode!=pic.TCM_CPU && !tcs.isZero()) {
      double dtc = (icyc*map.bytes + indx)/hr.dbpe;
      pic.setKeyD(dmac,pic.KEY_TCOFF,tcs.getWSec(),tcs.getFSec(),dtc);
    }
    hi.setDFS(0);
    if (asfn.isTop()) return false;       // back to top
    return true; 
  }

  private void updateRTF() {
    long rbyte = map.bytes*rcyc + rndx;
    hr.setOutByte(0,(double)rbyte);
    hr.ioh.write(hr.hb,88,8,88L); // update outByte field
  }

  private long availRTF() {
    long ibyte = map.bytes*icyc + indx;
    long inbyte = (long)hi.getInByte();
    long avail = inbyte - ibyte;
    if (throttle==tONDEMAND && prefill<0) avail -= bdelay;
    if (avail<archtl*bpa) hi.ioh.read(hi.hb,80,8,80L); // update inByte field
    if (inmem) updateRTF();	// hi==hr give outbyte feedback
    return avail;
  }

  public void setReplay (String value) {
    if (value.equalsIgnoreCase("STOP")) value = "STOPNOW";
    if (value.equalsIgnoreCase("START")) value = "CONTINUOUS";
    int nreplay = Parser.find(replayList,value,replay,replayListOffset);
    if (nreplay>replayListOffset) setReplay(nreplay);
    else warning("Illegal replay mode: "+value);
  }
  private void setReplay (int nreplay) {
    if (nreplay==0 && replay!=0) replay = rSTOPNOW;
    else if (nreplay==rRESTART && replay==0) replay = rCONTINUOUS;
    else if (nreplay==rONESHOT && replay==rCONTINUOUS) replay = rSTOPTOP;
    else if (nreplay==rRECONNECT) { openPorts(); replay=0; }
    else replay=nreplay;
    if (replay==rSTOPNOW) process(); // process now
    if (replay>=rABORT && slave!=null) setSlaveReplay(replay,pic.DMA_STOP);
    if (replay==rCONTINUOUS && nodma) replay = rSPIN;
  }
  private void setSlaveReplay (int value, int mode) {
    if (picSlave==null) picSlave = (PicSlave)M.registry.get(slave);
    if (picSlave==null) { warning("Slave="+slave+" does not exist"); return; }
    if (mode==pic.DMA_STOP && value<rABORT) value = rSTOPNOW;
    if (nodma && value==rSPIN) value = rCONTINUOUS;
    picSlave.setReplay(getReplay(value));
    for (;;) {
      int smode = picSlave.getDmaMode();
      if (smode==mode) break;
      if (smode==pic.DMA_ENABLE && mode==pic.DMA_CONTINUOUS) break;
      if (mode==pic.DMA_ENABLE && smode==pic.DMA_CONTINUOUS) break;
      if (mode==pic.DMA_ONDEMAND && smode==pic.DMA_CONTINUOUS) break;
      Time.sleep(pollTime); 
    }
  }
  public void setMonitor (String value) {
    monitor = Parser.find(monitorList,value,monitor,-1);
  }
  public void setChan (int chan) { 
    pic.setKeyD(dmac,pic.KEY_CHAN,chan); 
  }
  public void setRate (int value) { 
    rate=value; pic.setKeyL(dmac,pic.KEY_RATE,rate); 
  }
  public void setFreq (double value) { 
    freq=value; pic.setKeyD(dmac,pic.KEY_FREQ,fscale*freq); 
  }
  public void setDec (int value) { 
    dec=value; 
  }
  public void setGain (int value) { 
    gain=value; pic.setKeyL(dmac,pic.KEY_GAIN,gain); 
  }
  public void setSkip (int value) { 
    skip=value; 
  }
  public void setStats (int value) {
    stats = value;
  }
  public String setSFName (String value) {
    if (value!=null && value.length()>0) {
      if (value.startsWith("_")) {
        String fn = hi.getName().toString();
        int i = fn.lastIndexOf('_');
        if(i>0) value = fn.substring(0,i)+value;
      }
      asfn.setFormat(value);
    }
    if (archsf<0 && !asfn.isLegit()) archsf=0;
    return value;
  }
  public void setArchTop(double top) { if (archsf!=0) { asfn.setTop(top); openNextFile(true); } }
  public void setArchDur(double dur) { if (archsf!=0) asfn.setDur(dur); }

  private String getReplay (int replay) {
    return Parser.get(replayList,replay-replayListOffset);
  }
  public String getReplay() { return getReplay(replay); }
  public int getRate() { return rate; }
  public double getFreq() { return freq; }
  public int getDec() { return dec; }
  public int getGain() { return gain; }
  public int getSkip() { return skip; }
  public int getCycle() { return rcyc; }
  public long getIndex() { return rndx; }
  public double getProgress() { return doneReading? ((double)rndx)/map.bytes : hi.seek()/hi.size; }
  public String getMonitor() { return Parser.get(monitorList,monitor,-1); }
  public String getThrottle() { return Parser.get(throttleList,throttle,-1); }
  public TimeCode getTimeCode() { return tc; }
  public Time getTime() { return time; }
  public int getHBLost() { return lost; }
  public int getCBLost() { return lostc; }
  public int getPFull() { return pavg; }
  public ArchSFN getASFN() { return asfn; }
  public ArchSFN getArchiver() { return asfn; }
  public double getArchDur() { return (archsf!=0)? asfn.getLength() : hi.getLength(); }
  public double getArchOff() { return (archsf!=0)? asfn.getLengthTo(hi.getOffset()) : hi.getOffset()*hi.getDelta(); }
  public double getArchTop() { return (archsf!=0)? asfn.getTimeTop() : hi.getTimeAt(0); }
  public String getSFName() { return asfn.getSFN(); }
  public int getDmaMode() { return dmamode; }
  public double getOffset() { return hi.getOffset(); }
  public double getDelta() { return hi.getDelta(); }
  public MDevIce getDevIce() { return pic; }
  public float getMBPS() { return mbps; }
  public String getArchFN() {
    if (archfn == null) return null;
    int i=archfn.indexOf('{');
    if (i>0) return archfn.substring(0,i);
    return archfn;
  }


  public Table getTable (Table t) { return new Table( pic.getKeyTable(dmac,t.toString()) ); }

  // use the keyable interface to expose all of the PIC library keys

  private void setKeys (Table kt) {
    if (kt==null || kt.size()==0) return;
    for (Table.Iterator ti=kt.iterator(); ti.getNext(); ) {
      String sval = (ti.value==null)? "QUERY" : ti.value.toString();
      if (sval.length()==0 || sval.equals("QUERY")) keys.put(ti.key,ti.value); // to show in Query
      else setKey(ti.key,ti.value); // actually set
    }
  }

  @Override
  public String[] getKeys() {
    return keys.getKeys();
  }

  @Override
  public synchronized Object setKey (String key, Object value) {
         if (key.equals("REPLAY")) setReplay(Convert.o2s(value));
    else if (key.equals("RATE"))   setRate(Convert.o2l(value));
    else if (key.equals("FREQ"))   setFreq(Convert.o2d(value));
    else if (key.equals("DEC"))    setDec(Convert.o2l(value));
    else if (key.equals("GAIN"))   setGain(Convert.o2l(value));
    else if (key.equals("TABLE"))  value = pic.setKeyTable(dmac,value.toString());
    else {  // now expose all PIC library keys
      int ikey = pic.name2key(key);
      if (ikey<=0) return null;
      int ityp = pic.getKeyType(ikey);
      if (ikey==pic.KEY_IPCONN || ikey==pic.KEY_IPDISC) value = Integer.valueOf( pic.str2ip(Convert.o2s(value)) );
      if (ityp==DOUBLE) pic.setKeyD(dmac,ikey,Convert.o2d(value));
      else              pic.setKeyL(dmac,ikey,Convert.o2l(value));
      if (ikey==pic.KEY_RFFREQ) { hr.keywords.put("D:RFFREQ",pic.getKeyD(dmac,pic.KEY_RFFREQ)); hr.update(); }
      keys.put(key,value); // to show in Query
    }
    return value;
  }

  @Override
  public synchronized Object getKey (String key) {
    Object value = null;
         if (key.equals("REPLAY")) value=            getReplay();
    else if (key.equals("RATE"))   value=Convert.l2o(getRate());
    else if (key.equals("FREQ"))   value=Convert.d2o(getFreq());
    else if (key.equals("DEC"))    value=Convert.l2o(getDec());
    else if (key.equals("GAIN"))   value=Convert.l2o(getGain());
    else {  // now expose all PIC library keys
      int ikey = pic.name2key(key);
      if (ikey<=0) return null;
      int ityp = pic.getKeyType(ikey);
      if (ityp==DOUBLE) value=Convert.d2o(pic.getKeyD(dmac,ikey));
      else              value=Convert.l2o(pic.getKeyL(dmac,ikey));
    }
    return value;
  }

  private class DeArchiver implements Runnable {
    public void run() {
      while (!doneReading) {
        if (state==PROCESS && readBuffer());
        else Time.sleep(0.010);
      }
      async = false;
    }
  }

}
