package nxm.ice.prim;

import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.sys.libm.MMath;

import nxm.ice.lib.MDevIce;
import nxm.ice.lib.DevIce.*;
import nxm.ice.lib.IceHW;

/**
  NeXtMidas interface to the PIC family of ICE cards

  FUNC,DEV,NAME,VALUE

  @author Jeff Schoen
  @version %I%, %G%
*/
public class picdriver extends Primitive {

  public String replayList="-NShot,File,Stopped,OneShot,Continuous,"
    +"StopTop,Load,Spin,Archive,Restart,Abort";
  private int rFILE=-1,rSTOPPED=0,rONESHOT=1,rCONTINUOUS=2,
    rSTOPTOP=3,rLOAD=4,rSPIN=5,rARCHIVE=6,rRESTART=7,rABORT=8;
  private static int replayListOffset=-3;
  private int PREP=1,WAIT=2,DONE=3,SPECS=4,LOOPTC=5,PREDONE=9;

  private int status,offset,value,flags,key,dmamode,prcmode,tcmode,isfp;
  private int dmac,dmao,dir,bits,rate,tl,dec,gain,replay,block,kick,timeout;
  private double freq,wait,elapse,delta,tcoff,ratio,fscale;
  private String func,alias,label,name;
  private DataFile hcb;
  private Data data;
  private DmaMap map;
  private DmaSpec spec;
  private boolean tuner;
  private DmaMap[] maps;
  private DataFile[] hcbs;
  private MDevIce pic;
  private TimeCode tc = new TimeCode();
  private Time time = new Time();
  private int node,mnode,core;

  public int open() {

    func = MA.getU("FUNC");

    prcmode = PREDONE;
         if (func.equals("CREATE")) create();
    else if (func.equals("DETECT")) status = MDevIce.detect(0);
    else if (func.equals("NUMDEV")) status = MDevIce.detect(1);
    else if (func.equals("ACTIVE")) status = MDevIce.detect(2);
    else prcmode = DONE;
    if (prcmode==PREDONE) return (FINISH);
    tcmode = pic.TCM_OFF;

    alias = MA.getS("DEV");
    name = MA.getU("NAME");
    wait = MA.getD("/PAUSE");
    wait = MA.getD("/WAIT",wait);
    node = MA.getL("/NODE",0);  mnode = (node&0xFF);
    core = MA.getL("/CORE",0);
    dmac = (core>0)? -core : (node>0)? node : 0;
    dmac = MA.getL("/DMAC",dmac);
    isfp = MA.getL("/ISFP",0);

    pic = new MDevIce (MA,alias);
    status = pic.open();
    if (status<0) M.error("Problem opening device: "+alias);

    if (func.equals("SNIFF")) {
      boolean all = MA.getState("/ALL");
      if (all) flags = -1;
      pic.sniff(flags);
    }
    else if (func.equals("GETV")) {
      key = pic.name2key(name);
      int val = pic.getKeyV(dmac,key);
      label = MA.getU("VALUE");
      if (label.length()!=0) MR.put(label,val);
      else MT.writeln("Key "+name+" = "+val);
    }
    else if (func.equals("GET")) {
      boolean str = MA.find("/STR");
      boolean hex = MA.find("/HEX");
      key = pic.name2key(name);
      if (key<0) M.error("Key "+name+" not supported");
      data = new Data(0);
      if (str) data = new Data("TA",1);
      if (key==pic.KEY_NDEC) data = new Data(MA.getL("VALUE"));
      if (key==pic.KEY_NFREQ||key==pic.KEY_NRATIO) data = new Data(MA.getD("VALUE"));
      if (pic.getKey(dmac,key,data)<0) M.error("Key "+name+" not found");
      label = MA.getU("VALUE");
      Object val = data;
      if (hex) val = Convert.bb2hex(data.buf,0,data.size*data.bpe);
      if (str) {
        int i=data.size*data.bpe;
        for (;i>0 && data.buf[i-1]<=32;i--);
        val = new String(data.buf,0,i);
      }
      if (label.length()!=0) MR.put(label,val);
      else MT.writeln("Key "+name+" = "+val);
    }
    else if (func.equals("SET")) {
      key = pic.name2key(name);
      if (key<0) M.error("Key "+name+" not supported");
      data = MA.getData("VALUE");
      if (data==null) M.error("Value for "+name+" not found on command line");
      status = pic.setKey(dmac,key,data);
    }
    else if (func.equals("GPS")) {
      data = new Data(0);
      pic.getKey(dmac,pic.KEY_GPS,data);
      label = MA.getU("VALUE");
      double mjs = data.getD(0);
      double lat = data.getD(1);
      double lon = data.getD(2);
      double alt = data.getD(3);
      int  stage = (int)data.getD(4);
      int    nos = (int)data.getD(5);
      if (label.length()>0) {
	Table t = new Table();
	t.put("NOS",nos); t.put("STAT",stage); t.put("MJS",mjs);
	t.put("LAT",lat); t.put("LON",lon); t.put("ALT",alt);
        MR.put(label,t);
      }
      label = MA.getU("/SV");
      if (label.length()>0) {
        StateVector sv = new StateVector();
        sv.setTimeCode(mjs);
        sv.setGeo(alt,lat,lon);
        MR.put(label,sv);
      }
      label = MA.getU("/GSTAT");
      if (label.length()>0) {
	java.io.ByteArrayOutputStream baos = new java.io.ByteArrayOutputStream();
        java.io.PrintStream ps = new java.io.PrintStream(baos);
        ps.printf("GPS NoS=%d Status=%d MJS=%f Lat=%f Lon=%f Alt=%f",nos,stage,mjs,lat,lon,alt);
        MR.put(label,baos.toString());
      }
    }
    else if (func.equals("FPLAN")) {
      data = new Data("SD",4);
      data.setD(0,MA.getD("/SRATE"));
      data.setD(1,MA.getD("/RFFREQ"));
      pic.query(pic.QUERY_FPLAN,0,data.buf,16);
      System.out.printf("FreqPlan  ref=%21.18f  vco=%21.18f  srate=%21.18f  rffreq=%21.18f \n",
		data.getD(2),data.getD(3),data.getD(0),data.getD(1));
    }
    else if (func.equals("HALT")) {
      pic.reset(pic.FLG_DISABLE);
    }
    else if (func.equals("OVSR")) {
      int ovsr = MA.getL("VALUE",0);
      int port = MA.getL("PORT");
      status = pic.setKeyL(port,pic.KEY_OVSR,ovsr);
      if (status<0) M.warning ("Bad OVSR value: "+ovsr);
    }
    else if (func.equals("RESET") || func.equals("REBOOT")) {
      status = pic.reset( func.equals("REBOOT")? pic.FLG_BOOT:0 );
      if (status>=0) {
        String ioc = MA.getU("NAME"); // optional IOC load
        if (ioc.length()>0) pic.loadFile("*_"+ioc, pic.FLG_IOC);
      }
      else M.error("Problem resetting card");
    }
    else if (func.equals("TEST")) {
      int test = MA.getL("NAME");
      label = MA.getU("VALUE");
      flags = MA.getL("/DMAC", (label.length()==0)? 1 : 0);
      status = pic.test(test,flags);
      MR.put(label,status);
    }
    else if (func.equals("READ")) {
      offset = pic.name2offset(name,0);
      if (offset<0) M.error("Unknown register: "+name);
      value = pic.read(offset);
      label = MA.getU("VALUE");
      if (label.length()>0) MR.put(label,value);
      else if (MA.find("/HEX")) MT.writeln("Read "+name+"="+
	Convert.l2x(value)+" at "+ Convert.l2x(offset));
      else MT.writeln("Read "+name+"="+value+" at "+Convert.l2x(offset));
    }
    else if (func.equals("WRITE")) {
      offset = pic.name2offset(name,0);
      if (offset<0) M.error("Unknown register: "+name);
      value = MA.getL("VALUE");
      value = pic.write(offset,value);
    }
    else if (func.equals("LOADSHARC")) {
      pic.loadFile(MA.getU("NAME"), pic.FLG_SHARC);
    }
    else if (func.equals("LOADSTRATIX")) {
      pic.loadFile(MA.getU("NAME"), pic.FLG_STRATIX|mnode);
    }
    else if (func.equals("LOADIOC")) {
      pic.loadFile("*_"+MA.getU("NAME"), pic.FLG_IOC);
    }
    else if (func.equals("LOADPM")) {
      pic.loadFile(MA.getU("NAME"), pic.FLG_PM|mnode);
    }
    else if (func.equals("LOADVIRTEX")) {
      pic.loadFile(MA.getU("NAME"), pic.FLG_VIRTEX|mnode);
    }
    else if (func.equals("LOADPPC")) {
      pic.loadFile(MA.getU("NAME"), pic.FLG_PPC);
    }
    else if (func.equals("LOADMOD")) {
      int port = MA.getL("PORT") & 0xFF;
      pic.loadFile(MA.getU("NAME"), pic.FLG_MOD|port);
    }
    else if (func.equals("LOADFC")) {
      int port = MA.getL("PORT");
      hcb = MA.getDataFile("NAME","1000","SI,SF",0);
      hcb.open(hcb.INPUT); 
      int size = (int)hcb.size;
      int flags = (hcb.bps==4)? 0x20000000 : 0;
      data = hcb.getDataBuffer(size); 
      size = hcb.read(data);
      hcb.close();
      status = pic.loadFC (port, data.buf, size, flags);
    }
    else if (func.equals("CLEARFLASH")) {
      node = pic.getIntFlagDef("FLASHPORT",11);
      int nlf = pic.getIntFlagDef("NOLOAD",0);
      if (node==0) MT.writeln("Warning CLEARFLASH on node=0 not recommended");
      if (nlf<=0) M.error("The CLEARFLASH command should be called with NOLOAD flag");
      String ans = MT.getInput("Continue to ClearFlash on port="+node+".  Are you sure? ","N");
      if (ans==null || !(ans.startsWith("Y")||ans.startsWith("y"))) M.error("Aborting flash load");
      pic.setKeyL(0,pic.KEY_VERBOSE,-1);
      double start = Time.current();
      pic.loadFile("icek8m_rr", pic.FLG_FLASH);
      M.info("FlashClear took "+(Time.current()-start)+" sec");
    }
    else if (func.equals("LOADFLASH")) {
      node = pic.getIntFlagDef("FLASHPORT",0);
      if (node==0) MT.writeln("Do NOT interrupt this process ... you are reprogramming the boot loader.");
      MT.writeln("Rebooting card ...");
      pic.reset(pic.FLG_BOOT);
      if (MA.getState("/ASK",true)) {
        String ans = MT.getInput("Continue to LoadFlash "+MA.getU("NAME")+" on port="+node+".  Are you sure? ","N");
        if (ans==null || !(ans.startsWith("Y")||ans.startsWith("y"))) M.error("Aborting flash load");
      }
      pic.setKeyL(0,pic.KEY_VERBOSE,-1);
      double start = Time.current();
      pic.loadFile(MA.getU("NAME"), pic.FLG_FLASH);
      M.info("FlashLoad took "+(Time.current()-start)+" sec");
    }
    else if (func.equals("UNLOADFLASH")) {
      bits = MA.getL("/LEN",16*B1M);
      int bytes = bits/8, size = bits/32;
      byte[] buffer = new byte[bytes];
      pic.setKeyL(0,pic.KEY_VERBOSE,-1);
      pic.getKey (0,pic.KEY_FLASH,buffer,bytes);
      hcb = MA.getDataFile("NAME","1000","SL",0);
      hcb.setSize((double)size);
      hcb.open(hcb.OUTPUT);
      hcb.write(buffer,0,bytes);
      hcb.close();
    }
    else if (func.equals("CHECKFLASH")) {
      pic.setKeyL(0,pic.KEY_VERBOSE,-1);
      int chksum = pic.getKeyL (0,pic.KEY_FLASH);
      label = MA.getU("VALUE");
      if (label.length()!=0) MR.put(label,chksum);
      else MT.writeln("Flash ChkSum = "+ Convert.l2x(chksum));
    }
    else if (func.equals("CHECKFILE")) {
      status = pic.loadFile(MA.getU("NAME"), pic.FLG_FLASH|0x1);
    }
    else if (func.equals("ACQUIRE") || func.equals("PLAY")) {
      hcb = MA.getDataFile("NAME","1000","SP,SB,SI,SL,CB,CI,CL",0);
      if (func.equals("ACQUIRE")) dir=-1; else dir=1;
      hcb.open(hcb.INOUT|hcb.NATIVE); 
      if (hcb.bps>0) bits=hcb.bps*8; else bits = -hcb.bps;
      if (hcb.spa==2) bits = -bits;
      bits = MA.getL("/BITS",bits);
      int ptype = pic.getKeyL(0,pic.KEY_PTYPE);
      tuner = (ptype==pic.IOPT_TUNER) || (ptype==pic.IOPT_TBANK);
      if (tuner) dec = MA.getL("DEC"); else dec=1;
      delta = hcb.getXDelta();
      rate = (int)Math.round(dec/delta);
      ratio = MA.getD("/RATIO");
      if (ratio>0) pic.setKeyD(0,pic.KEY_RATIO,ratio);
      if (ratio>0) rate = (int)Math.round(dec/(delta*ratio));
      if (tuner) rate *= hcb.spa;
      rate = MA.getL("/SRATE",rate);
      fscale = MA.getState("/NYFREQ")? 1.0 : 2.0/rate;
      freq = MA.getD("FREQ",rate*.25) * fscale;
      gain = MA.getL("GAIN");
      block = MA.getL("/BLOCK",-1);
      kick = MA.getL("/KICK",-1);
      tcmode = pic.getKeyL(0,pic.KEY_TCMODE);
      if (tcmode==pic.TCM_CPU) tcoff=0; // CPU includes year
      else { time.fromCurrent(); tcoff = time.getYiS(); } // add current year
      tcoff = MA.getD("/TCOFF",tcoff);
      if (tcmode!=pic.TCM_OFF) hcb.setTime(0.0,0.0);
      status = pic.reset(0);
      int nchan = 1;
      if (ptype==pic.IOPT_TBANK) {
        nchan = pic.getKeyL(-1,pic.KEY_CHNS);
	if (pic.getKeyL(0,pic.KEY_PINDEX)!=3) nchan /= 2;
        nchan = MA.getL("/NCHN",nchan);
        pic.setKeyL(-1,pic.KEY_CHNS,nchan);
        int xfer = MA.getL("/TL",4096);
        int pktlen = MA.getL("/PKTLEN",xfer*hcb.bpa);
        pic.setKeyL(-1,pic.KEY_PKTLEN,pktlen);
        double dfreq = MA.getD("/DFREQ",(0.5/delta)) * fscale;
        pic.setKeyD(-1,pic.KEY_DFREQ,dfreq);
      }
      dmac = pic.ioPort(-1,-1,-1,dir,bits,rate,freq,dec,gain,flags);
      if (ratio>0) pic.setKeyD(dmac,pic.KEY_RATIO,ratio);
      replay = MA.getSelectionIndex("/REPLAY",replayList,-1,replayListOffset);
           if (replay==rCONTINUOUS) dmamode = pic.DMA_CONTINUOUS;
      else if (replay==rONESHOT) dmamode = pic.DMA_ONESHOT;
      else if (replay==rSPIN || replay==rLOAD) dmamode = pic.DMA_SPIN;
      else if (replay<=0) dmamode = -replay;
      else M.error("Illegal replay mode");
      map = pic.mapFile(hcb);
      if (replay==rLOAD) map.getVirtualAddress(0L,(int)map.bytes);
      status = pic.dmaSetup(dmac,dir,map,block,0);
      prcmode=PREP; 
      if (tuner && MA.getState("/TRESTART")) {
        pic.dmaFunc(dmac,dmamode);
        if (wait>0) Time.sleep(wait); 
        pic.dmaFunc(dmac,pic.DMA_STOP);
        pic.dmaFunc(dmac,pic.DMA_RESET);
      }
      label = MA.getU("/AFREQ");
      if (label!=null && label.length()>0) {
        double[] freqs = new double[nchan];
        data = new Data(freq);
	for (int n=0; n<nchan; n++) {
	  data.setD(0,freq);
          pic.setKeyL(dmac,pic.KEY_CHAN,n+1);
          pic.getKey(dmac,pic.KEY_NFREQ,data);
	  freqs[n] = data.getD(0)/fscale;
	}
	MR.put(label,new Data(freqs));
      }
    }
    else if (func.equals("CHAIN")) {
      Args.SplitArg arg = MA.getSplitArg(MA.getRaw("NAME"),'|');
      int nfiles = arg.getFields();
      DataFile hcbs[] = new DataFile[nfiles];
      DmaMap maps[] = new DmaMap[nfiles];
      for (int i=0; i<nfiles; i++) {
        String name = arg.getField(i);
        hcbs[i] = (DataFile)BaseFile.getInstanceFor(this,name);
        hcbs[i].open(DataFile.INOUT|DataFile.NATIVE); 
        maps[i] = pic.mapFile(hcbs[i]);
      }
      map = maps[0]; hcb = hcbs[0]; dir=1;
      bits = MA.getL("/BITS",hcb.bps*8);
      rate = (int)(1/hcb.getXDelta());
      replay = MA.getSelectionIndex("/REPLAY",replayList,-1,replayListOffset);
           if (replay==rCONTINUOUS) dmamode = pic.DMA_CONTINUOUS;
      else if (replay==rONESHOT) dmamode = pic.DMA_ONESHOT;
      else if (replay<=0) dmamode = -replay;
      else M.error("Illegal replay mode");
      status = pic.reset(0);
      dmac = pic.ioPort(-1,-1,-1,dir,bits,rate,0.0,0,0,0);
      status = pic.dmaSetup(dmac,dir,maps[0],block,0);
      boolean wrap = MA.getState("/WRAP");
      for (int i=1; i<nfiles; i++) {
        int next=i+1; 
        if (i==nfiles-1) next=(wrap)?1:0;
        pic.dmaChain (dmac,i,maps[i],dmamode,next);
        if (maps[i].mode!=DmaMap.RAMD) hcbs[i].read(maps[i].vaddr,0,(int)maps[i].bytes);
      }
      pic.dmaDump (dmac,0);
      prcmode=PREP; 
      MR.put(MA.getU("/DMAC"),dmac);
    }
    else if (func.equals("START")) {
      dir = MA.getL("/DIR",0);
      dmac = pic.ioPort (-1,-1,-1,dir,16,0,0.0,64,0,pic.FLG_DISABLE);
      pic.dmaFunc (dmac,pic.DMA_SPIN);
    }
    else if (func.equals("STOP")) {
      dir = MA.getL("/DIR",0);
      dmac = pic.ioPort (-1,-1,-1,dir,16,0,0.0,64,0,pic.FLG_DISABLE);
      pic.dmaFunc (dmac,pic.DMA_STOP);
      pic.dmaFunc (dmac,pic.DMA_CANCEL);
    }
    else if (func.equals("SPIN")) {
      dir = MA.getL("/DIR",1);
      dec = MA.getL("/DEC",1);
      bits = MA.getL("/BITS",16);
      rate = MA.getL("/RATE",10000000);
      dmac = pic.ioPort (-1,-1,-1,dir,bits,rate,0.0,dec,0,0);
      pic.dmaFunc (dmac,pic.DMA_ENABLE);
    }
    else if (func.equals("LOOP")) {
      bits = MA.getL("/BITS",16);
      rate = MA.getL("/RATE",10000000);
      // the DMAC=0 DIR=1 triggers the ALT flag behavior
      dmac = pic.ioPort (-1,-1,0,1,bits,rate,0.0,64,0,0);
    }
    else if (func.equals("LOOPIO")) {
      bits = MA.getL("/BITS",16);
      rate = MA.getL("/RATE",10000000);
      dmac = pic.ioPort (pic.IOPT_MODULE,1,-1,-1,bits,rate,0.0,1,0,0); // setup input
      dmao = pic.ioPort (pic.IOPT_MODULE,2, 0, 1,bits,rate,0.0,1,0,pic.FLG_LOOP); // start output
      pic.dmaFunc(dmac,pic.DMA_LOOP); // start input
      tcmode = pic.getKeyL(0,pic.KEY_TCMODE);
      if (tcmode!=0) prcmode=LOOPTC; 
      wait = MA.getD("/POLL");
      delta = 1.0/rate;
    }
    else if (func.equals("SPEC")) {
      todo = 1;
      int size = MA.getL("/SIZE",1024*1024);
      int block = MA.getL("/BLOCK",65536);
      map = pic.mapMem(size*4);
      spec = new DmaSpec();
      spec.block = block;
      if (MA.getState("/DUMP")) {
        todo = MA.getD("/TODO",10);
        hcb = MA.getDataFile("NAME","1000","SL",0);
        hcb.setSize (size);
        hcb.open (hcb.OUTPUT);
        pic.spec (map,spec,0);
        MT.writeln("Specs="+spec);
        hcb.write(map.vaddr,0,(int)map.bytes);
      } else {
        spec.block=-block; 
        pic.spec (map,spec,1);
        spec.block= block; 
        pic.spec (map,spec,1);
      }
    }
    else if (func.equals("SPECS")) {
      todo = MA.getD("/TODO",10);
      hcb = MA.getDataFile("NAME","3000","SF",0);
      hcb.addSubRec ("SRAT","SF");
      hcb.addSubRec ("BRAT","SF");
      hcb.addSubRec ("PRAT","SF");
      hcb.setSize (todo);
      hcb.open (hcb.OUTPUT);
      data = hcb.getDataBuffer(1);
      int size = MA.getL("/SIZE",1024*1024);
      map = pic.mapMem(size*4);
      spec = new DmaSpec();
      spec.block = MA.getL("/BLOCK",65536);
      prcmode=SPECS; 
      replay=rCONTINUOUS;
    }
    else if (func.equals("ROUTE")) {
      int route = pic.getKeyL(node,pic.KEY_ROUTE);
      boolean nopm = pic.getKeyL(0,pic.KEY_PMTYPE1)<=0 && pic.getKeyL(0,pic.KEY_PMTYPE2)<=0;
      String routeList = (node!=0)? pic.routeListPM : nopm? pic.routeListMBP : pic.routeListMB;
      MT.writeln("Node "+node+" Route   = "+Convert.l2x(route)+" = "+Parser.mask2s(routeList,route));
      int routf = pic.getKeyL(node,pic.KEY_ROUTF); 
      MT.writeln("Node "+node+" Routf   = "+Convert.l2x(routf)+" = "+Parser.mask2s(pic.routfList,routf));
      int ios = pic.getKeyL(node,pic.KEY_ROUTIOS); 
      int nios = ~ios;
      int iostat = ios&0XFFFF0000; 
      // now screen for false holdoffs
      if ((route&IceHW.ROUTE_X2PR)!=0) iostat |= (nios&0x0101);
      if ((route&IceHW.ROUTE_X2HP)!=0) iostat |= (nios&0x0202);
      if ((route&IceHW.ROUTE_X2HA)!=0) iostat |= (nios&0x0404);
      if ((route&IceHW.ROUTE_X2HB)!=0) iostat |= (nios&0x0808);
      if ((route&IceHW.ROUTE_X2CA)!=0) iostat |= (nios&0x0010);
      if ((route&IceHW.ROUTE_X2CB)!=0) iostat |= (nios&0x0020);
     if (node==0) {
      if ((route&IceHW.ROUTE_X2TA)!=0) iostat |= (nios&0x3040);
      if ((route&IceHW.ROUTE_X2TB)!=0) iostat |= (nios&0xC080);
     } else {
      if ((route&IceHW.ROUTE_Y2TA)!=0) iostat |= (nios&0x3040);
      if ((route&IceHW.ROUTE_Y2TB)!=0) iostat |= (nios&0xC080);
     }
      if ((route&IceHW.ROUTE_HA2X)==0) iostat &= (~0x40000);
      if ((route&IceHW.ROUTE_HB2X)==0) iostat &= (~0x80000);
      //if (route!=0) MT.writeln("Node "+node+" RoutIOs = "+Convert.l2x(ios)+" = "+Parser.mask2s(pic.iostatList,iostat));
      if (route!=0) MT.writeln("Node "+node+" RoutIOs = "+Convert.l2x(ios)+" = "+Parser.mask2s(pic.iostatList,ios));
    }
    else if (func.equals("DRVFLG")) {
      int drvflg = pic.getKeyL(node,pic.KEY_DRVFLG);
      MT.writeln("Node "+node+" DrvFlg  = "+Convert.l2x(drvflg)+" = "+Parser.mask2s(pic.drvFlgList,drvflg));
    }
    else if (func.equals("TRS")) {
      offset = (node==0)? 0x00100000 : 0x02000000;
      value = pic.rpb(node,offset); offset = (value<<16)>>24;
      MT.writeln("Node "+node+" TRS-Mem = "+Convert.l2x(value)+"  Offset = "+offset);
      offset = (node==0)? 0x40000000 : 0x08000000;
      value = pic.rpb(node,offset); offset = (value<<16)>>24;
      MT.writeln("Node "+node+" TRS-Hyp = "+Convert.l2x(value)+"  Offset = "+offset);
    }
    else if (func.equals("DMAC")) {
      int dmac = MA.getL("NAME");
      pic.dmaDump (dmac,0);
      if (dmac>0) {
        MT.writeln(" McfgList = "+pic.getMcfg(dmac));
        MT.writeln(" FlagList = "+pic.getFlag(dmac));
      }
    }
    else if (func.startsWith("NVR")) {
      pic.nvRam ("",-1);
    }
    else if (func.startsWith("NVM")) {
      int tn = MA.getL("NAME",1);
      pic.nvRam ("",tn);
    }
    else if (func.equals("MAP")) {
      int nbytes = MA.getL("VALUE");
      if (nbytes<=0) nbytes = MA.getL("NAME");
      if (nbytes<=0) nbytes = B1M;
      map = pic.mapMem(nbytes);
      M.info(map.toString());
      MT.getInput("Press return to unmap: ","OK",0);
    }
    else if (func.equals("TCGEN")) {
      hcb = MA.getDataFile("NAME","1000","SI",0);
      hcb.open(hcb.INOUT|hcb.NATIVE); 
      time = MA.getTime("VALUE",null);
      hcb.setTime(time);
      tc.offset=0;
      tc.delta=hcb.getDelta();
      tc.wsec=Math.floor(time.getSoY());
      tc.fsec=time.getFSec();
      map = pic.mapFile(hcb);
      flags = pic.BUF_MASK;
      if (MA.getState("/EXPAND")) flags |= pic.BUF_EXPAND; 
      pic.buffer(map,0,null,(int)map.bytes/2,flags);
      pic.tcInsert(map,tc,0);
    }
    else if (func.equals("WPM")) {
      offset = MA.getL("NAME");
      value = MA.getL("VALUE");
      pic.wpm(node,offset,value);
    }
    else if (func.equals("RPM")) {
      offset = MA.getL("NAME");
      value = pic.rpm(node,offset);
      label = MA.getU("VALUE");
      if (label.length()>0) MR.put(label,value);
      else if (MA.find("/HEX")) MT.writeln("Read "+Convert.l2x(value)+" at "+Convert.l2x(offset));
      else MT.writeln("Read "+value+" at "+Convert.l2x(offset));
    }
    else if (func.equals("WPB")) {
      offset = MA.getL("NAME");
      value = MA.getL("VALUE");
      pic.wpb(node,offset,value);
    }
    else if (func.equals("RPB")) {
      offset = MA.getL("NAME");
      value = pic.rpb(node,offset);
      label = MA.getU("VALUE");
      if (label.length()>0) MR.put(label,value);
      else if (MA.find("/HEX")) MT.writeln("Read "+Convert.l2x(value)+" at "+Convert.l2x(offset));
      else MT.writeln("Read "+value+" at "+Convert.l2x(offset));
    }
    else if (func.equals("WPC")) {
      offset = MA.getL("NAME");
      value = MA.getL("VALUE");
      int size = MA.getL("/SIZE",4);
      pic.wpc(node,offset,value,size);
    }
    else if (func.equals("RPC")) {
      offset = MA.getL("NAME");
      int size = MA.getL("/SIZE",4);
      value = pic.rpc(node,offset,size);
      label = MA.getU("VALUE");
      if (label.length()>0) MR.put(label,value);
      else if (MA.find("/HEX")) MT.writeln("Read "+Convert.l2x(value)+" at "+Convert.l2x(offset));
      else MT.writeln("Read "+value+" at "+Convert.l2x(offset));
    }
    else if (func.equals("MODPPC")) {
      data = MA.getData("VALUE");
      byte[] head = new byte[8];
      int bytes = data.buf.length;
      Convert.packI(head,0,(short)pic.PKTF_MODIFY);
      Convert.packI(head,2,(short)bytes);
      Convert.packL(head,4,0);
      pic.send(node,head,data.buf,bytes,pic.PKT_ACMD);
    }
    else if (func.equals("SEND")) {
      value = MA.getL("VALUE");
      byte[] head = new byte[8];
      byte[] buffer = new byte[1024];
      hcb = MA.getDataFile("NAME","1000,3000","S#,C#",0);
      hcb.open(hcb.INPUT); 
      int bytes = (int)(hcb.size*hcb.dbpe);
      Convert.packI(head,0,(short)pic.PKTF_MEM_WR);
      for (int ndone=0; ndone<bytes; ndone+=1024) {
        int ndo = Math.min(1024,bytes-ndone); 
        Convert.packI(head,2,(short)ndo);
        Convert.packL(head,4,value+ndone);
        hcb.read(buffer,0,ndo);
        pic.send(node,head,buffer,ndo,pic.PKT_ACMD);
      }
      hcb.close();
    }
    else if (func.equals("RECV")) {
      value = MA.getL("VALUE");
      byte[] head = new byte[8];
      byte[] buffer = new byte[1024];
      hcb = MA.getDataFile("NAME","1000,3000","S#,C#",0);
      hcb.open(hcb.INOUT); 
      int bytes = (int)(hcb.size*hcb.dbpe);
      Convert.packI(head,0,(short)pic.PKTF_MEM_RD);
      for (int ndone=0; ndone<bytes; ndone+=1024) {
        int ndo = Math.min(1024,bytes-ndone); 
        Convert.packI(head,2,(short)ndo);
        Convert.packL(head,4,value+ndone);
	int ndid = pic.sendrecv(node,head,buffer,0,ndo,pic.PKT_ACMD,100);
	if (ndid!=ndo) M.warning("Only got "+ndid+" of "+ndo+" bytes");
        hcb.write(buffer,0,ndo);
      }
      hcb.close();
    }
    else if (func.equals("DUMP")) {
      int start = MA.getL("/START",0x0000);
      int size  = MA.getL("/SIZE",0x1000);
      String form = MA.getS("/FORM","SL");
      hcb = MA.getDataFile("NAME","1000",form,0);
      hcb.setSize(size);
      hcb.open(hcb.OUTPUT); 
      data = hcb.getDataBuffer(size);
      int bytes = (int)(size*hcb.dbpe);
      int core  = MA.getL("/CORE");
      if (MA.getState("/PPC")) {
        start |= 0x20000000;
        for (int i=0; i<size; i++) {
  	  int val = pic.read(start+i);
          data.setL(i,val);
        }
        hcb.write(data);
      }
      else if (core>0) {
        for (int i=0; i<size; i++) {
  	  int val = pic.getKeyL(-core,pic.KEY_CORE+start+(i*4));
          data.setL(i,val);
        }
        hcb.write(data);
      }
      else {
        byte[] head = new byte[8];
        byte[] buffer = new byte[1024];
        Convert.packI(head,0,(short)pic.PKTF_MEM_RD);
        for (int ndone=0; ndone<bytes; ndone+=1024) {
          int ndo = Math.min(1024,bytes-ndone); 
          Convert.packI(head,2,(short)ndo);
          Convert.packL(head,4,start+ndone);
  	  int ndid = pic.sendrecv(node,head,buffer,0,ndo,pic.PKT_ACMD,100);
  	  if (ndid!=ndo) M.warning("Only got "+ndid+" of "+ndo+" bytes");
          hcb.write(buffer,0,ndo);
        }
      }
      hcb.close();
    }
    else if (func.startsWith("TRACE")) {
      int core  = MA.getL("/CORE");
      boolean root = (core==8) || (core==9);
      int mode = MA.getState("/RAW")? 3:1;
      int opts = MA.getL("/OPTS",0);
      if ((opts&0x1)!=0) mode = 3;
      if ((opts&0x2)!=0) mode |= 4;
      if ((opts&0x4)!=0) mode |= 8;
      if (MA.getState("/GATE")) mode |= 4;;
      if (MA.getState("/TINV")) mode |= 8;;
      int bit = MA.getL("/BIT",0);
      mode = MA.getL("/MODE",(bit<<4)|mode);
      int ctrl  = MA.getL("/CTRL",root?0x0:0x0FFC);
     if (func.equals("TRACE") || func.equals("TRACEARM")) {
      M.info(" DbgTraceMode="+Convert.l2x(mode));
      pic.setKeyL(-core,pic.KEY_CORE+ctrl,0);
      pic.setKeyL(-core,pic.KEY_CORE+ctrl,mode);
     }
     if (func.equals("TRACE")) {
      if (wait<0) wait = 1;
      Time.sleep(wait);
     }
     if (func.equals("TRACE") || func.equals("TRACEDUMP")) {
      int start = MA.getL("/START",root? 0x0:0x1000);
      int status = pic.getKeyL(-core,pic.KEY_CORE+start);	// readback count
      int size  = MA.getL("/SIZE",status);
      hcb = MA.getDataFile("NAME","1000","SL",0);
      hcb.setSize(size);
      hcb.open(hcb.OUTPUT); 
      data = hcb.getDataBuffer(size);
      int bytes = (int)(size*hcb.dbpe);
      int ocnt = 0;
      boolean compress = (mode&0x03)!=3;
      pic.setKeyL(-core,pic.KEY_CORE+ctrl,0);	// reset is also readback data mode
      for (int i=0; i<size; i++) {
  	int val = pic.getKeyL(-core,pic.KEY_CORE+start);
	data.setL(i,val);
      }
      if (!compress) { hcb.write(data); ocnt=size; }
      else for (int i=0; i<size; i++) {
	int val = data.getL(i);
	int count=(val>>28)&0xF;
        if (count==0);
	for (;count>0; count--) { hcb.write(data.buf,i*4,4); ocnt++; }
      }
      M.info(" DbgTraceDump read="+size+" write="+ocnt+" file="+hcb.getURL());
      hcb.close();
     }
    }
    else if (func.equals("EYESCAN")) {
      int lanes = MA.getL("/LANES",8);
      int lane0 = MA.getL("/LANE0",0);
      int gridp = MA.getL("/GRID",32);
      int grid = gridp-1;
      int bytes = Math.max(1,lanes)*grid*grid*2;
      double xdelta = 1.0/gridp;
      double xstart = -xdelta*(grid-1)/2;
      double ydelta = 1.0/grid;
      double ystart = -ydelta*(gridp-2)/2;
      byte[] buffer = new byte[bytes];
      buffer[0]=(byte)((lane0<<4)|lanes);
      buffer[1]=(byte)grid;
      hcb = MA.getDataFile("NAME","2000","SI",0);
      hcb.setFS(grid);
      hcb.setSize((double)grid);
      hcb.setXStart(xstart);
      hcb.setXDelta(xdelta);
      hcb.setYStart(ystart);
      hcb.setYDelta(ydelta);
      hcb.open(hcb.OUTPUT);
      status = pic.getKey (node,pic.KEY_EYESCAN,buffer,bytes);
      if (status>0) hcb.write(buffer,0,bytes);
      hcb.close();
    }
    else if (func.equals("STR2IP")) {
      String str = MA.getS("NAME");
      label = MA.getU("VALUE");
      value = pic.str2ip(str);
      if (label.length()!=0) {
        if (MA.find("/HEX")) MR.put(label,Convert.l2x(value));
        else MR.put(label,value);
      }
      else MT.writeln("IP Addr = "+ Convert.l2x(value));
    }
    else if (func.equals("JOIN")) {
      String str = MA.getS("NAME");
      value = pic.str2ip(str);
      pic.setKeyL(isfp,pic.KEY_IPCONN,value);
    }
    else if (func.equals("LEAVE")) {
      String str = MA.getS("NAME");
      value = pic.str2ip(str);
      pic.setKeyL(isfp,pic.KEY_IPDISC,value);
    }
    else if (func.equals("PING")) {
      String str = MA.getS("NAME");
      value = pic.str2ip(str);
      pic.setKeyL(isfp,pic.KEY_PING,value);
    }
    else {
      M.warning("Unsupported function: "+func);
    }
    if (prcmode==DONE) return (FINISH);
    else return (NORMAL);
  }

  public int process() {
    if (prcmode==PREP && dir>0) {
      if (map.mode!=map.RAMD) hcb.read(map.vaddr,0,(int)map.bytes);
      if (replay==rSPIN || replay==rLOAD) pic.dmaFunc(dmac,pic.DMA_LOAD);
      if (replay==rLOAD) { status=dmac; prcmode=DONE; }
    }
    if (prcmode==PREP) {
      if (wait>0) Time.sleep(wait); 
      elapse = Time.current();
      pic.dmaFunc(dmac,dmamode);
      if (kick>0) pic.dmaFunc(kick,pic.DMA_SPIN);
      if (replay==rSPIN) prcmode=DONE; else prcmode=WAIT;
      timeout = pic.getKeyL(0,pic.KEY_TIMEOUT);
    }
    if (tcmode!=pic.TCM_OFF) {
      int tcstat = pic.tc (dmac,0.0,delta,tc,pic.FLG_TCINTERP);
      if (tcstat>=0) {
        if (prcmode==LOOPTC) {
          pic.setKeyD(dmao,pic.KEY_TCOFF,tc.wsec,tc.fsec);
          if (wait>0) Time.sleep(wait); // loop forever
          else prcmode=DONE;		// stop after first
        } else {
          hcb.setTime(tc.wsec+tcoff,tc.fsec);
          tcmode = pic.TCM_OFF;
        }
      }
      else if (pic.dmaFunc(dmac,pic.DMA_POLL)==1) {
        if (timeout<0) return NOOP; // make pause
        if (Time.current()-elapse < timeout) return NOOP;
        pic.setKeyL(0,pic.KEY_TIMEOUT,0);
      }
      else M.warning("Bad PIC_TC status="+tcstat);
    }
    if (prcmode==WAIT) {
      if (replay==rCONTINUOUS) return NOOP;
      status = pic.dmaFunc(dmac,pic.DMA_WAIT);	// to check for buffer misses
      elapse = Time.current()-elapse;
      if (status==-1) M.warning ("Transfer timed out");
      if (status==-2) M.warning ("Missed "+pic.dmaFunc(dmac,pic.DMA_LOST)+" blocks");
      if (MA.getState("/SEQCHK")) {
        int seqfil = pic.getKeyL(dmac,pic.KEY_SEQFILL);
        int seqerr = pic.getKeyL(dmac,pic.KEY_SEQERR);
	M.info("SeqFill="+seqfil+" SeqErr="+seqerr);
      }
      pic.dmaFunc(dmac,pic.DMA_STOP);
      pic.dmaFunc(dmac,pic.DMA_CANCEL);
      float xtime = (float)elapse;
      double bytes = (double)map.bytes;
      if (dmamode>1) bytes *= dmamode;
      float xrate = (float)(bytes/elapse/1e6);
      if (verbose) M.info ("Stat="+status+"  Took="+xtime+"sec   Rate="+xrate+"Mby/sec");
      prcmode=DONE;
    }
    if (prcmode==DONE && dir<0) {
      if (map.mode!=map.RAMD) hcb.write(map.vaddr,0,(int)map.bytes);
    }
    if (prcmode==SPECS) {
      if (replay==rSPIN) return NOOP;
      spec.block = Math.abs(spec.block);
      if (prcmode!=DONE) status = pic.spec (map,spec,0);
      if (status<0) prcmode=DONE;
      data.packF(0,spec.srate);
      data.packF(4,spec.brate);
      spec.block = -spec.block;
      if (prcmode!=DONE) status = pic.spec (map,spec,0);
      if (status<0) prcmode=DONE;
      data.packF(8,spec.srate);
      hcb.write(data);
      if (prcmode==DONE) M.error("Bad PCI.SPEC status="+status);
      if (--todo==0) prcmode=DONE;
    }
    if (prcmode==DONE) return (FINISH);
    else return (NORMAL);
  }

  public int close() {
    if (replay==rCONTINUOUS) {
       pic.dmaFunc(dmac,pic.DMA_STOP);
       pic.dmaFunc(dmac,pic.DMA_CANCEL);
    }
    if (status>=0 && MA.find("/GOLDTEST")) status = testit();
    if (hcb!=null) hcb.close();
    if (map!=null) map.close();
    if (pic!=null) pic.close();
    if (maps!=null) for (int i=1; i<maps.length; i++) maps[i].close();
    if (hcbs!=null) for (int i=1; i<hcbs.length; i++) hcbs[i].close();
    MR.put(MA.getU("/STAT"),status);
    return (NORMAL);
  }

  public void setBreak (int value) {
    pic.setKeyL(dmac,pic.KEY_BREAK,value);
  }
  public void setReplay (String value) {
    replay = Parser.find(replayList,value,-1,replayListOffset);
  }
  public void setBlock (int value) { spec.block = value; }
  public int getBlock () { return spec.block; }

  private int testit() {
    DataFile tdf = MA.getDataFile("/GOLDTEST","1000","SB,SI,CB,CI",0);
    tdf.open(tdf.INPUT);
    Data dat = tdf.getDataBuffer((int)tdf.size);
    int nper = (int)(tdf.getDataSize()/4);
    int npass = (int)(hcb.getDataSize()/4/nper);
    int skip = (int)(tdf.keywords.getL("SKIP",0)*hcb.dbpe/4);
    int mask = ~tdf.keywords.getL("MASK",0);
    tdf.read(dat);
    tdf.close();
    int off, nbad=0;
    map.setBufferSize(nper*4);
    for (int ipass=0; ipass<npass; ipass++) {
      map.getBuffer (ipass*nper*4,nper*4);
      for (int iper=skip; iper<nper; iper++) {
	off = iper; 
	int tword = Convert.unpackL(dat.buf,off*4) & mask;
	int dword = Convert.unpackL(map.buf,off*4) & mask;
	if (dword!=tword) { 
	  if (++nbad>40) return -3;
	  String stword = Convert.bb2hex(dat.buf,off*4,4);
	  String sdword = Convert.bb2hex(map.buf,off*4,4);
	  String sxword = Convert.l2x(tword^dword);
	  String sxaddr = Convert.l2x(ipass*nper+off);
	  M.info ("Got "+sdword+" Expect "+stword+" Diff "+sxword+" at "+sxaddr+" "+(ipass*nper)+"+"+off);
	}
      }
    }
    if (nbad>1) return -3;
    else return 1;
  }

  private void create() {
    String format = MA.getS("NAME","CI");
    String ramaux = MR.getS("ENV.RAMAUX");
    double dbpe = Data.getBPA(format);
    if (dbpe<0) dbpe = -.125*dbpe;
    int multi = MA.getL("/MULTI",1);
    int round = MA.getL("/ROUND",65536);
    int spb = (int)Math.round(round/dbpe);
    int minr = MA.getL("/MINRBUF",1);
    double size = MA.getD("VALUE",1024.0*1024);
    double osize = spb*Math.max(minr,Math.floor(.5+size/spb));
    if (MA.getState("/POWER2")) osize = (double)MMath.power2GE((int)osize);
    if (MA.getState("/EXACT")) osize = size;
    double sr = MA.getD("PORT"); if (sr<0) sr=1e6;
    DataFile hcb = MA.getDataFile("DEV");
    hcb.setFormat(format);
    hcb.setSize(osize*multi);
    hcb.setXDelta(1.0/sr);
    hcb.setXUnits(Units.TIME);
    if (ramaux!=null) {
      hcb.setQualifier("CTG","1");
      hcb.setQualifier("DET","1");
      hcb.setAux(ramaux);
    }
    hcb.open(hcb.OUTPUT); 
    int status = hcb.isOpen()? 1 : -1;
    hcb.seek(hcb.size);
    hcb.update();
    hcb.close();
    MR.put(MA.getU("/STAT"),status);
  }
}
