package nxm.ice.prim;

import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.sys.libm.MMath;

import nxm.ice.lib.DevIce;
import nxm.ice.lib.MDevIce;
import nxm.ice.lib.FileIO;

import nxm.sys.libg.GMenu;
import nxm.sys.libg.MWindow;
import nxm.sys.libg.MJBrowser;
import nxm.sys.libg.MJFrame;
import nxm.sys.libg.MBox;

import java.io.File;

/**
  Utilities for ICE development.

  FUNC,P1,P2,P3

  @author Jeff Schoen
  @version %I%, %G%
*/
public class iceutil extends Primitive {


  public int open() {

    String func = MA.getU("FUNC");
    if (verbose) M.info("IceUtil f="+func+" p1="+MA.getCS("P1")+" p2="+MA.getCS("P2"));
         if (func.equals("GC")) System.gc();
    else if (func.equals("TTF2M")) ttf2m();
    else if (func.equals("BIT2M")) bit2m();
    else if (func.equals("FSTATS")) fstats();
    else if (func.equals("TSTATS")) tstats();
    else if (func.equals("DSTATS")) dstats();
    else if (func.equals("RBF2M")) rbf2m();
    else if (func.equals("ELF2M")) elf2m();
    else if (func.equals("STR2IP")) str2ip();
    else if (func.equals("IPX2IP")) ipx2ip();
    else if (func.equals("SETKEYS")) setkeys();
    else if (func.equals("MASK")) mask();
    else if (func.equals("CARETS")) carets();
    else if (func.equals("BUTTON")) button();
    else if (func.equals("RUN")) runit();
    else if (func.equals("JVM2XBR")) jvm2xbr();
    else if (func.equals("M2XBR")) m2xbr();
    else if (func.equals("M2VBR")) m2vbr();
    else if (func.equals("M2MIF")) m2mif();
    else if (func.equals("MBTST")) mbtst();
    else if (func.equals("JTAG2M")) jtag2m();
    else if (func.equals("LST2CSP")) lst2csp();
    else if (func.equals("SYM2PART")) sym2part();
    else if (func.equals("NET2UCF")) net2ucf();
    else if (func.equals("ENET2UCF")) enet2ucf();
    else if (func.equals("QNET2UCF")) qnet2ucf();
    else if (func.equals("VPP")) vppflow();			// Verilog PreProcessor
    else if (func.equals("TPP")) tppflow();			// Table PreProcessor
    else if (func.equals("UCF2XDC")) ucf2xdc();
    else if (func.equals("GERBER")) gerber();
    else if (func.equals("CRCGEN")) crcgen();
    else if (func.equals("MODARGS")) modargs();
    else if (func.equals("COMPRESS")) compress(1);
    else if (func.equals("UNCOMPRESS")) compress(-1);
    else if (func.equals("GETID")) getid();
    else if (func.equals("GREP")) grep();
    else if (func.equals("REPTBL")) reptbl();
    else if (func.equals("LOGGER")) logger();
    else if (func.equals("GETFILE")) getfile();
    else if (func.equals("DBGFILE")) dbgfile();
    else if (func.equals("TPUT")) tput();
    else if (func.equals("SCANUP")) scanup();
    else if (func.equals("GREPLOG")) greplog();
    else if (func.equals("KVI")) kvi();
    else M.error ("Unsupported function: "+func);

    return (FINISH);
  }


  private void getfile() {
    String path = MA.getCS("P1");
    File file = new File(path);
    if (file.exists()) {
      System.out.println("Exists "+ new File(path).exists());
    } else {
      System.out.println("NExists "+path);
    }
  }

  public static Logger logger;
  private void logger() {
    String str = MA.getS("P1");
    if (str.equals("PAUSE")) {
      logger = M.setLogger(null);
    }
    if (str.equals("RESUME")) {
      M.setLogger(logger);
    }
  }

  private void mask() {
    String str = MA.getS("P1");
    String mat = MA.getS("P2");
    String lab = MA.getU("P3");
    MR.put(lab,Parser.mask(str,mat,0));
  }

  private void carets() {
    String str = MA.getS("P1");
    String lab = MA.getU("P2");
    MR.put(lab,MA.evaluateCarets(str));
  }

  private void reptbl() {
    Table t = MA.getTable("P1");
    String rep = MA.getS("P2");
    t.clear();
    t.append(rep);
  }

  private void button() {
    String label = MA.getCS("P1");
    String list = MA.getCS("P2");
    int flags = GMenu.BUTTON|GMenu.TOFF|GMenu.ONEROW|GMenu.INPUT|GMenu.PERM;
    GMenu menu = new GMenu(null,"",list,0,flags,M.macro);
    menu.setMidas(M);
    menu.setLabel(label);
    WidgetPanel parent = (WidgetPanel)M.registry.get("PANEL");
    parent.addWidget(menu);
  }

  private void runit() {
    String cmd = MA.getRawArgs().trim().substring(4);
    Shell.runCommand(M, cmd);
  }

  private void setkeys() {
    Object obj = MA.getO("P1");
    Table  tbl = MA.getTable("P2");
    KeyObject.setKeys(obj,tbl);
  }

  private void dbgfile() {
    DataFile df = MA.getDataFile("P1","1000","SL",0);
    df.open(df.INPUT);
    Data db = df.getDataBuffer(4);
    int trig = MA.getL("/TRIG",20);
    int wind = MA.getL("/WIND",10);
    boolean totals = MA.getState("/TOTALS");
    boolean comps = MA.getState("/COMPS");
    int xind = wind; 
    int wtot = 0;
    int rtot = 0;
    int ttot = 0;
    int pass = 0;
    int i3l=0,len;
    for (int i=0; i<df.getSize(); i+=4) {
      df.read(db);
      int i0 = db.getL(0);
      int i1 = db.getL(1);
      int i2 = db.getL(2);
      int i3 = db.getL(3);
      if (i3l==0) i3l=i3;
      int tdif = ((i3-i3l)>>4)&0xFFFFFFF;
      ttot += tdif;
      if (tdif>trig && !comps) {
	if (xind<0) System.out.printf("... wtot %d    rtot %d    ttot %d   pass %d ...\n",wtot,rtot,ttot,pass);
	if (xind<0) { rtot=wtot=ttot=0; pass++; }
	xind = wind;
      }
      int rt = -1;
      String op = null;
      String adr = "     ";
      if ((i3&1)!=0) {
	rt = (i2>>11)&0x1F;
	op = (rt==0)? "mrd" : (rt==1)? "mwr" : (rt==12)? "msg" : null;
        len = (i2&0x7FF)*4;
	if (i1==0x0800) adr = "cque0";
	if (i1==0x0900) adr = "cque1";
	if (i1==0x0a00) adr = "sque0";
	if (i1==0x0c00) adr = "sque1";
	if (i1==0x1000) adr = "bram ";
	if (i1==0x2000) adr = "xmem ";
	if (i1==0x2000 && rt==1) wtot += len;
	if (i1==0x2000 && rt==0) rtot += len;
	if (i1==0x0900 && comps) { 
	  System.out.printf("Comp ... wtot %d    rtot %d    ttot %d   pass %d ...\n",wtot,rtot,ttot,pass);
	  rtot=wtot=ttot=0; pass++;
	}
      } else {
        op = "cpl";
        len = (i1&0x3FF)*4;
      }
      if (op==null) System.out.printf("Dbg [ %08x %08x %08x %08x ]   unknown Op\n",i0,i1,i2,i3);
      if (xind>0 && !totals) System.out.printf("Dbg [ %08x %08x %08x %08x ]    %s   %s   %4d   %d\n",i0,i1,i2,i3,op,adr,len,tdif);
      xind--;
      i3l = i3;
    }
    df.close();
  }


  private void ttf2m() {
    TextFile tf = MA.getTextFile("P1");
    tf.setExt("ttf");
    tf.open(tf.INPUT);
    DataFile df = MA.getDataFile("P2","1000","SB",0);
    df.setComment("ICEPIC IOC Load Program");
    df.open(df.OUTPUT);
    byte[] buffer = new byte[128];
    String line;
    while ( (line=tf.read()) != null) {
      int n = (line.length()+1)/4;
      if (n>128) M.error("Bad TTF line length");
      for (int i=0; i<n; i++) {
        int k = Convert.s2l(line.substring(i*4,i*4+3));
        if (k>127) buffer[i]=(byte)(k-256); else buffer[i]=(byte)k;
      }
      df.write (buffer,0,n);
    }
    tf.close();
    df.close();
  }

  private boolean chk (byte[] a, byte[] b, int n) {
    for (int i=0; i<n; i++) if (a[i]!=b[i]) return false;
    return true;
  }

  private void dstats() {
    String aux = MA.getS("P1");
    String out = MA.getU("P2");
    String path = M.io.getAuxPath(aux);
    try {
      IOResource res = IOResource.getInstance(M,0,path,0,null);
      Table stats = res.getStatFS();
      MR.put(out,stats);
    } catch (Exception ex) {
      M.warning("Unable to get disk statistics for AUX="+aux+" at path="+path+ " cause="+ex.getMessage());
    }
  }

  private void fstats() {
    DataFile bf = MA.getDataFile("P1");
    bf.setExt("mlog");
    bf.open(bf.INPUT);
    int size = (int)bf.getSize();
    DataFile df = MA.getDataFile("P2","3000","NH",0,0,null);
    df.setSubRecords("FAVG|SL,FMIN|SL,FMAX|SL");
    Data dfb = df.getDataBuffer(1);
    df.open(df.OUTPUT);
    String id = MA.getS("P3");
    for (int i=0; i<size; i++) {
      Table t = bf.getDataTable((double)i);
      if (t==null) continue;
      String from = t.getS("FROM");
      String type = t.getS("TYPE");
      String line = t.getS("TEXT");
      if (from!=null && from.equals(id) && type.equals("INFO") ) {
        int ti = line.indexOf("STATS=");
        if (ti>0) {
	  boolean hasM = line.indexOf("FMIN=") > 0; 
	  Table to = new Table(line.substring(ti+6));
	  int favg = to.getL("FAVG");
	  int fmin = hasM? to.getL("FMIN"):favg;
	  int fmax = hasM? to.getL("FMAX"):favg;
          dfb.packL(0,favg);
          dfb.packL(4,fmin);
          dfb.packL(8,fmax);
	  df.write(dfb);
        }
      }
    }
    bf.close();
    df.close();
  }

  private byte getTemp(String line, String item) {
    int i = line.indexOf(item+"=");
    if (i<0) return -1;
    line = line.substring(i+item.length()+1);
    int j = line.indexOf(" ");
    if (j>0) line = line.substring(0,j);
    return (byte)Convert.s2l(line);
  }


  private void tstats() {
    DataFile bf = MA.getDataFile("P1");
    bf.setExt("mlog");
    bf.open(bf.INPUT);
    int size = (int)bf.getSize();
    DataFile df = MA.getDataFile("P2","3000","NH",0,0,null);
    df.setSubRecords("XBAR|SB,IOM1|SB,PM1|SB,PM2|SB,ARX|SB");
    Data dfb = df.getDataBuffer(1);
    df.open(df.OUTPUT);
    String id = MA.getS("P3");
    double tstart=0,tdelta=60;
    for (int i=0; i<size; i++) {
      Table t = bf.getDataTable((double)i);
      if (t==null) continue;
      if (tstart==0) tstart = t.getD("TIME");
      String from = t.getS("FROM");
      String type = t.getS("TYPE");
      String line = t.getS("TEXT");
      if (type.equals("INFO") ) {
        int ti = line.indexOf("Temps:");
	if (ti>0) {
	  dfb.buf[0] = getTemp(line,"XBar");
	  dfb.buf[1] = getTemp(line,"IOM1");
	  dfb.buf[2] = getTemp(line,"PM1");
	  dfb.buf[3] = getTemp(line,"PM2");
        }
        ti = line.indexOf("AR26=");
        if (ti>0) {
	  dfb.buf[4] = getTemp(line,"Temp1");
	  df.write(dfb);
        }
      }
    }
    df.setXStart(tstart);
    df.setXDelta(tdelta);
    df.setXUnits(1);
    bf.close();
    df.close();
  }

  private void bit2m() {
    BaseFile bf = MA.getBaseFile("P1");
    bf.setExt("bit");
    bf.open(bf.INPUT);
    int bytes = (int)bf.getSize();
    int flash = MA.getL("/FLASH"); 
    int nosync = MA.getL("/NOSYNC"); 
    byte[] buf = new byte[bytes+16];
    byte[] syncpattern = { -1,-1,-1,-1,-86,-103,85,102};
  if (nosync>=0) {
    bf.read (buf,0,bytes);
    bf.close();
  } else {
    // find the sync pattern
    for (int t=0; ; t++) {
      for (int i=0; i<7; i++) buf[i]=buf[i+1];
      buf[7] = bf.readByte(); bytes--;
      if (chk(buf,syncpattern,8)) break;
      if (t>256) M.error("No sync found");
    }
    bytes -= 4; // to match old code
    bf.read (buf,8,bytes);
    bytes += 8;
    bf.close();
  }
    int quads = bytes/4;
  if (nosync<0) {
    Convert.swap4(buf,0,quads); 	// swap byte ordering for JTAG writes
  }
    for (;(quads%4)!=0;quads++,bytes+=4) System.arraycopy( buf,bytes-4, buf,bytes, 4); // extend to a 4 quad boundary
    int crc=0; for (int i=0; i<bytes; i+=4) crc = crc32(crc,Convert.unpackL(buf,i),32);
               for (int i=bytes; i<flash; i+=4) crc = crc32(crc,-1,32);
    DataFile df = MA.getDataFile("P2","1000","SL",0,quads,null);
    df.setComment("ICE XILINX FPGA Load Program");
    df.keywords.putMain("CRC",Convert.l2x(crc),false);
    df.open(df.OUTPUT);
    df.write (buf,0,bytes);
    df.close();
  }

  private void compress (int dir) {
    int ilen,olen,mlen;
    byte[] dbi,dbo,dbm;
    MDevIce p = new MDevIce(); p.load();

    DataFile dfi = MA.getDataFile("P1","1000","SB,SL",0);
    dfi.open();
    DataFile dfo = MA.getDataFile("P2",dfi,0);
    dfo.open();

    ilen = (int)(dfi.size*dfi.getBPE());
    dbi = new byte[ilen];
    dfi.read(dbi,0,ilen);
    dfi.close();

    if (dir>0) {
      int blen = ilen+512;
      dbm = new byte[blen];
      dbo = dbi;
      mlen = p.compress(dbi,ilen,dbm,blen,1);
      olen = p.compress(dbm,mlen,dbo,blen,1);
      dfo.keywords.putMain("ICSZ",""+ilen,false);
      dfo.keywords.putMain("ICSZM",""+mlen,false);
      String scrc = dfo.keywords.getMain("CRC");
      int crc = dfo.keywords.getL("CRC",0);
      if (scrc==null&&crc!=0) {
        dfo.keywords.delete("CRC");
        dfo.keywords.putMain("CRC",Convert.l2x(crc),false);
      }
      dfo.setFormat("SB");
    } else {
      olen = dfo.keywords.getL("ICSZ",0);
      mlen = dfo.keywords.getL("ICSZM",0);
      int blen = olen+512;
      dbm = new byte[blen];
      dbo = new byte[blen];
      p.compress(dbi,ilen,dbm,mlen,-1);
      p.compress(dbm,mlen,dbo,olen,-1);
      dfo.keywords.deleteMain("ICSZ");
      dfo.keywords.deleteMain("ICSZM");
      String comment = dfo.getComment();
      if (comment.indexOf("XILINX")>0) dfo.setFormat("SL");
    }
    dfo.write(dbo,0,olen);
    dfo.close();
  }


  private int crc32 (int crc, int data, int bits) {
    for (int i=0; i<bits; i++) {
      int val = ((data>>i)^(crc>>31))&0x1;
      crc = ((crc<<1)&0xFFFFFFFE) | val;
      if (val!=0) crc ^= 0xEDB88320;
    }
    return crc;
  }

  private int crc32r (int crc, int data, int bits) {
    for (int i=bits; i>=0; i--) {
      int val = ((data>>i)^(crc>>31))&0x1;
      crc = ((crc<<1)&0xFFFFFFFE) | val;
      if (val!=0) crc ^= 0xEDB88320;
    }
    return crc;
  }

  private void rbf2m() {
    BaseFile bf = MA.getBaseFile("P1");
    int maxpad = 32*2014;
    int st = MA.getL("P3");
    bf.setExt("rbf");
    bf.open(bf.INPUT);
    byte[] pad = new byte[maxpad];
    for (int i=0; i<maxpad; i++) pad[i] = -1;
    int wp = (st==1)? 22 : 0;
    int fp = (st==2)? 1048 : (st==4)? 396 : (st==5)? 32*1024 : 0;
    int bp = (st==2)? 4 : (st==3)? 1094 : (st==4)? 16 : 0;
    System.out.println("Padding for Stratix type="+st+" word="+wp+" front="+fp+" back="+bp);
    // align to 4 byte boundary
    int ibytes = (int)bf.getSize();
    int obytes = (ibytes+3)&0xFFFFFFFC;
    int off = obytes-ibytes;
    byte[] buf = new byte[ibytes];
    bf.read (buf,0,ibytes);
    bf.close();
    DataFile df = MA.getDataFile("P2","1000","SB",0,obytes,null);
    df.setComment("ALTERA FPGA Load Program");
    df.keywords.putMain("CRC",Convert.l2x(0),false);
    df.open(df.OUTPUT);
    df.write (pad,0,fp);	// front pad
    df.write (pad,0,off+wp);	// front pad
    df.write (buf,0,ibytes);
    if (st==2) { pad[0]=pad[2]=pad[3]=0; pad[1]=0x4; }
    df.write (pad,0,bp);
    df.close();
  }

  private void elf2m() {
    int bytes;
    BaseFile bf = MA.getBaseFile("P1");
    bf.setExt("elf");
    bf.open(bf.INPUT);
    bytes=(int)bf.getSize();
    byte[] buf = new byte[bytes];
    bf.read (buf,0,bytes);
    bf.close();
    DataFile df = MA.getDataFile("P2","1000","SB",0,bytes,null);
    df.setComment("ICE XILINX PPC Load Program");
    df.open(df.OUTPUT);
    df.write (buf,0,bytes);
    df.close();
    // create segments for DSRAM and ISRAM
    byte[] dsram = new byte[0x1000]; for (int i=0; i<0x1000; i++) dsram[i]=0;
    byte[] isram = new byte[0x1000]; for (int i=0; i<0x1000; i++) isram[i]=0;
    // analyse header
    Convert.swap4(buf,20,5);
    Convert.swap2(buf,40,6);
    String ident = Convert.unpackS(buf,0,16);
    int psecs = Convert.unpackI(buf,44);
    int hoffs = Convert.unpackL(buf,28);
    int hsize = Convert.unpackI(buf,42);
    M.info("Ident="+ident+" PSECs="+psecs+" Off="+hoffs+" Size="+hsize);
    for (int i=0; i<psecs; i++) {
      int soff = hoffs+i*hsize;
      Convert.swap4(buf,soff,8);
      int vaddr = Convert.unpackL(buf,soff+8);
      int filsz = Convert.unpackL(buf,soff+16);
      int memsz = Convert.unpackL(buf,soff+20);
      int poffs = Convert.unpackL(buf,soff+4);
      M.info("Sec="+i+" Vaddr="+Convert.l2x(vaddr)+" Offst="+Convert.l2x(poffs)+
                      " FilSz="+Convert.l2x(filsz)+" MemSz="+Convert.l2x(memsz));
      if ((vaddr&0xFFFF0000)==0xFFFF0000) System.arraycopy(buf,poffs, isram,vaddr&0xFFF, filsz);
      if ((vaddr&0xFFFF0000)==0xFD000000) System.arraycopy(buf,poffs, dsram,vaddr&0xFFF, filsz);
    }
    writeRamInit("nxm.ice.xfer.ppccode.h",isram,0,0x1000,3,2,4);
    writeRamInit("nxm.ice.xfer.ppcdata.h",dsram,0,0x1000,1,2,2);
    writeMemInit("nxm.ice.xfer.ppccode.mem",isram,0,0x1000,0xfffff000);
    writeMemInit("nxm.ice.xfer.ppcdata.mem",dsram,0,0x1000,0xfd000000);
  }

  /** Midas bit test program */
  private void mbtst() {
    DataFile df = MA.getDataFile("P1","1000","SB,SI,SL",0);
    df.open();
    int  nd = (int)df.getSize();
    Data db = df.getDataBuffer(nd);
    df.read(db);
    df.close();
    int bits = df.getBPA()*8;
    int[] tally = new int[32];
    int data=0;
    for (int i=0; i<nd; i++) {
      if (bits==8) data = db.getB(i);
      if (bits==16) data = db.getI(i);
      if (bits==32) data = db.getL(i);
      for (int b=0,m=1;b<16; b++,m<<=1) {
	if ((data&m)!=0) tally[b]++;
      }
    }
    MT.writeln("Analysis of "+nd+" samples from URL="+df.getURL());
    for (int i=0; i<bits; i++) {
      float avg = (float)tally[i]/nd;
      MT.writeln("Bit Number="+i+" Average="+avg);
    }
  }

  private void str2ip() {
    int i;
    String str = MA.getS("P1");
    String label = MA.getU("P2");
    String hwlab = MA.getU("P3");
    String ip = "";
    int ic = str.indexOf(":");
    if (ic>0) str = str.substring(0,ic);
    i = str.indexOf("."); ip += Convert.l2x(Convert.s2l(str.substring(0,i))).substring(8); str = str.substring(i+1);
    i = str.indexOf("."); ip += Convert.l2x(Convert.s2l(str.substring(0,i))).substring(8); str = str.substring(i+1);
    i = str.indexOf("."); ip += Convert.l2x(Convert.s2l(str.substring(0,i))).substring(8); str = str.substring(i+1);
                          ip += Convert.l2x(Convert.s2l(str)).substring(8);
    MR.put(label,ip);
    MR.put(hwlab,"1c:e0:"+ip.substring(0,2)+":"+ip.substring(2,4)+":"+ip.substring(4,6)+":"+ip.substring(6,8));
  }

  private void ipx2ip() {
    int i;
    String str = MA.getS("P1");
    int    ix = MA.getL("P2");
    String iplab = MA.getU("P3");
    String ip = str;
    int ic = str.indexOf("X");
    char c = (ic>0)? str.charAt(ic-1) : '.';
    if (ic<0);	// all done
    else if (c=='+' || c=='|') {
      int ib; for (ib=ic-2; ib>=0 && str.charAt(ib)!='.'; ib--);
      int num = Convert.s2l(str.substring(ib+1,ic-1));
      if (c=='+') num += ix;
      if (c=='|') num |= ix;
      ip = str.substring(0,ib+1) + num + str.substring(ic+1);
    }
    else ip=str.replaceAll("X",""+ix);
    MR.put(iplab,ip);
  }

  /** Midas 2 Xilinx Block Ram */
  private void m2xbr() {
    DataFile df = MA.getDataFile("P1","1000","S#,C#",0);
    df.open();
    int bytes = (int)df.getDataSize();
    byte[] buf = new byte[bytes];
    df.read(buf,0,bytes);
    df.close();
    boolean auto = MA.getState("/ENUM");
    boolean auto2 = MA.getState("/ENUM2");
    boolean special = MA.getState("/SPECIAL");
    String pre = MA.getCS("/PREFIX");
    pre = StringUtil.isNullOrEmpty(pre)? "" : pre+".";

    TextFile tf = MA.getTextFile("P2");
    tf.open(tf.OUTPUT);
    int banks = MA.getL("/BANKS",1);
    int bpe  = MA.getL("/BPA",df.bpa);
    int npb  = bpe*2/banks;
    int bper = banks*256/8;
    for (int i=0,l=0; i<bytes; i+=bper,l++) {
      String id = Convert.l2x(l).substring(8);
      String hdata = Convert.bb2hex(buf,i,bper);
      for (int j=0; j<banks; j++) {
        String data = ""; int u=0;
        for (int k=j*npb; k<bper*2; k+=(banks*npb)) {
	  int o = 2 + bper*2 - k - npb;
 	  if (u>0 && u%8==0) data="_"+data;
	  data = hdata.substring(o,o+npb) + data;
	  u+=npb; 
	}
        if (auto) tf.writeln("defparam "+pre+"dpw["+j+"].sdr.inst.ram.INIT_"+id+" = 256'h"+data+";");
        else if (auto2) tf.writeln("defparam "+pre+"ram"+j+".INIT_"+id+" = 256'h"+data+";");
	else if (special) tf.writeln("defparam "+pre+"INIT_"+id+" = 256'h"+data+";");
	else      tf.writeln("defparam "+pre+"ram"+j+".INIT_"+id+" = 256'h"+data+";");
      }
    }
    tf.close();
  }

  /** Midas 2 Verilog Block Ram */
  private void m2vbr() {
    DataFile df = MA.getDataFile("P1","1000","S#,C#",0);
    df.open();
    int bytes = (int)df.getDataSize();
    byte[] buf = new byte[bytes];
    df.read(buf,0,bytes);
    df.close();

    String ram = MA.getCS("/RAM","ram");
    TextFile tf = MA.getTextFile("P2");
    tf.open(tf.OUTPUT);
    int bpa  = MA.getL("/BPA",df.bpa);
    int bitpa = bpa*8;
    for (int i=0,l=0; i<bytes; i+=bpa,l++) {
      String data = "  assign "+ram+"["+l+"] = "+bitpa+"'h";
      data += Convert.bb2hex(buf,i,bpa).substring(2);
      tf.writeln(data+";");
    }
    tf.close();
  }

  /** Midas 2 Memory Info File Block Ram */
  private void m2mif() {
    DataFile df = MA.getDataFile("P1","1000","S#,C#",0);
    df.open();
    int bytes = (int)df.getDataSize();
    byte[] buf = new byte[bytes];
    df.read(buf,0,bytes);
    int bpa  = MA.getL("/BPA",df.bpa);
    int depth = bytes / bpa;
    int bitpa = bpa*8;
    df.close();

    TextFile tf = MA.getTextFile("P2");
    tf.open(tf.OUTPUT);
    tf.writeln("WIDTH="+bitpa+";");
    tf.writeln("DEPTH="+depth+";");
    tf.writeln("ADDRESS_RADIX=UNS;");
    tf.writeln("DATA_RADIX=HEX;");
    tf.writeln("CONTENT BEGIN");
    for (int i=0,l=0; i<bytes; i+=bpa,l++) {
      tf.writeln("  "+l+" : "+Convert.bb2hex(buf,i,bpa).substring(2)+";");
    }
    tf.writeln("END;");
    tf.close();
  }

  /** JVM Midas 2 Xilinx Block Ram */
  private void jvm2xbr() {
    DataFile df = MA.getDataFile("P1","1000","SI",0);
    df.open();
    int bytes = (int)df.getDataSize();
    int bper = 256/8;
    byte[] buf = new byte[bytes];
    byte[] xbuf = new byte[bper];
    df.read(buf,0,bytes);
    df.close();
    String pre = MA.getCS("/PREFIX","inst")+".";
    int size = MA.getL("/SIZE");
    TextFile tf = MA.getTextFile("P2");
    tf.open(tf.OUTPUT);
    for (int i=0,l=0,ir=0; i<size; i+=bper,ir++) {
      for (int ii=0; ii<bper; ii++,l+=2) xbuf[ii]=(l<bytes)?buf[l]:0;
      String hdata = Convert.bb2hex(xbuf,0,bper);
      String id = Convert.l2x(ir).substring(8);
      tf.writeln("defparam "+pre+"INIT_"+id+" = 256'h"+hdata.substring(2)+";");
    }
    int pbper = 256/1;
    for (int i=0,l=1,ir=0; i<size; i+=pbper,ir++) {
      for (int ii=0; ii<bper; ii++) xbuf[ii]=0;
      for (int ii=0; ii<pbper; ii++,l+=2) if (l<bytes && buf[l]==1) xbuf[ii>>3] |= (0x1<<(ii&7));
      String hdata = Convert.bb2hex(xbuf,0,bper);
      String id = Convert.l2x(ir).substring(8);
      tf.writeln("defparam "+pre+"INITP_"+id+" = 256'h"+hdata.substring(2)+";");
    }
    tf.close();
  }

  private void getid() {
    String dev = MA.getCS("P1");
    int fd = FileIO.fopen(dev,-1,0);
    long fid = 0;
    if (fd>0) {
      fid = FileIO.fid(fd);
      FileIO.fclose(fd);
    }
    System.out.printf("GetFileId file=%s fd=%d fid=%x\n",dev,fd,fid);
  }

  private void grep() {
    String value = "NULL";
    String name = MA.getCS("P1");
    TextFile tf = MA.getTextFile("P2");
    tf.open(tf.INPUT);
    for (;value.equals("NULL");) {
      String line=tf.read();
      if (line==null) break;
      if (line.indexOf(name)>=0) value=line;
    }
    tf.close();
    MR.put(MA.getU("P3"),value);
  }

  private void writeRamInit (String name, byte[] buf, int off, int size, 
					int bank, int banks, int banksz) {
    TextFile tf = new TextFile(name);
    tf.open(tf.OUTPUT);
    int bper = 256/8;
    Convert.swap4(buf,off,size/4);
    for (int i=0,l=0; i<size; i+=2*bper,l++) {
      String id = Convert.l2x(l).substring(8);
      for (int j=bank; j<bank+banks; j++) {
        String data = "";
        for (int k=j-bank; k<2*bper/banksz; k+=2) {
	  data = Convert.bb2hex(buf,off+i+k*banksz,banksz).substring(2) + data;
          data = "_"+data;
        }
        data = data.substring(1);
        tf.writeln("defparam ppe.ram"+j+".INIT_"+id+" = 256'h"+data+";");
      }
    }
    Convert.swap4(buf,off,size/4);
    tf.close();
  }

  private void writeMemInit (String name, byte[] buf, int off, int size, int addr) {
    TextFile tf = new TextFile(name);
    tf.open(tf.OUTPUT);
    int bper = 256/32;
    Convert.swap4(buf,off,size/4);
    tf.writeln("@"+Convert.l2x(addr).substring(2));
    for (int i=0; i<size; i+=bper*4) {
      String data = "";
      for (int j=i; j<i+bper*4; j+=4) {
	data += Convert.bb2hex(buf,off+j,4).substring(2);
        data += " ";
      }
      tf.writeln(data);
    }
    Convert.swap4(buf,off,size/4);
    tf.close();
  }

  private static String[] rowlabel = {"A","B","C","D","E","F","G","H","J","K",
    "L","M","N","P","R","T","U","V","W","Y","AA","AB","AC","AD","AE","AF","AG",
    "AH","AJ","AK","AL","AM","AN","AP","AR","AT","AU","AV","AW","AY","BA","BB"};

  private void lst2csp() {
    String line;
    int nrow = MA.getL("/ROWS");
    int ncol = MA.getL("/COLS",nrow);
    int npin = MA.getL("/PINS",ncol*nrow);
    boolean reorder = MA.getState("/REORDER");
    if (npin<0 || nrow<0) M.error("Must specify /ROWS= and /PINS=");
    TextFile tf = MA.getTextFile("P1"); tf.setExt("lst");
    if (!tf.open(tf.INPUT)) M.error("Couldnt open "+tf.getURL());
    TextFile xf = MA.getTextFile("P1"); xf.fn.setRoot("symbolpre"); xf.setExt("csa");
    if (!xf.open(xf.INPUT)) M.error("Couldnt open "+xf.getURL());
    TextFile sf = MA.getTextFile("P1"); sf.setExt("csa");
    if (!sf.open(sf.OUTPUT)) M.error("Couldnt open "+sf.getURL());
    TextFile pf = MA.getTextFile("P1"); pf.setExt("lib");
    if (!pf.open(pf.OUTPUT)) M.error("Couldnt open "+pf.getURL());
    String name = tf.fn.getRoot().toUpperCase();
    String[] spin  = new String[npin+1];
    String[] stext = new String[npin+1];
    int ip=0;
    while ( (line=tf.read()) != null) {
      int ie = line.indexOf('=');
      int i = reorder? Convert.s2l(line.substring(0,ie))-1 : ip;
      spin[i] = line.substring(0,ie);
      stext[i] = line.substring(ie+1);
      ip++;
    }
    if (ip!=npin) {
      M.warning("Number lines="+ip+" != mnumber pins="+npin);
      npin = ip;
    }
    int grid = 254000;
    int hgrid = grid/2;
    int hpin = npin/2;
    int ix1  = 10, ix2 = ix1+40, ixs = ix1+2;
    int iy1  = 10, iy2 = iy1+hpin+4;
    pf.writeln(name);
    while ( (line=xf.read()) != null && !line.equals("XXXX")) sf.writeln(line);
    sf.writeln("  (SYMDEF L0 \""+name+"\" \"\" (PT "+(ixs*grid)+" "+((iy2-1)*grid)+")");
    sf.writeln("   (FIGURE FIG0 LC25 NO_SHEET (OPENSHAPE");
    sf.writeln("    (PT "+(ix1*grid)+" "+(iy2*grid)+")");
    sf.writeln("    (PT "+(ix2*grid)+" "+(iy2*grid)+")");
    sf.writeln("    (PT "+(ix2*grid)+" "+(iy1*grid)+")");
    sf.writeln("    (PT "+(ix1*grid)+" "+(iy1*grid)+")");
    sf.writeln("    (PT "+(ix1*grid)+" "+(iy2*grid)+")");
    sf.writeln("   ))");
    sf.writeln("   (TEXTLOC SYMBOL_NAME TC2 (PT "+(ixs*grid)+" "+((iy2-2)*grid)+"))");
    sf.writeln("   (TEXTLOC PART_NAME TC2 (PT "+(ixs*grid)+" "+((iy2-3)*grid)+"))");
    for (int i=0; i<npin; i++) {
      int j = i+1;
      int ix = (i<hpin)? ix1 : ix2;
      int ixt = (i<hpin)? ix+1 : ix-12;
      int iy = (i<hpin)? iy1+(hpin-i) : iy1+1+(i-hpin);
      sf.writeln("   (TERMINAL "+j+" TC0 (PT "+(ix*grid)+" "+(iy*grid)+"))");
      sf.writeln("   (TEXT TX"+j+" \""+stext[i]+"\" TC2 NO_SHEET (PT "+(ixt*grid)+" "+(iy*grid-hgrid)+"))");
      int pn = (ncol==1)? i : getPinNumber(spin[i],nrow,ncol);
      addLine(pf,""+pn+".0");
    }
    sf.writeln("  )");
    while ( (line=xf.read()) != null) sf.writeln(line);
    flushLine(pf); 

    tf.close();
    xf.close();
    sf.close();
    pf.close();
  }

  private void sym2part() {

    Parser p; String line;
    boolean v4 = MA.getState("/V4");
    boolean v5 = MA.getState("/V5");
    TextFile tf = MA.getTextFile("P1");
    tf.setExt( v5? "pkg" : v4? "txt" : "pins" );
    if (!tf.open(tf.INPUT)) M.error("Couldnt open "+tf.getURL());
    int nrow = MA.getL("/ROWS");
    if (nrow<=0) M.error("You must specify /ROWS=n for this part up to "+rowlabel.length);
    M.info("Part Rows="+nrow);
    int npin = nrow*nrow;
    String[] rows = new String[npin+1];
    while ( (line=tf.read()) != null) {
      p = new Parser(line);
      String text = p.get( v5? 3 : v4? 6 : 2 );
      String spin = p.get( v5? 1 : v4? 4 : 3 );
      int pin = getPinNumber(spin,nrow,nrow);
      if (rows[pin]!=null) M.error("Got Lpin="+pin+" Pin="+spin+" Label="+text);
      rows[pin] = text;
    }
    tf.close();

    TextFile sf = MA.getTextFile("P1"); sf.setExt("sym");
    if (!sf.open(sf.INPUT)) M.error("Couldnt open "+sf.getURL());
    tf = MA.getTextFile("P1"); tf.setExt("lib");
    if (!tf.open(sf.OUTPUT)) M.error("Couldnt open "+tf.getURL());

    while ( (line=sf.read()) != null) {
      p = new Parser(line);
      if (p.elements()<=0) continue;
      M.info("Process "+line);
      String text = p.get(1);
      if (text.equals("GLOBAL")) { 
        String signal = p.get(2);
        flushLine(tf); 
        addLine(tf,p.get(3));
        for (int i=0; i<=npin; i++) {
          if (signal.equals(rows[i])) addLine(tf,""+i+".0");
        }
        flushLine(tf); 
      }
      else if (text.startsWith("PINS")) {
        for (int i=2; i<=p.elements(); i++) {
          String signal = p.get(i);
          int pin = findPin(signal,rows);
	  if (pin<=0) M.error("Signal "+signal+" not found");
          addLine(tf,""+pin+".0");
        }
      }
      else if (text.startsWith("IOBANK")) {
        String bank = p.get(2);
        for (int i=3; i<=p.elements(); i++) {
          String pin = p.get(i);
	  int in = findPin("IO_L"+pin+"N_"+bank,rows);
	  int ip = findPin("IO_L"+pin+"P_"+bank,rows);
	  int is = findPin("IO_L"+pin+"_"+bank,rows);
          if (is>0) addLine(tf,""+is+".0");
          if (in>0) addLine(tf,""+in+".0");
          if (ip>0) addLine(tf,""+ip+".0");
	  if (is<0 && in<0) M.error("No pin "+pin+" in bank "+bank);
	  if (is>0 && in>0) M.error("Bad pin "+pin+" in bank "+bank);
	  if (in>0 && ip<0) M.error("Mismatch pin "+pin+" in bank "+bank);
        }
      } 
    }
    tf.close();
    sf.close();
  }

  private int getPinNumber (String spin, int nrow, int ncol) {
    int irow=nrow;
    for (irow=nrow-1; irow>=0 && !spin.startsWith(rowlabel[irow]); irow--);
    if (irow<0) return -1;
    int pin = 0;
    for (int i=rowlabel[irow].length(); i<spin.length(); i++) pin = 10*pin + (spin.charAt(i)-'0');
    pin += irow*ncol;
    return pin;
  }

  private String getSPinNumber (int pin, int nrow) {
    int irow = (pin-1)/nrow;
    int icol = pin - (irow*nrow);
    return rowlabel[irow]+icol;
  }

  String outline = "";

  private void addLine (TextFile tf, String text) {
    if (outline.length()>73) {
      tf.writeln(outline+"&");
      outline = "";
    }
    outline += text+" ";
  }
  private void flushLine (TextFile tf) {
    if (outline.length()>0) tf.writeln(outline);
    outline = "";
  }

  private void net2ucf() {
    Parser p; String line;
    TextFile tf = MA.getTextFile("P1"); tf.setExt("frs");
    if (!tf.open(tf.INPUT)) M.error("Couldnt open "+tf.getURL());
    TextFile sf = MA.getTextFile("P1"); sf.setExt("ucfx");
    if (!sf.open(sf.OUTPUT)) M.error("Couldnt open "+sf.getURL());
    String part = MA.getS("/PART");
    if (part.length()<=0) M.error("You must specify /PART=name for this design");
    int nrow = MA.getL("/ROWS");
    if (nrow<=0) M.error("You must specify /ROWS=n for this part up to "+rowlabel.length);
    M.info("Part="+part+"  Rows="+nrow);
    int npin = nrow*nrow;
    String label = null;
    while ( (line=tf.read()) != null) {
      if (line.length()==0) continue;
      if (line.charAt(0) != '.') line = ".TER "+line;
      p = new Parser(line); p.clean();
      String cmd = p.get(1);
      if (cmd.equals(".ADD_TER")) { 
        label = p.get(4);
	label = label.substring(1,label.length()-1);
        int bus = 0;
	int ll = label.length();
        char c = label.charAt(label.length()-2);
        if (ll>1 && label.charAt(ll-1)>='0' && label.charAt(ll-1)<='9') bus=1;
        if (bus>0 && ll>2 && label.charAt(ll-2)>='0' && label.charAt(ll-2)<='9') bus=2;
        if (bus!=0) label = label.substring(0,ll-bus)+"<"+label.substring(ll-bus,ll)+">";
	int i = label.indexOf("<0");
	if (bus==2 && i>0) label = label.substring(0,i+1)+label.substring(i+2);
        cmd = ".TER";
       }
      if (!cmd.equals(".TER")) continue;
      String cpart = p.get(2);
      if (!cpart.equals(part)) continue;
      String cpin = p.get(3);
      String spin = getSPinNumber(Convert.s2l(cpin),nrow);
      M.info("Terminal "+cpart+" "+cpin+" "+label);
      sf.writeln("NET \""+label.toLowerCase()+"\" LOC = \""+spin.toLowerCase()+"\" ;");
    }
    tf.close();
    sf.close();
  }

  private int findPin (String text, String[] rows) {
    for (int i=1; i<rows.length; i++) {
      if (matchPin(text,rows[i])) return i;
    }
    return -1;
  }

  private boolean matchPin (String text, String label) {
    if (label==null) return false;
    return label.indexOf(text)>=0;
  }

  private void qnet2ucf() {
    Parser p; String line;
    TextFile tf = MA.getTextFile("P1"); tf.setExt("qnet");
    if (!tf.open(tf.INPUT)) M.error("Couldnt open "+tf.getURL());
    TextFile sf = MA.getTextFile("P1"); sf.setExt("ucfx");
    if (!sf.open(sf.OUTPUT)) M.error("Couldnt open "+sf.getURL());
    String part = MA.getS("/PART");
    if (part.length()<=0) M.error("You must specify /PART=name for this design");
    M.info("Part="+part);
    String label = null;
    while ( (line=tf.read()) != null) {
      if (line.length()==0) continue;
      p = new Parser(line); p.clean();
      String cmd = p.get(1);
      if (cmd.equals("NET_NAME")) { 
        label = null;
        String net = line=tf.read();
        net = net.substring(1,net.length()-1);
        if (net.indexOf(".")>0) continue;
        if (net.startsWith("VCC")) continue;
        if (net.startsWith("GND")) continue;
        if (net.equals("NC")) continue;
	label = convertNet(net);
      }
      if (label==null) continue;
      if (!cmd.equals("NODE_NAME")) continue;
      String cpart = p.get(2);
      if (!cpart.equals(part)) continue;
      String cpin = p.get(3);
      M.info("Terminal "+cpart+" "+cpin+" "+label);
      sf.writeln("NET \""+label.toLowerCase()+"\" LOC = \""+cpin.toLowerCase()+"\" ;");
    }
    tf.close();
    sf.close();
  }

  private void enet2ucf() {
    Parser p; String line;
    TextFile tf = MA.getTextFile("P1"); tf.setExt("enet");
    if (!tf.open(tf.INPUT)) M.error("Couldnt open "+tf.getURL());
    TextFile sf = MA.getTextFile("P1"); sf.setExt("ucfx");
    if (!sf.open(sf.OUTPUT)) M.error("Couldnt open "+sf.getURL());
    String part = MA.getS("/PART");
    if (part.length()<=0) M.error("You must specify /PART=name for this design");
    M.info("Part="+part);
    String lnet = null;
    for (int i=0; i<8; i++) tf.read();
    while ( (line=tf.read()) != null) {
      if (line.length()<43) { lnet = null; continue; }
      String net = line.substring(0,16).trim();
      String prt = line.substring(16,25).trim();
      String pad = line.substring(25,34).trim();
      String pin = line.substring(34,43).trim();
      if (net.length()==0) net = lnet;
      lnet = net;
      if (prt.equals(part)) {
        if (net.indexOf(".")>=0) continue;
        if (pin.indexOf("VCC")>=0) continue;
        if (pin.indexOf("VOUT")>=0) continue;
        if (pin.indexOf("GND")>=0) continue;
        net = net.toLowerCase();
        boolean hasPol = net.endsWith("_n") || net.endsWith("_p");
	int ln = hasPol? net.length()-2 : net.length();
        boolean hasBus = net.charAt(ln-1)>='0' && net.charAt(ln-1)<='9';
        boolean hasBut = net.charAt(ln-2)>='0' && net.charAt(ln-2)<='9';
        int li = hasBus? ln-1 : ln; if (hasBut) li--;
        if (hasPol) net = net.substring(0,li)+net.charAt(ln+1)+net.substring(li,ln);
        if (hasPol) { li++; ln++; }
        if (hasBus) net = net.substring(0,li)+'<'+net.substring(li,ln)+'>';
        M.info("Terminal "+part+" "+pad+" "+net);
        sf.writeln("NET \""+net.toLowerCase()+"\" LOC = \""+pad.toLowerCase()+"\" ;");
      }
    }
    tf.close();
    sf.close();
  }

  private String[] getSA (Parser p) {
    int size = p.elements();
    String[] sa = new String[size];
    for (int i=0; i<size; i++) {
      String s = p.get(i+1);
      if (s.length()>0 && s.charAt(0)=='"') s = s.substring(1,s.length()-1);
      sa[i] = s;
    }
    return sa;
  }
  private String subS (String line, String sub) {
    return StringUtil.replaceFirst(line,"^",sub);
  }
  private String subS (String line, String sub1, String sub2) {
    return subS(subS(line,sub1),sub2);
  }
  private String subS (String line, String sub1, String sub2, String sub3) {
    return subS(subS(subS(line,sub1),sub2),sub3);
  }

  private void ucf2xdc() {
    Parser p; String line; String[] san = null;
    TextFile tf = MA.getTextFile("P1"); tf.setExt("ucf");
    if (!tf.open(tf.INPUT)) M.error("Couldnt open "+tf.getURL());
    TextFile sf = MA.getTextFile("P1"); sf.setExt("xdc");
    if (!sf.open(sf.OUTPUT)) M.error("Couldnt open "+sf.getURL());
    String label = null;
    boolean xdcmode = false;
    Table nets = new Table();
    while ( (line=tf.read()) != null) {
      line = line.trim();
      if (line.length()==0) continue;
      int ic = line.indexOf(';');
      if (ic>0) line = line.substring(0,ic);
      while (line.indexOf("  ")>=0) line = StringUtil.replaceAll(line,"  "," ");
      line = StringUtil.replaceAll(line,"<","[");
      line = StringUtil.replaceAll(line,">","]");
      p = new Parser(line); 
      String[] sa = getSA(p);
      int na = sa.length;
      if (san!=null && !(sa[0].equals("TIMESPEC") && sa[3].equals("PERIOD"))) {	
        sf.writeln(subS("create_generated_clock -name ^ [get_nets ^]",san[4],san[1]));
      }
      san = null;
      if (sa[0].equals("##XDC##")) {
        sf.writeln(line.substring(8));
      }
      else if (sa[0].equals("##XDCON##")) {
        xdcmode = true;
      }
      else if (sa[0].equals("##XDCOFF##")) {
        xdcmode = false;
      }
      else if (xdcmode) {
        sf.writeln(line);
      }
      else if (sa[0].startsWith("##")) {
        sf.writeln(line.substring(1));
      }
      else if (sa[1].equals("DCI_CASCADE")) {
        String s=sa[3]; int i=s.indexOf(" ");
        sf.writeln(subS("set_property DCI_CASCADE {^} [get_iobanks ^]",s.substring(0,i),s.substring(i+1)));
      }
      else if (sa[2].equals("TNM_NET")) {
       nets.put(sa[4],sa[1]);
       san = sa;
      }
      else if (sa[0].equals("TIMESPEC") && sa[3].equals("PERIOD")) {
        double freq = Convert.s2d(sa[5],M);
        String nsec = Convert.d2s(1000.0/freq,0,"##.###");
        String net  = nets.getS(sa[4]);
        sf.writeln(subS("create_clock -name ^ -period ^ [get_nets ^]",sa[4],nsec,net));
        MR.put(sa[1],freq);
      }
      else if (sa[0].equals("TIMESPEC") && sa[3].equals("FROM") && sa[5].equals("TO") && sa[7].equals("TIG")) {
        sf.writeln(subS("set_false_path -from [get_clocks ^] -to [get_clocks ^]",sa[4],sa[6]));
      }
      else if (sa[0].equals("TIMESPECS") && sa[3].equals("FROM") && sa[5].startsWith("TO")) {
        boolean tof = sa[5].equals("TOFR");
        boolean max = sa[7].equals("MAX");
        boolean tig = sa[7].equals("TIG");
        Parser cq = new Parser(sa[4]);
        Parser cp = new Parser(sa[6]);
	for (int j=1; j<=cq.elements(); j++) { String from = cq.get(j);
	for (int i=1; i<=cp.elements(); i++) { String to = cp.get(i);
	 if (to.equals(from)) continue;
         if (tig) sf.writeln(subS("set_false_path -from [get_clocks ^] -to [get_clocks ^]",from,to));
         if (max) sf.writeln(subS("set_max_delay -from [get_clocks ^] -to [get_clocks ^] ^",from,to,sa[8]));
         if (!tof) continue;
         if (tig) sf.writeln(subS("set_false_path -from [get_clocks ^] -to [get_clocks ^]",to,from));
         if (max) sf.writeln(subS("set_max_delay -from [get_clocks ^] -to [get_clocks ^] ^",to,from,sa[8]));
        }
        }
      }
      else if (sa[0].equals("NET") && sa[2].equals("LOC")) {
        sf.writeln(subS("set_property PACKAGE_PIN ^ [get_ports ^]",sa[4].toUpperCase(),sa[1]));
        for (int i=5; i<na; ) {
          if (!sa[i].equals("|")) { System.out.println("Bad NET format: "+line+"  //  "+i+" "+sa[i]); break; }
          int ie = sa[i+1].indexOf('=');
          if (ie>0) {
            sf.writeln(subS("set_property ^ ^ [get_ports ^]",sa[i+1].substring(0,ie),sa[i+1].substring(ie+1),sa[1])); i+=2;
          }
          else if (i+2<na && sa[i+2].equals("=")) {
            sf.writeln(subS("set_property ^ ^ [get_ports ^]",sa[i+1],sa[i+3],sa[1])); i+=4;
          }
          else {
            sf.writeln(subS("set_property ^ TRUE [get_ports ^]",sa[i+1],sa[1])); i+=2;
          }
        }
      }
      else if (sa[0].equals("NET") && sa[2].equals("TIG")) {
        if (sa[1].indexOf('*')<0) sf.writeln(subS("set_false_path -through [get_nets -match_style ucf ^]",sa[1]));
        else sf.writeln(subS("set_false_path -through [get_nets -match_style ucf ^]",sa[1]));
      }
      else if (sa[0].equals("NET") && sa[2].equals("USELOWSKEWLINES")) {
        sf.writeln("##UCF? "+line);
      }
      else if (sa[0].equals("NET")) {
        int ie = sa[2].indexOf('=');
        if (ie>0) sf.writeln(subS("set_property ^ ^ [get_nets ^]",sa[2].substring(0,ie),sa[2].substring(ie+1),sa[1]));
        else sf.writeln(subS("set_property ^ ^ [get_nets ^]",sa[2],sa[3],sa[1]));
      }
      else if (sa[0].equals("INST") && sa[2].equals("LOC")) {
        sf.writeln(subS("set_property LOC ^ [get_cells ^]",sa[4],sa[1]));
      }
      else if (sa[0].equals("INST") && sa[2].equals("AREA_GROUP")) {
        sf.writeln(subS("create_pblock ^",sa[4]));
        sf.writeln(subS("add_cells_to_pblock [get_pblocks ^] [get_cells -quiet [list ^]]",sa[4],sa[1]));
      }
      else if (sa[0].equals("AREA_GROUP") && sa[2].equals("RANGE")) {
        sf.writeln(subS("resize_pblock [get_pblocks ^] -add {^}",sa[1],sa[4]));
      }
      else if (sa[0].equals("INST") && sa[2].equals("RANGE")) {
        sf.writeln(subS("set_property RANGE ^ [get_cells ^]",sa[4],sa[1]));
      }
      else {
        System.out.print("SA [ ");
        for (int i=0; i<na; i++) System.out.print(sa[i]+" ");
        System.out.println("]");
        sf.writeln("##UCF? "+line);
      }
    }
    tf.close();
    sf.close();
  }

  private String convertNet (String net) {
    int i,len = net.length();
    boolean tnum = net.charAt(len-2)>='0' && net.charAt(len-2)<='9';
    if (net.endsWith("+")||(net.endsWith("P")&&tnum)) 
	net = net.substring(0,len-2)+"P"+net.charAt(len-2);
    if (net.endsWith("-")||(net.endsWith("N")&&tnum)) 
	net = net.substring(0,len-2)+"N"+net.charAt(len-2);
    for (i=len; i>0; i--) {
      char c = net.charAt(i-1);
      if (c<'0' || c>'9') break;
    }
    if (i!=len && len-i<4) net = net.substring(0,i)+"<"+net.substring(i)+">";
    if (net.startsWith("DDR2_")) net = "M"+net.substring(5);
    return net;
  }

  boolean isAlpha (char c) {
    return (c>='a' && c<='z') || (c>='A' && c<='Z') || c=='_';
  }
  boolean isAlphaP (char c) {
    return (c>='a' && c<='z') || (c>='A' && c<='Z') || c=='_' || c=='!';
  }
  boolean isAlphaN (char c) {
    return (c>='a' && c<='z') || (c>='A' && c<='Z') || c=='_' || c=='!' || (c>='0' && c<='9');
  }
  boolean compare (KeyVector kv, String key1, String test, String key2) {
    Object key1o = kv.get(key1); if (key1o!=null) key1=key1o.toString();
    Object key2o = kv.get(key2); if (key2o!=null) key2=key2o.toString();
    //M.info("Test in VPP define: "+key1+" "+test+" "+key2);
    if (test.equals("==")) return key1.equals(key2);
    if (test.equals("!=")) return !key1.equals(key2);
    if (test.equals(">=")) return Convert.s2d(key1) >= Convert.s2d(key2);
    if (test.equals("<=")) return Convert.s2d(key1) <= Convert.s2d(key2);
    if (test.equals(">")) return Convert.s2d(key1) > Convert.s2d(key2);
    if (test.equals("<")) return Convert.s2d(key1) < Convert.s2d(key2);
    M.warning("Illegal test in VPP define: "+key1+" "+test+" "+key2);
    return false;
  }
  boolean isDefine (KeyVector kv, String key, String rest) {
    Parser p=null;
    int elem=1;
    if (rest!=null) {
      p = new Parser(key+" "+rest,true,false);
      elem = p.elements();
    }
    boolean state = false;
    for (int i=1; i<=elem; i++) {
      String keyn = (p==null)? key : p.get(i);
      boolean togglen = keyn.startsWith("!") && !keyn.equals("!=");
      if (togglen) keyn = keyn.substring(1);
      boolean isA = isAlpha(keyn.charAt(0));
      if (isA) state = (kv.findKey(keyn)>=0) ^ togglen;
      else if (keyn.equals("||")||keyn.equals("|")) { if (state) break; }
      else if (keyn.equals("&&")||keyn.equals("&")) { if (!state) break; }
      else { state = compare(kv,p.get(i-1),keyn,p.get(i+1)); i++; }
    }
//System.out.println("Defined: "+key+" "+rest+" = "+state);
    return state;
  }

  private void vppfile (KeyVector kv, TextFile tf, TextFile tfo) {
    boolean isOut = tfo!=null;
    FileName fn = tf.getFileName();
    boolean isV = fn.getExt().equals("v");
    if (!tf.open(tf.INPUT)) M.error("Couldnt open "+tf.getURL());
    if (isOut && !tfo.open(tfo.OUTPUT)) M.error("Couldnt open "+tfo.getURL());
    String line = null;
    int iflev=0;
    int[] ifdef = new int[16];	// 0=false 1=true -1=alldone
    ifdef[0]=1;
    while ( (line=tf.read()) != null) {
      char cl=' ';
      int il = line.length();
      int ic = line.indexOf("//");
      boolean isPP = false;
      for (int i=0; i<il; i++) {
        char c = line.charAt(i);
	if (ic>=0 && i>=ic) break;
	if (isAlpha(c)) {
	  int ib=i;
	  for (i++; i<il && isAlphaN(line.charAt(i)); i++);
	  String token = line.substring(ib,i);
	  if (cl=='`' || cl=='#') {
            isPP = true;
	    int i1; for (i1=i; i1<il && !isAlphaP(line.charAt(i1)); i1++);
	    int i2; for (i2=i1; i2<il && isAlphaN(line.charAt(i2)); i2++);
	    String key = (i2>i1)? line.substring(i1,i2) : "";
	    String rest = (i2<il)? line.substring(i2).trim() : null;
// System.out.println("Found token="+token+" key="+key+" rest="+rest);
            if (token.equals("define") && rest==null && cl=='#') rest = "NULL";
	    int cid = ifdef[iflev];
                 if (token.equals("define")){ if (ifdef[iflev]==1 && rest!=null) kv.put(key,rest.equals("NULL")?"":rest); }
            else if (token.equals("undef")) { if (ifdef[iflev]==1) kv.remove(key); }
            else if (token.equals("if"))    { ifdef[++iflev] = (cid<=0)? -1 : isDefine(kv,key,rest)?1:0; }
            else if (token.equals("ifdef")) { ifdef[++iflev] = (cid<=0)? -1 : isDefine(kv,key,rest)?1:0; }
            else if (token.equals("ifndef")){ ifdef[++iflev] = (cid<=0)? -1 : !isDefine(kv,key,rest)?1:0; }
            else if (token.equals("else"))  { ifdef[iflev] = (ifdef[iflev]==0)?1:0; }
            else if (token.equals("elif"))  { ifdef[iflev] = (ifdef[iflev]!=0)?-1:isDefine(kv,key,rest)?1:0; }
            else if (token.equals("elsif")) { ifdef[iflev] = (ifdef[iflev]!=0)?-1:isDefine(kv,key,rest)?1:0; }
            else if (token.equals("endif")) { iflev--; }
	    else isPP = false;
            if (token.equals("include") && ifdef[iflev]>0 && line.indexOf("mdefs_user")>0) { 
	      String ufn = fn.getFullName().replace("mdefs","mdefs_user");
	      //System.out.println("Including file: "+ufn);
	      vppfile(kv,new TextFile(ufn),null);
	    }
          } else {
	    Object val = kv.get(token);
	    Object val2 = (val==null)? null : kv.get(val.toString());
	    if (val!=null && val2==null && (!isPP || token.endsWith("_CPP"))) 
	      { line=line.substring(0,ib)+val+line.substring(i); il=line.length(); i=ib+val.toString().length(); }
	    if (token.equals("CARD_CPP") && val2==null) line += "\n`define "+token.replace("CARD_CPP","ICE_CARD")+" \""+val+"\"";
//	    if (token.endsWith("_CPP") && val2==null) line += "\n`define "+token.replace("_CPP","")+" \""+val+"\"";
	  }
	  cl = ' ';
	}
	else if (c!=' ') cl = c;
      }
      if (isOut && (isV || (!isPP && ifdef[iflev]==1))) tfo.writeln(line);
    }
    if (tf!=null) tf.close();
    if (tfo!=null) tfo.close();
  }

  private void vppflow() {
    KeyVector kv = KeyVector.fromString( MA.getCS("/PREDEFS"));
    if (MA.isPresent("/MDEFS")) vppfile (kv,MA.getTextFile("/MDEFS"),null);
    //kv.dump();
    vppfile (kv,MA.getTextFile("P1"),MA.getTextFile("P2"));
  }

  private void tppflow() {
    KeyVector kv = KeyVector.fromString( MA.getCS("/DEFINES"));
    kv.add("NMVERSION",MR.getS("ENV.NMVERSION"));
    kv.add("HOSTNAME",MR.getS("ENV.HOSTNAME"));
    kv.add("USER",MR.getS("ENV.USER"));
    boolean dbg = MA.getState("/DEBUG");
    TextFile tfx = new TextFile( dbg? "/tmp/snapapp_pp":"ram:snapapp_pp");
    vppfile (kv,MA.getTextFile("P1"),tfx);
    MR.put(MA.getU("P2"),new nxm.sys.lib.Table(tfx));
    if (!dbg) M.pipes.remove("snapapp_pp");
  }

  private void jtag2m() {
    int lbuf = 32768, offset=0;
    int ibits=0,obits=0;
    int iwords=0,owords=0;
    int tdiw=0,tdow=0;
    int state = 21;
    boolean oot,tot;

    oot = MA.getState("/OOT");
    tot = MA.getState("/TOT");

    BaseFile bf = MA.getBaseFile("P1");
    bf.open(bf.INPUT);
    byte[] buf = new byte[lbuf];

    DataFile dfc = MA.getDataFile("P2","1000","5L",0);
    dfc.open(dfc.OUTPUT);
    Data dfcb = dfc.getDataBuffer(1);

    DataFile dfi = MA.getDataFile("P3","1000","SL",0);
    dfi.open(dfi.OUTPUT);
    Data dfib = dfi.getDataBuffer(lbuf);

    DataFile dfo = MA.getDataFile("P4","1000","SL",0);
    dfo.open(dfo.OUTPUT);
    Data dfob = dfo.getDataBuffer(lbuf);

    int size=(int)bf.getSize();
    for (;size>0;) {
      int nbuf = Math.min(lbuf,size);
      bf.read (buf,0,nbuf);
      for (int i=0; i<nbuf; i++) {
        int data = buf[i];
        boolean tdi = (data&0x1)!=0;
        boolean tdo = (data&0x2)!=0;
        boolean tms = (data&0x4)!=0;
        tdiw = tdiw>>>1; if (tdi) tdiw |= 0x80000000; 
        tdow = tdow>>>1; if (tdo) tdow |= 0x80000000; 
        if (++ibits==32) { dfib.setL(iwords,tdiw); iwords++; ibits=0; }
        if (++obits==32) { dfob.setL(owords,tdow); owords++; obits=0; }
	switch (state) {
	  case 0: 
	    if (tms) state++;
	    break;
	  case 1: case 2:
	    if (tms) state++;
	    else state+=10;
	    break;
          case 3: 
	    if (!tms) state = 0;
	    break;
	  case 11: case 12:
            tdiw=tdow=ibits=obits=iwords=owords=0; 
	    if (oot) obits = -1;
	    if (tot) ibits = -1;
	    state+=10; 
	    break;
          case 21: case 22:
            if (!tms) break;
	    if (ibits!=0) { int shift = 32-ibits; tdiw >>>= shift; }
	    if (obits!=0) { int shift = 32-obits; tdow >>>= shift; }
	    dfcb.setL(0,state-20);
	    dfcb.setL(1,ibits);
	    dfcb.setL(2,iwords);
	    dfcb.setL(3,tdiw);
	    dfcb.setL(4,tdow);
	    dfc.write(dfcb);
	    if (iwords>0) dfi.write(dfib,iwords);
	    if (owords>0) dfo.write(dfob,owords);
	    state = -3;
	    break;
          case -3: 
	    state = 0;
	    break;
          default:
	    M.error("Illegal State "+state+" at "+offset);
        }
	offset++;
      }
      size -= nbuf;
    }
    bf.close();
    dfc.close();
    dfi.close();
    dfo.close();
  }

  static double e2m = 0.0254;
  static String form = "%-6s%-10s%8.2f%8.3f%8.2f%8.3f%4d%8.2f%8.3f  %-6s%2d%8.2f%8.3f  D%1d";

  public void gerber() {
    TextFile tfi = MA.getTextFile("P1");
    if (!tfi.open(tfi.INPUT)) M.error("Opening input URL="+tfi.getURL());
    M.info("Opened gerber input URL="+tfi.getURL());
    TextFile tfo = MA.getTextFile("P2");
    if (!tfo.open(tfo.OUTPUT)) M.error("Opening output URL="+tfo.getURL());
    tfo.writeln("! GAPFile Version 1.0");
    tfo.writeln("! Created by: NeXtMidas "+Installation.NM_VERSION);
    tfo.writeln("GAPFile    Version 1.0    (c) 1991 CAD Solutions, Inc.");
    tfo.writeln("!     Type            Width          Length     Rotate   Offset      Swap  Tool      Size      Legend");
    tfo.writeln("!                English  Metric English  Metric     English  Metric Code  No. English  Metric");
    tfo.writeln("|-----------------------+---------------+-------------------+-------------------------+--------------");
    while (true) {
      String line = tfi.read();
      if (line==null) break;
      Parser p = new Parser (line,true);
      String pos = p.get(1);
      if (pos==null || pos.length()==0) continue;
      char c = pos.charAt(0);
      if (c<'0' || c>'9') continue;
      String shape  = p.get(2);
      String usage  = p.get(3);
      double size   = Convert.s2d(p.get(4));
      int    orient = Convert.s2l(p.get(5));
      double length = Convert.s2d(p.get(6));
      if (shape.equals("CIRCLE")) shape = "ROUND";
      if (shape.equals("ANNULUS")) { shape = "DONUT"; length=orient; orient=0; }
      if (shape.equals("FINGER")) shape = "OBLONG";
      if (shape.equals("BULLET")) shape = "OBLONG";
      if (shape.equals("DIAMOND")) shape = "SQUARE";
      if (shape.equals("RECTANGLE") || shape.equals("OBLONG")) {
        length += size;
        if (orient==0 || orient==180) { double t=size; size=length; length=t; }
      }
      String dcode = "D"+pos;
      Double dsize = Double.valueOf(size);
      Double esize = Double.valueOf(size*e2m);
      Double dlength = Double.valueOf(length);
      Double elength = Double.valueOf(length*e2m);
      Integer iorient = Integer.valueOf(0);
      Integer izero = Integer.valueOf(0);
      Double dzero = Double.valueOf(0.0);
      Object[] args = { dcode,shape,dsize,esize,dlength,elength,
                        iorient,dzero,dzero,dcode,izero,dzero,dzero,izero };
//      M.info("String.format() only supported on 1.5 JDK");
      String s = String.format(form,args);
      System.out.println(s);
      tfo.writeln(s);
    }
    tfi.close();
    tfo.close();
    M.info("Closed GAP output URL="+tfo.getURL());
  }

  int evalcrc;

  private void crcgen() {

    int size = MA.getL("P1",32);
    int poly = MA.getL("P2",0x04C11DB7);
    int seed = MA.getL("P3",0xFFFFFFFF);
    int bits = MA.getL("P4",32);
    int data = MA.getL("P5",0x00000000);
    String label = MA.getU("P6");
    boolean flip = MA.getState("/FLIP");
    boolean comp = MA.getState("/COMP");
    boolean eqn  = MA.getState("/EQN");
    boolean red  = MA.getState("/RED");

    String[] eqs = new String[size];

    int crc = seed;
    int sizem = size-1;
    int mask = (size==32)? -1 : (0x1<<size)-1;
    int maskm = mask & 0xFFFFFFFE;
    int polym = poly & 0xFFFFFFFE;

    if (eqn) for (int b=0; b<size; b++) eqs[b] = "C"+b;

    for (int i=0; i<bits; i++) {
      int val = ((data>>i) ^ (crc>>sizem)) & 0x1;
      crc = ((crc<<1)&maskm) | val;
      if (val!=0) crc ^= polym;
      if (eqn) {
        String a = eqs[sizem]+"^D"+i;
        for (int b=sizem; b>0; b--) {
          eqs[b] = eqs[b-1];
	  if ( ((poly>>b)&0x1) != 0) eqs[b] += "^"+a;
        }
        eqs[0] = a;
      }
    }
    if (flip) {
      int crcf = 0;
      for (int i=0; i<size; i++) if (((crc>>i)&0x1)!=0) crcf |= ( 0x1 << (i^0x7) );
      crc = crcf;
    }
    if (comp) crc ^= mask;

    MT.writeln("CRC size="+size+" mask="+Convert.l2x(maskm)+" polym="+Convert.l2x(polym));
    MT.writeln("CRC in="+Convert.l2x(seed)+" data="+Convert.l2x(data)+" out="+Convert.l2x(crc));

    MR.put(label,crc);

    evalcrc = 0;
    if (red) for (int b=0; b<size; b++) eqs[b] = reduceEqs(bits,data,eqs[b],b);
    if (eqn) for (int b=0; b<size; b++) MT.writeln("crc["+b+"]<="+eqs[b]);
    if (red) MT.writeln("CRC eval="+Convert.l2x(evalcrc));

  }

  private String reduceEqs (int bits, int data, String eqs, int b) {
    int n,c,ones;
    eqs = eqs+"^"; // trailer for pattern matching
    for (c=n=0; c<eqs.length(); c++) if (eqs.charAt(c)=='C') n++;
    String out = "C"+(n%2);
    ones = n;
    for (int i=0; i<bits; i++) {
      String patt = "D"+i+"^"; int plen = patt.length();
      for (c=n=0; c<eqs.length(); c++) if (eqs.regionMatches(c,patt,0,plen)) n++;
      if (n%2==1) out += "^d["+i+"]";
      if (n%2==1 && ((data>>i)&0x1)==1) ones++;
    }
    if (ones%2==1) evalcrc |= (1<<b);
    return out+";";
  }

  private void modargs () {
    Args args1 = Args.parseCommand(M,MA.getS(2));
    Args args2 = (Args) MA.getO(3);
    args2.getKV().clear();
    args2.getKV().merge(args1.getKV());
    args2.nargs = args1.nargs;
    args2.nargsx = args1.nargsx;
  }

  private void tput () {
    Table tbl = MA.getTable("P1");
    DataFile hcb = MA.getDataFile("P2");
    hcb.open(DataFile.INOUT|DataFile.HEADERONLY);
    for (Table.Iterator ti=tbl.iterator(); ti.getNext(); ) {
      hcb.keywords.put(ti.key,ti.value);
    }
    hcb.close();
  }

  private void scanup () {
    DataFile hi = MA.getDataFile("P1");
    DataFile ho = MA.getDataFile("P2");
    hi.open(DataFile.INPUT);
    ho.open(DataFile.INOUT);
    int nhi = (int)hi.getSize(); int bpei=(int)hi.getBPE();
    int nho = (int)ho.getSize(); int bpeo=(int)ho.getBPE();
    double scf = hi.keywords.getD("RFCF",-1.0);
    double sbw = hi.keywords.getD("RFBW",-1.0), sbwh=sbw*0.4;
    Data dbi = hi.getDataBuffer(nhi);
    Data dbo = ho.getDataBuffer(nho);
    hi.read(dbi);
    ho.read(dbo);
    for (int o=0; o<nho; o++) {
      int offo = o*bpeo;
      float cfo = Convert.unpackF(dbo.buf,offo+4);
      float bwo = Convert.unpackF(dbo.buf,offo+8), bwoh=bwo*0.4F;
      boolean inband = (cfo>scf-sbwh) && (cfo<scf+sbwh);
      if (!inband) continue;
      int lhit = Convert.unpackL(dbo.buf,offo+0);
      int hit=0;
      float snr=0,cfi=0,bwi=0;
      for (int i=0; i<nhi && hit==0; i++) {
	int offi = i*bpei;
	cfi = Convert.unpackF(dbi.buf,offi+4);
	bwi = Convert.unpackF(dbi.buf,offi+8);
	snr = Convert.unpackF(dbi.buf,offi+12);
	if ((cfi > cfo-bwoh) && (cfi < cfo+bwoh)) hit=1;
      }
      Convert.packL(dbo.buf,offo+0,hit);
      if (hit>0) Convert.packF(dbo.buf,offo+12,snr);
      if (hit!=lhit) M.info("Track INDX="+o+" FREQ="+cfo+" BW="+bwo+" SNR="+snr+" UP="+hit);
    }
    ho.seek(0);
    ho.write(dbo);
    ho.close();
    hi.close();
  }

  private void greplog () {
    DataFile hi = MA.getDataFile("P1");
    hi.open();
    DataFile ho = MA.getDataFile("P2",hi,0);
    ho.open();
    String ptype = MA.getS("P3");
    Data db = hi.getDataBuffer(1);
    int size = (int)hi.getSize();
    for (int i=0; i<size; i++) {
      hi.read(db);
      String stype = Convert.unpackS(db.buf,24,8);
      if (ptype.indexOf(stype)>=0) ho.write(db);
    }
    ho.close();
    hi.close();
  }

  private void kvi() {
    KeyVector kv = (KeyVector)MA.getTable("P1").getCore();
    int index = MA.getL("P2");
    String resname = MA.getU("P3");
    MR.put(resname,kv.getIndex(index));
  }

}
