package nxm.ice.prim;

import java.security.MessageDigest;
import java.util.ArrayList;

import nxm.ice.lib.MDevIce;
import nxm.sys.lib.Data;
import nxm.sys.lib.DataFile;
import nxm.sys.lib.Primitive;
import nxm.sys.lib.Table;

public class icemeasure extends Primitive {
  /** function list */
  private static final String funcList = "Bandwidth,Characterize,Flash,Harmonics,Histogram,Mse,NoiseFloor,Notch,Npr,Sonet,Udp";
  private static final int BAND=1, CHAR=2, FLASH=3, HARM=4, HIST=5, MSE=6, NOISE=7, NOTCH=8, NPR=9, SONET=10, UDP=11;
  /** Size of ICE Diode Flash Memory */
  private static final int FLASHSIZE = 0x10000;

  /** Sonet mode, looking for a pattern */
  private static final int MATCH = 24;
  /** Sonet mode, calculating Bit Error Rate */
  private static final int BER = 25;
  /** function */
  private int func;  
  /** first input data file */
  private DataFile hi1;
  /** second input data file (if necessary) */
  private DataFile hi2;
  /** output data file */
  private DataFile ho;
  /** first input data buffer */
  private Data dbi1;
  /** second input data buffer (if necessary) */
  private Data dbi2;
  /** frequency the first signal */
  private double freq1;
  /** frequency the second signal */
  private double freq2;
  /** fractional bandwidth of signal (relative to sampling rate) */
  private double fbw;
  /** number of bins that DC occupies */
  private int dcend;
  /** first data array */
  private double[] data1;
  /** indices for 1st data array */
  private int[] inds1;
  /** length of data array 1 */
  private int len1 = 0;
  /** second data array */
  private double[] data2;
  /** indices for 2nd data array */
  private int[] inds2;
  /** length of data array 2 */
  private int len2;  
  /** additional data array */
  private double[] datan;
  /** additional index array */
  private int[] indsn;
  /** length of additional data array */
  private int lenn;
  /** adjustment to be made */
  private double adj_val;
  /** frequency/bin */
  private double fpb;  
  /** sample rate */
  private double fs;
  /** noise floor threshold */
  private double thresh;
  /** quick sort variable */
  private QuickSort qsort;
  /** spurs array */
  private ArrayList<IceSpur> spurs;
  /** complex/scalar data */
  private boolean complex;
  /** median index */
  private int medind;
  /** mode */
  private int mode;
  /** seek index for sonet test */ 
  private int seek_ind;
  /** number of frames to synchronize over */
  private int frames;
  /** number of matched frames */
  private int matched;
  /** number of bytes to skip on input */
  private long skip;
  /** general purpose string */
  private String str;
  /** array of real bits counters */ 
  private long[] bit_accumr;
  /** array of imaginary bit counters */
  private long[] bit_accumi;

  public int open () {
    hi1 = MA.getDataFile("IN1");
    hi1.open();
    hi1.setDFS(0);
    xfer = hi1.getFrameSize();
    complex = hi1.getXStart() < 0;

    hi2 = null;

    ho = MA.getDataFile("OUT", "3000", "NH", 0);

    // get function
    func = MA.getSelectionIndex("FUNC",funcList,0);

    int nargs = MA.getNArgs();
    str = null;

    double THRESH1, mfreq1, mfreq2, npoints;
    int tmpbeg, tmpend, m, n;

    switch (func) {
      case BAND:
        if ( nargs != 6 )
          M.error("Error! Incorrect number of arguments for function: Bandwidth! i.e. icemeasure band <out> <in1> <fs> <fbw> <thresh>");
        //M.error("Error! Bandwidth test requires more arguments! i.e. icemeasure band <out> <in1> <fs> <sbw> <thresh>");
        ho.setSubRecords("MEDIAN|SD,IN|SD,OUT|SD");

        // sampling rate
        fs = (MA.get("FS")==null) ? MA.getD(4) : MA.getD("FS");
        if ( fs <= 0 ) 
          M.error("You must supply a valid sampling rate!");
        // fractional bandwidth
        fbw = (MA.get("FBW")==null) ? MA.getD(5) : MA.getD("FBW");
        if ( fbw < 0 || fbw > 1)
          M.error("Fractional bandwidth (FBW) must be between 0.0 and 1.0!");
        // threshold
        THRESH1 = (MA.get("THRESH")==null) ? MA.getL(6) : MA.getL("THRESH");
        if ( THRESH1 < 0 )
          M.error("Threshold must be a logarithmic value!");
        thresh = Math.pow(10, THRESH1/10);

        dcend = detDcInds(xfer);
        // calculate the # of points
        npoints = fbw*xfer - 2*dcend;
        tmpbeg = (int) Math.floor(0.5*(xfer-npoints));
        tmpend = tmpbeg + (int) Math.ceil(npoints);

        len1 = tmpend-tmpbeg;
        // in-band 
        data1 = new double[len1];
        inds1 = new int[len1];
        // out-of-band
        len2 = xfer-len1;
        data2 = new double[len2];    
        inds2 = new int[len2];

        // build the index array (they will be cloned later for quicksort)
        for (m=0; m<len1; m++)
          inds1[m] = m+tmpbeg;
        for (m=0; m<tmpbeg; m++)
          inds2[m] = m;
        for (m=tmpend, n=tmpbeg; m<xfer; m++, n++)
          inds2[n] = m;

        medind = len1/2;
        qsort = new QuickSort();
        break;
      case CHAR:
        if ( nargs != 7 )
          M.error("Error! Error! Incorrect number of arguments for function: Characterize! i.e. icemeasure char <out> <in1> <fs> <fbw> <thresh> <freq1>");
        ho.setSubRecords("SNR|SD,SFDR|SD,NF|SD,THRESH|SD,SIG|SD,MAXSPUR|SD,MAXSPURFREQ|SD"); 

        // sampling rate
        fs = (MA.get("FS")==null) ? MA.getD(4) : MA.getD("FS");
        if ( fs <= 0 )
          M.error("You must supply a valid sampling rate!");
        // fractional bandwidth
        fbw = (MA.get("FBW")==null) ? MA.getD(5) : MA.getD("FBW");
        if ( fbw < 0 || fbw > 1)
          M.error("Fractional bandwidth (FBW) must be between 0.0 and 1.0!");
        // threshold 
        THRESH1 = (MA.get("THRESH")==null) ? MA.getL(6) : MA.getL("THRESH");
        if ( THRESH1 < 0 )
          M.error("Threshold must be a logarithmic value!");
        thresh = Math.pow(10, THRESH1/10);
        // get frequency 
        freq1 = (MA.get("FREQ1")==null) ? MA.getD(7) : MA.getD("FREQ1");
        mfreq1 = mapFreq(freq1);
        if ( mfreq1 > fs || mfreq1 < 0 )
          M.error("Specified frequency is outside of sampling rate!");

        // calculate frequency/bin
        fpb = (complex) ? fs/xfer : 0.5*fs/xfer;

        // noise floor adjustment
        adj_val = 10*Math.log10(fpb);      
        //dcinds = (int) Math.ceil(0.25*bw/fpb);
        dcend = detDcInds(xfer);
        // now map this to a start and end points
        npoints = fbw*xfer - 2*dcend;
        tmpbeg = (int) Math.floor(0.5*(xfer-npoints));
        tmpend = tmpbeg + (int) Math.ceil(npoints);
        len1 = tmpend-tmpbeg;
        data1 = new double[len1];
        inds1 = new int[len1];
        // build index array once, we will clone it on every sort
        for (m=0, n=tmpbeg; m<len1; m++, n++)
          inds1[m] = n;
        // index of median (into data/inds)
        medind = len1/2;
        // now determine the signal indices
        inds2 = fillIndices(mfreq1, dcend, 0);
        len2 = inds2.length;
        qsort = new QuickSort();
        break;
      case FLASH:
        if ( nargs != 5 )
          M.error("Error! Incorrect number of arguments for function: Bandwidth! i.e. icemeasure flash <out> <in1> <alias> <cardnum>");

        xfer = (int) hi1.getSize();
        if ( xfer != FLASHSIZE ) {
          M.error("File size ("+xfer+") not equal to "+FLASHSIZE);
          return FINISH;
        }
        if ( !hi1.getFormat().equals("SB") ) {
          M.error("Input type must be SB");
          return FINISH;
        }
        ho.setSubRecords("ERROR|SB,SHA1|8A,SHA2|8A");
        /* get alias */
        str = (MA.get("ALIAS")==null) ? MA.getCS(4) : MA.getCS("ALIAS");
        /* get card number */
        m = (MA.get("CARDNUM")==null) ? MA.getL(5) : MA.getL("CARDNUM");
        /* build complete alias */
        str = String.format(str, m-1);
        break;
      case HARM:      
        if ( nargs != 8 )
          M.error("Error!  Error! Incorrect number of arguments for function: Harmonics! i.e. icemeasure harm <out> <in1> <fs> <thresh> <freq1> <freq2> <order>");
        ho.setSubRecords("SNR|SD,S1S2|SD,SFDR|SD,NF|SD,THRESH|SD,MAXSPURFREQ|SL,MAXSPUR|SD");

        // sampling rate
        fs = (MA.get("FS")==null) ? MA.getD(4) : MA.getD("FS");
        if ( fs <= 0 )
          M.error("You must supply a valid sampling rate!");
        // threshold
        THRESH1 = (MA.get("THRESH")==null) ? MA.getL(5) : MA.getL("THRESH");
        if ( THRESH1 < 0 )
          M.error("Threshold must be a logarithmic value!");
        thresh = Math.pow(10, THRESH1/10);
        // first frequency
        freq1 = (MA.get("FREQ1")==null) ? MA.getD(6) : MA.getD("FREQ1");
        mfreq1 = mapFreq(freq1);
        if ( mfreq1 < 0 || mfreq1 > fs )
          M.error("First frequency is outside of sampling rate!");
        // second frequency
        freq2 = (MA.get("FREQ2")==null) ? MA.getD(7) : MA.getD("FREQ2");
        mfreq2 = mapFreq(freq2);
        if ( mfreq2 < 0 || mfreq2 > fs )
          M.error("Second frequency is outside of sampling rate!");
        // harmonic/intermod order
        int order = (MA.get("ORDER")==null) ? MA.getL(8) : MA.getL("ORDER");

        // DC null!
        dcend = detDcInds(xfer);
        lenn = xfer-2*dcend;
        datan = new double[lenn];
        indsn = new int[lenn];
        for (m=0, n=dcend; m<lenn; m++, n++) 
          indsn[m] = n;

        // calculate frequencies per bin and the NF adjustment
        fpb = (complex) ? fs/xfer : 0.5*fs/xfer;
        adj_val = 10*Math.log10(fpb);

        // signals
        inds1 = fillIndices(mfreq1, dcend, 0);
        len1 = inds1.length;
        inds2 = fillIndices(mfreq2, dcend, 0);
        len2 = inds2.length;

        spurs = new ArrayList<IceSpur>();
        IceSpur spur;
        double fh1, fh2, fh3, fh4, mult;
        // find the harmonics
        for (m=2; m<=order; m++) {
          // find the harmonics
          fh1 = mapFreq(m*freq1);
          fh2 = mapFreq(m*freq2);
          if ( fh1 <= fs && fh1 > 0 ) {
            spur = new IceSpur();
            spur.setIndices(fillIndices(fh1, dcend, dcend));
            spur.setColumn("H"+m+"F1");
            spur.setFreq(unmapFreq(fh1));
            spurs.add(spur);
            ho.addSubRec(spur.getColumn()+"FREQ", "SL");
            ho.addSubRec(spur.getColumn()+"POW", "SD");
            ho.addSubRec(spur.getColumn()+"DBC", "SD");
          }
          if ( fh2 <= fs && fh2 > 0 ) {
            spur = new IceSpur();
            spur.setIndices(fillIndices(fh2, dcend, dcend));
            spur.setColumn("H"+m+"F2");        
            spur.setFreq(unmapFreq(fh2));
            spurs.add(spur);
            ho.addSubRec(spur.getColumn()+"FREQ", "SL");
            ho.addSubRec(spur.getColumn()+"POW", "SD");
            ho.addSubRec(spur.getColumn()+"DBC", "SD");
          }
          // find the intermods
          if ( m%2==0 ) {
            mult = m/2;
            fh1 = mapFreq(Math.abs(mult*freq1-mult*freq2));
            fh2 = mapFreq(mult*freq1+mult*freq2);
            // put these outside
            fh3 = fs+1;
            fh4 = fs+1;
          }
          else {
            mult = (m-1) / 2;
            fh1 = mapFreq((mult+1)*freq1 - mult*freq2);
            fh2 = mapFreq((mult+1)*freq2 - mult*freq1);
            fh3 = mapFreq((mult+1)*freq1 + mult*freq2);
            fh4 = mapFreq((mult+1)*freq2 + mult*freq1);
          }

          if ( fh1 <= fs && fh1 > 0 ) {
            spur = new IceSpur();
            spur.setIndices(fillIndices(fh1, dcend, dcend));
            spur.setColumn("I"+m+"A");
            spur.setFreq(unmapFreq(fh1));
            spurs.add(spur);
            ho.addSubRec(spur.getColumn()+"FREQ", "SL");
            ho.addSubRec(spur.getColumn()+"POW", "SD");
            ho.addSubRec(spur.getColumn()+"DBC", "SD");
          }
          if ( fh2 <= fs && fh2 > 0 ) {
            spur = new IceSpur();
            spur.setIndices(fillIndices(fh2, dcend, dcend));
            spur.setColumn("I"+m+"B");
            spur.setFreq(unmapFreq(fh2));
            spurs.add(spur);
            ho.addSubRec(spur.getColumn()+"FREQ", "SL");
            ho.addSubRec(spur.getColumn()+"POW", "SD");
            ho.addSubRec(spur.getColumn()+"DBC", "SD");
          }
          if ( fh3 <= fs && fh3 > 0 ) {
            spur = new IceSpur();
            spur.setIndices(fillIndices(fh3, dcend, dcend));
            spur.setColumn("I"+m+"C");
            spur.setFreq(unmapFreq(fh3));
            spurs.add(spur);
            ho.addSubRec(spur.getColumn()+"FREQ", "SL");
            ho.addSubRec(spur.getColumn()+"POW", "SD");
            ho.addSubRec(spur.getColumn()+"DBC", "SD");
          }
          if ( fh4 <= fs && fh4 > 0 ) {
            spur = new IceSpur();
            spur.setIndices(fillIndices(fh4, dcend, dcend));
            spur.setColumn("I"+m+"D");
            spur.setFreq(unmapFreq(fh4));
            spurs.add(spur);
            ho.addSubRec(spur.getColumn()+"FREQ", "SL");
            ho.addSubRec(spur.getColumn()+"POW", "SD");
            ho.addSubRec(spur.getColumn()+"DBC", "SD");
          }
        }
        medind = xfer/2;      
        qsort = new QuickSort();
        break;
      case HIST:
        if ( nargs != 5 )
          M.error("Error! Incorrect number of arguments for function: Bandwidth! i.e. icemeasure stuck <out> <in1> <bits> <bitsoff>"); 
        // number of bits
        int nbits = ((MA.get("BITS")==null) ? MA.getL(4) : MA.getL("BITS") );
        int bitsoff = ((MA.get("BITSOFF")==null) ? MA.getL(5) : MA.getL("BITSOFF") );
        int bitpow = nbits + bitsoff - 1;
        adj_val = (int) Math.pow(2, bitpow+1);
        bit_accumr = new long[nbits];
        bit_accumi = new long[nbits];
        indsn = new int[nbits];
        // setup real outputs
        ho.addSubRec("REMSB", "SL");
        indsn[0] = (int) Math.pow(2, bitpow);
        for ( m=1; m<nbits; m++ ) {
          indsn[m] = (int) Math.pow(2, bitpow-(m));

          ho.addSubRec("RE"+m, "SL");
        }

        // setup imag outputs
        ho.addSubRec("IMMSB", "SL");
        for ( m=1; m<nbits; m++ )
          ho.addSubRec("IM"+m, "SL");      
        // set the transfer length for this case
        xfer = MA.getL("/TL", 16384);
        break;
      case MSE:
        if ( nargs != 5 )
          M.error("Error! Error! Incorrect number of arguments for function: Mean Square Error! i.e. icemeasure mse <out> <in1> <in2> <segments>");
        ho.setSubRecords("AVERAGE|SD,WORST|SD");

        // get additional input file
        hi2 = (MA.get("IN2")==null) ? MA.getDataFile(4) : MA.getDataFile("IN2");
        hi2.open();
        hi2.setDFS(0);
        dbi2 = hi2.getDataBuffer(xfer);
        // get the number of segments
        int nsegs = (MA.get("SEGMENTS")==null) ? MA.getL(5) : MA.getL("SEGMENTS");
        nsegs = ( nsegs < 0 ) ? 1 : nsegs;

        // DC null!
        dcend = detDcInds(xfer);
        len1 = len2 = xfer-2*dcend;
        data1 = new double[len1];
        data2 = new double[len1];
        inds1 = new int[len1];

        // build index array once, we will clone it on every sort
        for (m=0, n=dcend; m<len1; m++, n++)
          inds1[m] = n;

        // get the user input # of segments
        indsn = new int[nsegs];
        if ( nsegs>1 ) {
          // calculate the end points for each segment
          int pps = (int) Math.floor(1.0*len1/nsegs);
          for (m=0; m<nsegs-1; m++) 
            indsn[m] = pps*(m+1)+dcend;
        }
        // ensure that we process through the end
        indsn[nsegs-1] = len1+dcend;
        qsort = new QuickSort();

        break;
      case NOISE:
        if ( nargs != 6 )
          M.error("Error! Incorrect number of arguments for function: Noise Floor! i.e. icemeasure noise <out> <in1> <fs> <fbw> <thresh>");
        ho.setSubRecords("NF|SD,THRESH|SD,MAXSPUR|SD,MAXSPURFREQ|SD");

        // sampling rate
        fs = (MA.get("FS")==null) ? MA.getD(4) : MA.getD("FS");
        if ( fs <= 0 )
          M.error("You must supply a valid sampling rate!");
        // fractional bandwidth
        fbw = (MA.get("FBW")==null) ? MA.getD(5) : MA.getD("FBW");
        if ( fbw < 0 || fbw > 1)
          M.error("Fractional bandwidth (FBW) must be between 0.0 and 1.0!");
        // threshold 
        THRESH1 = (MA.get("THRESH")==null) ? MA.getL(6) : MA.getL("THRESH");
        if ( THRESH1 < 0 )
          M.error("Threshold must be a logarithmic value!");
        thresh = Math.pow(10, THRESH1/10);

        fpb = (complex) ? fs/xfer : 0.5*fs/xfer;
        // noise floor adjustment
        adj_val = 10*Math.log10(fpb);
        dcend = detDcInds(xfer);

        fbw = ( fbw>1 ) ? 1 : ( fbw<0 ) ? 0 : fbw;

        npoints = fbw*xfer - 2*dcend;
        tmpbeg = (int) Math.floor(0.5*(xfer-npoints));
        tmpend = tmpbeg + (int) Math.ceil(npoints);

        len1 = tmpend-tmpbeg;
        data1 = new double[len1];
        inds1 = new int[len1];
        for (m=0, n=tmpbeg; m<len1; m++, n++) 
          inds1[m] = n;
        medind = len1/2;
        qsort = new QuickSort();
        break;   
      case NOTCH:
        if ( nargs != 6 )
          M.error("Error! Error! Incorrect number of arguments for function: Notch! i.e. icemeasure notch <out> <in1> <fs> <nbw> <fbw>");
        ho.setSubRecords("MEDIAN|SD,NOTCH|SD,TONE|SD");

        // sampling rate
        fs = (MA.get("FS")==null) ? MA.getD(4) : MA.getD("FS");
        if ( fs <= 0 )
          M.error("You must supply a valid sampling rate!");
        // notch bandwidth
        double nbw = (MA.get("NBW")==null) ? MA.getD(5) : MA.getD("NBW");
        if ( nbw > fs || nbw < 0 )
          M.error("Notch bandwidth must be less than the sampling rate and greater than zero!");
        // fractional bandwidth
        fbw = (MA.get("FBW")==null) ? MA.getD(6) : MA.getD("FBW");
        if ( fbw < 0 || fbw > 1)
          M.error("Fractional bandwidth (FBW) must be between 0.0 and 1.0!");

        fpb = complex ? fs/xfer : 0.5*fs/xfer;

        // tone bandwidth ~ 500 Hz
        npoints = 500.0/fpb;
        tmpbeg = (int) Math.floor(0.5*(xfer-npoints));
        tmpend = (int) Math.ceil(0.5*(xfer+npoints));
        int tlen = (tmpend - tmpbeg);

        // calculate notch width 
        npoints = nbw/fpb;
        int pbeg = (int) Math.floor(0.5*(xfer-npoints));
        int pend = (int) Math.ceil(0.5*(xfer+npoints));
        len1 = (pend - pbeg) - tlen;
        inds1 = new int[len1];
        data1 = new double[len1];

        for (m=0, n=pbeg; n<tmpbeg; m++, n++)
          inds1[m] = n;
        for (n=tmpend; n<pend; m++, n++)      
          inds1[m] = n;

        // calculate what should be the noise floor
        npoints = ( fbw>1 ? 1 : fbw < 0 ? 0 : fbw)*xfer;
        pbeg = (int) Math.floor(0.5*(xfer-npoints));
        pend = (int) Math.ceil(0.5*(xfer+npoints));
        len2 = (pend - pbeg) - (tlen + len1) + 1;
        if ( len2 < 0 ) 
          M.error("FBW (" + fbw + ")value is too small! Please increase!");

        inds2 = new int[len2];
        data2 = new double[len2];
        for (m=0, n=pbeg; n<inds1[0]; m++, n++)
          inds2[m] = n;
        for (n=inds1[len1-1]; n<pend; m++, n++)
          inds2[m] = n;
        qsort = new QuickSort();
        break;
      case NPR:
        if ( nargs != 9 )
          M.error("Error! Error! Incorrect number of arguments for function: Noise Power Ratio! i.e. icemeasure npr <out> <in1> <freq1> <bw1> <freq2> <bw2> <freqn> <bwn>");
        ho.setSubRecords("SIGNAL|SD,NOTCH|SD");

        // 1st passband
        freq1 = (MA.get("FREQ1")==null) ? MA.getD(4) : MA.getD("FREQ1");
        if ( freq1 < 0 || freq1 > 1.0 )
          M.error("Please specify passband as a percentage into the signal (i.e. 0 < freq1 < 1.0)!");
        double bw1 = (MA.get("BW1")==null) ? MA.getD(5) : MA.getD("BW1");
        if ( bw1 < 0 || bw1 > 1.0 )
          M.error("Please specify passband width as a percentage into the signal (i.e. 0 < bw1 < 1.0)!");
        freq2 = (MA.get("FREQ2")==null) ? MA.getD(6) : MA.getD("FREQ2");
        if ( freq2 < 0 || freq2 > 1.0 )
          M.error("Please specify passband as a percentage into the signal (i.e. 0 < freq2 < 1.0)!");   
        double bw2 = (MA.get("BW2")==null) ? MA.getD(7) : MA.getD("BW2");
        if ( bw2 < 0 || bw2 > 1.0 )
          M.error("Please specify passband width as a percentage into the signal (i.e. 0 < bw2 < 1.0)!");
        double freqn = (MA.get("FREQN")==null) ? MA.getD(8) : MA.getD("FREQN");
        if ( freqn < 0 || freqn > 1.0 )
          M.error("Please specify passband as a percentage into the signal (i.e. 0 < freqn < 1.0)!");
        double bwn = (MA.get("BWN")==null) ? MA.getD(9) : MA.getD("BWN");
        if ( bwn < 0 || bwn > 1.0 )
          M.error("Please specify passband width as a percentage into the signal (i.e. 0 < bwn < 1.0)!");

        inds1 = detInds(freq1, bw1);
        inds2 = detInds(freq2, bw2);
        indsn = detInds(freqn, bwn);
        break;
      case SONET:
        if ( nargs != 8 )
          M.error("Error! Error! Incorrect number of arguments for function: SONET! i.e. icemeasure sonet <out> <in1> <in2> <skip2> <frame> <cor> <off>");
        ho.setSubRecords("ERRORS|SD,PROC|SD,BER|SD");

        xfer = MA.getL("/TL", 4096);

        // get additional input file
        hi2 = (MA.get("IN2")==null) ? MA.getDataFile(4) : MA.getDataFile("IN2");
        hi2.open();
        hi2.setDFS(0);

        // get the skip for the 2nd buffer
        int tmp_skip = (MA.get("SKIP2")==null) ? MA.getL(5) : MA.getL("SKIP2");
        // set transfer length to the frame size
        xfer = (MA.get("FRAME")==null) ? MA.getL(6) : MA.getL("FRAME");
        // grab how many frames they want to correlate over
        frames = (MA.get("COR")==null) ? MA.getL(7) : MA.getL("COR");
        // get the output result string
        str = (MA.get("OFF")==null) ? MA.getS(8) : MA.getS("OFF");
        MR.put(str,  -1);
        skip = tmp_skip*xfer;

        dbi2 = hi2.getDataBuffer(xfer);
        hi2.seek(skip);
        mode = MATCH;

        matched = 0;
        seek_ind = 0;
        break;     
      case UDP:
        if ( nargs != 4 )
          M.error("Error! Error! Incorrect number of arguments for function: UDP! i.e. icemeasure udp <out> <in1> <levels>");
        ho.setSubRecords("MAXDELTA|SD,ERRORS|SI");

        int levels = (MA.get("LEVELS")==null) ? MA.getL(4) : MA.getL("LEVELS");
        thresh = Math.pow(2, levels);
        xfer = (int) thresh;
        break;
      default:
        M.error("Error! Function " + "TBD" + " does not exist! Valid functions are " + funcList);
    }
    // open the input file
    ho.open(DataFile.OUTPUT);
    dbi1 = hi1.getDataBuffer(xfer);
    return NORMAL;
  }

  public int process () {
    int n1 = hi1.read(dbi1);
    if (n1<0) return (FINISH);
    double[] in1 = dbi1.castD(true);
    int m, n;
    double accum1, accum2, cur, cur_thresh, median, peak;
    int[] sinds1, sinds2, sindsn;
    switch (func) {
      case BAND:
        int tmpbeg = inds1[0];
        int tmpend = inds1[len1-1]+1;

        // do work on the in-band portion
        System.arraycopy(in1, tmpbeg, data1, 0, len1);
        // clone then sort
        sinds1 = inds1.clone();
        qsort.sort(data1, sinds1, len1);

        // get the median and the threshold value
        median = data1[medind];
        cur_thresh = median*thresh;

        // find first ind less than threshold
        for (m=0; m<medind; m++) {
          if (data1[m] < cur_thresh)
            break;
        }
        double pin = 100.0*m/len1;

        double pout = 0.0;      
        if ( len2 > 0 ) {
          // do work on the out of band portion
          System.arraycopy(in1, 0, data2, 0, tmpbeg);      
          System.arraycopy(in1, tmpend, data2, tmpbeg, xfer-tmpend);
          // clone then sort
          sinds2 = inds2.clone();      
          qsort.sort(data2, sinds2, len2);

          // find ind's greater than value of of band
          for (m=0; m<len2; m++) {
            if (data2[m] < cur_thresh )
              break;
            pout = 100.0*(m+1)/xfer;
          }
        }
        // do the output
        str = String.format("{MEDIAN=%f,IN=%f,OUT=%f}",
            10*Math.log10(median), pin, pout);
        break;
      case CHAR:
        // get the signal peak
        peak = Double.MIN_VALUE;
        for (m=0; m<len2; m++) {
          if ( peak < in1[inds2[m]])
            peak = in1[inds2[m]];
        }
        System.arraycopy(in1, inds1[0], data1, 0, len1);
        // clone the indices before sorting
        sinds1 = inds1.clone();
        // perform quick sort
        qsort.sort(data1, sinds1, len1);
        // search for max spur & location!
        int ib = inds2[0];
        int ie = inds2[len2-1];
        double max_spur = Double.MIN_VALUE;
        int max_ind = -1;
        // search for max spur! (just in case it was not an intermod)
        for (m=0; m<len1; m++) {
          // make sure that we are outside of our signal boundaries
          if ( sinds1[m] < ib || sinds1[m] > ie ) {
            max_spur = data1[m];
            max_ind = sinds1[m];
            break;
          }
        }
        median = data1[medind];
        cur_thresh = median*thresh;
        int ni = 0;
        for (m=0; m<medind; m++) {
          if (data1[m] < cur_thresh)
            break;
          // don't count our signal
          if ( sinds1[m] < ib || sinds1[m] > ie )
            ni++;
        }

        double cpow = 10*Math.log10(peak);
        double spow = 10*Math.log10(max_spur);
        double npow = 10*Math.log10(median) - adj_val;
        str = String.format("{SNR=%s,SFDR=%s,NF=%s,THRESH=%s,SIG=%s,MAXSPUR=%s,MAXSPURFREQ=%s}",
            cpow-npow, cpow-spow, npow, 100.0*ni/len1, cpow, spow, freqFromInd(max_ind));
        break;
      case FLASH:
        /* open pic device */
        MDevIce pic = new MDevIce(MA, str);
        if ( pic.open() <= 0 ) {
          M.error("Cannot open alias: "+str);
          return FINISH;
        }
        /* reset card */
        if ( pic.reset(0) <= 0 ) {
          M.error("Cannot reset card");
          return FINISH;
        }
        /* set flash */
        if ( pic.setKey(0, MDevIce.KEY_FLASH, dbi1.buf, FLASHSIZE) <= 0 ) {
          M.error("Cannot set flash");
          return FINISH;
        }
        /* get flash */
        byte[] buf = new byte[FLASHSIZE];
        if ( pic.getKey(0, MDevIce.KEY_FLASH, buf, FLASHSIZE) <= 0 ) {
          M.error("Cannot get flash");
          return FINISH;
        }
        /* close pic device */
        pic.close();
        /* get results */
        m = ( new String(dbi1.buf)).equals(new String(buf) ) ? 0 : 1;
        str = String.format("{ERROR=%s,SHA1=%s,SHA2=%s}",
            m, checkSum(dbi1.buf, FLASHSIZE), checkSum(buf, FLASHSIZE));
        break;
      case HARM:
        // get the peak inside the two signal index arrays
        double peak1 = Double.MIN_VALUE;
        for (m=0; m<len1; m++) {
          if ( peak1 < in1[inds1[m]])
            peak1 = in1[inds1[m]];
        }
        double peak2 = Double.MIN_VALUE;
        for (m=0; m<len2; m++) {
          if ( peak2 < in1[inds2[m]])
            peak2 = in1[inds2[m]];
        }
        // calculate the harmonics/intermods/spurs
        int[] tmpinds;
        double cp;
        String harmstr = "";
        // now do harmonics/spurs/intermods
        for (m=0; m<spurs.size(); m++) {
          cp = Double.MIN_VALUE;
          tmpinds = spurs.get(m).getIndices();
          for (n=0; n<tmpinds.length; n++) {
            if ( cp < in1[tmpinds[n]] )
              cp = in1[tmpinds[n]];
          } 
          spurs.get(m).setMaxPower(cp, -1);      
          harmstr = String.format("%s,%sFREQ=%d", harmstr, spurs.get(m).getColumn(), (int) spurs.get(m).getFreq());
          harmstr = String.format("%s,%sPOW=%.02f", harmstr, spurs.get(m).getColumn(), 10*Math.log10(spurs.get(m).getMaxPower()));
          harmstr = String.format("%s,%sDBC=%.02f", harmstr, spurs.get(m).getColumn(), -10*Math.log10(peak1/spurs.get(m).getMaxPower()));
        }

        // array copy
        System.arraycopy(in1, indsn[0], datan, 0, lenn);
        // clone the index array
        sindsn = indsn.clone();

        // perform quick sort
        qsort.sort(datan, sindsn, lenn);
        // search for max spur & location!
        int i1b = inds1[0];
        int i1e = inds1[len1-1];
        int i2b = inds2[0];
        int i2e = inds2[len2-1];
        double maxHarm = Double.MIN_VALUE;
        int maxInd = -1;
        // search for max spur! (just in case it was not an intermod)
        for (m=0; m<lenn; m++) {
          // make sure that we are outside of our signal boundaries
          if ( (sindsn[m] < i1b || sindsn[m] > i1e) && (sindsn[m] < i2b || sindsn[m] > i2e) ) {
            maxHarm = datan[m];
            maxInd = sindsn[m];
            break;
          }
        }

        int np = 0;
        median = datan[medind];
        cur_thresh = median*thresh;
        for (m=0; m<medind; m++) {
          if (datan[m] < cur_thresh)
            break;
          if ( (sindsn[m] < i1b || sindsn[m] > i1e) && (sindsn[m] < i2b || sindsn[m] > i2e) ) {
            np++;
          }
        }
        str = String.format("{SNR=%.02f,S1S2=%.02f,SFDR=%.02f,NF=%02f,THRESH=%.02f,MAXSPURFREQ=%d,MAXSPUR=%.02f%s}", 
            10*Math.log10(peak1/median) + adj_val, 10*Math.log10(peak1/peak2), 10*Math.log10(peak1/maxHarm), 10*Math.log10(median) - adj_val, 100.0*np/lenn, (int) unmapFreq(freqFromInd(maxInd)), 10*Math.log10(maxHarm), harmstr);
        break;
      case HIST:
        int bits = bit_accumr.length;
        // clear last 
        for ( m=0; m<bits; m++ ) {
          bit_accumr[m] = bit_accumi[m] = 0;
        } 
        int valr, vali; 
        for ( m=0; m<2*n1; m+=2 ) {        
          /*
        // grab the values (un-two's complement negative)
        // TODO this should really be what we want (but we haven't confirmed on an actual stuck bit yet)
        valr = (int) ( ( in1[m  ] >=0 ) ? in1[m  ] : adj_val + in1[m  ]);
        vali = (int) ( ( in1[m+1] >=0 ) ? in1[m+1] : adj_val + in1[m+1]);
           */
          valr = (int) ( ( in1[m  ] >=0 ) ? in1[m  ] : 0 );
          vali = (int) ( ( in1[m+1] >=0 ) ? in1[m+1] : 0 );

          for ( n=0; n<bits; n++ ) {
            if ( valr >= indsn[n] ) {
              bit_accumr[n]++;
              valr %= indsn[n];
            }
            if ( vali >= indsn[n] ) {
              bit_accumi[n]++;
              vali %= indsn[n];
            }
          }
        }

        // do the output
        str = "{REMSB="+bit_accumr[0]+",IMMSB="+bit_accumi[0];
        for ( m=1; m<bits; m++ )
          str += ",RE"+m+"="+bit_accumr[m]+",IM"+m+"="+bit_accumi[m];
        str += "}";
        break;
      case MSE:
        // read the 2nd file
        int n2 = hi2.read(dbi2);
        if (n2<0) return (FINISH);
        double[] in2 = dbi2.castD(true);      

        accum1 = accum2 = 0;

        // calculate PSD of each first
        for (m=0; m<len1; m++) {
          data1[m] = Math.sqrt(in1[inds1[m]]);
          data2[m] = Math.sqrt(in2[inds1[m]]);
          accum1 += data1[m]/len1;
          accum2 += data2[m]/len1;
        }

        // uncast 2nd input signal as we are done processing it
        dbi2.uncast(in2, false);

        double delta = accum1/accum2;
        double acc1, acc2;
        double worst = -1;
        double accum = 0;
        double invdenom = 1.0/(accum1*accum1*len1/indsn.length);
        // step through each segment
        int z;
        for (m=0, n=inds1[0], z=0; m<indsn.length; m++) {
          // accumulate the error in each segment
          acc1 = acc2 = 0;
          for ( ; n<indsn[m]; n++, z++) {
            acc1 += data1[z];
            acc2 += data2[z];
          }
          // normalized mean squared error for this segment
          cur = Math.pow(acc1 - delta*acc2, 2.0)*invdenom;

          // look for the worst
          if ( worst < cur ) 
            worst = cur;
          // accumulate
          accum += cur;
        }
        str = String.format("{AVERAGE=%f,WORST=%f}", accum/indsn.length, worst);

        break;
      case NOISE:
        // copy section of interest
        System.arraycopy(in1, inds1[0], data1, 0, len1);
        // clone the index array
        sinds1 = inds1.clone();
        // perform quick sort
        qsort.sort(data1, sinds1, len1);

        median = data1[medind];
        cur_thresh = thresh*median;
        for (m=0; m<medind; m++) {
          if ( data1[m] < cur_thresh )
            break;
        }
        str = String.format("{NF=%f,THRESH=%f,MAXSPUR=%f,MAXSPURFREQ=%f}",
            10*Math.log10(median)-adj_val, 100.0*(m+1)/len1, 10*Math.log10(data1[0]), freqFromInd(sinds1[0]));
        break;  
      case NOTCH:
        // copy notch
        System.arraycopy(in1, inds1[0], data1, 0, len1);
        // copy nf
        System.arraycopy(in1, inds2[0], data2, 0, len2);
        // sort the two arrays
        sinds1 = inds1.clone();
        qsort.sort(data1, sinds1, len1);
        sinds2 = inds2.clone();
        qsort.sort(data2, sinds2, len2);

        peak = Double.MIN_VALUE;
        for (m=inds1[0]; m<inds1[len1-1]; m++) {
          if ( peak < in1[m])
            peak = in1[m];
        }
        str =  String.format("{MEDIAN=%f,NOTCH=%f,TONE=%f}",
            10*Math.log10(data2[len2/2]), 10*Math.log10(data1[len1/2]), 10*Math.log10(peak));
        break;
      case NPR:
        accum1 = 0;
        for (m=0; m<inds1.length; m++)
          accum1 += in1[inds1[m]];
        accum1 /= inds1.length;

        accum2 = 0;
        for (m=0; m<inds2.length; m++)
          accum2 += in1[inds2[m]];
        accum2 /= inds2.length;

        double accumn = 0;
        for (m=0; m<indsn.length; m++)
          accumn += in1[indsn[m]];
        accumn /= indsn.length;

        str = String.format("{SIGNAL=%f,NOTCH=%f}", 10*Math.log10(accum1/accum2), 10*Math.log10(accum1/accumn));
        break;
      case SONET:
        // read the 2nd file
        n2 = hi2.read(dbi2);
        len1 = ( n2 > n1 ) ? n1 : n2;
        if ( len1 <= 0 ) 
          return FINISH;
        in2 = dbi2.castD(true);
        // Sonet test has 2 modes, 1 looks for sync, 1 calculates BER      
        switch ( mode ) {
          case MATCH:
            // look for match
            for ( m=0; m<len1; m++ ) {
              if ( in1[m] != in2[m] ) {
                matched = 0;
                seek_ind++;
                // seek to next frame for comparison
                hi1.seek(seek_ind*xfer);
                hi2.seek(skip);
                dbi1.uncast(in1, false);
                dbi2.uncast(in2, false);
                return NORMAL;
              }
            }
            dbi1.uncast(in1, false);
            dbi2.uncast(in2, false);
            matched++;
            // look for exit case
            if ( matched >= frames ) {
              // switch modes, sync & set result
              mode = BER;
              hi1.seek(seek_ind*xfer);
              hi2.seek(skip);
              MR.put(str, seek_ind*xfer);
            }

            return NORMAL;
          case BER:
            accum1 = accum2 = 0;
            for (m=0; m<len1; m++) {
              if ( in1[m] != in2[m] ) {          
                accum1++;
              }      
            }
            accum2 += len1;
            // uncast 2nd input signal as we are done processing it
            dbi2.uncast(in2, false);
            str = String.format("{ERRORS=%s,PROC=%s,BER=%.02f}", accum1, accum2, 100.0*(accum1/accum2));
            break;
        }
        break;
      case UDP: 
        double max_delta = 0;
        int errs = 0;      
        // loop through looking for errors
        for (m=1; m<in1.length; m++) {
          cur = Math.abs(in1[m] - in1[m-1]);
          // skip roll-over case
          if ( cur != 2*thresh-1 ) {
            // look for a max
            if ( cur > max_delta  )
              max_delta = cur;
            // look for an error
            if ( cur > 1 )
              errs++;
          }
        }
        str = String.format("{MAXDELTA=%f,ERRORS=%s}", max_delta, errs);
        break;
      default:
        str = null;
        M.error("Got through to default case!!");
    }
    ho.insertData(-1, new Table(str));
    dbi1.uncast(in1, false);
    return NORMAL;
  }

  public int close () {
    hi1.close();
    ho.close();
    if ( hi2 != null ) hi2.close();
    return FINISH;
  }

  private double mapFreq (double freq) {
    return complex ? freq+0.5*fs : freq;
  }

  private double unmapFreq (double freq) {
    return complex ? freq - 0.5*fs : freq;
  }

  private double freqFromInd (int ind) {
    return complex ? ind*fpb-0.5*fs : ind*fpb;
  }

  private int[] fillIndices (double freq, int np, int dcinds) {
    int[] out;
    if ( freq < 0 || freq >= fs ) {
      out = new int[1];
      out[0] = 0;
      return out;
    }
    // calculate how far in the center is
    double cb1 = freq/fpb;
    //double np1 = bw / fpb;
    int ib = (int) Math.floor(cb1 - 0.5*np);
    int ie = (int) Math.ceil(cb1 + 0.5*np);
    // make sure the mapped frequency doesn't overstep
    ib = ( ib < dcinds ) ? dcinds : ib;
    ie = ( ie >= (xfer-(dcinds+1)) ) ? xfer-(dcinds+1) : ie;
    if ( ib >= xfer || ie <= 0 || ie < ib ) {
      M.warning("Signal falls outside of sampling rate!!!");
      out = new int[1];
      out[0] = 0;
      return out;
    }
    out = new int[ie-ib+1];
    for (int m=0, n=ib; n<=ie; m++, n++)
      out[m] = n;
    return out;
  }

  private int[] detInds (double strt, double wid) {
    int beg = (int) Math.ceil(xfer*strt);
    int np = (int) Math.floor(xfer*wid);
    int[] points = new int[np];
    for (int m=0; m<np; m++) 
      points[m] = m+beg;
    return points;
  }

  private int detDcInds (int nfft) {
    return (int) Math.floor(0.01*nfft);
  }

  /** Perform a sha256 hash on an input buffer 

  @param in input buffer
  @param len length of input buffer
  @return sha256 hash of buffer
   */
  private String checkSum ( byte[] in, int len ) {
    try {
      /* init sha256 sum */
      MessageDigest hash = MessageDigest.getInstance("SHA-256");
      /* set the buffer */
      hash.update(in, 0, len);
      /* digest */ 
      byte[] dig = hash.digest();
      /* convert to string */
      StringBuffer sb = new StringBuffer();
      for ( int m=0; m<dig.length; m++ ) {
        sb.append(String.format("%02x", dig[m]&0xFF));
      }
      return sb.toString();
    } catch ( Exception e ) {
      e.printStackTrace();
    }
    return null;
  }

  /** private class for spurious signals
   */
  private class IceSpur {
    /** indices that this spur occupies */
    private int[] inds;
    /** Center frequency of the spur */
    private double freq;
    /** maximum spur amplitude */
    private double maxpow;
    /** index of the maximum spur amplitude */
    private int maxind;
    /** Column name for Midas table file output */
    private String col;

    public IceSpur () {
      inds = null;
      maxpow = freq = 0;
      maxind = -1;
      col = null;
    }

    public void setIndices (int[] in) { inds = in; }
    public void setMaxPower (double f, int d) { 
      maxpow = f;
      maxind = d;
    }
    public void setColumn (String s) { col = s; }
    public void setFreq (double f) { freq = f; }

    public int[] getIndices () { return inds; }
    public double getMaxPower () { return maxpow; }
    public int getMaxInd () { return maxind; }
    public String getColumn () { return col; }
    public double getFreq () { return freq; }
  }

  private class QuickSort {
    private double[] out;
    private int[] ind;

    public void sort(double[] vals, int[] inds, int len) {
      out = vals;
      ind = inds;
      quicksortInds(0, len-1);
    }

    public void sort(double[] vals, int len) {
      out = vals;
      quicksort(0, len-1);
    }

    private void quicksortInds(int low, int high) {
      int i = low, j = high;
      // Get the pivot element from the middle of the list
      double pivot = out[low + (high-low)/2];

      // partition
      while (i <= j) {
        while (out[i] > pivot) i++;
        while (out[j] < pivot) j--;
        if (i <= j) { swapInds(i, j); i++; j--; }
      }
      // recursively call
      if (low < j) quicksortInds(low, j);
      if (i < high) quicksortInds(i, high);
    }

    private void quicksort(int low, int high) {
      int i = low, j = high;
      // Get the pivot element from the middle of the list
      double pivot = out[low + (high-low)/2];

      // partition
      while (i <= j) {
        while (out[i] > pivot) i++;
        while (out[j] < pivot) j--;
        if (i <= j) { swap(i, j); i++; j--; }
      }
      // recursively call
      if (low < j) quicksort(low, j);
      if (i < high) quicksort(i, high);
    }

    private void swapInds(int i, int j) {
      double tval = out[i];
      int tind = ind[i];
      out[i] = out[j];
      out[j] = tval;
      ind[i] = ind[j];
      ind[j] = tind;
    }

    private void swap(int i, int j) {
      double tval = out[i];
      out[i] = out[j];
      out[j] = tval;
    }
  }

}
