package nxm.ice.net;

import nxm.ice.lib.IceSys;
import nxm.sys.lib.*;

/**
  A class to display system information in an Browser.

  Supported requests are:
   Get?resultName
   Set?resultName=value
   Top
   Log
   DiskStats/auxName
   File/path

  @author Jeff Schoen
  @version $Id: HSystem.java,v 1.8 2014/09/23 12:35:44 jgs Exp $

*/
public class HSystem extends HSource {

  private Midas M;

  public HSystem (String name, Midas M) {
    this.M = M;
    this.name = name+"/";
  }

  public void handleRequest (String uri, HPage hp) {

    String option = uri.substring(name.length()+1);

    if (option.startsWith("Get?")) {
      getPage (option.substring(4),hp);
    }
    else if (option.startsWith("Set?")) {
      setPage (option.substring(4),hp);
    }
    else if (option.startsWith("Cmd?")) {
      cmdPage ("Cmd",option.substring(4),hp);
    }
    else if (option.startsWith("Top")) {
      cmdPage ("Top","top -n 1 -b",hp);
    }
    else if (option.startsWith("Log")) {
      int len = option.endsWith("Tail")? 512 : 8100;
      cmdPage ("Log","dmesg -s "+len,hp);
    }
    else if (option.startsWith("Disks")) {
      diskPage ("ALL",hp);
    }
    else if (option.startsWith("DiskStats/")) {
      String aux = option.substring(10);
      diskPage (aux,hp);
    }
    else if (option.startsWith("Ice-Driver")) {
      cmdPage (option,"cat /proc/driver/icepic",hp);
    }
    else if (option.startsWith("Ice-Version")) {
      cmdPage (option,"/opt/ice/pkg/icexxx/jre/ice version",hp);
    }
    else if (option.startsWith("Ice-Health")) {
      cmdPage (option,"cat /var/log/icehealth.log",hp);
    }
    else if (option.startsWith("Ice-PICs")) {
      tablePage (option,IceSys.getPICs(),hp);
    }
    else if (option.startsWith("Ice-NICs")) {
      tablePage (option,IceSys.getNICs(),hp);
    }
    else if (option.startsWith("Ice-Disks")) {
      tablePage (option,IceSys.getDisks(),hp);
    }
    else {
      optPage (hp);
    }
  }

  /* Display system log information if available */
  public void getPage (String key, HPage hp) {
    Object value = M.results.get(key);
    hp.open();
    hp.writeln (key+"="+value);
    hp.close();
  }

  public void setPage (String key, HPage hp) {
    int ie = key.indexOf("=");
    if (ie>0) {
      String value = key.substring(ie+1);
      key = key.substring(0,ie);
      M.results.setKey(key,value);
    }
    getPage(key,hp);
  }

  /* Display output of command */
  public void cmdPage (String name, String cmd, HPage hp) {
    byte[] buf = new byte[8192];
    cmd = StringUtil.replaceAll(cmd,"%20"," ");		// spaces
    cmd = StringUtil.replaceAll(cmd,"%22","\"");	// quotes
    cmd = StringUtil.replaceAll(cmd,"%27","'");		// tic
    int bytes = IceSys.runCommand(cmd,0,buf);
    boolean tbl = name.endsWith(".tbl");
    if (tbl) {
      hp.open();
      Table t=textToTable(buf,bytes); 
      hp.write(t.toString());
      hp.close();
    } else {
      hp.openToBody("HSystem Cmd Page");
      hp.writeln ("<h3><a href='' onClick=\"location.reload(); return false;\">System Cmd="+name+"</a></h3>");
      hp.writeln ("<br><pre>");
      hp.write(buf,0,bytes);
      hp.writeln ("</pre>");
      hp.closeFromBody();
    }
  }
  
  private String trim (String s) {
    if (s.charAt(0)=='"' && s.charAt(s.length()-1)=='"') s = s.substring(1,s.length()-1);
    return s;
  }
  private Table textToTable (byte[] buf, int len) {
    Table t = new Table();
    String s = new String(buf,0,len);
    for (int i=0; i<len; ) {
      int il = s.indexOf("\n",i);
      if (il<0) il=len;
      String line = s.substring(i,il);
      int ile = line.indexOf("=");
      Parser p = new Parser(line,',',true);
      String p1 = p.get(1);
      String p2 = p.get(2);
      if (p2.equals(":")) {		// list like /proc/driver output
	String key = trim(p.get(1)); 
	String val = trim(p.get(3));
	Object obj = null;
	if (val.indexOf('=')<0) {
	  obj = Convert.s2o(val,"_",M);
	} else {
	  String st=val;
	  for (int ip=4; ip<=p.elements(); ip++) {
	    val = p.get(ip);
	    int ie = val.indexOf('=');
	    if (ie>0) st = st+","+val;
	  }
	  obj = new Table("{"+st+"}");
	}
	t.put(key,obj);
      }
      else if (p1.endsWith(":")) {	// JSON like /var/lob/iceblock.log
        p1 = p1.substring(0,p1.length()-1);
	String key = trim(p1); 
	String val = trim(p2);
	Object obj = Convert.s2o(val,"_",M);
	t.put(key,obj);
      }
      else if (ile>0) {
	String key = trim(line.substring(0,ile)); 
	String val = trim(line.substring(ile+1));
	t.put(key,val);
      }
      i=il+1;
    }
    return t;
  }

  /* Display table of info page */
  public void tablePage (String name, Table t, HPage hp) {
    boolean tbl = name.endsWith(".tbl");
    if (tbl) {
      hp.open();
      hp.write(t.toString());
      hp.close();
    } else {
      hp.openToBody("HSystem "+name+" Page");
      hp.writeln ("<h3><a href='' onClick=\"location.reload(); return false;\">"+name+"</a></h3>");
      hp.writeln ("<br><pre>");
      Table.Iterator ti = t.iterator();
      while (ti.hasNext()) {
        String key = ti.next();
        Object value = t.get(key);
        hp.writeln (key+"="+value);
      }
      hp.writeln ("</pre>");
      hp.closeFromBody();
    }
  }

  /* Display disk stats for a given Auxiliary */
  public void diskPage (String aux, HPage hp) {
    hp.openToBody("HSystem Disk Page");
    hp.writeln ("<h3><a href='' onClick=\"location.reload(); return false;\">Aux="+aux+" Disk Stats</a></h3>");
    String list = aux;
    if (aux.equals("ALL")) list = M.results.getS("AUX.READ");
    String[] auxes = StringUtil.cleanSplit(list, "|");
    hp.writeln("<table border='1' cellpadding='2' cellspacing='2' align='left'>");
    for (int i=0; i<auxes.length; i++) {
      aux = auxes[i];
      String path = M.io.getAuxPath(aux);
      Table tab = OS.getStatFS(path);
      hp.writeln ("<tr>");
      hp.writeln ("<td>Aux   = "+aux+"</td>");
      hp.writeln ("<td>Path  = "+path+"</td>");
      if (tab!=null) {
        hp.writeln ("<td>Files = "+tab.getL("FILES")+"</td>");
        hp.writeln ("<td>Total = "+tab.getL("TMBY")+" Mby</td>");
        hp.writeln ("<td>Free  = "+tab.getL("FMBY")+" Mby</td>");
        hp.writeln ("<td>Used  = "+tab.getL("USED")+"%</td>");
      }
      hp.writeln ("</tr>");
    }
    hp.closeFromBody();
  }

  /* Display menu of all available display pages */
  public void optPage (HPage hp) {
    hp.openToBody("HSystem Options Page");
    hp.writeln ("<center><h2>NeXtMidas System Information</h2></center>");
    hp.writeln ("<ul>");
    hp.writeln ("<li><a href='/nmroot/index.html'>JavaDocs</a> - System tree documentation</li>");
    hp.writeln ("<li><a href='LogTop'>System Log</a> - Top of System Log File</li>");
    hp.writeln ("<li><a href='LogTail'>System Log</a> - Tail of System Log File</li>");
    hp.writeln ("<li><a href='LogAll'>System Log</a> - All of System Log File</li>");
    hp.writeln ("<li><a href='TopCPU'>Top CPU/Mem</a> - List of Top CPU/Memory Processes on Server</li>");
    hp.writeln ("<li><a href='Disks'>Disk Aux=ALL</a> - All Disks Statistics</li>");
    hp.writeln ("<li><a href='Ice-Version'>Ice-Version</a> - ICE Software Version</li>");
    hp.writeln ("<li><a href='Ice-Driver'>Ice-Driver</a> - ICE Driver Statistics</li>");
    hp.writeln ("<li><a href='Ice-Health'>Ice-Health</a> - ICE Block Health and Status</li>");
//    hp.writeln ("<li><a href='Ice-Archivers'>Ice-Archivers</a> - ICE Archiver Statistics</li>");
    hp.writeln ("<li><a href='Ice-Disks'>Ice-Disks</a> - ICE Disk Statistics</li>");
    hp.writeln ("<li><a href='Ice-PICs'>Ice-PICs</a> - ICE PIC Card Statistics</li>");
    hp.writeln ("<li><a href='Ice-NICs'>Ice-NICs</a> - ICE Network Card Statistics</li>");
    hp.writeln ("</ul>");
    hp.closeFromBody();
  }

}

