package nxm.ice.libg;

import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.sys.libg.*;
import nxm.sys.lib.Native;
import java.awt.*;

/**
  A class to plot trace vector data from ICE core test capture files

  @author Jeff Schoen
  @version $Id: LayerTV.java,v 1.13 2002/10/22 16:09:57 schoenj Exp $

*/
public class LayerTV extends Layer {

  private int nx, ny, nchan, nfs, bpe;
  private double factor, delx, xframe;
  private DataFile df;
  private int select=-1,selected=-1;
  private int single=-1;
  private int[] lbuf,bits,offs;
  private boolean clear;
  private String[] names;
  private Color cgrid = new Color(0x505050);
  private Color cselect = null;

  private static int MAXFRAME=32768;

  /* Constructors */
  public LayerTV () { 
    super("TV"); 
  }

  /** setup to attach a file */
  public boolean setFile (PlotFile pf, Table tab, Args MA) {
    setFile(pf);
    df = (DataFile)pf;
    nfs = (int)df.getSize();
    bpe = (int)df.getBPE();
    isPaged = nfs>MAXFRAME;
    if (isPaged) nfs=MAXFRAME;
    setXAxis( pf.getXStart(), pf.getXDelta() );
    lbuf = new int[nfs];
    setNChan(bpe*8);
    setup();
    return true;
  }

  public void setNChan (int n) {
    nchan = n;
    nx = 1; 
    ny = nchan;
    bits = new int[n];
    offs = new int[n];
    names = new String[n];
    for (int i=0; i<n; i++) { offs[i]=i; bits[i]=1; names[i]="Bit"+i; }
    setYAxis(0.0,ny);
  }
  public int getNChan() { return nchan; }

  private void redo() {
    findRange();
    MP.findRange();
    MP.useRange();
    MP.setup();
    refresh();
  }

  private void setup() { 
    delx = df.getXDelta();
    setFrame(nfs);
    setSize(1);
    setData(df.getDataBuffer(nfs));
    setColor(1);
    ndata=0;
  }

  /** get the next available data from the input source */
  public int getData (int mode) {
    int stat = Drawable.FULL;
    if (df==null || !df.isOpen() || data==null || isDone) return stat;
    int ndo = (int)df.avail();
    if (ndo>0) {
      if (ndo>nfs) ndo=nfs;
      fseek = pf.seek(); // update for getTimeAt reference
      if (isPaged) {
        double s=df.getStart(), d=df.getDelta();
        ndo = Math.min(ndo,(int)((MP.rx2-s)/d-fseek+2));
        setXStart(s+fseek*d);
        if (isReset) xframe=xstart;
      }
    }
    if (ndo>0) {
      ndo = df.read(data,ndo);
      if (bpe==1) for (int i=0; i<ndo; i++) lbuf[i]=data.getB(i);
      if (bpe==2) for (int i=0; i<ndo; i++) lbuf[i]=data.getI(i);
      if (bpe==4) for (int i=0; i<ndo; i++) lbuf[i]=data.getL(i);
      ndata=ndo;
      if (isPaged) stat = Drawable.DATAX;
      else stat = Drawable.FULL;
      isReset=false;
    }
    else if (isPaged) {
      df.seek(df.getSize());
      setXStart(df.getStart());
      isDone=true;
      ndata = 0;
    }
    else if (!realtime) stat = Drawable.FULL;
    else if (df.isStream() && !df.isStreaming()) stat = Drawable.DONE;
//    if (isReset || !isPaged) clear = true;
    a1 = 0;
    a2 = ny;
    return stat;
  }

  /* find layer bounds */
  public void findRange() { 
    x1 = xstart;
    x2 = xstart + (nfs-2)*xdelta;
    y1 = 0;
    y2 = ny;
  }

  public void findFullRange () { findFullRange (pf); } 

  public void findFullRange (PlotFile pf) { 
    fx1 = x1; fx2 = x2;
    fy1 = y1; fy2 = y2;
    if (pf!=null && !realtime) {
      fx1 = pf.getXStart();
      fx2 = fx2 + pf.getXDelta()*pf.getSize();      
    }
  }

  /** dray the layer on the specified plot */
  public void draw (int flag) {
    needRefresh=false;
    if (!MP.is2D) return;
    int i,j,ix,iy,ix1,ix2,iy1,iy2;
    Color cfill = MP.getOption(MP.O_CONTRAST)? Color.black : MP.theme.cbg;
    double dx = MP.mxx*xdelta;
    // draw the channel grid
    if (flag != Drawable.DATAX || clear) { 
      MP.gc.setColor(cfill);
      MP.gc.fillRect(MP.ix1,MP.iy1,MP.ix21,MP.iy21);
      MP.setColor(cgrid);
      ix1 = MP.ix1;
      ix2 = MP.ix2;
      for (i=0; i<=ny; i++) {
        iy = (int)(MP.myy*i+MP.myb);
	if (iy>MP.iy1 && iy<MP.iy2) {
	  MP.gc.drawLine(MP.ix1,iy,MP.ix2,iy);
	}
	if (dx>5) for (j=0; j<nfs; j++) {
          ix = (int)(MP.mxx*(xframe+xdelta*j)+MP.mxb);
	  if (ix>MP.ix1 && ix<MP.ix2) MP.gc.drawLine(ix,iy+2,ix,iy-2);
	}
      }
      clear=false; 
    }
    if (ndata<=0) return;
    // draw the channel data
    for (int ichan=0; ichan<nchan; ichan++) {
      int ibits = bits[ichan];
      int ioff  = offs[ichan];
      int imask = (1<<ibits)-1;
      ix = 0;
      iy = ichan;
      i = (MP.myy>0)? iy : iy+1;
      iy1 = (int)(MP.myy*i+MP.myb)+3;
      i = (MP.myy>0)? iy+1 : iy;
      iy2 = (int)(MP.myy*i+MP.myb)-3;
      if (iy1>MP.iy2 || iy2<MP.iy1) continue;
      double ry1 = 0;
      double ry2 = imask;
      double myy = (double)(iy1-iy2) / (ry2 - ry1);
      double myb = iy2 - ry1*myy;
      int iymin = iy1+1;
      int iymax = iy2-1;
      int[] pax=MP.pix.x, pay=MP.pix.y;
      int nmax=pax.length;
      for (i=j=0; i<ndata; i++,j++) {
	if (j==nmax) { MP.drawPixels (pax,pay,nmax,line); j=0; }
        ix = (int)(MP.mxx*(xstart+xdelta*i)+MP.mxb);
	int ud = (lbuf[i]>>ioff)&imask;
        iy  = (int)( ud*myy + myb);
        if (iy<iymin) iy=iymin; else if (iy>iymax) iy=iymax;
        pax[j] = ix;
	pay[j] = iy;
      }
      MP.drawPixels (pax,pay,j,line);
    }
  }

  public void setTraceConfig (String cfg) {
    if (cfg.equals("NONE")) return;
    if (cfg.startsWith("{")) cfg = cfg.substring(1,cfg.length()-1);
    Parser p = new Parser(cfg);
    setNChan(p.elements());
    int tbits=0;
    for (int i=0; i<nchan; i++) {
      String name = p.get(nchan-i);
      names[i]=name;
      bits[i]=1;
      offs[i]=tbits;
      if (name.endsWith("]")) {
	int ib = name.indexOf("[");
	int ic = name.indexOf(":");
	if (ib>0 && ic>0) {
	  bits[i] = Convert.s2l(name.substring(ib+1,ic)) -  Convert.s2l(name.substring(ic+1,name.length()-1)) + 1;
	}
      }
      tbits += bits[i];
    }
    System.out.println("TraceConfig {"+cfg+"} nchan="+nchan+" Bits="+tbits);
  }

  private int getChan (Position pos) {
    int iy = (int)pos.y;
    if (iy<0 || iy>=ny) select = -1;
    else select = iy;
    return select;
  }

  public void select (Position pos) {
    select = getChan(pos);
  }
  public int select() {
    return select;
  }
  public int getSelect() {
    return select;
  }
  public void setSelect (int chan) {
    selected = chan;
  }
  public void setSelectColor (String cstr) {
    cselect = MColor.getColor(cstr);
  }

  public void single (int channel) {
    single = channel;
  }
  public int single() {
    return single;
  }

  public String getReadOut (Position mp) {
    int iy = (int)mp.y;
    int ix = (int)((mp.x-xframe)/xdelta);
    int ioff=iy, ibits=1, imask=1;
    String name = "Bit"+iy;
    int val = 0;
    if (iy>=0 && iy<nchan && ix>0 && ix<nfs) {
      name = names[iy];
      ibits = bits[iy];
      ioff  = offs[iy];
      imask = (1<<ibits)-1;
      val = (lbuf[ix]>>ioff)&imask;
    }
    String sval = ""+val;
    if (name.indexOf('[')>0) {
      sval = Convert.l2x(val);
      while (sval.length()>3 && sval.indexOf("0x0")==0) sval=sval.replace("0x0","0x");
    }
    String tmp = " TV ("+name+"="+sval+")";
    return tmp;
  }
}
