/**
  Implements ICE/SDDS/VRT packet functions 

  @author Jeff Schoen
  @version $Id: PKT.c,v 1.3 2005/03/29 14:56:00 jgs Exp $
*/

#define PKTFLG_TRIM 0x01
#define PKTFLG_NOSID 0x02
#define PKTFLG_NOCID 0x04
#define PKTFLG_CTXX  0x08
#define PKTFLG_DALT 0x10
#define PKTFLG_TC   0x20
#define PKTFLG_BIGE 0x40
#define PKTFLG_QALT 0x80

#define PKTBLK_SEQ  0x10010000
#define PKTBLK_HDR  0x10020000
#define PKTBLK_CTX  0x10030000

#define PKTSEQ_CALT 0x00100000
#define PKTSEQ_WAIT 0x00200000
#define PKTSEQ_DONE 0x00400000
#define PKTSEQ_CJMP 0x00800000

#define PKTSEQ_HSET 0x01000000
#define PKTSEQ_HADD 0x02000000
#define PKTSEQ_HSTR 0x04000000
#define PKTSEQ_CHDR 0x08000000

#define PKTSEQ_HSXE 0x10000000
#define PKTSEQ_HSXC 0x20000000
#define PKTSEQ_HSEC 0x40000000
#define PKTSEQ_HSEI 0x40001000
#define PKTSEQ_CDAT 0x80000000

#define PKTSEQ_NOOP 0x00000000
#define PKTSEQ_CLR  0x30000000

#define PKTFLG_NOENA  0
#define PKTFLG_CARRY  1
#define PKTFLG_CHECK  2

#define PROTO_UDP 17
#define PROTO_IGMP 2
#define PROTO_ICMP 1
#define PROTO_ARP  0x0806
#define PROTO_IP   0x0800

#define VCTX_CTX 0
#define VCTX_SEC 1

#define VCTX_BW 2
#define VCTX_IF 3
#define VCTX_RF 4
#define VCTX_REFL 7
#define VCTX_GAIN 8
#define VCTX_SR 10
#define VCTX_TEMP 13
#define VCTX_FMT 16
#define VCTX_GPS 17

#define VCTX_DSEC 20
#define VCTX_HSEC 21
#define VCTX_MSEC 22
#define VCTX_NOCID 23

/* CORE Plan handle */
typedef struct {
  HALO halo;
  int_4 pkty;
  int_4 form;
  int_4 size;
  int_u4 myip;
  int_u4 mcip;
  int_4 port;
  int_8 rate;
  int_8 freq;
  int_8 bw;
  int_8 edes;
  int_4 gain;
  int_4 chan;
  int_u4 flgs;
  int_u4 wsec;	/* whole seconds */
  int_u4 qsec;	/* qano (250e-12) seconds */
  int_4 ncps;	/* channels per stream */
  int_4 nfps;	/* fibers per stream */
  int_4 ncpa;	/* channels per network address */
  int_4 ouid;
  int_4 muxm;	/* GBE mux mode for high speed data */
  int_4 indx;	/* GBE mux mode index */
  int_4 temp;
} PKTPlan;

/*
  seq = 0xsr00cbaa

  aa = nibble address or hdr word address or cdata output count
  b  = nibble adjustment value
  c  = nibble test value or enable set values 

  calt = seq[20];  // alt startup if carry
  wait = seq[21];  // wait
  done = seq[22];  // done
  cjmp = seq[23];  // jump to address if carry

  hset = seq[24];  // set to dat
  hadd = seq[25];  // add to dat
  hstr = seq[26];  // store dat
  chdr = seq[27];  // start output from header

  hsxe = seq[28];  // set/clear enable to seq[12]
  hsxc = seq[29];  // set/clear carry to seq[13]
  hsec = seq[30];  // set enable=carry
  cdat = seq[31];  // start output from data

  flag=0  straight add
  flag=1  set enable based on previous carry
  flag=2  just check add for overflow - no store

*/
int_4 PKT_getBPS (int_4 type) {
  int_4 bps;
  switch (type) {
    case 'F': bps = 4; break;
    case 'D': bps = 8; break;
    case 'X': bps = 8; break;
    case 'L': bps = 4; break;
    case 'I': bps = 2; break;
    case 'J': bps = -12; break;
    case 'B': bps = 1; break;
    case 'N': bps =-4; break;
    case 'P': bps =-1; break;
    case 'M': bps = 1; break;
    case 'A': bps = 8; break;
    default : bps = 1; break;
  }
  return bps;
}

int_4 PKT_getSPA (int_4 mode) {
  int_4 spa;
  switch (mode) {
    case 'S': spa = 1; break;
    case 'C': spa = 2; break;
    case 'V': spa = 3; break;
    case 'Q': spa = 4; break;
    case 'M': spa = 9; break;
    case 'X': spa = 10; break;
    case 'T': spa = 16; break;
    case 'A': spa = 32; break; /* since 3.1.0 to match X-Midas */
    default : spa = 1; break;
  }
  return spa;
}

int_4 swap (int_4 data) {
  char c, *cbuf=(char*)(&data);
  c=cbuf[0]; cbuf[0]=cbuf[3]; cbuf[3]=c;
  c=cbuf[1]; cbuf[1]=cbuf[2]; cbuf[2]=c;
  return data;
}

#define CRC32POLY_LE 0xedb88320

int_4 PKT_checksum (int_4 *cb, int_4 n, int_4 bitoff, int_u4 crc) {
  int_4 i,j;
  int_u4 qw;
  for (j=0; j<n; j++) {
    qw = cb[j]>>bitoff;
    /*printf("d=%08x crc=%08x\n",qw,crc);*/
    for (i=bitoff; i<32; i++,qw>>=1) crc = (crc>>1) ^ ( ((crc^qw)&0x1)? CRC32POLY_LE:0);
    bitoff=0;
  }
  return crc;
}

int_4 oneschksum (int_2 *i2, int_4 len) {
  int_4 i,chksum=0;
  for (i=0; i<len; i++) {
    chksum += (0xFFFF&i2[i]);
    if (chksum&0x10000) chksum = (chksum&0xFFFF)+1;
  }
  return (~chksum)&0xFFFF;
}

#define MAXARPTBL 2048
int_8 PKT_getMACaddr (int_4 addr) {
  int_4 i,na,lat,tpl[6]; int_8 mac=0; 
  char fn[40],saddr[20],data[MAXARPTBL],*s;
  FILE *fp;
  strcpy(fn,"/var/etc/arptbl");
  fp = fopen(fn,"r"); 
  if (fp==NULL) return -1;
  lat = fread(data,1,MAXARPTBL,fp); 
  fclose(fp);
  if (lat<=0) { printf("Err: Reading arp table file=%s or file is empty.\n",fn); return -2; }
  data[lat]=0;
  sprintf(saddr,"%d.%d.%d.%d",(addr>>24)&0xFF,(addr>>16)&0xFF,(addr>>8)&0xFF,(addr>>0)&0xFF);
  s = strstr(data,saddr);
  if (s==NULL) return -1;
  s = strstr(s,"lladdr");
  if (s==NULL) { printf("Err: finding lladdr=%s keyword in arp table file=%s.\n",saddr,fn); return -2; }
  na = sscanf(s+7,"%02x:%02x:%02x:%02x:%02x:%02x ",tpl+5,tpl+4,tpl+3,tpl+2,tpl+1,tpl+0);
  if (na!=6) { printf("Err: converting %s in arp table file=%s.\n",s,fn); return -2; }
  for (i=5; i>=0; i--) mac = (mac<<8) | (tpl[i]&0xFF);
  return mac;
}

int_4 PKT_ipeth (PKTPlan *plan, int_4 *cb, int_4 n, int_4 nudp, int_4 npad, int_4 proto) {
  int_4 i,j,n2; int_2 *i2;
  int_8 esrc = (0x1CE000L<<24)|(plan->myip&0xFFFFFFFFL);
  int_8 edes = (0x01005EL<<24)|(plan->mcip&0x007FFFFFL);
  int_4 ipsz = (proto==PROTO_IGMP)? 24 : 20;
  int_4 upl = 8 + nudp;			// total UDP|ICMP|IGMP length
  int_4 ipl = ipsz + upl;		// total IP length
  int_4 neth = 2 + 14 + ipl + npad;	// total ethernet packet length in bytes + 2by start prep - CRC
  int_4 ports = plan->port | (plan->port<<16);
  if (plan->edes>0) edes = plan->edes;
  if (plan->mcip==-1 || proto==PROTO_ARP) edes = 0xFFFFFFFFFFFFLL;
  if (proto==PROTO_ARP) {		// just happens to be same length as 0 len UDP packet	
    cb[n++] = swap(edes>>32)|neth;	// ETH destAddr[5:4]
    cb[n++] = swap(edes);		// ETH destAddr[3:0]
    cb[n++] = swap(esrc>>16);		// ETH srcAddr[5:2]
    cb[n++] = swap((esrc<<16)|0x0806);	// ETH type=IP:srcAddr[1:0]
    cb[n++] = swap(0x00010800);
    cb[n++] = swap(0x06040001);
    cb[n++] = swap(esrc>>16);		
    cb[n++] = swap((esrc<<16)|(plan->myip>>16));
    cb[n++] = swap((plan->myip<<16)|0);
    cb[n++] = swap(0);
    cb[n++] = swap(plan->mcip);
    for (i=0; i<npad; i+=4) cb[n++]=0;	// padding for minimum frame
  } else {
    cb[n++] = swap(edes>>32)|neth;	// ETH destAddr[5:4]
    cb[n++] = swap(edes);		// ETH destAddr[3:0]
    cb[n++] = swap(esrc>>16);		// ETH srcAddr[5:2]
    cb[n++] = swap((esrc<<16)|0x0800);	// ETH type=IP:srcAddr[1:0]
  }
  i2=(int_2*)(cb+n);
  if (proto==PROTO_UDP) {	
    cb[n++] = swap(0x45000000|ipl);	// IP totalLength:typeOfService=00:ver=4:len=5
    cb[n++] = swap(0x01ce0000); 	// IP identifier:frag
    cb[n++] = swap(0x40000000|(proto<<16));// IP chksum:protocol:ttl
    cb[n++] = swap(plan->myip); 	// IP srcAddr
    cb[n++] = swap(plan->mcip); 	// IP destAddr
    i2[5]   = oneschksum(i2,10);
    cb[n++] = swap(ports); 		// UDP destPort:srcPort
    cb[n++] = swap(upl<<16); 		// UDP chksum=0:length
  }
  if (proto==PROTO_IGMP) {	
    cb[n++] = swap(0x46000000|ipl);	// IP totalLength:typeOfService=00:ver=4:len=6
    cb[n++] = swap(0x01ce4000); 	// IP identifier:frag
    cb[n++] = swap(0x01000000|(proto<<16));// IP chksum:protocol:ttl
    cb[n++] = swap(plan->myip); 	// IP srcAddr
    cb[n++] = swap(plan->mcip); 	// IP destAddr
    cb[n++] = swap(0x94040000); 	// Switch route
    i2[5]   = oneschksum(i2,12);
    i2=(int_2*)(cb+n); n2=n;
    cb[n++] = swap(0x17000000); 	// IGMP Leave checksum:code:type
    cb[n++] = swap(plan->mcip); 	// IGMP group
    i2[1] = oneschksum(i2,4);		// calulate chksum for Leave
    j = cb[n2];				// save off Leave substitute value
    cb[n2] = swap(0x16000000); 		// IGMP Join checksum:code:type 
    i2[1] = oneschksum(i2,4);		// calulate chksum for Join
    cb[n++] = j;			// store Leave cmd in 1st pad word
    for (i=4; i<npad; i+=4) cb[n++]=0;	// padding for minimum frame
  }
  if (proto==PROTO_ICMP) {	
    cb[n++] = swap(0x45000000|ipl);	// IP totalLength:typeOfService=00:ver=4:len=5
    cb[n++] = swap(0x01ce4000); 	// IP identifier:frag
    cb[n++] = swap(0x40000000|(proto<<16));// IP chksum:protocol:ttl
    cb[n++] = swap(plan->myip); 	// IP srcAddr
    cb[n++] = swap(plan->mcip); 	// IP destAddr
    i2[5]   = oneschksum(i2,10);
    i2=(int_2*)(cb+n); n2=n;
    cb[n++] = swap(0x08000000); 	// ICMP Ping checksum:code:type
    cb[n++] = swap(0x01ce0001); 	// ICMP Ping seqno:ident
    i2[1] = oneschksum(i2,4);		// calulate chksum 
    for (i=0; i<npad; i+=4) cb[n++]=0;	// padding for minimum frame
  }
  return n;
}

int_4 getnadr (int_4 adr, int_4 k, int_4 nibs, int_4 bige) {
  int nadr = adr+k;
  if (!bige);
  else if (nibs>8) nadr = (adr+15-(k^1));     // assume in int_8
  else if (nibs>4) nadr = (adr+7-(k^1));      // assume in int_4
  else if (nibs>2) nadr = (adr+3-(k^1));      // assume in int_2
  return nadr;
}

int_4 PKT_nibadd (int_4 *cb, int_4 n, int_4 adr, int_8 data, int_4 nibs, int_4 flag, int_4 bige) {
  int_4 k=0,func,nadr;		/* set ena=prevcarry     :        set ena clear carry */
  func = (flag==PKTFLG_CARRY)? PKTSEQ_HSXC|PKTSEQ_HSEC : PKTSEQ_HSXE|PKTSEQ_HSXC|0x1000;
  if (adr+nibs>=0x100) printf("PKT_cblock error: adr=%d + nibs=%d > 256\n",adr,nibs);
  if (flag==PKTFLG_CHECK) {
    cb[n++] = func|getnadr(adr,k,nibs,bige);
    for (; k<nibs; k++,data>>=4) cb[n++] = PKTSEQ_HADD | ((data&0xF)<<8) | getnadr(adr,k+1,nibs,bige);
  }
  else {
    for (; k<nibs-1; k+=2,data>>=8,func=0) {	/* by 2 if possible */
      cb[n++] = func                                         | getnadr(adr,k,nibs,bige);
      cb[n++] = PKTSEQ_HADD               | ((data&0x0F)<<8) | getnadr(adr,k+1,nibs,bige);
      cb[n++] = PKTSEQ_HADD | PKTSEQ_HSTR | ((data&0xF0)<<4) | getnadr(adr,k,nibs,bige);
      cb[n++] = PKTSEQ_HSTR                                  | getnadr(adr,k+1,nibs,bige);
    }
    for (; k<nibs; k++,data>>=4,func=0) {
      nadr = getnadr(adr,k,nibs,bige);
      cb[n++] = func|nadr;
      cb[n++] = PKTSEQ_HADD | ((data&0xF)<<8) | nadr;
      cb[n++] = PKTSEQ_HSTR | nadr;
    }
  }
  return n;
}

int_4 vctx_field (int_4 vctx, int_4 field, int_4 *cb, int_8 xval) {
  int size=0;
  if ((vctx>>field)&0x1) switch (field) {
  case VCTX_BW: case VCTX_IF: case VCTX_RF: case VCTX_SR: cb[0]=(xval>>12); cb[1]=(xval<<20); size=2; break;
  case VCTX_FMT: cb[0]=xval; cb[1]=0; size=2; break;
  case VCTX_REFL: case VCTX_GAIN: cb[0]=(xval<<7); size=1; break;
  case VCTX_TEMP: cb[0]=(xval<<6); size=1; break;
  case VCTX_GPS: cb[0]=xval; size=1; break;
  }
  return size;
} 

int_4 vctx_field_offset (int_4 vctx, int_4 field) {
  int_4 i,n=0, cb[2];
  for (i=2; i<field; i++) n += vctx_field (vctx,i,cb,0);
  return n;
}

int_4 PKT_cblock (PICSTRUCT *p, PKTPlan *plan, int_4 *cb) {
  int_4 i,npost=0,nseq=0,ns,nt,ne,nc,nch,nchi,ipl,upl,fid,fb,vfb,tb,hdr;
  int_4 swaps=0,nss=8,n=nss,ncc=0,nca=0,nce=0,nj=0,tadd=0;
  int_4 ieth=nss*8,icpa=2*8,iscnt=3*8;		// manipulatable nibble offsets
  int_4 bps = PKT_getBPS((plan->form>>8)&0xFF);
  int_4 spa = PKT_getSPA((plan->form>>0)&0xFF);
  int_4 bits = (bps<0)? -bps*spa : bps*spa*8;
  int_4 bpa = bits/8;
  int_4 pty = plan->pkty;
  int_4 npkt =  (pty==PKT_ICE)? 64 : (pty==PKT_SDDS)? 56 : (pty>=PKT_VRT)? 32 : 0;
  int_4 ndata = plan->size/4;
  int_4 elem = plan->size/bpa;
  int_4 sinc = plan->muxm;					// sequence increment for mux mode
  int_4 indx = plan->indx;					// sequence start for mux mode
  real_8 dinc = (1.0/plan->rate)*1e12*(plan->size/bpa);
  int_8 tinc = (int_8)(dinc+0.001);				// in picoseconds per packet
  int_8 tmod = (1LL<<40) - 1e12;
  int_4 bige = (plan->flgs&PKTFLG_BIGE)? 1:0;
  int_8 psec = (int_8)plan->qsec*250 + indx*tinc;
  int_u4 psecu = (int_4)(psec>>32);
  int_u4 psecl = (int_4)(psec&0xFFFFFFFF);
  int_4 uctx = (sinc>1 && indx>0)? 0 : p->vctx;			// use vctx - no context on helper fibers
  int_4 vctx = (uctx&0xFFFFC), nctx=0;
  int_4 by3 = 0;
  if (spa>1) bits /= spa; 
  if (plan->pkty>=PKT_VRT) {
    if (plan->flgs&PKTFLG_NOSID) npkt -= 4;
    if (plan->flgs&PKTFLG_NOCID) npkt -= 8;
    if (plan->flgs&PKTFLG_TRIM) npkt -= 4;
  }
  npkt += plan->size; 
  if (fabs(dinc-tinc)>0.001) {
    if (fabs(dinc-tinc-0.333)<0.001) by3 = 1;
    else if (fabs(dinc-tinc-0.666)<0.001) by3 = 2;
    else printf("Non integral picoseconds/packet of %f on dev=%d type=%d index=%d\n",dinc,p->devno,p->ptype,p->pindex);
  }
  p->regX = tinc; tinc *= sinc;					// store for tcup and now adjust for mux mode
  v2print("CBlock ty=%d size=%d myip=%08x mcip=%08x port=%d rate=%d freq=%d chan=%x flgs=%08x\n",
  plan->pkty,plan->size,plan->myip,plan->mcip,plan->port,(int_4)plan->rate,(int_4)plan->freq,plan->chan,plan->flgs);


  // handle ethernet header
  if (plan->myip!=0) n=PKT_ipeth(plan,cb,n,npkt,0,PROTO_UDP);
  // handle packet header
  ns = n;
  if (plan->pkty==PKT_UDP || plan->pkty==PKT_ETH) {
    nc=n;
  }
  else if (plan->pkty==PKT_ICE) {
    cb[n++] = 0x00676665;	// sync
    cb[n++] = plan->indx; 	// sequence count
    cb[n++] = elem;	 	// elements per packet
    cb[n++] = plan->chan<<16; 	// user & channel
    cb[n++] = (bpa<<24)|(69<<16); // bpa & rep
    cb[n++] = (plan->form&0xFFFF);
    nt=n;
    for (i=0;i<10;i++) cb[n++]=0;
    nc=n;
    ncc=14;			// channel number byte offset
  }
  else if (plan->pkty==PKT_SDDS) {
    fid = (bits<=4)? 0x80 : (bits<=8)? 0x81 : (bits<=16)? 0x82 : 0x00;
    fb  = bits; if (spa==2) fb |= 0x80;
    if (bige && bits>=16) swaps = 0x1;
    cb[n++] = (plan->indx<<16)|(fb<<8)|fid;	// frame=0:bits:fid
    nt=n;
    cb[n++] = 0; 		// tcinfo|tccorr 
    cb[n++] = 0; 		// tctics
    cb[n++] = 0; 		// tctics
    cb[n++] = 0; 		// tcext
    for (i=0;i<8;i++) cb[n++]=0;
    cb[n++] = plan->chan<<16;	// pack SID in end of AAD for debug
    nc=n;
    ncc=54;
  }
  else if (plan->pkty>=PKT_VRT) {
    tb = 0x80; 			// link efficient integer data
    tb |= (spa==2)? 0x10:0x00;	// complex cart or real
    if (plan->pkty==PKT_VRTX||plan->pkty==PKT_VRTW||plan->pkty==PKT_VRTD) tb |= 0x40;	// BE data
    if (tb&0x40) swaps = (bits==16)? 1 : (bits==32)? 2 : (bits==12)? 4 : 0;	// enable swap 
    fb = (bits-1)&0x3F;
    vfb = (((spa==2)?0xA:0x8)<<28)|(fb<<6)|fb; // data format field for context packet
    hdr = 0x0|(plan->indx<<16);	// header with zero size and starting sequence count
    hdr |= 0x00600000; 		// has timecode TSI=UTC and TSF=pSec 
    cb[n++] = hdr; 		// placeholder type:fields:size
    if ((plan->flgs&PKTFLG_NOSID)==0) {
      hdr |= 0x10000000;	// has StreamID
      cb[n++] = plan->chan;	// stream ID
    }
    if ((plan->flgs&PKTFLG_NOCID)==0) {
      hdr |= 0x08000000;	// has ClassID
      cb[n++] = plan->ouid;	// org unique ID
      i = (plan->flgs&PKTFLG_NOSID)? plan->chan : 0;
      cb[n++] = (i<<16)|(tb<<8)|fb; // class ID
    }
    nt = n;	
    cb[n++] = plan->wsec; 	// time stamp int sec
    cb[n++] = psecu;		// time stamp frac sec upper
    cb[n++] = psecl; 		// time stamp frac sec lower
    if ((plan->flgs&PKTFLG_TRIM)==0) {
      hdr |= 0x04000000;	// has trailer
      cb[n++] = 0x0; 		// trailer
      npost += 1;
    }
    ncc=(plan->flgs&PKTFLG_NOSID)?10:4;
    cb[ns] = hdr + (n-ns+ndata);// backfill type:fields:size in VRT header word
    if (bige) for (i=ns; i<n; i++) cb[i] = swap(cb[i]);
    nc=n;			// end of non-context packet

    if (uctx&0x100000) tadd = (int_4)((tinc<<24)/1e11);
    if (uctx&0x200000) tadd = (int_4)((tinc<<24)/1e10);
    if (uctx&0x400000) tadd = (int_4)((tinc<<24)/1e09);
    cb[n++] = 0x0;		// context repeat 24b counter and by3 counter at nch-1
    nch=n;			// start of context packet fields
    cb[n++] = 0x41600006;	// VRT context header 6w with no fields total
    cb[n++] = plan->chan;	// stream ID 
    if (((uctx>>VCTX_NOCID)&1)==0) {
      cb[nch] = 0x49600008;	// VRT context header 6w with no fields total
      cb[n++] = plan->ouid;	// org unique ID
      cb[n++] = 0;		
    }
    nchi = n;
    cb[n++] = bitrev4(vctx);	// indicator field
    n += vctx_field (vctx,VCTX_BW,cb+n,plan->bw);
    n += vctx_field (vctx,VCTX_IF,cb+n,plan->freq);
    n += vctx_field (vctx,VCTX_RF,cb+n,plan->freq);
    n += vctx_field (vctx,VCTX_REFL,cb+n,0);
    n += vctx_field (vctx,VCTX_GAIN,cb+n,plan->gain);
    n += vctx_field (vctx,VCTX_SR,cb+n,plan->rate);
    n += vctx_field (vctx,VCTX_TEMP,cb+n,plan->temp);
    n += vctx_field (vctx,VCTX_FMT,cb+n,vfb);
    n += vctx_field (vctx,VCTX_GPS,cb+n,0);
    nctx = (n-nchi-1); cb[nch] += nctx;	// add # fields to complete context header size */
    if (bige) for (i=nch; i<n; i++) cb[i] = swap(cb[i]);
    nca=n; n=PKT_ipeth(plan,cb,n,(nchi-nch+4+nctx)*4,0,PROTO_UDP);	// swapped to keep header fields < 256 nibs
    nce=n; p->pktnctxoff = nch;	// store offset of context data
  }
  else {
    printf("Unsupported packet type=%d in PKT_cblock - using raw\n",plan->pkty);
    nc=n;
  }
  if (n>=256) printf("Err: NIO packet header words=%d >256 word limit.\n",n);
  ne = n++;							// add PKTBLK_SEQ slot at end of HDR block
  p->pktntoff = nt;

  for (i=nss; i<(nc-npost); i++) cb[n++] = PKTSEQ_CHDR | i;	// copy header words from address i

  cb[n++] = PKTSEQ_CDAT | (swaps<<12) | ndata;			// start copy of data words

  // tuner bank channel breakout
  if (plan->ncps>1) {
    if (plan->ncps%plan->ncpa!=0) printf("NIO NCHAN=%d must be multiple of NCPA=%d\n",plan->ncps,plan->ncpa);
   if (plan->ncpa>1) {
    n=PKT_nibadd(cb,n,(ns+1)*8,0x1L,8,0,bige); 			// sequence count increment
    i = 0x100 - plan->ncpa; 
    n=PKT_nibadd(cb,n,icpa,0x1L,2,0,0);				// NCPA increment
    n=PKT_nibadd(cb,n,icpa,i,2,2,0);				// NCPA counter wrap check
    cb[n++] = PKTSEQ_HSEI; nj=n;				// set enable to !carry
    cb[n++] = PKTSEQ_CJMP; cb[n++] = PKTSEQ_NOOP; 
    n=PKT_nibadd(cb,n,icpa,i,2,0,0);				// NCPA counter wrap reset
    i = (int_4)(B4G-plan->ncpa); 
    n=PKT_nibadd(cb,n,(ns+1)*8,i,8,0,bige); 			// sequence count reset
   }
    n=PKT_nibadd(cb,n,ieth+2*7 ,plan->nfps,2,0,1);		// ETH dest addr increment
    n=PKT_nibadd(cb,n,ieth+2*35,plan->nfps,2,0,1);	 	// IP dest addr increment
    n=PKT_nibadd(cb,n,ieth+2*26,0x10000L-plan->nfps,4,0,1); 	// IP checksum increment
    if (nj!=0) cb[nj] |= n;
    cb[n++] = PKTSEQ_NOOP; 
    n=PKT_nibadd(cb,n,ns*8+2*ncc,plan->nfps,3,0,bige); 		// channel number increment
    i = 0x1000 - plan->ncps;
    n=PKT_nibadd(cb,n,ns*8+ncc*2,i,3,2,bige);	 		// channel number wrap to carry
    cb[n++] = PKTSEQ_HSEI; nj=n;				// set enable to !carry
    cb[n++] = PKTSEQ_CJMP; cb[n++] = PKTSEQ_NOOP; 
    n=PKT_nibadd(cb,n,ns*8+ncc*2,i,3,0,bige);	 		// channel number wrap 
    i = 0x100 - plan->ncps/plan->ncpa;
    n=PKT_nibadd(cb,n,ieth+2*7 ,i,2,0,1);			// ETH dest addr increment
    n=PKT_nibadd(cb,n,ieth+2*35,i,2,0,1);		 	// IP dest addr increment
    i = plan->ncps/plan->ncpa;
    n=PKT_nibadd(cb,n,ieth+2*26,i,4,0,1);		 	// IP checksum increment
  }
  if (plan->ncpa>1) sinc = plan->ncpa;

  // handle packet header field updates
  if (plan->pkty==PKT_ICE) {
    n=PKT_nibadd(cb,n,iscnt,elem,8,0,bige);			// samples since start count
    n=PKT_nibadd(cb,n,(ns+1)*8,sinc,8,0,bige); 			// sequence count (32 bits)
    if (plan->flgs&PKTFLG_TC)
    n=PKT_nibadd(cb,n,(nt+1)*8,-elem,8,0,bige);			// offset whole samples decrement
  }
  else if (plan->pkty==PKT_SDDS) {
    n=PKT_nibadd(cb,n,ns*8+0x04,sinc,4,0,bige); 		// sequence count (16 bits)
  }
  else if (plan->pkty>=PKT_VRT) {
    n=PKT_nibadd(cb,n,ns*8+(bige?2:4),sinc,1,0,0); 		// sequence count (4 bits)
    if (by3>0) {
      n=PKT_nibadd(cb,n,nch*8-0x02,14,1,2,0);			// by3 counter overflow ?
      n=PKT_nibadd(cb,n,nt*8+0x08,by3,10,1,bige);		// add by3 count if carry
      n=PKT_nibadd(cb,n,nch*8-0x02,14,1,2,0);			// by3 counter overflow ?
      n=PKT_nibadd(cb,n,nch*8-0x02,13,1,1,0);			// by3 counter overflow ? reset
      n=PKT_nibadd(cb,n,nch*8-0x02,1,1,0,0);			// by3 counter count
    }
    n=PKT_nibadd(cb,n,nt*8+0x08,tinc,10,0,bige);		// fractional seconds increment (40 bits) - with carry
    n=PKT_nibadd(cb,n,nt*8+0x08,tmod,10,2,bige);		// check for 1 sec overflow
    n=PKT_nibadd(cb,n,nt*8+0x08,tmod,10,1,bige);		// add the wrap amount if carry
    if (uctx&2) cb[n++] = PKTSEQ_CALT;				// set alternate to carry if OnSec bit
    n=PKT_nibadd(cb,n,nt*8+0x00,0x1L,8,1,bige); 		// increment integer seconds if carry
    if (tadd>0) {
      n=PKT_nibadd(cb,n,nch*8-0x08,tadd,6,0,0);			// packet count for context repeat modes
      cb[n++] = PKTSEQ_CALT;					// set alternate to carry if xSec bits
    }
  }
  if (nj!=0) cb[nj] |= n;					// add jump address to here
  cb[n++] = PKTSEQ_CLR|PKTSEQ_WAIT;				// unset Enable / clear carry / wait for data
  cb[n++] = PKTSEQ_WAIT;					// wait for data
  if (npost>0) cb[n++] = PKTSEQ_CHDR | (nc-npost);		// copy trailer
  cb[n++] = PKTSEQ_DONE;					// finish sequence
  cb[n++] = 0;
 
  // finalize structure
  cb[0]=((ne+1)<<16)|0x0001;					// starting execute sequence address
  cb[1]=PKTBLK_HDR|ne;
  for (i=2; i<nss; i++) cb[i]=0;				// prep timecode registers

  cb[ne]=PKTBLK_SEQ|nseq;
  
  // handle context packets - ns=StartOfDataPacket nch=startOfContextPacket
  if (uctx>0) {
    cb[0] |= (n<<24);						// alternate starting sequence addr for context packet
    if (uctx&1) cb[0] |= 0x20;					// kick off context packet at startup
    cb[n++] = PKTSEQ_CLR;					// unset Enable / clear carry 
    cb[n++] = PKTSEQ_CALT;					// set carry to alternate flag
    for (i=nca; i<nce; i++) cb[n++] = PKTSEQ_CHDR | i;		// copy ethernet+UDP header
    for (i=nch; i<nchi; i++) cb[n++] = PKTSEQ_CHDR | i;		// context header words
    for (i=0; i<=2; i++) cb[n++] = PKTSEQ_CHDR | (nt+i);	// TS from data packet
    cb[n++] = PKTSEQ_CHDR | (nchi);				// indicator field
    for (i=0; i<nctx; i++) cb[n++] = PKTSEQ_CHDR | (nchi+1+i);	// data fields
    n=PKT_nibadd(cb,n,nch*8+(bige?2:4),0x1L,1,0,0); 		// sequence count (4 bits)
    cb[n++] = PKTSEQ_DONE;					// finish sequence
    cb[n++] = 0;
  }
  if (plan->flgs&PKTFLG_DALT) cb[0] |= 0x40;

  nseq = n-ne;
  if (nseq>ndata) printf("Too many seq steps=%d to complete in data cycles=%d.\n",nseq,ndata);
  if (n>256) printf("Too many seq slots=%d to complete in blockram buffer=%d.\n",n,256);
  /* printf("CBlock nc=%d nt=%d ne=%d nseq=%d\n",nc,nt,ne,nseq); */

  return n;
}
