/****************************************************************************/
/*                                                                          */
/*  part_si5338.c                                                           */
/*                                                                          */
/****************************************************************************/

#include "part_si5338.h" 

/*                                                           */
/* Sane defaults for SI5338 registers based on Module type   */
/*                                                           */
int32_t pt_si5338_rgc_default(PT_SI5338_REG *R, int32_t modid)
{
  PT_SI5338_REG *rgdflt = (PT_SI5338_REG *)NULL;
  int32_t numreg = 0;
  int32_t i;

  switch (modid) {

    case (FPGA_MODID_LB2DM3):
      numreg = sizeof(si5338_lb2dm3_reg_defaults)/sizeof(PT_SI5338_REG);
      rgdflt = (PT_SI5338_REG *) (&si5338_lb2dm3_reg_defaults);
      break;

    case (FPGA_MODID_D2RF):  
      numreg = sizeof(si5338_d2rf_reg_defaults)/sizeof(PT_SI5338_REG);
      rgdflt = (PT_SI5338_REG *) (&si5338_lb2dm3_reg_defaults);
      break;

    case (FPGA_MODID_D2AWGM3):  
      numreg = sizeof(si5338_d2awgm3_reg_defaults)/sizeof(PT_SI5338_REG);
      rgdflt = (PT_SI5338_REG *) (&si5338_d2awgm3_reg_defaults);
      break;

    default:
      dbgerrormsg("Unknown module type: pt_si5338_rgc_default \n");
      return (-1);
  }

  if (numreg > SI5338_MAX_CFG_RG) {
    printf("ERROR: numreg > SI5338_MAX_CFG_RG !\n");
    return (-1);
  }
  /* First, default all registers to page zero setting */
  for (i=0; i<SI5338_MAX_CFG_RG; i++){
    R[i].ra = 255;           
    R[i].rv = 0x00;         
    R[i].rm = 0xFF;           
  }
  /* Next, load default register settings */
  if (numreg > SI5338_MAX_CFG_RG) {
    dbgwarnmsg("Si5338 initialization register count too large. Registers ignored!\n");
    numreg = SI5338_MAX_CFG_RG;
  }
  for (i=0; i<numreg; i++){
    R[i].ra = rgdflt[i].ra;
    R[i].rv = rgdflt[i].rv;
    R[i].rm = rgdflt[i].rm;
  }

  return (0);
}


int32_t pt_si5338_rgc_setbits(PT_SI5338_REG *R, uint8_t rgaddr, uint8_t rgmask,
                                uint8_t rgval, uint8_t pgnum)
{
  uint8_t pageref = 0x00;
  int32_t i;
  for (i=0; i<SI5338_MAX_CFG_RG; i++) {
      if (R[i].ra == 255) pageref = (0x01)&(R[i].rv);
      if ((R[i].ra == rgaddr) && (pgnum == pageref)) {
        R[i].rv = u8_setbits(R[i].rv, rgmask, rgval);
      }
  }
  return (0);
}

int32_t pt_si5338_msx_soln(PT_SI5338_REG *R, PT_SI5338_MSCFG *M)
{
  static const float64_t REF_FRQ_MIN =    5.0;
  static const float64_t REF_FRQ_MAX =  710.0;
  static const float64_t PFD_FRQ_MIN =    5.0;
  static const float64_t PFD_FRQ_MAX =   40.0;
  static const float64_t MSX_FRQ_MIN =    5.0;    
  static const float64_t MSX_FRQ_MAX =  355.0;
  static const float64_t VCO_FRQ_MIN = 2200.0;
  static const float64_t VCO_FRQ_MAX = 2840.0;
  static const float64_t p1p2div_tbl[6] = {1.0, 2.0, 4.0, 8.0, 16.0, 32.0};
  static const float64_t rdiv_msx_tbl[6] = {1.0, 2.0, 4.0, 8.0, 16.0, 32.0};
  float64_t vco_valid_freqs[SI5338_VCO_CHK_FREQ];     
  int32_t i,ii,itest;
  float64_t ftest,errmin,errtot,wght;

  /* Initialize input structure values assuming failure */

  /* Check input reference bounds */
  if ((M->inp_ref_frq < REF_FRQ_MIN) || (M->inp_ref_frq > REF_FRQ_MAX)) {
    dbgerrormsg("Invalid input reference frequency to Si5338.\n");
    return (-1);
  }

  /* Find lowest P1 divider that keeps PFD frequency in range     */
  /*   NOTE: Prefer a divide by two for 50% duty cycle comparison */
  /*   NOTE: Highest possible frequency produces lowest jitter    */
  M->p1dividx = 0;
  for (i=0; i<6; i++) {
    ftest = (M->inp_ref_frq)/(p1p2div_tbl[i]);
    if ((ftest >= PFD_FRQ_MIN ) && (ftest <= PFD_FRQ_MAX)) M->p1dividx = i;
  }
  ftest = (M->inp_ref_frq)/((float64_t) p1p2div_tbl[M->p1dividx]);
  if ((ftest < PFD_FRQ_MIN ) || (ftest > PFD_FRQ_MAX)) {
    dbgerrormsg("Invalid P1 divider frequency to Si5338.\n");
    return (-2);
  }
  M->p2dividx = M->p1dividx;
  M->pfd_frq = (M->inp_ref_frq)/(p1p2div_tbl[M->p1dividx]);

  /* Now, determine valid VCO frequencies as multiple of pfd_frq */
  for (i=0; i<SI5338_VCO_CHK_FREQ; i++) {
    vco_valid_freqs[i] = -1.0;
  }
  ii = 0;
  for (i=0; i<1000; i++) {
    ftest = ((float64_t) i)*(M->pfd_frq);
    if ((ftest >= VCO_FRQ_MIN ) && (ftest <= VCO_FRQ_MAX)) {
      if (ii < SI5338_VCO_CHK_FREQ) {
        vco_valid_freqs[ii] = ftest; 
      }
      ii++;
    }
  }

  /* Loop over all output frequencies and determine R-Dividers */
  for (i=0; i<SI5338_MSX_NUM; i++) {
    M->ms_rdividx[i] = 0;
    if (M->clk_frq_des[i] < MSX_FRQ_MIN) {
      M->ms_rdividx[i] = 5;
      for (ii=5; ii>=0; ii--) {
        ftest = (rdiv_msx_tbl[ii])*(M->clk_frq_des[i]);
        if ((ftest >= MSX_FRQ_MIN) && (ftest <= MSX_FRQ_MAX)) 
          M->ms_rdividx[i] = ii;
      }
    }
    M->ms_frq_des[i] = (rdiv_msx_tbl[M->ms_rdividx[i]])*(M->clk_frq_des[i]);
  }

  /* Compute total error for each MSx given each possible VCO frequency */
  errmin = 10000000000.0;
  M->vco_frq = VCO_FRQ_MIN;
  for (i=0; i<SI5338_VCO_CHK_FREQ; i++) {
    errtot = errmin;
    if (vco_valid_freqs[i] > 0.0) { 
      errtot = 0.0;
      for (ii=0; ii<SI5338_MSX_NUM; ii++) {
        wght = 1.0;
        wght  = (ii == M->ms_critical) ? (100.0) : (1.0);
        ftest = (vco_valid_freqs[i])/(M->ms_frq_des[ii]);
        errtot += (wght*(ftest - ((int32_t) ftest)));
      }
      dbgprint(M->dbglvl,DBG_LB,"MSX: %d VCO: %9.2f ERRTOT: %21.15f \n",
        ii,vco_valid_freqs[i],errtot);
    }
    if (errtot < errmin) {
      errmin = errtot;
      M->vco_frq = vco_valid_freqs[i];
    }
  }

  /* Calculate MSx = a + [b/c] multiplier to PFD reference */
  /*   a = {4,6,8,9,10,11,...,566}                         */
  /*   b = {0,1,...,(2^30 - 1)}                            */
  /*   c = {1,2,...,(2^30 - 1)}                            */
  /*                                                       */
  for (i=0; i<SI5338_MSX_NUM; i++) {
    ftest = (M->vco_frq) / (M->ms_frq_des[i]);
    itest = (int32_t) ftest;
    if ((itest < 4) || (itest == 5) || (itest == 7) || (itest > 566)) {
      dbgerrormsg("Invalid MSx_a coefficient. \n");
      return (-3);
    }
    M->ms_a[i] = (itest > 0) ? ((uint32_t) itest) : (0);            
    ftest = ftest - ((float64_t) itest);
    ftest = (ftest > 0.0) ? (ftest) : (0.0);
    ftest = (ftest < 1.0) ? (ftest) : (1.0);
    ftest = ftest*((float64_t) SI5338_MSX_FRACTION);
    itest = (int32_t) (ftest + 0.5);
    M->ms_b[i] = (itest > 0) ? ((uint32_t) itest) : (0);
    if (M->ms_b[i] == 0) 
      M->ms_c[i] = 1;
    else
      M->ms_c[i] = (uint32_t) SI5338_MSX_FRACTION;

    /* Update the actual MSx frequency for precision apps */
    ftest = ((float64_t) M->ms_a[i]) + 
              ((float64_t) M->ms_b[i])/((float64_t) M->ms_c[i]);
    ftest = (ftest > 0.0) ? (ftest) : (1.0);
    M->ms_frq_act[i] = (M->vco_frq)/ftest;
    M->clk_frq_act[i] = (M->ms_frq_act[i])/(rdiv_msx_tbl[M->ms_rdividx[i]]);

    /* Finally, create the configuration register values */
    ftest = ((float64_t) M->ms_a[i])*((float64_t) M->ms_c[i]) + 
              ((float64_t) M->ms_b[i]);
    ftest = (128.0*ftest)/((float64_t) M->ms_c[i]);
    ftest = ftest - 512.0;
    ftest = (ftest > 0.0) ? (ftest) : (0.0);
    M->ms_p1[i] = (ftest > 0.0) ? ((uint32_t) ftest) : (0);
    M->ms_p2[i] = (128*(M->ms_b[i]))%(M->ms_c[i]);
    M->ms_p3[i] = M->ms_c[i];
  }
 
  /* Configure the PLL */
  M->k     = 925.0;
  M->rsel  = 0x00;
  M->bwsel = 0x00;
  if (M->pfd_frq < 15.0) {
    M->k     = 325.0;
    M->rsel  = 0x01;
    M->bwsel = 0x01;
  }
  if (M->pfd_frq < 8.0) {
    M->k     = 185.0;
    M->rsel  = 0x03;
    M->bwsel = 0x02;
  }
  M->q = 3;
  M->vco_gain = 0x00;
  if (M->vco_frq < 2425.0) {
    M->q = 4;
    M->vco_gain = 0x01;
  }
  ftest = 2500.0/(M->vco_frq);
  ftest = ((M->k)/(533.0*M->q))*((M->vco_frq)/(M->pfd_frq))*(ftest*ftest*ftest);
  ftest = 0.5+ftest;
  ftest = (ftest < 0.0) ? (0.0) : (ftest);
  ftest = (ftest > 127.0) ? (127.0) : (ftest);
  M->pll_kphi = (uint8_t) ftest;
  ftest = ((-6.67)*(M->vco_frq/1000.0))+20.67;
  ftest = 0.5+ftest;
  ftest = (ftest < 0.0) ? (0.0) : (ftest);
  ftest = (ftest > 63.0) ? (63.0) : (ftest);
  M->mscal  = (uint8_t) ftest;
  M->ms_pec = 0x07;

  /* Update the input divider fields */
  pt_si5338_rgc_setbits(R,29,0x07,((0x07)&(M->p1dividx)),0);
  pt_si5338_rgc_setbits(R,30,0x07,((0x07)&(M->p2dividx)),0);

  /* Update the MSx register fields */
  pt_si5338_rgc_setbits(R,53,0xFF,((0xFF)&(M->ms_p1[0]))      ,0);
  pt_si5338_rgc_setbits(R,54,0xFF,((0xFF)&(M->ms_p1[0] >> 8)) ,0);
  pt_si5338_rgc_setbits(R,55,0x03,((0x03)&(M->ms_p1[0] >> 16)),0);
  pt_si5338_rgc_setbits(R,55,0xFC,((0x3F)&(M->ms_p2[0]))      ,0);
  pt_si5338_rgc_setbits(R,56,0xFF,((0xFF)&(M->ms_p2[0] >>  6)),0);
  pt_si5338_rgc_setbits(R,57,0xFF,((0xFF)&(M->ms_p2[0] >> 14)),0);
  pt_si5338_rgc_setbits(R,58,0xFF,((0xFF)&(M->ms_p2[0] >> 22)),0);
  pt_si5338_rgc_setbits(R,59,0xFF,((0xFF)&(M->ms_p3[0]))      ,0);
  pt_si5338_rgc_setbits(R,60,0xFF,((0xFF)&(M->ms_p3[0] >>  8)),0);
  pt_si5338_rgc_setbits(R,61,0xFF,((0xFF)&(M->ms_p3[0] >> 16)),0);
  pt_si5338_rgc_setbits(R,62,0x3F,((0x3F)&(M->ms_p3[0] >> 24)),0);

  pt_si5338_rgc_setbits(R,64,0xFF,((0xFF)&(M->ms_p1[1]))      ,0);
  pt_si5338_rgc_setbits(R,65,0xFF,((0xFF)&(M->ms_p1[1] >> 8)) ,0);
  pt_si5338_rgc_setbits(R,66,0x03,((0x03)&(M->ms_p1[1] >> 16)),0);
  pt_si5338_rgc_setbits(R,66,0xFC,((0x3F)&(M->ms_p2[1]))      ,0);
  pt_si5338_rgc_setbits(R,67,0xFF,((0xFF)&(M->ms_p2[1] >>  6)),0);
  pt_si5338_rgc_setbits(R,68,0xFF,((0xFF)&(M->ms_p2[1] >> 14)),0);
  pt_si5338_rgc_setbits(R,69,0xFF,((0xFF)&(M->ms_p2[1] >> 22)),0);
  pt_si5338_rgc_setbits(R,70,0xFF,((0xFF)&(M->ms_p3[1]))      ,0);
  pt_si5338_rgc_setbits(R,71,0xFF,((0xFF)&(M->ms_p3[1] >>  8)),0);
  pt_si5338_rgc_setbits(R,72,0xFF,((0xFF)&(M->ms_p3[1] >> 16)),0);
  pt_si5338_rgc_setbits(R,73,0x3F,((0x3F)&(M->ms_p3[1] >> 24)),0);

  pt_si5338_rgc_setbits(R,75,0xFF,((0xFF)&(M->ms_p1[2]))      ,0);
  pt_si5338_rgc_setbits(R,76,0xFF,((0xFF)&(M->ms_p1[2] >> 8)) ,0);
  pt_si5338_rgc_setbits(R,77,0x03,((0x03)&(M->ms_p1[2] >> 16)),0);
  pt_si5338_rgc_setbits(R,77,0xFC,((0x3F)&(M->ms_p2[2]))      ,0);
  pt_si5338_rgc_setbits(R,78,0xFF,((0xFF)&(M->ms_p2[2] >>  6)),0);
  pt_si5338_rgc_setbits(R,79,0xFF,((0xFF)&(M->ms_p2[2] >> 14)),0);
  pt_si5338_rgc_setbits(R,80,0xFF,((0xFF)&(M->ms_p2[2] >> 22)),0);
  pt_si5338_rgc_setbits(R,81,0xFF,((0xFF)&(M->ms_p3[2]))      ,0);
  pt_si5338_rgc_setbits(R,82,0xFF,((0xFF)&(M->ms_p3[2] >>  8)),0);
  pt_si5338_rgc_setbits(R,83,0xFF,((0xFF)&(M->ms_p3[2] >> 16)),0);
  pt_si5338_rgc_setbits(R,84,0x3F,((0x3F)&(M->ms_p3[2] >> 24)),0);

  pt_si5338_rgc_setbits(R,86,0xFF,((0xFF)&(M->ms_p1[3]))      ,0);
  pt_si5338_rgc_setbits(R,87,0xFF,((0xFF)&(M->ms_p1[3] >> 8)) ,0);
  pt_si5338_rgc_setbits(R,88,0x03,((0x03)&(M->ms_p1[3] >> 16)),0);
  pt_si5338_rgc_setbits(R,88,0xFC,((0x3F)&(M->ms_p2[3]))      ,0);
  pt_si5338_rgc_setbits(R,89,0xFF,((0xFF)&(M->ms_p2[3] >>  6)),0);
  pt_si5338_rgc_setbits(R,90,0xFF,((0xFF)&(M->ms_p2[3] >> 14)),0);
  pt_si5338_rgc_setbits(R,91,0xFF,((0xFF)&(M->ms_p2[3] >> 22)),0);
  pt_si5338_rgc_setbits(R,92,0xFF,((0xFF)&(M->ms_p3[3]))      ,0);
  pt_si5338_rgc_setbits(R,93,0xFF,((0xFF)&(M->ms_p3[3] >>  8)),0);
  pt_si5338_rgc_setbits(R,94,0xFF,((0xFF)&(M->ms_p3[3] >> 16)),0);
  pt_si5338_rgc_setbits(R,95,0x3F,((0x3F)&(M->ms_p3[3] >> 24)),0);

  pt_si5338_rgc_setbits(R, 97,0xFF,((0xFF)&(M->ms_p1[4]))      ,0);
  pt_si5338_rgc_setbits(R, 98,0xFF,((0xFF)&(M->ms_p1[4] >> 8)) ,0);
  pt_si5338_rgc_setbits(R, 99,0x03,((0x03)&(M->ms_p1[4] >> 16)),0);
  pt_si5338_rgc_setbits(R, 99,0xFC,((0x3F)&(M->ms_p2[4]))      ,0);
  pt_si5338_rgc_setbits(R,100,0xFF,((0xFF)&(M->ms_p2[4] >>  6)),0);
  pt_si5338_rgc_setbits(R,101,0xFF,((0xFF)&(M->ms_p2[4] >> 14)),0);
  pt_si5338_rgc_setbits(R,102,0xFF,((0xFF)&(M->ms_p2[4] >> 22)),0);
  pt_si5338_rgc_setbits(R,103,0xFF,((0xFF)&(M->ms_p3[4]))      ,0);
  pt_si5338_rgc_setbits(R,104,0xFF,((0xFF)&(M->ms_p3[4] >>  8)),0);
  pt_si5338_rgc_setbits(R,105,0xFF,((0xFF)&(M->ms_p3[4] >> 16)),0);
  pt_si5338_rgc_setbits(R,106,0x3F,((0x3F)&(M->ms_p3[4] >> 24)),0);

  /* Configure output clock dividers */
  pt_si5338_rgc_setbits(R,31,0x1C,((0x03)&(M->ms_rdividx[0])),0);
  pt_si5338_rgc_setbits(R,32,0x1C,((0x03)&(M->ms_rdividx[1])),0);
  pt_si5338_rgc_setbits(R,33,0x1C,((0x03)&(M->ms_rdividx[2])),0);
  pt_si5338_rgc_setbits(R,34,0x1C,((0x03)&(M->ms_rdividx[3])),0);

  /* Configure the PLL */
  pt_si5338_rgc_setbits(R,48,0x3F,((0x3F)&(M->pll_kphi)),0);
  pt_si5338_rgc_setbits(R,49,0x03,((0x03)&(M->bwsel)),0);    
  pt_si5338_rgc_setbits(R,49,0x0C,((0x03)&(M->rsel)),0);    
  pt_si5338_rgc_setbits(R,49,0x70,((0x07)&(M->vco_gain)),0);
  pt_si5338_rgc_setbits(R,50,0x3F,((0x3F)&(M->mscal)),0);
  pt_si5338_rgc_setbits(R,51,0x07,((0x07)&(M->ms_pec)),0);

#ifdef BOGUS
  /* "Miscellaneous register writes to ensure proper device function" */
  pt_si5338_rgc_setbits(R,47,0xFC,0x05,0);
  
#endif


  /* Debug output */
  dbgprintmsg(M->dbglvl,DBG_LB,"*** Si5338 Setup ***\n");
  dbgprint(M->dbglvl,DBG_LB,"dbglvl:     0x%8.8X \n",M->dbglvl);
  dbgprint(M->dbglvl,DBG_LB,"ms_critical %1.1i \n",M->ms_critical);
  dbgprint(M->dbglvl,DBG_LB,"ref_frq: %21.15f \n",M->inp_ref_frq);
  dbgprint(M->dbglvl,DBG_LB,"pfd_frq: %21.15f \n",M->pfd_frq);     
  dbgprint(M->dbglvl,DBG_LB,"vco_frq: %21.15f \n",M->vco_frq);
  dbgprint(M->dbglvl,DBG_LB,"p1dividx:   0x%2.2X \n",M->p1dividx);  
  dbgprint(M->dbglvl,DBG_LB,"p2dividx:   0x%2.2X \n",M->p2dividx);  
  dbgprint(M->dbglvl,DBG_LB,"k:       %21.15f \n",M->k);
  dbgprint(M->dbglvl,DBG_LB,"q:       %21.15f \n",M->q);
  dbgprint(M->dbglvl,DBG_LB,"pll_kphi:   0x%2.2X \n",M->pll_kphi);  
  dbgprint(M->dbglvl,DBG_LB,"rsel:       0x%2.2X \n",M->rsel);      
  dbgprint(M->dbglvl,DBG_LB,"bwsel:      0x%2.2X \n",M->bwsel);     
  dbgprint(M->dbglvl,DBG_LB,"vco_gain:   0x%2.2X \n",M->vco_gain);  
  dbgprint(M->dbglvl,DBG_LB,"mscal:      0x%2.2X \n",M->mscal);     
  dbgprint(M->dbglvl,DBG_LB,"ms_pec:     0x%2.2X \n",M->ms_pec);    

  for (i=0; i<SI5338_MSX_NUM; i++) {
    dbgprint(M->dbglvl,DBG_LB,"...MultiSynth%1.1i \n",i);
    dbgprint(M->dbglvl,DBG_LB,"     clk_frq_des[%1.1i]: %21.15f\n",i,M->clk_frq_des[i]);
    dbgprint(M->dbglvl,DBG_LB,"     clk_frq_act[%1.1i]: %21.15f\n",i,M->clk_frq_act[i]);
    dbgprint(M->dbglvl,DBG_LB,"     ms_frq_des[%1.1i]:  %21.15f\n",i,M->ms_frq_des[i]);
    dbgprint(M->dbglvl,DBG_LB,"     ms_frq_act[%1.1i]:  %21.15f\n",i,M->ms_frq_act[i]);
    dbgprint(M->dbglvl,DBG_LB,"     ms_rdividx[%1.1i]:  %10u \n",i,M->ms_rdividx[i]);
    dbgprint(M->dbglvl,DBG_LB,"     ms_a[%1.1i]:   %10u  0x%8.8X \n",i,M->ms_a[i],M->ms_a[i]);
    dbgprint(M->dbglvl,DBG_LB,"     ms_b[%1.1i]:   %10u  0x%8.8X \n",i,M->ms_b[i],M->ms_b[i]);
    dbgprint(M->dbglvl,DBG_LB,"     ms_c[%1.1i]:   %10u  0x%8.8X \n",i,M->ms_c[i],M->ms_c[i]);
    dbgprint(M->dbglvl,DBG_LB,"     ms_p1[%1.1i]:  %10u  0x%8.8X \n",i,M->ms_p1[i],M->ms_p1[i]);
    dbgprint(M->dbglvl,DBG_LB,"     ms_p2[%1.1i]:  %10u  0x%8.8X \n",i,M->ms_p2[i],M->ms_p2[i]);
    dbgprint(M->dbglvl,DBG_LB,"     ms_p3[%1.1i]:  %10u  0x%8.8X \n",i,M->ms_p3[i],M->ms_p3[i]);
  }

  return (0);
}


/*********************************************************************/
#ifdef _BUILD_UNIT_TEST

int main(void)
{
  int32_t rtn = 0;
  PT_SI5338_REG   S[SI5338_MAX_CFG_RG];
  PT_SI5338_MSCFG M;

  pt_si5338_rgc_default(&S[0], FPGA_MODID_LB2DM3, 0);

  /* Configure a divider solution for the MSx outputs */
  M.dbglvl      = DBG_LB;
  M.ms_critical = 3;                  /* MS3 used for zerodly  */
  M.inp_ref_frq    = 10.0;            /* 10MHz ref clock       */
  M.clk_frq_des[0] = 30.0;            /* STV6111 reference     */
  M.clk_frq_des[1] = M.inp_ref_frq;   /* Unused output         */
  M.clk_frq_des[2] = 125.0;                /* ADC clock             */
  M.clk_frq_des[3] = M.inp_ref_frq;   /* Zero delay mode clock */
  M.clk_frq_des[4] = M.inp_ref_frq;   /* Unused output         */
  rtn = pt_si5338_msx_soln(&S[0], &M);
  if (rtn<0) {
    dbgerror("Invalid clock configuration specified. Bad... \n");
    return(-2);
  }

  return (rtn);
}

#endif
/*********************************************************************/
