// !/bin/bash
// 
//
// eeprom code
// 9-24-2012 sjn changed frequency for preselector switch from 1500 MHz to 1300 MHz
//
//
//


#define CLK_TIMEOUT 100

void i2c_start (PICSTRUCT *p, int_4 mport) {
	rfxd_mwr(p, mport, FEMTO_SPI, FEMTO25_PRG_I2C);
	rfxd_mwr(p, mport, FEMTO_SPI, 0);
}

// FIX check this code
int_4 i2c_wait_for_ready (PICSTRUCT *p, int_4 mport) {
	int_4 count;
	for (count = CLK_TIMEOUT; count !=0; count--)  {
		if((rfxd_mrd(p, mport, FEMTO_SPI) & FEMTO_SPI_STATUS) == 0) {
			break;
		}
	}
	if (count == 0) {
		v3print("timeout on wait for ready\n");
	}
	return count;
}


// FIX check this code
int_4 i2c_wait_for_done (PICSTRUCT *p, int_4 mport) {
	int_4 count;
	for (count = CLK_TIMEOUT; count !=0; count--)  {
		if((rfxd_mrd(p, mport, FEMTO_SPI) & FEMTO_SPI_STATUS) == 0) {
			break;
		}
	}
	if (count == 0) {
		v3print("timeout on wait for ready\n");
	}
	return count;
}


// currently the write must be 3 bytes
// device address, memory address, data

void i2c_wr (PICSTRUCT *p, int_4 mport, int_4 slave, int_4 add, int_4 dat) {
	rfxd_mwr(p, mport, FEMTO_SPI1, slave);
	rfxd_mwr(p, mport, FEMTO_SPI2, add);
	if (slave == 0x30) {
		// temp sensor 
		rfxd_mwr(p, mport, FEMTO_SPI3, (dat >> 8) & 0xff);
		rfxd_mwr(p, mport, FEMTO_SPI4,  dat       & 0xff);
	} else if (slave == 0xa0) {
		// EEPROM
		rfxd_mwr(p, mport, FEMTO_SPI3, dat);
	}
	// wait for I2C ready
	i2c_wait_for_ready(p, mport);
	i2c_start(p, mport);
}

int_4 i2c_rd (PICSTRUCT *p, int_4 mport, int_4 slave, int_4 add) {
	int_4 dat;
	// wait for I2C ready
	i2c_wait_for_ready(p, mport);
	rfxd_mwr(p, mport, FEMTO_SPI1, slave);
	rfxd_mwr(p, mport, FEMTO_SPI2, add);
	i2c_start(p, mport);
	// wait for done
	i2c_wait_for_done(p, mport);
	if (slave == 0x30) {
		dat = (rfxd_mrd(p, mport, FEMTO25_I2C_HI) << 8) | rfxd_mrd(p, mport, FEMTO25_I2C_LO);
	} else {
		dat = rfxd_mrd(p, mport, FEMTO25_I2C_LO);
	}
	return dat;
}

//
// Temperature sensor/ EEPROM code
//
// temperature sensor is 16 bit registers

int_4 temp_rd (PICSTRUCT *p, int_4 mport, int_4 address) {
	int_4 val;
	val = i2c_rd(p, mport, 0x31, address);
	return val;
}

void temp_wr (PICSTRUCT *p, int_4 mport, int_4 address, int_4 data) {
	i2c_wr(p, mport, 0x30, address, data);
}

// dump the temp sensor registers
void temp_dump (PICSTRUCT *p, int_4 mport) {
	int i, t;
	for (i = 0; i < 8; i++) {
		t = temp_rd(p, mport, i);
		v3print("temp reg: %d val: %x\n", i, t);
	}
}

// return the temperature in degrees
int_4 temp_temp (PICSTRUCT *p, int_4 mport) {
	int_4 t;
	
	t = temp_rd(p, mport, 0x05);
	if (t & 0x1000) t |= 0xe000; // sign extend if needed
	t = t*.065;
	v3print("temperature: %d\n",t);
	return t;
}

void ee_read (PICSTRUCT *p, int_4 mport, int_4 address, unsigned char * data, int_4 count) {
	int_4 i;
	
	for (i = 0; i < count; i++) {
		 data[i] = i2c_rd(p, mport, 0xa0, address++);
	}
}

void ee_write (PICSTRUCT *p, int_4 mport, int_4 address, unsigned char * data, int_4 count) {
	int_4 i;	
	for (i = 0; i < count; i++) {
		i2c_wr(p, mport, 0xa0, address++, *data++);
	}
}

//software write protection allows protecting serial number, model number etc.

void ee_protect (PICSTRUCT *p, int_4 mport, int_4 val) {
	v3print("ee protect not implemented yet\n");
}


//
//Synth code
//

// FIX - need to figure out how to store these!!
// init values for registers at 4000 MHz;
//int_4 SYN_REG_DEF[6] = { 0x00640000, 0x08008641, 0x02004ec2, 0x000004b3,0x008a003c, 0x00580005 };
int_4 SYN_REG_DEF[6] = { 0x00640000, 0x08008641, 0x020042c2, 0x000004b3,0x008a003c, 0x00580005 };
int_4 SYN_REG[6];

//  phase detector frequency;
#define SYNTH_REF 20.
#define BANDS 5
float OUTSEL_LF[BANDS]   = { 2200.0, 1100.0,  550.0, 275.0, 137.5};
float OUTSEL_UF[BANDS]   = { 4401.0, 2200.0, 1100.0, 550.0, 275.0};
int_4  OUTSEL_DIV[BANDS]  = {      1,      2,      4,     8,   16};
int_4  OUTSEL_DIVB[BANDS] = {      0,      1,      2,     3,    4};
#define SYN25MAXFREQ 2200.0
#define SYN25MINFREQ 68.75

void syn25_start_spi (PICSTRUCT *p, int_4 mport) {
// 	puts "start_synth_spi addr:FEMTO_SPI data:FEMTO_PRG_SYN ";
	rfxd_mwr(p, mport, FEMTO_SPI, FEMTO_PRG_SYN);
	rfxd_mwr(p, mport, FEMTO_SPI, 0);
}

int_4 syn25_wait_for_ready(PICSTRUCT *p, int_4 mport){
	int_4 count;
	for (count = 100; count != 0; count--) {
		if ((rfxd_mrd(p, mport, FEMTO_SPI) & 0x80) == 0) {
			break;
		}
	}
	if (count == 0) {
		v3print("Timeout on syn25 wait for busy\n");
	}
	return count;
}

void syn25wr(PICSTRUCT *p, int_4 mport, int_4 address, int_4 data) {
	v3print("syn25wr - addr:%d data: %08x\n", address, data);
	data = data | address;
	syn25_wait_for_ready(p, mport);
	rfxd_mwr(p, mport, FEMTO_SPI4, (data >> 24) & 0xff);
	rfxd_mwr(p, mport, FEMTO_SPI3, (data >> 16) & 0xff);
	rfxd_mwr(p, mport, FEMTO_SPI2, (data >>  8) & 0xff);
	rfxd_mwr(p, mport, FEMTO_SPI1,  data        & 0xff);
	syn25_start_spi(p, mport);
}

//     ---------------------------------------------------------;
//     This program computes the GCD of two positive integers;
//     using the Euclid method.  Given a and b, a >= b, the;
//     Euclid method goes as follows:  (1) dividing a by b yields;
//     a reminder c; (2) if c is zero, b is the GCD; (3) if c is;
//     no zero, b becomes a and c becomes c and go back to;
//     Step (1).  This voidess will continue until c is zero.;
//     ---------------------------------------------------------;

int_4 calcGCF (int_4 a, int_4 b) {
	int_4 c;
	if (a < b) {
		c = a;
		a = b;
		b = c;
	}

	while (a >= 0) {
		c = a % b;
		//v3print("a: %d b: %d c: %d\n", a,b,c);
		if (c == 0) {
			return b;
		}
		a = b;
		b = c;
	}
	// never reaches here
	return 0;
}


real_8 syn25calcfr (PICSTRUCT *p, int_4 mport, real_8 freq) {
	int i;
	int_4 divsel, divselb;
	int_4 fbint, fbnum, fbden, prescale, gcf, num, den, LDF;
    	real_8 vcofreq, fbdiv, fbfrac, lo, hi, actual_freq;
	real_8 thisEps, bestEps;
	int_4 numerator, denominator;	
	
	freq = freq * 2.;
	// pick output dividers;
	v3print("SYN25CALCFR: %f\n", freq);
	for (i = 0; i < BANDS; i++) {
		lo = OUTSEL_LF[i];
		hi = OUTSEL_UF[i];
		v3print("lo: %f hi: %f\n", lo, hi);
		if ( ( freq >= lo ) & ( freq < hi ) ) {
			divsel  = OUTSEL_DIV[i];
			divselb = OUTSEL_DIVB[i];
			v3print("SYNTH_DIV: %d\n",divsel);
			break;
		}
	}
	if (i == BANDS) {
		v3print("Bad frequency, limits are: 137.5 - 4400.0\n");
		return 0;
	}
	// pick VCO;
	vcofreq = freq * divsel ;
	v3print("VCOFREQ: %f\n", vcofreq);

	fbdiv = vcofreq/SYNTH_REF;
	fbint  = (int_4)(fbdiv);
	fbfrac = fbdiv - fbint;
	// mod (den) is limited to 12 bits (4095);
	// num is limited to den-1;
	fbden = 100000;
	fbnum = (int_4)(fbfrac*fbden+.5);
	// preset to integer mode;
	LDF = 1;
	num = 0;
	den = 1;
	// prescaler must be 8/9 (1) when over 3 GHz, min count is 75 (40 MHz PD), else min count is 23;
	if (vcofreq >= 3000.) {
		prescale = 1;
	} else {
		prescale = 0;
	}
	v3print("fbdiv: %f fbint: %d fbfrac: %f fbnum: %d prescale: %d\n", fbdiv, fbint, fbfrac, fbnum, prescale);
	// FIX - do other variables, clock div etc need change based on freq?;
	// FIX - error checking on number bounds;
	if (fbfrac != 0) {
		gcf = calcGCF(fbnum, fbden);
		num = fbnum/gcf;
		den = fbden/gcf;
		//change to fractional mode;
		v3print("gcf: %d num: %d den: %d\n", gcf, num,den);
		if (den > 4095){
			v3print("syn25fr: den too big\n");
			// do a brute force search for best fit 
			bestEps=1000000.;
			for (den=4095;den >=2048; den--){
				num=(int)(fbfrac*(double)den+.5);
				thisEps=fabs(((double)num/(double)den)-fbfrac);
				if(thisEps<bestEps){
					numerator = num;
					denominator = den;
					bestEps=thisEps;
				}
			}
			num = numerator;
			den = denominator;
			v3print("gcf: %d num: %d den: %d\n", gcf, num,den);
		}		
	}
	actual_freq = ((double)fbint+(double)num/(double)den)*SYNTH_REF/divsel/2.;
	v3print("SYN25CALCFR: actual RF freq %f\n",actual_freq);
	return actual_freq;
}

// calculate divisors for ADF4350 synthesizer;
// fractional part is limited to 12 bits (4095 for modulus) and 12 bits for numerator;
// numerator must be less than denominator;
// this limits us to 5 KHz channel spacing at the LO freq, 2.5 KHz at RF;
// The software does not automatically pick up an error here;
// FIX need to return actual frequency that was set or error??


int syn25fr (PICSTRUCT *p, int_4 mport, real_8 * reqfreq) {
	int i;
	int_4 divsel, divselb;
	int_4 fbint, fbnum, fbden, prescale, gcf, num, den, LDF;
    	real_8 vcofreq, fbdiv, fbfrac, lo, hi, freq, actual_freq;
	real_8 thisEps, bestEps;
	int_4 numerator, denominator;	
	int_4 CYCLE_SLIP = 0;
	int_4 LOW_SPUR = 0;
	int_4 retval = 0;
	
	freq = *reqfreq;
	/* chack the frequency bounds */
	if (freq < SYN25MINFREQ) {freq = SYN25MINFREQ; retval =-1;}
	if (freq > SYN25MAXFREQ) {freq = SYN25MAXFREQ; retval =-1;}
	freq = freq * 2.;
	// pick output dividers;
	v3print("synfr: %f\n", freq);
	for (i = 0; i < BANDS; i++) {
		lo = OUTSEL_LF[i];
		hi = OUTSEL_UF[i];
		v3print("lo: %f hi: %f\n", lo, hi);
		if ( ( freq >= lo ) & ( freq < hi ) ) {
			divsel  = OUTSEL_DIV[i];
			divselb = OUTSEL_DIVB[i];
			v3print("SYNTH_DIV: %d\n",divsel);
			break;
		}
	}
	// pick VCO;
	vcofreq = freq * divsel ;
	v3print("VCOFREQ: %f\n", vcofreq);

	fbdiv = vcofreq/SYNTH_REF;
	fbint  = (int_4)(fbdiv);
	fbfrac = fbdiv - fbint;
	// mod (den) is limited to 12 bits (4095);
	// num is limited to den-1;
	fbden = 100000;
	fbnum = (int_4)(fbfrac*fbden+.5);
	// preset to integer mode;
	LDF = 1;
	num = 0;
	den = 1;
	// prescaler must be 8/9 (1) when over 3 GHz, min count is 75 (40 MHz PD), else min count is 23;
	if (vcofreq >= 3000.) {
		prescale = 1;
	} else {
		prescale = 0;
	}
	v3print("fbdiv: %f fbint: %d fbfrac: %f fbnum: %d prescale: %d\n", fbdiv, fbint, fbfrac, fbnum, prescale);
	// FIX - do other variables, clock div etc need change based on freq?;
	// FIX - error checking on number bounds;
	if (fbfrac != 0) {
		gcf = calcGCF(fbnum, fbden);
		num = fbnum/gcf;
		den = fbden/gcf;
		//change to fractional mode;
		LDF = 0;
		v3print("gcf: %d num: %d den: %d\n", gcf, num,den);
		if (den > 4095){
			v3print("syn25fr: den too big\n");
			// do a brute force search for best fit 
			bestEps=1000000.;
			for (den=4095;den >=2048; den--){
				num=(int)(fbfrac*(double)den+.5);
				thisEps=fabs(((double)num/(double)den)-fbfrac);
				if(thisEps<bestEps){
					numerator = num;
					denominator = den;
					bestEps=thisEps;
				}
			}
			num = numerator;
			den = denominator;
			v3print("gcf: %d num: %d den: %d\n", gcf, num,den);
		}		
	}
	actual_freq = ((double)fbint+(double)num/(double)den)*SYNTH_REF/divsel/2.;
	*reqfreq = actual_freq;
	v3print("SYN25FR: actual RF freq %f\n",actual_freq);
	
	SYN_REG[0] = (fbint << 15) | (num <<3) ;
	SYN_REG[1] = (prescale << 27) | (1 << 15) | (den << 3) | 1;
	SYN_REG[2] = (SYN_REG_DEF[2] | (LDF << 8) | (LOW_SPUR<<29));
	SYN_REG[3] = (SYN_REG_DEF[3] | (CYCLE_SLIP << 18));
	SYN_REG[4] = (SYN_REG_DEF[4] | (divselb << 20));
	syn25wr(p, mport, 4, SYN_REG[4]);
	syn25wr(p, mport, 3, SYN_REG[3]);
	syn25wr(p, mport, 2, SYN_REG[2]);
	syn25wr(p, mport, 1, SYN_REG[1]);
	syn25wr(p, mport, 0, SYN_REG[0]);
	// setup input switch
	if(freq/2. > 1300.) rfxd_mmsk(p, mport, FEMTO_DAC_CTRL, ~FEMTO25_BAND, 0);
	else                rfxd_mmsk(p, mport, FEMTO_DAC_CTRL, ~FEMTO25_BAND, FEMTO25_BAND);
	return retval;
}
//FIX this relies on setting of reg2 , not reentrant
void syn25mux ( PICSTRUCT *p, int_4 mport, int_4 val ) {
	SYN_REG[2] = (SYN_REG[2] & ~(7 << 26)) | ((val & 7) << 26);
	syn25wr(p, mport, 2, SYN_REG[2]);
}

void syn25csr ( PICSTRUCT *p, int_4 mport, int_4 val ) {
	SYN_REG[3] = (SYN_REG[3] & ~(1 << 18)) | ((val & 1) << 18);
	syn25wr(p, mport, 3, SYN_REG[3]);
}

void syn25lowspur ( PICSTRUCT *p, int_4 mport, int_4 val ) {
	SYN_REG[2] = (SYN_REG[2] & ~(3 << 29)) | ((val & 3) << 29);
	syn25wr(p, mport, 2, SYN_REG[2]);
	v3print("mod = %d\n", (SYN_REG[1]>>3) & 0xfff);
}

void syn25pwr ( PICSTRUCT *p, int_4 mport, int_4 val ) {
	SYN_REG[4] = (SYN_REG[4] & ~(3 << 3)) | ((val & 3) << 3);
	syn25wr(p, mport, 4, SYN_REG[4]);
}

void syn25phase ( PICSTRUCT *p, int_4 mport, int_4 val ) {
	SYN_REG[1] = (SYN_REG[1] & ~(0xfff << 15)) | (((val >> 16) & 0xfff) << 15);
	syn25wr(p, mport, 1, SYN_REG[1]);
	syn25wr(p, mport, 0, SYN_REG[0]);  // double buffered value
}

void syn25cp ( PICSTRUCT *p, int_4 mport, int_4 val ) {
	SYN_REG[2] = (SYN_REG[2] & ~(0xf << 9)) | ((val & 0xf) << 9);
	syn25wr(p, mport, 2, SYN_REG[2]);
	syn25wr(p, mport, 0, SYN_REG[0]);  // double buffered value
}

// there is no reset on the synthesizer;
void syn25reset(PICSTRUCT *p, int_4 mport) {

}

int syn25init (PICSTRUCT *p, int_4 mport) {
	int i;

	// turn on power
	rfxd_mmsk(p, mport, FEMTO_BOARD_CTRL, ~FEMTO25_RSYN_CE, FEMTO25_RSYN_CE);
	syn25reset(p, mport);

	for (i=5; i>=0; i--){
		// the default values;
		SYN_REG[i] = SYN_REG_DEF[i];
		syn25wr(p, mport, i, SYN_REG[i] );
	}
	// turn on output
	rfxd_mmsk(p, mport, FEMTO_BOARD_CTRL, ~FEMTO25_RSYN_PD, FEMTO25_RSYN_PD);
	v3print("Synth25 init DONE\n");
	return 0;
}

//
// ADC code
//
int_4 ADC_REG[5] = {0x80, 0, 0, 1, 1}; 

void adc_start_spi(PICSTRUCT *p, int_4 mport) {
	rfxd_mwr(p, mport, FEMTO_SPI, FEMTO25_PRG_ADC );
	rfxd_mwr(p, mport, FEMTO_SPI, 0); 
}

int_4 adc_wait_for_ready(PICSTRUCT *p, int_4 mport) {
	int_4 count;
	for (count = 100; count != 0; count--) {
		if ((rfxd_mrd(p, mport, FEMTO_SPI) & 0x80) == 0) {
			break;
		}
	}
	if (count == 0) {
		v3print("Timeout on ADC wait for busy\n");
	}
	return count;
}

void adcwr (PICSTRUCT *p, int_4 mport, int_4 address, int_4 data) {
	v3print("ADCWR - a: %x d: %x\n", address, data);
	adc_wait_for_ready(p, mport);
	rfxd_mwr(p, mport, FEMTO_SPI2, address & 0x7f);
	rfxd_mwr(p, mport, FEMTO_SPI1, data);
	adc_start_spi(p, mport);
}

int_4 adcrd (PICSTRUCT *p, int_4 mport, int_4 address) {
	int_4 dat=0;
	v3print( "call to adcrd, not implemented yet\n");
	return dat;
}

void adclvds (PICSTRUCT *p, int_4 mport) {
	adcwr(p, mport, 3, 0x01);
	adcwr(p, mport, 2, 0x0b);
}

void adccmos(PICSTRUCT *p, int_4 mport) {
	adcwr(p, mport, 3, 0x00);
	adcwr(p, mport, 2, 0x00);
}

// put outputs in to a test mode 
void adctest (PICSTRUCT *p, int_4 mport, int_4 val) {
	int_4 temp;
	// valid modes are 0,1,3,5,7
	if ( ( val==0) | ((val & 1) == 1) ) {
		temp = (ADC_REG[4] & ~(7 << 3)) | ((val & 7) << 3);
		adcwr( p, mport, 4, temp);
	} else {
		v3print("invalid value for test register\n");
	}
}

void adcreset (PICSTRUCT *p, int_4 mport) {
	adcwr(p, mport, 0, 0x80);
	udelay(1000);  // delay since we can't read the reset bit
}

void adcinit (PICSTRUCT *p, int_4 mport) {
	int i;
	adcreset(p, mport);
	// set the default values and write the register
	for(i=1; i<5;i++){
		adcwr( p, mport, i, ADC_REG[i]);
	}	
	// FIX - only one will be needed when we are done
	if((rfxd_mrd(p, mport, FEMTO_VERSION) & 0xf0) == 0xc0) {
		adccmos(p, mport);
		v3print( "ADC init DONE, cmos mode\n");
	} else {
		adclvds(p, mport);
		v3print( "ADC init DONE, lvds mode\n");
	}
}

//
// clock code
//


//SI5338 Register Map File

// code for 100 MHz out on clock 0, 10 MHz (LVDS) on clkc1, 10 MHz (CMOS) on clk2
// Addr(Hex)     Value(Bin)    Value(Hex)
// Register map for use with AN428 (JumpStart)
// http://www.silabs.com/products/clocksoscillators/pages/default.aspx 
// Copyright 2010 Silicon Laboratories
// Generated by ClockBuilder Desktop software version 3.1
// Date: Tuesday, July 19, 2011 3:12 PM
// Input Frequency (MHz): 10.000000000

#define NUM_REGS_MAX 350

typedef struct Reg_Data{
   unsigned char Reg_Addr;
   unsigned char Reg_Val;
   unsigned char Reg_Mask;
} Reg_Data;

Reg_Data xint_5338[NUM_REGS_MAX] = {
{  0,0x00,0x00},
{  1,0x00,0x00},
{  2,0x00,0x00},
{  3,0x00,0x00},
{  4,0x00,0x00},
{  5,0x00,0x00},
{  6,0x04,0x1D},
{  7,0x31,0x00},
{  8,0x20,0x00},
{  9,0x12,0x00},
{ 10,0x0E,0x00},
{ 11,0xC7,0x00},
{ 12,0x00,0x00},
{ 13,0x00,0x00},
{ 14,0x00,0x00},
{ 15,0x00,0x00},
{ 16,0x00,0x00},
{ 17,0x00,0x00},
{ 18,0x00,0x00},
{ 19,0x00,0x00},
{ 20,0x00,0x00},
{ 21,0x00,0x00},
{ 22,0x00,0x00},
{ 23,0x00,0x00},
{ 24,0x00,0x00},
{ 25,0x00,0x00},
{ 26,0x00,0x00},
{ 27,0x70,0x80},
{ 28,0x28,0xFF},
{ 29,0x30,0xFF},
{ 30,0xA8,0xFF},
{ 31,0xC0,0xFF},
{ 32,0x02,0xFF},
{ 33,0x02,0xFF},
{ 34,0xE3,0xFF},
{ 35,0x05,0xFF},
{ 36,0x06,0x1F},
{ 37,0x06,0x1F},
{ 38,0x13,0x1F},
{ 39,0x06,0x1F},
{ 40,0x84,0xFF},
{ 41,0x5C,0x7F},
{ 42,0x23,0x3F},
{ 43,0x00,0x00},
{ 44,0x00,0x00},
{ 45,0x00,0xFF},
{ 46,0x00,0xFF},
{ 47,0x14,0x3F},
{ 48,0x2F,0xFF},
{ 49,0x05,0xFF},
{ 50,0xC3,0xFF},
{ 51,0x07,0xFF},
{ 52,0x10,0x7F},
{ 53,0x00,0xFF},
{ 54,0x0B,0xFF},
{ 55,0x00,0xFF},
{ 56,0x00,0xFF},
{ 57,0x00,0xFF},
{ 58,0x00,0xFF},
{ 59,0x01,0xFF},
{ 60,0x00,0xFF},
{ 61,0x00,0xFF},
{ 62,0x00,0x3F},
{ 63,0x10,0x7F},
{ 64,0x00,0xFF},
{ 65,0x80,0xFF},
{ 66,0x00,0xFF},
{ 67,0x00,0xFF},
{ 68,0x00,0xFF},
{ 69,0x00,0xFF},
{ 70,0x01,0xFF},
{ 71,0x00,0xFF},
{ 72,0x00,0xFF},
{ 73,0x00,0x3F},
{ 74,0x10,0x7F},
{ 75,0x00,0xFF},
{ 76,0x80,0xFF},
{ 77,0x00,0xFF},
{ 78,0x00,0xFF},
{ 79,0x00,0xFF},
{ 80,0x00,0xFF},
{ 81,0x01,0xFF},
{ 82,0x00,0xFF},
{ 83,0x00,0xFF},
{ 84,0x00,0x3F},
{ 85,0x10,0x7F},
{ 86,0x80,0xFF},
{ 87,0xFE,0xFF},
{ 88,0x03,0xFF},
{ 89,0x00,0xFF},
{ 90,0x00,0xFF},
{ 91,0x00,0xFF},
{ 92,0x01,0xFF},
{ 93,0x00,0xFF},
{ 94,0x00,0xFF},
{ 95,0x00,0x3F},
{ 96,0x10,0x00},
{ 97,0x00,0xFF},
{ 98,0x80,0xFF},
{ 99,0x00,0xFF},
{100,0x00,0xFF},
{101,0x00,0xFF},
{102,0x00,0xFF},
{103,0x01,0xFF},
{104,0x00,0xFF},
{105,0x00,0xFF},
{106,0x80,0xBF},
{107,0x00,0xFF},
{108,0x00,0x7F},
{109,0x00,0xFF},
{110,0x40,0xFF},
{111,0x00,0xFF},
{112,0x00,0x7F},
{113,0x00,0xFF},
{114,0x40,0xFF},
{115,0x00,0xFF},
{116,0x80,0xFF},
{117,0x00,0xFF},
{118,0x40,0xFF},
{119,0x00,0xFF},
{120,0x00,0xFF},
{121,0x00,0xFF},
{122,0x40,0xFF},
{123,0x00,0xFF},
{124,0x00,0xFF},
{125,0x00,0xFF},
{126,0x00,0xFF},
{127,0x00,0xFF},
{128,0x00,0xFF},
{129,0x00,0x0F},
{130,0x00,0x0F},
{131,0x00,0xFF},
{132,0x00,0xFF},
{133,0x00,0xFF},
{134,0x00,0xFF},
{135,0x00,0xFF},
{136,0x00,0xFF},
{137,0x00,0xFF},
{138,0x00,0xFF},
{139,0x00,0xFF},
{140,0x00,0xFF},
{141,0x00,0xFF},
{142,0x00,0xFF},
{143,0x00,0xFF},
{144,0x00,0xFF},
{145,0x00,0x00},
{146,0xFF,0x00},
{147,0x00,0x00},
{148,0x00,0x00},
{149,0x00,0x00},
{150,0x00,0x00},
{151,0x00,0x00},
{152,0x00,0xFF},
{153,0x00,0xFF},
{154,0x00,0xFF},
{155,0x00,0xFF},
{156,0x00,0xFF},
{157,0x00,0xFF},
{158,0x00,0x0F},
{159,0x00,0x0F},
{160,0x00,0xFF},
{161,0x00,0xFF},
{162,0x00,0xFF},
{163,0x00,0xFF},
{164,0x00,0xFF},
{165,0x00,0xFF},
{166,0x00,0xFF},
{167,0x00,0xFF},
{168,0x00,0xFF},
{169,0x00,0xFF},
{170,0x00,0xFF},
{171,0x00,0xFF},
{172,0x00,0xFF},
{173,0x00,0xFF},
{174,0x00,0xFF},
{175,0x00,0xFF},
{176,0x00,0xFF},
{177,0x00,0xFF},
{178,0x00,0xFF},
{179,0x00,0xFF},
{180,0x00,0xFF},
{181,0x00,0x0F},
{182,0x00,0xFF},
{183,0x00,0xFF},
{184,0x00,0xFF},
{185,0x00,0xFF},
{186,0x00,0xFF},
{187,0x00,0xFF},
{188,0x00,0xFF},
{189,0x00,0xFF},
{190,0x00,0xFF},
{191,0x00,0xFF},
{192,0x00,0xFF},
{193,0x00,0xFF},
{194,0x00,0xFF},
{195,0x00,0xFF},
{196,0x00,0xFF},
{197,0x00,0xFF},
{198,0x00,0xFF},
{199,0x00,0xFF},
{200,0x00,0xFF},
{201,0x00,0xFF},
{202,0x00,0xFF},
{203,0x00,0x0F},
{204,0x00,0xFF},
{205,0x00,0xFF},
{206,0x00,0xFF},
{207,0x00,0xFF},
{208,0x00,0xFF},
{209,0x00,0xFF},
{210,0x00,0xFF},
{211,0x00,0xFF},
{212,0x00,0xFF},
{213,0x00,0xFF},
{214,0x00,0xFF},
{215,0x00,0xFF},
{216,0x00,0xFF},
{217,0x00,0xFF},
{218,0x00,0x00},
{219,0x00,0x00},
{220,0x00,0x00},
{221,0x0D,0x00},
{222,0x00,0x00},
{223,0x00,0x00},
{224,0xF4,0x00},
{225,0xF0,0x00},
{226,0x00,0x00},
{227,0x00,0x00},
{228,0x00,0x00},
{229,0x12,0x00},
{231,0x00,0x00},
{232,0x00,0x00},
{233,0x00,0x00},
{234,0x00,0x00},
{235,0xF6,0x00},
{236,0x00,0x00},
{237,0x00,0x00},
{238,0x14,0x00},
{239,0x00,0x00},
{240,0x40,0x00},
{242,0x00,0x02},
{243,0xF0,0x00},
{244,0x00,0x00},
{245,0x00,0x00},
{247,0x0D,0x00},
{248,0x00,0x00},
{249,0xA8,0x00},
{250,0x00,0x00},
{251,0x80,0x00},
{252,0x00,0x00},
{253,0x00,0x00},
{254,0x00,0x00},
{255,   1,0xFF},
{  0,0x00,0x00},
{  1,0x00,0x00},
{  2,0x00,0x00},
{  3,0x00,0x00},
{  4,0x00,0x00},
{  5,0x00,0x00},
{  6,0x00,0x00},
{  7,0x00,0x00},
{  8,0x00,0x00},
{  9,0x00,0x00},
{ 10,0x00,0x00},
{ 11,0x00,0x00},
{ 12,0x00,0x00},
{ 13,0x00,0x00},
{ 14,0x00,0x00},
{ 15,0x00,0x00},
{ 16,0x00,0x00},
{ 17,0x01,0x00},
{ 18,0x00,0x00},
{ 19,0x00,0x00},
{ 20,0x90,0x00},
{ 21,0x31,0x00},
{ 22,0x00,0x00},
{ 23,0x00,0x00},
{ 24,0x01,0x00},
{ 25,0x00,0x00},
{ 26,0x00,0x00},
{ 27,0x00,0x00},
{ 28,0x00,0x00},
{ 29,0x00,0x00},
{ 30,0x00,0x00},
{ 31,0x00,0xFF},
{ 32,0x00,0xFF},
{ 33,0x01,0xFF},
{ 34,0x00,0xFF},
{ 35,0x00,0xFF},
{ 36,0x90,0xFF},
{ 37,0x31,0xFF},
{ 38,0x00,0xFF},
{ 39,0x00,0xFF},
{ 40,0x01,0xFF},
{ 41,0x00,0xFF},
{ 42,0x00,0xFF},
{ 43,0x00,0x0F},
{ 44,0x00,0x00},
{ 45,0x00,0x00},
{ 46,0x00,0x00},
{ 47,0x00,0xFF},
{ 48,0x00,0xFF},
{ 49,0x01,0xFF},
{ 50,0x00,0xFF},
{ 51,0x00,0xFF},
{ 52,0x90,0xFF},
{ 53,0x31,0xFF},
{ 54,0x00,0xFF},
{ 55,0x00,0xFF},
{ 56,0x01,0xFF},
{ 57,0x00,0xFF},
{ 58,0x00,0xFF},
{ 59,0x00,0x0F},
{ 60,0x00,0x00},
{ 61,0x00,0x00},
{ 62,0x00,0x00},
{ 63,0x00,0xFF},
{ 64,0x00,0xFF},
{ 65,0x01,0xFF},
{ 66,0x00,0xFF},
{ 67,0x00,0xFF},
{ 68,0x90,0xFF},
{ 69,0x31,0xFF},
{ 70,0x00,0xFF},
{ 71,0x00,0xFF},
{ 72,0x01,0xFF},
{ 73,0x00,0xFF},
{ 74,0x00,0xFF},
{ 75,0x00,0x0F},
{ 76,0x00,0x00},
{ 77,0x00,0x00},
{ 78,0x00,0x00},
{ 79,0x00,0xFF},
{ 80,0x00,0xFF},
{ 81,0x01,0xFF},
{ 82,0x00,0xFF},
{ 83,0x00,0xFF},
{ 84,0x90,0xFF},
{ 85,0x31,0xFF},
{ 86,0x00,0xFF},
{ 87,0x00,0xFF},
{ 88,0x01,0xFF},
{ 89,0x00,0xFF},
{ 90,0x00,0xFF},
{ 91,0x00,0x0F},
{ 92,0x00,0x00},
{ 93,0x00,0x00},
{ 94,0x00,0x00},
{255,   0,0xFF}
};

// changed registers for using external clock
//Addr(Hex)     Value(Bin)    Value(Hex)
Reg_Data xext_ch[4] = {
{  6, 0x08, 0x1D},
{ 29, 0x08, 0xFF},
{ 32, 0x22, 0xFF},
{ 33, 0x22, 0xFF}
};

// changed registers for using internal clock
//Addr(Hex)     Value(Bin)    Value(Hex)
Reg_Data xint_ch[4] = {
{  6, 0x04, 0x1D},
{ 29, 0x28, 0xFF},
{ 32, 0x02, 0xFF},
{ 33, 0x02, 0xFF}
};

// FIX these registers are not correct if we can even do this function
// changed registers for using int ref clock, ext sample clock
//Addr(Hex)     Value(Bin)    Value(Hex)
Reg_Data xintref_extfs_ch[4] = {
{  6, 0x04, 0x1D},
{ 29, 0x28, 0xFF},
{ 32, 0x02, 0xFF},
{ 33, 0x02, 0xFF}
};



// need for compatibility
void clk25_reset(PICSTRUCT *p, int_4 mport) {
}

void write_si5338 (PICSTRUCT *p, int_4 mport, int_4 a, int_4 d) {
	// wait for ready
	i2c_wait_for_ready(p, mport);
	// lsb to be 0 for write
	rfxd_mwr(p, mport, FEMTO_SPI1, 0xe0);
	rfxd_mwr(p, mport, FEMTO_SPI2, a);
	rfxd_mwr(p, mport, FEMTO_SPI3, d);
	// send data
	i2c_start(p, mport);
}

// pass in an array that holds the device address and all data to be sent to the device

int_4 read_si5338 (PICSTRUCT *p, int_4 mport, int_4 a) {
	int_4 d;
	// wait for ready
	i2c_wait_for_ready(p, mport);
	// set rd bit
	rfxd_mwr(p, mport, FEMTO_SPI1, 0xe1);
	rfxd_mwr(p, mport, FEMTO_SPI2, a);
	// send data
	i2c_start(p, mport);
	// wait for ready
	i2c_wait_for_ready(p, mport);
	// get returned data
	d = rfxd_mrd(p, mport, FEMTO25_I2C_LO);
	return d;
}

void clk25_wr(PICSTRUCT *p, int_4 mport, int_4 addr, int_4 data, int_4 mask) {
	int_4 curr_val, clear_curr_val, clear_new_val, combined, err;
		
	// ignore registers with masks of 0x00
	if ( mask != 0x00) {
		//v3print("clk25wr a:%03d d:%02x m: %02x\n", addr,data,mask);
		if ( mask == 0xff) {
		// do a regular I2C write to the register
		// at addr with the desired data value
			write_si5338(p, mport, addr, data);
			err = rfxd_mrd(p, mport, 5);
			//puts "  write a:addr d:data m:mask e: err"
		} else {
			// do a read-modify-write using I2C and
			// bit-wise operations
			// get the current value from the device at the
			// register located at addr
			//curr_val 0x[read_si5338 addr]
			curr_val = read_si5338(p, mport, addr);
			err = rfxd_mrd(p, mport, 5);
			//puts "  read a:addr d:curr_val e: err"
			// clear the bits that are allowed to be
			// accessed in the current value of the register
			clear_curr_val = curr_val &  (mask ^ 0xff);
			// clear the bits in the desired data that
			// are not allowed to be accessed
			clear_new_val = data & mask;
			// combine the cleared values to get the new
			// value to write to the desired register
			combined = clear_curr_val | clear_new_val;
			//puts "  write  masked register: clear_curr_val new_value: clear_new_val"
			write_si5338(p, mport, addr, combined);
			err = rfxd_mrd(p, mport, 5);
			//puts "  write a:addr d:combined e: err"
		}
	}
}

void clk25_list_wr (PICSTRUCT *p, int_4 mport, Reg_Data * x, int_4 length) {
	int_4 a, d, m, i;
	v3print("clk list write length %d\n", length);

	for (i = 0; i < length; i++) {
		a = x->Reg_Addr;
		d = x->Reg_Val;
		m = x->Reg_Mask;
		clk25_wr(p, mport, a, d, m);
		x++;
	}
}

void clk25_dump (PICSTRUCT *p, int_4 mport, int_4 diff) {
	int_4 a, d, m, l, i, dat;
	Reg_Data * x;
	int count = 0;

	x  = xint_5338;
	l = sizeof(xint_5338)/sizeof(Reg_Data);
	v3print("Clk dump length %d\n", l);
	write_si5338(p, mport, 255, 0);
	for (i = 0; i < l; i++) {
		a = x->Reg_Addr;
		d = x->Reg_Val;
		m = x->Reg_Mask;
		x++;
		if (m !=0 ) {
			//puts "addr: a data: d mask: m"
			//dat 0x[read_si5338 a]
			dat = read_si5338(p, mport, a);
			if (diff) {
				if ( dat != d ) {
					v3print("dump a: %d table: %x read: %x\n",a, d, dat);
				}
			} else {
				v3print("dump a: %d data %x\n",a, d);
			}
		}
		if (a == 255 ) {
			// change the page
			write_si5338(p, mport, 255, 1);
		}
	}
	// change the page
	write_si5338(p, mport, 255, 0);
}

void clk25_lst_dump (PICSTRUCT *p, int_4 mport) {
	int_4 a, d, m, l, i;
	Reg_Data * x;

	x  = xint_5338;
	l = sizeof(xint_5338)/sizeof(Reg_Data);
	v3print("length %d\n", l);
	for (i = 0; i < l; i++) {
		a = x->Reg_Addr;
		d = x->Reg_Val;
		m = x->Reg_Mask;
		if (m !=0 ) {
			v3print("dump a: %d data %x\n",a, d);
		}
	}
}

int_4 clk25_wait_for_status (PICSTRUCT *p, int_4 mport, int_4 mask) {
	int status, count;
	for (count = CLK_TIMEOUT; count != 0; count--) {
		status = read_si5338(p, mport, 218);
		if ((status & mask) == 0) {
		       break;
	       }
		udelay(100);	// wait 100 uS
	}
	if (count == 0) {
		v3print("Timeout waiting for clk status mask, status = %x\n", status);
	}
	return status;
}

void clk25_en (PICSTRUCT *p, int_4 mport ) {
	// enable outputs
	clk25_wr(p, mport, 230, 0x00, 0x10);
}

void clk25_dis (PICSTRUCT *p, int_4 mport) {
	// page 0
	clk25_wr(p, mport, 255, 0, 0xff);
	// Disable all outputs
	clk25_wr(p, mport, 230, 0x10, 0x10);
}

int_4 clk25_cal(PICSTRUCT *p, int_4 mport) {
	int_4 err, x;
	
	// wait for clock input to be valid - IN4
	err = clk25_wait_for_status(p, mport, 0x08);
	// configure PLL for locking
	clk25_wr(p, mport, 49, 0x00, 0x80);
	// Initiate PLL locking
	clk25_wr(p, mport, 246, 0x02, 0x02);
	// Restart LOL
	clk25_wr(p, mport, 241, 0x00, 0x80);
	clk25_wr(p, mport, 241, 0x65, 0x7f); // this is snow in init also
	// wait 25 ms
	rfxd_delay(p, mport, 25000);
	// wait for PLL to be locked by looking at PLL_LOL, SYS_CAL and other alarms
	// FIX is this dependent on which clock is selected?
	err = clk25_wait_for_status(p, mport, 0x19);
	// copy FCAL values to active registers
	x = read_si5338(p, mport, 237);
	clk25_wr(p, mport, 47, x, 3);
	x= read_si5338( p, mport, 236);
	clk25_wr(p, mport, 46, x, 0xff);
	x = read_si5338(p, mport, 235);
	clk25_wr(p, mport, 45, x, 0xff);
	clk25_wr(p, mport, 47, 0x14, 0xfc);
	//PLL to use FCAL values
	clk25_wr(p, mport, 49, 0x80, 0x80);
	return err;
}

// initialize the clock to use the internal reference clock an FS 100 MHz
// this voidedure is from Figure 9 in the data sheeet
void clk25_init (PICSTRUCT *p, int_4 mport) {
	int_4 l, err;

	// Disable all outputs
	clk25_dis(p, mport);
	// Pause LOL
	//clk25_wr(p, mport, 241, 0x65, 0x7f); // not in table for some reason
	clk25_wr(p, mport, 241, 0x80, 0x80);
	// write list of commands to configure clock gen
	l = sizeof(xint_5338)/sizeof(Reg_Data);
	clk25_list_wr(p, mport, xint_5338,l);
	// call the calibrate functions
	err = clk25_cal(p, mport);
	//enable selected outputs
	clk25_en(p, mport);
	v3print("FEMTO 2.5 clkinit done - status = %x\n", err);
}

// change the clock reference source
// pass in 0 for internal clock, non zero for external clock
void clk25_ref (PICSTRUCT *p, int_4 mport, int_4 refsrc) {
	Reg_Data * x;
	int_4 l;

	if (refsrc == 0) {
		v3print("FEMTO 2.5 internal reference clock\n");
		x  = xint_ch;
		l = sizeof(xint_ch)/sizeof(Reg_Data);
	} else {
		v3print("FEMTO 2.5 external reference clock\n");
		x = xext_ch;
		l = sizeof(xext_ch)/sizeof(Reg_Data);
	}
	clk25_list_wr(p, mport, x, l);
	// FIX do we need to recal ?
	// Do we need to reset DCM?
	// Doo we need to do anything to synth?
}

// 3 modes of operation
// 1. internal reference, internal sample clock
// 2. external reference, internal sample clock
// 3. internal reference, external sample clock

void clk25_fs (PICSTRUCT *p, int_4 mport, int_4 intext) {
	Reg_Data * x;
	int_4 l;
	if (intext == 0) {
		v3print("int ref clock, int sample clock\n");
		x  = xint_ch;
		l = sizeof(xint_ch)/sizeof(Reg_Data);
	} else if (intext == 1) {
		v3print("ext ref clock, int sample clock\n");
		// don't use reference
		x = xext_ch;
		l = sizeof(xext_ch)/sizeof(Reg_Data);
	} else if (intext == 2) {
		x = xintref_extfs_ch;
		l = sizeof(xintref_extfs_ch)/sizeof(Reg_Data);
		v3print("int ref clock, ext sample clock\n");
		// don't use reference
	} else {
		v3print("Invalid sample src option\n");
	}
	clk25_list_wr(p, mport, x, l);
	// FIX do we need to recal ?
	// Do we need to reset DCM?
	// Do we need to do anything to synth?
}

// write to address a, multisynth values p1, p2, p3 with appropriate shifts
// write function is address data mask

void MSX_write (PICSTRUCT *p, int_4 mport, int_4 a, int_4 p1, int_4 p2, int_4 p3) {
	clk25_wr(p, mport, a++,   p1      & 0xff,  0xff);
	clk25_wr(p, mport, a++,  (p1>> 8) & 0xff,  0xff);
	clk25_wr(p, mport, a++, ((p1>>16) & 0x3) | ((p2 << 2) & 0xfc), 0xff);
	clk25_wr(p, mport, a++,  (p2>> 6) & 0xff,  0xff);
	clk25_wr(p, mport, a++,  (p2>>14) & 0xff,  0xff);
	clk25_wr(p, mport, a++,  (p2>>22) & 0xff,  0xff);
	clk25_wr(p, mport, a++,   p3      & 0xff,  0xff);
	clk25_wr(p, mport, a++, ((p3>> 8) & 0xff), 0xff);
	clk25_wr(p, mport, a++, ((p3>>16) & 0xff), 0xff);
	clk25_wr(p, mport, a++, ((p3>>24) & 0xff), 0x3f);
}

// max frequency is 125 so starting vco at 2600 MHZ guarentees result will be greater than 2200 MHz
// lowest freq has to fit in multisynth integer part from 2600 MHz 
// !! need to see how these all fit into the actual registers
// !! need to add code to calc BOTH output multisynth and feedback multisynth
// FIX - check Silabs rules for frequency settings
// FIX - do some recaluation to make sure floating point is correct
// Note: Frequency is in MHz
//
void clk25_div (PICSTRUCT *p, int_4 mport, double fs) {
	double ref = 10.;
	int_4 MSx_P1, MSx_P2, MSx_P3, outdiv, fbint, fbnum, RSEL, BWSEL;
	int_4 PLL_KPHI, Q, VCO_GAIN, MSCAL;
	double vco, fbdiv, fbfrac, fbden, denf, K; 	
	int_4  gcf, num, den;

	outdiv = (int_4)(2600./fs)/2*2;
	MSx_P1 = (int_4)((outdiv*128) - 512);
	MSx_P2 = 0;
	MSx_P3 = 1;
	MSX_write(p, mport, 53, MSx_P1, MSx_P2, MSx_P3);
	v3print("OUTDIV - P1: %d p2: %d P3: %d\n", MSx_P1,MSx_P2,MSx_P3);
	vco = fs*outdiv;
	fbdiv = vco/ref;
	// test value
	//fbdiv 99.5328
	fbint  = (int_4)(fbdiv);
	fbfrac = fbdiv - fbint;
	fbden = 100000;
	fbnum = (int_4)(fbfrac*fbden+.5);
	v3print("fs: %f vco: %f outdiv: %d fbdiv: %f fbint: %d fbfrac: %f fbnum: %d\n", fs, vco, outdiv, fbdiv, fbint, fbfrac, fbnum);
	if (fbfrac != 0) {
		gcf = calcGCF(fbnum, fbden);
	} else {
		gcf = 1;
	}
	num = fbnum/gcf;
	den = fbden/gcf;
	v3print( "gcf: %d num: %d den: %d\n", gcf, num, den);
	denf = den * 1.;
	MSx_P1 = (int_4)((((fbint*denf+num)*128)/denf) -512);
	MSx_P2 = (num*128) % den;
	MSx_P3 = den;
	MSX_write(p, mport, 97, MSx_P1, MSx_P2, MSx_P3);
	v3print( "FBDIV - P1: %d p2: %d P3: %d\n", MSx_P1, MSx_P2, MSx_P3);
	// now calculate PLL values - see Figure 9, AN411
	// following are based on Fpfd = 10 MHz
	K = 325.;
	RSEL = 1;
	BWSEL = 1;
	// compute PLL_KPHI (register 48)
	if (vco > 2425) {
		Q =3;
		VCO_GAIN = 0;
	} else {
		Q = 4;
		VCO_GAIN = 1;
	}
	PLL_KPHI = (int_4)(K/(525.*Q)*(vco/ref)*pow(2500./vco,3.)+.5);
	//calculate MSCAL
	MSCAL = (int_4)(-6.67*(vco/1000.) + 20.67 +.5);
	clk25_wr(p, mport, 48, PLL_KPHI, 0x7f);
	clk25_wr(p, mport, 49, BWSEL + (RSEL << 2) + (VCO_GAIN<<4), 0x7f);
	clk25_wr(p, mport, 50, MSCAL, 0x3f);
}

//FIX 
// need to add error checking
// always have to rethe FPGA DCM

// the frequency in MHz
// assumes clk25_init has been run and chip is setup
// this changes the dividers and does the PLL calibration
// 
// FIX proably needs too return actual sample rate used
//
void fs (PICSTRUCT *p, int_4 mport, double rate) {
	// check the rate

	//disable the clock
	clk25_dis(p, mport);
	// Pause LOL
	clk25_wr(p, mport, 241, 0x80, 0x80);
	// calculate the new dividers and program the dividers
	clk25_div(p, mport, rate);
	// calibrate
	clk25_cal(p, mport);
	//enable selected output	
	clk25_en(p, mport);
}
//
// integer interface version
//

void clk25_fsi (PICSTRUCT *p, int_4 mport, int_4 rate) {
	double frate;
	frate = ((double)rate)/1e6;
	fs(p, mport, frate);
}
	


