/* 
@(#)$Id: radio.c 527 2008-01-04 16:40:29Z nossen(
*/
/* <BEGINTEXT>
 *
 * @(#)$Id: radio.c 527 2008-01-04 16:40:29Z nossen(
 *
 * CPCI:	
 * SUBPROGRAM:	HARDWARE INTERFACE
 * MODULE:	$RCSfile: radio.c,v(
 * 
 * PROGRAMMER: Steve Nossen
 *
 * FUNCTIONAL DESCRIPTION:
 *
 *  Useful functions for Radio Interface
 * 
 * 1/25/11 sjn add synrd/wr debug, cahnge synth monitor to lock detect
 * 3/24/11 sjn add ability to divide reference for synthesizer
 * 3/24/11 sjn add vco limits setting, called during syninit
 * 5/17/11 sjn make RefDiv local, read from Synth when needed
 *             place vco limits table in FPGA memory and retrieve when selecting VCO
 *             Add code to select synth Reg E bias based on freq (it appears repeatable over units and temp)
 *             Add return value from synrd vio command (don't know if it will be sent to GUI)
 * 12/28/2011 sjn change RFBIAS to 2 volts (reduces 2nd harmonic, 3rd harmonic)
 *                change synfr to set to min/max if out of bound
 */

#define LGSLIB_DEBUG 0

#define LNA_HI 3
#define LNA_LO 6

// define a data space for state variables needed for new commands
#if MEMSTRUCT
#define MAXDEV	12
#define MAXPORT	3

LGSSTRUCT femto[MAXDEV][MAXPORT];
#else
#define RAM_OFFSET 0xa0
#define RFXD_WR(p, port, x, addr, data)  rfxd_wrram_c(p, port, addr, data)
#define RFXD_WR2(p, port, x, addr, data)  rfxd_wrram_s(p, port, addr, data)
#define RFXD_WRI(p, port, x, addr, data)  rfxd_wrram_i(p, port, addr, data)
#define RFXD_WRF(p, port, x, addr, data)  rfxd_wrram_f(p, port, addr, data)
#define RFXD_WRD(p, port, x, addr, data)  rfxd_wrram_d(p, port, addr, data)

#define  RFXD_RD(p, port, x, addr)       rfxd_rdram_c(p, port, addr)
#define RFXD_RD2(p, port, x, addr)       rfxd_rdram_s(p, port, addr)
#define RFXD_RDI(p, port, x, addr)       rfxd_rdram_i(p, port, addr)
#define RFXD_RDF(p, port, x, addr)       rfxd_rdram_f(p, port, addr)
#define RFXD_RDD(p, port, x, addr)       rfxd_rdram_d(p, port, addr)
#endif


// data is right aligned, shifted msb first
// mask indicates the first bit that is used, also indirectly the count i.e. 0x8 equals 5 bits
// xor allows the data specified to be "logical", i.e 0 is smallest but fixes that data for IC needs 
// [00] b0..b2 = LNA Gain
// [01] b3..b4 = Filter mode
// [02] b5..b9 = WCDMA/GSM filter corner frequency
// [03] b10..b14 = VGA gain vcontrol
// [04] b15..b16 = Filter gain control 1
// [05] b17..b19 = Filter gain control 2
// [06] b20..b22 = Filter gain control 3
// [07] b23..b25 = Filter gain control 4
// [08] b26..b27 = Filter gain control 5
// [09] b28..b31 = WCDMA bias control
// [00] b32..b35 = GSM bias control
// [11] b36..b39 = 20 MHz filter corner frequency
// [12] b40..b43 = 20 MHz opamp bias1
// [13] b44..b47 = 20 MHz opamp bias2
// [14] b48..b51 = 20 MHz opamp bias3
// [15] b52..b55 = 20 MHz opamp bias4
// [16] b56..b59 = 20 MHz opamp bias5
// [17] b60..b63 = 20 MHz opamp bias6

// CDMA default

unsigned char rxreg_mask[NUMRFIC_REGS]      = {0x04,0x02,0x10,0x10,0x02,0x04,0x04,0x04,0x02,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08};
unsigned char rxreg_xor [NUMRFIC_REGS]      = {   0,0x00,0x10,0x01,   0,   0,   0,   0,   0,0x08,   0,0x08,   0,   0,   0,   0,   0,   0};
unsigned char rxreg_data[NUMRFIC_REGS]      = {0x01,0x00,0x10,0x01,   0,   0,   0,   0,   0,0x08,   0,0x08,   0,   0,   0,   0,   0,   0};
// init data holds most recent bias settings
unsigned char rxreg_init_data[NUMRFIC_REGS] = {0x01,0x00,0x10,0x01,0x00,0x00,0x00,0x00,0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08};

char * reg_name[] = {
	"LNA Gain",
	"Filter Mode",
	"WCDMA/GSM Freq",
	"VGA Gain",
	"Filt Gain 1",
	"Filt Gain 2",
	"Filt Gain 3",
	"Filt Gain 4",
	"Filt Gain 5",
	"WCDMA Bias",
	"GSM Bias",
	"20 MHz Freq",
	"20 MHz Bias 1",
	"20 MHz Bias 2",
	"20 MHz Bias 3",
	"20 MHz Bias 4",
	"20 MHz Bias 5",
	"20 MHz Bias 6",
};
	
/* keep a local copy of what we write to RFIC
 *  to be replaced by real reads in the future
 */

struct freqlst freqtbl[] = {
  {7,1,16500,19500}, 
  {8,1,19000,22000}, 
  {1,2,8250,9750},   
  {2,2,9500,11000},  
  {4,4,4125,4875},   
  {5,4,4750,5500}    
};

// need a radio table or a function to calulate parameters from channel

char * rad_error_string[] = {
	"No error",
	"Illegal Radio Board Number"
};

/****************************************************************************/
/*                                                                          */
/*  This is where the interface to ICELIB is located                        */
/*                                                                          */
/****************************************************************************/


int rfxd_mrdq (PICSTRUCT *p, int_4 mport, int_4 addr) {	
  int data;
  //v3print("MRD - addr: %02x\n", addr);
  data  = (pic_jpmemrd(p, mport, addr+3, 1) & 0xff);
  data |= (pic_jpmemrd(p, mport, addr+2, 1) & 0xff) << 8;
  data |= (pic_jpmemrd(p, mport, addr+1, 1) & 0xff) << 16;
  data |= (pic_jpmemrd(p, mport, addr  , 1) & 0xff) << 24;
  return data;
}

int rfxd_mrdw (PICSTRUCT *p, int_4 mport, int_4 addr) {	
  int data;
  //v3print("MRD - addr: %02x\n", addr);
  data  = (pic_jpmemrd(p, mport, addr+1, 1) & 0xff);
  data |= (pic_jpmemrd(p, mport, addr  , 1) & 0xff) << 8;
  return data;
}

int rfxd_mrd (PICSTRUCT *p, int_4 mport, int_4 addr) {	
  int data;
  //v3print("MRD - addr: %02x\n", addr);
  data = pic_jpmemrd(p, mport, addr, 1);
  return data;
}

int rfxd_mmsk (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 dand, int_4 dxor) {
  /* just do a write */
  //v3print("MMSK - addr: %02x and: %02x xor: %02x\n", addr, dand, dxor);
  pic_jpmemmsk (p, mport, addr, dand, dxor, 1);
  return 0;
}

int rfxd_mwr (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 data) {
  //v3print("MWR - addr: %02x data: %02x\n", addr, data);
  pic_jpmemwr(p, mport, addr, data, 1);
  return 0;
}	

int rfxd_mwrdword (PICSTRUCT *p, int_4 mport, int addr, int data) {
  rfxd_mwr(p, mport, addr++, (data >> 24) & 0xff);
  rfxd_mwr(p, mport, addr++, (data >> 16) & 0xff);
  rfxd_mwr(p, mport, addr++, (data >>  8) & 0xff);
  rfxd_mwr(p, mport, addr++, (data      ) & 0xff);
  return 0;
}

int rfxd_mwrword (PICSTRUCT *p, int_4 mport, int addr, int data) {
  rfxd_mwr(p, mport, addr++, (data >>  8) & 0xff);
  rfxd_mwr(p, mport, addr++, (data      ) & 0xff);
  return 0;
}

int rfxd_dumpreg (PICSTRUCT *p, int_4 mport, int start,int stop){
  int i, temp;
  for(i=start;i<stop;i++){
    temp = rfxd_mrd(p,mport,i);
    vprint("FPGA reg: %02x data: %02x\n", i,temp);
  }
  return 0;
}

#if MEMSTRUCT
#else
// write RAM and read RAM need a semaphore mechanism to prevent anotehr instance from 
// interuppting an operation. If only read or write are happening in an app
// then semaphore is not needed although partial results are possible.
// EG - reading freq could occur while writing freq, an incorrect value will be read

int rfxd_wrram_c(PICSTRUCT *p, int_4 port, int_4 addr, int_4 data) {
	//v3print("WRRAM: addr %02x data: %02x\n", addr, data);	
	addr += RAM_OFFSET;
	rfxd_mwr(p, port, addr, (data & 0xff));
	return 0;
}

int rfxd_wrram_s(PICSTRUCT *p, int_4 port, int_4 addr, int_4 data) {
	//v3print("WRRAM: addr: %02x data: %08x\n", addr, data);	
	addr += RAM_OFFSET;
	rfxd_mwrword(p, port, addr, data);
	return 0;
}

int rfxd_wrram_i(PICSTRUCT *p, int_4 port, int_4 addr, int_4 data) {
	//v3print("WRRAM: addr: %02x data: %08x\n", addr, data);	
	addr += RAM_OFFSET;
	rfxd_mwrdword(p, port, addr, data);
	return 0;
}

// should size be fixed to 4 here ? Is float promoted to double??
int rfxd_wrram_f(PICSTRUCT *p, int_4 port, int_4 addr, real_4 data) {
	union flt {
		real_4 a;
		int_4 b;
	} d;

	d.a = data;
	//v3print("WRRAMF: addr %02x data:%08x\n", addr, d.b);	
	addr += RAM_OFFSET;
	rfxd_mwrdword(p, port, addr, d.b);
	return 0;
}

int rfxd_wrram_d(PICSTRUCT *p, int_4 port, int_4 addr, real_8 data) {
	union dbl {
		real_8 a;
		int_8 b;
	} d;

	d.a = data;
	//v3print("WRRAMD: addr %02x data_f: %f data: %16llx\n", addr, d.a, d.b);	
	addr += RAM_OFFSET;
	// write MSB first
	rfxd_mwrdword(p, port, addr,  (int_4)(d.b >> 32));
	rfxd_mwrdword(p, port, addr+4,(int_4)(d.b));
	return 0;
}

int_4 rfxd_rdram_c(PICSTRUCT *p, int_4 port, int_4 addr) {
	int_4 data;	
	addr += RAM_OFFSET;
	data = (int_4) (rfxd_mrd(p, port, addr) & 0xff);
	//v3print("RDRAM: addr: %02x data: %02x\n", addr-RAM_OFFSET, data);	
	return data;
}

int_4 rfxd_rdram_s(PICSTRUCT *p, int_4 port, int_4 addr) {
	int_4 data;	
	addr += RAM_OFFSET;
	data = (int_4)rfxd_mrdw(p, port, addr) ;
	//v3print("RDRAM: addr: %02x data: %08x\n", addr-RAM_OFFSET, data);	
	return data;
}

int_4 rfxd_rdram_i(PICSTRUCT *p, int_4 port, int_4 addr) {
	int_4 data;	
	addr += RAM_OFFSET;
	data = rfxd_mrdq(p, port, addr) ;
	//v3print("RDRAM: addr: %02x data: %08x\n", addr-RAM_OFFSET, data);	
	return data;
}

real_4 rfxd_rdram_f(PICSTRUCT *p, int_4 port, int_4 addr) {
	union flt {
		real_4 a;
		int_4 b;
	} d;

	d.b = 0;

	addr += RAM_OFFSET;	
	d.b = (int_4) (rfxd_mrdq(p, port, addr));
	//v3print("RDRAMF: addr: %02x data: %08x\n", addr-RAM_OFFSET, d.b);	
	return d.a;
}

real_8 rfxd_rdram_d(PICSTRUCT *p, int_4 port, int_4 addr) {
	union dbl {
		real_8 a;
		int_8 b;
	} d;

	addr += RAM_OFFSET;		// need to read MSB first
	d.b =               (int_8) (rfxd_mrdq(p, port, addr)   & 0x000000000ffffffff);
	d.b = (d.b << 32) | (int_8) (rfxd_mrdq(p, port, addr+4) & 0x000000000ffffffff);
	//v3print("RDRAMD: addr: %02x data_f: %f data_d: %16llx\n", addr-RAM_OFFSET, d.a, d.b);	
	return d.a;
}

// this version does a read though mrd function
int rfxd_dumpram (PICSTRUCT *p, int_4 mport){
  int i, temp, temp1;
  for(i=0;i<96;i++){
    temp1 = p->verbose;  // save the current debug setting
    p->verbose = 0;      // turn off debug
    temp = rfxd_mrd(p,mport,i+RAM_OFFSET);
    p->verbose = temp1;  // restore debug setting
    if(i%16 == 0) v3print("%02x: ", i);
    v3print("%02x ", temp);
    if(i%16 == 7) v3print(" ");
    if(i%16 == 15) v3print("\n");
//    v3print("addr: %02x data: %02x\n", i, temp);
  }
  return 0;
}

// this version does a read through readram function
int rfxd_dumpram2 (PICSTRUCT *p, int_4 mport){
  int i, temp, temp1;
  for(i=0;i<96;i++){
    temp1 = p->verbose;  // save the current debug setting
    p->verbose = 0;      // turn off debug
    temp = rfxd_rdram_c(p,mport,i);
    p->verbose = temp1;  // restore debug setting
    if(i%16 == 0) v3print("%02x: ", i);
    v3print("%02x ", temp);
    if(i%16 == 7) v3print(" ");
    if(i%16 == 15) v3print("\n");
//    v3print("addr: %02x data: %02x\n", i, temp);
  }
  return 0;
}

#endif

/*                      
 * clock programming
 *
 *
 * AD9516/17/18 Register Map File, Rev 1.0
 *
 * change to FA for 25 MHz clock 
 * {0011          01100100      64}
 * {0011          11111010      FA}
 *
 * {0010          01111101      7D}
 * {0016          00000101      05}
 *
 * AD9516/17/18 Register Map File, Rev 1.0
 *  Femto AM2 10 MHZ ref, 9517-0
 *  Addr(Hex)     Value(Bin)    Value(Hex)
*/
int xint[] = {
0x0000, 0x18, \
0x0001, 0x00, \
0x0002, 0x10, \
0x0003, 0x11, \
0x0004, 0x00, \
0x0010, 0x7C, \
0x0011, 0x64, \
0x0012, 0x00, \
0x0013, 0x00, \
0x0014, 0x1c, \
0x0015, 0x07, \
0x0016, 0x05, \
0x0017, 0x00, \
0x0018, 0x06, \
0x0019, 0x00, \
0x001A, 0x00, \
0x001B, 0x00, \
0x001C, 0x02, \
0x001D, 0x00, \
0x001E, 0x00, \
0x001F, 0x0E, \
0x00A0, 0x01, \
0x00A1, 0x00, \
0x00A2, 0x00, \
0x00A3, 0x01, \
0x00A4, 0x00, \
0x00A5, 0x00, \
0x00A6, 0x01, \
0x00A7, 0x00, \
0x00A8, 0x00, \
0x00A9, 0x01, \
0x00AA, 0x00, \
0x00AB, 0x00, \
0x00F0, 0x0B, \
0x00F1, 0x0B, \
0x00F2, 0x08, \
0x00F3, 0x0A, \
0x00F4, 0x0B, \
0x00F5, 0x0B, \
0x0140, 0x43, \
0x0141, 0x42, \
0x0142, 0x4A, \
0x0143, 0x43, \
0x0190, 0x00, \
0x0191, 0x80, \
0x0192, 0x00, \
0x0193, 0xBB, \
0x0194, 0x00, \
0x0195, 0x00, \
0x0196, 0x00, \
0x0197, 0x00, \
0x0198, 0x00, \
0x0199, 0x32, \
0x019A, 0x00, \
0x019B, 0x00, \
0x019C, 0x00, \
0x019D, 0x00, \
0x019E, 0x32, \
0x019F, 0x00, \
0x01A0, 0x00, \
0x01A1, 0x00, \
0x01A2, 0x00, \
0x01A3, 0x00, \
0x01E0, 0x00, \
0x01E1, 0x02, \
0x0230, 0x00, \
0x0231, 0x00, \
0x0232, 0x00
};

// Addr(Hex)     Value(Bin)    Value(Hex)
int xext [] = { 
0x0000, 0x18,
0x0001, 0x00,
0x0002, 0x10,
0x0003, 0x53,
0x0004, 0x00,
0x0010, 0x71,
0x0011, 0x01,
0x0012, 0x00,
0x0013, 0x00,
0x0014, 0x03,
0x0015, 0x00,
0x0016, 0x06,
0x0017, 0x00,
0x0018, 0x0E,
0x0019, 0x00,
0x001A, 0x00,
0x001B, 0x00,
0x001C, 0x00,
0x001D, 0x00,
0x001E, 0x00,
0x001F, 0x0E,
0x00A0, 0x01,
0x00A1, 0x00,
0x00A2, 0x00,
0x00A3, 0x01,
0x00A4, 0x00,
0x00A5, 0x00,
0x00A6, 0x01,
0x00A7, 0x00,
0x00A8, 0x00,
0x00A9, 0x01,
0x00AA, 0x00,
0x00AB, 0x00,
0x00F0, 0x0B,
0x00F1, 0x0B,
0x00F2, 0x08,
0x00F3, 0x0A,
0x00F4, 0x0B,
0x00F5, 0x0B,
0x0140, 0x43,
0x0141, 0x42,
0x0142, 0x4A,
0x0143, 0x43,
0x0190, 0x00,
0x0191, 0x80,
0x0192, 0x00,
0x0193, 0xBB,
0x0194, 0x00,
0x0195, 0x00,
0x0196, 0x00,
0x0197, 0x00,
0x0198, 0x00,
0x0199, 0x22,
0x019A, 0x00,
0x019B, 0x11,
0x019C, 0x30,
0x019D, 0x00,
0x019E, 0x22,
0x019F, 0x00,
0x01A0, 0x11,
0x01A1, 0x30,
0x01A2, 0x00,
0x01A3, 0x00,
0x01E0, 0x02,
0x01E1, 0x01,
0x0230, 0x00,
0x0231, 0x00,
0x0232, 0x00
};


int rfxd_delay(PICSTRUCT *p, int_4 mport, int count){
	udelay(count);
	return 0;
}

void clk_start_spi (PICSTRUCT *p, int_4 mport) {
	rfxd_mwr (p, mport, FEMTO_SPI, FEMTO_PRG_PLL );
	rfxd_mwr (p, mport, FEMTO_SPI, 0); 
}

void clk_wr (PICSTRUCT *p, int_4 mport, int addr, int data) {
	// rfxd_mwrdword( 0x1d + 0x80, (addr << 8) | data);
	rfxd_mwr (p, mport, FEMTO_SPI3, addr >> 8);
	rfxd_mwr (p, mport, FEMTO_SPI2, addr & 0xff);
	rfxd_mwr (p, mport, FEMTO_SPI1, data);
	clk_start_spi(p, mport);
}

void clk_do (PICSTRUCT *p, int_4 mport) {
	clk_wr(p, mport, 0x232, 0x01);
}

void clk_cal (PICSTRUCT *p, int_4 mport) {
	clk_wr(p, mport, 0x18, 0x00);
	clk_do(p, mport);
	clk_wr(p, mport, 0x18, 0x01);
	clk_do(p, mport);
}

int clk_lock_wait (PICSTRUCT *p, int_4 mport) {
	int i = 1000;		// allow up to 1 second to lock
	clk_wr(p, mport, 0x1a, 0x00);	// put lock detect to lock pin
	clk_do(p, mport);
	while ( --i) {
		if (rfxd_mrd (p, mport, FEMTO_STATUS) & FEMTO_PLL_LOCK) {
			break;
		}
		rfxd_delay(p, mport, 1000); // wait 1 ms
	}
	return i;
}

void clk_init (PICSTRUCT *p, int_4 mport) {
	int a, d, i, len, src;
	int * x;
	
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
        src = l->clkfssrc;
#else
        src = RFXD_RD(p, mport, LGSSTRUCT, clkfssrc);
#endif
	v3print ("FEMTO clkinit begin\n");

        // Set the PLL set bit to allow clock to run
        rfxd_mmsk(p, mport, FEMTO_RESET, ~FEMTO_PLL_SYNC, FEMTO_PLL_SYNC);

	if(src == 0) {
		v3print ("internal FS\n");
		x = xint;
		len = sizeof(xint)/sizeof(int);
	}
	else {
		v3print ("external FS\n");
		x = xext;
		len = sizeof(xext)/sizeof(int);
	}
	v3print(" length = %d\n", len);

	for (i = 0; i < len; i+=2){
		// get the address
		a = x[i];
		// get the data
		d  = x[i+1];
		//v3print("addr: %03x data: %02x\n", a, d);
		clk_wr(p, mport, a, d);
	}
	clk_do(p, mport);
	clk_cal(p, mport);
	v3print ("FEMTO clkinit done\n");
}

void clkXinit (PICSTRUCT *p, int_4 mport) {
  	v3print("Clock init device %d\n", p->mrev[mport-1]);
	if (p->mrev[mport-1]==FEMTO25) clk25_init(p,mport);
	else {
  		rfxd_mmsk(p, mport, FEMTO_BOARD_CTRL, ~FEMTO_REF_IN_EXT, 0);         // internal clock
		clk_init(p,mport);
	}
}

// set the parameters realted to sample rate change
// assumes that clk_init was called previously
// requires that a clk_lock_wait and dcm_init be called before clock is used

int clk_set_rate_params(PICSTRUCT *p, int_4 mport, int r, int b, int a, int n0, int n1, int n2) {
	int r_lo, r_hi, b_lo, b_hi, n1_hi, n1_lo, n2_hi, n2_lo;
	int temp;

	r_lo  =  r       & 0xff;
	r_hi  = (r >> 8) & 0xff;
	b_lo  =  b       & 0xff;
	b_hi  = (b >> 8) & 0xff;
	n2_hi = n2/2;
	n2_lo = n2 - n2_hi;	// lo will be 1 more than hi if odd
	n1_hi = n1/2;
	n1_lo = n1 - n1_hi;
	n2_hi--; n2_lo--;
	n1_hi--; n1_lo--;

	v3print("r=%d rhi=%02x rlo=%02x b=%d bhi=%02x blo=%02x a=%d\n", r, r_hi, r_lo, b, b_hi, b_lo, a);
	v3print("n0=%d n1=%02x n1hi=%02x n1lo=%02x n2=%02x n2hi=%02x n2lo=%02x\n", n0, n1, n1_hi, n1_lo, n2, n2_hi, n2_lo);
#if 1
	clk_wr(p, mport, 0x11, r_lo);
	clk_wr(p, mport, 0x12, r_hi);
	clk_wr(p, mport, 0x13, a);
	clk_wr(p, mport, 0x14, b_lo);
	clk_wr(p, mport, 0x15, b_hi);

	// set up vo divider
	clk_wr(p, mport, 0x1e0, n0-2);
	// setup n1 divider even if it is bypassed
	clk_wr(p, mport, 0x199, (n1_hi<<4) | n1_lo);
	clk_wr(p, mport, 0x19e, (n1_hi<<4) | n1_lo);
	// and n2 divider
	clk_wr(p, mport, 0x19b, (n2_hi<<4) | n2_lo);
	clk_wr(p, mport, 0x1a0, (n2_hi<<4) | n2_lo);
	// check if need for bypass
	temp = 0;
	if (n1 == 1) temp = temp | 0x10;
	if (n2 == 1) temp = temp | 0x20;
	clk_wr(p, mport, 0x19c, temp);
	clk_wr(p, mport, 0x1a1, temp);

	clk_do(p, mport);
	clk_cal(p, mport);
#endif
	v3print ("FEMTO rate change done\n");
	return 0;
}

int iseven (int n) {
	return ((n-n/2) == 0);
}

#define ULIMIT	2950000000.
#define LLIMIT	2550000000.
#define PRESCALE 16

int clk_rate ( PICSTRUCT *p, int_4 mport, int rate ) {
	int n, n0, n1, n2, a, b, r;
	int fpd, sq;
	double fvco;

	v3print("Clock rate %d\n", rate);

	for(n0 = 2; n0 <= 6; n0++){
		for (n2 = 1; n2 <= 32; n2++){
			for ( n1 = 1; n1 <= 32; n1++){
				n = n0*n1*n2;
				fvco = n*(double)rate;
				if (( fvco >= LLIMIT) && (fvco <= ULIMIT)){
					if      ( (rate%25000) == 0)
						fpd = 100000;
					else if ( (rate%20000) == 0)
						fpd = 80000;
					else if ( (rate%10000) == 0)
						fpd = 40000;
					else 
						continue;
					sq = ((n2%2) == 0) || ( ((n1%2)==0) && (n2==1) );
					n = fvco/fpd;
					b = n/PRESCALE;
					a = n - (b*PRESCALE);
					r = 10000000/fpd;
					if (sq & (b < 8192)){
						clk_set_rate_params(p, mport, r, b, a, n0, n1, n2);
						return 0;
					}
				}
			}
		}
	}
	return -1;
}


// change the reference input source
// if external and internal clock is not the same then 9517 needs new parameters
//
// The SPI clocks will come from the 8305 output
// If a detector is implemented to detect loss of external clock then the
// SPI clock should be switched to the internal clock, FEMTO_SPI_CLK_SEL=0 
// Note the flashing LED is off for ext ref as it comes from the internal clock
// before the 8305


int clk25_ref_src ( PICSTRUCT *p, int_4 mport, int intext ) {

	if ( intext == 0 ) {
		// turn on internal osc
		rfxd_mmsk (p, mport, FEMTO_BOARD_CTRL, ~FEMTO_OSCEN, FEMTO_OSCEN);
		// select internal 10 MHz clock
		//rfxd_mmsk (p, mport, FEMTO_BOARD_CTRL, ~FEMTO_REF_IN_EXT, 0);
		// select clock from 8305 driver
		rfxd_mmsk (p, mport, FEMTO_RESET, ~FEMTO_SPI_CLK_SEL, FEMTO_SPI_CLK_SEL);
		vprint("Select internal clock\n");
	} else {
		// select external clock and set freq to 10 MHz clk
		rfxd_mmsk (p, mport, FEMTO_BOARD_CTRL, ~FEMTO_REF_IN_EXT, FEMTO_REF_IN_EXT);
		// select clock from 8305 driver
		//rfxd_mmsk (p, mport, FEMTO_RESET, ~FEMTO_SPI_CLK_SEL, FEMTO_SPI_CLK_SEL);
		// turn off internal osc
		rfxd_mmsk (p, mport, FEMTO_BOARD_CTRL, ~FEMTO_OSCEN, 0);
		vprint("Select external clock\n");
	}
        return (0);
}


int clk_ref_src ( PICSTRUCT *p, int_4 mport, int intext ) {

	if ( intext == 0 ) {
		// turn on internal osc
		rfxd_mmsk (p, mport, FEMTO_BOARD_CTRL, ~FEMTO_OSCEN, FEMTO_OSCEN);
		// select internal 10 MHz clock
		rfxd_mmsk (p, mport, FEMTO_BOARD_CTRL, ~FEMTO_REF_IN_EXT, 0);
		// select clock from 8305 driver
		rfxd_mmsk (p, mport, FEMTO_RESET, ~FEMTO_SPI_CLK_SEL, FEMTO_SPI_CLK_SEL);
		vprint("Select internal clock\n");
	} else {
		// select external clock and set freq to 10 MHz clk
		rfxd_mmsk (p, mport, FEMTO_BOARD_CTRL, ~FEMTO_REF_IN_EXT, FEMTO_REF_IN_EXT);
		// select clock from 8305 driver
		rfxd_mmsk (p, mport, FEMTO_RESET, ~FEMTO_SPI_CLK_SEL, FEMTO_SPI_CLK_SEL);
		// turn off internal osc
		rfxd_mmsk (p, mport, FEMTO_BOARD_CTRL, ~FEMTO_OSCEN, 0);
		vprint("Select external clock\n");
	}
        return (0);
}

// this doesn't do anything yet 
int clk_fs_src (PICSTRUCT *p, int_4 mport, int intext) {
	int temp;
	int rate;
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	l->clkfssrc = (char) intext;
	rate = l->clkrate;
#else
	RFXD_WR(p, mport, LGSSTRUCT, clkfssrc, intext);
	rate = 	RFXD_RDI(p, mport, LGSSTRUCT, clkrate);
 
#endif
	if (intext == 0) {
		v3print("internal clock\n");
		// setup al the registers
		clkXinit(p, mport);
		// change the rate, waits are done in calling routine
		clk_rate(p, mport, rate);
	} else {
		v3print("external clock\n");
		clkXinit(p, mport);
	}
	return 0;
}
	
// DAC programming

// check this conversion since the numbers are left justified in the 16 bits
int v2int (PICSTRUCT *p, int_4 mport, float val) {
	int LVL;
	int version;
	
	version = (int)rfxd_mrd(p,mport, FEMTO_REV);
 
	if((version >= 4) && (version <= 7)) LVL = (int) ((val * 65535) / 5) ;
	else                                 LVL = (int) ((val * 65535) / 3.3) ;
	//printf ("level: %f LVL: %04x\n", val, LVL);
	return LVL;
}

void dac_spi_start (PICSTRUCT *p, int_4 mport) {
        rfxd_mwr (p, mport, FEMTO_SPI, FEMTO_PRG_DAC);
        rfxd_mwr (p, mport, FEMTO_SPI, 0);
}

void dac_wr (PICSTRUCT *p, int_4 mport, int_4 val) {
	rfxd_mwrdword (p, mport, FEMTO_SPI4, val);
	dac_spi_start(p, mport);
}


// this can be done in one command
void dac_ldacn (PICSTRUCT *p, int_4 mport) {
	rfxd_mmsk (p, mport, FEMTO_DAC_CTRL, ~FEMTO_LDACN, 0);
        rfxd_mmsk (p, mport, FEMTO_DAC_CTRL, ~FEMTO_LDACN, FEMTO_LDACN);
}

void dac_soft_reset (PICSTRUCT *p, int_4 mport) {
	v3print("soft reset\n");
	rfxd_mmsk (p, mport, FEMTO_DAC_CTRL, ~(FEMTO_DC_RST_N | FEMTO_LDACN), FEMTO_DC_RST_N | FEMTO_LDACN);
	// Reset
	dac_wr (p, mport, 0x07000000);
}

void dac_control_register (PICSTRUCT *p, int_4 mport) {
	int version;
	version = rfxd_mrd(p,mport,FEMTO_REV);
	if((version >= 4) && (version <= 7))
		dac_wr (p, mport, 0x08000001); // turn on internal reference
	else    dac_wr (p, mport, 0x08000000); // turn on external reference
	v3print("control reg write, Version = %d\n", version);
}

void dac_data_register ( PICSTRUCT *p, int_4 mport, int CHN,  int x ) {
	// 4 dont care, 4 command bits, 4 address bits, 16 data bits
	dac_wr (p, mport, (CHN << 20) | (x << 4)); 
}

int dac_setchan ( PICSTRUCT *p, int_4 mport, int chan, float level) {
	int LVL;
	if ( (chan < 0) || (chan > 7) ) {
		v3print("Error: bad channnel\n");
		return 1;
	}
	LVL = v2int(p, mport, level);
	if ( (LVL < 0) || (LVL > 65535 )) {
		v3print("Error: bad level, LVL= %04x\n",LVL);
		return 1;
	}
	v3print("Channel %d, Value: %d\n", chan, LVL);
	dac_data_register(p, mport, chan, LVL);
	dac_ldacn(p, mport);
	return 0;
}

// DAC storage is floats - is this a problem??
void dac_setall(PICSTRUCT *p, int_4 mport) {
#if MEMSTRUCT
	LGSSTRUCT *l;

	l = &femto[p->devno][mport];
	dac_setchan(p, mport, 0, l->VCM1);
	dac_setchan(p, mport, 1, l->VCM2);
	dac_setchan(p, mport, 2, l->VCMWB);
	dac_setchan(p, mport, 3, l->REF_TUNE);
	dac_setchan(p, mport, 4, l->VREFP);
	dac_setchan(p, mport, 5, l->VREFN);
	dac_setchan(p, mport, 6, l->LOBIAS);
	dac_setchan(p, mport, 7, l->RFBIAS);
#else
	dac_setchan(p, mport, 0, RFXD_RDF(p, mport, LGSSTRUCT, VCM1));
	dac_setchan(p, mport, 1, RFXD_RDF(p, mport, LGSSTRUCT, VCM2));
	dac_setchan(p, mport, 2, RFXD_RDF(p, mport, LGSSTRUCT, VCMWB));
	dac_setchan(p, mport, 3, RFXD_RDF(p, mport, LGSSTRUCT, REF_TUNE));
	dac_setchan(p, mport, 4, RFXD_RDF(p, mport, LGSSTRUCT, VREFP));
	dac_setchan(p, mport, 5, RFXD_RDF(p, mport, LGSSTRUCT, VREFN));
	dac_setchan(p, mport, 6, RFXD_RDF(p, mport, LGSSTRUCT, LOBIAS));
	dac_setchan(p, mport, 7, RFXD_RDF(p, mport, LGSSTRUCT, RFBIAS));
#endif
}

void dac_init  (PICSTRUCT *p, int_4 mport) {
	int CHN, x;

#if MEMSTRUCT
	LGSSTRUCT *l;

	l = &femto[p->devno][mport];
	l->VCM1 = 1.2;     	/* VCM1 */
	l->VCM2	= 1.6; 		/* VCM2 */
	l->VCMWB = 2.05;      	/* VCMWB */
     	l->REF_TUNE = 1.65;	/* REF_TUNE */
     	l->VREFP = 1.45;	/* VREFP */
     	l->VREFN = 1.55;	/* VREFN */
     	l->LOBIAS = 1.5;	/* LOBIAS */
	l->RFBIAS = 2.0;     	/* set RFBIAS */
#else
	RFXD_WRF(p, mport, LGSSTRUCT, VCM1,  1.2);
	RFXD_WRF(p, mport, LGSSTRUCT, VCM2,  1.6);
	RFXD_WRF(p, mport, LGSSTRUCT, VCMWB, 2.05);
	RFXD_WRF(p, mport, LGSSTRUCT, REF_TUNE, 1.65);
	RFXD_WRF(p, mport, LGSSTRUCT, VREFP,  1.45);
	RFXD_WRF(p, mport, LGSSTRUCT, VREFN,  1.55);
	RFXD_WRF(p, mport, LGSSTRUCT, LOBIAS, 1.5);
	RFXD_WRF(p, mport, LGSSTRUCT, RFBIAS, 2.0);
#endif
	// Soft reset
	dac_soft_reset(p, mport);
	// Control register
	dac_control_register(p, mport);
	// data
	dac_setall(p, mport);
	v3print("DACINIT done\n");
}


// #####
//RFIC programming


void disp (PICSTRUCT *p, int val, int count) {
	int mask, i;
	
	mask = 1 << (count-1);
	for (i = 0; i < count;  i++) {
		if ((val & mask) != 0) {
			v3print("1");
		} else {
			v3print("0");
		}
		mask  = mask >> 1;
	}
	v3print(" ");
}

void  rad_disp (PICSTRUCT * p, int_4 mport) {
#if MEMSTRUCT
	LGSSTRUCT *l;
	
	l = &femto[p->devno][mport];
	disp(p, l->LNA, 3);
	disp(p, l->BAND, 2);
	disp(p, l->FREQ, 5);
	disp(p, l->VGA, 5);
	disp(p, l->FIL1, 2);
	disp(p, l->FIL2, 3);
	disp(p, l->FIL3, 3);
	disp(p, l->FIL4, 3);
	disp(p, l->FIL5, 2);
	disp(p, l->CBIAS, 4);
	disp(p, l->GBIAS, 4);
	disp(p, l->WFREQ, 4);
	disp(p, l->BIAS1, 4);
	disp(p, l->BIAS2, 4);
	disp(p, l->BIAS3, 4);
	disp(p, l->BIAS4, 4);
	disp(p, l->BIAS5, 4);
	disp(p, l->BIAS6, 4);
	v3print("\n");
#else
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, LNA),   3);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, BAND),  2);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, FREQ),  5);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, VGA),   5);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, FIL1),  2);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, FIL2),  3);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, FIL3),  3);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, FIL4),  3);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, FIL5),  2);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, CBIAS), 4);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, GBIAS), 4);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, WFREQ), 4);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, BIAS1), 4);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, BIAS2), 4);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, BIAS3), 4);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, BIAS4), 4);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, BIAS5), 4);
	disp(p, RFXD_RD(p, mport, LGSSTRUCT, BIAS6), 4);
	v3print("\n");
#endif
}

void rfic_start_spi (PICSTRUCT *p, int_4 mport) {
	rfxd_mwr (p, mport, FEMTO_SPI, FEMTO_PRG_RF );
	rfxd_mwr (p, mport, FEMTO_SPI, 0); 
}

int rficload (PICSTRUCT *p, int_4 mport) {
	int RFIC_WORD1, RFIC_WORD2;
#if MEMSTRUCT
	LGSSTRUCT *l;

	l = &femto[p->devno][mport];
	RFIC_WORD1 =         ( l->LNA          << (61-32)) \
	                   | ( l->BAND         << (59-32)) \
			   | ((l->FREQ ^ 0x10) << (54-32)) \
			   | ((l->VGA  ^ 0x01) << (49-32)) \
			   | ( l->FIL1         << (47-32)) \
			   | ( l->FIL2         << (44-32)) \
			   | ( l->FIL3         << (41-32)) \
			   | ( l->FIL4         << (38-32)) \
			   | ( l->FIL5         << (36-32)) \
			   | ((l->CBIAS ^ 0x08)<< (32-32)) ;

	RFIC_WORD2 =         ( l->GBIAS         << 28) \
			   | ((l->WFREQ ^ 0x08) << 24) \
			   | ( l->BIAS1         << 20) \
			   | ( l->BIAS2         << 16) \
			   | ( l->BIAS3         << 12) \
			   | ( l->BIAS4         << 8 ) \
			   | ( l->BIAS5         << 4 ) \
			   | ( l->BIAS6         << 0 ) ;
	
#else
	RFIC_WORD1 =         ( RFXD_RD(p, mport, LGSSTRUCT, LNA)          << (61-32)) \
	                   | ( RFXD_RD(p, mport, LGSSTRUCT, BAND)         << (59-32)) \
			   | ((RFXD_RD(p, mport, LGSSTRUCT, FREQ) ^ 0x10) << (54-32)) \
			   | ((RFXD_RD(p, mport, LGSSTRUCT, VGA)  ^ 0x01) << (49-32)) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, FIL1)         << (47-32)) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, FIL2)         << (44-32)) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, FIL3)         << (41-32)) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, FIL4)         << (38-32)) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, FIL5)         << (36-32)) \
			   | ((RFXD_RD(p, mport, LGSSTRUCT, CBIAS) ^ 0x08)<< (32-32)) ;

	RFIC_WORD2 =         ( RFXD_RD(p, mport, LGSSTRUCT, GBIAS)         << 28) \
			   | ((RFXD_RD(p, mport, LGSSTRUCT, WFREQ) ^ 0x08) << 24) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, BIAS1)         << 20) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, BIAS2)         << 16) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, BIAS3)         << 12) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, BIAS4)         << 8 ) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, BIAS5)         << 4 ) \
			   | ( RFXD_RD(p, mport, LGSSTRUCT, BIAS6)         << 0 ) ;
#endif	   
	v3print("rfic1: %08X rfic2: %08x\n", RFIC_WORD1, RFIC_WORD2);
	rad_disp(p, mport);
	rfxd_mwrdword (p, mport, FEMTO_SPI8, RFIC_WORD1);
	rfxd_mwrdword (p, mport, FEMTO_SPI4, RFIC_WORD2);
	rfic_start_spi(p, mport);
	return 0;
}

int radlna (PICSTRUCT *p, int_4 mport, int lna) {
#if MEMSTRUCT
	LGSSTRUCT *l;

	l = &femto[p->devno][mport];

	l->LNA = lna & 0x7;
#else
	RFXD_WR(p, mport, LGSSTRUCT, LNA, lna & 0x7);
#endif
	rficload(p, mport);
	return 0;
}

int radlna_noupdate (PICSTRUCT *p, int_4 mport, int lna) {
#if MEMSTRUCT
	LGSSTRUCT *l;

	l = &femto[p->devno][mport];

	l->LNA = lna & 0x7;
#else
	RFXD_WR(p, mport, LGSSTRUCT, LNA, lna & 0x7);
#endif
	return 0;
}

int radfl (PICSTRUCT *p, int_4 mport, int filter) {
#if MEMSTRUCT
	LGSSTRUCT *l;

	l = &femto[p->devno][mport];

	l->FREQ = filter & 0x1f;
//	l->WFREQ = filter & 0xf;
#else
	RFXD_WR(p, mport, LGSSTRUCT, FREQ, filter & 0x1f);
#endif
	rficload(p, mport);
	return 0;
}

void radreset (PICSTRUCT *p, int_4 mport) {
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	// assert reset
	rfxd_mmsk (p,  mport, FEMTO_BOARD_CTRL, ~FEMTO_RF_CLRN, 0x00);
	// clear reset
	rfxd_mmsk (p,  mport, FEMTO_BOARD_CTRL, ~FEMTO_RF_CLRN, FEMTO_RF_CLRN);
	// set all variables to 0 state, presumed reset value of the bits
	l->LNA = 3;
	// CDMA
	l->BAND =0x00;
	// nominal bandwdith
	l->FREQ = 0x0f;
	// VGA to min GAIN
	l->VGA = 0x01;
	//filter states to MAX gain
	l->FIL1 = 0x00;
	l->FIL2 = 0x00;
	l->FIL3 = 0x00;
	l->FIL4 = 0x00;
	l->FIL5 = 0x00;
	// CDMA BIAS (mid)
	l->CBIAS = l->SCBIAS = 0x8;
	// GSM BIAS (off)
	l->GBIAS = l->SGBIAS = 0x00;
	// Wideband freq control - half band
	l->WFREQ = 0x8;
	// 20 MHz Bias
	l->BIAS1 = l->SBIAS1 = 0x00;
	l->BIAS2 = l->SBIAS2 = 0x00;
	l->BIAS3 = l->SBIAS3 = 0x00;
	l->BIAS4 = l->SBIAS4 = 0x00;
	l->BIAS5 = l->SBIAS5 = 0x00;
	l->BIAS6 = l->SBIAS6 = 0x00;
#else
	// assert reset
	rfxd_mmsk (p,  mport, FEMTO_BOARD_CTRL, ~FEMTO_RF_CLRN, 0x00);
	// clear reset
	rfxd_mmsk (p,  mport, FEMTO_BOARD_CTRL, ~FEMTO_RF_CLRN, FEMTO_RF_CLRN);
	// set all variables to 0 state, presumed reset value of the bits
	RFXD_WR(p, mport, LGSSTRUCT, LNA, 3);
	// CDMA
	RFXD_WR(p, mport, LGSSTRUCT, BAND, 0x00);
	// nominal bandwdith
	RFXD_WR(p, mport, LGSSTRUCT, FREQ, 0x0f);
	// VGA to min GAIN
	RFXD_WR(p, mport, LGSSTRUCT, VGA, 0x01);
	//filter states to MAX gain
	RFXD_WR(p, mport, LGSSTRUCT, FIL1, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, FIL2, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, FIL3, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, FIL4, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, FIL5, 0x00);
	// CDMA BIAS (mid)
	RFXD_WR(p, mport, LGSSTRUCT, CBIAS, 0x08);
	RFXD_WR(p, mport, LGSSTRUCT, SCBIAS, 0x08);
	// GSM BIAS (off)
	RFXD_WR(p, mport, LGSSTRUCT, GBIAS, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, SGBIAS, 0x00);
	// Wideband freq control - half band
	RFXD_WR(p, mport, LGSSTRUCT, FIL1, 0x00);
	// 20 MHz Bias
	RFXD_WR(p, mport, LGSSTRUCT, BIAS1, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, BIAS2, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, BIAS3, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, BIAS4, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, BIAS5, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, BIAS6, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, SBIAS1, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, SBIAS2, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, SBIAS3, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, SBIAS4, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, SBIAS5, 0x00);
	RFXD_WR(p, mport, LGSSTRUCT, SBIAS6, 0x00);
#
#endif
}

int radmode (PICSTRUCT *p, int_4 mport, int mode) {
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	
	if (mode == 2) {		// GSM
		l->BIAS1 = 0;
		l->BIAS2 = 0;
		l->BIAS3 = 0;
		l->BIAS4 = 0;
		l->BIAS5 = 0;
		l->BIAS6 = 0;
		l->CBIAS = 0;
		l->GBIAS = l->SGBIAS;
		l->BAND  = 2;
		// add DAC specific code here
	} else if (mode == 0) {		// WCDMA
		l->BIAS1 = 0;
		l->BIAS2 = 0;
		l->BIAS3 = 0;
		l->BIAS4 = 0;
		l->BIAS5 = 0;
		l->BIAS6 = 0;
		l->CBIAS = l->SCBIAS;
		l->GBIAS = 0;
		l->BAND  = 0;
		// add DAC specific code here
	} else if (mode == 1) {		// Wide band
		l->BIAS1 = l->SBIAS1;
		l->BIAS2 = l->SBIAS2;
		l->BIAS3 = l->SBIAS3;
		l->BIAS4 = l->SBIAS4;
		l->BIAS5 = l->SBIAS5;
		l->BIAS6 = l->SBIAS6;
		l->CBIAS = 0;
		l->GBIAS = 0;
		l->BAND  = 1;
		// add DAC specific code here
	} else {			// bypass
		l->BIAS1 = 0;
		l->BIAS2 = 0;
		l->BIAS3 = 0;
		l->BIAS4 = 0;
		l->BIAS5 = 0;
		l->BIAS6 = 0;
		l->CBIAS = 0;
		l->GBIAS = 0;
		l->BAND = 3;
		v3print("invalid mode, bypass enabled \n");
	}
#else
	if (mode == 2) {		// GSM
		RFXD_WR(p, mport, LGSSTRUCT, BIAS1, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS2, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS3, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS4, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS5, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, CBIAS, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, GBIAS, RFXD_RD(p, mport, LGSSTRUCT, SGBIAS));
		RFXD_WR(p, mport, LGSSTRUCT, BAND,  0x02);
		// add DAC specific code here
	} else if (mode == 0) {		// WCDMA
		RFXD_WR(p, mport, LGSSTRUCT, BIAS1, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS2, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS3, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS4, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS5, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, GBIAS, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, CBIAS, RFXD_RD(p, mport, LGSSTRUCT, SCBIAS));
		RFXD_WR(p, mport, LGSSTRUCT, BAND,  0x00);
		// add DAC specific code here
	} else if (mode == 1) {		// Wide band
		RFXD_WR(p, mport, LGSSTRUCT, BIAS1, RFXD_RD(p, mport, LGSSTRUCT, SBIAS1));
		RFXD_WR(p, mport, LGSSTRUCT, BIAS2, RFXD_RD(p, mport, LGSSTRUCT, SBIAS2));
		RFXD_WR(p, mport, LGSSTRUCT, BIAS3, RFXD_RD(p, mport, LGSSTRUCT, SBIAS3));
		RFXD_WR(p, mport, LGSSTRUCT, BIAS4, RFXD_RD(p, mport, LGSSTRUCT, SBIAS4));
		RFXD_WR(p, mport, LGSSTRUCT, BIAS5, RFXD_RD(p, mport, LGSSTRUCT, SBIAS5));
		RFXD_WR(p, mport, LGSSTRUCT, CBIAS, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, GBIAS, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BAND,  0x01);
		// add DAC specific code here
	} else {			// bypass
		RFXD_WR(p, mport, LGSSTRUCT, BIAS1, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS2, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS3, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS4, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BIAS5, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, CBIAS, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, GBIAS, 0x00);
		RFXD_WR(p, mport, LGSSTRUCT, BAND,  0x03);
		v3print("invalid mode, bypass enabled \n");
	}
#endif
	rficload(p, mport);
	return 0;
}

// structure of gain table is attenuation, VGA, FIL1, FIL2, FIL3, FIL4, FIL5
// in terms of attenuation
// another lookup is done for each stage to determine device setting
// this allows easy change of gain plan
// should add LNA setting to this table!

int gaintbl0 [ ] = {
	0,   0, 0,  0,  0,  0, 0, \
	6,   0, 0,  0,  0,  0, 6, \
	12,  0, 0,  0,  0,  6, 6, \
	18,  0, 0,  0,  0, 12, 6, \
	24,  0, 0,  0,  6, 12, 6, \
	30,  0, 0,  0, 12, 12, 6, \
	36,  0, 0,  6, 12, 12, 6, \
	42,  0, 0, 12, 12, 12, 6, \
	48,  0, 6, 12, 12, 12, 6, \
	54,  6, 6, 12, 12, 12, 6, \
	60, 12, 6, 12, 12, 12, 6, \
	66, 18, 6, 12, 12, 12, 6, \
	72, 24, 6, 12, 12, 12, 6
};
int gaintbl1 [ ] = {
	0,   0, 0,  0,  0,  0, 0, \
	6,   0, 0,  0,  0,  0, 6, \
	12,  0, 0,  0,  0,  6, 6, \
	18,  0, 0,  0,  6,  6, 6, \
	24,  0, 0,  0, 12,  6, 6, \
	30,  0, 0,  6, 12,  6, 6, \
	36,  0, 0,  6, 12, 12, 6, \
	42,  0, 0, 12, 12, 12, 6, \
	48,  0, 6, 12, 12, 12, 6, \
	54,  6, 6, 12, 12, 12, 6, \
	60, 12, 6, 12, 12, 12, 6, \
	66, 18, 6, 12, 12, 12, 6, \
	72, 24, 6, 12, 12, 12, 6
};
int gaintbl2 [ ] = {
	0,   0, 0,  0,  0,  0, 0, \
	6,   0, 0,  0,  0,  0, 6, \
	12,  0, 0,  0,  0,  6, 6, \
	18,  0, 0,  0,  6,  6, 6, \
	24,  0, 0,  0, 12,  6, 6, \
	30,  0, 0,  6, 12,  6, 6, \
	36,  0, 0, 12, 12,  6, 6, \
	42,  0, 6, 12, 12,  6, 6, \
	48,  6, 6, 12, 12,  6, 6, \
	54, 12, 6, 12, 12,  6, 6, \
	60, 18, 6, 12, 12,  6, 6, \
	66, 24, 6, 12, 12,  6, 6, \
	72, 24, 6, 12, 12, 12, 6
};
int gaintbl3 [ ] = {
	0,   0, 0,  0,  0,  0, 0, \
	6,   0, 0,  0,  0,  0, 6, \
	12,  0, 0,  0,  0,  6, 6, \
	18,  0, 0,  0,  6,  6, 6, \
	24,  0, 0,  6,  6,  6, 6, \
	30,  0, 6,  6,  6,  6, 6, \
	36,  6, 6,  6,  6,  6, 6, \
	42,  6, 6,  6, 12,  6, 6, \
	48,  6, 6, 12, 12,  6, 6, \
	54, 12, 6, 12, 12,  6, 6, \
	60, 18, 6, 12, 12,  6, 6, \
	66, 24, 6, 12, 12,  6, 6, \
	72, 24, 6, 12, 12, 12, 6
};
int gaintbl4 [ ] = {
	0,   0, 0,  0,  0,  0, 0, \
	6,   0, 6,  0,  0,  0, 0, \
	12,  0, 6,  6,  0,  0, 0, \
	18,  0, 6, 12,  0,  0, 0, \
	24,  0, 6, 12,  6,  0, 0, \
	30,  0, 6, 12, 12,  0, 0, \
	36,  0, 6, 12, 12,  6, 0, \
	42,  0, 6, 12, 12, 12, 0, \
	48,  0, 6, 12, 12, 12, 6, \
	54,  6, 6, 12, 12, 12, 6, \
	60, 12, 6, 12, 12, 12, 6, \
	66, 18, 6, 12, 12, 12, 6, \
	72, 24, 6, 12, 12, 12, 6
};
int gaintbl5 [ ] = {
	0,   0, 0,  0,  0,  0, 0, \
	6,   0, 6,  0,  0,  0, 0, \
	12,  0, 6,  6,  0,  0, 0, \
	18,  0, 6, 12,  0,  0, 0, \
	24,  0, 6, 12,  6,  0, 0, \
	30,  0, 6, 12, 12,  0, 0, \
	36,  0, 6, 12, 12,  6, 0, \
	42,  0, 6, 12, 12,  6, 6, \
	48,  6, 6, 12, 12,  6, 6, \
	54, 12, 6, 12, 12,  6, 6, \
	60, 18, 6, 12, 12,  6, 6, \
	66, 24, 6, 12, 12,  6, 6, \
	72, 24, 6, 12, 12, 12, 6
};

int * gaintblptr[] = {
	gaintbl0, \
	gaintbl1, \
	gaintbl2, \
	gaintbl3, \
	gaintbl4, \
	gaintbl5 
};

int gaintblindex = 2;

int radvga(int val){
	if      (val ==  0) val = 0x10;
	else if (val ==  6) val = 0x08;
	else if (val == 12) val = 0x04;
	else if (val == 18) val = 0x02;
	else                val = 0x01;
	return val;
}
	
int radgn1(int val){
	if (val == 6) 	val = 3;
	else 		val = 0;
	return val;
}

int radgn2(int val){
	if      (val ==  6)	val = 6;
	else if (val == 12)	val = 5;
	else 			val = 0;
	return val;
}

int radgni ( PICSTRUCT *p, int_4 mport, int_4 vga, int_4 fil1, int_4 fil2, int_4 fil3, int_4 fil4, int_4 fil5) {
	v3print("RADGNIN  - VGA:%02x FIL1: %d FIL2: %d FIL3: %d FIL4: %d FIL5: %d\n", \
       		vga, fil1,fil2,fil3,fil4,fil5);
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	l->VGA  = radvga(vga);
	l->FIL1 = radgn1(fil1);
	l->FIL2 = radgn2(fil2);
	l->FIL3 = radgn2(fil3);
	l->FIL4 = radgn2(fil4);
	l->FIL5 = radgn1(fil5);
#else
	RFXD_WR(p, mport, LGSSTRUCT, VGA,  radvga(vga));
	RFXD_WR(p, mport, LGSSTRUCT, FIL1, radgn1(fil1));
	RFXD_WR(p, mport, LGSSTRUCT, FIL2, radgn2(fil2));
	RFXD_WR(p, mport, LGSSTRUCT, FIL3, radgn2(fil3));
	RFXD_WR(p, mport, LGSSTRUCT, FIL4, radgn2(fil4));
	RFXD_WR(p, mport, LGSSTRUCT, FIL5, radgn1(fil5));
#endif
	rficload(p, mport);
	v3print("RADGNOUT - VGA:%02x FIL1: %d FIL2: %d FIL3: %d FIL4: %d FIL5: %d\n", \
       		radvga(vga), radgn1(fil1),radgn2(fil2),radgn2(fil3),radgn2(fil4),radgn1(fil5));
	return 0;
}

int radgn ( PICSTRUCT *p, int_4 mport, int gain ) {
	int i, ptr;
	int * gaintbl;
	int_4 vga, fil1, fil2, fil3, fil4, fil5;
	// pick the desired gain table
	gaintbl = gaintblptr[gaintblindex];
	for (i = 0; i < 13; i++){
		ptr = i * 7;
		if (gain == gaintbl[ptr++]) {
			v3print ("RADGN -  attn: %d ptr: %d tble#: %d \n", gain, ptr, gaintblindex);
			vga  = (gaintbl[ptr++]);
			fil1 = (gaintbl[ptr++]);
			fil2 = (gaintbl[ptr++]);
			fil3 = (gaintbl[ptr++]);
			fil4 = (gaintbl[ptr++]);
			fil5 = (gaintbl[ptr++]);
			radgni(p, mport, vga, fil1, fil2, fil3, fil4, fil5);
			return 0;
		}
	}
	v3print("Error in gain setting\n");
	return -1;
}

int radgndis(PICSTRUCT *p, int_4 mport, int vga, int fil1, int fil2, int fil3, int fil4, int fil5) {
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	l->VGA  = radvga(vga);
	l->FIL1 = radgn1(fil1);
	l->FIL2 = radgn2(fil2);
	l->FIL3 = radgn2(fil3);
	l->FIL4 = radgn2(fil4);
	l->FIL5 = radgn1(fil5);
	rficload(p, mport);
	v3print("VGA:%02x FIL1: %d FIL2: %d FIL3: %d FIL4: %d FIL5: %d", l->VGA, l->FIL1, l->FIL2, l->FIL3, l->FIL4, l->FIL5);
#else
	RFXD_WR(p, mport, LGSSTRUCT, VGA,  radvga(vga));
	RFXD_WR(p, mport, LGSSTRUCT, FIL1, radgn1(fil1));
	RFXD_WR(p, mport, LGSSTRUCT, FIL2, radgn2(fil2));
	RFXD_WR(p, mport, LGSSTRUCT, FIL3, radgn2(fil3));
	RFXD_WR(p, mport, LGSSTRUCT, FIL4, radgn2(fil4));
	RFXD_WR(p, mport, LGSSTRUCT, FIL5, radgn1(fil5));
	rficload(p, mport);
	v3print("VGA:%02x FIL1: %d FIL2: %d FIL3: %d FIL4: %d FIL5: %d\n", \
       		RFXD_RD(p, mport, LGSSTRUCT, VGA),  \
       		RFXD_RD(p, mport, LGSSTRUCT, FIL1), \
       		RFXD_RD(p, mport, LGSSTRUCT, FIL2), \
       		RFXD_RD(p, mport, LGSSTRUCT, FIL3), \
       		RFXD_RD(p, mport, LGSSTRUCT, FIL4), \
       		RFXD_RD(p, mport, LGSSTRUCT, FIL5) );
#endif
	return 0;
}

		
int radbias (PICSTRUCT *p, int_4 mport, int_4 bias1,int_4 bias2,int_4 bias3,int_4 bias4,int_4 bias5,int_4 bias6) {
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	
	v3print("Set bias: B1: %d B2: %d B3: %d B4: %d B5: %d B6: %d BAND: %d\n", bias1, bias2, bias3, bias4, bias5, bias6, l->BAND);
	if (l->BAND == 2) {		//GSM
		l->SGBIAS = bias1;
		l->GBIAS = l->SGBIAS;
	} else if (l->BAND == 0) {	// WCDMA
		l->SCBIAS = bias1;
		l->CBIAS = l->SCBIAS;
	} else if (l->BAND == 1) {	// Wide Band
		l->SBIAS1 = bias1;
		l->SBIAS2 = bias2;
		l->SBIAS3 = bias3;
		l->SBIAS4 = bias4;
		l->SBIAS5 = bias5;
		l->SBIAS6 = bias6;
		l->BIAS1 = l->SBIAS1;
		l->BIAS2 = l->SBIAS2;
		l->BIAS3 = l->SBIAS3;
		l->BIAS4 = l->SBIAS4;
		l->BIAS5 = l->SBIAS5;
		l->BIAS6 = l->SBIAS6;
	}
	rficload(p, mport);
#else
	int band;
	
	band = RFXD_RD(p, mport, LGSSTRUCT,  BAND);
	v3print("Set bias: B1: %d B2: %d B3: %d B4: %d B5: %d B6: %d BAND: %d\n", bias1, bias2, bias3, bias4, bias5, bias6, band);
	if (band == 2) {		//GSM
		RFXD_WR(p, mport, LGSSTRUCT,  GBIAS, bias1);
		RFXD_WR(p, mport, LGSSTRUCT, SGBIAS, bias1);
	} else if (band == 0) {	// WCDMA
		RFXD_WR(p, mport, LGSSTRUCT,  CBIAS, bias1);
		RFXD_WR(p, mport, LGSSTRUCT, SCBIAS, bias1);
	} else if (band == 1) {	// Wide Band
		RFXD_WR(p, mport, LGSSTRUCT,  BIAS1, bias1);
		RFXD_WR(p, mport, LGSSTRUCT, SBIAS1, bias1);
		RFXD_WR(p, mport, LGSSTRUCT,  BIAS2, bias2);
		RFXD_WR(p, mport, LGSSTRUCT, SBIAS2, bias2);
		RFXD_WR(p, mport, LGSSTRUCT,  BIAS3, bias3);
		RFXD_WR(p, mport, LGSSTRUCT, SBIAS3, bias3);
		RFXD_WR(p, mport, LGSSTRUCT,  BIAS4, bias4);
		RFXD_WR(p, mport, LGSSTRUCT, SBIAS5, bias4);
		RFXD_WR(p, mport, LGSSTRUCT,  BIAS5, bias5);
		RFXD_WR(p, mport, LGSSTRUCT, SBIAS5, bias5);
		RFXD_WR(p, mport, LGSSTRUCT,  BIAS6, bias6);
		RFXD_WR(p, mport, LGSSTRUCT, SBIAS6, bias6);
	}
	rficload(p, mport);
#endif
	return 0;
}

// synthesizer programming
//

// Reference divider ratio (10 MHz in)
//	#define R (10000/200)
// Reference divider ratio (25 MHz in)
// Control (CPx=0, Ax=1, PSC=16)
#define SYNTH_PLL_A 3
#define SYNTH_CP 0
#define SYNTH_PSC 0
#define SYNTH_PRESCALE 16

//definitions for STW81101
int SYNTH_MO[6]   = {      7,      8,     1,      2,     4,     5 };
int SYNTH_MULT[6] = {      1,      1,     2,      2,     4,     4 };
float SYNTH_LO[6] = { 1650.0, 1900.0, 825.0,  950.0, 412.5, 475.0 };
float SYNTH_HI[6] = { 1950.0, 2200.0, 975.0, 1100.0, 487.5, 550.0 };

void start_spi_syn1 (PICSTRUCT *p, int_4 mport) {
	rfxd_mwr (p, mport, FEMTO_SPI,FEMTO_PRG_SYN); 
	rfxd_mwr (p, mport, FEMTO_SPI,0 );
}

void syn1wr(PICSTRUCT *p, int_4 mport, int val){
	rfxd_mwrdword (p, mport, FEMTO_SPI4,  val);
	start_spi_syn1(p, mport);
}

int syn1fr (PICSTRUCT *p, int_4 mport, real_8 freq) {
#if MEMSTRUCT
	LGSSTRUCT *l;
#endif

        // set the reference frequency
        int clkref = 10;

	int i, ifreq, MPY ;
	float lo, hi;
	int SYNTH_B, SYNTH_A, SYNTH_R, SYNTH_0, SYNTH_1, SYNTH_MODE;

	SYNTH_R = (clkref*1000/200);

	for (i = 0; i < 6; i++){
		lo = SYNTH_LO[i];
		hi = SYNTH_HI[i];
		//printf("lo: %f hi: %f\n", lo,hi);
		if ( ( freq >= lo ) && ( freq <= hi ) ) {
			SYNTH_MODE = SYNTH_MO[i];
			MPY = SYNTH_MULT[i];
			break;
		}
	}
	if (i == 6) {
		v3print("Bad frequency, limits are: \n");
		for (i = 0; i < 6; i++) {
			lo = SYNTH_LO[i];
			hi = SYNTH_HI[i];
			v3print("lo: %f hi: %f\n", lo,hi);
		}
		return -1;
	}

// FIX this may need special treatment because it is a double
#if MEMSTRUCT
	l = &femto[p->devno][mport];
	l->RFFREQ = freq;
#else
	RFXD_WRD(p, mport, LGSSTRUCT,  RFFREQ, freq);
#endif
	//	// round the frequency up (1930.2 gets passed in as 1930.199951)
//	freq = freq + .05;
	freq = freq * 10 * MPY;
	ifreq = (int)freq;			// need to convert to int here
	SYNTH_B = ifreq / SYNTH_PRESCALE;
	SYNTH_A = ifreq - SYNTH_B * SYNTH_PRESCALE;
	v3print ("freq: %d B: %d A:%d\n", ifreq,SYNTH_B,SYNTH_A);
	SYNTH_0 = (0 << 24) | (SYNTH_R << 14) | (SYNTH_PLL_A << 12) | (SYNTH_CP << 9) | (SYNTH_PSC << 8) ;
	SYNTH_1 = (1 << 24) | (SYNTH_MODE << 17) | (SYNTH_B << 5) | SYNTH_A ;

	v3print("syn: %08x %08x\n", SYNTH_0, SYNTH_1);

	syn1wr (p, mport, SYNTH_0);
	syn1wr (p, mport, SYNTH_1);

	// VCO Calibrate
	syn1wr (p, mport, SYNTH_0 | 0x40 );
	syn1wr (p, mport, SYNTH_0 );
	return 0;
}

// call this after changing clk_ref above. This will reprogram the synth

int synXref (PICSTRUCT *p, int_4 mport) {
	real_8 freq;
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	freq = l->RFFREQ; 
#else
	freq = RFXD_RDD(p, mport, LGSSTRUCT, RFFREQ);
#endif

  	v3print("Clock ref change %d\n", p->mrev[mport-1]);
	if      (p->mrev[mport-1]==FEMTO2)  synfr  (p,mport, &freq);
	else if (p->mrev[mport-1]==FEMTO25) syn25fr(p,mport, &freq);
	else                                syn1fr (p,mport, freq);
	return 0;
}

int synXinit (PICSTRUCT *p, int_4 mport ) {
	real_8 freq = 1000.;

	// set up default frequency
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	l->RFFREQ = freq;
#else
	RFXD_WRD(p, mport, LGSSTRUCT,  RFFREQ, freq) ;
#endif
  	v3print("Syn init %d\n", p->mrev[mport-1]);
	if (p->mrev[mport-1]==FEMTO2) syninit(p,mport);
	else if (p->mrev[mport-1]==FEMTO25){ 
		syn25init(p,mport);
		syn25fr(p,mport,&freq);
	}
	else syn1fr(p, mport, freq);
	v3print ("Synth init DONE\n");
	return 0;
}

int minmax_reset(PICSTRUCT *p, int_4 mport){
	rfxd_mmsk( p, mport, FEMTO_RESET, ~FEMTO_MINMAXRES,FEMTO_MINMAXRES);
	rfxd_mmsk( p, mport, FEMTO_RESET, ~FEMTO_MINMAXRES,0);
	return 0;
}
// return min and max values scaled to +- 32768 as full scale
int minmax_get(PICSTRUCT *p, int_4 mport, int_4 *imax, int_4 *imin, int_4 *qmax, int_4 *qmin){
	int imaxhi;
	imaxhi = rfxd_mrd (p, mport, I_MAXHI);
	*imax = ((rfxd_mrd(p, mport, I_MAXHI) << 24) | (rfxd_mrd(p, mport, I_MAXLO) << 16))/65536;
	*imin = ((rfxd_mrd(p, mport, I_MINHI) << 24) | (rfxd_mrd(p, mport, I_MINLO) << 16))/65536;
	*qmax = ((rfxd_mrd(p, mport, Q_MAXHI) << 24) | (rfxd_mrd(p, mport, Q_MAXLO) << 16))/65536;
	*qmin = ((rfxd_mrd(p, mport, Q_MINHI) << 24) | (rfxd_mrd(p, mport, Q_MINLO) << 16))/65536;
	return 0;
}

int minmax_print (PICSTRUCT *p, int_4 mport)
{
	int_4 imax, imin, qmax, qmin;
	int_4 imaxhi, imaxlo;
	int_4 iavg, qavg;

	minmax_reset(p, mport);
	// allow time to see some mins and maxes
	rfxd_delay(p, mport, 10000);
	// first need to capture data
	minmax_get(p,  mport, &imax, &imin, &qmax, &qmin);
	iavg = (imax + imin)/2;
	qavg = (qmax + qmin)/2;
	//v3print("IMAX: %08x IMIN: %08x IAVG: %08x QMAX: %08x QMIN: %08x QAVG: %08x\n", imax, imin, iavg, qmax, qmin, qavg);
	vprint("IMAX: %d IMIN: %d IAVG: %d QMAX: %d QMIN: %d QAVG: %d\n", imax, imin, iavg, qmax, qmin, qavg);
	return 0;
}
//
// the following routine is used to adjust the mixer offset voltage to the RFIC
// this is used in wideband filter mode. The algorithm uses a min/max peak detect
// in the FPGA to determine a "DC" level. It then uses successive approximation
// to adjust the offset voltage
//
#define OLD_MINMAX	0


int radcal (PICSTRUCT *p, int_4 mport){
	int i;
	volatile int j,k;
	float idacv = 1.65;
	float qdacv = 1.65;
	float daci = .825;
	int_4 imax, imin, qmax, qmin;
	int_4 imaxhi, imaxlo;
	int_4 iavg, qavg;
	int temp;
	int_1 verbose;

	verbose=p->verbose;	
	p->verbose=0;
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
#endif
	for (i = 0; i < 9; i++) {
		vprint("About to set dac chan4: %f volts, chan5: %f volts\n", qdacv, idacv);
		dac_setchan(p, mport, 4, qdacv);
		dac_setchan(p, mport, 5, idacv);
		// let the DACS settle a little
		rfxd_delay(p, mport, 100);
		// reset the min, max registers
		minmax_reset(p, mport);
		// allow time to see some mins and maxes
		rfxd_delay(p, mport, 10000);
		// first need to capture data
		minmax_get(p, mport, &imax, &imin, &qmax, &qmin);
		// divide by 2 
		iavg = (imax + imin)/2;
		qavg = (qmax + qmin)/2;
		
		//v3print("IMAX: %08x IMIN: %08x IAVG: %08x QMAX: %08x QMIN: %08x QAVG: %08x\n", imax, imin, iavg, qmax, qmin, qavg);
		v3print("IMAX: %d IMIN: %d IAVG: %d QMAX: %d QMIN: %d QAVG: %d\n", imax, imin, iavg, qmax, qmin, qavg);
		if (iavg > 0) {
			idacv = idacv - daci;
		} else {
			idacv = idacv + daci;
		}
		if (qavg > 0) {
			qdacv = qdacv + daci;
		} else {
			qdacv = qdacv - daci;
		}
		daci = daci/2;
	}
	p->verbose = verbose;
	v3print("DCCAL: qdac: %f idac: %f\n", qdacv, idacv);

	return 0;
}

#if 1

int off_setup (PICSTRUCT * p, int_4 port, int gain, float offset, int delay, int_4 *I_avg, int_4 *Q_avg) {
        int_4 I_max, I_min, Q_max, Q_min;

        // FIX - may have to alter the gain to be real gain!!!
	//radgn(p, port, gain);
        rfxd_set_gain(p, port, gain);
	dac_setchan(p, port, 4, offset);
        dac_setchan(p, port, 5, offset);
        minmax_reset(p, port);
        rfxd_delay(p, port, delay);
        minmax_get(p, port, &I_max, &I_min, &Q_max, &Q_min);
        *I_avg = (I_max+I_min)/2;
        *Q_avg = (Q_max+Q_min)/2;
	return 0;
}
/*
   function to set DC offset to minimize change of DC across gain settings
   the algorithm works by measuroing DC offset at extremes of the DAC setting
   at 2 gain settings. Then it finds the intersection of the 2 lines and sets the
   dac to provide that intersection. Testing has shown that there are 2 clusters
   of intersects; 1) related to back end gain and 2) related to VGA gain
*/
int dc_offset_intersect (PICSTRUCT * p, int_4 port) {
	int_4 saved_gain;
        int_4 I_avg, Q_avg;
        int y1I, y1Q, y2I, y2Q;
        int b72I, b72Q, b36I, b36Q;
        float m72I, m36I, m72Q, m36Q;
        float xI, xQ;
        float dac_min = 0.0;
        float dac_max = 3.0;
        int print_flag=0;

	//int gain1 = 72;
        //int gain2 = 36;
	saved_gain = rfxd_get_gain(p, port);
	int gain1 = 54; // attn of 18
        int gain2 = 18; // attn of 54, This includes 1st VGA reduction 
	int delay = 500;

        // get DC at dac_min 
	off_setup(p, port, gain1, dac_min, delay, &I_avg, &Q_avg);
        b72I = y1I = I_avg;
        b72Q = y1Q = Q_avg;
        // get DC at dac_max
	off_setup(p, port, gain1, dac_max, delay, &I_avg, &Q_avg);
        y2I = I_avg;
        y2Q = Q_avg;
        // compute slope
	m72I = (y2I-y1I)/(dac_max-dac_min);
        m72Q = (y2Q-y1Q)/(dac_max-dac_min);
        printf("gain1: %d %d %d %f %d %d %f\n", gain1, y1I, y2I, m72I, y1Q, y2Q, m72Q);

        off_setup(p, port, gain2, dac_min, delay, &I_avg, &Q_avg);
        b36I = y1I = I_avg;
        b36Q = y1Q = Q_avg;
        off_setup(p, port, gain2, dac_max, delay, &I_avg, &Q_avg);
        y2I = I_avg;
        y2Q = Q_avg;
        m36I = (y2I-y1I)/(dac_max-dac_min);
        m36Q = (y2Q-y1Q)/(dac_max-dac_min);
        printf("gain2: %d %d %d %f %d %d %f\n", gain2, y1I, y2I, m36I, y1Q, y2Q, m36Q);

        // compute intersection of 2 lines
	xI = (b36I-b72I)/(m72I-m36I);
        xQ = (b36Q-b72Q)/(m72Q-m36Q);
        printf("xI %f xQ %f\n", xI, xQ);

        dac_setchan(p, port, 5, xI);
        dac_setchan(p, port, 4, xQ);
	rfxd_set_gain(p, port, saved_gain);

        return 0;
}
#endif


void dcm_init (PICSTRUCT *p, int_4 mport) {
	rfxd_mmsk(p, mport, FEMTO_RESET, ~FEMTO_DCM_RESET, FEMTO_DCM_RESET);    // DCM reset
	rfxd_mmsk(p, mport, FEMTO_RESET, ~FEMTO_DCM_RESET, 0              );    // DCM reset
}

int dcm_lock_wait (PICSTRUCT *p, int_4 mport) {
	int i = 1000;		// allow up to .1 second to lock
	while ( --i) {
		if (rfxd_mrd (p, mport, FEMTO_STATUS) & FEMTO_DCM_LOCK) {
			break;
		}
		rfxd_delay(p, mport, 100); // wait 100 us
	}
	return i;
}


/*
-----------------------------------------------------------------------
  PRIVATE INTERFACE FUNCTIONS:
    int_4 rfxd_clkref_set (PICSTRUCT *p, int_4 mport, int_4 intext)
    int_4 rfxd_a2dclksrc_set (PICSTRUCT *p, int_4 mport, int_4 clksrc)
    int_4 rfxd_dcs_set (PICSTRUCT *p, int_4 mport, int_4 dcs_sw)
    int_4 rfxd_ais_set (PICSTRUCT *p, int_4 mport, int_4 ais_sw, real_8 *aiscoeff)
    int_4 rfxd_lna_set (PICSTRUCT *p, int_4 mport, int_4 lna_sw)

  REVISIONS:
  2008/01/01    sn      baseline
  2009/09/15    rdh     standardized to ICE RF interface
-----------------------------------------------------------------------
*/

int_4 rfxd_refsel_set( PICSTRUCT *p, int_4 port, int_4 intext) {
	int samplesrc, temp=0;
// determine sample clock source for later
// and set reference clock source
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][port];
	samplesrc = l->clkfssrc;
	l->clkrefsrc = (char)intext;
#else
	samplesrc = RFXD_RD(p, port, LGSSTRUCT,  clkfssrc) ;
	RFXD_WR(p, port, LGSSTRUCT, clkrefsrc, intext);
#endif
	v3print("Set refsel %d device %d\n",intext, p->mrev[port-1]);
	if (p->mrev[port-1]==FEMTO25) {
		clk25_ref (p, port, intext);
		//////////////////clk25_ref_src(p, port, intext);
		// FIX don't know if we have to recal SI5338 or if we have to wait for loop to lock
	} else {
		clk_ref_src(p, port, intext);
		// if using the internal sample clock then we have to recal, wait for lock
		if (temp == 0) {
			clk_cal(p, port);
			temp = clk_lock_wait(p, port);	// wait until it is locked
			v3print("CLK_LOCK - %d mSec\n", 1000-temp);
		}
	}
	// since the clock phase changed, reinitn the DCM
	dcm_init(p, port);		// init the FPGA DCM after clock is stable
	temp = dcm_lock_wait(p, port);	// wait until it is locked
	v3print("DCM_LOCK - %d uSec\n", (1000-temp)*100);
	// tell synth to reprogram
	synXref(p, port);
	return 0;
}

int_4 rfxd_fssel_set( PICSTRUCT *p, int_4 port, int_4 intext) {
	int temp;
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][port];
	temp = l->clkfssrc;
#else
	temp = RFXD_RD(p, port, LGSSTRUCT,  clkfssrc) ;
#endif
	v3print("Set sample select %d device %d\n",intext, p->mrev[port-1]);
	v3print("FSSEL - PTR: %p DEVICE: %x PORT: %x\n", p, p->devno, port);
	if (p->mrev[port-1]==FEMTO25) {
		v3print("External sample clock not supported on FEMTO2.5 (yet)\n");
		return 0;
	} else {
		clk_fs_src(p, port, intext);
		// if using the internal clock then we have to recal, wait for lock
		if (temp == 0) {
			temp = clk_lock_wait(p, port);	// wait until it is locked
				v3print("CLK_LOCK - %d mSec\n", 1000-temp);
		}
	}
	// since the clock phase changed, reinit the DCM
	dcm_init(p, port);		// init the FPGA DCM after clock is stable
	temp = dcm_lock_wait(p, port);	// wait until it is locked
	v3print("DCM_LOCK - %d uSec\n", (1000-temp)*100);
	return 0;
}

int_4 rfxd_rate_set( PICSTRUCT *p, int_4 port, int_4 rate) {
	int temp;

	v3print("RATE - PTR: %p DEVICE: %x PORT: %x RATE: %d\n", p, p->devno, port, rate);
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][port];
	l->clkrate = rate;
#else
	RFXD_WRI(p, port, LGSSTRUCT,  clkrate, rate) ;
#endif
	v3print("Set sample rate device %d\n", p->mrev[port-1]);
	if (p->mrev[port-1]==FEMTO25) {
		clk25_fsi (p, port, rate);
	} else {
		temp = clk_rate(p, port, rate);
		v3print("Clock rate return = %d\n", temp);
		temp = clk_lock_wait(p, port);	// wait until it is locked
		v3print("CLK_LOCK - %d mSec\n", 1000-temp);
	}
	dcm_init(p, port);		// init the FPGA DCM after clock is stable
	temp = dcm_lock_wait(p, port);	// wait until it is locked
	v3print("DCM_LOCK - %d uSec\n", (1000-temp)*100);
	return 0;
}

int_4 rfxd_set_vio( PICSTRUCT *p, int_4 port, int_4 address, int_4 data, int_4 len) 
{
#if MEMSTRUCT
	LGSSTRUCT *l;
#endif
	int i, ptr, t1, t2, t3;
	int_4 bias1, bias2, bias3, bias4, bias5, bias6;
	int_4 vga, fil1, fil2, fil3, fil4, fil5;
	int version;
	float LVL;
	//int_4 retval = 0;
	int_4 retval = data;
	real_8 freq;

	v3print("Virtual IO address - %08x data - %08x len - %d\n",address, data, len);
	switch (address) {
		case 0x80:
			v3print("Help\n");
			v3print("  0x80 - print help menu\n");
			v3print("  0x81 - IQ compensation, 0 = off, 1 = on\n");
			v3print("  0x82 - Set filter bandwidth\n");
			v3print("  0x83 - Set gain discretely\n");
			v3print("  0x84 - Force DC offset calibration\n");
			v3print("  0x85 - Select reference source, 0 = int, 1 = ext\n");
			v3print("  0x86 - FEMTO 2.5 control\n");
			v3print("  0x87 - set LNA\n");
			v3print("  0x88 - set DAC chan (high word) to setting (low word)\n");
			v3print("  0x89 - synrd, synwr - address, data, address msb =1 -> write\n");
			v3print("  0x8a - set wideband filter bias discretely\n");
#if !MEMSTRUCT
			v3print("  0x8b - dump RAM parameters\n");
#endif
			v3print("  0x8c - AGC: set alpha, beta, reset AGC\n");
			v3print("  0x8d - print AGC registers\n");
			v3print("  0x8e - execute AGC\n");
			v3print("  0x8f - hardware tests\n");
		 	break;
		case 0x81:
			if (data == 0){
				rfxd_mwr(p, port, FEMTO_LVDS, 9);
				rfxd_mwr(p, port, 0x19, 0);
			} else if (data == 1) {
				rfxd_mwr(p, port, FEMTO_LVDS, 11);
				rfxd_mwr(p, port, 0x19, 1);
			} else if (data == 2) {
				int tmp, tmp1;
				rfxd_mwr(p, port, 0x19, 0);
				tmp = rfxd_mrd(p, port, 0x29);
				v3print("Version = %02x\n", tmp);
				rfxd_mwr(p, port, 0x19, 0x10);
				tmp = rfxd_mrd(p, port, 0x29);
				v3print("Status  = %02x\n", tmp);
				tmp  = rfxd_mrdw(p, port, 0x2a);
				tmp1 = rfxd_mrdw(p, port, 0x2c);
				v3print("taps    - %04x %04x\n", tmp, tmp1);
				tmp  = rfxd_mrdw(p, port, 0x2e);
				tmp1 = rfxd_mrdw(p, port, 0x30);
				v3print("diag    - %04x %04x\n", tmp, tmp1);
				tmp  = rfxd_mrdw(p, port, 0x32);
				tmp1 = rfxd_mrdw(p, port, 0x34);
				v3print("dc      - %04x %04x\n", tmp, tmp1);
				rfxd_mwr(p, port, 0x19, 3);
				rfxd_mwr(p, port, 0x19, 0);
				rfxd_mwr(p, port, FEMTO_LVDS, 11);
				rfxd_mwr(p, port, 0x19, 1);
			}
			break;
		case 0x82:
			radfl (p, port, data); 
			break;
		case 0x83:
			vga  = (((data & 0x00f00000) * 6) >> 20);
			fil1 = (((data & 0x000f0000) * 6) >> 16);
			fil2 = (((data & 0x0000f000) * 6) >> 12);
			fil3 = (((data & 0x00000f00) * 6) >>  8);
			fil4 = (((data & 0x000000f0) * 6) >>  4);
			fil5 =  ((data & 0x0000000f) * 6);
			radgni(p, port, vga, fil1, fil2, fil3, fil4, fil5);
			break;
		case 0x84:
			if      (data == 0) radcal(p,port);	// calibrate DC offset
			else if (data == 1) dc_offset_intersect(p, port); 
			else minmax_print(p, port);
			break;
		case 0x85:
			if(data & 0x2) rfxd_refsel_set( p, port, data&1); // set reference source
			else           rfxd_fssel_set ( p, port, data&1); // set sample rate source
			break;
		case 0x86:
#ifdef FEMTO25
			t1 = (data >> 24) & 0xff;
			t2 = (data >> 16) & 0xff;
			t3 = data & 0xffff;
			if      (t1 == 0xe0)      write_si5338(p, port, t2, t3);
			else if (t1 == 0xe1) t3 = read_si5338(p, port, t2);
			else if (t1 == 0x30)      i2c_wr(p, port, t1, t2, t3);
			else if (t1 == 0x31) t3 = i2c_rd(p, port, t1, t2);
			else if (t1 == 0xa0)      i2c_wr(p, port, t1, t2, t3);
			else if (t1 == 0xa1) t3 = i2c_rd(p, port, t1, t2);
			else if (t1 == 0x00)      adcwr(p, port, t2, t3);
			else if (t1 == 0x02)      gaintblindex=t3;
			else if (t1 == 0x10){
			      adcwr(p, port, 4, 3);
			      rfxd_mmsk(p, port, 2, 0xf7, 8);
			} 
			else if (t1 == 0x12){
			      adcwr(p, port, 4, 1);
			      rfxd_mmsk(p, port, 2, 0xf7, 0);
			} 
			else if (t1 == 0x14){
			      freq=rfxd_get_freq(p, port);
			      v3print("Freq = %f\n", freq);
			} 
			if(t1&1) v3print("Read from slave %x address %x data %x\n", t1,t2,t3);
#endif
			break;
		case 0x87:
      			radlna(p, port, data & 0x7);
			break;
		case 0x88:
			t1 = data >> 16; // get channel
			t2 = data & 0xffff;
			version = (int)rfxd_mrd(p,port, FEMTO_REV);
			if((version >= 4) && (version <= 7)) LVL = (t2 * 5.0)/65535.;
			else                                 LVL = (t2 * 3.3)/65535.;
			dac_setchan(p, port, t1, LVL);
			v3print("Set dac channel %d to %f volts\n", t1, LVL);
			break;
		case 0x89:
  			if(p->mrev[port-1]==FEMTO25){
				syn25wr(p, port, data&0x7, data);
			} else {
				t1 = (data >> 8) & 0xff; // get address
				t2 = data & 0xff;	// data
				if(data & 0x10000){
					synwr(p, port, t1, t2);
					v3print("synwr - address: %02x data: %02x\n", t1, t2);
				} else if(data & 0x20000){
					syndump(p, port);
				} else {
					retval = synrd(p,port,t1);
					v3print("synrd - address: %02x data: %02x\n", t1, retval);
				}
			}
			break;
		case 0x8a:
			bias1 = (data & 0x00f00000) >> 20;
			bias2 = (data & 0x000f0000) >> 16;
			bias3 = (data & 0x0000f000) >> 12;
			bias4 = (data & 0x00000f00) >>  8;
			bias5 = (data & 0x000000f0) >>  4;
			bias6 = (data & 0x0000000f) >>  0;
			radbias(p,port,bias1,bias2,bias3,bias4,bias5,bias6);
			break;
#if !MEMSTRUCT
		case 0x8b:
			rfxd_dumpram2 (p, port);
			break;
#endif
		case 0x8c:
			t1 = (data & 0xff00) >> 8;
			t2 = (data & 0x00ff) >> 0;
			rfxd_agc_init(p, port, t1, t2);
			break;
		case 0x8d:
			//t1= rfxd_mrd(p, port, FEMTO_PWRHI);
			//t2= rfxd_mrd(p, port, FEMTO_PWRLO);
			//v3print("\nAlpha: %02x Beta: %02x PwrH: %02x PwrL: %02x\n\n", rfxd_mrd(p, port, 0x36), rfxd_mrd(p,port, 0x37), t1, t2);
			rfxd_agc_get_pwr(p, port);
			break;
		case 0x8e:
			rfxd_agc_update (p, port);
			break;
		case 0x8f:
			if(data == 1) {
				synth2_limits(p,port);
			}
			else if(data == 2) {
				rfxd_mwr(p, port, 7, 8); // start RSSI reading
				rfxd_mwr(p, port, 7, 0);
				rfxd_delay(p, port, 10);
				t1 = rfxd_mrdw(p, port, 0x14);
				t2 = rfxd_get_rfpwr(p, port);
				v3print("RSSI raw: %d RSSI dBm: %d\n", t1, t2);

			} else if(data == 3) {
				t1 = rfxd_get_rfpwr(p, port);
				v3print("RSSI %d\n", t1);
			} else if((data & 0xf0) == 0x80) {
				 syn25csr(p, port, data & 0x1);
			} else if((data & 0xf0) == 0x90) {
				 syn25lowspur(p, port, data & 0x3);
			} else if((data & 0xf0) == 0xa0) {
				 syn25pwr(p, port, data & 0x3);
			} else if((data & 0xf0) == 0xb0) {
				 syn25phase(p, port, data );
			} else if((data & 0xf0) == 0xc0) {
				 syn25cp(p, port, data );
			}
			break;

		default:
			break;
	}
	return retval;
}



/*
-----------------------------------------------------------------------
  PUBLIC INTERFACE FUNCTIONS:
    int_4 rfxd_setup(PICSTRUCT *p, int_4 mport, int_4 dir, int_4 rate,

    int_4 rfxd_set_freq (PICSTRUCT *p, int_4 mport, real_8 freq_mhz)
    int_4 rfxd_set_bw (PICSTRUCT *p, int_4 mport, int_4 bw_hz)
    int_4 rfxd_set_gain (PICSTRUCT *p, int_4 mport, int_4 gain)
    int_4 rfxd_set_agctc (PICSTRUCT *p, int_4 mport, real_8 tcval);
    int_4 rfxd_set_agccf (PICSTRUCT *p, int_4 mport, real_8 cfval);
    int_4 rfxd_set_opts (PICSTRUCT *p, int_4 mport, int_4 flags)
    int_4 rfxd_agc_init (PICSTRUCT *p, int_4 mport, int_4 alpha, int_4 beta)
    int_4 rfxd_agc_reset (PICSTRUCT *p, int_4 mport)

    real_8 rfxd_get_freq (PICSTRUCT *p, int_4 mport)  (in MHz)
    int_4 rfxd_get_bw (PICSTRUCT *p, int_4 mport)     (in Hz)
    int_4 rfxd_get_gain (PICSTRUCT *p, int_4 mport)
    int_4 rfxd_get_flags (PICSTRUCT *p, int_4 mport)
    int_4 rfxd_get_pwr (PICSTRUCT *p, int_4 mport)
    int_4 rfxd_get_rfpwr (PICSTRUCT *p, int_4 mport)

  REVISIONS:
  2008/01/01    sn      baseline
  2009/09/15    rdh     standardized to ICE RF interface
-----------------------------------------------------------------------
*/
#define RFXD_RFPWR_CALTBL_SZ 60
/*---------------------------------------------------------------------------*/
/* RFXD- Query A/D attached to AD8313 RMS Power Detector                 */
int_4 rfxd_get_rfpwr(PICSTRUCT *p, int_4 mport)
{
// FEMTO2 RSSI Measurements,,
// "[LO @1900 MHz, RF in @951MHz] [LO@3800MHz, RF in @1901MHz]",,
// RF input level (dBm),RF in @951MHz,RF in @1901MHz
  int_4 caltbl[RFXD_RFPWR_CALTBL_SZ][2] = {
{ 10,28181}, {  9,28014}, {  8,27836}, {  7,27591}, {  6,27303},
{  5,26985}, {  4,26651}, {  3,26311}, {  2,25969}, {  1,25629},
{  0,25289}, { -1,24954}, { -2,24622}, { -3,24305}, { -4,23977},
{ -5,23650}, { -6,23316}, { -7,22983}, { -8,22656}, { -9,22339},
{-10,22039}, {-11,21745}, {-12,21448}, {-13,21147}, {-14,20835},
{-15,20508}, {-16,20164}, {-17,19809}, {-18,19448}, {-19,19080},
{-20,18706}, {-21,18328}, {-22,17945}, {-23,17559}, {-24,17174},
{-25,16783}, {-26,16395}, {-27,16000}, {-28,15606}, {-29,15208},
{-30,14815}, {-31,14421}, {-32,14028}, {-33,13650}, {-34,13288},
{-35,12948}, {-36,12630}, {-37,12353}, {-38,12098}, {-39,11888},
{-40,11700}, {-41,11546}, {-42,11417}, {-43,11308}, {-44,11221},
{-45,11153}, {-46,11094}, {-47,11052}, {-48,11011}, {-49,10978}
};
  int_4  ii,idBm,ipmin,iperr;
  int_u4 mdata = 0;
  int_u4 sumpwr = 0;
  int_4 num = 4;
  int_4 correction = 3;

  /* Average 4 power measurements */
  for(ii=0;ii<num;ii++)
    {
      /* Start the ADC */
      rfxd_mwr(p,mport,FEMTO_SPI, FEMTO_PRG_RSSI);
      rfxd_mwr(p,mport,FEMTO_SPI, 0);
      udelay(10);
      mdata = rfxd_mrdw(p,mport,RSSI_HI);
      sumpwr += mdata;           
    }
  sumpwr /= num;
  /* Compute total input power in dBm as function of A/D reading */
  idBm  = caltbl[0][0];
  ipmin = 2147483647; 
  for (ii=0; ii<RFXD_RFPWR_CALTBL_SZ; ii++) {
    iperr = abs(sumpwr - caltbl[ii][1]);
    if (iperr < ipmin) {
      ipmin = iperr;
      idBm = caltbl[ii][0];
    }
  }
  idBm += correction;
// sn  v3print("RFXD RFPWR: %d (dBm)  %d (ADC)  \n",idBm,sumpwr);

  return(idBm);
}


/* note: freq and bw currently return the requested values 
   gain and flags return actual set value 
   FEMTO25 - stored value is requested freq but returned value is recalculated for actual freq
*/

real_8 rfxd_get_freq (PICSTRUCT *p, int_4 mport) {
  real_8 freq;

#if MEMSTRUCT
  LGSSTRUCT *l;
  l = &femto[p->devno][mport];
  freq = l->RFFREQ;
#else
  freq = RFXD_RDD(p, mport, LGSSTRUCT, RFFREQ);
#endif
  if(p->mrev[mport-1] == FEMTO25){
     freq = syn25calcfr(p,mport,freq);
  }
  v3print("RFXD_GET_FREQ: freq = %f\n", freq);
  return freq;
}

int_4 rfxd_get_bw (PICSTRUCT *p, int_4 mport) {
#if MEMSTRUCT
  LGSSTRUCT *l;
  l = &femto[p->devno][mport];
  return l->SAVED_BW;
#else
  return RFXD_RDI(p, mport, LGSSTRUCT, SAVED_BW);
#endif
}

/* saving frequency is done based on actual set frequency
 * synfr returns the actual set freq in calling var, error in return value
 */

int_4 rfxd_set_freq( PICSTRUCT *p, int_4 port, real_8 freq) {
  int_4 retval;
v3print("Set frequency %f device %d\n",freq, p->mrev[port-1]);
//  v3print("PTR: %p DEVICE: %x PORT: %x\n", p, p->devno, port);
  if      (p->mrev[port-1]==FEMTO2)  retval = synfr(p, port, &freq);
  else if (p->mrev[port-1]==FEMTO25) retval = syn25fr(p, port, &freq);
  else                               retval = syn1fr(p, port, freq);
#if MEMSTRUCT
  LGSSTRUCT *l;
  l = &femto[p->devno][port];
  l->RFFREQ = freq;
#else
  RFXD_WRD(p, port, LGSSTRUCT, RFFREQ, freq);
#endif
  return retval;
}

int_4 rfxd_set_bw( PICSTRUCT *p, int_4 port, int_4 bw) {
    int_4 bw_to_idx[32] = { 
               23749030, 24092014, 24498074, 24967212, 25499424, 26094714,
               26753080, 27474520, 28259038, 29106634, 30017304, 30991052,
               32027874, 33127774, 34290748, 35516804, 36805932, 38158136,
               39573416, 41051776, 42593208, 44197716, 45865304, 47595968,
               49389704, 51246520, 53166412, 55149380, 57195424, 59304544,
               61476740, 63712016 };
    int_4 i,idx,bwtest,bwerr;
    idx = 0;
#if MEMSTRUCT
    LGSSTRUCT *l;
    l = &femto[p->devno][port];
    l->SAVED_BW = bw;
#else
    RFXD_WRI(p, port, LGSSTRUCT, SAVED_BW, bw);	// write request BW
#endif
    bwerr = bw_to_idx[31];
    for (i=0; i<32; i++) {
      bwtest = abs(bw - bw_to_idx[i]);
      if (bwtest < bwerr) {
        bwerr = bwtest;
        idx = i;
      }
    }
    v3print("Set bandwidth %d  index %d \n",bw,idx);  
    v3print("PTR: %p DEVICE: %x PORT: %x\n", p, p->devno, port);
    radfl(p,port,idx);	
    return 0;
}

int_4 rfxd_get_gain (PICSTRUCT *p, int_4 mport) {
#if MEMSTRUCT
  LGSSTRUCT *l;
  l = &femto[p->devno][mport];
  return l->SAVED_GAIN;
#else
  return RFXD_RD2(p, mport, LGSSTRUCT, SAVED_GAIN);
#endif
}

int_4 rfxd_set_gain( PICSTRUCT *p, int_4 port, int_4 gain) {
        int_4 atten, adjgain;
	atten = 72 - (6*((int_4) (gain/6)));
        atten = (atten < 0)  ?  (0) : (atten);
        atten = (atten > 72) ? (72) : (atten);
	adjgain = 72 - atten;
#if MEMSTRUCT
        LGSSTRUCT *l;
	l = &femto[p->devno][port];
	l->SAVED_GAIN = adjgain;
#else
	RFXD_WR2(p, port, LGSSTRUCT, SAVED_GAIN, adjgain);
#endif
	v3print("Set gain %d  atten %d \n",gain,atten);
        v3print("PTR: %p DEVICE: %x PORT: %x\n", p, p->devno, port);
	radgn(p, port, atten);
	return 0;
}

/* RFXD AGC parameter settings
     NOTE: For now, arguments are ignored and these are simply wrappers to 
           the initialization and update routines.
           In a final implementation, rfxd_set_agctc() updates the AGC
           adaptation time constant and rfxd_set_agccf() sets the backoff 
           from full-scale A/D loading or "crest factor".
 */
int_4 rfxd_set_agctc( PICSTRUCT *p, int_4 port, real_8 tcval)
{
  rfxd_agc_update(p,port);
  return (0);
}

int_4 rfxd_set_agccf( PICSTRUCT *p, int_4 port, real_8 cfval)
{
  rfxd_agc_init(p,port,0xff,0xff);
  return (0);
}



/* RFXD Module Options */          
int_4 rfxd_get_opts (PICSTRUCT *p, int_4 mport)
{
#if MEMSTRUCT
  LGSSTRUCT *l;
  l = &femto[p->devno][mport];
  return l->SAVED_FLAGS;
#else
  return RFXD_RDI(p, mport, LGSSTRUCT, SAVED_FLAGS);
#endif
}

int_4 rfxd_set_opts (PICSTRUCT *p, int_4 mport, int_4 i4flags)
{
  int_u4 u4flags,iregval,oldflags;
  u4flags = 0x00000000 | i4flags;
  v3print("RFXD(W) FLAGS: %8.8X \n",u4flags);
#if MEMSTRUCT
  LGSSTRUCT *l;
  l = &femto[p->devno][mport];
  oldflags = l->SAVED_FLAGS;
  l->SAVED_FLAGS = i4flags;
#else
  oldflags = RFXD_RDI(p, mport, LGSSTRUCT, SAVED_FLAGS);
  RFXD_WRI(p, mport, LGSSTRUCT, SAVED_FLAGS, i4flags);
#endif

  /* Make certain options are enabled */
  if (!(u4flags & RFOPT_ENABLE))
    return (0); 

  /* Internal/External 10MHz Reference - only call if a change*/                     
  if( (oldflags ^ u4flags) & RFOPT_XREF){
    if (u4flags & RFOPT_XREF) 
      rfxd_refsel_set(p,mport,1); 
    else 
      rfxd_refsel_set(p,mport,0); 
  }

  /* Check for DCS (DC Suppression) option */
  if( (oldflags ^ u4flags) & RFOPT_DCS){
    if( u4flags & RFOPT_DCS)
      radcal(p,mport);
  }

  /* Check for AIS (Adaptive Image Suppression) option */
  if (u4flags & RFOPT_AIS) {
    rfxd_mwr(p, mport, FEMTO_LVDS, 11);
    rfxd_mwr(p, mport, 0x19, 1);
  }
  else {
    rfxd_mwr(p, mport, FEMTO_LVDS, 9);
    rfxd_mwr(p, mport, 0x19, 0);
  }

  /* Check for AGC (Automatic Gain Control) option */

  if (u4flags & RFOPT_AGC)
    rfxd_agc_set(p,mport,1,0);
  else
    rfxd_agc_set(p,mport,0,0);

  /* Check for LNA (Low Noise Amp) option */
  if( (oldflags ^ u4flags) & RFOPT_LNA){
    if (u4flags & RFOPT_LNA)
      radlna(p, mport, LNA_HI);		// set to max gain
    else
      radlna(p, mport, LNA_LO);		// set to min gain
  }
  return (0);
}


int_4 rfxd_setup (PICSTRUCT *p, int_4 port, int_4 dir, 
                       int_4 rate, int_4 gain) 
{
  int_4 bwHz,temp, FPGAver;
  int_4 opts = 0x00000000;
  double freqMHz;
#if MEMSTRUCT
  LGSSTRUCT *l;
#endif 
  
  /* Get RFOPT flags from open string */
  opts = RFOPT_ENABLE|RFOPT_DCS|RFOPT_AIS; /* default options */
  if (findflag("RFOPT_XREF",p->config) >= 0) { 
    opts |= RFOPT_XREF; 
    printf("RFOPT_XREF deprecated, use RFOPTS=(a|XREF|c) syntax.\n"); 
  }
  if (findflag("RFOPT_DCS",p->config) >= 0) { 
    opts |= RFOPT_DCS; 
    printf("RFOPT_DCS deprecated, use RFOPTS=(a|DCS|c) syntax.\n"); 
  }
  if (findflag("RFOPT_AIS",p->config) >= 0) { 
    opts |= RFOPT_AIS; 
    printf("RFOPT_AIS deprecated, use RFOPTS=(a|AIS|c) syntax.\n"); 
  }
  if (findflag("RFOPT_LNA",p->config) >= 0) { 
    opts |= RFOPT_LNA; 
    printf("RFOPT_LNA deprecated, use RFOPTS=(a|LNA|c) syntax.\n"); 
  }
  opts = findmaskflagdef("RFOPTS",p->config,RFOPTLIST,opts);
  if (opts!=0) opts |= RFOPT_ENABLE;

  /* Initialize the module and get version/revision */
  v3print("RFXD_SETUP: LGSLIB Date: %s %s\n", __DATE__, __TIME__);
  v3print("RFXD_SETUP: PTR: %p DEVICE: %x PORT: %x DIR: %d RATE: %d GAIN: %d\n", p, p->devno, port, dir, rate, gain);
  rfxd_mwr(p, port, FEMTO_RESET, FEMTO_DCM_RESET | FEMTO_FPGA_RESET);	// reset device
  rfxd_mmsk(p, port, FEMTO_RESET, ~FEMTO_FPGA_RESET, 0);                // clear FPGA RESET
  rfxd_delay(p, port, 1000);
  rfxd_mwr(p,port, FEMTO_LED_OUT, 0x00);            // turn off the LEDs
  FPGAver = rfxd_mrd(p, port, FEMTO_VERSION);
  temp = rfxd_mrd(p, port, FEMTO_REV);
  v3print("RFXD_SETUP: FPGA Version: %02x Board Version %d\n",FPGAver,temp);
  v3print("RFXD_SETUP: board rev from ICELIB %d\n",p->mrev[port-1]);
  dac_init(p, port);
  
  /* Setup RF synthesizer and A/D clock                    */
  /*   -set the clks to be local reference and sample rate */
#if MEMSTRUCT
  l = &femto[p->devno][port];
  l->clkrefsrc = 0;
  l->clkfssrc = 0;
#else
  RFXD_WR(p, port, LGSSTRUCT, clkrefsrc, 0);
  RFXD_WR(p, port, LGSSTRUCT, clkfssrc, 0);
#endif
  rfxd_mmsk(p, port, FEMTO_BOARD_CTRL, ~FEMTO_OSCEN,FEMTO_OSCEN);     // oscillator on

  rfxd_mwr(p,port, FEMTO_LED_OUT, 0x01);      // turn on LED1
  rfxd_mwr(p,port,FEMTO_OUT,1);		// look at FPGA ref clock during clock init
  /* init the sample clock generator */	
  clkXinit(p, port);		// init the clock generator
  /* for FEMTO 25 have to init the ADC before the DCM that gets clock from ADC */
  if (p->mrev[port-1]==FEMTO25){
	adcinit(p,port);	// init ADC
        // FIX do we need delay here?
	udelay(10000);
  }
  // the rate set code sets clock freq, init dcm and waits for dcm locked
  temp = rfxd_rate_set ( p, port, rate ); 
  v3print("RFXD_SETUP: rfxd_rate_set return = %d\n", temp);
  rfxd_mwr(p,port, FEMTO_LED_OUT, 0x02);// turn off LED1, on LED2

  /* init the synthesizer */
  synXinit(p, port);	// set up default synthesiser config once

  /* init the radio */
  radreset(p, port);		// reset the RFIC
  
  /* Set LNA Gain to min/max gain */
  gain = LNA_LO;  
  if ((opts & RFOPT_ENABLE) && (opts & RFOPT_LNA)) gain = LNA_HI; 
  radlna(p, port, gain);
  v3print("RFXD_SETUP: LNA gain = %d\n",gain);
  
  /* Set RFGain to minimum gain */
  gain = findintflag("RFGAIN",p->config);
  if (findintflag("RFOPT_GAIN",p->config) >= 0) {
    gain = findintflag("RFOPT_GAIN",p->config);
    printf("RFOPT_GAIN flag deprecated, use RFGAIN=value syntax.\n"); 
  }
  gain = (gain > 72) ? (72) : (gain);  // 72=max_gain
  gain = (gain < 0) ? (0) : (gain);  // 0=min_gain
  rfxd_set_gain(p,port,gain);
  v3print("RFXD_SETUP: gain= %d\n",gain);

  /* Set bandwidth mode and value  */
  radmode(p,port,1);		/* set to wideband */
  radbias(p,port,2,2,2,2,2,2);		/* set to near minimum bias */
  // sjn 10/4/12 radbias(p,port,1,1,1,1,1,1);		/* set to near minimum bias */
  bwHz = (int)(finddblflag("RFBW",p->config) * 1000000);
  if (findintflag("RFOPT_BW",p->config) >= 0) {
    bwHz = (int)(finddblflag("RFOPT_BW",p->config) * 1000000);
    printf("RFOPT_BW flag deprecated, use RFBW=value syntax.\n"); 
  }
  rfxd_set_bw(p,port,bwHz);
  v3print("RFXD_SETUP: bw= %d\n",bwHz);

  /* Set up internal / external reference */
  if (opts & RFOPT_XREF) {
    v3print("RFXD_SETUP: external ref\n");
#if MEMSTRUCT
    l->clkrefsrc = 1;
#else
    RFXD_WR(p, port, LGSSTRUCT, clkrefsrc, 1);
#endif
    rfxd_refsel_set(p,port,1); 
  }
  else {
    v3print("RFXD_SETUP: internal ref\n");
#if MEMSTRUCT
    l->clkrefsrc = 0;
#else
    RFXD_WR(p, port, LGSSTRUCT, clkrefsrc, 0);
#endif
   rfxd_refsel_set(p,port,0); 
  }

  /* Set synthesizer RF frequency */
  freqMHz = finddblflag("RFFREQ",p->config);
  if (finddblflag("RFOPT_FREQ",p->config) >= 0.0) {
    freqMHz = finddblflag("RFOPT_FREQ",p->config);
    printf("RFOPT_FREQ flag deprecated, use RFFREQ=value syntax.\n"); 
  }
  freqMHz = (freqMHz >   62.5) ? (freqMHz) : (900.0);
  freqMHz = (freqMHz < 4000.0) ? (freqMHz) : (900.0);
  rfxd_set_freq(p,port,freqMHz); 

  /* Setup the LVDS data path (DDR I/Q, Swap Bytes) */
  rfxd_mwr(p, port, FEMTO_LVDS, 1);	// output DDR I,Q
  rfxd_mmsk(p, port, FEMTO_LVDS, ~FEMTO_BYTE_SWAP, FEMTO_BYTE_SWAP);
  rfxd_mwr(p,port, FEMTO_LED_OUT, 0x05);      // turn on LED1, LED3

  /* turn off external output */
  rfxd_mwr(p,port,FEMTO_OUT,6);

  /* Calibrate DC offset */                    
  if (opts & RFOPT_DCS) {
    v3print("SETUP: DC Offset\ni");
	radcal(p,port);
    rfxd_mwr(p,port, FEMTO_LED_OUT, 0x08);      // turn off LED1. on LED4
  }
 
  /* AIS - image compensation */
  if (opts & RFOPT_AIS) {                       
    v3print("SETUP: AIS\n");
    rfxd_mwr(p,port, FEMTO_LVDS, 0x03);    // enable compensated output
    rfxd_mwr(p,port, FEMTO_IQ_COMP, 0x01); // start compensation
    rfxd_mmsk(p, port, FEMTO_LVDS, ~0x8, 0x08); 	// swap bytes
  }
  else {
    rfxd_mwr(p, port, FEMTO_LVDS, 1);	     // output DDR I,Q 
    rfxd_mmsk(p, port, FEMTO_LVDS, ~0x8, 0x08);   // swap bytes
  }
  v3print("RFXD_SETUP: DONE\n\n");

  /* apply runtime options */
  rfxd_set_opts (p,port,opts);

  /* setup agc */
  rfxd_agc_init(p,port,0x7f,0x01);

  return (0);
}


#define DEBUG_PRINT 1
#define SYNTHDEBUG 1

// array to store linearized cap values
unsigned char cal_array[] = {				
  0, 32, 64, 96,  8, 40, 72,104, 16, 48, 80,112,	
  1, 33, 65, 97,  9, 41, 73,105, 17, 49, 81,113,	
  2, 34, 66, 98, 10, 42, 74,106, 18, 50, 82,114,	
  3, 35, 67, 99, 11, 43, 75,107, 19, 51, 83,115,	
  4, 36, 68,100, 12, 44, 76,108, 20, 52, 84,116,	
  5, 37, 69,101, 13, 45, 77,109, 21, 53, 85,117,	
  6, 38, 70,102, 14, 46, 78,110, 22, 54, 86,118,	
  7, 39, 71,103, 15, 47, 79,111, 23, 55, 87,119		
};

/********************************************************************

	LGS SYNTH programming

 ********************************************************************/

void synth_reset (PICSTRUCT *p, int_4 port)
{
	rfxd_mmsk( p, port, FEMTO_BOARD_CTRL, ~FEMTO_SYN_RST,FEMTO_SYN_RST);
	rfxd_mmsk( p, port, FEMTO_BOARD_CTRL, ~FEMTO_SYN_RST,0);
	rfxd_mmsk( p, port, FEMTO_BOARD_CTRL, ~FEMTO_SYN_RST,FEMTO_SYN_RST);
}

void synth_start_spi (PICSTRUCT *p, int_4 mport) {
	rfxd_mwr (p, mport, FEMTO_SPI,FEMTO_PRG_SYN); 
	rfxd_mwr (p, mport, FEMTO_SPI,0 );
}

void synwr(PICSTRUCT *p, int_4 port, unsigned char address, unsigned char control){
	int val;
	
#if DEBUG_PRINT
//	vprint("Synth set - A: %02d D: %02x\n", (int)address, (int)control);
#endif
	val = (unsigned)(address | 0x80) << 8 | (unsigned)control;
	//rfxd_mwrword (p, port, FEMTO_SPI2,  val);
	rfxd_mwr(p, port, FEMTO_SPI2,  (address | 0x80));
	rfxd_mwr(p, port, FEMTO_SPI1,  control);
	synth_start_spi(p, port);
}

int synbusy(PICSTRUCT *p, int_4 port)
{
	int i = 1000;
	
	while (i--) {
		if( (rfxd_mrd (p, port, FEMTO_SPI) & FEMTO_SPI_STATUS) == 0) break;
	}
	return 0;
}


char synrd(PICSTRUCT *p, int_4 port, unsigned char address)
{
	int temp;
	
	temp = (unsigned)(address) << 8 | 0;
	//rfxd_mwrword (p, port, FEMTO_SPI2,  temp);
	rfxd_mwr (p, port, FEMTO_SPI2,  address);
	rfxd_mwr (p, port, FEMTO_SPI1,  0);
	synth_start_spi(p, port);
	synbusy(p, port); 	// wait for SPI to be done
	temp = rfxd_mrd (p, port, SYNTH_SPI_READ);
#if DEBUG_PRINT
//	vprint("Synth get - A: %02d D: %02x\n",  (int)address, (int)temp);
#endif
	return temp;
}

int synth_stat(PICSTRUCT *p, int_4 port)
{
	return 0;
}

int synth_ld(PICSTRUCT *p, int_4 port)
{
	int val;
	
	synwr(p, port, SYN_MUXREG, 4);		// lock detect mode
	val = (int)synrd(p, port, FEMTO_STATUS);
	return (val & FEMTO_SYNNLD);
}

int syndump (PICSTRUCT *p, int_4 mport ) {
	int i, data;

	for(i=1;i<30;i++){
		data = synrd(p,mport,i);
		vprint("SYNDUMP - ADDR: %02x DATA: %02x\n", i, data);
	}
	return 0;
}

#if 1 
// delay in micro seconds using PIC delay

int syn_delay(PICSTRUCT *p, int_4 port, int count)
{
	rfxd_delay(p, port, count);
	return 0;
}
#endif

#if 0
// use delay by counting in FPGA
int syn_delay(PICSTRUCT *p, int_4 port, int count)
{
	int i = 100000;
	int err = 1;
	int savecount = count;

	count = count * 10;   // to get microseconds
	rfxd_mwr(p, port, FEMTO_TIMERHI, (count >> 8) & 0xff);
	rfxd_mwr(p, port, FEMTO_TIMERLO,  count       & 0xff);
	rfxd_mwr(p, port, FEMTO_LED_OUT, 4);
	rfxd_mmsk(p,port, FEMTO_RESET, ~FEMTO_START_TIMER,0);
	rfxd_mmsk(p,port, FEMTO_RESET, ~FEMTO_START_TIMER,FEMTO_START_TIMER);
	while(i--) {
		if( rfxd_mrd(p, port, FEMTO_STATUS) & FEMTO_TIMER_DONE) {
			err = 0;
			break;
		}
	}
	rfxd_mwr(p, port, FEMTO_LED_OUT, 0);
	vprint("Syndelay xcount: %d error: %d\n", savecount, err);
	return 0;
}
#endif

// cal code for rev2 of the synthesizer
// this will all be replaced by hardware in synth rev3

#if LGSSYNTH2

// measure the frequency of the VCO using the synth2 built in counters
unsigned synth2_measure_freq(PICSTRUCT *p, int_4 port, unsigned char print_flag){
	unsigned char countlo, countmid, counthi, reg2e;
	unsigned count;
	char RefDiv;

	synwr(p, port, 0x16, 0x80);
	synwr(p, port, 0x16, 0xc0);
	synwr(p, port, 0x16, 0x80);
	RefDiv = synrd(p, port, SYN_REFREG) + 1;
	if(RefDiv == 1) syn_delay(p, port, 1000);
	else            syn_delay(p, port, 1500);
	// read the status
	countlo  = synrd(p, port,0x21);
	countmid = synrd(p, port,0x22);
	counthi  = synrd(p, port,0x23);
	reg2e = synrd(p, port,0x2e);
	count = (unsigned)((unsigned char)countlo) + ((unsigned)((unsigned char)countmid) << 8);

	if(print_flag)
		vprint("cap: %02x cnt %d cntlo %02x cntmid %02x cnthi %02x\n", \
		(int)reg2e, count, (unsigned)countlo, (unsigned) countmid, (unsigned)counthi);
	return count;
}
#endif

#define SYNTH_TEST 1
#ifdef SYNTH_TEST

int synth2_measure(PICSTRUCT *p, int_4 port, unsigned char vco, unsigned char dac, unsigned char cap, int print_flag)
{	
	unsigned char save_reg1, save_reg8, save_regb, save_regc;
	int count;
	unsigned char mode;

	// put in VCO mode
	save_reg1 = synrd(p, port,1);
	synwr (p, port, 1 , 0x80 | vco);
	// set reg 8 to use DAC as voltage source
	save_reg8 = synrd(p, port,8);
	synwr (p, port, 8, dac);
	// set output divider to 32
	save_regb = synrd(p, port, 0xb);
	save_regc = synrd(p, port, 0xc);
	synwr (p, port, 0xb, 0x1f);
	synwr (p, port, 0xc, 0x20);
	// set the KSHIFT
	synwr (p, port, 0x15, KSHIFT);
	// set KWAIT
	synwr (p, port, 0x17, 0x8);
	synwr (p, port, 0x18, 0);

	// set the cap and write
	synwr(p, port, 7, cap);
	count = (int)synth2_measure_freq(p, port, print_flag);

	if(print_flag) vprint ("count: %d\n", count);
	
	// restore DAC register
	synwr (p, port, 8 , save_reg8);
	//restore output dividers
	synwr(p, port, 0x0b, save_regb);
	synwr(p, port, 0x0c, save_regc);
	// restore reg 1
	synwr (p, port, 1, save_reg1);
	return count;
}	

int synth2_capswp(PICSTRUCT *p, int_4 port, unsigned char print_flag, char vco, unsigned char dac)
{
	unsigned char cap;
	int count;

	for(cap = 0; cap<96; cap++){
		count = synth2_measure(p, port, vco, dac, cap, 0);
			vprint(	"vco: %02x dac: %02x cap: %02x count: %d\n", (int)vco, (int)dac, (int)cap, count);
	}
	return 0;
}

int synth2_dacswp(PICSTRUCT *p, int_4 port, unsigned char print_flag, char vco, unsigned char cap)
{
	unsigned char dac;
	int count;

	for(dac = 0; dac<32; dac++){
		count = synth2_measure(p, port, vco, dac, cap, 0);
			vprint(	"vco: %02x dac: %02x cap: %02x count: %d\n", (int)vco, (int)dac, (int)cap, count);
	}
	return 0;
}

double synth2_test(PICSTRUCT *p, int_4 port, unsigned char vco, unsigned char dac, unsigned char cap)
{
	int count;
	float gate, freq;

	count = synth2_measure(p, port, vco, dac, cap, 0);
	gate = 1/10000000.*4096;
	freq = count*(1.0/gate)*32*4;
	vprint("count: %d freq: %f\n", count, freq);
	return freq;
}

int synth2_limits(PICSTRUCT *p, int_4 port)
{
	unsigned char vco, dac, cap;
	unsigned char print_flag = 0;
	int count;
	double freq;

	for(vco = 8; vco < 0x80; vco<<=1){
		for(dac = 0; dac<32; dac+=31){
			for(cap = 0; cap<96; cap+=95){
				//count = synth2_measure(p, port, vco, dac, cap, 0);
				freq = synth2_test(p, port, vco, dac, cap);
				vprint(	"vco: %02x dac: %02x cap: %02x count: %d freq %f\n", (int)vco, (int)dac, (int)cap, count, freq);
			}
		}
	}
	return 0;
}

// pass in an index and set the corresponding cap

unsigned char synth_index(PICSTRUCT *p, int_4 port, unsigned char index, unsigned char print_flag)
{
	unsigned char b1 = 0;
	unsigned char b2 = 0;
	unsigned char b3 = 0;
	unsigned char current, temp;
	
	current = 0;
	for (b1 = 0;b1 <= 7; b1++) {
		for (b2 = 0;b2 <= 2; b2++) {
			for (b3 = 0;b3 <= 3 ; b3++) {
				// set the cap;
				if (index == current){
					goto L1;
				}
				current++;
			}
		}
	}
	
L1:	temp = (0x80 | b1 | (b2<<3) | (b3<<5));		
	if(print_flag)
		vprint("SYNCAP b1: %x b2: %x b3: %x reg7: %x\n", (int)b1, (int)b2, (int)b3, (int)temp);
	synwr (p, port, 7, temp);
	return temp;
}

// return the index corresponding to a capacitor setting

int synth_cap(PICSTRUCT *p, int_4 port, unsigned char cap)
{
	unsigned char b1 = 0;
	unsigned char b2 = 0;
	unsigned char b3 = 0;
	unsigned char index, temp;
	
	cap &= 0x7f;			// remove MSB
	index = 0;
	for (b1 = 0;b1 <= 7; b1++) {
		for (b2 = 0;b2 <= 2; b2++) {
			for (b3 = 0;b3 <= 3 ; b3++) {
				temp = (b1 | (b2<<3) | (b3<<5));		
				if (cap == temp){
					goto L1;
				}
				index++;
			}
		}
	}
	
L1:	vprint("SYNINDEX b1: %x b2: %x b3: %x index: %d\n", (int)b1, (int)b2, (int)b3, (int)index);
	return 0;
}

int vco_cap_test(PICSTRUCT *p, int_4 port, unsigned char vco, unsigned char index)
{
	unsigned count;
	unsigned char cap;
	unsigned char print_flag = 1;
	
	synwr(p, port, 1, 0x80 | vco);
	//cap = synth_index(p, port, index, 0);	// lowest cap
	cap = cal_array[index];
	synwr(p, port, 7, cap | 0x80);
	cap &= 0x7f;
	syn_delay(p, port, 1000);
	// frequency measurement
	count = synth2_measure_freq(p, port, print_flag);
	vprint("INDEX: %x CAP: %x count: %d\n", (int)index, (int)cap, count);
	return 0;
}



// calibration sweep using internal frequency counter of Synth version 2
unsigned char capval[96];
unsigned freqcnt[96];

int synth2_cal_sweep(PICSTRUCT *p, int_4 port, unsigned char start, unsigned char stop)
{
	unsigned char cap, index, i;
	unsigned count;

	for (index = start; index < stop; index++) {
		// set the cap and write;
		cap = cal_array[index];
		count = synth2_measure(p, port, 0x08, 0x10, cap | 0x80, 0);
		capval[index] = cap;
		freqcnt[index] = count;		
		vprint("INDEX: %x CAP: %x count: %d\n", (int)index, (int)cap, count);
	}
#if 1
	// sort the list
	for (i = stop; i > start; i--) {
		for(index = start; index < (i-1); index++){
			if(freqcnt[index] > freqcnt[index+1]){
				cap = capval[index];
				capval[index] = capval[index+1];
				capval[index+1] = cap;
				count = freqcnt[index];
				freqcnt[index] = freqcnt[index+1];
				freqcnt[index+1] = count;
			}
		}
	}
	// print the new table
	vprint("\n");
	for (index = start; index < stop; index++) {
		vprint("INDEX: %x CAP: %x count: %d\n", (int)index, (int)capval[index], freqcnt[index]);
	}
	vprint("\n");

#endif

	return 0;
}

#endif // SYNTH_TEST

int synth2_binary_search(PICSTRUCT *p, int_4 port, int target,unsigned char print_flag)
{
	int count;
	unsigned char b1;
	unsigned char b2;
	unsigned char b3;
	int index, delta;

	index = 48;
	delta = 24;

	while(delta){
		synwr(p, port, 7, cal_array[index] | 0x80);
		// wait a little - FIX need to check if this is enough
		syn_delay(p, port, 100);
		// frequency measurement
		count = (int)synth2_measure_freq(p, port, print_flag);
		if(print_flag){
			b1 =  cal_array[index]       & 0x7;
	    		b2 = (cal_array[index] >> 3) & 0x3;
		 	b3 = (cal_array[index] >> 5) & 0x3;
			vprint("SYNCAL1 b1: %x b2: %x b3: %x temp: %x count: %d index; %d delta %d\n", (int)b1, (int)b2, (int)b3, (int)cal_array[index], count, (int)index, (int)delta);
		}
		if ((count-target) > 0){     
			index += delta;
		} else {
			index -= delta;
		}
		delta /= 2;
	}
	return index;
}

int synth2_linear_search(PICSTRUCT *p, int_4 port, unsigned char start, unsigned char stop, int target, unsigned char print_flag)
{
	//unsigned char  cap;
	int i, count;
	int min_residue = 0x7fffffff;
	int max_residue = 0x7fffffff;
	unsigned char min_index;
	
	for(i = start; i < stop; i++){
	// set the cap and write;
		//cap = synth_index(p, port, i, print_flag);
		synwr(p, port, 7, cal_array[i] | 0x80);
// for synth non metal fix
		syn_delay(p, port, 1000);
		count = (int)synth2_measure_freq(p, port, print_flag);
		if (( count > target) && (count < max_residue)) {
			max_residue = count;
			min_index = i;
			if(print_flag) vprint("lin_search: count %d i %d\n", max_residue, min_index);
		}
	}
	return min_index;
}

//#endif  // LGSSYNTH2

// use device as a mode to select cal method
// 0; manual, 1, automatic, 2 step
// MSB set is a print flag for debug

#define SEARCH_RANGE 5

int synth_cal(PICSTRUCT *p, int_4 port, float vcofreq)
{
	
	unsigned char save_reg8, save_reg1, save_regb, save_regc;
	unsigned char countlo, countmid, counthi, reg25, reg2e;
	unsigned char status, cap, min_index;
	unsigned char temp = 0;
	int count;
	float freq, gate, period;
	int target;
//	int min_residue = 0x7fff;
	unsigned char mode, flags;
	unsigned char print_flag, search;
	char start, stop;
	int RefDiv;

	mode = 0;  // manual
	search = 1;  // 0:linear 1:binary
	print_flag = 0; // print

	// save reg 1, later function puts into correct mode
	save_reg1 = synrd(p, port,1) & 0xfc;
	// set reg 8 to use DAC as voltage source
	save_reg8 = synrd(p, port,8);
	synwr (p, port, 8, 0x10);
	// set output divider to 32
	save_regb = synrd(p, port, 0xb);
	save_regc = synrd(p, port, 0xc);
	synwr (p, port, 0xb, 0x1f);
	synwr (p, port, 0xc, 0x20);
	// set the KSHIFT
	synwr (p, port, 0x15, KSHIFT);
	// set KWAIT
	synwr (p, port, 0x17, 0x8);
	synwr (p, port, 0x18, 0);
	// set the target frequency
	//gate = 1./10000000.*KSHIFT*8*4;
	RefDiv = synrd(p, port, SYN_REFREG) + 1;
	gate = 1./(10000000./RefDiv)*KSHIFT*8*4;
	freq = vcofreq * 1000000.;
	period = 1./freq * 32. * 4.;
	target = (int)(gate/period);
	vprint ("freq %f period %f gate %f target %d\n", freq, period, gate, target);
	synwr(p, port, 0x19,  target & 0xff);
	synwr(p, port, 0x1a, (target >> 8) & 0xff);
	synwr(p, port, 0x1b, 0);
	// dump reg before cal

// modes 0: manual, 1 automatic, 2: step

	if (mode == 0){
		// put in VCO mode
		synwr (p, port, 1 , save_reg1 & 0xf8);
		// go though all the caps and measure frequency
		// FIX 7/19/2010 this delay is a stop gap. We 
		// don't need it on other platforms but don't know what is different yet
		syn_delay(p, port, 100000);
		if (search == 0){
			vprint("linear search\n");
			start = (char)0;
			stop = (char)96;
			min_index = synth2_linear_search(p, port, start, stop, target, print_flag);
		} else {
			vprint("binary search\n");
			// do a binary search, then a linear search around answer
			min_index = synth2_binary_search(p, port, target, print_flag);
			start = (char)min_index -SEARCH_RANGE;
			if (start < 0) start = 0;
			stop = (char)min_index + SEARCH_RANGE;
			if (stop > 96 ) stop = 96;
			min_index = synth2_linear_search(p, port, start, stop, target, print_flag);
		}
		// set to the capacitor that gave the best result
		//cap = synth_index(p, port, min_index, 0);
		synwr(p, port, 7, cal_array[min_index] | 0x80);
		//synth2_measure_freq(p, port, print_flag);
		synth2_measure_freq(p, port, 1);
	}
	else if (mode == 1) {
		// start the FLL, keep in FLL mode FIX when new synth is avail, check times
		synwr ( p, port, 0x1, save_reg1| 0x3);
		syn_delay(p, port, 10000);
		synwr ( p, port, 0x1, save_reg1| 0x7);
		syn_delay(p, port, 10000);
		synwr ( p, port, 0x1, save_reg1| 0x3);
		syn_delay(p, port, 10000);
		status = synrd(p, port, 0x20);
		countlo  = synrd(p, port,0x21);
		countmid = synrd(p, port,0x22);
		counthi  = synrd(p, port,0x23);
		reg25 = synrd(p, port,0x25);
		reg2e = synrd(p, port,0x2e);
		count = (unsigned)((unsigned char)countlo) + ((unsigned)((unsigned char)countmid) << 8);
#if 0
		vprint("status: %0x cap: %02x cnt %d cntlo %02x cntmid %02x cnthi %02x reg25 %02x\n", \
	  	(int)status, (int)reg2e, count, (unsigned)countlo, (unsigned) countmid, (unsigned)counthi, (int) reg25);
#endif
		vprint ("count: %d status: %02x\n", count, (int)status);
	}
	else if ( mode == 2) {
		vprint ("Step mode not implemented yet\n");
	}
	// restore DAC register
	synwr (p, port, 8 , save_reg8);
	//restore output dividers
	synwr(p, port, 0x0b, save_regb);
	synwr(p, port, 0x0c, save_regc);
	// put in pll mode
	synwr(p, port, 1, save_reg1 | 0x01);
		
	return temp;
}	


// FIX These should be structures

int outsel[] = {
	4000, 8000, 0, 1,
	2000, 4000, 1, 2,
	1000, 2000, 3, 4,
	 500, 1000, 7, 8,
	 250,  500, 15, 16,
	 125,  250, 31, 32
};
#if 0
int vcosel[] = {
	4000, 4800, 1,
	4800, 5600, 2,
	5600, 6500, 4,
	6500, 8000, 8
};
#endif
#if 0
int synth_div(PICSTRUCT *p, int_4 port, int div)
{
	char divsel = -1;
	char pathsel;
	
	switch (div){
		case 1:
			divsel = 0;
			pathsel = 1;
			break;
		case 2:
			divsel = 1;
			pathsel = 2;
			break;
		case 4:
			divsel = 3;
			pathsel = 4;
			break;
		case 8:
			divsel = 7;
			pathsel = 8;
			break;
		case 16:
			divsel = 15;
			pathsel = 16;
			break;
		case 32:
			divsel = 31;
			pathsel = 32;
			break;
		default:
			break;
	}
	if ( divsel >= 0){
		synwr(p, port, SYN_OUTREG1, divsel);
		synwr(p, port, SYN_OUTREG2, pathsel);
	}
	return divsel;
}
#endif

#if 1


float syn_measure_vco(PICSTRUCT *p, int_4 port, int_4 vco, int_4 cap, int_4 dac, int_4 print_flag) {
        int_4 count;
        float gate, freq;

        // measure freq
        count = synth2_measure(p, port, vco, dac, cap, print_flag);
        gate = 1./10000000.*KSHIFT*8*4;
        freq = count*1./gate*32*4;
        if (print_flag) {
                vprint("vco: %2x cap: %2x dac: %d count: %d freq: %f\n", vco, cap, dac, count, freq);
        }
        return freq;
}


int_4 syn_vco_set_limits (PICSTRUCT *p, int_4 port) {
        int_4 save_reg1, save_reg8, save_regb, save_regc;
        float vco1_hi, vco1_lo, vco2_hi, vco2_lo, vco3_hi, vco3_lo, vco4_hi, vco4_lo;
        int_2 vco1, vco2, vco3;
	int dac = 0x10;
	int_4 err = 0;
        int print_flag = 0;

        // measure freq vco 1
        vco1_hi = syn_measure_vco(p, port, 0x88,  0, dac, print_flag);
        vco1_lo = syn_measure_vco(p, port, 0x88, 95, dac, print_flag);
        // measure freq vco 2
        vco2_hi = syn_measure_vco(p, port, 0x90,  0, dac, print_flag);
        vco2_lo = syn_measure_vco(p, port, 0x90, 95, dac, print_flag);
        // measure freq vco 3
        vco3_hi = syn_measure_vco(p, port, 0xa0,  0, dac, print_flag);
        vco3_lo = syn_measure_vco(p, port, 0xa0, 95, dac, print_flag);
        // measure freq vco 4
        vco4_hi = syn_measure_vco(p, port, 0xc0,  0, dac, print_flag);
        vco4_lo = syn_measure_vco(p, port, 0xc0, 95, dac, print_flag);

        vco1 = (vco1_hi+vco2_lo)/2./1000000.;
        vco2 = (vco2_hi+vco3_lo)/2./1000000.;
        vco3 = (vco3_hi+vco4_lo)/2./1000000.;
	RFXD_WR2(p, port, LGSSTRUCT, VCO1, vco1);
	RFXD_WR2(p, port, LGSSTRUCT, VCO2, vco2);
	RFXD_WR2(p, port, LGSSTRUCT, VCO3, vco3);
        if (print_flag) {
                vprint("lo: %f %f %f %f\n", vco1_lo, vco2_lo, vco3_lo, vco4_lo);
                vprint("hi: %f %f %f %f\n", vco1_hi, vco2_hi, vco3_hi, vco4_hi);
        }
        if ( (vco1_lo < 4000000000.) && (vco1_hi > vco2_lo) && \
	     (vco2_hi > vco3_lo)     && (vco3_hi > vco4_lo) && \
	     (vco4_hi > 8000000000.)) {
                vprint( "VCOs cover range\n");
        } else {
                err = 1;
                vprint( "VCOs do not cover range\n");
        }
        vprint("limits: %d %d %d\n", vco1, vco2, vco3);
        return err;
}

#endif


// FIX need to reconcile the variable types here

int pow2[] = {1,2,4,8,16,32,64,128,256,512,1024};

#define VCO_LL  5000.
#define VCO_UL  5200.
#define RFFREQ_REGE_LL (2*875)
#define RFFREQ_REGE_UL (2*975)
#define SYNMINFREQ 62.5 
#define SYNMAXFREQ 4000.0

int syn_set_freq(PICSTRUCT *p, int_4 port, real_8 * reqfreq)
{
	int divsel = 0;
	int pathsel = 0;
	int vco = 0;
	int lf, uf;
	int ni;
	int w;
	int temp;
	double vcofreq, Fref, freq, actualfreq;
	double n, nf;
	long nfl;
	unsigned char i, nf1, nf2, nf3;
//	unsigned char cap;
	unsigned char nc = 0;
	int RefDiv, DivReg;
	int_2 vcosel[5];
	int Rege;
	int retval = 0;

	/* printf("**** SETTING LGS SYNTH2: %f\n",freq); */

	// software assumes LO freq for synth2. RF freq is passed in
	freq = *reqfreq;
	actualfreq = freq; /* FIX need to calculate actual freq */
	if(freq < SYNMINFREQ) {freq = SYNMINFREQ; retval = -1;}
	if(freq > SYNMAXFREQ) {freq = SYNMAXFREQ; retval = -1;}
	freq = 2 * freq;	
	// determine output frequency range with respect to VCOs
	for(i=0; i< 6;i++){
		lf = outsel[4*i];
		uf = outsel[4*i+1];
		if( (freq >= lf) && ( freq < uf) ){
			divsel = outsel[4*i+2];
			pathsel = outsel[4*i+3];
			// calculate the VCO frequency
			vcofreq = freq * pathsel;
			break;
		}
	}
	vprint ("SYNFR: FREQ: %f LF: %d UF: %d  DIV: %x PATH: %x VCOFREQ: %f\n", freq, lf, uf, divsel, pathsel, vcofreq);
	// pick a VCO
	vcosel[0] = 4000;
	vcosel[1]= RFXD_RD2(p, port, LGSSTRUCT, VCO1);
	vcosel[2]= RFXD_RD2(p, port, LGSSTRUCT, VCO2);
	vcosel[3]= RFXD_RD2(p, port, LGSSTRUCT, VCO3);
	vcosel[4] = 8000;
	for(i=0; i< 4;i++){
		lf = vcosel[i];
		uf = vcosel[i+1];
		if( (vcofreq >= lf) && ( vcofreq < uf) ){
 			vco = 1 << i;
			break;
		}
	}
	vprint("VCOL: %d VCOH: %d VCO: %d\n", lf, uf, vco);
	// lookup the needed cap
//	cap = 64;
	// compute integer part
	// Fout = 2 * Fref * (Ni+Nf+2^(w+4))
	// N = Fout/(2*Fref)
	// Ni+Nf+2^(w+4) = Fout/(2*Fref)
	// Ni+Nf = Fout/(2*Fref)-2^(w+4)
	// 0 <= w <= 6
	// if the vco freq is between 5000 and 5200 MHZ, set refdiv to 3 to avoid N around 0
	if ((vcofreq > VCO_LL) && (vcofreq < VCO_UL)) 	RefDiv = 3;
	else                                            RefDiv = 1;
	vprint("Ref divider %d\n", RefDiv);
	// fix bias current if we are in dead zone of divider chain
	if ((freq > RFFREQ_REGE_LL) && (freq   < RFFREQ_REGE_UL)) Rege = 0x88;
	else                                                          Rege = 0x8f;
	vprint("Rege %x\n", Rege);
	DivReg = RefDiv -1;
	Fref = 10./RefDiv;
	n = ((float)vcofreq)/(2*Fref);
	ni = (int) n;
	vprint("Synth: NI %d\n", ni);
	nf = n - (float)ni;
	vprint("Synth: NF %f\n", nf);
	for(w=6;w>=0;w--){
		temp = pow2[w+4];
		vprint("temp: %d\n",temp);
		if( (ni - pow2[w+4]) > 0) break;
	}
	ni = ni - pow2[w+4];
	// compute fractional part
	nf = nf * (float)(0x00ffffff);
	vprint("Synth:NF %f\n", nf);
	nfl = (long)(nf+.5);
	vprint("Synth:NF %ld\n", nfl);
	if(nf != 0)
		nc = 0x60;			// put in 3rd order fractional mode
	nf1 = (nfl >> 16) & 0xff;
	nf2 = (nfl >>  8) & 0xff;
	nf3 = (nfl >>  0) & 0xff;
#if 1
//	vprint("Synth: Freq %d, NI %0x, NF %f, nf1 %02x nf2 %02x nf3 %02x w %02x VCO %03x\n", freq, ni, nf, nf1, nf2, nf3, w, vco);
	vprint("Synth: Freq %f\n", freq);
	vprint("Synth: N %f\n", n);
	vprint("Synth: NI %d\n", ni);
	vprint("Synth: NF %ld\n", nfl);
	vprint("Synth: w %d\n", w);
	vprint("Synth: VCO %d\n", vco);
#endif
	// program the chip
//	synwr(p, port, SYN_CAPREG,  cap);
//	
	synwr(p, port, SYN_BIASREG2, Rege);
	synwr(p, port, SYN_REFREG, DivReg);
	synwr(p, port, SYN_MODREG,  (vco <<3) | 0x80 | 0x01); // power VCO, PLL mode, VCO#
	synwr(p, port, SYN_DSMFRI1, nc);
	synwr(p, port, SYN_DSMFRI2, ni);
	synwr(p, port, SYN_DSMFRF1, nf1);
	synwr(p, port, SYN_DSMFRF2, nf2);
	synwr(p, port, SYN_DSMFRF3, nf3);
	synwr(p, port, SYN_DIVREG,  (unsigned char)w);
	synwr(p, port, SYN_OUTREG1, divsel);
	synwr(p, port, SYN_OUTREG2, pathsel);

	// calibrate
	synth_cal(p, port,  vcofreq);
	// PLL mode
	i = synrd(p, port, 1) | 0x1;			// put in PLL mode
	synwr(p, port, 1, i);		
	//syndump(p,port);
	*reqfreq = actualfreq; /* FIX this needs to be calulated */
	return retval ;
}

real_8 syn_get_freq(PICSTRUCT *p, int_4 mport)
{
	real_8 freq;
#if MEMSTRUCT
	LGSSTRUCT *l;

	l = &femto[p->devno][mport];
	freq = l->RFFREQ;
#else
	freq = RFXD_RDD(p, mport, LGSSTRUCT, RFFREQ);
#endif
	return freq;
}


unsigned char syn_4GHz_rev2[] = {
0x01, 0x88,	// VCO mode, VCO 0 
0x02, 0x00,	// integer mode
0x03, 0x48,	// N[7:0]
0x04, 0x00,	// frac
0x05, 0x00,	// frac
0x06, 0x00,	// frac
0x07, 0xa6,	// starting cap val
0x08, 0x4f,	// REF SEL, mid val for DAC
0x09, 0xed,	// 
0x0a, 0x03,	// 
0x0b, 0x00,	// set divider to 1
0x0c, 0x01,	// set divider to 1
0x0d, 0x10,	// 
0x0e, 0x8f,
0x0f, 0xaf,
0x10, 0x00,
0x11, 0x00,
0x12, 0x00,
0x13, 0x00,
0x14, 0x04,
0x15, 0x80,	// kshift[7:0]
0x16, 0xe0,	// kshift[11:8], ..
0x17, 0x00,	// delay time lsb
0x18, 0x80,	// delay time, msb (32768 * 1/1.25 MHz = 26 mS)
0x19, 0x7a,
0x1a, 0x00,
0x1b, 0x00,
0x1c, 0x00,
0x30, 0x00,
0x31, 0x00,
0xff
};


// FIX make sure reg 8 is set correct after setting freq

int synth_init(PICSTRUCT *p, int_4 port, unsigned char * param_ptr)
{
	unsigned char a, d;

	d = synrd(p, port, 0x7f);	// read chip id
	vprint("Synthesizer id: %x\n", (int)d);
	if (param_ptr == 0) {
		if ((d == 0x12) || (d == 0x13) || (d == 0x14) ) {
			param_ptr = syn_4GHz_rev2;
		} else {
			param_ptr = 0;
			return -1;		// error for wrong synth
		}
	}
	
	while ( (a = *param_ptr++) != 0xff){
		d = *param_ptr++;
		synwr(p, port, a, d);
	}
	return 0;
}

// call this after changing clk_ref. This will reprogram the synth

int synfr (PICSTRUCT *p, int_4 mport, real_8 * freq ) {
	int retval;
	vprint("SYNFR: freq = %f\n", *freq);
	retval = syn_set_freq(p, mport, freq);
	return retval;
}

int synref (PICSTRUCT *p, int_4 mport) {
	real_8 freq;
#if MEMSTRUCT
	LGSSTRUCT *l;

	l = &femto[p->devno][mport];
	freq = l->RFFREQ; 
#else
	freq = RFXD_RDD(p, mport, LGSSTRUCT, RFFREQ);
#endif
	vprint("SYNFR: freq = %f\n", freq);
	synfr(p, mport, &freq);
	return 0;
}

int syntest (PICSTRUCT *p, int_4 mport ) {
	int i,d;

	rfxd_mwr(p, mport, FEMTO_CHOPHI,  0x55);
	d = rfxd_mrd(p, mport, FEMTO_CHOPHI);
	vprint("CHOPHI = %02x, should be 0x55\n", d);
	rfxd_mwr(p, mport, FEMTO_CHOPHI,  0xAA);
	d = rfxd_mrd(p, mport, FEMTO_CHOPHI);
	vprint("CHOPHI = %02x, should be 0xaa\n", d);

	for(i=0;i<5;i++) {
		rfxd_mwr(p, mport, FEMTO_LED_OUT, 4);
		syn_delay(p, mport, 50000);
		rfxd_mwr(p, mport, FEMTO_LED_OUT, 0);
		syn_delay(p, mport, 50000);
	}	
	return 0;
}

int syninit (PICSTRUCT *p, int_4 mport ) {
	int d;
	int temp;
	real_8 freq = 1000.0;
	
	vprint ("Synth init BEGIN\n");
	temp = p->verbose;
	synth_reset(p, mport);
	syn_delay(p, mport, 100);	// doesn't seem to help
	d = synrd(p, mport, 0x7f);	// read chip id
	vprint("Synthesizer id: %x\n", (int)d);
	synth_init(p, mport, 0);
	// now set a freq so it is stable
	//p->verbose = 1;		// don't print out MRD, MWR commands
	//syndump(p,mport);
	// the limit test does not work here, but does work after first call to set freq!
	//syn_vco_set_limits(p, mport);
// 11/10/2011 sjn changed order of next 2 commands	
	syn_vco_set_limits(p, mport);
	synfr(p, mport, &freq);
	vprint ("Synth init DONE\n");
	p->verbose = temp;
	return 0;
}

