/****************************************************************************/
/*                                                                          */
/*   LB2DM3 module interface routines for the ICEPIC device target          */
/*                                                                          */
/****************************************************************************/

#include "iomlib_lb2dm3x.h"

#define LB2DM3_STATE 0xFD000000  /* Base address for state structure register */
#define LB2DM3_SIZE sizeof(LB2DM3_State)

typedef struct {
  int_u4 chksum;    /* 32 bit checksum|size of packet */
  int_u4 toggles;
  real_8 samprate;  /* A/D sample rate in MHz */
  real_8 vcoref;
  real_8 rffreq;
  real_8 rfbw;
  real_8 rfgain;
  int_4  rfopts;
  int_4  refclk;    /* Reference clock rate in MHz */
  int_4  mtgooff;
  int_1  bits;      /* Module output bits {-16,-8,8,16} */
  int_1  dcsbn;
  int_1  statbn;
  int_1  hw_ver;
  int_1  coeset;
  int_1  presel;
  int_1  fill[2];
} LB2DM3_State;

int_4 lb2dm3_dump_state (PICSTRUCT *p, int_4 mport, LB2DM3_State *SS)                                                
{                                                                                                                     
  printf("SS.chksum:    %08X   \n",SS->chksum);                                            
  printf("SS.refclk:    %i     \n",SS->refclk);                                            
  printf("SS.samprate:  %lf    \n",SS->samprate);                                          
  printf("SS.vcoref:    %lf    \n",SS->vcoref);                          
  printf("SS.rffreq:    %lf    \n",SS->rffreq);                          
  printf("SS.rfbw:      %lf    \n",SS->rfbw);                          
  printf("SS.rfgain:    %lf    \n",SS->rfgain);                          
  printf("SS.rfopts:    %08X   \n",SS->rfopts);                           
  printf("SS.toggles:   %08X   \n",SS->toggles);                           
  printf("SS.bits:      %i     \n",SS->bits);                             
  printf("SS.hw_ver:    %02X   \n",SS->hw_ver&0xFF);                             
  return 0;                                                                
}                                                                               

/*---------------------------------------------------------------------------*/
/* Internal Private Interfaces                                               */
/*                                                                           */
/*   NOTE: These routines should NEVER be called by external libraries.      */
/*         The API to these routines is fluid and is only self-consistent    */
/*         within a release series.                                          */
/*                                                                           */
/*---------------------------------------------------------------------------*/

static const int_4 lb2dm3_hbfir_coefs0 [] = {
    -1,2,-3,5,-8,12,-17,23,-31,41,-54,69,-87,109,-135,165,-201,242,
    -291,347,-413,489,-579,684,-810,963,-1153,1394,-1715,2168,-2862,
    4087,-6902,20844
  };
static const int_4 lb2dm3_hbfir_coefs1 [] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
  };
static const int_4 lb2dm3_hbfir_coefs2 [] = {
    0,1,-1,2,-3,5,-7,11,-16,22,-30,41,-55,71,-92,118,-148,185,-230,283,
    -346,420,-509,615,-743,898,-1092,1339,-1666,2126,-2828,4063,-6887,20839
  };

static const int_4 lb2dm3_cfir_nfilt = 25;
static const int_4 lb2dm3_cfir_taps  =  7;
static const int_4 lb2dm3_cfir_coefs[25][7] = {
    {  0,   0,    0,     0,    0,     0, 32767},
    {  0,   0,    0,     0,    0,     0, 32767},
    {  0,   0,    0,     0,    0,     0, 32767},
    {  0,  -4,    3,   -27,   12,  -304, 32767},
    {  0,  -7,    7,   -42,   29,  -407, 32767},
    {  2,  -7,    7,   -42,   23,  -501, 32767},
    {  2,  -6,    6,   -43,   19,  -590, 32767},
    {  1,  -6,    4,   -49,   15,  -672, 32767},
    {  1,  -8,    5,   -59,   18,  -759, 32767},
    {  2,  -11,  13,   -75,   39,  -873, 32767},
    {  6,  -16,  27,   -92,   76, -1007, 32767},
    {  0,  -16,  16,  -107,   63, -1091, 32767},
    {  1,  -20,  25,  -129,   91, -1208, 32767},
    {  6,  -32,  46,  -165,  152, -1349, 32767},
    {  5,  -38,  61,  -204,  203, -1506, 32767},
    { 14,  -58, 101,  -270,  321, -1730, 32767},
    { 16,  -66, 122,  -316,  376, -1907, 32767},
    { 25,  -87, 166,  -385,  503, -2153, 32767},
    { 16,  -90, 184,  -470,  598, -2402, 32767},
    { 33, -128, 260,  -577,  797, -2705, 32767},
    { 25, -125, 270,  -650,  859, -3011, 32767},
    { 71, -190, 419,  -810, 1191, -3677, 32767},
    { 74, -239, 419,  -836, 1442, -4089, 32767},
    { 11, -168, 408, -1039, 1666, -4513, 32767},
    {-33,  -48, 237,  -956, 1708, -4781, 32767}
  };

static const int_4 lb2dm3_wbcfir_coefs[2][7] = {
    {0,  0,  0,    0,   0,     0, 32767},
    {7,-17, 58, -176, 533, -2104, 32767}
  };

int_4 lb2dm3_presel_decode (PICSTRUCT *p, int_4 mport, int_u4 *psel0, int_u4 *psel1, LB2DM3_State *SS)
{
  int_4 pselrg; 
  int_u4 (*pseltbl_p)[2];
  enum LB2DM3_PRESEL_LIST {
    LB2DM3_PRESEL_AUTO      =  0, /* Auto select filter based on Fc     */
    LB2DM3_PRESEL_LOWBAND   =  1, /* Force Select lowband filter        */
    LB2DM3_PRESEL_MIDBAND   =  2, /* Force Select midband filter        */
    LB2DM3_PRESEL_HIGHBAND  =  3, /* Force Select highband filter       */
    LB2DM3_PRESEL_BYPASS    =  4, /* Force Bypass all filters           */
    LB2DM3_PRESEL_TERMINATE =  5  /* Force Terminate input (no signal)  */
  };
  int_u4 pseltbl[6][2] = { {0x02,0x02},{0x00,0x03},{0x01,0x00},{0x00,0x01},{0x02,0x02},{0x03,0x00} };
  int_u4 pseltblv3[6][2] = { {0x01,0x02},{0x01,0x02},{0x00,0x03},{0x01,0x00},{0x02,0x01},{0x03,0x00} }; /* fixed */
  pseltbl_p = ((SS->hw_ver&0xF) == 0x7)? pseltblv3 : pseltbl;
  
  *psel0 = pseltbl_p[LB2DM3_PRESEL_BYPASS][0];
  *psel1 = pseltbl_p[LB2DM3_PRESEL_BYPASS][1];
  pselrg = (SS->toggles & LB2DM3_TGLSS_PRESEL) ? SS->presel : (LB2DM3_PRESEL_BYPASS);

  switch (pselrg) {
    case (LB2DM3_PRESEL_LOWBAND):
      *psel0 = pseltbl_p[LB2DM3_PRESEL_LOWBAND][0];
      *psel1 = pseltbl_p[LB2DM3_PRESEL_LOWBAND][1];
    break;
    case (LB2DM3_PRESEL_MIDBAND):
      *psel0 = pseltbl_p[LB2DM3_PRESEL_MIDBAND][0];
      *psel1 = pseltbl_p[LB2DM3_PRESEL_MIDBAND][1];
    break;
    case (LB2DM3_PRESEL_HIGHBAND):
      *psel0 = pseltbl_p[LB2DM3_PRESEL_HIGHBAND][0];
      *psel1 = pseltbl_p[LB2DM3_PRESEL_HIGHBAND][1];
    break;
    case (LB2DM3_PRESEL_BYPASS):
      *psel0 = pseltbl_p[LB2DM3_PRESEL_BYPASS][0];
      *psel1 = pseltbl_p[LB2DM3_PRESEL_BYPASS][1];
    break;
    case (LB2DM3_PRESEL_TERMINATE):
      *psel0 = pseltbl_p[LB2DM3_PRESEL_TERMINATE][0];
      *psel1 = pseltbl_p[LB2DM3_PRESEL_TERMINATE][1];
    break;
    default:
      if (SS->rffreq < 3000.0) {
          *psel0 = pseltbl_p[LB2DM3_PRESEL_HIGHBAND][0];
          *psel1 = pseltbl_p[LB2DM3_PRESEL_HIGHBAND][1];
      }
      if (SS->rffreq < 1705.0) {
          *psel0 = pseltbl_p[LB2DM3_PRESEL_MIDBAND][0];
          *psel1 = pseltbl_p[LB2DM3_PRESEL_MIDBAND][1];
      }
      if (SS->rffreq < 1100.0) {
          *psel0 = pseltbl_p[LB2DM3_PRESEL_LOWBAND][0];
          *psel1 = pseltbl_p[LB2DM3_PRESEL_LOWBAND][1];
      }
  }
  v3print("-------PRESEL: %4.4i  0x%8.8X  0x%8.8X \n", pselrg,*psel0,*psel1);
  return (0);
}

int_4 lb2dm3_toggles (PICSTRUCT *p, int_4 mport, LB2DM3_State* SS)
{
  int_4 pinval, pinreg=LB2DM3_PIN_NULL;
  pinreg = u32_setbits(pinreg,LB2DM3_PIN_PPSTAG, (LB2DM3_TGLSS_PPSTAG & SS->toggles)? 1:0 );
  pinreg = u32_setbits(pinreg,LB2DM3_PIN_LSBP,   (LB2DM3_TGLSS_LSBP   & SS->toggles)? 1:0 );
  pinreg = u32_setbits(pinreg,LB2DM3_PIN_RSEL,   (LB2DM3_TGLSS_RAMP   & SS->toggles)? 1:0 );
  pinreg = u32_setbits(pinreg,LB2DM3_PIN_MTGO,   (LB2DM3_TGLSS_MTGO   & SS->toggles)? 1:0 );
  pinreg = u32_setbits(pinreg,LB2DM3_PIN_HBFBYP, (LB2DM3_TGLSS_HBFBYP & SS->toggles)? 1:0 );
  iom_prt_cmd(p,mport,LB2DM3_MODULE,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
  iom_prt_cmd(p,mport,LB2DM3_MTGO,PFUNC_RWR,0x00,SS->mtgooff,0xFFFF);
  pinreg = 0;
  if (LB2DM3_TGLSS_DIO & SS->toggles) pinreg = 0x7;
  if (LB2DM3_TGLSS_LFSR & SS->toggles) pinreg = 0xF;
  iom_prt_cmd(p,mport,LB2DM3_FXLH42245,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
  return (0);
}

int_4 lb2dm3_adclm (PICSTRUCT *p, int_4 mport, int_4 flags, LB2DM3_State* SS)
{
  int_4 adclm_B1  =  (9)&0xFF;
  int_4 adclm_B2  = (15)&0xFF;
  int_4 adclm_dec = (10)&0xFF;
  int_4 rtn=0, stat_func, stat_ID, stat_addr, stat_val;

  if (flags>=0) {
    iom_prt_cmd(p,mport,LB2DM3_ADCLM,PFUNC_RWR,0x01,adclm_B1,0xFF);
    iom_prt_cmd(p,mport,LB2DM3_ADCLM,PFUNC_RWR,0x02,adclm_B2,0xFF);
    iom_prt_cmd(p,mport,LB2DM3_ADCLM,PFUNC_RWR,0x03,adclm_dec,0xFF);
  }
  if (flags<=0) {
    iom_prt_cmd(p,mport,LB2DM3_ADCLM,PFUNC_RRD,0,0,0xFFFFFFFF);
    iom_prt_sts(p,mport,LB2DM3_ADCLM,&stat_func,&stat_ID,&stat_addr,&stat_val);
    rtn = adclm_to_cf_i32(stat_val);
  }
  return (rtn);
}

int_4 lb2dm3_ais (PICSTRUCT *p, int_4 mport, LB2DM3_State* SS)
{
  int_4 ais_cf1 = 0;
  int_4 ais_cf2 = 0x80000000;
  int_4 ais_cf1_ll  = (int_4) (-0.09*2147483648.0);
  int_4 ais_cf1_ul  = (int_4) (0.09*2147483648.0);
  int_u4 ais_cf2_ll = (int_u4) (0.93*2147483648.0);
  int_u4 ais_cf2_ul = (int_u4) (1.07*2147483648.0);
  int_4 pinval = (RFOPT_AIS & SS->rfopts) ? (0x00) : (0x01);
  int_4 pinreg = u32_setbits(AIS_PIN_NULL,AIS_PIN_BYPASS,pinval);
  iom_prt_cmd(p,mport,LB2DM3_AIS,PFUNC_RWR,0x00,ais_cf1,0xFFFFFFFF);
  iom_prt_cmd(p,mport,LB2DM3_AIS,PFUNC_RWR,0x01,ais_cf2,0xFFFFFFFF);
  iom_prt_cmd(p,mport,LB2DM3_AIS,PFUNC_RWR,0x05,ais_cf1_ul,0xFFFFFFFF);
  iom_prt_cmd(p,mport,LB2DM3_AIS,PFUNC_RWR,0x06,ais_cf1_ll,0xFFFFFFFF);
  iom_prt_cmd(p,mport,LB2DM3_AIS,PFUNC_RWR,0x07,ais_cf2_ul,0xFFFFFFFF);
  iom_prt_cmd(p,mport,LB2DM3_AIS,PFUNC_RWR,0x08,ais_cf2_ll,0xFFFFFFFF);
  iom_prt_cmd(p,mport,LB2DM3_STATIS,PFUNC_RWR,0x00,SS->statbn,0x0F);
  iom_prt_cmd(p,mport,LB2DM3_AIS,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
  return (0);
}

int_4 lb2dm3_lna (PICSTRUCT *p, int_4 mport, LB2DM3_State* SS)
{
  int_4 pinval = (RFOPT_LNA & SS->rfopts) ? (0x00) : (0x01);
  int_4 pinreg = u32_setbits(VMMK2103_PIN_NULL,VMMK2103_PIN_BYPASS,((SS->hw_ver&0xF)==0x7)? !pinval : pinval);
  int_4 rtn = iom_prt_cmd (p,mport,LB2DM3_VMMK2103,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
  return rtn;
}

int_4 lb2dm3_rfpwr (PICSTRUCT *p, int_4 mport, LB2DM3_State* SS)
{
  int_4 stat_func, stat_ID, stat_addr, stat_val;
  real_4 adcval;

  iom_prt_cmd(p,mport,LB2DM3_LTC2362,PFUNC_PWR,0,1,0xFFFFFFFF);
  iom_prt_dly(p,mport,200);
  iom_prt_cmd(p,mport,LB2DM3_LTC2362,PFUNC_RRD,0,0,0xFFFFFFFF);
  iom_prt_sts(p,mport,LB2DM3_LTC2362,&stat_func,&stat_ID,&stat_addr,&stat_val);
  iom_prt_cmd(p,mport,LB2DM3_LTC2362,PFUNC_PWR,0,0,0xFFFFFFFF);

  /* Three segment piecewise linear fit to dBm */
  adcval = (real_4) stat_val;
  if (adcval < 0.0) adcval = 0.0;
  if (adcval > 3000.0) adcval = 3000.0;

  if      (adcval > 2513.0) adcval = (0.05836689*adcval) - 146.35;
  else if (adcval > 356.0)  adcval = (0.02300209*adcval) - 57.95;
  else                      adcval = (0.09740266*adcval) - 85.68;

  if (SS->rfopts & RFOPT_LNA) adcval = adcval - 0.5; 
  else                        adcval = adcval + 16.3;
    
  if (adcval < -45.0) adcval = -45.0;
  if (adcval > 10.0) adcval = 10.0;
  adcval = (adcval < 0.0) ? (adcval-0.5) : (adcval+0.5);

  return (int_4) adcval;
}

int_4 lb2dm3_hbf (PICSTRUCT *p, int_4 mport, LB2DM3_State* SS)
{
  int_4 i, coef, coeset = SS->coeset;
  int_4 size = sizeof(lb2dm3_hbfir_coefs0) / sizeof(lb2dm3_hbfir_coefs0[0]);

  for(i=0; i<size; ++i) {
    coef = (coeset==2)? lb2dm3_hbfir_coefs2[i] : 
	   (coeset==1)? lb2dm3_hbfir_coefs1[i] : 
			lb2dm3_hbfir_coefs0[i];
    iom_prt_cmd(p,mport,LB2DM3_HBDECFIR_0,PFUNC_RWR,i, coef&0xFFFF,0xFFFF);
    iom_prt_cmd(p,mport,LB2DM3_HBDECFIR_1,PFUNC_RWR,i, coef&0xFFFF,0xFFFF);
  }

  return (0);
}

int_4 lb2dm3_cfir (PICSTRUCT *p, int_4 mport, LB2DM3_State* SS)
{
  int_4 i, coefval, coefset;
  int_4 cfir_nfilt = lb2dm3_cfir_nfilt;
  int_4 cfir_taps = lb2dm3_cfir_taps;
  real_8 srate = SS->samprate;
  int_4 wbcoef = iom_getcfgl(p,mport,"HBFBYP",(SS->toggles&LB2DM3_TGLSS_HBFBYP)?2:0);

  wbcoef = (wbcoef==0)? 0: wbcoef-1;

  /* Choose equalization index (zero based) to nearest 5MHz sample rate */
  coefset = (int_4) (0.5 + ((1.0/5.0)*srate));
  coefset -= 1;
  coefset = (coefset >= 0) ? (coefset) : (0);
  coefset = (coefset <= (cfir_nfilt-1)) ? (coefset) : (cfir_nfilt-1);

  for (i=0; i<cfir_taps; i++) {
    coefval = lb2dm3_cfir_coefs[coefset][i];
    iom_prt_cmd(p,mport,LB2DM3_FIR_0,PFUNC_RWR,i,(coefval&0xFFFF),0xFFFF);
    iom_prt_cmd(p,mport,LB2DM3_FIR_1,PFUNC_RWR,i,(coefval&0xFFFF),0xFFFF);

    coefval = lb2dm3_wbcfir_coefs[wbcoef][i];
    iom_prt_cmd(p,mport,LB2DM3_FIR_2,PFUNC_RWR,i,(coefval&0xFFFF),0xFFFF);
    iom_prt_cmd(p,mport,LB2DM3_FIR_3,PFUNC_RWR,i,(coefval&0xFFFF),0xFFFF);
  }

  iom_prt_cmd(p,mport,LB2DM3_FIR_0,PFUNC_RWR,64,86408,0xFFFFFF);
  iom_prt_cmd(p,mport,LB2DM3_FIR_1,PFUNC_RWR,64,86408,0xFFFFFF);

  iom_prt_cmd(p,mport,LB2DM3_FIR_2,PFUNC_RWR,64,131072,0xFFFFFF);
  iom_prt_cmd(p,mport,LB2DM3_FIR_3,PFUNC_RWR,64,131072,0xFFFFFF);

  return (0);
}

int_4 lb2dm3_ads42b49 (PICSTRUCT *p, int_4 mport, LB2DM3_State *SS)
{
  /* Reset ADC */
  iom_prt_cmd(p,mport,LB2DM3_ADS42B49,PFUNC_PWR,0,1,0xFFFF);
  iom_prt_dly(p,mport,10000);
  iom_prt_cmd(p,mport,LB2DM3_ADS42B49,PFUNC_PWR,0,0,0xFFFF);
  iom_prt_dly(p,mport,10000);

  /* Configure ADC TPH settings */
  iom_prt_cmd(p,mport,LB2DM3_ADS42B49,PFUNC_RWR,0x3D,0x20,0xFFFF);
  iom_prt_cmd(p,mport,LB2DM3_ADS42B49,PFUNC_RWR,0x3F,0x12,0xFFFF);
  iom_prt_cmd(p,mport,LB2DM3_ADS42B49,PFUNC_RWR,0x40,0x34,0xFFFF);

  /* TPMA: 0=Normal, 3=Fixed Pat, 4=Ramp Set ADC Channel-A gain to maximum (6dB) */
  iom_prt_cmd(p,mport,LB2DM3_ADS42B49,PFUNC_RWR,0x25,0xC0,0xFFFF);
  iom_prt_dly(p,mport,10000);
  /* TPMB: 0=Normal, 3=Fixed Pat, 4=Ramp Set ADC Channel-B gain to maximum (6dB) */
  iom_prt_cmd(p,mport,LB2DM3_ADS42B49,PFUNC_RWR,0x2B,0xC0,0xFFFF);
  iom_prt_dly(p,mport,10000);
  /* Enable ADC Digital Output data = 0x0000; Don't enable test ramp... */
  iom_prt_cmd(p,mport,LB2DM3_ADS42B49,PFUNC_RWR,0x44,0x01,0xFFFF);
  iom_prt_dly(p,mport,10000);
  return (0);
}

int_4 lb2dm3_si5338 (PICSTRUCT *p, int_4 mport, LB2DM3_State *SS, int_4 check)
{
  PT_SI5338_REG SIR[SI5338_MAX_CFG_RG];
  PT_SI5338_MSCFG MS;
  int_4 i,rtn;
  int_4 stat_func, stat_pin, stat_addr, stat_data;
  int_4 reg_data45, reg_data46, reg_data47;
  real_8 rate = SS->samprate;

  /* Check for digital HBF in setup */
  if ((SS->toggles & LB2DM3_TGLSS_HBFBYP) == 0x00) rate *= 2.0;

  /* Initialize the Si5338 configuration */
  pt_si5338_rgc_default(&SIR[0], FPGA_MODID_LB2DM3);

  /* Configure internal / external reference (assume internal) */
  if (SS->toggles & LB2DM3_TGLSS_EXTREF) {
    pt_si5338_rgc_setbits(&SIR[0],28,0x1C,((0x07)&(0x00)),0);
    pt_si5338_rgc_setbits(&SIR[0],29,0x18,((0x03)&(0x00 >> 3)),0);
  }
  else {
    pt_si5338_rgc_setbits(&SIR[0],28,0x1C,((0x07)&(0x0A)),0);
    pt_si5338_rgc_setbits(&SIR[0],29,0x18,((0x03)&(0x0A >> 3)),0);
  }
  pt_si5338_rgc_setbits(&SIR[0],28,0x03,0x00,0);

  /* Configure input to P2_DIVIDER (Zerodly clock reference) */
  pt_si5338_rgc_setbits(&SIR[0],28,0x20,((0x01)&(0x00)),0);
  pt_si5338_rgc_setbits(&SIR[0],30,0x18,((0x03)&(0x00 >> 1)),0);

  /* Configure a divider solution for the MSx outputs                */
  /*   By preselecting only VCO multiples of 5MHz, guarantee Zerodly */
  /*   path is integer multiple, so we use Samplerate path as the    */
  /*   critical path choice                                          */
  MS.dbglvl = 0;            /* MS.dbglvl = DBG_LB for debug output */
  MS.inp_ref_frq    = SS->refclk;      /* default 10MHz ref clock */
  MS.clk_frq_des[0] = SS->vcoref;      /* STV6111 reference     */
  MS.clk_frq_des[1] = MS.inp_ref_frq;  /* Unused output         */
  MS.clk_frq_des[2] = rate;            /* ADC clock             */
  MS.clk_frq_des[3] = MS.inp_ref_frq;  /* Zero delay mode clock */
  MS.clk_frq_des[4] = MS.inp_ref_frq;  /* Unused output         */
  MS.ms_critical    = 2;               /* Critical path is sample rate */
  rtn = pt_si5338_msx_soln(&SIR[0], &MS);

  SS->vcoref = MS.clk_frq_act[0];
  rate = MS.clk_frq_act[2];
  if ((SS->toggles & LB2DM3_TGLSS_HBFBYP) == 0x00) rate *= 0.5; /* Digital HBF in setup */
  SS->samprate = rate;
  if (check) return rtn;
  if (rtn<0) {
    printf("Si5338 Invalid clock configuration specified. Bad... \n"); return(-2);
  }

  /* Update actual rates / references for precision queries */
  if (MS.clk_frq_act[0] != MS.clk_frq_des[0]) {
    vprint("Requested ADC sample rate requires non-integer RF VCO reference. Degraded RF phase noise may result.\n");
  }

  if (MS.clk_frq_act[2] != MS.clk_frq_des[2]) {
    vprint("Unable to obtain exact sample rate. True sample rate is: %21.16f MHz \n",rate);
  }

  for (i=0; i<SI5338_MAX_CFG_RG; i++) { 
    v3print(" SI5338_REG: %3.3u  %2.2X  0x%2.2X  0x%2.2X  0x%2.2X\n", 
      SIR[i].ra,(SIR[i].rv ^ si5338_lb2dm3_reg_defaults[i].rv),
      si5338_lb2dm3_reg_defaults[i].rv,SIR[i].rv,SIR[i].rm);
  }

  /* Disable outputs and pause LOL */
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,230,1<<4,1<<4); 
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,241,1<<7,1<<7);

  /* Configure all the registers on the part */
  for(i=0; i<SI5338_MAX_CFG_RG; i++){
    iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,SIR[i].ra,SIR[i].rv,SIR[i].rm);
  }

  /* Cfg PLL for Lock, initiate PLL lock, Restart LOL, Wait 25ms */
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,49,0,1<<7);     
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,246,1<<1,1<<1);
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,241,0x65,0xFF);
  iom_prt_dly(p,mport,25000);                             

  /* Validate Input Clock Status */
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RRD,218,0,0xFFFFFFFF);
  iom_prt_sts(p,mport,LB2DM3_SI5338,&stat_func,&stat_pin,&stat_addr,&stat_data);
  v3print("Si5338 Clock Status = %x\n",stat_data);

  /* Confirm PLL Lock */
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RRD,218,0,0xFF);
  iom_prt_sts(p,mport,LB2DM3_SI5338,&stat_func,&stat_pin,&stat_addr,&stat_data);
  v3print("Si5338 PLL Lock Status = %x\n",stat_data);

  /* Copy FCAL Values To Active Registers */
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RRD,237,0,0xFF);
  iom_prt_sts(p,mport,LB2DM3_SI5338,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data47 = stat_data & 0x03;
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RRD,236,0,0xFF);
  iom_prt_sts(p,mport,LB2DM3_SI5338,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data46 = stat_data & 0xFF;
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RRD,235,0,0xFF);
  iom_prt_sts(p,mport,LB2DM3_SI5338,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data45 = stat_data & 0xFF;

  /* Check lock status */
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,47,reg_data47,0x03);
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,46,reg_data46,0xFF);
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,45,reg_data45,0xFF);

  /* Set PLL To Use FCAL Values */
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,49,1<<7,1<<7);

  /* Enable Outputs */
  iom_prt_cmd(p,mport,LB2DM3_SI5338,PFUNC_RWR,230,0,1<<4);
  iom_prt_dly(p,mport,25000);

  return (0);
}

int_4 lb2dm3_stv6111 (PICSTRUCT *p, int_4 mport, LB2DM3_State *SS)
{
  int_4 i, stat_func, stat_pin, stat_addr, stat_data;
  int_u4 psel0,psel1, rfatn = PE43204_ATTN_00DB;
  PT_STV6111_REG STV;

  pt_stv6111_rgc_default(&STV);

  /* RF Frequency configuration */
  if (pt_stv6111_rgc_rffreq_(&STV, &SS->rffreq, SS->vcoref) != 0) {
    SS->rffreq = 900.0;
    print("Invalid RFFREQ. Using %f \n",SS->rffreq);
    pt_stv6111_rgc_rffreq_(&STV, &SS->rffreq, SS->vcoref);
  }

  /* RF Gain configuration */
  if (pt_stv6111_rgc_rfgain_(&STV, &SS->rfgain, &rfatn) != 0) {
    SS->rfgain = 0.0;
    print("Invalid RFGAIN. Using %f \n",SS->rfgain);
    pt_stv6111_rgc_rfgain_(&STV, &SS->rfgain, &rfatn);
  }

  /* Configure the attenuator */
  iom_prt_cmd(p,mport,LB2DM3_PE43204,PFUNC_PWR,0x00,rfatn,0xFFFF);

  /* Configure the tuner */
  for (i=0; i<STV6111_NUM_RG; i++) {
    if (i==0x09) continue;
    v2print("xSTV6111(RFFREQ)-> RG: 0x%2.2X  RV: 0x%2.2X \n",i,STV.rg[i]);
    iom_prt_cmd(p,mport,LB2DM3_STV6111,PFUNC_RWR,i,STV.rg[i],0xFF);
  }
  iom_prt_dly(p,mport,2000);

  /* Initiate VCO calibration */
  STV.rg[0x09] = u8_setbits(STV.rg[0x09],STV6111_RGM_09_VCOCAL,0x01);
  iom_prt_cmd(p,mport,LB2DM3_STV6111,PFUNC_RWR,0x09,STV.rg[0x09],0x0F);
  iom_prt_dly(p,mport,20000);

  /* Initiate VCO loop filter calibration (BW must = 23MHz!) */
  STV.rg[0x09] = u8_setbits(STV.rg[0x09],STV6111_RGM_09_BBFCAL,0x01);
  iom_prt_cmd(p,mport,LB2DM3_STV6111,PFUNC_RWR,0x09,STV.rg[0x09],0x0F);
  iom_prt_dly(p,mport,5000);

  /* Check VCO calibration status */
  iom_prt_cmd(p,mport,LB2DM3_STV6111,PFUNC_RRD,0x09,0,0xFF);
  iom_prt_dly(p,mport,25000);
  iom_prt_sts(p,mport,LB2DM3_STV6111,&stat_func,&stat_pin,&stat_addr,&stat_data);
  v2print("STV6111 REG 0x09 = 0x%8.8X FUNC=%02X ADDR=%04X\n",stat_data,stat_func,stat_addr);

  /* Now set RF BW and initiate calibration */       
  pt_stv6111_rgc_rfbw_(&STV, &SS->rfbw);
  iom_prt_cmd(p,mport,LB2DM3_STV6111,PFUNC_RWR,0x01,STV.rg[0x01],0xFF);
  iom_prt_cmd(p,mport,LB2DM3_STV6111,PFUNC_RWR,0x08,STV.rg[0x08],0xFF);
  iom_prt_dly(p,mport,5000);
  STV.rg[0x09] = u8_setbits(STV.rg[0x09],STV6111_RGM_09_BBFCAL,0x01);
  iom_prt_cmd(p,mport,LB2DM3_STV6111,PFUNC_RWR,0x09,STV.rg[0x09],0x0F);
  iom_prt_dly(p,mport,5000);
 
  /* Check lock status */ 
  iom_prt_cmd(p,mport,LB2DM3_STV6111,PFUNC_RRD,0x09,0,0xFF);
  iom_prt_dly(p,mport,25000);
  iom_prt_sts(p,mport,LB2DM3_STV6111,&stat_func,&stat_pin,&stat_addr,&stat_data);
  v2print("STV6111 REG 0x09 = 0x%8.8X FUNC=%02X ADDR=%04X\n",stat_data,stat_func,stat_addr);

  /* Check PLL lock status */
  iom_prt_cmd(p,mport,LB2DM3_STV6111,PFUNC_RRD,0x08,0,0xFF);
  iom_prt_dly(p,mport,25000);
  iom_prt_sts(p,mport,LB2DM3_STV6111,&stat_func,&stat_pin,&stat_addr,&stat_data);
  v2print("STV6111 PLL LOCK = %x\n",((0x01)&(stat_data)));
  iom_prt_dly(p,mport,2000);

  /* set preselectors */
  lb2dm3_presel_decode(p,mport,&psel0,&psel1,SS);
  iom_prt_cmd(p,mport,LB2DM3_SKY13384_0,PFUNC_PWR,0,psel0,0xFFFF);
  iom_prt_cmd(p,mport,LB2DM3_SKY13384_1,PFUNC_PWR,0,psel1,0xFFFF);

  return (0);
}

int_4 lb2dm3_dcs (PICSTRUCT *p, int_4 mport, LB2DM3_State* SS)
{
  int_4 pinval = (RFOPT_DCS & SS->rfopts) ? (0x00) : (0x01);
  int_4 pinreg = u32_setbits(DCS_PIN_NULL,DCS_PIN_BYPASS,pinval);
  iom_prt_cmd(p,mport,LB2DM3_DCS,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
  iom_prt_cmd(p,mport,LB2DM3_DCS,PFUNC_RWR,0x00,SS->dcsbn,0x0F);
  return 0;
}

int_4 lb2dm3_set_rfopts (PICSTRUCT *p, int_4 mport, int_4 rfopts, LB2DM3_State* SS)
{
  int_4 rtn = 0;
  int_4 rfold,rfdif;

  if ((rfopts & RFOPT_ENABLE) == 0) return (rtn);	/* no options enabled */

  rfold = SS->rfopts;		/* currently configured options */
  SS->rfopts = rfopts;
  rfdif = rfopts ^ rfold;	/* any differences ? */

  if (rfdif & RFOPT_LNA) rtn=lb2dm3_lna(p,mport,SS);
  if (rfdif & RFOPT_DCS) rtn=lb2dm3_dcs(p,mport,SS);
  if (rfdif & RFOPT_AIS) rtn=lb2dm3_ais(p,mport,SS);

  return (rtn);
}

int_4 lb2dm3_clkstat (PICSTRUCT *p, int_4 mport, LB2DM3_State* SS)
{
  int_4 status=0, extref;
  int_4 stat_func,stat_pin,stat_addr,stat_data;

  iom_prt_cmd (p,mport,LB2DM3_MODULE,PFUNC_PRD,1,0,0xFF);
  iom_prt_sts (p,mport,LB2DM3_MODULE,&stat_func,&stat_pin,&stat_addr,&stat_data);

  extref = SS->toggles & LB2DM3_TGLSS_EXTREF;
  if (extref && (stat_data&0x1)==0) status |= 0x1;
  if ((stat_data&0x2)==0) status |= 0x2;

  return (status);
}

/*   The "_reset" function name is an HH interface retrain  */
int_4 lb2dm3_reset(PICSTRUCT *p, int_4 mport)
{
  int_4 jvm=iom_set_prt_jvm(p,mport,0);
  iom_prt_cmd(p,mport,LB2DM3_ROOT,PFUNC_PWR,0,0x0005,0xFFFF);	
  iom_prt_dly(p,mport,10000);
  iom_prt_cmd(p,mport,LB2DM3_ROOT,PFUNC_PWR,0,0x0000,0xFFFF);
  iom_prt_dly(p,mport,10000);
  iom_set_prt_jvm(p,mport,jvm);
  return (0);
}

int_4 lb2dm3_enable (PICSTRUCT *p, int_4 mport, int_4 ena)
{
  int_4 pinval = (ena>0) ? (0x01) : (0x00);
  int_4 pinreg = u32_setbits(ROOT_PIN_NULL,ROOT_PIN_ENABLE,pinval);
  int_4 jvm=iom_set_prt_jvm(p,mport,0);
  iom_prt_cmd(p,mport,LB2DM3_ROOT,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
  iom_set_prt_jvm(p,mport,jvm);
  return 0;
}

int_4 lb2dm3_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags) 
{
  LB2DM3_State *SS;
  if (mport==3) return lb2dm3_setup(p,1,dir,bits,rate,gain,flags)+
                       lb2dm3_setup(p,2,dir,bits,rate,gain,flags);

  /* populate State Structure with startup values - alloc_state zero fills */
  SS = (LB2DM3_State*)iom_alloc_state(p,mport,LB2DM3_SIZE);                                     
  SS->refclk	= 10;                                                   
  SS->vcoref	= 30.0;                                                   
  SS->rffreq	= iom_getcfgd (p,mport,"RFFREQ",1000.0);                                                
  SS->rfbw	= iom_getcfgd (p,mport,"RFBW",100.0);                                                
  SS->rfgain	= iom_getcfgd (p,mport,"RFGAIN",0.0);                                                
  SS->rfopts    = iom_getcfgm (p,mport,"RFOPTS",RFOPT_ENABLE,RFOPTLIST);
  SS->toggles	= 0;
  SS->samprate	= rate*1e-6;
  SS->bits	= bits;
  SS->mtgooff   = iom_getcfgl(p,mport,"MTGOOFF",0);
  SS->dcsbn     = iom_getcfgl(p,mport,"DCSBN",14);
  SS->statbn    = iom_getcfgl(p,mport,"STATBN",22);
  SS->coeset	= iom_getcfgl(p,mport,"COESEL",0);
  SS->presel	= iom_getcfgl(p,mport,"PRESEL",0);

  SS->hw_ver    = pic_jpmemrd(p,mport,MODDETECT_ADDR,1);

  if (iom_getcfgl(p,mport,"PREFX",0)>0)      SS->rfopts |= RFOPT_XREF;
  if (iom_getcfgl(p,mport,"RFOPT_XREF",0)>0) SS->rfopts |= RFOPT_XREF;
  if (iom_getcfgl(p,mport,"RFOPT_LNA",0)>0)  SS->rfopts |= RFOPT_LNA;
  if (iom_getcfgl(p,mport,"RFOPT_DCS",0)>0)  SS->rfopts |= RFOPT_DCS;
  if (iom_getcfgl(p,mport,"RFOPT_AIS",0)>0)  SS->rfopts |= RFOPT_AIS;
  if (iom_getcfgl(p,mport,"RFOPT_AGC",0)>0)  SS->rfopts |= RFOPT_AGC;

  if (iom_getcfgl(p,mport,"PPSTAG",0)>0)     SS->toggles |= LB2DM3_TGLSS_PPSTAG;
  if (iom_getcfgl(p,mport,"LSBP",0)>0)       SS->toggles |= LB2DM3_TGLSS_LSBP;
  if (iom_getcfgl(p,mport,"HBFBYP",0)>0)     SS->toggles |= LB2DM3_TGLSS_HBFBYP;
  if (iom_getcfgl(p,mport,"SPINV",0)>0)      SS->toggles |= LB2DM3_TGLSS_SPINV;
  if (iom_getcfgl(p,mport,"MTGO",0)>0)       SS->toggles |= LB2DM3_TGLSS_MTGO;
  if (iom_getcfgl(p,mport,"PRESELBYP",0)==0) SS->toggles |= LB2DM3_TGLSS_PRESEL;
  if (iom_getcfgl(p,mport,"PREFX",0)>0)      SS->toggles |= LB2DM3_TGLSS_EXTREF;
  if (iom_getcfgl(p,mport,"RAMPOUT",0)>0)    SS->toggles |= LB2DM3_TGLSS_RAMP;
  if (iom_getcfgl(p,mport,"LB2D_NEQ",0)==0)  SS->toggles |= LB2DM3_TGLSS_EQ;
  if (iom_getcfgl(p,mport,"LB2D_DIO",0)>0)   SS->toggles |= LB2DM3_TGLSS_DIO;
  if (iom_getcfgl(p,mport,"LB2D_LFSR",0)>0)  SS->toggles |= LB2DM3_TGLSS_LFSR;

  if (SS->samprate>125) SS->toggles |= LB2DM3_TGLSS_HBFBYP;

  iom_init (p,mport);                                                                               

  iom_set_prt_jvm(p,mport,1);	/* use JVM as commander */

  /* Reset the FPGA */
  SS->toggles |= LB2DM3_TGLSS_RSTFPGA;
  lb2dm3_toggles (p,mport,SS);
  SS->toggles &= ~LB2DM3_TGLSS_RSTFPGA;
  iom_set_state (p,mport);                         

  /* Call the module setup functions */
  lb2dm3_toggles (p,mport,SS);
  lb2dm3_hbf (p,mport,SS);   
  lb2dm3_cfir (p,mport,SS);   
  lb2dm3_si5338 (p,mport,SS,0);   
  lb2dm3_ads42b49 (p,mport,SS);
  lb2dm3_stv6111 (p,mport,SS);
  lb2dm3_lna (p,mport,SS);
  lb2dm3_dcs (p,mport,SS);
  lb2dm3_ais (p,mport,SS);
  lb2dm3_adclm (p,mport,1,SS);
  iom_set_state (p,mport);	/* also performs iom_exec */

  if (p->verbose>1) lb2dm3_dump_state(p,mport,SS);

  return (0); 
}

int_4 lb2dm3_getkeyl (PICSTRUCT *p, int_4 mport, int_4 key)       
{                                                                              
  int_4 rtn=-1;                                                               
  LB2DM3_State* SS = (LB2DM3_State*) iom_get_state (p,mport);                         
       if (SS==NULL)        rtn = -1;
  else if (key==KEY_RFOPTS) rtn = SS->rfopts;
  else if (key==KEY_RFBW)   rtn = SS->rfbw;
  else if (key==KEY_RFGAIN) rtn = SS->rfgain;
  else if (key==KEY_RFPWR)  rtn = lb2dm3_rfpwr (p,mport,SS);
  else if (key==KEY_ADLM)   rtn = lb2dm3_adclm (p,mport,-1,SS);
  else if (key==KEY_DCSBN)  rtn = SS->dcsbn;
  else if (key==KEY_CLKSTAT) rtn = lb2dm3_clkstat (p,mport,SS);
  else printf("Unhandled getkeyl=%d in lb2dm3\n",key);
  return rtn;
}                                                

real_8 lb2dm3_getkeyd (PICSTRUCT *p, int_4 mport, int_4 key)       
{                                                                              
  real_8 rtn=-1.0;                                                               
  LB2DM3_State* SS = (LB2DM3_State*) iom_get_state (p,mport);                         
       if (SS==NULL)        rtn =  -1;
  else if (key==KEY_RFOPTS) rtn = SS->rfopts;
  else if (key==KEY_RFFREQ) rtn = SS->rffreq;
  else if (key==KEY_RFBW)   rtn = SS->rfbw;
  else if (key==KEY_RFGAIN) rtn = SS->rfgain;
  else if (key==KEY_RFPWR)  rtn = lb2dm3_rfpwr (p,mport,SS);
  else if (key==KEY_DRATE)  rtn = SS->samprate*1e6;
  else printf("Unhandled getkeyd=%d in lb2dm3\n",key);
  return rtn;
}                                                

int_4 lb2dm3_setkeyl (PICSTRUCT *p, int_4 mport, int_4 key, int_4 i4val)       
{                                                                              
  int_4 rtn = 0;                                                               
  LB2DM3_State* SS = (LB2DM3_State*) iom_get_state (p,mport);                         
  if (SS==NULL) return -1;
       if (key==KEY_RFOPTS) {                   rtn = lb2dm3_set_rfopts (p,mport,i4val,SS); }
  else if (key==KEY_DCSBN)  { SS->dcsbn=i4val;  rtn = lb2dm3_dcs (p,mport,SS); }
  else if (key==KEY_RFGAIN) { SS->rfgain=i4val; rtn = lb2dm3_stv6111 (p,mport,SS); }
  else printf("Unhandled setkeyl=%d in lb2dm3\n",key);
  iom_set_state (p,mport);                         
  return rtn;
}                                                

int_4 lb2dm3_setkeyd (PICSTRUCT *p, int_4 mport, int_4 key, real_8 r8val)       
{                                                                              
  int_4 i4val = r8val;
  int_4 rtn = 0;                                                               
  LB2DM3_State* SS = (LB2DM3_State*) iom_get_state (p,mport);                         
  if (SS==NULL) return -1;
       if (key==KEY_RFFREQ) { SS->rffreq=r8val; rtn = lb2dm3_stv6111 (p,mport,SS); }
  else if (key==KEY_RFBW)   { SS->rfbw=r8val;   rtn = lb2dm3_stv6111 (p,mport,SS); }
  else if (key==KEY_RFGAIN) { SS->rfgain=r8val; rtn = lb2dm3_stv6111 (p,mport,SS); }
  else printf("Unhandled setkeyd=%d in lb2dm3\n",key);
  iom_set_state (p,mport);                         
  return rtn;
}                                                

int_4 lb2dm3_fplan (PICSTRUCT *p, int_4 mport, real_8 *dval, int_4 len) 
{
  int_4 rtn1,rtn2;
  LB2DM3_State SS;
  PT_STV6111_REG STV;

  vfill (0,(int_4*)&SS,sizeof(SS));
  SS.refclk	= 10;                                                   
  SS.vcoref	= 30.0;                                                   
  SS.samprate	= dval[0];
  SS.rffreq	= dval[1];
  rtn1=lb2dm3_si5338 (p,0,&SS,1);   

  vfill (0,(int_4*)&STV,sizeof(STV));
  pt_stv6111_rgc_default(&STV);
  rtn2=pt_stv6111_rgc_rffreq_(&STV, &SS.rffreq, SS.vcoref);

  dval[0]=SS.samprate;
  dval[1]=SS.rffreq;
  dval[2]=SS.refclk;
  dval[3]=SS.vcoref;
  
  return rtn1;
}
