/****************************************************************************/
/*   GPS module interface routines                                          */
/*   sysreg = {selpps,selirb, fixed,apflg,apena,good}                       */
/****************************************************************************/

#define GPSOPTLIST "NPPS,APWR,AFVP,PTST,DPPS,MANUAL"
#define GPSOPT_NPPS 0x01
#define GPSOPT_APWR 0x02
#define GPSOPT_AFVP 0x04
#define GPSOPT_PTST 0x08
#define GPSOPT_DPPS 0x10
#define GPSOPT_MANU 0x20
#define GPSFLG_SETTIME

int_4 pic_setup_gps (PICSTRUCT *p, int_4 port) {
  int_4 i,mode,opts,sysreg,stat;
  mode = findintflagdef("GPSMODE",p->config,1);
  opts = (mode==0)? 0x1 : 0x0;	/* if MODE=0 turn off PPS */
  opts = findmaskflagdef("GPSOPTS",p->config,GPSOPTLIST,opts);
  sysreg = 0x00041C00; /* mrev|navg=4|offset=28|sysreg */
  if (opts&GPSOPT_APWR) sysreg |= 0x02;
  if (opts&GPSOPT_AFVP) sysreg |= 0x04;
  if (mode==0)          sysreg |= 0x08;
  if (opts&GPSOPT_PTST) sysreg |= 0x10;
  if (opts&GPSOPT_NPPS) sysreg |= 0x20;
  if (opts&GPSOPT_DPPS) sysreg |= 0x40;
  if (opts&GPSOPT_MANU) sysreg |= 0x80;
  sysreg = findintflagdef("GPSREG",p->config,sysreg);
  i = (port==9)? 0 : port-1;
  sysreg |= p->mrev[i]<<24;
  pic_jpmemwr (p,port,0x0000,sysreg,1);		/* only 1st byte is FPGA sysreg */
  pic_jpmemwr (p,port,0xFD000008,sysreg,4);
  pic_loadfile (p, "icegps", FLG_JVM|port);
  stat=pic_jpmemrd (p,port,0xFD000008,4);
  printf("GPS setup on port=%d sysreg=%08x rb=%08x\n",port,sysreg,stat);
  return 0;
}

real_8 dmi2deg (int val) {
  int deg, min, sign = (val<0)? -1 : 1;
  val *= sign;
  deg = val/1000000;
  min = val - deg*1000000;
  return sign * (deg + min/600000.0);
}

#define MJS_1970_TO_1950 631152000

int_4 dump_gps (PICSTRUCT *p, int_4 side, int_4 flag, real_8 *dval) {
  int_4 i,j,k,len,lfn,addr,year,doy,sec,stage=0,date=0,time=0,nos=0,dop=0,week,status=0,retry=8;
  int_4 cmd,istat,jstat,kstat; int_4 data[512]; char stmp[80],ctmp[32], *c=(char*)data; 
  real_8 soy,lat,lon,alt,rtsec,rtmsec,sow,mjs,smjs; 
  int_8 cmjs; int_4 delay,ptime=1000;	/* processing time usec for chrony settime call */
  struct timeval tv;
  FILE *f;

  cmd = findintflagdef("GPSCMD",p->config,0);
  if (cmd!=0) pic_jpmemwr (p,side,0xFD00000C,cmd,4);
  lfn = findstrflag("GPSDFN",p->config,ctmp,0);

  DO_OVER:
  istat = pic_jpmemrd(p,side,0xFD000100,1);

             i = pic_jpmemrd(p,side,0xFD000104,4); week=i>>16; dop=(i>>12)&0xF; stage=(i>>8)&0xF; nos=i&0xFF;
  sow = 1.0e-3 * pic_jpmemrd(p,side,0xFD000108,4);
  alt = 1.0e-2 * pic_jpmemrd(p,side,0xFD00010C,4);
  lat = dmi2deg( pic_jpmemrd(p,side,0xFD000110,4) );
  lon = dmi2deg( pic_jpmemrd(p,side,0xFD000114,4) );

  jstat = pic_jpmemrd(p,side,0xFD000100,1);

  i = pic_jpmemrd (p,side,0x1000,4);    // read status to lock counts into rtmsec register
  j = pic_jpmemrd (p,side,0x1004,4);    // rtmsec of status read
  k = ((j>>28)&0xF)*100 + ((j>>24)&0xF)*10 + ((j>>20)&0xF)*1;
  rtmsec = 0.001 * k;
  k += (j&0x1FFFF)*1000;
  rtsec = 0.001 * k;
  sow += rtmsec;

    // make sure reads were consistent with JVM parsing at .3 msec into second 0=pre 1=dur 2=after
       if (istat==0 && jstat==0 && rtmsec<0.5) sow += 1;
  else if (istat==2 && jstat!=1 && rtmsec>0.3) ;
  else if (--retry>0) { usleep(1000); goto DO_OVER; }

  if (retry==0) {
    printf("GPS device on port=%d not responding. Aborting operation.\n",side);
    status = -1;
  }

  soy = 86400.0*7*(week-2048) + sow;
  mjs = 2185747182.0 + soy;	/* based on the second rollover in 2019 */

  i = findintflagdef("GPS2SYS_WIN",p->config,-1);
  j = findintflagdef("GPS2CHRONY",p->config,-1);
  if (i<0 && j<0);
  else if (status!=0 || nos==0 || stage<=0 || mjs<2e9) 
    printf("GPS not adjusting system clock due to Status=%d NoS=%d Stage=%d MJS=%d\n",status,nos,stage,mjs);
  else {
    gettimeofday(&tv,0);
    smjs = (double)tv.tv_sec + 1.0e-6*tv.tv_usec + MJS_1970_TO_1950;
    rtsec = mjs-smjs; rtmsec = rtsec*1000;
    if (j>=0 && abs(rtsec)<60.0) {	/* dont wait for an adjust of more than 60 seconds */
      cmjs = (int_8)(mjs+0.99); k = cmjs%86400;		/* chrony time */
      delay = ((cmjs-mjs)*1000000)-ptime; udelay(delay);
      sprintf(ctmp,"%02d:%02d:%02d",k/3600,(k/60)%60,k%60);
      sprintf(stmp,"/usr/bin/chronyc settime %s",ctmp);
      kstat=system(stmp);
      printf("GPS to chrony at %s off=%f delay=%d status=%d\n",ctmp,rtsec,delay,kstat);
    }
    else if (rtmsec <= -i || rtmsec >= i) {
      tv.tv_sec = floor(mjs) - MJS_1970_TO_1950;
      tv.tv_usec = (mjs-floor(mjs)) * 1e6;
      istat=settimeofday(&tv,0);
      if (istat!=0) perror("settimeofday: ");
      printf("GPS adjusting system clock by %f sec. Tolr=%dmS Stat=%d\n",rtsec,i,istat);
    }
  }

  if (dval!=NULL) {
    dval[0] = mjs;
    dval[1] = lat;
    dval[2] = lon;
    dval[3] = alt;
    dval[4] = stage;
    dval[5] = nos;
  } else {
    print("GPS NoS=%d DoP=%d Stage=%d MJS=%f Lat=%f Lon=%f Alt=%f\n",nos,dop,stage,mjs,lat,lon,alt);
  }
  if (lfn>0) {
    f = fopen(ctmp,"a");
    if (f==NULL) printf("Problem opening GPSDFN=%s\n",ctmp);
    else {
      fprintf(f,"GPS ");
      for (i=0x20; i<0x48; i+=4) { j=pic_jpmemrd(p,side,0xFD000000+i,4); fprintf(f,"0x%08x ",j); }
      fprintf(f,"\n");
      fclose(f);
    } 
  }
  if (p->verbose==1) {
    print("Cfg    = %08x\n",pic_jpmemrd(p,side,0xFD000008,4));
    i = pic_jpmemrd(p,side,0xFD000020,4);
    print("Stage  = %08x\n",(i>>24)&0xFF);
    print("Offset = %08x\n",pic_jpmemrd(p,side,0xFD000028,4));
    print("Diff   = %08x\n",pic_jpmemrd(p,side,0xFD00002C,4));
    print("VCO    = %08x\n",pic_jpmemrd(p,side,0xFD000038,4));
  }
  if (p->verbose==2) {
    len = pic_jpmemrd(p,side,0xFD000118,2); len=min(len,1024);
    for (i=0,addr=0x2000; i<(len+3)/4; i++,addr+=4) data[i] = pic_jpmemrd (p,side,addr,4); c[len]=0;
    print("GPS len=%d nema=\n%s\n",len,c);
    vprint("Cfg    = %08x\n",pic_jpmemrd(p,side,0xFD000008,4));
    i = pic_jpmemrd(p,side,0xFD000020,4);
    print("Stage  = %08x\n",(i>>24)&0xFF);
    print("Count  = %08x\n",(i>>16)&0xFF);
    print("Status = %08x\n",pic_jpmemrd(p,side,0xFD000024,4));
    print("Offset = %08x\n",pic_jpmemrd(p,side,0xFD000028,4));
    print("Diff   = %08x\n",pic_jpmemrd(p,side,0xFD00002C,4));
    print("AccOff = %08x\n",pic_jpmemrd(p,side,0xFD000030,4));
    print("AccDif = %08x\n",pic_jpmemrd(p,side,0xFD000034,4));
    print("VCO    = %08x\n",pic_jpmemrd(p,side,0xFD000038,4));
    print("Calib  = %08x\n",pic_jpmemrd(p,side,0xFD00003C,4));
    print("dPVCO  = %08x\n",pic_jpmemrd(p,side,0xFD000040,4));
    print("dFVCO  = %08x\n",pic_jpmemrd(p,side,0xFD000044,4));
    print("Lat    = %08x\n",pic_jpmemrd(p,side,0xFD000050,4));
    print("Lon    = %08x\n",pic_jpmemrd(p,side,0xFD000054,4));
    print("Alt    = %08x\n",pic_jpmemrd(p,side,0xFD000058,4));
    print("DoP    = %08x\n",pic_jpmemrd(p,side,0xFD00005C,4));
    print("NoS    = %08x\n",pic_jpmemrd(p,side,0xFD000060,4));
    print("Week   = %08x\n",pic_jpmemrd(p,side,0xFD000064,4));
    print("SoW    = %08x\n",pic_jpmemrd(p,side,0xFD000068,4));
    print("SoG    = %08x\n",pic_jpmemrd(p,side,0xFD00006C,4));
  }
  return status;
}

int_4 pic_dump_gps (PICSTRUCT *p, int_4 port) {
  return dump_gps (p, port, 0, NULL);
}
