/****************************************************************************/
/*                                                                          */
/*   D2AWGM3 module interface routines for the ICEPIC device target          */
/*                                                                          */
/****************************************************************************/

#include "iomlib_d2awgm3x.h"

#define D2AWGM3_STATE 0xFD000000  /* Base address for state structure register */
#define D2AWGM3_SIZE sizeof(D2AWGM3_State)

typedef struct {
  int_u4 chksum;	/* 32 bit checksum|size of packet */
  int_4  sysreg;	/* system options register */
  real_8 samprate;	/* A/D sample rate in MHz */
  real_8 dacrate;
  real_8 rffreq;
  int_4  rfgain;
  int_4  rfattn;
  int_4  rfopts;
  int_4  mtgooff;
  int_2  interp;	/* FPGA interpolation */
  int_2  dacint;	/* DAC interpolation */
  int_1  hsmode;	/* high speed mode */
  int_1  rg_attn;
  int_1  rg_vga;
  int_1  mixpd;		/* mixer power down */
} D2AWGM3_State;

int_4 d2awgm3_dump_state (PICSTRUCT *p, int_4 mport, D2AWGM3_State *SS)                                                
{                                                                                                                     
  printf("SS.chksum:    %08X   \n",SS->chksum);                                            
  printf("SS.sysreg:    %08X   \n",SS->sysreg);                                            
  printf("SS.dacrate:   %lf    \n",SS->dacrate);                                          
  printf("SS.samprate:  %lf    \n",SS->samprate);                                          
  printf("SS.rffreq:    %lf    \n",SS->rffreq);                          
  printf("SS.rfgain:    %d     \n",SS->rfgain);                          
  printf("SS.rfopts:    %08X   \n",SS->rfopts);                           
  return 0;                                                                
}                                                                               

/*---------------------------------------------------------------------------*/
/* Internal Private Interfaces                                               */
/*                                                                           */
/*   NOTE: These routines should NEVER be called by external libraries.      */
/*         The API to these routines is fluid and is only self-consistent    */
/*         within a release series.                                          */
/*                                                                           */
/*---------------------------------------------------------------------------*/

int_4 d2awgm3_calculate_rfpwr (int_4 adc_rfpwr, int_4 atten_qdb, real_8 freqmhz)
{
  /* RF power meter is log-linear meter, but the solution curve varies with */
  /* frequency. This table is used to get the linear (slope / y-intercept)  */
  /* fit coefficients, and freqmhz is used to interpolate the surface       */
  /* between frequencies. The result is accurate to about 1dB.              */
  enum { RFPWR_TBL_SZ = 26 };
  static real_8 d2awg_m3_rfpwr_tbl[RFPWR_TBL_SZ][3] = {
     {   0.0, 4.6821315E-02, -59.22189}, { 100.0, 4.7154218E-02, -61.85349},
     { 200.0, 4.7427673E-02, -62.97015}, { 300.0, 4.7674309E-02, -63.54081},
     { 400.0, 4.8336752E-02, -64.10522}, { 500.0, 4.8874997E-02, -64.74007},
     { 600.0, 4.9617149E-02, -65.32240}, { 700.0, 5.0067808E-02, -65.99976},
     { 800.0, 5.0547868E-02, -66.55621}, { 900.0, 5.0625544E-02, -66.93533},
     {1000.0, 5.0322998E-02, -67.01509}, {1100.0, 4.9655993E-02, -66.91844},
     {1200.0, 4.8734497E-02, -66.74814}, {1300.0, 4.7813214E-02, -66.36892},
     {1400.0, 4.6602238E-02, -65.97525}, {1500.0, 4.5890063E-02, -65.36910},
     {1600.0, 4.5359418E-02, -64.87202}, {1700.0, 4.4941839E-02, -64.26146},
     {1800.0, 4.4212226E-02, -63.76649}, {1900.0, 4.3545038E-02, -63.30266},
     {2000.0, 4.2666040E-02, -62.93183}, {2100.0, 4.2282421E-02, -62.67810},
     {2200.0, 4.1794870E-02, -62.33878}, {2300.0, 4.1251764E-02, -61.96976},
     {2400.0, 4.0923487E-02, -61.50374}, {2500.0, 4.0676787E-02, -61.17500}
  };

  int_4  i,idx_low,idx_high;
  real_8 x1, x2, y1, y2, ym, yb;
  real_8 m_rf, b_rf, rfpwr_dbm, rffreq;

  /* Constrain input frequency to prevent overflow */
  rffreq = (freqmhz < d2awg_m3_rfpwr_tbl[0][0]) ?  (d2awg_m3_rfpwr_tbl[0][0]) : (freqmhz);
  rffreq = (freqmhz > d2awg_m3_rfpwr_tbl[RFPWR_TBL_SZ-1][0]) ?  (d2awg_m3_rfpwr_tbl[RFPWR_TBL_SZ-1][0]) : (freqmhz);

   /* Find frequency range of interest (in 1st nyquist zone) */
  idx_low = 0;
  for (i=0; i<RFPWR_TBL_SZ; i++) {
      idx_low = (rffreq > d2awg_m3_rfpwr_tbl[i][0]) ? (i) : (idx_low);
  }
  idx_high = idx_low + 1;
  if (idx_high >= RFPWR_TBL_SZ) {
      idx_high = RFPWR_TBL_SZ - 1;
      idx_low  = RFPWR_TBL_SZ - 2;
  }

  /* Linearly interpolate the slope coefficient */
  x2 = d2awg_m3_rfpwr_tbl[idx_high][0];
  x1 = d2awg_m3_rfpwr_tbl[idx_low][0];
  y2 = d2awg_m3_rfpwr_tbl[idx_high][1];
  y1 = d2awg_m3_rfpwr_tbl[idx_low][1];
  ym = (y2-y1)/(x2-x1);
  yb = y2 - ym*x2;
  m_rf = ym*rffreq + yb;

  /* Linearly interpolate the intercept coefficient */
  x2 = d2awg_m3_rfpwr_tbl[idx_high][0];
  x1 = d2awg_m3_rfpwr_tbl[idx_low][0];
  y2 = d2awg_m3_rfpwr_tbl[idx_high][2];
  y1 = d2awg_m3_rfpwr_tbl[idx_low][2];
  ym = (y2-y1)/(x2-x1);
  yb = y2 - ym*x2;
  b_rf = ym*rffreq + yb;

  /* Calculate RF power at input of the attenuator */
  rfpwr_dbm = m_rf*((real_8) adc_rfpwr) + b_rf;

  /* Translate the measurement to output of attenuator (4x in dB) */
  rfpwr_dbm = rfpwr_dbm - (0.25*((real_8) atten_qdb));

  /* Limit the result to realistic integer range to prevent overflow */
  rfpwr_dbm = (rfpwr_dbm < -100.0) ? (100.0) : (rfpwr_dbm);
  rfpwr_dbm = (rfpwr_dbm >   40.0) ?  (40.0) : (rfpwr_dbm);

#ifdef DBG_PRINT
  printf("D2AWGM3_RFPWR_DEBUG: %d %d %d %20.15f %20.15f %20.15f \n",
           idx_low,idx_high,adc_rfpwr,atten_qdb,rffreq,m_rf,b_rf);
#endif

  return ((int_4) rfpwr_dbm);
}

int_4 d2awgm3_rfpwr (PICSTRUCT *p, int_4 mport, D2AWGM3_State* SS)
{
  int_4  cmfunc, cmpin, cmaddr, cmdata;
  int_4  adc_rfpwr, rfpwrdbm;

  /* Read RF power meter ADC */
  iom_prt_cmd(p,mport,D2AWGM3_MCP3421,PFUNC_RRD,0,0,0xFFFFFFFF);
  iom_prt_sts(p,mport,D2AWGM3_MCP3421,&cmfunc,&cmpin,&cmaddr,&cmdata);
  /* printf("CM: 0x%8.8X  0x%8.8X \n",cmaddr,cmdata); */
  adc_rfpwr = (cmaddr == 0x00000000) ? ((cmdata & 0x0FFF0000)>>16) : (0);

  /* Calculate RF output power in dBm using calibrated table */
  rfpwrdbm = d2awgm3_calculate_rfpwr(adc_rfpwr, SS->rg_attn, SS->rffreq);

  return (rfpwrdbm);
}

int_4 d2awgm3_max5868_pin (PICSTRUCT *p, int_4 mport, int_4 pinmask, int_4 rstate)
{
  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_PWR,0x00,(rstate)?pinmask:0,pinmask);
  if (pinmask&MAX5868_MUTE)	/* ok MUTE is a pin and a register */
    iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x00,(rstate)?0x04:0,0x04);
  return 0;
}

real_8 d2awgm3_max5868_tune (PICSTRUCT *p, int_4 mport, int_8 rate, real_8 tfreq, int_4 specinv)
{
  real_8 fcw_full,fcw_rem;
  int_8 fcw,nfw,dfw;
  int_8 scale;
  int_4 cfncoe[5];
  int_4 cfgdsp;
  real_8 act_freq,act_vco;
  int_4 i;

  scale = pow(2,33);

  fcw_full = (tfreq / (real_8)rate)*scale;
  fcw = (int_8)fcw_full;
  fcw_rem = fcw_full - (real_8)fcw;
  nfw = (int_8)(fcw_rem * pow(2,18)-1);
  dfw = (int_8)((real_8)nfw/fcw_rem);

  act_vco = (real_8)fcw + (real_8)nfw / (real_8)dfw;
  act_freq = (((real_8)fcw + (real_8)nfw/(real_8)dfw) * (real_8)rate / scale);

  cfncoe[0] = nfw & 0xFF;
  cfncoe[1] = (nfw & 0xFF00)>>8;
  cfncoe[2] = ((dfw & 0x0003F)<<2) | ((nfw & 0x030000)>>16);
  cfncoe[3] = ((dfw & 0x03FC0)>>6);
  cfncoe[4] = ((dfw & 0x7C000)>>14);

/*
  printf("MAX5868 Tune: Desired Freq = %f, Actual Freq = %f\n", tfreq,act_freq);
  printf("MAX5868 Tune: fcw_full=%f, fcw=%ld, fcw_rem=%f, nfw=%ld, dfw=%ld\n",fcw_full,fcw,fcw_rem,nfw,dfw);
*/

  for (i=0; i<4; i++) {
    iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x02+i,(fcw>>(i*8))&0xFF,0xFFFF);
  }

  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x60,0x18,0xFFFF);

  for (i=0; i<5; i++) {
    iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x6B+i,cfncoe[i],0xFFFF);
  }

  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x60,0x00,0xFFFF);
  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x01,0x01,0x0001);
  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x00,0x00,0x01);
  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x51,specinv<<1,0x02);

  return (act_freq);
}

int_4 d2awgm3_lmx2582_mix (PICSTRUCT *p, int_4 mport, int_4 mixpd)
{
  int_4 (*lmx2582_regs)[2];

  lmx2582_regs = &lmx2582_base_table[0];

  lmx2582_regs[6][1]  = (lmx2582_regs[6][1] & 0xFF7F) | (mixpd<<7);
  lmx2582_regs[21][1] = (lmx2582_regs[21][1] & 0xFBFF) | (mixpd<<10);

  iom_prt_cmd(p,mport,D2AWGM3_LMX2582,PFUNC_RWR,lmx2582_regs[6][0],lmx2582_regs[6][1],0xFFFF);
  iom_prt_cmd(p,mport,D2AWGM3_LMX2582,PFUNC_RWR,lmx2582_regs[21][0],lmx2582_regs[21][1],0xFFFF);

  return(0);
}

int_4 d2awgm3_setgain (PICSTRUCT *p, int_4 mport, int_4 gain, int_4 key, D2AWGM3_State* SS)
{
  int_4  gain_index, rgval;
  real_8 x,gain_correction, freq=SS->rffreq;

  if (key==KEY_RFATTN) {
    if (p->xcvr!=0) { SS->rfattn = doXcvrFunc(p,"ATTN",(double)gain); }
    else { SS->rg_attn = gain; iom_prt_cmd(p,mport,D2AWGM3_PE43704,PFUNC_RWR,0x00,gain,0xFFFF); }
    return 0;
  }
  if (key==KEY_MGAIN) {
    iom_prt_cmd(p,mport,D2AWGM3_LMH6401,PFUNC_RWR,0x02,gain,0xFFFF);
    SS->rg_vga = gain;
    return 0;
  }

  /* Linear plus 5rd order curve fit to gain slope correction: freq in MHz */
  x = (freq-1500)/950;
  gain_correction = 2.6*pow(x,5) + 3.4*pow(x,4) - 1.8*pow(x,3) - 2.5*pow(x,2) - 0.81*x + 0.0045*freq + 11.25;
  /*v3print("freq = %f, gain correction = %f\n",freq,gain_correction);*/

  gain_correction = (gain_correction <  0.0) ?  (0.0) : (gain_correction);
  /* gain_correction = (gain_correction > 18.0) ? (18.0) : (gain_correction); */

  gain_index = gain + ((int_4)(gain_correction + 0.5));
  gain_index = (gain_index < 0) ? (0) : gain_index;

  if (gain_index > (GAINTBLSIZE-1) && key==KEY_RFGAIN) {
    printf("WARNING: Maximum Normalized Gain Reached\n");
    gain_index = GAINTBLSIZE-1;
  }

  /* Set attenuator register value */
  rgval = (d2awgm3_gaintbl[gain_index][0])<<2;
  rgval = findintflagdef("M3ATTN",p->config,rgval);
  iom_prt_cmd(p,mport,D2AWGM3_PE43704,PFUNC_RWR,0x00,rgval,0xFFFF);
  SS->rg_attn = rgval;

  /* Set amplifier register value */
  rgval =  d2awgm3_gaintbl[gain_index][1];
  rgval = findintflagdef("M3VGA",p->config,rgval);
  iom_prt_cmd(p,mport,D2AWGM3_LMH6401,PFUNC_RWR,0x02,rgval,0xFFFF);
  SS->rg_vga = rgval;

  SS->rfgain = gain;

  return(0);
}

int_4 d2awgm3_setmtoff (PICSTRUCT *p, int_4 mport, real_8 mtgooffns, D2AWGM3_State* SS)
{
  int_4 mtgo_fir_ind,mtgooff,mtgooff_;
  int_8 fpga_srate = SS->samprate*1e6;
  int_4 polyfir = SS->hsmode? 0:1;
  int_4 awgoutdel = findintflagdef("AWGOUTDEL",p->config,-1);
  int_4 fpga_interp = SS->interp;
  int_4 mtgo = findintflag("MTGO",p->config);
  real_8 mtgooffns_act;

  if (polyfir) {
         if(fpga_interp == 1) mtgo_fir_ind = 0;
    else if(fpga_interp == 2) mtgo_fir_ind = 1;
    else if(fpga_interp == 3) mtgo_fir_ind = 2;
    else if(fpga_interp == 4) mtgo_fir_ind = 3;
    else if(fpga_interp == 8) mtgo_fir_ind = 4;
    else if(fpga_interp == 10) mtgo_fir_ind = 5;
    else if(fpga_interp == 12) mtgo_fir_ind = 6;
    else if(fpga_interp == 16) mtgo_fir_ind = 7;
    else if(fpga_interp == 20) mtgo_fir_ind = 8;
    else if(fpga_interp == 32) mtgo_fir_ind = 9;
    else if(fpga_interp == 48) mtgo_fir_ind = 10;
    else if(fpga_interp == 50) mtgo_fir_ind = 11;
    else if(fpga_interp == 64) mtgo_fir_ind = 12;
    else if(fpga_interp == 72) mtgo_fir_ind = 13;
    else if(fpga_interp == 84) mtgo_fir_ind = 14;
    else if(fpga_interp == 96) mtgo_fir_ind = 15;
    else if(fpga_interp == 128) mtgo_fir_ind = 16;
  } else {
         if(fpga_interp == 1) mtgo_fir_ind = 0;
    else if(fpga_interp == 2) mtgo_fir_ind = 1;
    else if(fpga_interp == 4) mtgo_fir_ind = 2;
    else if(fpga_interp == 8) mtgo_fir_ind = 3;
    else if(fpga_interp == 16) mtgo_fir_ind = 4;
    else if(fpga_interp == 32) mtgo_fir_ind = 5;
    else if(fpga_interp == 64) mtgo_fir_ind = 6;
    else if(fpga_interp == 128) mtgo_fir_ind = 7;
  }

  if (awgoutdel >= 0) {
    mtgooff = mtgooff_ = fpga_interp*awgoutdel;
    mtgooffns_act = (real_8)(1e9/fpga_srate)*mtgooff;
  } else if (mtgooffns == -1000000000.0) {					/* not present */
    mtgooff_ = mtgooffns = 0;
  } else {
    mtgooff = (int_4)(mtgooffns/1e9*fpga_srate+((mtgooffns<0)?-0.5:0.5));	/* ns to samples */
    mtgooffns_act = (real_8)(1e9/fpga_srate)*mtgooff;
    if (polyfir) mtgooff_ = -129 - d2awgm3_mtgo_tbl_pp[mtgo_fir_ind] + mtgooff;
    else         mtgooff_ = -82 + mtgooff;
    if (SS->dacint==12) mtgooff_ -= 12;			/* include adjustment for diff upsamp filter on chip */
    if (SS->dacint<=10) mtgooff_ -= 16;			/* include adjustment for diff upsamp filter on chip */
    if (mtgo<2) mtgooff_ += fpga_srate;			/* now include samples to next PPS */
  }
  if (mtgooff_!=0) vprint("Requested MTGO OFFSET=%f ns, Actual MTGO OFFSET=%f ns,  Dly=%d samp\n",mtgooffns,mtgooffns_act,mtgooff_);

  iom_prt_cmd(p,mport,D2AWGM3_MTGO,PFUNC_RWR,0,mtgooff_,0xFFFFFFFF);

  return 0;
}

int_4 d2awgm3_setfreq (PICSTRUCT *p, int_4 mport, real_8 rffreq, D2AWGM3_State* SS)
{
  int_8 dac_rate = SS->dacrate*1e6;
  real_8 datfreq = rffreq*1e6;
  real_8 act_freq;
  int_4 mixpd;
  int_4 dblen;
  int_4 specinv;

  if (p->xcvr) datfreq = 1e6 * doXcvrFunc(p,"FREQ",rffreq);	/* pass to Rx and return IF */

  if (datfreq > 3*(dac_rate/2)) {
    mixpd = 0;
    dblen = 1;
    specinv = (datfreq > 2*dac_rate)? 0 : 1;
    datfreq = fabs(2*(real_8)dac_rate - datfreq);
  } else if (datfreq > (dac_rate/2)) {
    mixpd = 0;
    dblen = 0;
    specinv = (datfreq > dac_rate)? 0 : 1;
    datfreq = fabs((real_8)dac_rate - datfreq);
  } else {
    mixpd = 1;
    dblen = 0;
    specinv = 0;
  }
  if (p->xena>0 && p->xinv>0) specinv = (1-specinv);		/* Apply AT18 inverts ? */
  specinv = findintflagdef("XCFINV",p->config,specinv);		/* Override auto */

  if (mixpd) {
    iom_prt_cmd(p,mport,D2AWGM3_PE42553_0,PFUNC_PWR,0x00,0x00,0xFFFF); /* First Switch */
    iom_prt_cmd(p,mport,D2AWGM3_PE42553_1,PFUNC_PWR,0x00,0x01,0xFFFF); /* Second Switch */
    iom_prt_cmd(p,mport,D2AWGM3_LTC5549,  PFUNC_PWR,0x00,0x00,0xFFFF); /* Mixer */
  } else {
    iom_prt_cmd(p,mport,D2AWGM3_PE42553_0,PFUNC_PWR,0x00,0x01,0xFFFF); /* First Switch */
    iom_prt_cmd(p,mport,D2AWGM3_PE42553_1,PFUNC_PWR,0x00,0x00,0xFFFF); /* Second Switch */
    iom_prt_cmd(p,mport,D2AWGM3_LTC5549,  PFUNC_PWR,0x00,(dblen<<1)|0x01,0xFFFF); /* Mixer */
  }

  d2awgm3_lmx2582_mix(p,mport,mixpd);

  act_freq = d2awgm3_max5868_tune(p,mport, dac_rate, datfreq, specinv);

  SS->rffreq = act_freq*1e-6;
  SS->mixpd = mixpd;

  d2awgm3_setgain(p,mport,SS->rfgain,KEY_RFGAIN,SS);

  return(0);
}

int_4 d2awgm3_findrate (int_4 rate, int_8 *dac_rate, int_8 *actrate, int_8 *fpga_srate, int_4 *dac_interp, int_4 *fpga_interp, int_4 hsmode, int_4 polyfir)
{
  int_8 max_fpgarate;
  int_4 i,j,tblszd,tblszf;
  int_4 *dac_interp_;
  int_4 *fpga_interp_;
  int_8 interp_rate;
  int_8 cur_interp_rate;
  int_8 fpga_rate_max;
  int_8 inrate,cur_rate;
  int_8 error,cur_error;
  int_8 fpga_rate;
  int_4 fixdac = *dac_interp;

  /* max_fpgarate = (hsmode)? 500000000 : 315000000; */
  max_fpgarate = (hsmode)? MAXFPGARATEHS : MAXFPGARATELS;
  dac_interp_ = d2awgm3_dac_interp;
  fpga_interp_ = (polyfir)? d2awgm3_fpga_interp_pp : d2awgm3_fpga_interp;

  tblszd = sizeof(d2awgm3_dac_interp)/sizeof(d2awgm3_dac_interp[0]);
  tblszf = (polyfir)? sizeof(d2awgm3_fpga_interp_pp)/sizeof(d2awgm3_fpga_interp_pp[0]) :
                      sizeof(d2awgm3_fpga_interp)/sizeof(d2awgm3_fpga_interp[0]);

  if (hsmode) { /* last entry is interp=1 */
    fpga_interp_ = (polyfir)? &d2awgm3_fpga_interp_pp[tblszf-1] :
                              &d2awgm3_fpga_interp[tblszf-1];
    tblszf = 1;
  }

  inrate = rate;
  interp_rate = 0;
  fpga_rate_max = 0;
  error = -1;
  for (i=0; i<tblszd; i++) {
    if (fixdac>0 && dac_interp_[i]!=fixdac) continue;
    for (j=0; j<tblszf; j++) {
      cur_rate = inrate;
      cur_error = 0;

      fpga_rate = inrate * fpga_interp_[j];
      if (fpga_rate > max_fpgarate) {
         if (error == 0) continue;
         fpga_rate = max_fpgarate;
         cur_rate = fpga_rate / fpga_interp_[j];
         cur_error = inrate-cur_rate;
      }

      cur_interp_rate = fpga_rate * dac_interp_[i];
      if (cur_interp_rate > MAXDACRATE) {
         if (error == 0) continue;
         cur_interp_rate = MAXDACRATE;
         cur_rate = cur_interp_rate / dac_interp_[i];
         cur_rate /= fpga_interp_[j];
         cur_error = inrate-cur_rate;
      } else if (cur_interp_rate < MINDACRATE) {
         if (error == 0) continue;
         cur_interp_rate = MINDACRATE;
         cur_rate = cur_interp_rate / dac_interp_[i];
         if (cur_rate > max_fpgarate) continue;
         cur_rate /= fpga_interp_[j];
         cur_error = cur_rate-inrate;
      }

      if (cur_error != 0) {
         if (error > 0 && cur_error>=error) continue;
      } else if (error <= 0) {
         if (fpga_rate <= fpga_rate_max) continue;
      }

      error = cur_error;
      interp_rate = cur_interp_rate;
      if (error == 0) fpga_rate_max = fpga_rate;
      *dac_interp = dac_interp_[i];
      *fpga_interp = fpga_interp_[j];
    }
  }
  *dac_rate = interp_rate;
  *actrate = interp_rate/(*dac_interp)/(*fpga_interp);
  *fpga_srate = *dac_rate/(*dac_interp);
  if (*actrate != rate) {
    printf("WARNING: Unable to meet rate requirement of %f Msps, setting rate to %f Msps\n", ((double)rate)/1e6, ((double)(*actrate))/1e6);
  }

  return(0);
}

int_4 d2awgm3_findrate_fixdac (int_8 dac_rate, int_4 rate, int_8 *actrate, int_8 *fpga_srate, int_4 *dac_interp, int_4 *fpga_interp, int_4 hsmode, int_4 polyfir)
{
  int_4 (*d2awgm3_interp)[2];
  int_4 interp;
  int_8 max_fpgarate;
  int_4 tblszf,tblszd;
  int_4 *dac_interp_;
  int_4 *fpga_interp_;
  int_8 fpga_rate;
  int_4 i,j;
  int_4 rdiff,rdiff_,ind;
  int_4 tmin1,tmin2;
  int_4 dac_int,fpga_int,arate;


  max_fpgarate = (hsmode)? MAXFPGARATEHS : MAXFPGARATELS;
  dac_interp_ = d2awgm3_dac_interp;
  fpga_interp_ = (polyfir)? d2awgm3_fpga_interp_pp : d2awgm3_fpga_interp;
  rdiff = 500000000;

  tblszd = sizeof(d2awgm3_dac_interp)/sizeof(d2awgm3_dac_interp[0]);
  tblszf = (polyfir)? sizeof(d2awgm3_fpga_interp_pp)/sizeof(d2awgm3_fpga_interp_pp[0]) :
                      sizeof(d2awgm3_fpga_interp)/sizeof(d2awgm3_fpga_interp[0]);

  if (hsmode) {
    fpga_interp_ = (polyfir)? &d2awgm3_fpga_interp_pp[tblszf-1] :
                              &d2awgm3_fpga_interp[tblszf-1];
    tblszf = 1;
  }

  for (i=0;i<tblszd;i++) {
    if ((dac_rate/dac_interp_[i]) > max_fpgarate) continue;
    for (j=0;j<tblszf;j++) {

      interp = dac_interp_[i] * fpga_interp_[j];
      rdiff_ = abs(rate - dac_rate/interp);
      /* printf("JJB::: %d : desired = %d, calculated = %d, interp = %d, mindiff = %d, diff = %d\n",i,rate,dac_rate/interp,interp,rdiff,rdiff_); */
      if (rdiff_ < rdiff) {
        rdiff = rdiff_;
        *dac_interp = dac_interp_[i];
        *fpga_interp = fpga_interp_[j];
      }

    }
  }

  arate = dac_rate/((*dac_interp) * (*fpga_interp));

  if (arate != rate) printf("WARNING: Desired Rate of %f Msps cannot be met, using %f Msps\n\n",rate/1e6,arate/1e6);

  /* printf("JJB::: %d: dac_interp = %d, fpga_interp = %d, desired = %d, actual = %d\n",ind,d2awgm3_interp[ind][1],d2awgm3_interp[ind][0],rate,arate);  */
  /* printf("JJB::: D2AWGM3, DAC Rate = %lu, DAC Interp = %d, FPGA Interp = %d, FPGA Rate = %lu\n", dac_rate, *dac_interp, *fpga_interp, dac_rate/(*dac_interp)); */

  *actrate = arate;
  *fpga_srate = dac_rate/(*dac_interp);

  return(0);
}

int_4 d2awgm3_coewr (PICSTRUCT *p, int_4 mport, int_4 part_addr, int_4 *coe, int_4 taps, int_4 interp)
{

  int_4 part_addr0 = 0;
  int_4 part_addr1 = 0;
  int_4 part_reg0;
  int_4 part_reg1;
  int_4 *coeptr;
  int_4 i,j;
  real_8 time;

  part_addr0 = (part_addr<<8);
  part_addr1 = (part_addr<<8)+4;
  part_reg1 = (PFUNC_RWR<<12)|14;

  iom_exec(p,mport);
  time = gettime();
  coeptr = coe;
  for (i=0;i<taps;i++){
    for (j=0;j<interp;j++){
      if (p->type==ICEPIC7 || p->type==ICEPIC8) {
        iom_prt_wr(p,mport, part_addr, part_reg1, (*(coeptr++) & 0x3FFFF)<<8|(i<<26)|j );
      } else {
        pic_jpmemwr(p,mport,part_addr0,(*(coeptr++) & 0x3FFFF)<<8|(i<<26)|j,4);
        pic_jpmemwr(p,mport,part_addr1,part_reg1,4);
      }
    }
  }
  iom_prt_flush(p,mport,part_addr);
  time = gettime()-time;
  v3print("PolyFilt load ntap=%d interp=%d took=%f\n",taps,interp,time);
  return(0);
}

int_4 d2awgm3_polyfilt (PICSTRUCT *p, int_4 mport, int_4 interp)
{
  int_4 i,j;
  int_4 coe_wradr,coe_wrdat,coe_wr;
  int_4 *coefs;
  int_4 taps = 22;
  int_4 stat_func, stat_pin, stat_addr, stat_data;

       if (interp==1) coefs = d2awgm3_polyfir_1;
  else if (interp==2) coefs = d2awgm3_polyfir_2;
  else if (interp==3) coefs = d2awgm3_polyfir_3;
  else if (interp==4) coefs = d2awgm3_polyfir_4;
  else if (interp==8) coefs = d2awgm3_polyfir_8;
  else if (interp==10) coefs = d2awgm3_polyfir_10;
  else if (interp==12) coefs = d2awgm3_polyfir_12;
  else if (interp==16) coefs = d2awgm3_polyfir_16;
  else if (interp==20) coefs = d2awgm3_polyfir_20;
  else if (interp==32) coefs = d2awgm3_polyfir_32;
  else if (interp==48) coefs = d2awgm3_polyfir_48;
  else if (interp==50) coefs = d2awgm3_polyfir_50;
  else if (interp==64) coefs = d2awgm3_polyfir_64;
  else if (interp==72) coefs = d2awgm3_polyfir_72;
  else if (interp==84) coefs = d2awgm3_polyfir_84;
  else if (interp==96) coefs = d2awgm3_polyfir_96;
  else if (interp==128) coefs = d2awgm3_polyfir_128;
  else                  coefs = d2awgm3_polyfir_2;

  iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_RWR,10,16,0xFFFF);
  iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_PWR,0,0,0xFFFF);
  iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_RWR,0,interp,0xFFFF);

  d2awgm3_coewr(p,mport, D2AWGM3_HBINTFIR, coefs, taps,interp);

  return(0);
}


int_4 d2awgm3_lmx2582 (PICSTRUCT *p, int_4 mport, int_8 rate, int_4 inter)
{
  int_4 i;
  int_4 ratediv;
  /* int_4 (*lmx2582_regs)[2]; */
  int_4 lmx2582_regs[LMX2582_CONFIG_REGS][2];
  int_4 Fpd = 160000000;
  int_4 PLL_N_PRE = 2;
  int_4 PLL_DEN = 1000000;
  real_8 Fvco_, Fvco_rem, rate_act;
  real_8 rate_des = rate;
  int_4 PLL_N,PLL_NUM;

  /* lmx2582_regs = &lmx2582_base_table[0]; */
  for (i=0;i<LMX2582_CONFIG_REGS;++i) {
    lmx2582_regs[i][0] = lmx2582_base_table[i][0];
    lmx2582_regs[i][1] = lmx2582_base_table[i][1];
  }

  iom_prt_cmd(p,mport,D2AWGM3_LMX2582,PFUNC_PWR,0,0x01,0xFFFF);

  if (rate > MAXDACRATE){ /*4960000000){ */
    printf("WARNING: %d Msps is an Invalid DAC Rate, Maximum DAC Rate is %d Msps\n",(int_4)(rate/1e6),(int_4)(MAXDACRATE/1e6));
    rate = MAXDACRATE; /*4960000000;*/
  } else if (rate < MINDACRATE){ /*3550000000){*/
    printf("WARNING: %d Msps is an Invalid DAC Rate, Minimum DAC Rate is %d Msps\n",(int_4)(rate/1e6),(int_4)(MINDACRATE/1e6));
    rate = MINDACRATE; /* 3550000000;*/
  }
  ratediv=1;
  if (rate < MINVCORATE) {
    rate = 2*rate;
    ratediv=2;
    /* Power Down VCO to Output Buffer : Reg 31, bit 9=1 A, bit 10=1 B */
    /* Power Up VCO to Channel Divider : Reg 31, bit 7=0 */
    lmx2582_regs[21][1] = (lmx2582_regs[21][1] & 0xF97F) | (3<<9);

    /* Enable Channel Divider : Reg 34, bit 5=1 */
    lmx2582_regs[18][1] = (lmx2582_regs[18][1] & 0xFFDF) | (1<<5);

    /* Enable SEG1 Channel Divider : Reg 35, bit 1=1 */
    /* Disable SEG2 Channel Divider : Reg 35, bit 7=0 */
    /* Disable SEG3 Channel Divider : Reg 35, bit 8=0 */
    /* Set Channel Divider (SEG1) to Divide by 2 : Reg 35, bit 2=0 */
    lmx2582_regs[17][1] = (lmx2582_regs[17][1] & 0xFE79) | (1<<1);

    /* Enable Buffer from Channel Divider to Selection Mux : Reg 36, bit 10=1 A, bit 11=1 B*/
    /* Configure Channel Divider Mux for SEG1 : Reg 36, bit 4:6=1 */
    lmx2582_regs[16][1] = (lmx2582_regs[16][1] & 0xFF8F) | (1<<4);

    /* Select Ouput Mux A : Reg 47, bit 11:12=0 */
    lmx2582_regs[5][1] = (lmx2582_regs[5][1] & 0xE7FF);

    /* Select Ouput Mux B : Reg 48, bit 0:1=0 */
    lmx2582_regs[4][1] = (lmx2582_regs[4][1] & 0xFFFC);
  }

  Fvco_ = ((real_8)rate) / ((real_8)(Fpd*PLL_N_PRE));
  Fvco_rem = Fvco_ - floor(Fvco_);

  PLL_N = (int_4)Fvco_;
  PLL_NUM = round(Fvco_rem * PLL_DEN);
  rate_act = (((real_8)PLL_N + ((real_8)PLL_NUM)/PLL_DEN)*Fpd*PLL_N_PRE)/ratediv;
  if(rate_act != rate_des) printf("WARNING: RATE Mismatch, Desired = %f Hz, Actual = %f Hz\n", rate_des/inter,rate_act/inter);

  lmx2582_regs[14][1] = (PLL_N & 0xFFF)<<1;
  lmx2582_regs[7][1]  = (PLL_NUM & 0x0000FFFF);
  lmx2582_regs[8][1]  = (PLL_NUM & 0xFFFF0000)>>16;
  /* Handle the Mix Path Powerdown Case */
/*
  lmx2582_regs[6][1]  = (lmx2582_regs[6][1] & 0xFF7F) | (mixpd<<7);
  lmx2582_regs[21][1] = (lmx2582_regs[21][1] & 0xFBFF) | (mixpd<<10);
*/

  for (i=0; i<LMX2582_CONFIG_REGS; i++) {
    /* printf("JJB::: LMX2582 : ADDR=0x%x, Data=0x%x\n",lmx2582_regs[i][0],lmx2582_regs[i][1]); */
    iom_prt_cmd(p,mport,D2AWGM3_LMX2582,PFUNC_RWR,lmx2582_regs[i][0],lmx2582_regs[i][1],0xFFFF);
  }

  return(0);
}

#ifdef OLD_D2AWG_PART
int_4 d2awgm3_si5344 (PICSTRUCT *p, int_4 mport, real_8 rate, int_4 extref)
{
  PT_SI5344_REG SIR[SI5344_MAX_CFG_RG];
  int_4 i,rtn,last_page;
  int_4 stat_func, stat_pin, stat_addr, stat_data;

  /* Initialize the Si5344 configuration */
  pt_si5344_rgc_default(&SIR[0], FPGA_MODID_D2AWGM3);

  /* Configure all the registers on the part */
  last_page = 255;
  for (i=0; i<SI5344_MAX_CFG_RG; i++){
    if (SIR[i].rp != last_page) {
      /* Need to update page register */
      /* printf("UPDATE PAGE to 0x%x\n",SIR[i].rp); */
      iom_prt_cmd(p,mport,D2AWGM3_SI5344,PFUNC_RWR,SI5344_RG_PAGE,SIR[i].rp,0xFF);
      last_page = SIR[i].rp;
    }
    /* printf("SI5344: 0x%0.2x:0x%0.2x\n",SIR[i].ra,SIR[i].rv); */
    iom_prt_cmd(p,mport,D2AWGM3_SI5344,PFUNC_RWR,SIR[i].ra,SIR[i].rv,0xFF);
  }

  if (extref) {
    /* printf("Enable External Reference...\n"); */
    iom_prt_cmd(p,mport,D2AWGM3_SI5344,PFUNC_RWR,SI5344_RG_PAGE,0x05,0xFF);
    iom_prt_cmd(p,mport,D2AWGM3_SI5344,PFUNC_RWR,0x2A,0x05,0xFF);
  }

  return (0);
}
#endif

int_4 d2awgm3_loadcoefs (PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 sub_ID, int_4 coeset)
{
  int_4 i;
  int_4 size;
  real_8 coe_tmp;

  iom_prt_cmd(p,mport,part_ID,PFUNC_RWR,256,0,0xFFFF);

  size = sizeof(d2awgm3_hbfir0)/sizeof(d2awgm3_hbfir0[0]);

  for (i=0;i<size;++i) {
    coe_tmp = (real_8)d2awgm3_hbfir0[i];
    iom_prt_cmd(p,mport,part_ID,PFUNC_RWR,i,(int_4)coe_tmp,0xFFFF);
  }

  iom_prt_cmd(p,mport,part_ID,PFUNC_RWR,256,1,0xFFFF);

  size = sizeof(d2awgm3_hbfir1)/sizeof(d2awgm3_hbfir1[0]);
  for (i=0;i<size;++i) {
    iom_prt_cmd(p,mport,part_ID,PFUNC_RWR,i,d2awgm3_hbfir1[i],0xFFFF);
  }

  return(0);
}

int_4 d2awgm3_set_rfopts (PICSTRUCT *p, int_4 mport, int_4 rfopts, D2AWGM3_State* SS)
{
  int_4 rtn = 0;
  int_4 rfold,rfdif;

  if ((rfopts & RFOPT_ENABLE) == 0) return (rtn);	/* no options enabled */

  rfold = SS->rfopts;		/* currently configured options */
  SS->rfopts = rfopts;
  rfdif = rfopts ^ rfold;	/* any differences ? */

  if (rfdif & RFOPT_MUTE) d2awgm3_max5868_pin(p,mport,MAX5868_MUTE,(rfopts&RFOPT_MUTE)?1:0);

  return (rtn);
}

int_4 d2awgm3_rdtemp (PICSTRUCT *p, int_4 mport)
{
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  iom_prt_cmd (p,mport,D2AWGM3_MIC280,PFUNC_RRD,1,0,0xFFFFFFFF);
  iom_prt_sts (p,mport,D2AWGM3_MIC280,&stat_func,&stat_pin,&stat_addr,&stat_data);
  return stat_data;
}

int_4 d2awgm3_clkstat (PICSTRUCT *p, int_4 mport)
{
  int_4 status=0, extref;
  int_4 stat_func,stat_pin,stat_addr,stat_data;

  iom_prt_cmd (p,mport,D2AWGM3_MODULE,PFUNC_PRD,1,0,0xFF);
  iom_prt_sts (p,mport,D2AWGM3_MODULE,&stat_func,&stat_pin,&stat_addr,&stat_data);
  iom_prt_cmd (p,mport,D2AWGM3_LTC6957,PFUNC_PRD,0,0,0xFF);
  iom_prt_sts (p,mport,D2AWGM3_LTC6957,&stat_func,&stat_pin,&stat_addr,&extref);

  status = 0;
  if ((extref==0) && (stat_data&0x1)==0) status |= 0x1;
  if ((stat_data&0x2)==0) status |= 0x2;

  return (status);
}

int_4 d2awgm3_reset (PICSTRUCT *p, int_4 mport)
{
  int_4 jvm=iom_set_prt_jvm(p,mport,0);
  iom_prt_cmd(p,mport,D2AWGM3_ROOT,PFUNC_PWR,0,0x05,0xFF);	
  iom_prt_dly(p,mport,500);
  iom_prt_cmd(p,mport,D2AWGM3_ROOT,PFUNC_PWR,0,0x00,0xFF);
  iom_prt_dly(p,mport,500);
  iom_set_prt_jvm(p,mport,jvm);
  return (0);
}

int_4 d2awgm3_enable (PICSTRUCT *p, int_4 mport, int_4 ena)
{
  D2AWGM3_State* SS = (D2AWGM3_State*) iom_get_state (p,mport);                         
  int_4 poly,jvm;
  if (SS==NULL) return -1;
  poly = (SS->hsmode==0)? 1 : 0;
  jvm=iom_set_prt_jvm(p,mport,0);
  if (ena==0) {
    d2awgm3_max5868_pin(p,mport,MAX5868_MUTE,1);
    if (poly) iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_PWR,0x00,0x00,0xFFFF);
    iom_prt_cmd(p,mport,D2AWGM3_ROOT,PFUNC_PWR,0x00,0x00,0xFF);
    iom_prt_cmd(p,mport,D2AWGM3_ROOT,PFUNC_PWR,0x00,0x05,0xFF);
    iom_prt_cmd(p,mport,D2AWGM3_ROOT,PFUNC_PWR,0x00,0x00,0xFF);
  } else {
    iom_prt_cmd(p,mport,D2AWGM3_MODULE,PFUNC_PWR,0x00,SS->sysreg,0xFF);
    iom_prt_cmd(p,mport,D2AWGM3_ROOT,PFUNC_PWR,0x00,0x02,0xFF);
    if (poly) iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_PWR,0x00,0x01,0xFFFF);
    d2awgm3_max5868_pin(p,mport,MAX5868_MUTE,(SS->rfopts&RFOPT_MUTE)?1:0);
  }
  iom_set_prt_jvm(p,mport,jvm);
  return 0;
}

real_4 d2awgm3_ntpoff (PICSTRUCT *p, int_4 mport)
{
  double time,frac=-1; 
  int ppstat,cpps,ccur;
  ppstat = iom_rd(p,mport,0x5F00,4);
  time = pic_time();
  cpps = (ppstat>>16)&0xFFFF;
  ccur = (ppstat>> 0)&0xFFFF;
  if (cpps>10000) {
    frac = (double)ccur/cpps;
    frac = fmod(time-frac,1.0);
    if (frac>0.5) frac-=1.0;
  }
  return frac;
}

int_4 d2awgm3_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags) 
{
  D2AWGM3_State *SS;
  int_4 i,sysreg,tone,fpga_interp,dac_interp,cfgdsp;
  int_4 extref,mute_offset,specinv,max_dlatency,polyfir;
  int_4 hbidbg,hbinter,hbistage,hb_prescale,hb_maxclip,hb_minclip;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_8 dac_rate,actrate,fpga_srate;
  real_8 ratef,mtgooffns;

  if (mport==3) return d2awgm3_setup(p,1,dir,bits,rate,gain,flags)+
                       d2awgm3_setup(p,2,dir,bits,rate,gain,flags);

  SS = (D2AWGM3_State*)iom_alloc_state(p,mport,D2AWGM3_SIZE);

  SS->hsmode    = p->qdrx & mport;
  polyfir	= (SS->hsmode)? 0 : 1;
  if ((SS->hsmode==0) && (rate > MAXFPGARATELS)) {
    rate = 240e6;
    printf("WARNING: For Rates > %f MSPS, Enable Flag QDRX.  Setting Rate to %f MSPS\n",1.0e-6*MAXFPGARATELS,1.0e-6*rate);
  }

  /* populate State Structure with startup values - alloc_state zero fills */
  SS->rffreq	= iom_getcfgd (p,mport,"RFFREQ",1000.0); 
  SS->dacrate	= iom_getcfgd (p,mport,"DACRATE",0.0);
  SS->rfgain	= iom_getcfgl (p,mport,"RFGAIN",0);
  SS->rfattn	= iom_getcfgl (p,mport,"RFATTN",0);
  SS->rfopts    = iom_getcfgm (p,mport,"RFOPTS",RFOPT_ENABLE,RFOPTLIST);
  dac_interp	= iom_getcfgl (p,mport,"DACINT",-1);

  mtgooffns     = iom_getcfgd(p,mport,"MTOFF",-1000000000.0);
  mtgooffns     = iom_getcfgd(p,mport,"MTGOOFF",mtgooffns);

  if (iom_getcfgl(p,mport,"PREFX",0)>0)      SS->rfopts |= RFOPT_XREF;
  if (iom_getcfgl(p,mport,"RFOPT_XREF",0)>0) SS->rfopts |= RFOPT_XREF;
  if (iom_getcfgl(p,mport,"RFOPT_MUTE",0)>0) SS->rfopts |= RFOPT_MUTE;

  /* find the best frequency plan that is available */
  if (SS->dacrate>0.0) {
    dac_rate = (int_8)(SS->dacrate*1e6);
    d2awgm3_findrate_fixdac(dac_rate, rate, &actrate, &fpga_srate, &dac_interp, &fpga_interp, SS->hsmode,polyfir);
  } else {
    d2awgm3_findrate(rate, &dac_rate, &actrate, &fpga_srate, &dac_interp, &fpga_interp, SS->hsmode,polyfir);
  }
  SS->interp  = fpga_interp;
  SS->dacint  = dac_interp;
  SS->dacrate = dac_rate*1e-6;
  SS->samprate= fpga_srate*1e-6;

  dac_interp  = findintflagdef("DACINT",p->config,dac_interp);
  hbinter     = findintflagdef("HBINT",p->config,fpga_interp);
  hbidbg      = findintflagdef("HBDBG",p->config,7);
  hb_prescale = findintflagdef("HBPRE",p->config,0x00FF);
  hb_maxclip  = findintflagdef("HBCMAX",p->config, 32700)<<8;
  hb_minclip  = findintflagdef("HBCMIN",p->config,-32700)<<8;
  hbistage    = (int_4)round(log2(hbinter));

  extref      = (SS->rfopts&RFOPT_XREF)?1:0;
  ratef       = finddblflagdef("CRATE",p->config,320.0);

  sysreg = 0;
  if (findintflag("RSEL",p->config) > 0) sysreg |= 1<<0;
  if (findintflag("TSEL",p->config) > 0) sysreg |= 1<<1;
  if (findintflag("MTGO",p->config) > 0) sysreg |= 1<<3;
  i = findintflagdef("TSS",p->config,0);  sysreg |= (i&0x0F)<<4;
  if (findintflag("CALEN",p->config) > 0) sysreg |= 1<<8;
  if (findintflag("TLRS",p->config) > 0)  sysreg |= 1<<9;
  if (findintflag("AWGTONE",p->config) > 0) sysreg |= 1<<13;
  if (SS->hsmode)                           sysreg |= 1<<14;
  if (findintflag("MTGO",p->config) > 1)    sysreg |= 1<<15;
  SS->sysreg = sysreg;

  iom_init (p,mport);				/* initialize IOM transport */
#if NEWSTUFF == 2
  iom_set_prt_jvm(p,mport,1);			/* use JVM as commander ? */
#else
  iom_set_prt_jvm(p,mport,SS->hsmode?0:1);	/* use JVM as commander ? */
#endif

  iom_prt_cmd(p,mport,D2AWGM3_MODULE,PFUNC_PWR,0x00,SS->sysreg,0xFF);	/* System setup register */
  d2awgm3_reset(p,mport);			/* Reset the FPGA */

  vprint("D2AWGM3 SRATE=%llu, DACRATE=%llu, FPGARATE=%llu, FPGA_INTERP=%d, DAC_INTERP=%d\n",actrate,dac_rate,fpga_srate,hbinter,dac_interp);

  d2awgm3_setmtoff (p,mport,mtgooffns,SS);

  if (polyfir) {
    d2awgm3_polyfilt(p,mport, hbinter);
  } 
  else if (SS->hsmode==0) {
    /* Set Prescalar Value */
    iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_PWR,1,hb_prescale,0xFFFF);
    /* Set Clip Value */
    iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_PWR,2,hb_maxclip,0xFFFFFFFF);
    iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_PWR,3,hb_minclip,0xFFFFFFFF);
    /* Set Oversample Factor to 4 */
    iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_RWR,0,hbinter,0xFF);
    /* Set Output Stage Select to 2 - always log2(OversampFactor) */
    iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_RWR,1,hbistage,0xFF);
    iom_prt_cmd(p,mport,D2AWGM3_HBINTFIR,PFUNC_RWR,2,hbidbg,0xFF);
  }

  if (extref) {
    iom_prt_cmd(p,mport,D2AWGM3_LTC6957,PFUNC_PWR,0,0x00,0xFFFF);
    iom_prt_cmd(p,mport,D2AWGM3_OSC,PFUNC_PWR,0,0x03,0xFFFF);
  } else {
    iom_prt_cmd(p,mport,D2AWGM3_LTC6957,PFUNC_PWR,0,0x04,0xFFFF);
    iom_prt_cmd(p,mport,D2AWGM3_OSC,PFUNC_PWR,0,0x00,0xFFFF);
  }
  iom_prt_cmd(p,mport,D2AWGM3_SI5335,PFUNC_PWR,0,0x05,0xFFFF);
  iom_prt_dly(p,mport,1000);
  iom_prt_cmd(p,mport,D2AWGM3_SI5335,PFUNC_PWR,0,0x01,0xFFFF);
  iom_prt_dly(p,mport,1000);

  d2awgm3_lmx2582 (p,mport,dac_rate,hbinter*dac_interp);

  /* setup the DAC part */
       if (dac_interp==24) { max_dlatency = 1317; cfgdsp = 0xF0; }
  else if (dac_interp==20) { max_dlatency = 1137; cfgdsp = 0xE0; }
  else if (dac_interp==16) { max_dlatency = 864;  cfgdsp = 0xD0; }
  else if (dac_interp==12) { max_dlatency = 764;  cfgdsp = 0xC0; }
  else if (dac_interp==10) { max_dlatency = 673;  cfgdsp = 0x00; }
  else if (dac_interp==8)  { max_dlatency = 496;  cfgdsp = 0xB0; }
  else if (dac_interp==6)  { max_dlatency = 420;  cfgdsp = 0xA0; }
  else if (dac_interp==5)  { max_dlatency = 382;  cfgdsp = 0x90; }
  else                     { max_dlatency = 288;  cfgdsp = 0x80; }

  mute_offset = (max_dlatency * fpga_srate)/dac_rate;

  d2awgm3_max5868_pin(p,mport,MAX5868_ALL,0);
  d2awgm3_max5868_pin(p,mport,MAX5868_MUTE,1);
  d2awgm3_max5868_pin(p,mport,MAX5868_RESET,1);
  d2awgm3_max5868_pin(p,mport,MAX5868_CLKDIS,1);
  d2awgm3_max5868_pin(p,mport,MAX5868_DATDIS,1);
  d2awgm3_max5868_pin(p,mport,MAX5868_ALL,0);
  iom_prt_dly(p,mport,10000);

  d2awgm3_max5868_pin(p,mport,MAX5868_CLKINV,findintflagdef("DACCLKINV",p->config,0));
  d2awgm3_max5868_pin(p,mport,MAX5868_SYNCTY,findintflagdef("DACSYNCT",p->config,1));

  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x100,mute_offset,0xFFFF);
  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x01,(cfgdsp|0x0C),0xFFFF);
  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x01,(cfgdsp|0x04),0xFFFF);
  d2awgm3_max5868_pin(p,mport,MAX5868_CLKDIS,0);

  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x01,(cfgdsp|0x00),0xFFFF);
  d2awgm3_setfreq (p,mport,SS->rffreq,SS);
  iom_prt_cmd(p,mport,D2AWGM3_MAX5868,PFUNC_RWR,0x01,(cfgdsp|0x01),0xFFFF);
  d2awgm3_max5868_pin(p,mport,MAX5868_DATDIS,0);

  iom_prt_cmd(p,mport,D2AWGM3_LMH6401,PFUNC_RWR,0x02,0x20,0xFFFF);
  iom_prt_cmd(p,mport,D2AWGM3_PE43704,PFUNC_RWR,0x00,0x30,0xFFFF);

  d2awgm3_setgain(p,mport,SS->rfgain,KEY_RFGAIN,SS);
  if (p->xcvr) d2awgm3_setgain(p,mport,SS->rfattn,KEY_RFATTN,SS);

  iom_prt_cmd (p,mport,D2AWGM3_MODULE,PFUNC_PRD,1,0,0xFF);
  iom_prt_sts (p,mport,D2AWGM3_MODULE,&stat_func,&stat_pin,&stat_addr,&stat_data);
  if (stat_data&1) printf("Warn: Failed lock to %sternal 10MHz reference on port=%d\n",extref?"ex":"in",mport);

  iom_set_state (p,mport);	/* also performs iom_exec */

  if (p->verbose>1) d2awgm3_dump_state(p,mport,SS);

  return (0); 
}

int_4 d2awgm3_getkeyl (PICSTRUCT *p, int_4 mport, int_4 key)
{
  int_4 rtn = -1;
  D2AWGM3_State* SS = (D2AWGM3_State*) iom_get_state (p,mport);
       if (key==KEY_TEMP )  rtn = d2awgm3_rdtemp (p,mport);
  else if (key==KEY_CLKSTAT) rtn = d2awgm3_clkstat (p,mport);
  else if (SS==NULL)        rtn = -1;	/* not available */
  else if (key==KEY_RFOPTS) rtn = SS->rfopts;
  else if (key==KEY_RFGAIN) rtn = SS->rfgain;
  else if (key==KEY_RFATTN) rtn = (p->xcvr)? SS->rfattn : SS->rg_attn;
  else if (key==KEY_MGAIN)  rtn = SS->rg_vga;
  else if (key==KEY_RFPWR)  rtn = d2awgm3_rfpwr (p,mport,SS);
  else printf("Unhandled getkeyl=%d in d2awgm3\n",key);
  return rtn;
}

real_8 d2awgm3_getkeyd (PICSTRUCT *p, int_4 mport, int_4 key)
{
  real_8 rtn = -1.0;
  D2AWGM3_State* SS = (D2AWGM3_State*) iom_get_state (p,mport);
       if (key==KEY_NTPOFF) rtn = d2awgm3_ntpoff (p,mport);
  else if (SS==NULL)        rtn = -1;
  else if (key==KEY_RFOPTS) rtn = SS->rfopts;
  else if (key==KEY_RFGAIN) rtn = SS->rfgain;
  else if (key==KEY_RFFREQ) rtn = (p->xena)? p->xfreq+SS->rffreq-p->xcf : SS->rffreq;
  else if (key==KEY_MTOFF)  rtn = SS->mtgooff;
  else if (key==KEY_RFPWR)  rtn = d2awgm3_rfpwr (p,mport,SS);
  else if (key==KEY_DRATE)  rtn = SS->samprate*1e6;
  else if (key==KEY_RFBW)   rtn = SS->dacrate;
  else printf("Unhandled getkeyd=%d in d2awgm3\n",key);
  return rtn;
}

int_4 d2awgm3_setkeyl (PICSTRUCT *p, int_4 mport, int_4 key, int_4 i4val)
{
  int_4 rtn = 0;
  D2AWGM3_State* SS = (D2AWGM3_State*) iom_get_state (p,mport);
  if (SS==NULL) return -1;
       if (key==KEY_RFOPTS) { rtn = d2awgm3_set_rfopts (p,mport,i4val,SS); }
  else if (key==KEY_RFGAIN) { rtn = d2awgm3_setgain (p,mport,i4val,key,SS); }
  else if (key==KEY_RFATTN) { rtn = d2awgm3_setgain (p,mport,i4val,key,SS); }
  else if (key==KEY_MGAIN)  { rtn = d2awgm3_setgain (p,mport,i4val,key,SS); }
  else if (key==KEY_MTOFF)  { rtn = d2awgm3_setmtoff (p,mport,(real_8)i4val,SS); }
  else printf("Unhandled setkeyl=%d in d2awgm3\n",key);
  iom_set_state (p,mport);
  return rtn;
}

int_4 d2awgm3_setkeyd (PICSTRUCT *p, int_4 mport, int_4 key, real_8 r8val)
{
  int_4 i4val = r8val;
  int_4 rtn = 0;
  D2AWGM3_State* SS = (D2AWGM3_State*) iom_get_state (p,mport);
  if (SS==NULL) return -1;
       if (key==KEY_RFGAIN) { rtn = d2awgm3_setgain (p,mport,i4val,key,SS); }
  else if (key==KEY_RFFREQ) { rtn = d2awgm3_setfreq (p,mport,r8val,SS); }
  else if (key==KEY_MTOFF)  { rtn = d2awgm3_setmtoff (p,mport,r8val,SS); }
  else printf("Unhandled setkeyd=%d in d2awgm3\n",key);
  iom_set_state (p,mport);
  return rtn;
}
