/****************************************************************************/
/*                                                                          */
/*   IO module common interface routines                                    */
/*                                                                          */
/****************************************************************************/
#define IOM_CFG_STB    0xFD000000  /* Base address for module state block */
#define IOM_CFG_CRC    0xFD000000  /* Base address for module state block crc */
#define IOM_CFG_SBS    0xFD000004  /* Base address for module state block size */
#define IOM_CFG_CMD    0xFD000400  /* Base address for command count */
#define IOM_CFG_PAR    0xFD000404  /* Base address for command parity */
#define IOM_CFG_RET    0xFD000408  /* Base address for return status */
#define IOM_CFG_BUF    0xFD000410  /* Base address for command data */
#define IOM_CFG_HBT    0xFD00FFFC  /* Base address for heartbeat */

#define IOM_RD   0x1	/* read up to 1 qword at address */
#define IOM_WR   0x2	/* write up to 1 qword at address */
#define IOM_WRB  0x3	/* write 1 byte at address */
#define IOM_WRM  0x4	/* write multiple qword to the same address */
#define IOM_DLY  0x5	/* delay N microseconds */
#define IOM_POLL 0x6	/* poll address for certain value with timeout */
#define IOM_RWM  0x7	/* write same address and read same other address multiple times */
#define IOM_CRC  0x8	/* get CRC of current state */
#define IOM_RDP  0x9	/* read part */
#define IOM_WRP  0xA	/* write part */
#define IOM_PRT_CMD 0xB /* part command */
#define IOM_PRT_STS 0xC /* part status */

#define MAXIOMBUF 250	/* max as JVM has 2K ram buffer on IO modules */

int_4 iom_jvm_rd (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 len) {
  int_4 sel,data;
  if ((addr&0xFF000000)!=IOM_CFG_STB) printf("Uh oh. Bad address=%08x for iom_jvm_rd\n",addr);
  sel = getJTAGsel(p,mport) | 0x0100; /* JVM mode */
  pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,addr,2);
  data = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,len);
  return data;
}
int_4 iom_jvm_wr (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 data, int_4 len) {
  int_4 sel;
  if ((addr&0xFF000000)!=IOM_CFG_STB) printf("Uh oh. Bad address=%08x for iom_jvm_wr\n",addr);
  sel = getJTAGsel(p,mport) | 0x0100; /* JVM mode */
  pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,addr,2);
  pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,data,len);
  return data;
}
int_4 iom_jvm_wrm (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 *data, int_4 len) {
  int_4 i,sel,ilen=len>>2;
  if ((addr&0xFF000000)!=IOM_CFG_STB) printf("Uh oh. Bad address=%08x for iom_jvm_wrm\n",addr);
  sel = getJTAGsel(p,mport) | 0x0100; /* JVM mode */
  pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,addr,2);
  if (p->type==ICEPIC8 && len>16) {
    pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,0,len);
    pic_wfifo (p, data, len, FLG_SYNC);
  }
  else for (i=0; i<ilen; i++) {
    pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,data[i],4);
  }
  return len;
}

int_4 iom_init (PICSTRUCT *p, int_4 mport) {
  p->iom_cnt=0;
  if (p->iom_buf==0) p->iom_buf = (int_4 *) malloc(MAXIOMBUF*4);
  return 0;
}

int_4 iom_exec (PICSTRUCT *p, int_4 mport) {
  int_2 s=0; int_4 i,timeout=1000,par,retry=4,hbt1,hbt2;
  if (p->iom_cnt==0) return 0;
  par = 0x1ce00000 | p->iom_cnt;
  for (i=0; i<p->iom_cnt; i++) par ^= p->iom_buf[i];
  pic_lock(p,LOCK_ALLOC);
  RETRY:
  iom_jvm_wr (p,mport,IOM_CFG_CMD,0x1ce00000,4);	/* clear command */
  iom_jvm_wrm (p,mport,IOM_CFG_BUF,p->iom_buf,p->iom_cnt*4);
  iom_jvm_wr (p,mport,IOM_CFG_PAR,par,4);
  iom_jvm_wr (p,mport,IOM_CFG_CMD,p->iom_cnt,1);
  for (i=0; timeout>0; timeout--) {			/* poll for completion */
    i = iom_jvm_rd (p,mport,IOM_CFG_CMD,4); s = i;
    if ((i>>20)!=0x1ce || s<0) {
      printf("Problem with IO port=%d configuration block exec=%d hex=%08x\n",mport,s,i);
      if (--retry>0) goto RETRY;
    }
    if (s<=0) break;
    udelay(1000);
  }
  pic_lock(p,LOCK_FREE);
  if (s>0) {
    hbt1 = iom_jvm_rd (p,mport,IOM_CFG_HBT,4);
    hbt2 = iom_jvm_rd (p,mport,IOM_CFG_HBT,4);
    printf("Err: timeout from JVM command cnt=%d i=%d hbt=%x hbt=%x\n",p->iom_cnt,i,hbt1,hbt2);
  }
  p->iom_cnt=0;
  return s;
}

int_4 iom_set_prt_jvm (PICSTRUCT *p, int_4 mport, int_4 onof) {
  int_4 rtn = p->iom_prt_jvm[mport-1];
  iom_exec(p,mport);
  p->iom_prt_jvm[mport-1]=onof;
  return rtn;
}

int_4 iom_rd (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 len) {
  p->iom_buf[p->iom_cnt++] = (len<<4) | (addr<<8) | IOM_RD; 
  iom_exec (p,mport);
  return iom_jvm_rd (p,mport,IOM_CFG_RET,len);
}

int_4 iom_rdb (PICSTRUCT *p, int_4 mport, int_4 addr) {
  return iom_rd(p,mport,addr,1);
}

int_4 iom_rwm (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 *data, int_4 len) {
  int i, wlen=len>>2;
  p->iom_buf[p->iom_cnt++] = (addr<<8) | IOM_RWM;
  p->iom_buf[p->iom_cnt++] = data[0];
  p->iom_buf[p->iom_cnt++] = data[1];
  p->iom_buf[p->iom_cnt++] = wlen;
  iom_exec (p,mport);
  for (i=0; i<wlen; i++) data[i] = iom_jvm_rd (p,mport,IOM_CFG_BUF+(i*4),4);
  return len;
}

int_4 iom_assert (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 len, int_4 expect, int_4 timeout) {
  if (p->iom_cnt>MAXIOMBUF-4) iom_exec (p,mport);
  p->iom_buf[p->iom_cnt++] = (len<<4) | (addr<<8) | IOM_RD; 
  p->iom_buf[p->iom_cnt++] = (timeout<<8) | IOM_POLL; 
  p->iom_buf[p->iom_cnt++] =  expect; 
  return iom_exec (p,mport);
}

int_4 iom_wr (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 data, int_4 len) {
  int_4 lnone=-1, *lcmd=(p->iom_cnt<2)? &lnone:p->iom_buf+(p->iom_cnt-2);
  int_4 lfunc = (*lcmd>>0)&0xF;
  int_4 lcnt  = (*lcmd>>4)&0xF;
  int_4 laddr = (*lcmd>>8);
  if (p->iom_cnt>MAXIOMBUF-4) iom_exec (p,mport);
  if (len==1 && (addr>>16)==0)					/* optimize to 8b data  16b addr */
    { p->iom_buf[p->iom_cnt++] = (data<<24) | (addr<<8) | IOM_WRB; }
  else if (len==4 && laddr==addr && lfunc==IOM_WR)		/* change WR to WRM */
    { p->iom_buf[p->iom_cnt++] = data;  (*lcmd)=(addr<<8)|0x10|IOM_WRM; }
  else if (len==4 && laddr==addr && lfunc==IOM_WRM && lcnt<15)	/* up last WRM cnt */
    { p->iom_buf[p->iom_cnt++] = data;  (*lcmd)+=0x10; }
  else  							/* normal qword write */
    { p->iom_buf[p->iom_cnt++] = (len<<4) | (addr<<8) | IOM_WR; p->iom_buf[p->iom_cnt++] = data; }
  return len;
}

int_4 iom_wrb (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 data) {
  return iom_wr (p,mport,addr,data,1);
}

int_4 iom_dly (PICSTRUCT *p, int_4 mport, int_4 usec) {
  if (p->iom_cnt>MAXIOMBUF-2) iom_exec (p,mport);
  p->iom_buf[p->iom_cnt++] = (usec<<8) | IOM_DLY; 
  return 0;
}

int_4 iom_crc (PICSTRUCT *p, int_4 mport, int_4 len) {
  p->iom_buf[p->iom_cnt++] = (len<<8) | IOM_CRC; 
  iom_exec (p,mport);
  return iom_jvm_rd (p,mport,IOM_CFG_RET,1);
}

int_4 lpaddr;
int_4 iom_prt_flush (PICSTRUCT *p, int_4 mport, int_4 prt) {
  int_4 sel,len=p->iom_cnt*4;
  sel = getJTAGsel(p,mport) | (prt<<8); 	/* page = prt */
  if (len>0) {
    pic_acmd (p,ADON_WRIOP,sel,lpaddr,len);
    pic_wfifo (p, p->iom_buf, len, FLG_SYNC);
  }
  p->iom_cnt=0;
  return len;
}

int_4 iom_prt_wait (PICSTRUCT *p, int_4 mport, int_4 part_ID)
{
  int_4 part_addr = (part_ID<<8);
  int_4 part_busy;
  int_4 timeout = 5000;
  do {
    part_busy = (pic_jpmemrd(p,mport,part_addr+4,4)>>15)&0x1;
  } while (part_busy && --timeout>0);
  if (timeout==0) printf("iom_prt_wait timeout\n"); 
  return part_busy;
}

int_4 iom_prt_cmd (PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 func, int_4 addr, int_4 data, int_u4 mask)
{
  /* Part Address Resolution */
  int_4 part_addr = (part_ID<<8);
  int_4 pin = (func==PFUNC_PWR)? data:0;

if (p->iom_prt_jvm[mport-1]) {
  if (p->iom_cnt>MAXIOMBUF-4) iom_exec (p,mport);
  /* queue part command and args to JVM */
  p->iom_buf[p->iom_cnt++] = (part_ID<<24) | (func<<20) | (addr<<8) | IOM_PRT_CMD;
  p->iom_buf[p->iom_cnt++] = mask; 
  p->iom_buf[p->iom_cnt++] = data; 
} else {
  pic_lock(p,LOCK_ALLOC);
  /* Write the Mask for Part Commands */
  pic_jpmemwr(p,mport,part_addr+0,mask,4);
  pic_jpmemwr(p,mport,part_addr+4,(PFUNC_RWR<<12)|0xFFF,4);
  iom_prt_wait(p,mport,part_ID);
  /* Perform Write|Read Command */
  pic_jpmemwr(p,mport,part_addr+0,data,4);
  pic_jpmemwr(p,mport,part_addr+4,(pin<<16)|(func<<12)|(addr&0xFFF),4);
  iom_prt_wait(p,mport,part_ID);
  pic_lock(p,LOCK_FREE);
}
  return(0);
}

int_4 iom_prt_sts (PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 *func, int_4 *pin, int_4 *addr, int_4 *data)
{
  int_4 part_addr,cmd_stat0,cmd_stat1;
  part_addr = (part_ID<<8);
  pic_lock(p,LOCK_ALLOC);
  iom_exec (p,mport);	/* exec outstanding requests */
  cmd_stat0 = pic_jpmemrd(p,mport,part_addr+0,4);
  cmd_stat1 = pic_jpmemrd(p,mport,part_addr+4,4);
  pic_lock(p,LOCK_FREE);
  *data = cmd_stat0;
  *addr = cmd_stat1&0xFFF;
  *func = (cmd_stat1>>12)&0xF;
  *pin  = (cmd_stat1>>16)&0xFFFF;
  return(0);
}

int_4 iom_prt_std (PICSTRUCT *p, int_4 mport, int_4 part_ID)
{
  int_4 part_addr,cmd_stat0;
  part_addr = (part_ID<<8);
  pic_lock(p,LOCK_ALLOC);
  iom_exec (p,mport);	/* exec outstanding requests */
  cmd_stat0 = pic_jpmemrd(p,mport,part_addr+0,4);
  pic_lock(p,LOCK_FREE);
  return (cmd_stat0);
}

int_4 iom_prt_crd (PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 func, int_4 addr, int_4 *data, int_u4 mask)
{
  iom_prt_cmd (p, mport, part_ID, func, addr, *data, mask);
  *data = iom_prt_std (p, mport, part_ID);	/* pin and reg returned in data slot 0 */
  return(0);
}

int_4 iom_prt_wr (PICSTRUCT *p, int_4 mport, int_4 prt, int_4 addr, int_4 data) {
  lpaddr = addr;
  p->iom_buf[p->iom_cnt++] = data;
  if (p->iom_cnt>MAXIOMBUF-2) iom_prt_flush (p,mport,prt);
  return 0;
}
int_4 iom_prt_rd (PICSTRUCT *p, int_4 mport, int_4 prt, int_4 addr, int_4 data) {
  return 0;
}
int_4 iom_prt_dly (PICSTRUCT *p, int_4 mport, int_4 usec) {
  if (p->iom_prt_jvm[mport-1]) iom_dly (p,mport,usec);
  else udelay(usec);
  return 0;
}

/* Generic IO Module state store/retrieve routines */

int_4 *iom_alloc_state (PICSTRUCT *p, int_4 mport, int_4 size)
{
  int_4 i = (mport==2)? 1:0;
  int_4 lsize = (size>=1020)? 255 : (size+3)/4;
  int_4 *iom_state = (int_4 *) malloc(lsize*4);
  if (lsize>=255) print("IOM state size=%d > max, trunc to %d\n",size,lsize*4);
  vfill (0, iom_state, lsize*4);
  iom_state[0] = 0xFFFFFF00|lsize; /* CRC */
  p->iom_state[mport-1] = iom_state;
  return iom_state;
}

int_4 iom_dump_state (PICSTRUCT *p, int_4 mport)
{
  int_4 i = (mport==2)? 1:0, id=0;
  int_4 *iom_state = iom_get_state(p,mport);
  int_4 size = iom_state[0]&0xFF;
  int_4 hbt  = iom_jvm_rd (p,mport,IOM_CFG_HBT,4);
  if (p->mtype[i]==IOMT_D2AWG && p->mrev[i]==3) id = getSP6id(p,mport);
  printf("IOM State Structure for port=%d type=%s id=%02x addr=%08x hbt=%08x\n",
		mport,getIomName(p->mtype[i],p->mrev[i]),id,IOM_CFG_STB,hbt);
  for (i=0; i<size; i++) {
    printf("  Offset=%04x %s=%08x\n",i, (i==0)?"ChkSum":"OpData",pic_jpmemrd (p,mport,IOM_CFG_STB+i*4,4));
  }
  return size;
}

int_4 *iom_get_state (PICSTRUCT *p, int_4 mport)
{
  int_4 i,crc,size,*iom_state;

  if (mport<1 || mport>2) { printf("Illegal port=%d in iom_get_state, using port=1.\n",mport); mport=1; }

  /* init local structure */
  if (p->iom_state[mport-1]==0) {
    crc = pic_jpmemrd (p,mport,IOM_CFG_CRC,4);
    size = (crc&0xFF);
    if (size==0 || size==0xFF) return NULL;
    iom_state = iom_alloc_state(p,mport,size*4);
  } else {
    iom_state = p->iom_state[mport-1];
    crc = iom_state[0];
  }
  size = (crc&0xFF);
  if (size<1) size=0x3FC;

  if (p->iom_buf==0) p->iom_buf = (int_4 *) malloc(MAXIOMBUF*4);

  /* get checksum from PCIe fast register */
  RD(REG_IOMCRCS+(mport-1)*4,crc);
  /* get checksum from module */
  if (crc==0) crc = pic_jpmemrd (p,mport,IOM_CFG_CRC,4);

  /* if not equal to local checksum reload from module */
  if (iom_state[0]!=crc) {
    for (i=0; i<size; i++) {
      iom_state[i] = pic_jpmemrd (p,mport,IOM_CFG_STB+i*4,4);
    }
  }
  return iom_state;
}

int_4 iom_set_state (PICSTRUCT *p, int_4 mport)
{
  int_4 i,crc,size,*iom_state;

  if (mport<1 || mport>2) { printf("Illegal port=%d in iom_set_state\n",mport); return -1; }
  iom_state = p->iom_state[mport-1];
  if (iom_state==0) { printf("No current state for port=%d in iom_set_state \n",mport); return -1; }
  crc = iom_state[0];
  size = (crc&0xFF);

  /* flush current actions */
  iom_exec (p,mport);

  /* calculate checksum for the local buffer */
  iom_state[0] = size;
  crc = crc32_le(0xFFFFFFFF,iom_state,size*4);
  iom_state[0] = (crc&0xFFFFFF00) | size;

  /* get checksum from module */
  RD(REG_IOMCRCS+(mport-1)*4,crc);
  if (crc==0) crc = pic_jpmemrd (p,mport,IOM_CFG_CRC,4);

  /* if not equal to local checksum - load into module */
  if (iom_state[0]!=crc) {
    for (i=0; i<size; i++) {
      pic_jpmemwr (p,mport,IOM_CFG_STB+i*4,iom_state[i],4);
    }
    WR(REG_IOMCRCS+(mport-1)*4,iom_state[0]);
  }
  return size;
}

/* Generic IO Module parameter  routines */

/* get generic config key */
int_4 iom_getcfg (PICSTRUCT *p, int_4 mport, char* key, void *data) {
  int_4 iport = (mport==2)? 1:0;
  int_4 *lval = (int_4*)data;
  real_8 *dval = (real_8*)data;
  int_4 stat = -1;
  char type = 'L';
  if (stat<0) return 0;
  if (key[1]==':') { type=key[0]; key+=2; }
  stat = findflag(key,p->config);
  if (stat<0) ;
  else if (type=='L') { *lval=findintflag(key,p->config); stat=4; }
  else if (type=='D') { *dval=finddblflag(key,p->config); stat=8; }
  return stat;
}

/* get existence of config key */
int_4 iom_getcfgx (PICSTRUCT *p, int_4 mport, char* key) {
  return (findflag(key,p->config)>=0)?1:0;
}
/* get boolean config key */
int_4 iom_getcfgb (PICSTRUCT *p, int_4 mport, char* key) {
  return (findintflagdef(key,p->config,0)!=0)?1:0;
}
/* get integer config key */
int_4 iom_getcfgl (PICSTRUCT *p, int_4 mport, char* key, int_4 def) {
  int_4 iport = (mport==2)? 1:0;
  if (strcmp(key,"RATE")==0) return p->iomparam[iport].rate;
  if (strcmp(key,"BITS")==0) return p->iomparam[iport].bits;
  if (strcmp(key,"GAIN")==0) return p->iomparam[iport].gain;
  return findintflagdef(key,p->config,def);
}
/* get integer mask config key */
int_4 iom_getcfgm (PICSTRUCT *p, int_4 mport, char* key, int_4 def, char *list) {
  return findmaskflagdef(key,p->config,list,def);
}
/* get double config key */
real_8 iom_getcfgd (PICSTRUCT *p, int_4 mport, char* key, real_8 def) {
  int_4 iport = (mport==2)? 1:0;
  if (strcmp(key,"RATE")==0) return (real_8)p->iomparam[iport].rate;
  return finddblflagdef(key,p->config,def);
}

int_4 iom_getkey (PICSTRUCT *p, int_4 mport, int_4 key, void *data) {
  int_4 iport = (mport==2)? 1:0;
  int_4 mtype = p->mtype[iport];
  int_4 mrev  = p->mrev[iport];
  int_4 stat = -1;
  switch (mtype) {
    case IOMT_A2D:
    switch (mrev) {
      case 14: stat=a2dm14_getkey(p,mport,key,data); break;
    }
    break;
    case IOMT_D2AWG:
    break;
  }
  if (stat == -1) { /* then get from config string */
    stat = iom_getcfg (p,mport,pic_key2name(p,key),data);
  }
  return stat;
}
  
int_4 iom_setkey (PICSTRUCT *p, int_4 mport, int_4 key, void *data) {
  int_4 iport = (mport==2)? 1:0;
  int_4 mtype = p->mtype[iport];
  int_4 mrev  = p->mrev[iport];
  int_4 stat = -1;
  switch (mtype) {
    case IOMT_A2D:
    switch (mrev) {
      case 14: stat=a2dm14_setkey(p,mport,key,data); break;
    }
    break;
    case IOMT_D2AWG:
    break;
  }
  return stat;
}
