
/****************************************************************************/
/*                                                                          */
/*    A2DM20 module interface routine                                       */
/*                                                                          */
/****************************************************************************/

#include "iomlib_a2dm20x.h" 

int_4 a2dm20_wr (PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 func, int_4 addr, int_4 data, int_u4 mask)
{
  int_4 cport = 1;

  if (part_ID==AD6688_ID) {
    if (func==PFUNC_PWR){
      iom_prt_cmd(p,cport,part_ID,func,0,data,mask);
    } else {
      iom_prt_cmd(p,cport,part_ID,func,1,addr,0xFFFF);
      iom_prt_cmd(p,cport,part_ID,func,0,data,mask);
    }
  } else if (part_ID==JESD_AVS_ID) {
    if (func==PFUNC_PWR){
      iom_prt_cmd(p,cport,part_ID,func,0,data,mask);
    } else {
      iom_prt_cmd(p,cport,part_ID,func,1,addr,0xFFFF);
      iom_prt_cmd(p,cport,part_ID,func,0,data,mask);
    }
  } else if (part_ID==JESD_CFG_ID) {
    if (func==PFUNC_PWR){
      iom_prt_cmd(p,cport,part_ID,func,0,data,0);
    } else if (addr==0xFFF) {
      iom_prt_cmd(p,cport,part_ID,func,0,data,0);
    } else {
      iom_prt_cmd(p,cport,part_ID,func,1,addr,0);
      iom_prt_cmd(p,cport,part_ID,func,2,data,0);
    }
  } else {
    iom_prt_cmd(p,cport,part_ID,func,addr,data,mask);
  }
  return(0);
}

int_4 a2dm20_rd (PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 func, int_4 addr, int_4 *data, int_u4 mask)
{
  int_4 cport = 1;

  if(part_ID==AD6688_ID) {
    iom_prt_cmd(p,cport,part_ID,PFUNC_RWR,1,addr,0xFFFF);
    iom_prt_crd(p,cport,part_ID,PFUNC_RRD,2,data,0xFFFF);
  } else if(part_ID==JESD_AVS_ID) {
    if (func==PFUNC_PRD){
      iom_prt_crd(p,mport,part_ID,func,addr,data,mask);
    } else {
      iom_prt_cmd(p,cport,part_ID,PFUNC_RWR,1,addr,0xFFFF);
      iom_prt_crd(p,cport,part_ID,PFUNC_RRD,2,data,0xFFFF);
    }
  } else if(part_ID==JESD_CFG_ID) {
    iom_prt_cmd(p,cport,part_ID,PFUNC_RWR,1,addr,0xFFFF);
    iom_prt_crd(p,cport,part_ID,PFUNC_RRD,3,data,0xFFFF);
  } else {
    iom_prt_crd(p,mport,part_ID,func,addr,data,mask);
  }
  return(0);
}

int_4 a2dm20_ud (PICSTRUCT *p, int_4 cport, int_4 usec)
{
  iom_prt_dly (p,cport,usec);
  return 0;
}

int_4 a2dm20_get_linkstat (PICSTRUCT *p, int_4 mport)
{
  int_4 status;
  int_4 stat_data;
  int_4 cport=1;
  int_4 extref;
  int_4 pps_stat, pps_cnt;

  a2dm20_rd(p,cport,OSC_ID,PFUNC_PRD,0,&extref,0xFFFF);
  a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,(extref|0x04),0xFFFF); /* Reset link clk counter */
  a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,(extref&0xFFFD),0xFFFF); /* Clear reset */
  a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,(extref|0x02),0xFFFF); /* Enable link clk counter */

  udelay(20000); 

  a2dm20_rd(p,cport,OSC_ID,PFUNC_PRD,1,&stat_data,0xFFFF);

  return(stat_data);
}

int_4 a2dm20_get_clkstat (PICSTRUCT *p, int_4 mport)
{
  int_4 status;
  int_4 stat_data;
  int_4 cport=1;
  int_4 extref;
  int_4 pps_stat, pps_cnt;

  a2dm20_rd(p,cport,OSC_ID,PFUNC_PRD,0,&extref,0xFFFF);
  a2dm20_rd(p,cport,MOD_ID,PFUNC_PRD,1,&stat_data,0xFFFF);
  a2dm20_rd(p,cport,MTGO_ID,PFUNC_RRD,1,&pps_stat,0xFFFF);
  a2dm20_rd(p,cport,MTGO_ID,PFUNC_RRD,2,&pps_cnt,0xFFFF);

  extref &= 0x1;
  status = 0;
  if (extref && (stat_data&0x1000)==0) status |= 0x1;
  if ((stat_data&0x2000)==0) status |= 0x2;

  return(status);
}

void ad6688_initjesd(PICSTRUCT *p, int_4 mport)
{
/* 7 Initialize JESD link - */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_START_REG, 0x4F, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_START_REG, 0x0F, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_CAL_REG, 0x00, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_CAL_REG, 0x04, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_CAL_REG, 0x00, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_LOL_REG, 0x08, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_LOL_REG, 0x00, 0xFFFF);
}

int_4 ad6688_cfg(PICSTRUCT *p, int_4 mport, struct AD6688_obj *ad6688_inst)
{
  real_8 jesd_rate; 
  int_4 ad6688_reg = 0;
  int_4 Qignore,Isrc,Qsrc;
  int_4 i;

  int_4 DCS1 = findintflagdef("AD6688_DCS1",p->config,0);
  int_4 DCS2 = findintflagdef("AD6688_DCS2",p->config,3);
  int_4 adcfs = findintflagdef("AD6688_ADCFS",p->config,0);
  if(adcfs>0) adcfs=0x0F;

/* RESET AD6688... */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0002,0x03,0xFFFF);
  a2dm20_ud(p,mport,5000);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0002,0x00,0xFFFF);
  a2dm20_ud(p,mport,5000);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0000,0x01,0xFFFF);
  a2dm20_ud(p,mport,5000);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0001,0x02,0xFFFF);
  a2dm20_ud(p,mport,5000);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x011C,DCS1,0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x011E,DCS2,0xFFFF);
  a2dm20_ud(p,mport,5000);
/* Config Clock Divider */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0108,0x01,0xFFFF);
  a2dm20_ud(p,mport,5000);
/* CLOCK DUTY CYCLE STABLIZER CIRCUITS */
  //a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x011C,DCS1<<1&DCS1,0xFFFF);
  //a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x011E,DCS2<<1&DCS2,0xFFFF);
/* Config ADC Bias */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x1910,adcfs,0xFFFF);
/* Config SYSREF */
  ad6688_reg = AD6688_SYSREF_MODE_SEL(0); /* Disable SYSREF during setup */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_SYS1_CNTL_REG,ad6688_reg,0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0109,0x00,0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x010A,0x00,0xFFFF);
  ad6688_reg = AD6688_SYSREF_MODE_SEL(1); /* Enable Continuous SYSREF */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_SYS1_CNTL_REG,ad6688_reg,0xFFFF);
/* Config Clock Delay Control */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0110,0x00,0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0111,0x00,0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0112,0x00,0xFFFF);
/* 1 Power down the link */
  /*  1a - 0x0571 0x15 */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_JESD_CTL1_REG, 0x15, 0xFFFF);

  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_SMPL_MODE_REG, 0x01, 0xFFFF);

  switch (ad6688_inst->chip_dec) {
    case 1: ad6688_reg = AD6688_CHIP_DEC(0); break; 
    case 2: ad6688_reg = AD6688_CHIP_DEC(1); break; 
    case 3: ad6688_reg = AD6688_CHIP_DEC(8); break; 
    case 4: ad6688_reg = AD6688_CHIP_DEC(2); break; 
    case 5: ad6688_reg = AD6688_CHIP_DEC(5); break; 
    case 6: ad6688_reg = AD6688_CHIP_DEC(9); break; 
    case 8: ad6688_reg = AD6688_CHIP_DEC(3); break; 
    case 10: ad6688_reg = AD6688_CHIP_DEC(6); break; 
    case 12: ad6688_reg = AD6688_CHIP_DEC(10); break; 
    case 15: ad6688_reg = AD6688_CHIP_DEC(7); break; 
    case 16: ad6688_reg = AD6688_CHIP_DEC(4); break; 
    case 20: ad6688_reg = AD6688_CHIP_DEC(13); break; 
    case 24: ad6688_reg = AD6688_CHIP_DEC(11); break; 
    case 30: ad6688_reg = AD6688_CHIP_DEC(14); break; 
    case 40: ad6688_reg = AD6688_CHIP_DEC(15); break; 
    case 48: ad6688_reg = AD6688_CHIP_DEC(12); break; 
    default: ad6688_reg = AD6688_CHIP_DEC(0);
  } 
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_CHIP_DEC_REG, ad6688_reg, 0xFFFF); 

  Qignore = 0;
  Isrc = 0;
  Qsrc = 0;
  for(i=0;i<ad6688_inst->ddc_en;++i) {
    Qignore = 0;
    if(ad6688_inst->DDC[i].src == 2) {
      Isrc = 0;
      Qsrc = 1;
    } else {
      Isrc = ad6688_inst->DDC[i].src;
      Qsrc = ad6688_inst->DDC[i].src;
    }

    switch (ad6688_inst->DDC[i].dec) {
     case 2: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(3);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 3: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(7) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 4: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(0);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 6: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(4);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 8: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(1);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 10: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(2) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 12: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(5);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 15: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(8) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 16: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(2);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 20: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(3) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 24: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(6);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 30: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(9) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 40: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(4) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 48: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     default: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(1);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
    }
  }
  if(ad6688_inst->ddc_en > 2) {
    ad6688_reg = AD6688_CHIP_QIGN(Qignore) | AD6688_CHIP_MODE(3);
  } else {
    ad6688_reg = AD6688_CHIP_QIGN(Qignore) | AD6688_CHIP_MODE(ad6688_inst->ddc_en);
  }
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_CHIP_MODE_REG, ad6688_reg, 0xFFFF); 

/* 2 Set JESD link configuration options */
  jesd_rate = (ad6688_inst->M * ad6688_inst->NP * 10 * ad6688_inst->fs) / (8 * ad6688_inst->chip_dec * ad6688_inst->L);
  vprint("A2DM20 JESD Rate calculated as %f MHz\n",jesd_rate);
  if(jesd_rate < 1.6875e3) {
    printf("JESD Lane rate too low, something is messed up...\n");
    return(-1);
  }else if(jesd_rate < 3.375e3){
    ad6688_reg = AD6688_LANE_RATE(5); 
  }else if(jesd_rate < 6.75e3){
    ad6688_reg = AD6688_LANE_RATE(1); 
  }else if(jesd_rate < 13.5e3){
    ad6688_reg = AD6688_LANE_RATE(0); 
  }else if(jesd_rate > 15.5e3){
    printf("JESD Lane rate too high, something is messed up...\n");
    return(-1);
  }else {
    ad6688_reg = AD6688_LANE_RATE(3); 
  }
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_CNTL_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = (ad6688_inst->L==1)? 0xFE : (ad6688_inst->L==2)? 0xFC : (ad6688_inst->L==4)? 0xF0 : 0x00;
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_JESD_LPD_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_SCR(1) | AD6688_L(ad6688_inst->L-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_SCR_L_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_F(ad6688_inst->F-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_F_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_K(ad6688_inst->K-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_K_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_M(ad6688_inst->M-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_M_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_CS(ad6688_inst->CS) | AD6688_N(ad6688_inst->N-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_CS_N_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_SUBCLS(ad6688_inst->SUBCLS) | AD6688_NP(ad6688_inst->NP-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_SUBCLS_NP_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_S(ad6688_inst->S-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_S_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_HD(0) | AD6688_CF(0);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_HD_CF_REG, ad6688_reg, 0xFFFF);

/* 3 Configure the detailed options (?) */
/* 4 Set output lane mapping (Default good) */
/* 5 set additional driver configuration options (?) */
/* 6 Power up the link */
  /*   6a - 0x0571 0x14 */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_JESD_CTL1_REG, 0x14, 0xFFFF);

/* 7 Initialize JESD link - */
  ad6688_initjesd(p,mport);

  return (0);
}

int_4 a2dm20_lmx2592(PICSTRUCT *p, int_4 mport, real_8 rate, int_4 div, int_4 pwrlvl, int_4 chrgpump, int_4 chrggn, int_4 vco_idac, int_4 vco_capctrl)
{
  int_4 i;
  int_4 (*lmx2582_regs)[2];
  int_4 Fpd = 160000000;
  int_4 PLL_N_PRE = 2;
  int_4 PLL_DEN = 1000000;
  real_8 Fvco_, Fvco_rem;
  int_4 PLL_N,PLL_NUM;
  int_4 cmdbuf_data;
  int_4 M,N;
  int_4 vco_sel;

  int_4 lmx2582_base_table[LMX_TBL_SIZE][2] = {
    {0x40, 0x00AF},{0x3E, 0x0000},{0x3D, 0x0001},{0x3B, 0x0000},
    {0x30, 0x03FC},{0x2F, 0x08E0},{0x2E, 0x2021},{0x2D, 0xF080},
    {0x2C, 0x02FA},{0x2B, 0x0020},{0x2A, 0x0000},{0x29, 0xE100},
    {0x28, 0x05F5},{0x27, 0x8104},{0x26, 0x0018},{0x25, 0x4000},
    {0x24, 0x0848},{0x23, 0x039B},{0x22, 0xC3F0},{0x21, 0x4210},
    {0x20, 0x4210},{0x1F, 0x0401},{0x1E, 0x0034},{0x1D, 0x0084},
    {0x1C, 0x2924},{0x19, 0x0000},{0x18, 0x0509},{0x17, 0x8842},
    {0x16, 0x2300},{0x14, 0x012C},{0x13, 0x0965},{0x0E, 0x0420},
    {0x0D, 0x4000},{0x0C, 0x7002},{0x0B, 0x0028},{0x0A, 0x10D8},
    {0x09, 0x0B02},{0x08, 0x1084},{0x07, 0x28B2},{0x04, 0x1943},
    {0x02, 0x0500},{0x01, 0x0809},{0x00, 0x231C}
  };

  lmx2582_regs = &lmx2582_base_table[0];

  a2dm20_wr(p, mport, LMX2582_ID, PFUNC_PWR, 0, 1, 0xFFFF);
  a2dm20_ud(p, mport, 500000);
  a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR, 0x00,0x01,0xFFFF);

  Fvco_ = rate / ((real_8)(Fpd*PLL_N_PRE));
  Fvco_rem = Fvco_ - (int_4)Fvco_;

  PLL_N = (int_4)Fvco_;
  PLL_NUM = (int_4)(Fvco_rem * PLL_DEN);

  M = PLL_NUM;
  N = PLL_DEN;

  lmx2582_regs[11][1]  = (N & 0x0000FFFF);
  lmx2582_regs[12][1]  = (N & 0xFFFF0000)>>16;
  lmx2582_regs[14][1] = (PLL_N & 0xFFF)<<1;
  lmx2582_regs[7][1]  = (M & 0x0000FFFF);
  lmx2582_regs[8][1]  = (M & 0xFFFF0000)>>16;
  lmx2582_regs[5][1]  = (lmx2582_regs[5][1] & 0xFFC0) | (pwrlvl & 0x003F);
  lmx2582_regs[6][1]  = (lmx2582_regs[6][1] & 0xC0FF) | ((pwrlvl<<8) & 0x3F00); 
  lmx2582_regs[31][1]  = (lmx2582_regs[31][1] & 0xF00F) | ((chrgpump<<2) & 0x007C) | ((chrgpump<<7) & 0x0F80) | (chrggn & 0x0003); 
  
  for (i=0; i<LMX_TBL_SIZE-1; i++) { 
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR, lmx2582_regs[i][0],lmx2582_regs[i][1],0xFFFF);
  } 

  a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  8, LMX_VCO_IDAC_OVR(1), LMX_VCO_IDAC_OVR(1));
  a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR, 19, LMX_VCO_IDAC(vco_idac), LMX_VCO_IDAC(0x1FF));

  if(div == 12) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(0), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(4), LMX_SEG3(0xF));
  } else if(div == 16) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(1), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(2), LMX_SEG3(0xF));
  } else if(div == 24) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(1), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(4), LMX_SEG3(0xF));
  } else if(div == 32) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(1), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(8), LMX_SEG3(0xF));
  } else if(div == 48) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(2), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(4), LMX_SEG3(0xF));
  } else if(div == 64) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(2), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(8), LMX_SEG3(0xF));
  } else if(div == 72) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(4), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(4), LMX_SEG3(0xF));
  } else if(div == 96) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(8), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(4), LMX_SEG3(0xF));
  } else if(div == 128) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(8), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(8), LMX_SEG3(0xF));
  } else if(div == 144) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(1), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(4), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(8), LMX_SEG3(0xF));
  } else {
    printf("ERROR: LMX divide of %d not supported...\n",div);
  }

  a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR, lmx2582_regs[i][0],lmx2582_regs[i][1], 0xFFFF);

  return(0);
}

/*---------------------------------------------------------------------------*/
/* A2DM20 - Public Interfaces for Module Initialization, Setup, and Adjust   */
/*---------------------------------------------------------------------------*/

int_4 frac2rat(real_8 x, int* M, int* N)
{
  real_8 a,b,e;
  int_4 h1,h2,k1,k2,ux;
  int_4 iter,j;

    iter = 16;
    b = x;
    h1 = 1;  h2 = 0;  k1 = 0;  k2 = 1;
    for(j=0;j<iter;j++) {
      a = (int)b;
      ux = h1;
      h1 = a*h1 + h2;
      h2 = ux;
      ux = k1;
      k1 = a*k1 + k2;
      k2 = ux;
      e = x-((real_8)h1/(real_8)k1);

      *M = h1;
      *N = k1;
      if(fabs(e*1e15) < 1) return(0);
      b = 1/(b-a);
    }
    return(-1);
}

int_4 DDC_setfreq (PICSTRUCT *p, int_4 mport, struct AD6688_obj *ad6688_inst)
{
  real_8 fs,fc,x,fa,xcf=0,xcfb=0;
  int_4 ddc_num,cport,offset;
  int_4 M,N,i,inv;
  real_8 tmp;
  int_8 ftw,maw,mbw;

  fs = ad6688_inst->fs;
  fc = ad6688_inst->DDC[mport-1].fc;
  inv = ad6688_inst->spinv;

  if (p->xcvr) {
    if (p->xena && p->xinv) xcf = p->xcf;
  } else {
    xcf = p->xcf;		/* the center freq of down converted signal in MHz spectrally flipped */
    xcfb = p->xcfb;		/* the center freq of down converted signal in MHz to switch to bypass */
    if (xcfb > 0.0 && fc < xcfb) xcf = 0.0;
  }
  if (xcf != 0.0) fc -= 2*xcf;	/* flip the XCVR center freq */

  ddc_num = mport-1;
  cport = 1;

  if (inv) fc = -fc;
  x = fmod(fc,fs)/fs;
  if(frac2rat(x,&M,&N) < 0) printf("ERROR:: Tuning to %0.16f not possible... M=%d,N=%d\n",fc,M,N); 

  tmp = (real_8)M/(real_8)N;
  ftw = (int_8)(tmp * 2.81474976710656E14);
  maw = (((int_8)M % (int_8)N) * ((int_8)2.81474976710656E14 % (int_8)N)) % (int_8)N;
  mbw = (int_8)N;

  fa = (((real_8)ftw+((real_8)maw/(real_8)mbw))*fs) / 2.81474976710656E14;
  if (inv) fa = -fa;
  if (xcf != 0.0) fa += 2*xcf;	/* flip the XCVR center freq */

/*
  printf("Fc = %f, Fs = %f\n",fc,fs);
  printf("x = %f\n",x);
  printf("M = %d, N = %d\n",M,N);
  printf("ftw = %lld, maw = %lld, mbw = %lld\n", ftw,maw,mbw);
  printf("Desired Freq = %f, Actual Freq = %f\n", fc,fa);
*/
  ad6688_inst->DDC[mport-1].fc = fa;

  offset = ddc_num*AD6688_DDCx_OFFSET;
  for (i=0; i<6; i++) a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,offset+AD6688_DDC_PINC+i, (ftw>>(i*8))&0xFF, 0xFFFF);
  offset = ddc_num*AD6688_MxW_OFFSET;
  for (i=0; i<6; i++) a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,offset+AD6688_DDC_MAW+i, (maw>>(i*8))&0xFF, 0xFFFF);
  for (i=0; i<6; i++) a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,offset+AD6688_DDC_MBW+i, (mbw>>(i*8))&0xFF, 0xFFFF);

  return(0);
}

real_8 a2dm20_get_adcrate(PICSTRUCT *p, int_4 mport)
{
  real_8 adcrate;
  struct AD6688_obj *ad6688_ptr;
  int_4 cport = 1;

  ad6688_ptr = (struct AD6688_obj*) iom_get_state(p,cport);
  adcrate = ad6688_ptr->fs;

  return(adcrate);
}

real_8 a2dm20_get_freq(PICSTRUCT *p, int_4 mport)
{
  real_8 freq;
  struct AD6688_obj *ad6688_ptr;
  int_4 cport = 1;

  ad6688_ptr = (struct AD6688_obj*) iom_get_state(p,cport);

  if(ad6688_ptr->M == 2) freq = ad6688_ptr->DDC[0].fc;
  else                   freq = ad6688_ptr->DDC[mport-1].fc;

  return(freq);
}

int_4 a2dm20_set_freq (PICSTRUCT *p, int_4 mport, real_8 freq, struct AD6688_obj *ad6688_ptr)
{
  real_8 freq_;
  int_4 cport = 1;
  
  if (p->xcvr) freq = doXcvrFunc(p,"FREQ",freq);
  if(ad6688_ptr->a2dports == 5 ) mport=1;
  ad6688_ptr->DDC[mport-1].fc = freq;
  DDC_setfreq (p, mport, ad6688_ptr);
  freq_ = ad6688_ptr->DDC[mport-1].fc;
  if(freq_ != freq) printf("WARNING: Desired Frequency %0.16f MHz cannot be met, using %0.16f MHz\n",freq,freq_);

  return (0);
}

int_4 a2dm20_get_gain(PICSTRUCT *p, int_4 mport)
{
  int_4 cport = 1;
  int_4 gain;
  int_4 gain_;

  pic_lock(p,LOCK_ALLOC);

  if(mport == 1) {
    a2dm20_rd(p,cport,LMH6401_0_ID,PFUNC_RRD,2,&gain_,0xFFFF);
  }else { 
    a2dm20_rd(p,cport,LMH6401_1_ID,PFUNC_RRD,2,&gain_,0xFFFF);
  }

  gain = 31 - gain_;

  pic_lock(p,LOCK_FREE);

  return(gain);
}

int_4 a2dm20_get_attn(PICSTRUCT *p, int_4 mport)
{
  int_4 cport = 1;
  int_4 attn;
  real_8 attn_;

  pic_lock(p,LOCK_ALLOC);

  if(mport == 1) {
    a2dm20_rd(p,cport,PE43704_0_ID,PFUNC_RRD,0,&attn,0xFFFF);
  }else { 
    a2dm20_rd(p,cport,PE43704_1_ID,PFUNC_RRD,0,&attn,0xFFFF);
  }

  attn_ = (real_8)attn/4;
  attn = round(attn_);

  pic_lock(p,LOCK_FREE);

  return(attn);
}

int_4 a2dm20_get_adclm(PICSTRUCT *p, int_4 mport)
{
  int_4 adclm = 0;
  int_4 adclm_ = 0;
  int_4 cport = 1;

  pic_lock(p,LOCK_ALLOC);

  if(mport == 1) {
    a2dm20_rd(p,cport,ADCLM_0_ID,PFUNC_RRD,0,&adclm_,0xFFFF);
  } else {
    a2dm20_rd(p,cport,ADCLM_1_ID,PFUNC_RRD,0,&adclm_,0xFFFF);
  }

  adclm = adclm_to_cf_i32(adclm_);

  pic_lock(p,LOCK_FREE);

  return(adclm);
}

int_4 a2dm20_get_temp(PICSTRUCT *p, int_4 mport)
{
  int_4 bufaddr;
  int_4 t1,t2;
  int_4 cmdata;
  int_4 i = 0;
  int_4 cport=1;

  pic_lock(p,LOCK_ALLOC);

  /* Read from converter (t1) */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x18E6,0x01,0xFFFF);
  a2dm20_ud(p,cport,15000);
  a2dm20_rd(p,cport,MCP3421_0_ID,PFUNC_RRD,0,&t1,0);
  t1 = (t1 & 0xFFFF0000)>>16;

  /* Read from converter (t2) */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x18E6,0x02,0xFFFF);
  a2dm20_ud(p,cport,15000);
  a2dm20_rd(p,cport,MCP3421_0_ID,PFUNC_RRD,0,&t2,0);
  t2 = (t2 & 0xFFFF0000)>>16;

  /* calculate temperature
  temp(degrees C) = abs(t2-t1) * 3.874 - 273.15 */
  cmdata = round( abs(t2-t1)*3.874 - 273.15 );

  pic_lock(p,LOCK_FREE);
  return(cmdata);
}

int_4 a2dm20_get_rfpwr(PICSTRUCT *p, int_4 mport)
{
  int_4 bufaddr;
  int_4 rfpwr;
  int_4 cport = 1;
  real_8 pwrdb_;
  int_4 pwrdb;
  int_4 attn;
  real_8 attn_;

  pic_lock(p,LOCK_ALLOC);

  if(mport==1) {
    a2dm20_rd(p,cport,MCP3421_1_ID,PFUNC_RRD,0,&rfpwr,0);
    a2dm20_rd(p,cport,PE43704_0_ID,PFUNC_RRD,0,&attn,0xFFFF);
  } else {
    a2dm20_rd(p,cport,MCP3421_2_ID,PFUNC_RRD,0,&rfpwr,0);
    a2dm20_rd(p,cport,PE43704_1_ID,PFUNC_RRD,0,&attn,0xFFFF);
  }

  rfpwr = (rfpwr & 0xFFFF0000)>>16;
  attn_ = (real_8)attn/4;

  pwrdb_ = 30*rfpwr*1e-3 - 70 + attn_;
  pwrdb = round(pwrdb_);

  pic_lock(p,LOCK_FREE);
  return(pwrdb);
}

int_4 a2dm20_set_attn (PICSTRUCT *p, int_4 mport, int_4 attn,  struct AD6688_obj *ad6688_ptr)
{
  int_4 cport = 1;
  if (p->xcvr) { doXcvrFunc(p,"ATTN",(double)attn); return 0; }
  attn = 4*attn;
  a2dm20_wr(p,cport,(mport==2)?PE43704_1_ID:PE43704_0_ID,PFUNC_RWR,0,attn,0xFFFF);
  ad6688_ptr->rfattn[mport-1]=attn;
  return(0);
}

/* 
  total 63 db range, 0->31 in var attn, -6->26 in var gain amp
  must keep output of attn <= -15db to avoid compression in +15db fixed amp
  6db loss in components after fixed amp and another unknown 6dB is made up in var amp
  full scale input range of A2D is 1.9V peak-to-peak or ~6dB into 50 ohms
  apparent gain between attn and amp = +15 -2 -2 -2 -6 = +3
*/
int_4 a2dm20_set_gain (PICSTRUCT *p, int_4 mport, int_4 gain, struct AD6688_obj *ad6688_ptr)
{
  int_4 cport=1,rfgain=0,rfattn=31,gm=4;
  int_4 expert = findintflagdef("RFEXPERT",p->config,0);

 if (expert) {
  rfattn = -1;
  rfgain = (gain<-6)? -6 : (gain>26)? 26 : gain;	/* bounds check (-6 to 26) */
  if (rfgain!=gain) printf("A2Dm20 expert mode gain=%d out of bounds (-6,26)\n",gain);
 } else {
       if (gain<=-34){ rfattn=31; rfgain=-6; printf("A2Dm20 gain=%d < min=-35\n",gain); } /* below lowest zone */
  else if (gain<=-6) { rfattn=31; rfgain=gain-gm+rfattn; }
  else if (gain<=24) { rfgain=21; rfattn=rfgain+gm-gain; }
  else if (gain<=30) { rfattn=0 ; rfgain=gain-gm+rfattn; }
  else               { rfgain=26; rfattn=0; printf("A2Dm20 gain=%d > max=30\n",gain); }	/* above highest zone */
 }
 vprint("A2DM20 GAIN mport=%d expert=%d gain=%d rfattn=%d rfgain=%d\n",mport,expert,gain,rfattn,rfgain);

  /* dont tweak attn in expert mode */
  if (expert==0) {
    a2dm20_wr(p,cport,(mport==2)? PE43704_1_ID:PE43704_0_ID, PFUNC_RWR, 0,4*rfattn,0xFFFF);
    ad6688_ptr->rfattn[mport-1]=rfattn;
  }
  a2dm20_wr(p,cport,(mport==2)? LMH6401_1_ID:LMH6401_0_ID, PFUNC_RWR, 2,26-rfgain,0xFFFF);
  ad6688_ptr->rfgain[mport-1]=(expert)? rfgain:gain;
  return(0);
}

int_4 a2dm20_enable (PICSTRUCT *p, int_4 mport, int_4 ena)
{
  int_4 cport = 1;

  if (ena) a2dm20_wr(p,cport,MOD_ID,PFUNC_PWR,0,0x03,0xFFFF); 
  else if (pic_acmd(p,ADON_RD,DMA_GMCFG+(3-mport),0,0)!=0); /* other side still active */
  else a2dm20_wr(p,cport,MOD_ID,PFUNC_PWR,0,0x01,0xFFFF);
  iom_exec (p,cport);
  if (ena==0 && p->xcvr>0) doXcvrFunc(p,"FREQ",0.0);

  return(0);
}

int_4 get_rststat (PICSTRUCT *p, int_4 mport)
{
  int_4 rst_dat0;
  int_4 rst_dat1;
  int_4 jesd_rst,link_val,pll_locked,digital_rst,analog_rst,waitrequest;
  int_4 rx_ready,cal_busy,lockedtodata;

  a2dm20_rd(p,mport,JESD_RST_ID,PFUNC_PRD,1,&rst_dat0,0xFFFF);
  a2dm20_rd(p,mport,JESD_RST_ID,PFUNC_PRD,2,&rst_dat1,0xFFFF);

  jesd_rst = JESD_STAT_RST(rst_dat0);
  link_val = JESD_STAT_RXVAL(rst_dat0);
  pll_locked = JESD_STAT_PLLLOC(rst_dat0);
  digital_rst = JESD_STAT_DIGRST(rst_dat0);
  analog_rst = JESD_STAT_ALGRST(rst_dat0);
  waitrequest = JESD_STAT_WTREQ(rst_dat0) ;
  rx_ready = JESD_STAT_RXRDY(rst_dat0) ;
  cal_busy = JESD_STAT_CALBSY(rst_dat1);
  lockedtodata = JESD_STAT_RXLOC(rst_dat1) ;

  printf("JESD RST STAT[0] = 0x%x\n",rst_dat0);
  printf("JESD RST STAT[1] = 0x%x\n",rst_dat1);
  printf("jesd_rst = 0x%x\n",jesd_rst);
  printf("link_val = 0x%x\n",link_val);
  printf("pll_locked = 0x%x\n",pll_locked);
  printf("digital_rst = %d, analog_rst = %d\n",digital_rst,analog_rst);
  printf("waitrequest = %d\n",waitrequest);
  printf("rx_ready = 0x%x\n",rx_ready);
  printf("cal_busy = 0x%x\n",cal_busy);
  printf("lockedtodata = 0x%x\n",lockedtodata);
 
  return(0);
}

int_4 xcvr_setup (PICSTRUCT *p, int_4 mport, struct AD6688_obj *ad6688_inst)
{
  int_4 i,channel,offset,timecnt;
  int_4 xcvr_rddata;
  int_4 xcvr_addr,xcvr_mask,xcvr_wrdata;
  int_4 xcvr_rmw;
  int_4 config_sel;
  int_4 cmdbuf_data;
  int_4 timeout = 100;

#if JESD_SOFT_PCS 
  switch(ad6688_inst->pll_config) {
    case 12:  config_sel = -1; break;
    case 16:  config_sel = 0; break;
    case 24:  config_sel = 0; break;
    case 32:  config_sel = 1; break;
    case 40:  config_sel = 2; break;
    case 48:  config_sel = 3; break;
    case 60:  config_sel = 4; break;
    case 64:  config_sel = 5; break;
    case 72:  config_sel = -1; break;
    case 80:  config_sel = 6; break;
    case 96:  config_sel = 7; break;
    case 120: config_sel = 8; break;
    case 128: config_sel = 8; break;
    case 144: config_sel = -1; break;
    default:  config_sel = 0; break;
  }
#else
  switch(ad6688_inst->chip_dec) {
    case 2:  config_sel = 7; break; /* 7500 */
    case 3:  config_sel = 5; break; /* 5000 */
    case 4:  config_sel = 7; break; /* 7500 */ 
    case 6:  config_sel = 5; break; /* 5000 */
    case 8:  config_sel = 7; break; /* 7500 */
    case 10: config_sel = 6; break; /* 6000 */
    case 12: config_sel = 5; break; /* 5000 */
    case 15: config_sel = 4; break; /* 4000 */
    case 16: config_sel = 3; break; /* 3750 */
    case 20: config_sel = 2; break; /* 3000 */
    case 24: config_sel = 1; break; /* 2500 */
    case 30: config_sel = 0; break; /* 2000 */
    case 40: config_sel = 2; break; /* 3000 */
    case 48: config_sel = 1; break; /* 2500 */
    default: config_sel = 0; break;
  }
#endif
  for(channel=0;channel<8;++channel) {
  /* Get AVMM Interface */
    xcvr_addr   = 0x000;
    xcvr_mask   = 0xFF;
    xcvr_wrdata = 0x02;
    xcvr_addr = xcvr_addr | (channel << XCVR_CFG_DPRIO_ADDR_WIDTH);
    if(xcvr_mask == 0xFF) {
      a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,xcvr_addr,xcvr_wrdata,0xFFFF);
    } else {
/* printf("%d: Reading AVMM Interface\n",channel); */
      a2dm20_rd(p,mport,JESD_CFG_ID,PFUNC_RRD,xcvr_addr,&xcvr_rddata,0xFFFF);
      xcvr_rmw = (xcvr_rddata & ~xcvr_mask) | (xcvr_wrdata & xcvr_mask);
/* printf("   Writing to get AVMM Interface: 0x%x\n",xcvr_rmw); */
      a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,xcvr_addr,xcvr_rmw,0xFFFF);
    }
/*    a2dm20_rd(p,mport,JESD_RST_ID,PFUNC_PRD,1,&cmdbuf_data,0xFFFF); */
/* printf("RECONFIG WAITREQUEST: %d\n",JESD_STAT_WTREQ(cmdbuf_data)); */

    v2print("Using PLL Configuration %d, DEC = %d\n",config_sel,ad6688_inst->DDC[0].dec);
    a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,0xFFF,((channel<<8)|config_sel),0xFFFF);

  /* Initiate recalibration of CDR PLL */
    xcvr_addr   = 0x100;
    xcvr_mask   = 0x02;
    xcvr_wrdata = 0x02;
    xcvr_addr = xcvr_addr | (channel << XCVR_CFG_DPRIO_ADDR_WIDTH);
/* printf("   Reading Calibrate CDR\n"); */
    a2dm20_rd(p,mport,JESD_CFG_ID,PFUNC_RRD,xcvr_addr,&xcvr_rddata,0xFFFF);
    xcvr_rmw = (xcvr_rddata & ~xcvr_mask) | (xcvr_wrdata & xcvr_mask);
/* printf("   Writing to Calibrate CDR: 0x%x\n",xcvr_rmw); */
    a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,xcvr_addr,xcvr_rmw,0xFFFF);

    timecnt = 0;
    do{
      a2dm20_rd(p,mport,JESD_RST_ID,PFUNC_PRD,2,&cmdbuf_data,0xFFFF);
      /* printf("XCVR RECONFIG CALIBRATION STAT(0 = DONE): 0x%x\n",JESD_STAT_CALBSY(cmdbuf_data)); */
      /* udelay(2000); */
    }while((JESD_STAT_CALBSY(cmdbuf_data) != 0) && (timecnt++ < timeout));

  /* Release AVMM Interface to PreSICE */
    xcvr_addr   = 0x000;
    xcvr_mask   = 0xFF;
    xcvr_wrdata = 0x01;
    xcvr_addr = xcvr_addr | (channel << XCVR_CFG_DPRIO_ADDR_WIDTH);
    a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,xcvr_addr,xcvr_wrdata,0xFFFF);

/* printf("   Reading return AVMM Interface\n"); */
    /* a2dm20_rd(p,mport,JESD_CFG_ID,PFUNC_RRD,xcvr_addr,&xcvr_rddata,0xFFFF); */
    /* xcvr_rmw = (xcvr_rddata & ~xcvr_mask) | (xcvr_wrdata & xcvr_mask); */
/* printf("   Writing to return AVMM Interface: 0x%x\n",xcvr_rmw); */
    /* a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,xcvr_addr,xcvr_rmw,0xFFFF); */
  }

  return(0);
}

real_8 jesd_findLMF (PICSTRUCT *p, int_4 mport, int_4 rate, struct AD6688_obj *ad6688_inst, int_4 *lmx_div, int_4 *link_clk_pll, int_4 *dec_idx)
{
  int_4 i,idx;
  real_8 fs,fs_fix,rate_,rate_err;
  real_8 jesd_lane_rate,link_clk_rate,sysref_ip,sysref_ad;
  int_4 sysref_ip_div,sysref_ad_div;
  int_4 D;
  int_4 F;
  int_4 M = 4; /* Two DDCs Default configuration, change only if necessary */
  int_4 S = 2;
  int_4 Np = 16;
  int_4 L = M;

  fs_fix = finddblflagdef("ADCRATE",p->config,0.0);
  if(fs_fix > 0.0) {
    idx = 0;
    rate_ = fs_fix/a2dm20x_dec_table[0];
    rate_err = abs(rate_ - ((real_8)rate)/1e6);
    /* printf("JJB: fs_fix = %f, rate=%d, rate_=%f, rate_err = %f, dec = %d\n",fs_fix, rate, rate_, rate_err, a2dm20x_dec_table[0]); */
    for(i=1;i<M20x_DECTBL_SIZE;++i) {
      rate_ = fs_fix/a2dm20x_dec_table[i];
      if(rate_err > abs(rate_ - ((real_8)rate)/1e6)) { idx = i; rate_err = abs(rate_ - ((real_8)rate)/1e6); }
      /* printf("JJB: fs_fix = %f, rate=%d, rate_=%f, rate_err = %f, idx = %d, dec = %d\n",fs_fix, rate, rate_, rate_err, idx, a2dm20x_dec_table[idx]); */
    }
    i = idx;
    fs = fs_fix;
    rate_ = fs_fix/a2dm20x_dec_table[i];
    if(rate_err > 0.0) {
      printf("WARNING: Unable to meet desired rate of %f MHz, using %f MHz instead\n",((real_8)rate)/1e6,rate_);
      rate = (int_4)(rate_*1e6);
    }
  } else {
    for(i=0;i<M20x_DECTBL_SIZE;++i) {
      fs = a2dm20x_dec_table[i]*((real_8)rate)/1e6;
      if ((fs <= M20_RATE_MAX) && (fs >= M20_RATE_MIN)) break;
  }
  }
  if (i==M20x_DECTBL_SIZE) {
    printf("ERROR: No valid Configuration found for rate=%d\n",rate);
    /* Do something here that doesn't blow up the M20...*/
    return(-1);
  } else D = a2dm20x_dec_table[i];
  *dec_idx = (i==M20x_DECTBL_SIZE)? 10 : i;

  jesd_lane_rate = (M * Np * 10 * fs) / (8 * D * L);
  if (jesd_lane_rate > JESD_RATE_MAX) {
    L=2*M;
    jesd_lane_rate = (M * Np * 10 * fs) / (8 * D * L);
    if (jesd_lane_rate > JESD_RATE_MAX) {
      M=M/2;
      jesd_lane_rate = (M * Np * 10 * fs) / (8 * D * L);
    }
  }
  if (jesd_lane_rate < JESD_RATE_MIN) {
    M=M*2;
    jesd_lane_rate = (M * Np * 10 * fs) / (8 * D * L);
  }

  if ( (jesd_lane_rate > JESD_RATE_MAX) || (jesd_lane_rate < JESD_RATE_MIN) || (M>8) || (L>8) ) {
    printf("ERROR: Unable to find suitable LMF for rate=%d\n",rate);
    L=0;M=0;F=0;
    return(-1);
  } else F = M*S*Np/(8*L);

  link_clk_rate = jesd_lane_rate/40;

  *lmx_div = (int_4)(2*fs/link_clk_rate);

  ad6688_inst->L = L;
  ad6688_inst->M = M;
  ad6688_inst->F = F;
  ad6688_inst->N = 16;
  ad6688_inst->NP = Np;
  ad6688_inst->K = 32;
  ad6688_inst->CS = 0;
  ad6688_inst->S = S;
  ad6688_inst->HD = 0;
  ad6688_inst->CF = 0;
  ad6688_inst->chip_dec = D;
  ad6688_inst->ddc_en = M/2;
  sysref_ip_div = (32*M)/L;
  sysref_ad_div = sysref_ip_div;
  a2dm20_wr(p,mport,JESD_SYS_ID,PFUNC_PWR,0,((((sysref_ad_div/2)-1)&0x00FF)<<8)|(((sysref_ip_div/2)-1)&0x00FF),0xFFFF);

  sysref_ip = link_clk_rate/sysref_ip_div;
  sysref_ad = (link_clk_rate * (*lmx_div)) / 128;
  v2print("JESD Link Parameters::::\n");
  v2print("Desired Rate = %d\n",rate);
  v2print("Calculated Sample Rate = %f, Decimation = %d\n",fs,D);
  v2print("LMF = %d%d%d\n",L,M,F);
  v2print("JESD lane rate = %f\n",jesd_lane_rate);
  v2print("Link clk = %f, LMX_divide = %f\n",link_clk_rate,2*fs/link_clk_rate);
  v2print("FPGA SYSREF calculation: D=32*M/L=%d, FPGA sysref = %f, ADC sysref = %f, ratio=%f\n",sysref_ip_div,sysref_ip,sysref_ad,sysref_ad/sysref_ip);

  *link_clk_pll = 0;
  ad6688_inst->pll_config = *lmx_div;
  if (*lmx_div == 120) {
    *link_clk_pll = 1;
    *lmx_div = 96;
  } else if (*lmx_div == 80) {
    *link_clk_pll = 1;
    *lmx_div = 64;
  } else if (*lmx_div == 60) {
    *link_clk_pll = 1;
    *lmx_div = 48;
  } else if (*lmx_div == 40) {
    *link_clk_pll = 1;
    *lmx_div = 32;
  }

  return(fs);  
}

int_4 jesd_setup (PICSTRUCT *p, int_4 mport, struct AD6688_obj *ad6688_inst)
{
  int_4 jesd_ilas1;
  int_4 jesd_ilas2;
  int_4 jesd_ilas12;
  int_4 jesd_lane[8];
  int_4 fxk;
  int_4 i;

  jesd_ilas1  = 0;
  jesd_ilas2  = 0;
  jesd_ilas12 = 0;

  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,0x00,0x00,0xFFFF);
  for(i=0;i<8;i++) {
    jesd_lane[i] = 0x04;
    if (i>=ad6688_inst->L) jesd_lane[i]|=0x02;
    /* printf("Setting JESD Register 0x%x = 0x%x\n",(i+1)<<2,jesd_lane[i]); */
    a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,i+1,jesd_lane[i],0xFFFF);
  }

  jesd_ilas1 |= ((ad6688_inst->L-1) & 0x1F);
  jesd_ilas1 |= (1<<7); /* Enable SCR */
  jesd_ilas1 |= ((ad6688_inst->F-1) & 0xFF)<<8;
  jesd_ilas1 |= ((ad6688_inst->K-1) & 0x1F)<<16;
  jesd_ilas1 |= ((ad6688_inst->M-1) & 0xFF)<<24;
  
  /* printf("JESD ILAS_DATA1 = 0x%x\n",jesd_ilas1); */

  jesd_ilas2 |= ((ad6688_inst->N-1) & 0x1F);
  jesd_ilas2 |= ((ad6688_inst->CS) & 0x03)<<6;
  jesd_ilas2 |= ((ad6688_inst->NP-1) & 0x1F)<<8;
  jesd_ilas2 |= ((ad6688_inst->SUBCLS) & 0x07)<<13; /* Subclass 1 */
  jesd_ilas2 |= ((ad6688_inst->S-1) & 0x1F)<<16;
  jesd_ilas2 |= (1)<<21; /* JESD204B */
  jesd_ilas2 |= ((ad6688_inst->CF) & 0x1F)<<24;
  jesd_ilas2 |= ((ad6688_inst->HD) & 0x01)<<31;

  /* printf("JESD ILAS_DATA2 = 0x%x\n",jesd_ilas2); */

  fxk = ad6688_inst->F * ad6688_inst->K;
  jesd_ilas12 |= (fxk-1)&0x3FF;

  /* printf("JESD ILAS_DATA12 = 0x%x\n",jesd_ilas12); */

  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,0x25,jesd_ilas1,0xFFFF);
  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,0x26,jesd_ilas2,0xFFFF);
  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,0x30,jesd_ilas12,0xFFFF);
  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,(0x54)>>2,0x3F5,0xFFFF);

  return(0);
}

int_4 a2dm20_set_jesd(PICSTRUCT *p, int_4 mport, int_4 addr, int_4 data)
{
  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,addr,data,0xFFFF);
  return(0);
}

int_4 a2dm20_get_jesd(PICSTRUCT *p, int_4 mport, int_4 addr)
{
  int_4 x;
  a2dm20_rd(p,mport,JESD_AVS_ID,PFUNC_RWR,addr,&x,0xFFFF);
  return(x);
}

int_4 a2dm20_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags)
{ 
  real_8 rfrate,rfratef;
  struct AD6688_obj *ad6688_inst;
  int_4 cmdbuf_data;
  int_4 cport = 1;
  int_4 extref;
  real_8 rffreq,rffreq2,rffreq3,rffreq4;
  int_4 rfdec,rfdecf;
  int_4 rfattn;
  int_4 rfgain;
  int_4 hbt;
  int_4 link_clk_pll;
  int_4 lmx_idac,lmx_capctrl,lmx_pwr;
  int_4 ddcsrc,ddcsrc2,ddcsrc3,ddcsrc4;
  int_4 testmode,testsel;
  int_4 i,err,indx,reconfig_cnt,jesdsync_cnt;
  int_4 lmx_div;
  int_4 a2dports,bus_rate;
  int_4 adclmB1 = 0; 
  int_4 adclmB2 = 0; 
  int_4 adclmD  = 0; 
  int_4 timeout = 10;
  int_4 oldM20,adcgainsel;
  int_4 a2dports_,pack8,pack12,pack8dbg;
  int_4 jesd_subc,jesdlmf,dec_idx;
  int_4 ferror,ferror_,findx;
  int_4 lsbp,ppstag,mtgo,pps_offset,tcdel;
  int_4 lanes,calbusy,rxlock,link_val;
  int_4 stat_data;
  int_4 ddc0_dbg,ddc1_dbg,ddc2_dbg,ddc3_dbg;
  int_4 pfir,impsel,spinv;
  int_4 rate_;
  int_4 expert;
  real_8 time;
  time = gettime();

  expert = findintflagdef("RFEXPERT",p->config,0);
  extref = findintflagdef("PREFX",p->config,0);
  rfratef = finddblflagdef("ADCRATE",p->config,0.0);
  adcgainsel = findintflagdef("ADCGAINSEL",p->config,1);
  rffreq = finddblflagdef("RFFREQ",p->config,625.0); 
  rffreq2 = finddblflagdef("RFFREQ2",p->config,rffreq);
  rffreq3 = finddblflagdef("RFFREQ3",p->config,rffreq);
  rffreq4 = finddblflagdef("RFFREQ4",p->config,rffreq);
  rfdecf  = findintflagdef("RFDEC",p->config,0);
  rfattn = findintflagdef("RFATTN",p->config,expert?31:0);
  rfgain = findintflagdef("RFGAIN",p->config,0);
  ddcsrc = findintflagdef("DDCSRC",p->config,0);
  ddcsrc2 = findintflagdef("DDCSRC2",p->config,1);
  ddcsrc3 = findintflagdef("DDCSRC3",p->config,0);
  ddcsrc4 = findintflagdef("DDCSRC4",p->config,1);
  testmode = findintflagdef("TESTMODE",p->config,0);
  testsel = findintflagdef("TESTSEL",p->config,0);
  lmx_pwr = findintflagdef("LMXPWR",p->config,0);
  lmx_idac = findintflagdef("LMXIDAC",p->config,300);
  lmx_capctrl = findintflagdef("LMXCAP",p->config,32);
  adclmB1 = findintflagdef("ADCLMB1",p->config,9)&0xFF;
  adclmB2 = findintflagdef("ADCLMB2",p->config,15)&0xFF;
  adclmD  = findintflagdef("ADCLMD", p->config,10)&0xFF;
  oldM20  = findintflagdef("OM20", p->config,0);
  pack8dbg = findintflagdef("PACK8DBG", p->config,0);
  a2dports_ = findintflagdef("A2DPORTSDBG", p->config,-1);
  lsbp = findintflagdef("LSBP", p->config,0);
  ppstag = findintflagdef("PPSTAG", p->config,0);
  mtgo = findintflagdef("MTGO", p->config,0);
  pack12 = (bits==-12)? 1:0;
  pack8  = (bits==-8)? 1:0;
  pack12 = findintflagdef("PACK12", p->config,pack12);
  jesd_subc = findintflagdef("JESDSC",p->config,0);
  ddc0_dbg = findintflagdef("DDC0T",p->config,0);
  ddc1_dbg = findintflagdef("DDC1T",p->config,0);
  ddc2_dbg = findintflagdef("DDC2T",p->config,0);
  ddc3_dbg = findintflagdef("DDC3T",p->config,0);
  pfir = findintflagdef("A2DPFIR", p->config,0);
  impsel = findintflagdef("IMPSEL", p->config,0);
  spinv = findintflagdef("SPINV", p->config,0);

  iom_init (p,cport);		/* initialize IOM transport */
#if NEWSTUFF == 2
  iom_set_prt_jvm(p,cport,findintflagdef("JVMCTRL",p->config,1));	/* use JVM as commander ? */
#else
  iom_set_prt_jvm(p,cport,findintflagdef("JVMCTRL",p->config,0));	/* use JVM as commander ? */
#endif

  if (flags&FLG_SGO) {
    vprint("A2DM20 setup SGO port\n");
    ad6688_inst = (struct AD6688_obj *) iom_get_state (p,cport);
    if (ad6688_inst==NULL) ad6688_inst = (struct AD6688_obj *)iom_alloc_state(p,cport,sizeof(struct AD6688_obj));
    if (ad6688_inst==NULL) return -1;
    a2dm20_set_gain(p,mport,rfgain,ad6688_inst);
    a2dm20_set_freq(p,mport,rffreq,ad6688_inst);
    if (expert||p->xena) a2dm20_set_attn(p,mport,rfattn,ad6688_inst);
    iom_set_state(p,cport);
    return (ad6688_inst->a2dports);
  }

  ad6688_inst = (struct AD6688_obj *)iom_alloc_state(p,cport,sizeof(struct AD6688_obj));
  ad6688_inst->SUBCLS = jesd_subc;

  rate_ = rate;
  jesdsync_cnt = 0;

for (reconfig_cnt=1; reconfig_cnt<=3; reconfig_cnt++) {

  /* SET MAX ATTN, MIN GAIN ON START */
  a2dm20_set_gain(p,1,-30,ad6688_inst);
  a2dm20_set_gain(p,2,-30,ad6688_inst);

  /* SET UP INTERNAL/EXTERNAL CLOCKING */
  if(extref) {
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x101,0xFFFF);
    a2dm20_ud(p,cport,100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x501,0xFFFF);
    a2dm20_ud(p,cport,100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x101,0xFFFF);
    a2dm20_ud(p,cport,100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x1101,0xFFFF);
  } else {
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x110,0xFFFF);
    a2dm20_ud(p,cport,100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x510,0xFFFF);
    a2dm20_ud(p,cport,100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x110,0xFFFF);
    a2dm20_ud(p,cport,100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x1110,0xFFFF);
  }
  /* check clock status */
  a2dm20_rd(p,cport,MOD_ID,PFUNC_PRD,1,&stat_data,0xFFFF);
  if (stat_data&0x1000) { printf("Warn: Failed lock to %sternal 10MHz reference\n",extref?"ex":"in"); continue; }

  /* DETERMINE RATE, DECIMATION AND JESD LINK PARAMETERS */
  if(pfir==1 && rate_ >= 100000000 && rate_<=600000000) {
    rate=rate_*2; /* FPGA 90% Half Band Filter is enabled*/
    v2print("Using 90%% FIR\n");
  }else {
    rate = rate_;
    pfir = 0;
    v2print("Using 80%% FIR\n");
  }
  
  rfrate = jesd_findLMF(p,cport,rate,ad6688_inst,&lmx_div,&link_clk_pll,&dec_idx);
  if(rfrate<0) { printf("ERROR: Invalid A2D Setup, Change Rate or Sample Rate\n"); a2dports=-1; goto DONE; }
  if(ad6688_inst->L==4) lanes = 0x0F;
  else if(ad6688_inst->L==8) lanes = 0xFF;

  link_clk_pll = findintflagdef("LINK_SEL", p->config,link_clk_pll);

  rfdec = ad6688_inst->chip_dec;
  bus_rate = (rfrate/rfdec) * 2*((pack12)? 12 : (pack8)? 8 : 16) / 8;
  bus_rate = (pfir)? bus_rate/2 : bus_rate;
  a2dports = (a2dports_>=0)? a2dports_ : (bus_rate > 2000)? 5 : 4;

  if (p->xcvr>0) rffreq = doXcvrFunc(p,"FREQ",rffreq);	/* set Tx and return IF */

  ad6688_inst->a2dports = a2dports;
  ad6688_inst->fs = rfrate; /* Sample Rate in MHz */
  ad6688_inst->gain = adcgainsel;
  ad6688_inst->DDC[0].fc = rffreq; /* Center Frequency in MHz */
  ad6688_inst->DDC[0].dec = rfdec; 
  ad6688_inst->DDC[0].src = ((a2dports==5) && (mport==2))? 1 : ddcsrc;
  ad6688_inst->DDC[1].fc = rffreq2; /* Center Frequency in MHz */
  ad6688_inst->DDC[1].dec = rfdec; 
  ad6688_inst->DDC[1].src = ddcsrc2;
  ad6688_inst->DDC[2].fc = rffreq3; /* Center Frequency in MHz */
  ad6688_inst->DDC[2].dec = ad6688_inst->chip_dec;
  ad6688_inst->DDC[2].src = ddcsrc3;
  ad6688_inst->DDC[3].fc = rffreq4; /* Center Frequency in MHz */
  ad6688_inst->DDC[3].dec = ad6688_inst->chip_dec;
  ad6688_inst->DDC[3].src = ddcsrc4;

  ad6688_inst->spinv = spinv;

  /* CONFIGURE FPGA DATA ROUTING */
  a2dm20_wr(p,cport,ROUTE_ID,PFUNC_PWR,0,(a2dports<<13)|(impsel<<12)|(pfir<<11)|(link_clk_pll<<8)|(pack8dbg<<6)|(pack8<<5)|(pack12<<4)|(ppstag<<2)|(mtgo<<1)|lsbp,0xFFFF);

  vprint("A2DM20 RFRATE = %f, RFDEC = %d, RFFREQ = %f, DDCSRC = %d, M=%d\n", rfrate,rfdec,rffreq,ddcsrc,ad6688_inst->M);
  v2print("A2DM20 LVDS Rate=%d MB/s, A2DPORTS=%d\n",bus_rate,a2dports);

  /* POWER DOWN LMX */
  a2dm20_wr(p, cport, LMX2582_ID, PFUNC_PWR, 0, 0, 0xFFFF);
  a2dm20_ud(p,cport,5000);

  /* RESET JESD IP CORE */
  a2dm20_wr(p,cport,MOD_ID,PFUNC_PWR,0,0x0000,0xFFFF);
  a2dm20_wr(p,cport,JESD_CFG_ID,PFUNC_PWR,0,0x0001,0xFFFF); /* set reconfig resets */
  a2dm20_wr(p,cport,JESD_RST_ID,PFUNC_PWR,0,0x0006,0xFFFF); /* set analog and digital resets */

  /* CONFIGURE LMX CLOCK CHIP */
  a2dm20_lmx2592(p,cport,1e6*ad6688_inst->fs*2,lmx_div,lmx_pwr,8,0,lmx_idac,lmx_capctrl);

  /* CONFIGURE AD6688 */
  ad6688_cfg(p,cport,ad6688_inst);
  DDC_setfreq (p, mport, ad6688_inst);
  /* Use Test Mode to verify JESD is properly aligned on startup */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0550,0x06,0xFFFF); /* PN Sequence Testmode */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,0x05,0xFFFF); /* DDC0 I/Q Testmode */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,0x05,0xFFFF); /* DDC0 I/Q Testmode */
#ifdef BOGUS
  if(testmode) {
  /* Set User Test Pattern */  
    if(testmode>=7) {
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0551, 0xAA, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0552, 0xAA, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0553, 0xAA, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0554, 0xAA, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0555, 0xAA, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0556, 0xAA, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0557, 0xAA, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0558, 0xAA, 0xFFFF);
    } else { 
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0551, 0x00, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0552, 0x11, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0553, 0x22, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0554, 0x33, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0555, 0x44, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0556, 0x55, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0557, 0x66, 0xFFFF);
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0558, 0x77, 0xFFFF);
    }

    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0550,0x00,0xFFFF); /* Ramp Output for Testmode */
    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0573,0x00,0xFFFF); /* User Pattern output for Testmode */
    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,0,0xFFFF);
    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,0,0xFFFF);

    if(testsel==1)  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0550,0x0F,0xFFFF); /* Ramp Output for Testmode */
    if(testsel==2)  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0550,0x08,0xFFFF); /* User Pattern output for Testmode */
    if(testsel==3)  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0573,0x08,0xFFFF); /* JESD Ramp Output for Testmode */
    if(testsel==4)  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0573,0x0E,0xFFFF); /* JESD User Pattern output for Testmode */

    if(testmode==1) a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,1,0xFFFF); /* DDC0 I Testmode */
    if(testmode==2) a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,4,0xFFFF); /* DDC0 Q Testmode */
    if(testmode==3) a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,5,0xFFFF); /* DDC0 I/Q Testmode */
    if(testmode==4) a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,1,0xFFFF); /* DDC1 I Testmode */
    if(testmode==5) a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,4,0xFFFF); /* DDC1 Q Testmode */
    if(testmode==6) a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,5,0xFFFF); /* DDC1 I/Q Testmode */
    if(testmode==7) {
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,5,0xFFFF); /* DDC0 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,5,0xFFFF); /* DDC1 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0367,5,0xFFFF); /* DDC2 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0387,5,0xFFFF); /* DDC3 I/Q Testmode */
    }
    if(testmode==8) {
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,4,0xFFFF); /* DDC0 Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,5,0xFFFF); /* DDC1 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0367,5,0xFFFF); /* DDC2 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0387,5,0xFFFF); /* DDC3 I/Q Testmode */
    }
    if(testmode==9) {
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,1,0xFFFF); /* DDC0 I Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,5,0xFFFF); /* DDC1 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0367,5,0xFFFF); /* DDC2 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0387,5,0xFFFF); /* DDC3 I/Q Testmode */
    }
    if(testmode==10) {
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,5,0xFFFF); /* DDC1 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0367,5,0xFFFF); /* DDC2 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0387,5,0xFFFF); /* DDC3 I/Q Testmode */
    }
    if(testmode==11) {
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,ddc0_dbg,0xFFFF); /* DDC0 I Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,ddc1_dbg,0xFFFF); /* DDC1 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0367,ddc2_dbg,0xFFFF); /* DDC2 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0387,ddc3_dbg,0xFFFF); /* DDC3 I/Q Testmode */
    }
  }
  else {
    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0550,0,0xFFFF);
    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,0,0xFFFF);
    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,0,0xFFFF);
  }
#endif

  /* CONFIGURE RF GAIN/ATTN */
  if (expert||p->xena) a2dm20_set_attn(p,mport,rfattn,ad6688_inst);
  a2dm20_set_gain(p,mport,rfgain,ad6688_inst);

  /* Verify CORE PLL is locked */
  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,1,&cmdbuf_data,0xFFFF);
  v2print("A2DM20 CORE PLL NEEDS TO BE LOCKED: %d\n",JESD_STAT_PLLLOC(cmdbuf_data));

  /* Verify Xcvr Calibration Completed */
  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,2,&cmdbuf_data,0xFFFF);
  v2print("A2DM20 XCVR CALIBRATION NEEDS TO COMPLETE (0 = DONE): 0x%x\n",JESD_STAT_CALBSY(cmdbuf_data));
  /* if(JESD_STAT_CALBSY(cmdbuf_data) != 0) return(0); */

  /* Program Xcvr PLL?? */
  v2print("A2DM20 Configure XCVR PLL\n");
  a2dm20_wr(p,cport,JESD_CFG_ID,PFUNC_PWR,0,0x0000,0xFFFF);
  xcvr_setup(p,cport,ad6688_inst);
  a2dm20_ud(p,cport,5000);

  /* Release Xcvr Resets */
  v2print("A2DM20 Releasing Xcvr Resets\n");
  a2dm20_ud(p,cport,500);
  a2dm20_wr(p,cport,JESD_RST_ID,PFUNC_PWR,0,0x0004,0xFFFF); /* release analog reset */
  a2dm20_ud(p,cport,5000);
  a2dm20_wr(p,cport,JESD_RST_ID,PFUNC_PWR,0,0x00,0xFFFF); /* release digital reset */

i = 0;
do {
  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,1,&cmdbuf_data,0xFFFF);
  link_val = JESD_STAT_RXVAL(cmdbuf_data);
  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,2,&cmdbuf_data,0xFFFF);
  calbusy = JESD_STAT_CALBSY(cmdbuf_data);
  rxlock = JESD_STAT_RXLOC(cmdbuf_data)&lanes;
  ++i;
  v2print("XCVR Setup %d: calbusy = %x, rxlock = %x, link_val = 0x%x\n",i,calbusy,rxlock,link_val);
} while(((calbusy!=0x00) || (rxlock!=lanes)) && (i < 10));

  /* Relese AVS Reset */
  v2print("A2DM20 Releasing AVS Reset\n");
  a2dm20_wr(p,cport,JESD_AVS_ID,PFUNC_PWR,0,0x01,0xFFFF); 

  /* Configure JESD Core */
  v2print("A2DM20 Configuring JESD Core\n");
  jesd_setup(p,cport,ad6688_inst);

  /* Release Link/Frame Resets */
  v2print("A2DM20 Releasing Link Reset\n");
  a2dm20_wr(p,cport,MOD_ID,PFUNC_PWR,0,0x01,0xFFFF);

  /* Enable SYSREF */
  v2print("A2DM20 Enabling SYSREF\n");
  if(ad6688_inst->SUBCLS==1) a2dm20_wr(p,cport,JESD_RST_ID,PFUNC_PWR,0,0x80,0xFFFF);

  /* Set up ADCLM */
  a2dm20_wr(p,cport,ADCLM_0_ID,PFUNC_RWR,0x01,adclmB1,0xFFFF);
  a2dm20_wr(p,cport,ADCLM_0_ID,PFUNC_RWR,0x02,adclmB2,0xFFFF);
  a2dm20_wr(p,cport,ADCLM_0_ID,PFUNC_RWR,0x03,adclmD,0xFFFF);
  a2dm20_wr(p,cport,ADCLM_1_ID,PFUNC_RWR,0x01,adclmB1,0xFFFF);
  a2dm20_wr(p,cport,ADCLM_1_ID,PFUNC_RWR,0x02,adclmB2,0xFFFF);
  a2dm20_wr(p,cport,ADCLM_1_ID,PFUNC_RWR,0x03,adclmD,0xFFFF);

  a2dm20_rd(p,cport,AD6688_ID,PFUNC_RRD,0x011b,&cmdbuf_data,0xFFFF);
  v2print("A2DM20 AD6688 Clock Status: 0x%x\n",cmdbuf_data);
  a2dm20_rd(p,cport,AD6688_ID,PFUNC_RRD,0x056F,&cmdbuf_data,0xFFFF);
  v2print("A2DM20 AD6688 PLL Status: 0x%x\n",cmdbuf_data);

  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,2,&cmdbuf_data,0xFFFF);
  v2print("A2DM20 XCVR CALIBRATION STAT(0 = DONE): 0x%x\n",JESD_STAT_CALBSY(cmdbuf_data));

  /* Check if JESD Link has been properly establised */
  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,1,&cmdbuf_data,0xFFFF);
  link_val = JESD_STAT_RXVAL(cmdbuf_data);
  v2print("JESD RX Link status = %d\n",link_val);
  if (link_val!=0) {
    /* Check if JESD is aligned properly */
    a2dm20_rd(p,cport,JESD_SYS_ID,PFUNC_PRD,1,&cmdbuf_data,0xFFFF);
    v2print("JESD Sync = %d\n",cmdbuf_data);
    if(cmdbuf_data<3) {
      jesdsync_cnt += 1;
      reconfig_cnt -= 1;
      if(jesdsync_cnt > 3) {
        printf("ERROR: JESD Not in Sync (%d)!!\n",cmdbuf_data);
        break;
      } else {
        vprint("JESD Not in Sync (%d)!! Retry...\n",cmdbuf_data);
      }
    } else {
      /* Enable normal output */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,0x00,0xFFFF); /* DDC0 I/Q Testmode */
      a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0347,0x00,0xFFFF); /* DDC0 I/Q Testmode */
      break;
    }
    printf("A2DM20 JESD not locked, reconfiguring %d\n",reconfig_cnt); 
  }
}
  if(reconfig_cnt>3) { printf("Error setting up A2DM20, check reference\n"); a2dports=-1; goto DONE; }

  a2dm20_rd(p,cport,JESD_AVS_ID,PFUNC_PRD,0x01,&cmdbuf_data,0xFFFF);
  jesdlmf = (cmdbuf_data&0xE000)>>13;
  a2dm20_rd(p,cport,AD6688_ID,PFUNC_RRD,AD6688_SYS1_STAT_REG,&cmdbuf_data,0xFFFF);
  v2print("SYSREF Status 1 = 0x%x\n",cmdbuf_data);

  /* Disable LMX Ref Clock */
  if (findintflagdef("LMXDISABLE",p->config,0)) {
    v2print("Disabling LMX clk for JESD reference\n");
    a2dm20_wr(p, cport, LMX2582_ID, PFUNC_RWR, 46, 0xA1, 0xFFFF);
  }
  /* SET PPS OFFSET */
  pps_offset = a2dm20x_pps_table[jesdlmf-1][dec_idx];
  if (pack12) pps_offset = 25;
  pps_offset += (int_4)(1.0e-9*rate*findintflagdef("MTGOOFF", p->config,pps_offset));
  v2print("PPS Offset: jesdlmf = %d, idx = [%d][%d], offset = %d\n",jesdlmf,jesdlmf-1,dec_idx,pps_offset);
  a2dm20_wr(p,cport,MTGO_ID,PFUNC_RWR,0,pps_offset,0xFFFF);

  /* SET IOM STATE */  
  DONE:
  iom_set_state(p,cport);

  time = gettime()-time;
  vprint("A2DM20 setup elapsed time=%f sec\n",time);
  return(a2dports);
}

int_4 a2dm20_get_ad6688 (PICSTRUCT *p, int_4 mport)
{
  int_4 cmdbuf_data;
  int_4 cport=1;
  a2dm20_rd(p,cport,AD6688_ID,PFUNC_RRD,AD6688_SYS1_STAT_REG,&cmdbuf_data,0xFFFF);
  v2print("SYSREF Status 1 = 0x%x\n",cmdbuf_data);
  return(0);
}

real_4 a2dm20_ntpoff (PICSTRUCT *p, int_4 mport)
{
  double time,frac=-1; 
  int ppstat,cpps,ccur;
  ppstat = iom_rd(p,mport,0x5F00,4);
  time = pic_time();
  cpps = (ppstat>>16)&0xFFFF;
  ccur = (ppstat>> 0)&0xFFFF;
  if (cpps>10000) {
    frac = (double)ccur/cpps;
    frac = fmod(time-frac,1.0);
    if (frac>0.5) frac-=1.0;
  }
  return frac;
}

int_4 a2dm20_testset (PICSTRUCT *p)
{
  int_4 cport = 1;
  int_4 id   = findintflagdef("M20PID", p->config,AD6688_ID);
  int_4 pin  = findintflagdef("M20PIN", p->config,0);
  int_4 addr = findintflagdef("M20ADDR",p->config,0);
  int_4 data = findintflagdef("M20DATA",p->config,0);
  int_4 jesdrst = findintflagdef("M20JESDRST",p->config,0);

  if(jesdrst) {
    vprint("JJB: Running initjesd\n");
    ad6688_initjesd(p,cport);
  } else if(pin) {
    vprint("JJB: in a2dm20_testset PIN WR: id=%d, addr=0x%x, data=0x%x\n",id,addr,data);
    a2dm20_wr(p,cport,id,PFUNC_PWR,addr,data,0xFFFF);
  }else {
    vprint("JJB: in a2dm20_testset REG WR: id=%d, addr=0x%x, data=0x%x\n",id,addr,data);
    a2dm20_wr(p,cport,id,PFUNC_RWR,addr,data,0xFFFF);
  }
  return 0;
}
int_4 a2dm20_testget (PICSTRUCT *p)
{
  int_4 cport = 1;
  int_4 id   = findintflagdef("M20PID", p->config,AD6688_ID);
  int_4 pin  = findintflagdef("M20PIN", p->config,0);
  int_4 addr = findintflagdef("M20ADDR",p->config,0);
  int_4 data;

  if(pin) {
    a2dm20_rd(p,cport,id,PFUNC_PRD,addr,&data,0xFFFF);
    vprint("JJB: in a2dm20_testget PIN RD: id=%d, addr=0x%x, data=0x%x\n",id,addr,data);
  }else {
    a2dm20_rd(p,cport,id,PFUNC_RRD,addr,&data,0xFFFF);
    vprint("JJB: in a2dm20_testget REG RD: id=%d, addr=0x%x, data=0x%x\n",id,addr,data);
  }
  return data;
}

int_4 a2dm20_getkeyl (PICSTRUCT *p, int_4 mport, int_4 key)
{
  int_4 cport=1, rtn=-1;
  A2DM20_State* SS = (A2DM20_State*) iom_get_state (p,cport);
       if (key==KEY_TEMP)   rtn = a2dm20_get_temp (p,mport);
  else if (key==KEY_CLKSTAT) rtn = a2dm20_get_clkstat (p,mport);
  else if (SS==NULL)        rtn = -1;
  else if (key==KEY_RFOPTS) rtn = SS->rfopts;
  else if (key==KEY_RFGAIN) rtn = SS->rfgain[mport-1];
  else if (key==KEY_RFATTN) rtn = SS->rfattn[mport-1];
  else if (key==KEY_RFPWR)  rtn = a2dm20_get_rfpwr (p,mport);
  else if (key==KEY_ADLM)   rtn = a2dm20_get_adclm (p,mport);
  else if (key==KEY_MTOFF)  rtn = a2dm20_get_ad6688 (p,mport);
  else printf("Unhandled getkeyl=%d in a2dm20\n",key);
  return rtn;
}

real_8 a2dm20_getkeyd (PICSTRUCT *p, int_4 mport, int_4 key)
{
  int_4 cport=1; real_8 rtn=-1.0;
  A2DM20_State* SS = (A2DM20_State*) iom_get_state (p,cport);
       if (key==KEY_NTPOFF) rtn = a2dm20_ntpoff (p,mport);
  else if (SS==NULL)        rtn =  -1;
  else if (key==KEY_RFFREQ) rtn = (p->xena)? p->xfreq+SS->DDC[mport-1].fc-p->xcf : SS->DDC[mport-1].fc;
  else if (key==KEY_RFBW)   rtn = SS->fs;
  else printf("Unhandled getkeyd=%d in a2dm20\n",key);
  return rtn;
}

/* already have lock because always called from pic_setkey */

int_4 a2dm20_setkeyl (PICSTRUCT *p, int_4 mport, int_4 key, int_4 i4val)
{
  int_4 cport=1, rtn=0;
  A2DM20_State* SS = (A2DM20_State*) iom_get_state (p,cport);
       if (SS==NULL) return -1;
  else if (key==KEY_RFOPTS) { SS->rfopts=i4val; }
  else if (key==KEY_RFGAIN) { a2dm20_set_gain(p,mport,i4val,SS); }
  else if (key==KEY_RFATTN) { a2dm20_set_attn(p,mport,i4val,SS); }
  else printf("Unhandled setkeyl=%d in a2dm20\n",key);
  iom_set_state (p,cport);
  return rtn;
}

int_4 a2dm20_setkeyd (PICSTRUCT *p, int_4 mport, int_4 key, real_8 r8val)
{
  int_4 cport=1, i4val=r8val;
  int_4 rtn = 0;
  A2DM20_State* SS = (A2DM20_State*) iom_get_state (p,cport);
       if (SS==NULL) return -1;
  else if (key==KEY_RFFREQ) { a2dm20_set_freq(p,mport,r8val,SS); }
  else printf("Unhandled setkeyd=%d in a2dm20\n",key);
  iom_set_state (p,cport);
  return rtn;
}

