package nxm.ice.lib;

import nxm.sys.lib.*;
import nxm.ice.lib.DevIce.DmaMap;
import nxm.ice.lib.FileIO;
import java.io.File;

/** 
  Resource for interacting with an Ice Block Device using native code.
 
  @author Jeff Schoen
  @version $Id: IceBlockResource.java,v 1.5 2003/10/27 22:30:08 schoenj Exp $
*/
public class IceBlockResource extends IOResource {

  private int fb;
  private File file;
  private static int PGSZ=8192,CBSZ=131072;
  private long paddr=-1, psize;
  private long rdaddr, rdsize;
  private String rdpath;
  public  MDevIce pic;
  public  DmaMap map;
  public  int type,devno;
  public  int clen=0;
  public  byte[] cbuf=null;

  public static String typeList = "Raw,Ram,IceFS,PIC,NIC,PAC,IceBlk";
  public static int RAW=0, RAM=1, IFS=2, PIC=3, NIC=4, PAC=5, BLK=6;

  public IceBlockResource() { super(); }

  /** Test for the existence of the resource. */
  public boolean exists () { return file.exists(); }

  public boolean isFile() { return true; }

  /** Set the URL that this resource points to. */
  public void setURL (String str) {
    super.setURL(str);
    int i;
         if (str.startsWith("ramd:"))  { type=RAM; str=str.substring(5); }	// specify RAM disk
    else if (str.startsWith("ice:"))   { type=RAM; str=str.substring(4);	// default to RAM disk
      if ((i=str.indexOf("/icefs"))>0) { type=IFS; devno=Convert.s2l(str.substring(i+6,i+7))-1; }
      if ((i=str.indexOf("/pic"))>0)   { type=PIC; devno=Convert.s2l(str.substring(i+4,i+5))-1; }
      if ((i=str.indexOf("/nic"))>0)   { type=NIC; devno=Convert.s2l(str.substring(i+4,i+5))-1; }
    }
    else if (str.startsWith("file:"))  { type=RAW; str=str.substring(5); }
    else type=RAW;
    if (type==RAW && str.indexOf("/icefs")>=0) type=IFS;
    file = new File (str);
    //System.out.println("IceBlk type="+type+":"+devno+" File="+file);
  }

  /** open the resource */
  public boolean open () { 
         if (type==RAM) openRAM();
    else if (type==PIC) openPIC();
    else if (type==IFS) openFIO();
    else if (type==RAW) openFIO();
    else M.error("IceBlockResource type="+type+" not handled yet");
    offset=0;
    return isOpen;
  }

  public boolean openRAM () { 
    isOpen = false;
    String devn=null;
    String auxstr=null;
    String path = urlstr.substring(0,urlstr.lastIndexOf('/')+1);
    rdpath = path.substring(4);
    Table auxtbl = (Table)M.results.get("AUX");
    for (Table.Iterator ti=auxtbl.iterator(); auxstr==null && ti.getNext(); ) {
      String s = ti.value.toString(); if (s.startsWith(path)) auxstr = s; 
    }
    if (auxstr==null) M.error("IceBlock AUX entry not found for: "+urlstr);
    Parser p = new Parser(auxstr);
    rdpath = p.get(1).substring(5);
    devn   = p.get(2);
    rdaddr = Convert.s2xl (p.get(3));
    rdsize = Convert.s2xl (p.get(4));
    if (urlstr.endsWith("/")) return false;	// just the directory
    pic = new MDevIce(M,"ICERAM,DEVNO=0");	// default driver entry point
    long maxalloc = (long)M.results.getL("ENV.ICE.MAXALLOC",-1);
    if (maxalloc>0) pic.setMaxAlloc(maxalloc<<20);
    if (rdaddr==0 && rdsize!=0) {
      pic.open();
      rdaddr = pic.phys2long( pic.getKeyL(0,MDevIce.KEY_QALLOC) );
      pic.close();
    }
    byte[] detbuf = new byte[16];
    if (dir<=0) {
      int fd = Native.fopen (file.toString(),dir);
      Native.freadb (fd,detbuf,0,16);
      if (fd!=0) Native.fclose(fd); 
      paddr = Convert.unpackX(detbuf,0);
      psize = Convert.unpackX(detbuf,8);
      length = Math.min(length,psize);
    } else {
      lockFileAlloc();
      int fd = Native.fopen (file.toString(),dir);
      // add temp allocation to keep overwrite algorithm simple
      Convert.packX(detbuf,0,-1L);
      Convert.packX(detbuf,8,1L);
      Native.fwriteb (fd,detbuf,0,16);
      Native.fseek(fd,0L);
      psize = (length+PGSZ-1)/PGSZ*PGSZ;
      getAllocation();
      Convert.packX(detbuf,0,paddr);
      Convert.packX(detbuf,8,psize);
      Native.fwriteb (fd,detbuf,0,16);
      if (fd!=0) Native.fclose(fd); 
      lockFileFree();
    }
    map = pic.getMap();
    map.mode  = DmaMap.RAMD;
    map.bytes = length;
    map.paddr = getPhysicalAddress();
    map.dev   = pic;
    map.devn  = null;
    map.setPaged(Shell.ostag.startsWith("lin"));
    isOpen = true;
    return isOpen;
  }

  public boolean openPIC () { 
    isOpen = false;
    fb = FileIO.fopen ("/dev/"+file.toString(),dir);
    FileIO.fseek(fb,getPhysicalAddress());
    isOpen = (fb > 0);
    return isOpen;
  }

  public boolean openFIO () { 
    isOpen = false;
    int flags=0;
    boolean direct = urlstr.endsWith(".det") && (dir>0) && (type!=IFS);
    boolean cache  = urlstr.endsWith(".pkt") && (dir>0);
    boolean thread = false;
    if (qualTable!=null) direct = qualTable.getState("USEDIRECT",direct);
    if (qualTable!=null) thread = qualTable.getState("USETHREADED",thread);
    if (direct) flags |= 4;
    if (thread) flags |= 8;
    fb = FileIO.fopen (file.toString(),dir,flags);
    isOpen = (fb > 0);
    length = file.length();
    if (cache) {
      cbuf = new byte[CBSZ+4096];
      clen = 0;
    }
    return isOpen;
  }
 
  /** close the resource */
  public boolean close() { 
    if (clen>0) flushCache();
    if (isOpen && type==PIC) pic.close();
    if (!isOpen) return false;
    if (fb>0) FileIO.fclose(fb); 
    if (map!=null) map.close(DmaMap.NOCHK);
    isOpen = false; 
    return true;
  }

  /** delete the resource */
  public boolean delete() { return file.delete(); }

  /** get length of resource */
  public long getLength() { return length; }

  /** seek to the specified offset in file */
  public boolean seek (long offset) { 
    if (clen>0) flushCache();
    if (offset==this.offset) return true;
    if (fb>0 && FileIO.fseek(fb,offset)!=offset) return false;
    this.offset = offset;
    return true; 
  }

  /** get currently available length of resource */
  public long avail() { return length-offset; } 

  /** read data from resource */
  public int read (byte[] buf, int boff, int bytes) { 
    bytes = (int)Math.min(bytes,length-offset);
    if (map!=null) {
      long vaddr = map.getVirtualAddress(offset,bytes);
      Native.p2ja (vaddr,0, buf,boff, bytes);
    } else {
      bytes=FileIO.freadb(fb,buf,boff,bytes);
    }
    if (bytes>0) offset += bytes;
    return bytes;
  }

  private void flushCache() {
    FileIO.fwriteb(fb,cbuf,0,clen);
    clen=0;
  }

  /** write data to resource */
  public int write (byte[] buf, int boff, int bytes) { 
    if (map!=null) {
      bytes = (int)Math.min(bytes,length-offset);
      long vaddr = map.getVirtualAddress(offset,bytes);
      Native.ja2p (buf,boff, vaddr,0, bytes);
    } else if (cbuf!=null) {
      if (clen>=CBSZ) flushCache();
      System.arraycopy(buf,boff, cbuf,clen, bytes); clen+=bytes;
    } else {
      bytes=FileIO.fwriteb(fb,buf,boff,bytes);
    }
    if (bytes>0) offset += bytes;
    return bytes;
  }

  /** read data from resource */
  public int read (long lbuf, int boff, int bytes) { 
    bytes = (int)Math.min(bytes,length-offset);
    if (map!=null) {
      long vaddr = map.getVirtualAddress(offset,bytes);
      Native.p2p (vaddr,0, lbuf,boff, bytes);
    } else {
      bytes=FileIO.fread(fb,lbuf,boff,bytes);
    }
    if (bytes>0) offset += bytes;
    return bytes;
  }

  /** write data to resource */
  public int write (long lbuf, int boff, int bytes) { 
    if (map!=null) {
      bytes = (int)Math.min(bytes,length-offset);
      long vaddr = map.getVirtualAddress(offset,bytes);
      Native.p2p (lbuf,boff, vaddr,0, bytes);
    } else {
      bytes=FileIO.fwrite(fb,lbuf,boff,bytes);
    }
    if (bytes>0) offset += bytes;
    return bytes;
  }

  /** Return a printable string containing information about this object. */
  public String toString () { return "IceBlockResource: " + getURL(); }

  private synchronized void getAllocation() {
    String[] lf = M.io.listFiles(rdpath,"*","det");
    int ndet = lf.length;
    long[] paddrs = new long[ndet+1];
    long[] psizes = new long[ndet+1];
    byte[] detbuf = new byte[16];
    for (int i=0; i<ndet; i++) {
      String fname = rdpath+lf[i];
      int fd = Native.fopen (fname,-1);
      Native.freadb (fd,detbuf,0,16);
      Native.fclose(fd);
      long paddri = Convert.unpackX(detbuf,0);
      long psizei = Convert.unpackX(detbuf,8);
      // now sort into ascending order
      int j=0; for (; j<i && paddri>paddrs[j]; j++);
      for (int k=i-1; k>=j; k--) { paddrs[k+1]=paddrs[k]; psizes[k+1]=psizes[k]; }
      paddrs[j]=paddri; psizes[j]=psizei;
    }
    paddr = -1; 
    long minsize = paddrs[ndet] = rdsize;
    for (int n=ndet-1; n>=0; n--) {
      long size = paddrs[n+1] - (paddrs[n]+psizes[n]);
      if (size>=psize && size<=minsize) { paddr=paddrs[n]+psizes[n]; minsize=size; }
    }
    if (paddr<0) M.error("Could not allocate IceBlock buffer of "+psize+" bytes");
  }

  private long extent = 0x01000000-1;

  public int checkAllocation (int bytes) {
    long next = offset + bytes;
    if (length >= next) return bytes;
    length = next;
    if (psize >= length) return bytes;
    psize = (next+bytes+extent) & ~extent;
    byte[] detbuf = new byte[16];
    Convert.packX(detbuf,0,paddr);
    Convert.packX(detbuf,8,psize);
    int fd = Native.fopen (file.toString(),dir);
    Native.fwriteb (fd,detbuf,0,16);
    Native.fclose(fd);
    return bytes;
  }

  public long getPhysicalAddress() { return rdaddr+paddr; }

  public Table getStatFS() {
    if (!isOpen) open();
    String[] lf = M.io.listFiles(rdpath,"*","det");
    int files = lf.length;
    long total = 0;
    byte[] detbuf = new byte[16];
    for (int i=0; i<files; i++) {
      String fname = rdpath+lf[i];
      int fd = Native.fopen (fname,-1);
      Native.freadb (fd,detbuf,0,16);
      Native.fclose(fd);
//    long paddri = Convert.unpackX(detbuf,0);
      long psizei = Convert.unpackX(detbuf,8);
      total += psizei;
    }
    int tmby = (int)(rdsize/1000000);
    int fmby = tmby - (int)(total/1000000);
    int used = (100-100*fmby/tmby);
    String stats = "{PATH="+rdpath+",FILES="+files+",TMBY="+tmby+",FMBY="+fmby+",USED="+used+"}";
    return new Table(stats);
  }

  File lockFile;
  private void lockFileAlloc() {
    lockFile = new File(rdpath+"allocate.LOCK");
    for (int i=0; i<100; i++) {
      try { if (lockFile.createNewFile()) return; } catch (Exception e) { }
      Time.sleep(0.010);
    }
    System.out.println("Overriding stale lock file "+lockFile);
  }

  private void lockFileFree() {
    lockFile.delete();
  }

}
