package nxm.ice.lib;

/**
  A class to handle communications with an ICE DSP card.

  @author Jeff Schoen
  @version %I%, %G%

*/
public class DevIce extends Object {

  protected long handle;
  protected static boolean useNative;
  protected long MAXALLOC=0x10000000L;

  public final static int 
	ICEPIC1=1, ICEPIC2=2, ICEPIC3=3, ICEPIC4=4, ICEPIC5=5, ICEPIC6=6, ICEPIC7=7, ICEPIC8=8, ICEPIC9=9;
  public static String cardList = "ICEPIC1,ICEPIC2,ICEPIC3,ICEPIC4,ICEPIC5,ICEPIC6,ICEPIC7,ICEPIC8,ICEPIC9";

  public final static int 
	IOPT_NONE=0, IOPT_STREAM=1, IOPT_LINK=2, IOPT_MODULE=3, IOPT_TUNER=4, 
	IOPT_INTERNAL=5, IOPT_EXTERNAL=6, IOPT_CORE=7, IOPT_MCORE=8, IOPT_TBANK=9, IOPT_CBANK=10;
  public static String ioptList = "STREAM,LINK,MODULE,TUNER,INTERNAL,EXTERNAL,CORE,MCORE,TBANK,CBANK";

  public final static int 
	IOMT_UNKNOWN=-101, IOMT_TEST=-100, IOMT_AUTO=-99, IOMT_NONE=0;

  public final static int 
	IOMT_E2D=-1, IOMT_LV2D=-2, IOMT_A2D=-3, IOMT_CXD=-4, IOMT_T2D=-5, IOMT_UXD=-6, IOMT_VXD=-7, 
	IOMT_DR2D=-8, IOMT_PSE2D=-9, IOMT_DXFPDP=-10, IOMT_DXFPQC=-11, IOMT_DXUDP=-12, IOMT_SNTXD=-13, IOMT_SDDSXD=-14, 
	IOMT_GXD=-15, IOMT_NFXD=-16, IOMT_CDR2D=-17, IOMT_LB2D=-18, IOMT_FLZRXD=-19, IOMT_RFXD=-20, IOMT_MSASXD=-21, 
	IOMT_DSFPXD=-22, IOMT_SFPXD=-23, IOMT_SFPPXD=-24, IOMT_SE2D=-25, IOMT_TGSDDS2D=-26, IOMT_AWG2D=-27, IOMT_UFLZXD=-28,
	IOMT_GPSXD=-29, IOMT_DIODEI=-30, IOMT_TGVXD=-31, IOMT_RF2D=-32;
  public static String imtList = 
	"E2D,LV2D,A2D,CXD,T2D,UXD,VXD,DR2D,PSE2D,FPDPXD,FPQCXD,UDPXD,SNTXD,SDDSXD,"+
	"GXD,NFXD,CDR2D,LB2D,FLZRXD,RFXD,MSASXD,DSFPXD,SFPXD,SFPPXD,ES2D,TGSDDSXD,AWG2D,UFLZXD,GPS,DIODEI,TGVXD,RF2D";

  public final static int 
	IOMT_D2E=1, IOMT_D2LV=2, IOMT_D2A=3, IOMT_DXC=4, IOMT_D2T=5, IOMT_DXU=6, IOMT_DXV=7, 
	IOMT_D2DR=8, IOMT_D2PSE=9, IOMT_FPDPXD=10, IOMT_FPQCXD=11, IOMT_UDPXD=12, IOMT_DXSNT=13, IOMT_DXSDDS=14, 
	IOMT_DXG=15, IOMT_DXNF=16, IOMT_D2CDR=17, IOMT_D2LB=18, IOMT_DXFLZR=19, IOMT_DXRF=20, IOMT_DXMSAS=21,
	IOMT_DXDSFP=22, IOMT_DXSFP=23, IOMT_DXSFPP=24, IOMT_D2SE=25, IOMT_D2TGSDDS=26,  IOMT_D2AWG=27, IOMT_DXUFLZ=28,
	IOMT_GPS=29, IOMT_DIODE=30, IOMT_DXTGV=31, IOMT_D2RF=32;
  public static String omtList = 
	"D2E,D2LV,D2A,DXC,D2T,DXU,DXV,D2DR,D2PSE,DXFPDP,DXFPQC,DXUDP,DXSNT,DXSDDS,"+
	"DXG,DXNF,D2CDR,D2LB,DXFLZR,DXRF,DXMSAS,DXDSFP,DXSFP,DXSFPP,DXD2SE,TGSDDSXD,D2AWG,DXUFLZ,GPS,DIODE,DXTGV,D2RF";

  public final static int 
	PMT_DTDM=1, PMT_DTDMX=2, PMT_ZPPM=3, PMT_S6M=4, PMT_V5M=5, PMT_V6M=6, PMT_V7M=7, PMT_K8M=8;
  public static String pmList = "DTDM,DTDMX,ZPPM,S6M,V5M,V6M,V7M,K8M";

  public final static int DMA_LOOPNSPIN=-23, DMA_INDEX=-22, 
	DMA_ALLOCATE=-21, DMA_ENABLE=-20, DMA_LOADNSPIN=-19, DMA_STATUS_W64=-18, DMA_ONDEMAND=-17,
	DMA_BURY=-16, DMA_KILL=-15,  DMA_LOOP=-14, DMA_POLL=-13, DMA_ACTIVE=-12, DMA_RESHOT=-11,
	DMA_LOST=-10, DMA_CANCEL=-9, DMA_RESET=-8, DMA_LOAD=-7,  DMA_CYCLE=-5,   DMA_STATUS=-4, 
	DMA_WAIT=-3,  DMA_CONTINUOUS=-2, DMA_SPIN=-1, DMA_STOP=0, DMA_ONESHOT=1;

  public final static int 
	FLG_TEST   =0x00000001, FLG_NOBRK  =0x00000002, FLG_BIGEND =0x00000004, FLG_HS     =0x00000008,
	FLG_MUXCLK =0x00000010, FLG_INTCLK =0x00000020, FLG_TC     =0x00000040, FLG_LOOP   =0x00000080,
	FLG_NOLOCK =0x00000100, FLG_DISABLE=0x00000200, FLG_VERBOSE=0x00000400, FLG_NCCLK  =0x00000800,
	FLG_BOOT   =0x00001000, FLG_TUNER  =0x00002000, FLG_CLKI   =0x00004000, FLG_MSBI   =0x00008000,
	FLG_VHS	   =0x00010000, FLG_DUAL   =0x00020000, FLG_MUX    =0x00040000, FLG_ALT    =0x00080000,
	FLG_SGO	   =0x00100000, FLG_TGO	   =0x00200000, FLG_GGO    =0x00400000, FLG_XGO    =0x00800000,
	FLG_OP1	   =0x01000000, FLG_OP2	   =0x02000000, FLG_OP3    =0x04000000, FLG_OP4    =0x08000000,
	FLG_BIT0   =0x10000000, FLG_BIT4   =0x20000000, FLG_RGO    =0x40000000, FLG_BACKEND=0x80000000, 
        FLG_BIT1   =0x30000000, FLG_NIO    =0x80000000;
  public final static String flagList = 
	"Test,NoBrk,BigEnd,HS,MuxClk,IntClk,TC,Resv,NoLock,Disable,Verbose,NcClk,"+
	"Boot,Tuner,ClkInv,MsbInv,VHS,Dual,Mux,Alt,SGo,TGo,GGo,XGo,Op1,Op2,Op3,Op4,Bit0,Bit4,RGo,NIO";
  public final static int FLG_XTGO = FLG_XGO|FLG_TGO;
  public final static int FLG_STGO = FLG_SGO|FLG_TGO;
  public final static int FLG_XSTGO = FLG_XGO|FLG_SGO|FLG_TGO;

  public final static int 
	FLG_NOWAIT  =0x01000000, FLG_LINK   =0x02000000,
	FLG_FTUN    =0x01000000, FLG_DTUN   =0x02000000,
	FLG_GTUN    =0x04000000, FLG_MTUN   =0x08000000,
	FLG_TCINTERP=0x01000000, FLG_TCDELTA=0x02000000,
	FLG_INFO    =0x01000000, FLG_XFER   =0x02000000;

  public final static int 
	FLG_SHARC=0x01000000, FLG_IOC=0x02000000, FLG_PM=0x03000000,
	FLG_FC	 =0x04000000, FLG_PPC=0x05000000, FLG_FLASH=0x06000000,
	FLG_MOD  =0x08000000, FLG_STRATIX=0x09000000, FLG_VIRTEX=0x0A000000;

  public final static int 
	FLG_UFILT=0x0001, FLG_UCFIR=0x0002, FLG_NCFIR=0x0004, FLG_PFIR4=0x0008, 
	FLG_AOVSR=0x0010, FLG_POVSR=0x0020, FLG_ITDEC=0x0040, FLG_RESAMP=0x0080,
	FLG_DSYNC=0x0100, FLG_TALT1=0x0200, FLG_FSYNC=0x0400, FLG_GSYNC=0x0800,
	FLG_TBANK=0x1000, FLG_URFIR=0x2000, FLG_PRESAMP=0x4000, FLG_FIRONLY=0x8000,
	FLG_ITCPC=0x10000, FLG_ITFMT=0x20000, FLG_PMWBR=0x40000, FLG_PMWBT=0x80000;
  public final static String tflagList = 
	"UFilt,UCFir,NCFir,PFir4,AOVSR,POVSR,ITDec,Resamp,DSync,TAlt1,FSync,GSync,TBank,URFir,PreSamp,FirOnly,ITCPC,ITFmt,PMWBR,PMWBT";

  public final static int
	FLG_PPC_MEM=0x0000, FLG_PPC_BUS=0x0001, FLG_CTL_BUS=0x0002;

  public final static String drvFlgList = 
	"RefX,PPSX,IrbX,,,,,,Ref1,PPS1,,,Ref2,PPS2";

  public final static String rfOptsList = 
	"Enable,XRef,XADClk,Atten,LNA,DCS,AIS,AGC,BBAmp,Mute";

  public final static int 
	KEY_RATE=1, KEY_FREQ=2, KEY_DEC=3, KEY_GAIN=4, KEY_OVSR=5, 
	KEY_FRAME=6, KEY_ALG=7, KEY_FEED=8, KEY_TCOFF=9, KEY_FLAG=10, 
	KEY_SWAP=11, KEY_CHAIN=12, KEY_RATIO=13, KEY_MCFG=14, KEY_BREAK=15,
	KEY_CHAN=16, KEY_ACTIVE=17, KEY_DELAY=18, KEY_PKTLEN=19, KEY_DFREQ=20, KEY_FRAMEDEC=28,
	KEY_PRCCLK=21, KEY_IOCCLK=22, KEY_ICLK=23, KEY_CCLK=24, KEY_MCLK=25, KEY_PCLK=26, KEY_TUNCLK=27, KEY_DRATE=29,
	KEY_CLKI=31, KEY_MSBI=32, KEY_BIT=33, KEY_BITS=34, KEY_CLKDLY=35, KEY_ATCCALIB=36, 
	KEY_NODMA=37, KEY_TODO=38, KEY_GAP=39,
	KEY_FLGBSET=41, KEY_FLGBCLR=42, KEY_TFLGBSET=43, KEY_TFLGBCLR=44, KEY_STIME=47,
	KEY_NFREQ=51, KEY_NDEC=52, KEY_HUNG=53, KEY_CRC32=54, 
	KEY_SEQERR=55, KEY_SEQFILL=56, KEY_PKTHDR=57, KEY_NRATIO=58, KEY_INBYTE=59, KEY_INBYTE_W64=60,
	KEY_RFFREQ=61, KEY_RFGAIN=62, KEY_RFBW=63, KEY_RFOPTS=64, KEY_RFATTN=65,
	KEY_RFPWR=66, KEY_RFAIS=67, KEY_RFCAIS=68, KEY_TCXO=69, KEY_AGCTC=70, KEY_MCORES=79,
	KEY_AGCCF=71, KEY_MGAIN=72, KEY_MFREQ=73, KEY_NTPOFF=74, KEY_GPS=75, KEY_SYSMON=76, 
	KEY_GPSMODE=77, KEY_LOG=78, KEY_NFGAIN=81, KEY_NYBW=82, KEY_FBWF=83,
	KEY_CTYPE=101, KEY_PTYPE=102, KEY_PINDEX=103, KEY_MTYPE=104, KEY_MREV=105, KEY_TEMP=106,
	KEY_CHNS=107, KEY_CPC=108, KEY_STATUS=109, KEY_VERSION=110, KEY_MCHNS=111, KEY_CBUFSZ=112,
	KEY_IOCTYPE=113, KEY_TINC=114, KEY_DRIVER=115, KEY_TCMODE=116,
	KEY_IDCODE=117, KEY_UIDCODE=118, KEY_USER1=119, KEY_USER2=120, KEY_PMSTAT=121, KEY_BSCAN=122,
	KEY_PMTYPE=125, KEY_PMCODE=126, KEY_CONFIG=127, KEY_DETECT=128, KEY_QALLOC=129, 
	KEY_VERBOSE=130, KEY_RSTIME=131, KEY_TPOE=132, KEY_ROUTE=133, KEY_TIMEOUT=134, KEY_PMHBT=135, 
	KEY_GCHIP=136, KEY_JTAGPORT=137, KEY_FPGAREV=138, KEY_FPGASIG=139, 
	KEY_TPD=140, KEY_TPDX=141, KEY_RMPL=142, KEY_WMPL=143, KEY_ROUTF=144, KEY_DMACHK=145, KEY_DMAOFF=146, KEY_STATE=147, 
	KEY_DUMP=148, KEY_MBITS=149, KEY_TPSR=150, KEY_MPCIO=151, KEY_CUID=152, KEY_EMT=153, KEY_EYESCAN=154, KEY_ADLM=155, KEY_NAME=156,
	KEY_HWREV=157, KEY_EVCNT=158, KEY_RTCLK=159, KEY_MGTDUMP=160, KEY_DRVFLG=161, KEY_DCSBN=162, KEY_CLKSTAT=163, 
	KEY_MTOFF=164, KEY_ENABLE=165, KEY_SPEEDS=166, KEY_FPGADSG=167, KEY_ROUTIOS=168, KEY_DSGTYPE=169, 
	KEY_NVME=170, KEY_NIOP=171, KEY_NVMS=172, KEY_NIOS=173, KEY_VCTL=174, KEY_NPM=175,
	KEY_MTYPE1=191, KEY_MTYPE2=192, KEY_MREV1=193, KEY_MREV2=194,
	KEY_PMTYPE1=195, KEY_PMTYPE2=196, KEY_PM1CODE=197, KEY_PM2CODE=198, KEY_PMINDEX=199,
	KEY_PCICLK=201, KEY_PCIBUS=202, KEY_PCIBW=203, KEY_PCIREV=204, KEY_PFIFO=205, 
	KEY_AFIFO=206, KEY_PCINACK=207, KEY_PCIGEN=208, KEY_PCILANES=209,
	KEY_IPADDR=401, KEY_IPCONN=402, KEY_IPDISC=403, KEY_IPVLAN=404, KEY_IPSOCK=405, KEY_PING=408,
	KEY_IOC=0x1000, KEY_IOCIOM=0x1000, KEY_IOCALG=0x1010, KEY_IOCRAM=0x1020,
	KEY_MOD=0x2000, KEY_APP=0x4000, KEY_FLASH=0x8000, KEY_CORE=0x01000000, KEY_MCORE=0x02000000, 
	KEY_ARGS=300, KEY_TABLE=350, KEY_TRACE=360, KEY_CNAME=370, KEY_QSTAT=380, KEY_NULL=-1;

  public final static int QUERY_FPLAN=1;

  protected static String[] keys = {"ACTIVE","CTYPE","VERSION","RATE","FREQ","DEC","GAIN"};

  public final static int 
	ALG_NULL=0, ALG_NOOP=1, ALG_USER=2, ALG_SWAP=3, ALG_LUT=4, 
	ALG_AM=5, ALG_FM=6, ALG_PM=7, ALG_PSK=8, ALG_QPSK=9, ALG_TTDM=10;
  public final static String algList = "NOOP,USER,SWAP,LUT,AM,FM,PM,PSK,QPSK";

  public final static int
	TCM_OFF=0, TCM_CPU=1, TCM_ZTC=2, TCM_SDN=3, TCM_SMS=4, 
	TCM_DTL=5, TCM_IRB=6, TCM_SDDS=7, TCM_ICE=8, TCM_VRT=9, TCM_FILE=10, TCM_STC=12;
  public final static String tcmodeList = "CPU,ZTC,SDN,SMS,DTL,IRB,SDDS,ICE,VRT,FILE,,STC";

  public final static int
	TC_OKNC=2,TC_OK=1, TC_INACTIVE=-1, TC_BADPARAM=-2, TC_NOBITS=-3, 
	TC_NOBARKER=-4, TC_BCDERR=-5, TC_NAN=-6, TC_NOCLOCK=-7;
  public final static String tcerrList = "InActive,BadParam,NoBits,NoBarker,BCDErr,NAN,NoClock";

  public final static int 	// Bit Reverse of Vita CIF 0 mask 
	VCTX_CHX=0, VCTX_SEC=1, VCTX_BW=2, VCTX_IF=3, VCTX_RF=4, VCTX_REFL=7, VCTX_GAIN=8, VCTX_SR=10, VCTX_TEMP=13, 
	VCTX_FMT=16, VCTX_GPS=17, VCTX_DSEC=20, VCTX_HSEC=21, VCTX_MSEC=22, VCTX_NOCID=23, 
	VCTX_XSR=24, VCTX_XIF=25, VCTX_XRF=26, VCTX_XGAIN=27;
  public final static String vctxList = "OnChx,OnSec,BW,IF,RF,,,REFL,GAIN,,SR,,,TEMP,,,FMT,GPS,,---,dSec,cSec,mSec,NoCID,xSR,xIF,xRF,xGAIN";

  public final static int
	BUF_MASK=0x1, BUF_EXPAND=0x2; 

  public final static int
	PKT_RMIF=0x00100000, PKT_ACMD=0x00200000; 

  public final static int
	PKTF_BUS_RD=1, PKTF_BUS_WR=2, PKTF_MEM_RD=3, PKTF_MEM_WR=4, PKTF_MEM_TEST=5,
	PKTF_INIT=6, PKTF_MODIFY=7, PKTF_RUN=8, PKTF_RESET=9;

  public final static String mcfgXList = 
	"ENA,LEN,TEN,SEN,PLY,COE,MCOE,MCEN,8BIT,1BIT,BIT1,BIT4,LSBX,UOPT,CLKI,MSBI,"+
	"SGO,TGO,GGO,XGO,DUAL,MUX,VHS,ALT,LP1,LP2,LP3,SP1,IOM1,IOM2,?,?";

  public final static String mcfgYList = 
	"ENA,LEN,TEN,SEN,PLY,COE,MCOE,MCEN,8BIT,1BIT,BIT1,BIT4,LSBX,UOPT,CLKI,MSBI,"+
	"SGO,TGO,GGO,XGO,IQ,MUX,VHS,ALT,LP1,LP2,FLIP,CLKM,IOM1,IOM2,?,BMUX";

  public final static String routeListMB = 
	"CA2HA,CB2HA,CA2HB,CB2HB,HA2CA,HB2CA,HA2CB,HB2CB,"+
	"CA2TB,CA2TA,CB2TB,CB2TA,HA2TA,HB2TA,HA2TB,HB2TB,"+
	"NP2R,HP2R,HA2R,HB2R,CA2R,CB2R,TA2R,TB2R,"+
	"R2NP,R2HP,R2HA,R2HB,R2CA,R2CB,R2TA,R2TB";
  public final static String routeListMBP = 
	"CA2HA,CB2HA,CA2HB,CB2HB,HA2CA,HB2CA,HA2CB,HB2CB,"+
	"TA2HA,CA2TA,CB2TB,TB2HB,HA2TA,HB2TA,HA2TB,HB2TB,"+
	"NP2R,HP2R,HA2R,HB2R,CA2R,CB2R,TA2R,TB2R,"+
	"R2NP,R2HP,R2HA,R2HB,R2CA,R2CB,R2TA,R2TB";
  public final static String routeListPM = 
	"CA2HA,TA2CA,TB2CB,CB2HB,HA2CA,HB2CA,HA2CB,HB2CB,"+
	"TA2HA,CA2TA,CB2TB,TB2HB,HA2TA,HB2TA,HA2TB,HB2TB,"+
	"NP2R,TC2R,HA2R,HB2R,CA2R,CB2R,TA2R,TB2R,"+
	"R2NP,R2TC,R2HA,R2HB,R2CA,R2CB,R2TA,R2TB";
  public final static String routfList = 
	"TMuxA,TMuxB,TMuxC,TxOut,HPFDP,HXFDP,CXFDP,TXFDP,"+
	"CA2CB,CB2CA,TA2TB,TB2TA,TX2HX,CX2HX,TX2CX,CXAOB,"+
	"HXAOB,TXAOB,TA2HX,TB2HX,DIO2P,P2DIO,CXMIX,TXTAG,"+
	"NIO2M,M2NIO,NV2M,M2NV,NIO2HX,HX2NIO,NV2HX,HX2NV";
  public final static String iostatList = 
	"iPR,iHP,iHA,iHB,iCA,iCB,iTA,iTB,iNIO,iDIO,iHA1,iHB2,iTA1,iTA2,iTB1,iTB2,"+
	"oPR,oHP,oHA,oHB,oCA,oCB,oTA,oTB,oNIO,oDIO,oHA1,oHB2,oTA1,oTA2,oTB1,oTB2";

  // default do nothing constructor 
  public DevIce() {
  }

  // load the native libraries 
  public static boolean load() {
    if (!useNative) try { System.loadLibrary("DevIce"); useNative=true; }
    catch (Exception e) { throw new RuntimeException (e); }
    return useNative;
  }

  // detect any cards on the system 
  public static int detect() {
    load();
    return detect(0);
  }
  public native static int detect (int flags);

  public native int open (String config, int flags);
  public native int close ();
  public native int reset (int flags);
  public native int sniff (int flags);
  public native int test (int test, int flags);
  public native int spec (DmaMap map, DmaSpec spec, int flags);
  public native int read (int offset);
  public native int write (int offset, int value);
  public native int writem (int offset, int value, int mask);
  public native int name2offset (String name, int flags);
  public native int name2key (String name);

  public native int sendrecv (int node, byte[] head, byte[] data, int bytes, int maxbytes, int flags, int timeout);
  public native int send (int node, byte[] head, byte[] data, int bytes, int flags);
  public native int recv (int node, byte[] head, byte[] data, int bytes, int flags);

  public native byte getKeyType (int key);
  public native int getKeySize (int key);
  public native int setKey (int dmac, int key, byte[] buf, int len);
  public native int getKey (int dmac, int key, byte[] buf, int len);
  public native int getKeyV (int dmac, int key);
  public native int getKeyL (int dmac, int key);
  public native int getKeyL (int dmac, int key, int data);
  public native int setKeyL (int dmac, int key, int data);
  public native double getKeyD (int dmac, int key);
  public native double getKeyD (int dmac, int key, double data);
  public native int    setKeyD (int dmac, int key, double data);

  public native int getIntFlagDef (String flagname, int def);
  public native double getDblFlagDef (String flagname, double def);
  public native int getRawFlag (String flagname, byte[] buffer, int len);

  public native int wpm (int node, int offset, int value);
  public native int rpm (int node, int offset);
  public native int wpb (int node, int offset, int value);
  public native int rpb (int node, int offset);
  public native int wpc (int node, int offset, int value, int size);
  public native int rpc (int node, int offset, int size);

  public native int ioPort (int ptype, int pindex, int dmac, int dir, 
	int bits, int rate, double fcny, int dec, int gain, int flags);

  public native int mapMem (DmaMap map, int psize, int mode);
  public native int dmaSetup (int dmac, int dir, DmaMap map, int nblock, int flags);
  public native int dmaChain (int dmac, int index, DmaMap map, int todo, int next);
  public native int dmaFunc (int dmac, int func);
  public native long dmaFuncX (int dmac, int func);
  public native int dmaStat (int dmac, DmaStat dmastat);
  public native int dmaDump (int dmac, int flags);
  public native int loadFile (String name, int flags);
  public native int loadFC (int port, byte[] buf, int size, int flags);
  public native int timer (int port, int rate);
  public native int nvRam (String fname, int mode);
  public native int tc (int dmac, double offset, double delta, TimeCode tc, int flags);
  public native int tcInsert (DmaMap map, TimeCode tc, int flags);
  public native int buffer (DmaMap map, int offset, byte[] buf, int length, int mode);
  public native int str2ip (String str);
  public native int setMapData (long vaddr, int moff, byte[] buf, int boff, int bytes);
  public native int getMapData (long vaddr, int moff, byte[] buf, int boff, int bytes);

  public native int compress (byte[] ibuf, int ilen, byte[] obuf, int olen, int dir);
  public native int cvt12 (long vaddr, byte[] data, int bytes, int dir);
  public native int query (int type, int info, byte[] buf, int len);

  public DmaMap mapMem (int psize) {
    DmaMap map = new DmaMap();
    map.mode = DmaMap.MEM;
    map.bytes = phys2long(psize);
    if (mapMem(map,psize,1)<0) return null;
    map.dev = this;
    return map;
  }

  public DmaMap getMap() {
    return new DmaMap();
  }

  public static class DmaStat { 
    public int index, cycle; 
  }

  public class DmaMap { 
    public final static long PGSZ=8192, PGMSK=PGSZ-1;
    public final static int MEM=1, RAMD=2, DISK=3, COPY=4;
    public final static int NOCHK=10;
    public long vaddr=0; 
    public long paddr=0;
    public long offset=0;
    public long bytes=0;
    public long vbytes=0;
    public int mode=0;
    public boolean paged;
    public byte[] buf; // scratch buffer
    public String devn;
    protected DevIce dev;

    public void close() { close(0); }
    public void close (int flag) { 
      int psize = long2phys(bytes);
      if (mode==RAMD) {
        if (vbytes>0) { dev.mapMem(this,0,paged? -4 : -3); vbytes=0; }
        if (flag!=NOCHK) dev.mapMem(this,0,0); 
      }
      else dev.mapMem(this,psize,-1); 
    }
    public String toString() { 
      return "DmaMap bytes="+bytes+" paddr="+paddr+" vaddr="+vaddr; 
    }
    public void setBufferSize (int bytes) {
      buf = new byte[bytes];
    }
    public int setBuffer (int moff, int bytes) {
      long addr = getVirtualAddress(moff,bytes);
      return dev.setMapData (addr, 0, buf, 0, bytes);
    }
    public int getBuffer (int moff, int bytes) {
      long addr = getVirtualAddress(moff,bytes);
      return dev.getMapData (addr, 0, buf, 0, bytes);
    }
    public void setOffset (int offset) {
      this.offset = offset;
    }
    public DmaMap getSubMap (int chan, int multi) {
      DmaMap map = new DmaMap();
      if (multi<=0) multi=1;
      map.bytes = bytes/multi;
      map.paddr = paddr;
      map.vaddr = vaddr;
      map.offset = map.bytes*chan;
      map.dev = dev;
      map.devn = devn;
      map.paged = paged;
      map.mode  = mode;
      return map;
    }
    public void copyTo (DmaMap map) {
      map.dev   = this.dev;
      map.mode  = this.mode;
      map.paddr = this.paddr;
      map.vaddr = this.vaddr;
      map.bytes = this.bytes;
      map.offset = this.offset;
      map.vbytes = this.vbytes;
    }
    public long getVirtualAddress (long off, long len) {
      if (vbytes==bytes) return vaddr+off;	// fully mapped - not paged
      if (off<offset || off+len>offset+vbytes) {
        if (paged) { 
          if (vbytes>0) dev.mapMem(this,0,-4); 
          offset = off & ~(MAXALLOC-1);
          vbytes = (Math.min(MAXALLOC+len,bytes-offset)+PGMSK) & ~PGMSK;
          dev.mapMem(this,0,4); 
        } else {
          vbytes = bytes;
          dev.mapMem(this,0,3);
        }
      }
      return vaddr+(off-offset);
    }
    public void setPaged (boolean onoff) {
      paged = onoff;
    }
  }

  public void setMaxAlloc (long maxalloc) {
    MAXALLOC = maxalloc;
  }

  public static class DmaSpec { 
    public int block, total, sjump, bjump; 
    public float srate, brate;
    public String toString() {
      return "Spec Total="+total+" SRate="+srate+" BRate="+brate;
    }
  }

  public static class TimeCode { 
    public double offset, delta, wsec, fsec; 
    public double getSeconds() { return wsec+fsec; }
    public void update (TimeCode tc) { 
      offset=tc.offset;
      delta=tc.delta;
      wsec=tc.wsec;
      fsec=tc.fsec;
    }
    public void computeMetrics (TimeCode tc, double defDelta) { 
      double tdiff = diff(tc);
      double sdiff = offset - tc.offset;
      if (tc.offset>=0 && tdiff>0 && sdiff>0) delta = (tdiff/sdiff);
//      else delta = -1;
    }
    public double diff (TimeCode tc) { 
      double diff = (wsec-tc.wsec);
      diff += (fsec-tc.fsec);
      return diff;
    }
    public double err (TimeCode tc) { 
      double diff = diff(tc);
      diff -= (offset-tc.offset)*delta;
      return diff;
    }
    public void initMetrics() {
      offset = -1; delta = -1; wsec=0; fsec=0;
    }
    public String toString() {
      return "TC Offset="+offset+" WSec="+wsec+" FSec="+fsec+" Delta="+delta;
    }
  }

  public class IoPort { 
    public int type, index, dmac, dir, flags, pad;
    public double rate, fcny, dec, gain;
  }

  protected int long2phys (long l) {
    if ((l>>32)!=0) l = ((l>>5)&0xFFFFFFFE) | 0x1;
    else l = (l&0xFFFFFFFC);
    return (int)l;
  }
  protected long phys2long (int i) {
    long l=i;
         if ((i&0x01)!=0) l = (l&0xFFFFFFFE)<<5;
    else if ((i&0x10)!=0) l = (l&0xFFFFFFFC)<<10;
    else                  l = (l&0xFFFFFFFC)<<0;
    return l;
  }
  protected int int2phys (int i) {
    i = (i&0xFFFFFFFC);
    return i;
  }
  protected int phys2int (int i) {
    if ((i&0x1)!=0) i=-1;      // error code 
    return i;
  }
  protected int phys2word (int i) {
         if ((i&0x01)!=0) i = (i<<3)&0xFFFFFFF0;
    else if ((i&0x10)!=0) i = (i<<8)&0xFFFFFC00;
    else i = (i>>2)&0x3FFFFFFF;
    return i;
  }
  protected int word2phys (int i) {
    if ((i&0xC0000000)!=0) i = ((i>>3)&0x1FFFFFFE) | 0x1;
    else i = i<<2;
    return i;
  }
  protected int phys2page (int i) {
         if ((i&0x01)!=0) i = (i<<5)&0xFFFFFFC0;
    else if ((i&0x10)!=0) i = (i<<0)&0xFFFFFFFC;
    else i = (i>>10)&0x003FFFFC;

    if ((i&0x1)!=0) i = (i<<8)&0xFFFFFE00;
    else i = (i>>8)&0x00FFFFFF;
    return i;
  }
  protected int page2phys (int i) {
    if ((i&0xFFC00000)!=0) i = (i&0xFFFFFFFC)|0x10;
    else i = i<<10;
    return i;
  }
  protected int long2word (long l) {
    l = (l>>2)&0xFFFFFFFF;
    return (int)l;
  }
  protected long word2long (int i) {
    long l=i; l = (l&0xFFFFFFFF)<<2;
    return l;
  }

  public void unpack12xxx (byte[] buf, int bytes) {
    int nout=bytes/4, nin=nout*3;
    for (int i=nin,j=nout; i>0;) {
      i-=3; j-=4;
      buf[j+0] = (byte)(buf[i+0]<<4);
      buf[j+1] = (byte)((buf[i+0]<<4)|((buf[i+1]>>4)&0xF));
      buf[j+2] = (byte)(buf[i+1]&0xF0);
      buf[j+3] = buf[i+2];
    }
  }

  public String getStrFlagDef (String flagname, String def) {
    int len=128; byte[] buf = new byte[len];
    int ls = getRawFlag(flagname,buf,len);
    if (ls<=0) return def;
    return new String(buf,0,ls);
  }

  public String getKeyS (int dmac, int key, int len) {
    byte[] buf = new byte[len];
    int ls = getKey (dmac, key, buf, len);
    while (ls>0 && buf[ls-1]<32) ls--;
    if (ls<=0) return null;
    return new String(buf,0,ls);
  }
  public int setKeyS (int dmac, int key, String string) {
    int len = string.length(); string += "0";
    byte[] buf = string.getBytes(); buf[len]=0;	// terminate
    return setKey (dmac,key,buf,len+1);
  }

}
