package nxm.ice.lib;

import nxm.sys.inc.DataTypes;
import nxm.sys.lib.Data;
import nxm.sys.lib.Shell;
import nxm.sys.lib.Convert;
import nxm.sys.lib.MidasException;

/**
  Structure for a Native Core 

  Native code libraries should have the following methods:
    alloc()  -  allocates the handle to an instance of this library
    setkey() -  sets a parameter
    getkey() -  gets a parameter
    init()   -  initialize the library for the current set of parameters
    work()   -  perform the desired function on a segment of data
    free()   -  release all resources

  @author Jeff Schoen
  @version $Id: CoreNative.java,v 1.3 2011/08/12 12:47:37 jgs Exp $
*/
public class CoreNative extends Core {

  protected long handle;
  protected String config;

  public native long alloc (); 
  public native int config (long handle, int key, int value);
  public native int init   (long handle);
  public native int setkey (long handle, String key, char ctype, byte[] value, int off, int len);
  public native int getkey (long handle, String key, char ctype, byte[] value, int off, int len);
  public native int work   (long handle, long si, long sip, long so, long sop);
  public native int free   (long handle);

  public native int sysrd  (long handle, int addr);
  public native int syswr  (long handle, int addr, int data);

  public static synchronized native int open   (long handle);
  public static synchronized native int poll   (long handle);
  public static synchronized native int upload (long handle);

  public static native long newStream (int size, int IBW, int OBW);
  public static native void freeStream (long handle);
  public static native long loadStreamBuffer (long handle);
  public static native int  loadStreamAvail  (long handle);
  public static native int  loadStreamStatus (long handle, int n);
  public static native long unloadStreamBuffer (long handle);
  public static native int  unloadStreamAvail (long handle);
  public static native int  unloadStreamStatus (long handle, int n);
  public static native int  xgetL (long handle, int off);
  public static native void xwrL (long handle, int x);
  public static native void fp2fptx (long lbuf, int bytes);
  public static native void fptx2fp (long lbuf, int bytes);

  public static boolean useNative = false;

  /** Create an instance of this function with native implementation */
  public CoreNative () {
    useNative |= loadNative("nxm.ice.lib.CoreNative");
    loadNative(getClass().getName());
  }

  /** load the native libraries */
  private static boolean loadNative (String libName) {
    boolean ok = Shell.loadLibrary(libName);
    if (!ok) throw new MidasException ("Problem with load of native library: "+libName);
    return ok;
  }

  public int setFormats (String fmt1, String fmt2) {
    return config(handle,FORM,getFormats(fmt1,fmt2));
  }
  public int setMCID (int index) {
    return config(handle,MCID,index);
  }
  public int setTraceMode (int mode) {
    return config(handle,FLGS,(mode<<24));
  }
  public int getTraceData (int addr) {
    return sysrd(handle,0xFFC);
  }

  public void allocNative (String config) {
    this.config = config;
    handle = alloc();
    if (handle==0) throw new MidasException ("Problem with alloc of native config="+config);
    config(handle,IMPL,halo.impl);
    config(handle,FLOW,halo.flow);
    config(handle,FLGS,halo.flgs);
    config(handle,MCID,halo.mcid);
    config(handle,VBPR,halo.vbpr);
    init(handle);
  }

  public synchronized int set (String key, Data value) {
    char ctype = (char)value.type; if (ctype=='A') ctype='S';
    int len = setkey (handle,key,ctype,value.buf,0,value.size*value.bpe);
    if (len>=0) return len;
    return super.set(key,value);
  }

  public synchronized Data get (String key, byte type) {
    byte[] buffer = new byte[256];
    char ctype = (char)type; 
    if (key.length()>2 && key.charAt(1)==':') { ctype=key.charAt(0); key=key.substring(2); }
    int len = getkey (handle,key,ctype,buffer,0,buffer.length);
    if (type==DataTypes.STRING) {
      type = DataTypes.ASCII;
      for (;len%8!=0;) buffer[len++]=0;
    }
    if (len>0) return new Data(buffer,0,len,type,1);
    return super.get(key,type);
  }

  public synchronized int get (String key, Data data) {
    char ctype = (char)data.type; 
    if (key.length()>2 && key.charAt(1)==':') { ctype=key.charAt(0); key=key.substring(2); }
    int len = getkey (handle,key,ctype,data.buf,0,data.buf.length);
    if (data.type==DataTypes.ASCII) {
      for (;len%8!=0;) data.buf[len++]=0;
    }
    if (len>0) return len;
    return super.get(key,data);
  }

  public synchronized int get (String key, Data data, int off, int len) {
    char ctype = (char)data.type; 
    if (key.length()>2 && key.charAt(1)==':') { ctype=key.charAt(0); key=key.substring(2); }
    len = getkey (handle,key,ctype,data.buf,off,len);
    if (data.type==DataTypes.ASCII) {
      for (;len%8!=0;) data.buf[len++]=0;
    }
    return len;
  }

  public int open() {
    return open(handle);
  }

  public synchronized int process (Stream so) {
    return work (handle, 0L,0L, so.handle,0L);
  }
  public synchronized int process (Stream si, Stream so) {
    return work (handle, si.handle,0L, so.handle,0L);
  }
  public synchronized int process (Buffer bi, Buffer bo) {
    return work (handle, bi.handle,0L, bo.handle,0L);
  }
  public synchronized int process (Buffer bi, Stream so) {
    return work (handle, bi.handle,0L, so.handle,0L);
  }
  public synchronized int process (Buffer bi, Stream sip, Stream so, Stream sop) {
    return work (handle, bi.handle,sip.handle, so.handle,sop.handle);
  }

  public int poll() {
    return poll(handle);
  }

  public int close() {
    return free(handle);
  }

}
