package nxm.ice.lib;

import nxm.sys.inc.DataTypes;
import nxm.sys.lib.Data;
import nxm.sys.lib.DataFile;
import nxm.sys.lib.Convert;
import nxm.sys.lib.MidasException;

/**
  Structure for a Code-Once-Run-Everywhere library function with stream IO, state without public class interface

  These functions have special "handle" strings to instruct the CORE pre-processor on how to construct verilog code for them.

  $  is replaced with the instance name and underscore
  $$ is replaced with the instance name
  %0 is replaced with the variable name assigned the return value
  %n is replaced with argument #n
  x=N sets delay for following xen assigns
  s=var synchronizes sequence on this command
  a=arg adds first additional argument
  b=arg adds second additional argument

  Arguments:
  i: input variable
  b: buffer specific
  c:
  d: dma specific
  s: state variable
  o: output vaviable

  @author Jeff Schoen
  @version $Id: CoreComponent.java,v 1.2 2011/05/24 13:20:30 jgs Exp $
*/
public class CoreComponent extends CoreCommon {

  // A Value is a variable storing one or more words of data
  public static class Value {

    public int bps,spa,bpa,boff,len;
    public byte mode,type;
    public byte[] buf;
    private int scalars;

    public Value (String format, int size, int BW) {
      this(format,size);
    }

    public Value (String format, int size) {
      mode = (byte)format.charAt(0);
      type = (byte)format.charAt(1);
      spa  = Data.getSPA(mode);
      bps  = Data.getBPS(type);
      bpa  = spa*bps;
      len = bpa*size;
      scalars = len/bps;
      buf = new byte[len];
    }

    public Value (byte[] buf, byte mode, byte type, int boff, int elem) {
      this.buf=buf; this.mode=mode; this.type=type; this.boff=boff;
      spa = Data.getSPA(mode);
      bps = Data.getBPS(type);
      bpa = spa*bps;
      len = bpa*elem;
      scalars = len/bps;
    }

    public void setSize (int size) {
      int nlen = size*bpa;
      if (nlen>len) System.out.println("Value setSize="+size+" with BPA="+bpa+" greater than allocated len="+len);
      len = nlen;
    }
    public void setBPA (int bpa) {
      if (bpa>this.bpa) System.out.println("Value setBPA="+bpa+" greater than allocated BPA="+this.bpa);
      this.bpa = bpa;
    }
 
    public void fromL (int value) { Convert.packL(buf,boff,value); }
    public void fromF (float value) { Convert.packF(buf,boff,value); }
    public void fromD (double value) { Convert.packD(buf,boff,value); }
    public void fromS (String value) { setS(value); }

    public int   toL () { return (type==LONG)? Convert.unpackL(buf,boff) : getxL(0); }
    public float toF () { return (type==FLOAT)? Convert.unpackF(buf,boff) : (float)getxD(0); }
    public double toD () { return (type==DOUBLE)? Convert.unpackD(buf,boff) : getxD(0); }
    public String toS() { 
      int alen=len;
      for (;alen>0 && buf[alen-1]<=32;alen--);
      return new String(buf,0,alen);
    }

    public String toString() { 
      if (type==STRING || type==ASCII) return toS();
      if (type==DOUBLE) return Convert.d2s(toD()); 
      if (type==FLOAT) return Convert.d2s(toF()); 
      return Convert.l2s(toL()); 
    }

    public void fromList (String list, int index) { 
      String s=getListItem(list,index,-2); len=s.length();
      Convert.packS(buf,0,len,s);
    }
    public int toItem (String list) { 
      String sitem = Convert.unpackS(buf,0,len);
      int item = findListItem(list,sitem,-2);
      if (item<-1) System.out.println("Item="+sitem+" not found in list="+list);
      return item;
    }
    public int toMask (String list) { 
      int mask=0;
      String sitem = Convert.unpackS(buf,0,len);
      if (sitem.equalsIgnoreCase("None")) return 0;
      if (sitem.equalsIgnoreCase("All")) return -1;
      for (int i=0; i<32; i++) {
        String s=getListItem(sitem,i,0); len=s.length();
	if (s==null || s.length()==0) continue;
	if (s.equals("Undefined")) break;
        int bit = findListItem(list,s,0);
	if (bit<0) System.out.println("Item="+s+" not found in list="+list);
	else mask |= (1<<bit);
      }
      return mask;
    }
    public int[] toLA() {
      int[] ia = new int[scalars];
      Convert.bb2ja (buf,0,type, ia,0,INT, scalars);
      return ia;
    }
    public float[] toFA() {
      float[] fa = new float[scalars];
      Convert.bb2ja (buf,0,type, fa,0,FLOAT, scalars);
      return fa;
    }
    public double[] toDA() {
      double[] da = new double[scalars];
      Convert.bb2ja (buf,0,type, da,0,DOUBLE, scalars);
      return da;
    }
    public double getxD (int off) {
      switch (type) {
      case BYTE: return (double)getB(off);
      case INT: return (double)getI(off);
      case LONG: return (double)getL(off);
      case FLOAT: return (double)getF(off);
      case DOUBLE: return getD(off);
      }
      return 0.0;
    }
    public int getxL (int off) {
      switch (type) {
      case BYTE: return (int)getB(off);
      case INT: return (int)getI(off);
      case LONG: return getL(off);
      case FLOAT: return (int)getF(off);
      case DOUBLE: return (int)getD(off);
      }
      return 0;
    }

    public void setS (String s) {
      type=STRING; bpa=spa=bps=1;
      buf = s.getBytes();
      len = buf.length;
    }

    public void setB (int off, byte value) { buf[off]=value; }
    public void setI (int off, short value) { off<<=1; Convert.packI(buf,off,value); }
    public void setL (int off, int value) { off<<=2; Convert.packL(buf,off,value); }
    public void setF (int off, float value) { off<<=2; Convert.packF(buf,off,value); }
    public void setD (int off, double value) { off<<=3; Convert.packD(buf,off,value); }

    public void setCF (int off, CxFloat value) { off<<=3; Convert.packF(buf,off,value.x); Convert.packF(buf,off+4,value.y); }
    public void setCFxy (int off, float x, float y) { off<<=3; Convert.packF(buf,off,x); Convert.packF(buf,off+4,y); }
    public void setCFIxy (int off, float x, float y) { off<<=2; Convert.packI(buf,off,(short)Math.rint(x*0x8000)); 
	    							Convert.packI(buf,off+2,(short)Math.rint(y*0x8000)); }
    public void setCFI (int off, CxFloat value) { off<<=2; Convert.packI(buf,off,(short)Math.rint(value.x*0x8000)); 
	    						   Convert.packI(buf,off+2,(short)Math.rint(value.y*0x8000)); }

    public void setCFIS2 (int shf, CxFloat value1, CxFloat value2) { int scl=(shf<0)? 0x8000>>-shf : 0x8000<<shf; 
	    Convert.packI(buf,0,(short)Math.rint(value1.x*scl)); Convert.packI(buf,2,(short)Math.rint(value1.y*scl)); 
	    Convert.packI(buf,4,(short)Math.rint(value2.x*scl)); Convert.packI(buf,6,(short)Math.rint(value2.y*scl)); }

    public byte  getB (int off) { return (type==BYTE)? buf[off] : (byte)getxL(off); }
    public short getI (int off) { return (type==INT)? Convert.unpackI(buf,off<<1) : (short)getxL(off); }
    public int   getL (int off) { return (type==LONG)? Convert.unpackL(buf,off<<2) : getxL(off); }
    public float getF (int off) { return (type==FLOAT)? Convert.unpackF(buf,off<<2) : (float)getxD(off); }
    public double getD (int off) { return (type==DOUBLE)? Convert.unpackD(buf,off<<3) : getxD(off); }

    public void  getCF (int off, CxFloat value) { off<<=3; value.x=Convert.unpackF(buf,off); value.y=Convert.unpackF(buf,off+4); }
    public float getCFx (int off) { off<<=3; return Convert.unpackF(buf,off); }
    public float getCFy (int off) { off<<=3; return Convert.unpackF(buf,off+4); }
    public void  getCIF (int off, CxFloat value) { off<<=2; value.x=Convert.unpackI(buf,off)/32768F; value.y=Convert.unpackI(buf,off+2)/32768F; }
    public float getCIFx (int off) { off<<=2; return Convert.unpackI(buf,off)/32768F; }
    public float getCIFy (int off) { off<<=2; return Convert.unpackI(buf,off+2)/32768F; }

    public char charAt (int off) { return (char)buf[off]; }

    public final static String handleArgs = "i:wen,i:8:woff,i:BPW:wbus,i:ren,i:8:roff,o:BPW:rbus";
    public final static String handleAuto = "getB=i,getI=i,getL=i,getF=i,getCF=io,getCIF=io,getCIFx=i,getCIFy=i,setB=ii,setI=ii,setL=ii,setF=ii,setCF=ii,setCFxy=iii,setCFI=ii,setCFIS2=iii";

    public final static String handleMethod_getB = "$ren=T,$roff=%1,x=1,%0=$rbus,o=1";
    public final static String handleMethod_getI = "$ren=T,$roff=%1,x=1,%0=$rbus,o=1";
    public final static String handleMethod_getL = "$ren=T,$roff=%1,x=1,%0=$rbus,o=1";
    public final static String handleMethod_getF = "$ren=T,$roff=%1,x=1,%0=$rbus,o=1";
    public final static String handleMethod_getCF = "$ren=T,$roff=%1,x=1,%2=$rbus,o=1";
    public final static String handleMethod_getCIF = "$ren=T,$roff=%1,x=1,%2=CI2CF($rbus),o=1";

    public final static String handleMethod_setB = "$wen=T,$woff=%1,$wbus=%2,x=0,o=1";
    public final static String handleMethod_setI = "$wen=T,$woff=%1,$wbus=%2,x=0,o=1";
    public final static String handleMethod_setL = "$wen=T,$woff=%1,$wbus=%2,x=0,o=1";
    public final static String handleMethod_setF = "$wen=T,$woff=%1,$wbus=%2,x=0,o=1";
    public final static String handleMethod_setCF = "$wen=T,$woff=%1,$wbus=%2,x=0,o=1";
    public final static String handleMethod_setCFxy = "$wen=T,$woff=%1,$wbus=muxf(%2,%3),x=0,o=1";
    public final static String handleMethod_setCFI = "$wen=T,$woff=%1,$wbus=CF2CI(%2),x=0,o=1";
    public final static String handleMethod_setCFIS2 = "$wen=T,$woff=0,$wbus=CF2CIS2(%1,%2,%3),x=0,o=1";
  }

  public interface ValueI {
    public final static String handleMethod_getCF = "$ren=T,$roff=%1,x=1,%0=$rbus,o=1";
    public final static String handleAuto = "getCF=io";
  }
  public interface ValueO {
    public final static String handleAuto = "setCF=ii,setCFxy=iii";
    public final static String handleMethod_setCF = "$wen=T,$woff=%1,$wbus=%2,x=0,o=1";
    public final static String handleMethod_setCFxy = "$wen=T,$woff=%1,$wbus=muxf(%2,%3),x=0,o=1";
  }
  public interface ValueIF {
    public final static String handleMethod_getCF = "$.ren=T,$.roff=%1,x=1,%0=$.rbus,o=1";
    public final static String handleMethod_setCF = "$.wen=T,$.woff=%1,$.wbus=%2,x=0,o=1";
    public final static String handleMethod_setCFxy = "$.wen=T,$.woff=%1,$.wbus=muxf(%2,%3),x=0,o=1";
  }

  public short  V2s (Value v) { return v.getI(0); }
  public int    V2i (Value v) { return v.getL(0); }
  public float  V2f (Value v) { return v.getF(0); }
  public double V2d (Value v) { return v.getD(0); }

  // A Stream is a sample continuous data source/sink
  public static class Stream {
    public int len,bib,woff,roff,wboff,rboff,wbpe,rbpe,wper,rper,mtl,mask,chn,cvt;
    public boolean wok,rok,useNative; 
    public byte[] b,c;
    public long handle;
    public CxFloat cxt;

    public Stream () {
      this(1024);
    }
    public Stream (int size) {
      this(size,32,32);
    }
    public Stream (int size, int IBW, int OBW) {
      this(size,IBW,OBW,false);
    }
    public Stream (int size, int IBW, int OBW, boolean useNative) {
      len = size;
      mtl = len/2;
      mask = len-1;
      wok = true;
      rper = rbpe = (IBW/8);
      wper = wbpe = (OBW/8);
      if (useNative) handle = CoreNative.newStream(size,IBW,OBW);
      else b = new byte[len]; 
      this.useNative = useNative;
    }
    public int load (DataFile df) {
      int n=0, avail = roff-woff;
      if (avail<0 || bib==0) avail = len-woff;
      if (avail>mtl) avail = mtl;
      if (cvt==1) {		// 8-bit
	n = df.read(b,woff,avail>>1)<<1;
	for (int ib=n>>1,ii=n; ib>0;) { ib--; ii--; 
	  b[woff+ii]=b[woff+ib]; ii--; b[woff+ii]=0;
	}
      }
      else if (cvt==2) {	// 12-bit
	int m = avail>>2;
	n = df.read(b,woff,m*3)/3*4;
	for (int i=0,ij=woff+m*3,ii=woff+m*4; i<m; i++) { ij-=3; ii-=4;
	  b[ii+0] = (byte)(b[ij+0]<<4);    b[ii+1] = (byte)(b[ij+1]<<4 | ((b[ij+0]>>4)&0xF) );
	  b[ii+2] = (byte)(b[ij+1]&0xF0);  b[ii+3] = b[ij+2];
	}
      }
      else n = df.read(b,woff,avail);
      if (n==0 && !df.isStream()) n = -1;
      if (n>0) update(n);
      return n;
    }
    public int unload (DataFile df) {
      int avail = woff-roff;
      if (avail<0 || bib==len) avail = len-roff;
      if (avail==0) return 0;
      int n = df.write(b,roff,avail);
      if (n>0) update(-n);
      return n;
    }
    public void update (int xfer) {
      bib += xfer;
      if (xfer>0) {
	if (c!=null) c[woff>>2]=(byte)chn;
        woff += xfer; 
        if (woff>=len) woff-=len;
      } else {
	if (dbgmod!=0) {
	  if (dbgcnt%dbgmod==0) System.out.printf("Stream cnt=%x value=%x\n",dbgcnt,Convert.unpackL(b,roff));
	  dbgcnt -= xfer;
	}
        roff += -xfer; 
        if (roff>=len) roff-=len;
      }
      rok = (bib>=rper);
      wok = (len-bib>=wper);
    }
    public void rd (Value val) {
      System.arraycopy(b,roff, val.buf,0, val.len);
      update(-val.len);
    }
    public void rdx (Value val, int n) {
      int bpe = n*val.bps*val.spa;
      System.arraycopy(b,roff, val.buf,0, bpe); 
      update(-bpe);
    }
    public float rdF() {
      float x=Convert.unpackF(b,roff);
      update(-4);
      return x;
    }
    public int rdL() {
      int x=Convert.unpackL(b,roff);
      update(-4);
      return x;
    }
    public int rdI() {
      short x=Convert.unpackI(b,roff);
      update(-2);
      return x;
    }
    public byte rdB() {
      byte x = b[roff];
      update(-1);
      return x;
    }
    public int rdBits (int n) {
      int x=0;
      for (int i=0; i<n; i++) {
	if ((b[roff]&(1<<rboff))!=0) x |= (1<<i);
	if (++rboff==8) { update(-1); rboff=0; }
      }
      return x;
    }
    public void rdCI (CxShort val) {
      val.x=Convert.unpackI(b,roff+0);
      val.y=Convert.unpackI(b,roff+2);
      update(-4);
    }
    public void rdCF (CxFloat val) {
      val.x=Convert.unpackF(b,roff+0);
      val.y=Convert.unpackF(b,roff+4);
      update(-8);
    }
    public void rdCIF (CxFloat val) {
      val.x=Convert.unpackI(b,roff+0)/32768F;
      val.y=Convert.unpackI(b,roff+2)/32768F;
      update(-4);
    }
    public void rdCIFA (CxFloat[] val, int n) {
      for (int i=0; i<n; i++) rdCIF(val[i]);
    }
    public void rdCIFx (Value val, int n) {
      if (cxt==null) cxt = new CxFloat();
      for (int i=0; i<n; i++) { rdCIF(cxt); val.setCF(i,cxt); }
    }

    public void wr (Value val) {
      System.arraycopy(val.buf,0, b,woff, val.len);
      update(val.len);
    }
    public void wrx (Value val, int n) {
      if (n==0) return;
      int bpe = n*val.bps*val.spa;
      System.arraycopy(val.buf,0, b,woff, bpe);
      update(bpe);
    }
    public void wrX (long x) {
      Convert.packX(b,woff,x);
      update(8);
    }
    public void wrF (float x) {
      Convert.packF(b,woff,x);
      update(4);
    }
    public void wrL (int x) {
      Convert.packL(b,woff,x);
      update(4);
    }
    public void wrI (short x) {
      Convert.packI(b,woff,x);
      update(2);
    }
    public void wrB (byte x) {
      b[woff]=x;
      update(1);
    }
    public void wrLCS (int x, int chn, int swaps) {
      if (c==null) c = new byte[len>>2];
      this.chn = (byte)chn;
           if (swaps==1) x = bswap2s(x);
      else if (swaps==2) x = bswap4(x);
      Convert.packL(b,woff,x);
      update(4);
    }
    public void wrBits (int n, int x) {
      for (int i=0; i<n; x>>=1,i++) {
	if ((x&1)==0) b[woff] &= ~(1<<wboff);
	else          b[woff] |=  (1<<wboff);
	if (++wboff==8) { update(1); wboff=0; }
      }
    }
    public void wrCI (CxShort val) {
      Convert.packI(b,woff+0,val.x);
      Convert.packI(b,woff+2,val.y);
      update(4);
    }
    public void wrCF (CxFloat val) {
      Convert.packF(b,woff+0,val.x);
      Convert.packF(b,woff+4,val.y);
      update(8);
    }
    public void wrCFx (CxFloat val, int n) {
      if (n<=0) return;
      Convert.packF(b,woff+0,val.x);
      Convert.packF(b,woff+4,val.y);
      update(8);
    }
    public void wrCFI (CxFloat val) {
      Convert.packI(b,woff+0,(short)Math.round(val.x*32768F));
      Convert.packI(b,woff+2,(short)Math.round(val.y*32768F));
      update(4);
    }
    public void wrCFIA (CxFloat[] val, int n) {
      for (int i=0; i<n; i++) wrCFI(val[i]);
    }
    public void wrCFIx (Value val, int n) {
      if (cxt==null) cxt = new CxFloat();
      for (int i=0; i<n; i++) { val.getCF(i,cxt); wrCFI(cxt); }
    }
    public void wrCFIS (CxFloat val, int shf) {
      int scl = 0x8000<<shf;
      Convert.packI(b,woff+0,(short)Math.round(val.x*scl));
      Convert.packI(b,woff+2,(short)Math.round(val.y*scl));
      update(4);
    }
    public void wrCFISA (CxFloat[] val, int shf, int n) {
      for (int i=0; i<n; i++) wrCFIS(val[i],shf);
    }
    public void wrCFB (CxFloat val) {
      Convert.packB(b,woff+0,(byte)Math.round(val.x*128F));
      Convert.packB(b,woff+1,(byte)Math.round(val.y*128F));
      update(2);
    }
    public void wrCFBA (CxFloat[] val, int n) {
      for (int i=0; i<n; i++) wrCFB(val[i]);
    }
    public void wrCFxy (float x, float y) {
      Convert.packF(b,woff+0,x);
      Convert.packF(b,woff+4,y);
      update(8);
    }
    public void wrCIxy (int x, int y) {
      Convert.packI(b,woff+0,(short)x);
      Convert.packI(b,woff+2,(short)y);
      update(4);
    }

    public void setC (int chn) { 
      if (c==null) c = new byte[len>>2];
      this.chn=chn;
    }
    public int getC () { 
      return (c!=null)? c[roff>>2] : this.chn;
    }

    public int  xgetL (int ioff) { 
      if (useNative) return CoreNative.xgetL(handle,ioff);
      else return Convert.unpackL(b,(woff+ioff)&mask);
    }
    public void xwrL (int x) {
      if (useNative) CoreNative.xwrL(handle,x);
      else wrL(x);
    }

    private int dbgmod,dbgcnt;

    public void setDebug (int cnt) {
      dbgmod = cnt;
      dbgcnt=0;
    }

    public String toString() { return "Stream len="+len+" woff="+woff+" wok="+wok+" roff="+roff+" rok="+rok; }
    public final static String handleArgs = "b:swaps,b:wok,b:wen,b:#2/8:wsel,b:#2:wbus,b:rok,b:ren,b:#3/8:rsel,b:#3:rbus";
    public final static String handleMethod_rd = "$ren=T,x=2,%1_wen==xn,%1_wbus==$rbus,o=2";
    public final static String handleMethod_wr = "x=0,$wen=T,%1_ren==$wen,$wbus==%1_rbus";
  }

  public interface StreamI {
    public final static String handleArgs = "o:rok,i:ren,o:BWP:rsel,o:BW:rbus";
    public final static String handleAuto = "rd=o,rdx=oi,rdCI=o,rdCF=o,rdCIF=o,rdCIFA=oi,rdCIFx=oi,rdBits=i,rdB=,rdL=,rdF=,getC=";
    public final static String handleMethod_rd = "$ren=T,x=2,%1_wen==xn,%1_wbus==$rbus,o=2";
    public final static String handleMethod_rdx = "$ren=T,x=2,%1_wen==xn,%1_wbus==$rbus,o=2";
    public final static String handleMethod_rdCI = "$ren=T,x=2,%1=$rbus,o=3";
    public final static String handleMethod_rdCF = "$ren=T,x=2,%1=$rbus,o=3";
    public final static String handleMethod_rdCIF = "$ren=T,x=2,%1=CI2CF($rbus),o=3";
    public final static String handleMethod_rdCIFA = "$ren=T,x=2,t=CIB2CFA_task(%2)($rbus,%1),o=3";
    public final static String handleMethod_rdCIFx = "$ren=T,x=2,t=CIB2CFx_task(%2)($rbus,%1_wbus),o=3";
    public final static String handleMethod_rdBits = "$ren=T,$bits=%1,x=2,%0=$rbus,o=3";
    public final static String handleMethod_rdB = "$ren=T,x=2,%0=$rbus,o=3";
    public final static String handleMethod_rdI = "$ren=T,x=2,%0=$rbus,o=3";
    public final static String handleMethod_rdL = "$ren=T,x=2,%0=$rbus,o=3";
    public final static String handleMethod_rdF = "$ren=T,x=2,%0=$rbus,o=3";
    public final static String handleMethod_getC = "x=0,%0=$rsel";
  }

  public interface StreamO {
    public final static String handleArgs = "o:wok,i:wen,i:BWP:wsel,i:BW:wbus";
    public final static String handleAuto = "wr=i,wrx=ii,wrCI=i,wrCF=i,wrCFx=ii,wrCFB=i,wrCFBA=ii,wrCFI=i,wrCFIA=ii,wrCFIAX=iii,wrCFIx=ii,wrCFIS=ii,wrCFISA=iii,wrCFxy=ii,wrCIxy=ii,wrBits=ii,wrB=i,wrL=i,wrLCS=iii,wrX=i,wrF=i,setC=i";
    public final static String handleMethod_wr = "x=0,$wen=T,%1_ren==$wen,$wbus==%1_rbus";
    public final static String handleMethod_wrx = "x=0,$wen=%2,%1_ren==$wen,$wbus==%1_rbus";
    public final static String handleMethod_wrCI = "x=0,$wen=T,$wbus=%1";
    public final static String handleMethod_wrCF = "x=0,$wen=T,$wbus=%1";
    public final static String handleMethod_wrCFx = "x=0,$wen=%2,$wbus=%1";
    public final static String handleMethod_wrCFB = "x=0,$wen=T,$wbus=CF2CB(%1)";
    public final static String handleMethod_wrCFBA = "x=0,$wen=T,t=CFA2CBB_task(%2)(%1,$wbus)";
    public final static String handleMethod_wrCFI = "x=0,$wen=T,$wbus=CF2CI(%1)";
    public final static String handleMethod_wrCFIA = "x=0,$wen=T,t=CFA2CIB_task(%2)(%1,$wbus)";
    public final static String handleMethod_wrCFIAX = "x=0,$wen=T,t=CFA2CIB_task(%2,%3)(%1,$wbus)";
    public final static String handleMethod_wrCFIx = "x=0,$wen=T,t=CFx2CIB_task(%2)(%1_rbus,$wbus)";
    public final static String handleMethod_wrCFIS = "x=0,$wen=T,$wbus=CF2CIS(%1,%2)";
    public final static String handleMethod_wrCFISA = "x=0,$wen=T,t=CFA2CIBS_task(%3)(%1,%2,$wbus)";
    public final static String handleMethod_wrCFxy = "x=0,$wen=T,$wbus=muxf(%1,%2)";
    public final static String handleMethod_wrCIxy = "x=0,$wen=T,$wbus=muxs(%1,%2)";
    public final static String handleMethod_wrBits = "x=0,$wen=T,$bits=%1,$wbus=%2";
    public final static String handleMethod_wrB = "x=0,$wen=T,$wbus=%1";
    public final static String handleMethod_wrI = "x=0,$wen=T,$wbus=%1";
    public final static String handleMethod_wrL = "x=0,$wen=T,$wbus=%1";
    public final static String handleMethod_wrF = "x=0,$wen=T,$wbus=%1";
    public final static String handleMethod_wrX = "x=0,$wen=T,$wbus=%1";
    public final static String handleMethod_wrLCS = "x=0,$wen=T,$wbus=%1,$wsel=%2,$swaps=%3";	// write Long with Channel and Swaps
    public final static String handleMethod_setC = "x=0,$wsel=%1";
  }

  //
  // A Buffer is a block continuous data source/sink
  public static class Buffer extends Stream {
    public Buffer (int size, int IBW, int OBW, boolean useNative) { super(size,IBW,OBW,useNative); }
    public void setBPE (int n) { rbpe=n; wbpe=n; update(0); }
    public void rd (int ioff, Value val) { System.arraycopy(b,roff+ioff, val.buf,0, val.len); }
    public void wr (int ioff, Value val) { System.arraycopy(val.buf,0, b,woff+ioff, val.len); }
    public byte getB (int ioff) { return Convert.unpackB(b,(roff+ioff)&mask); }
    public void setB (int ioff, byte val) { Convert.packB(b,(woff+ioff)&mask,val); }
    public int  getL (int ioff) { return Convert.unpackL(b,(roff+ioff)&mask); }
    public void setL (int ioff, int val) { Convert.packL(b,(woff+ioff)&mask,val); }
    public float getF(int ioff) { return Convert.unpackF(b,(roff+ioff)&mask); }
    public void setF (int ioff, float val) { Convert.packF(b,(woff+ioff)&mask,val); }
    public long getX (int ioff) { return (((long)Convert.unpackL(b,(roff+ioff+4)&mask))<<32) + Convert.unpackL(b,(roff+ioff)&mask); }
    public void setX (int ioff, long val) { Convert.packX(b,(woff+ioff)&mask,val); }
    public void rokLen (int len) { rper=len; }
    public void wokLen (int len) { wper=len; }
    public void rnxt (int n) { update(-n); }
    public void wnxt (int n) { update(n); }
    public String toString() { return "Buffer len="+len+" woff="+woff+" wok="+wok+" roff="+roff+" rok="+rok; }
  }

  public interface BufferI {
    public final static String handleArgs = "o:rok,i:rd,d:32:dma,i:nxt,i:AW:adr,o:AW:len,o:BW:bus,o:8:busb";
    public final static String handleAuto = "rd=io,rnxt=i,getX=i,getF=i,getL=i,getB=i,setC=i,rptr=i,setBPE=i,rokLen=i";
    public final static String handleMethod_rd = "$rd=T,$adr=%1,x=2,%2.bus=$bus,o=3";
    public final static String handleMethod_rnxt = "$nxt=T,$adr=%1";
    public final static String handleMethod_getX = "$rd=T,$adr=%1,x=2,%0=$bus,o=3";
    public final static String handleMethod_getL = "$rd=T,$adr=%1,x=2,%0=$bus,o=3";
    public final static String handleMethod_getF = "$rd=T,$adr=%1,x=2,%0=$bus,o=3";
    public final static String handleMethod_getB = "$rd=T,$adr=%1,x=2,%0=$busb,o=3";
    public final static String handleMethod_setC = "x=0,$sel=%1";
    public final static String handleMethod_setBPE = "$nxt=F,$adr=%1";
    public final static String handleMethod_rokLen = "$len=%1";
  }

  public interface BufferO {
    public final static String handleArgs = "o:wok,i:wr,c:32:dma,i:nxt,i:AW:adr,o:AW:len,i:BW:bus";
    public final static String handleAuto = "wr=ii,wnxt=i,setX=ii,setF=ii,setL=ii,setB=ii,setC=i,wptr=i,setBPE=i,wokLen=i";
    public final static String handleMethod_wr = "x=0,$wr=T,$adr=%1,$bus=%2.bus";
    public final static String handleMethod_wnxt = "$nxt=T,$adr=%1";
    public final static String handleMethod_setX = "x=0,$wr=T,$adr=%1,$bus=%2";
    public final static String handleMethod_setF = "x=0,$wr=T,$adr=%1,$bus=%2";
    public final static String handleMethod_setL = "x=0,$wr=T,$adr=%1,$bus=%2";
    public final static String handleMethod_setB = "x=0,$wr=T,$adr=%1,$bus=%2";
    public final static String handleMethod_setC = "x=0,$sel=%1";
    public final static String handleMethod_setBPE = "$nxt=F,$adr=%1";
    public final static String handleMethod_wokLen = "$len=%1";
  }

  // A Queue is a small fast FIFO
  public static class Queue {
    public int len,roff,woff,bpe;
    public int shft,mask;
    public int afull,aempty;
    public boolean wok,rok;
    public byte[] b;

    public Queue (int size, int bw, int AFULL, int AEMPTY) {
      bpe = (bw+7)/8;
      len = size*bpe;
      mask = len-1;
      for (shft=0; (1<<shft)<bpe; shft++);
      woff = roff = 0;
      rok = false;
      wok = true;
      afull = AFULL;
      aempty = AEMPTY;
      b = new byte[len];
    }

    public void update (int rw) {
      if (rw>0) woff=(woff+bpe)&mask;
      if (rw<0) roff=(roff+bpe)&mask;
      int nin = ((woff-roff)&mask)>>shft;
      wok = nin<=afull;
      rok = nin>aempty;
    }
    public boolean rok() { return rok; }
    public boolean wok() { return wok; }

    public void wr (Value v) { System.arraycopy(v.buf,0, b,woff, bpe); update(1); }
    public void wrB (int v)      { b[woff]=(byte)v; update(1); }
    public void wrI (int v)      { Convert.packI(b,woff,(short)v); update(1); }
    public void wrL (int v)      { Convert.packL(b,woff,v); update(1); }
    public void wrF (float v)    { Convert.packF(b,woff,v); update(1); }
    public void wrD (double v)   { Convert.packD(b,woff,v); update(1); }
    public void wrCF (CxFloat v) { Convert.packF(b,woff,v.x); Convert.packF(b,woff+4,v.y); update(1); }

    public void   rd (Value v) { System.arraycopy(b,roff, v.buf,0, bpe); update(-1); }
    public byte   rdB() { byte   v=b[roff]; update(-1); return v; }
    public short  rdI() { short  v=Convert.unpackI(b,roff); update(-1); return v; }
    public int    rdL() { int    v=Convert.unpackL(b,roff); update(-1); return v; }
    public float  rdF() { float  v=Convert.unpackF(b,roff); update(-1); return v; }
    public double rdD() { double v=Convert.unpackD(b,roff); update(-1); return v; }
    public void   rdCF (CxFloat v) { v.x=Convert.unpackF(b,roff); v.y=Convert.unpackF(b,roff+4); update(-1); }

    public String toString() { return "Queue len="+len+" woff="+woff+" wok="+wok+" roff="+roff+" rok="+rok; }
    public final static String handleArgs = "o:wok,i:wen,i:BPW:wbus,o:rok,i:ren,o:BPW:rbus";
    public final static String handleAuto = "rd=o,rdB,rdI,rdL,rdF,rdD,rdCF=o,rok,wr=i,wrB=i,wrI=i,wrL=i,wrF=i,wrD=i,wrCF=i,wok";
    public final static String handleMethod_wr  = "x=0,$wen==T,$wbus==%1_rbus";
    public final static String handleMethod_wrB = "x=0,$wen==T,$wbus==%1";
    public final static String handleMethod_wrI = "x=0,$wen==T,$wbus==%1";
    public final static String handleMethod_wrL = "x=0,$wen==T,$wbus==%1";
    public final static String handleMethod_wrF = "x=0,$wen==T,$wbus==%1";
    public final static String handleMethod_wrD = "x=0,$wen==T,$wbus==%1";
    public final static String handleMethod_wrCF= "x=0,$wen==T,$wbus==%1";
    public final static String handleMethod_rd  = "$ren==T,x=0,%1_wen=T,%1_wbus==$rbus,o=0";
    public final static String handleMethod_rdB = "$ren==T,x=0,%0==$rbus,o=0";
    public final static String handleMethod_rdI = "$ren==T,x=0,%0==$rbus,o=0";
    public final static String handleMethod_rdL = "$ren==T,x=0,%0==$rbus,o=0";
    public final static String handleMethod_rdF = "$ren==T,x=0,%0==$rbus,o=0";
    public final static String handleMethod_rdD = "$ren==T,x=0,%0==$rbus,o=0";
    public final static String handleMethod_rdCF= "$ren==T,x=0,%1==$rbus,o=0";
    public final static String handleMethod_wok = "x=0,%0==$wok,o=0";
    public final static String handleMethod_rok = "x=0,%0==$rok,o=0";
  }

  // A RamB is a Block Ram in the FPGA
  public static class Ram {
    public byte[] b;
    public int OL;
    public int WA,WB;
    private int AL,ML;
    private int len;
    public Ram (int orderLength, int widthA, int widthB) {
      OL = orderLength;  AL=1<<OL; ML=AL-1;
      WA = widthA;
      WB = widthB;
      len = 1<<OL;
      b = new byte[len];
    }
    public void rd (int off, Value val) {
      System.arraycopy(b,off, val.buf,0, val.len);
    }
    public void wr (int off, Value val) {
      System.arraycopy(val.buf,0, b,off, val.len);
    }
    public byte rdB (int off) {
      off &= ML; return b[off];
    }
    public short rdI (int off) {
      off &= ML; return Convert.unpackI(b,off);
    }
    public int rdL (int off) {
      off &= ML; return Convert.unpackL(b,off);
    }
    public float rdF (int off) {
      off &= ML; return Convert.unpackF(b,off);
    }
    public void wrB (int off, int val) {
      off &= ML; b[off]=(byte)val;
    }
    public void wrI (int off, short val) {
      off &= ML; Convert.packI(b,off,val);
    }
    public void wrL (int off, int val) {
      off &= ML; Convert.packL(b,off,val);
    }
    public void wrF (int off, float val) {
      off &= ML; Convert.packF(b,off,val);
    }
    public String toString() { return "Ram len="+(1<<OL)+" widthA="+WA+" widthB="+WB; }

    public final static String handleArgs = "i:wr,c:32:wdma,i:#1:wadr,i:#2:wbus,NL2,"+
					    "i:rd,d:32:rdma,i:#1:radr,o:#3:rbus,o:8:rbusb";
    public final static String handleAuto = "rd=io,wr=ii,rdB=i,wrB=ii,rdI=i,wrI=ii,rdL=i,wrL=ii,rdF=i,wrF=ii,wrX=ii";
    public final static String handleMethod_wr = "x=0,$wr=T,$wadr=%1,$wbus=%2";
    public final static String handleMethod_rd = "$rd=T,$radr=%1,x=2,%2=$rbus,o=3";
    public final static String handleMethod_wrB = "x=0,$wr=T,$wadr=%1,$wbus=%2";
    public final static String handleMethod_rdB = "$rd=T,$radr=%1,x=2,%0=$rbusb,o=3";
    public final static String handleMethod_wrI = "x=0,$wr=T,$wadr=%1,$wbus=%2";
    public final static String handleMethod_rdI = "$rd=T,$radr=%1,x=2,%0=$rbus,o=3";
    public final static String handleMethod_wrL = "x=0,$wr=T,$wadr=%1,$wbus=%2";
    public final static String handleMethod_rdL = "$rd=T,$radr=%1,x=2,%0=$rbus,o=3";
    public final static String handleMethod_wrF = "x=0,$wr=T,$wadr=%1,$wbus=%2";
    public final static String handleMethod_rdF = "$rd=T,$radr=%1,x=2,%0=$rbus,o=3";
    public final static String handleMethod_wrX = "$wr=T,$wadr=%1,$wbus=%2,x=1,$wr=T,$wadr=%1+4,$wbus=%2>>32";
  }


  // A RamRWC is a Channelized simultaneous Read/Write block ram
  public static class RamRWC {
    public byte[] b;
    public int WA,WB;
    public int OL,OC;
    private int AL,AC,ML,MC;
    private int chn,len;
    public RamRWC (int orderLength, int widthA, int widthB, int orderChannel) {
      WA = widthA;
      WB = widthB;
      OL = orderLength;  AL=1<<OL; ML=AL-1;
      OC = orderChannel; AC=1<<OC; MC=AC-1;
      len = 1<<(OL+OC);
      b = new byte[len];
      chn = 0;
    }
    public void setC (int chn) {
      this.chn = chn;
    }
    public void rd (int off, Value val) {
      System.arraycopy(b,(chn<<OL)|off, val.buf,0, val.len);
    }
    public void wr (int off, Value val) {
      System.arraycopy(val.buf,0, b,(chn<<OL)|off, val.len);
    }
    public byte rdB (int off) {
      off &= ML; return b[(chn<<OL)|off];
    }
    public float rdF (int off) {
      off &= ML; return Convert.unpackF(b,(chn<<OL)|off);
    }
    public void wrB (int off, int val) {
      off &= ML; b[(chn<<OL)|off]=(byte)val;
    }
    public void wrF (int off, float val) {
      off &= ML; Convert.packF(b,(chn<<OL)|off,val);
    }
    public String toString() { return "Ram len="+(1<<OL)+" chn="+(1<<OC); }

    public final static String handleArgs = "i:wr,c:32:wdma,i:#4:wchn,i:#1:wadr,i:#2:wbus,NL2,"+
					    "i:rd,d:32:rdma,i:#4:rchn,i:#1:radr,o:#3:rbus,o:8:rbusb";
    public final static String handleAuto = "rd=io,wr=ii,rdB=i,wrB=ii,rdF=i,wrF=ii,setC=i";
    public final static String handleMethod_setC = "x=0,$rchn=%1,$wchn=%1";
    public final static String handleMethod_wr = "x=0,$wr=T,$wadr=%1,$wbus=%2";
    public final static String handleMethod_rd = "$rd=T,$radr=%1,x=2,%2=$rbus,o=3";
    public final static String handleMethod_wrB = "x=0,$wr=T,$wadr=%1,$wbus=%2";
    public final static String handleMethod_rdB = "$rd=T,$radr=%1,x=2,%0=$rbusb,o=3";
    public final static String handleMethod_wrF = "x=0,$wr=T,$wadr=%1,$wbus=%2";
    public final static String handleMethod_rdF = "$rd=T,$radr=%1,x=2,%0=$rbus,o=3";
  }


  // DMA engines - Rams autowrap to implement circular buffers, Buffers do not
  public static class Dma {
    public static int SWP2=0x1, SWP4=0x2, C12T16=0x4, C16T12=0x8;
    public int flags;
    public Dma (int AW, int BW) {
    }
    private void ba2ba (byte[] bi, int offi, int leni,  byte[] bo, int offo, int leno, int len) {
      int maski=leni-1, masko=leno-1;
      boolean swap = (flags&SWP2)!=0;
      while (len>0) {
        int ido = len;
        offi &= maski;
        offo &= masko;
	if (offi+ido>leni) ido = leni-offi;
	if (offo+ido>leno) ido = leno-offo;
        System.arraycopy(bi,offi, bo,offo, ido);
        if (swap) Convert.swap2(bo,offo,ido>>1);
	offi += ido;
	offo += ido;
        len  -= ido;
      }
    }
    private void ba2bs (byte[] bi, int offi, int leni,  byte[] bo, int offo, int leno, int len, Stream so) {
      int i, maski=leni-1, masko=leno-1;
      offi &= maski;
      offo &= masko;
      if ((flags&SWP2)!=0) {
        for (i=0; i<len; i+=2) {
	  bo[offo] = bi[offi+1];
	  bo[offo+1] = bi[offi];
          offi += 2; offi &= maski;
          offo += 2; offo &= masko;
	}
      }
      else if ((flags&C16T12)!=0) {
        for (i=0; i<len; i+=4) {
	  bo[offo++] = (byte)(((bi[offi+1]<<4)&0xF0) | ((bi[offi+0]>>4)&0x0F)); offo &= masko;
	  bo[offo++] = (byte)(( bi[offi+2]    &0xF0) | ((bi[offi+1]>>4)&0x0F)); offo &= masko;
	  bo[offo++] =  bi[offi+3]; 						offo &= masko;
          offi += 4; offi &= maski;
	}
	len = (len/4)*3;
      }
      else if ((flags&C12T16)!=0) {
        byte b0,b1,b2;
        for (i=0; i<len; i+=3) {
	  b0=bi[offi++]; offi &= maski;
	  b1=bi[offi++]; offi &= maski;
	  b2=bi[offi++]; offi &= maski;
	  bo[offo++] = (byte)(b0<<4);
	  bo[offo++] = (byte)(((b1<<4)&0xF0)|((b0>>4)&0x0F));
	  bo[offo++] = (byte)(b1&0xF0); 
	  bo[offo++] = b2;
          offo &= masko;
	}
	len = (len/3)*4;
      }
      else {
        for (i=0; i<len; i+=1) {
	  bo[offo++] = bi[offi++];
	  offi &= maski;
          offo &= masko;
	}
      }
      so.update(len);
    }
    public void r2b (RamRWC ri,int off,  Buffer bo,int boff, int len) {
      off &= ri.ML; int xlen = off+len-ri.AL; if (xlen>0) len-=xlen; else xlen=0;
      System.arraycopy(ri.b,(ri.chn<<ri.OL)|off, bo.b,bo.woff+boff, len);
      if (xlen>0) System.arraycopy(ri.b,(ri.chn<<ri.OL), bo.b,bo.woff+boff+len, xlen);
    }
    public void b2r (Buffer bi,int boff, RamRWC ro,int off, int len) {
      off &= ro.ML; int xlen = off+len-ro.AL; if (xlen>0) len-=xlen; else xlen=0;
      System.arraycopy(bi.b,bi.roff+boff, ro.b,(ro.chn<<ro.OL)|off, len);
      if (xlen>0) System.arraycopy(bi.b,bi.roff+boff+len, ro.b,(ro.chn<<ro.OL), xlen);
    }
    public void r2r (RamRWC ri,int offi,  RamRWC ro,int offo, int len) {
      ba2ba (ri.b,(ri.chn<<ri.OL)|offi,ri.len, ro.b,(ro.chn<<ro.OL)|offo,ro.len, len);
    }
    public void b2b (Buffer bi,int boffi, Buffer bo,int boffo, int len) {
      ba2ba (bi.b,bi.roff+boffi,bi.len, bo.b,bo.woff+boffo,bo.len, len);
    }
    public void b2s (Buffer bi,int boffi, Stream so,int soffo, int len) {
      ba2bs (bi.b,bi.roff+boffi,bi.len, so.b,so.woff,so.len, len, so);
    }
    public void r2s (RamRWC ri,int offi, Stream so,int soffo, int len) {
      ba2ba (ri.b,(ri.chn<<ri.OL)|offi,ri.len, so.b,so.woff,so.len, len);
    }
    public void setFlags (int flags) { 
      this.flags = flags;
    }
    public static void sync() { }
    public static void addReader (Object o) { }
    public final static String handleArgs = "i:start,i:#1:cnt,s:8:flags,o:shrd,o:busy,NL2,i:rok,i:#1:radr,i:#2:rbus,d:32:rdma,NL2,i:wok,i:#1:wadr,o:#2:wbus,o:32:wdma";
    public final static String handleAuto = "r2r=iioii,r2b=iioii,b2r=iioii,b2b=iioii,b2s=iioii,r2s=iioii,addReader=i,sync=,setFlags=i";
    public final static String handleMethod_r2r = "$start=T,$radr=%2,$wadr=%4,$cnt=%5,x=0,$rok=%1_rok,%1_rdma==$rdma,$rbus=%1_rbus,$wok=%3_wok,%3_wdma=$wdma,%3_wbus=$wbus,a=$busy";
    public final static String handleMethod_r2b = "$start=T,$radr=%2,$wadr=%4,$cnt=%5,x=0,$rok=%1_rok,%1_rdma==$rdma,$rbus=%1_rbus,$wok=%3_wok,%3_dma=$wdma,%3_bus=$wbus,a=$busy";
    public final static String handleMethod_b2r = "$start=T,$radr=%2,$wadr=%4,$cnt=%5,x=0,$rok=%1_rok,%1_dma==$rdma,$rbus=%1_bus,$wok=%3_wok,%3_wdma=$wdma,%3_wbus=$wbus,a=$busy";
    public final static String handleMethod_b2b = "$start=T,$radr=%2,$wadr=%4,$cnt=%5,x=0,$rok=%1_rok,%1_dma==$rdma,$rbus=%1_bus,$wok=%3_wok,%3_dma=$wdma,%3_bus=$wbus,a=$busy";
    public final static String handleMethod_b2s = "$start=T,$radr=%2,$wadr=%4,$cnt=%5,x=0,$rok=%1_rok,%1_dma==$rdma,$rbus=%1_bus,$wok=%3_wok,%3_wen=$wdma[31],%3_wbus=$wbus,a=$busy";
    public final static String handleMethod_r2s = "$start=T,$radr=%2,$wadr=%4,$cnt=%5,x=0,$rok=%1_rok,%1_rdma==$rdma,$rbus=%1_rbus,$wok=%3_wok,%3_wen=$wdma[31],%3_wbus=$wbus,a=$busy";
    public final static String handleMethod_addReader = "$shrd==%1_rd";
    public final static String handleMethod_sync = "s=1,a=$busy";
    public final static String handleMethod_setFlags = "$flags=%1";
  }

  // A RomF is a 1D array of floats implemented in 1<<M depth lut rams
  public static class RomF {
    int N,M,L;
    public float[] data;
    public RomF (int oM, int RP) {
      M = (1<<oM); L=M+2;
      data = new float[L];
    }
    public void clear () { int i; for (i=0; i<L; i++) data[i]=0; }
    public void set (int m, float d) { data[m] = d; }
    public float get (int m) { return data[m]; }
    public String toString() { return "RomF M="+M; }
    public final static String handleArgs = "sets,noinit,i:ren,i:#1:rM[#2],o:fptx:rbus[#2]";
    public final static String handleAuto = "get=i,set=ii";
    public final static String handleMethod_get = "$ren==T,$rM[IX]==%1,x=0,%0==$rbus[IX],o=1";
  }

  // A RomF2D is a 2D array of floats implemented in N x M depth block rams
  // ow is the output width for the HW implementation
  public static class RomF2D {
    int N,M,L;
    public float[] data;
    public RomF2D (int oN, int oM, int ow) {
      N = (1<<oN); M = (1<<oM); L=N*M+4;
      data = new float[L];
    }
    public void clear () { int i; for (i=0; i<L; i++) data[i]=0; }
    public void set (int n, int m, float d) { data[n+m*N] = d; }
    public float get (int n, int m) { return data[n+m*N]; }
    public String toString() { return "RomF2D N="+N+" M="+M; }
    public final static String handleArgs = "sets,noinit,i:ren,i:#1:rN,i:#2:rM[#3],o:fptx:rbus[#3]";
    public final static String handleAuto = "get=ii,set=iii";
    public final static String handleMethod_get = "$ren==T,$rN==%1,$rM[IX]==%2,x=0,%0==$rbus[IX],o=1";
  }

  // A RomF2D is a 2D array of floats implemented in N x M depth block rams with interpolation between taps
  // ow is the output width for the HW implementation
  public static class RomF2Di {
    int N,M,L,MC;
    public float[] data;
    public RomF2Di (int oN, int oM, int ow) {
      N = (1<<oN); M = (1<<oM); 
      L = N*M+1; MC = ow/2;
      data = new float[L+1];
    }
    public void clear () { int i; for (i=0; i<L; i++) data[i]=0; }
    public void set (int n, int m, float d) {
      int i=n+m*N;
      if (i>=L) { System.out.println("RomF2D set out of range: N="+N+" M="+M+" n="+n+" m="+m); return; }
      data[i] = d;
      if (++i==L) data[i] = 2*data[i-1] - data[i-2];	// replicate final slope
    }
    public float getrN (float n, int m) {
      float nN = n*N;
      int n0 = (int)nN;
      int mn = n0+m*N;
      if (mn>=L) return data[L];
      float p = nN-n0;
      float d0 = data[mn];
      float d1 = data[mn+1];
      return d0 + (d1-d0)*p;
    }
    public void getrNc (float n, int m, boolean cx, CxFloat c) {
      c.x = getrN (n,m);
      c.y = cx? getrN (n,m+MC) : c.x;
    }
    public String toString() { return "RomF2Di N="+N+" M="+M; }
    public final static String handleArgs = "sets,noinit,i:renr,i:fptx:rNr,i:#1:rN,i:#2:rM[#3],o:fptx:rbus[#3]";
    public final static String handleAuto = "getrN=ii,getrN_NP=iii,getrNc=iiio,set=iii";
    public final static String handleMethod_getrN = "$renr==T,$rNr==%1,$rM[IX]==%2,x=1,%0==$rbus[IX],o=2";
    public final static String handleMethod_getrN_NP = "$renr==T,$rNr[%3]==%1,$rM[IX]==%2,x=1,%0==$rbus[%3][IX],o=2";
    public final static String handleMethod_getrNc = "$renr==T,$rNr==%1,$rM[IX]==%2,$cx==%3,x=1,%4==$rbus[IX],o=3";
  }
  public static class RomF2Di_NP extends RomF2Di {
    public RomF2Di_NP (int oN, int oM, int ow, int nps, int nla) { super(oN,oM,ow); }
    public final static String handleArgs = "sets,noinit,i:renr,i:fptx:rNr[#4],i:#1:rN,i:#2:rM[#3],o:fptx:rbus[#4][#3]";
  }
  public static class RomF2Di_CX extends RomF2Di {
    public RomF2Di_CX (int oN, int oM, int ow) { super(oN,oM,ow); }
    public final static String handleArgs = "sets,noinit,i:renr,i:fptx:rNr,i:#1:rN,i:#2:rM[#3],i:cx,o:CxFptx:rbus[#3]";
  }


  // A Tap Buffer is an array of values shifted in from 0 to N so the Nth tap is earlier in time
  public static class TapBuf {
    int N,IW,n;
    public float[] data;
    public TapBuf (int ord, int iw, int ow, int dly) {
      N = (1<<ord);
      IW = iw;
      n = N+IW;
      data = new float[n];
    }
    public void setLength (int len) {
      n = imin(len,N);
      if (len>N) System.out.println ("TapBuffer limited from "+len+" to "+n+" taps");
      n = n+IW;
    }
    public void load (float v) {
      for (int i=n-1; i>0; i--) data[i]=data[i-1];
      data[0] = v;
    }
    public void loadA (float[] v, int nv) {
      for (int i=n-1; i>=nv; i--) data[i]=data[i-nv];
      for (int i=nv-1,j=0; i>=0; i--,j++) data[i]=v[j];
    }
    public float get (int i) {
      return data[i];
    }
    public String toString() { return "TapBufSF N="+N; }
    public final static String handleArgs = "noinit,i:load,i:fptx:wbus,i:ren[#3],i:#1:radr[#3],o:fptx:rbus[#3]";
    public final static String handleAuto = "load=i,loadA=ii,get=i";
    public final static String handleMethod_load = "$load==T,$wbus==%1";
    public final static String handleMethod_loadA = "$load==T,$wbus==%1";
    public final static String handleMethod_get = "$ren[IX]==T,$radr[IX]==%1,x=0,%0=$rbus[IX],o=1";
  }

  public static class CxTapBuf {
    int N,IW,n,NH;
    public float[] datax,datay;
    public CxTapBuf (int ord, int iw, int ow, int dly) {
      N = (1<<ord);
      NH = N>>1;
      IW = iw;
      n = N+IW;
      datax = new float[n];
      datay = new float[n];
    }
    public void setLength (int len) {
      n = imin(len,N);
      if (len>N) System.out.println ("CxTapBuffer limited from "+len+" to "+n+" taps");
      n = n+IW;
    }
    public void setStride (int len) {
      if (len>N) System.out.println ("CxTapBuffer stride limited from "+len+" to "+N+" taps");
    }
    public void load (CxFloat v) {
      for (int i=n-1; i>0; i--) { datax[i]=datax[i-1]; datay[i]=datay[i-1]; }
      datax[0]=v.x; datay[0]=v.y;
    }
    public void loadA (CxFloat[] v, int nv) {
      for (int i=n-1; i>=nv; i--) { datax[i]=datax[i-nv]; datay[i]=datay[i-nv]; }
      for (int i=nv-1,j=0; i>=0; i--,j++) { datax[i]=v[j].x; datay[i]=v[j].y; }
    }
    public void get (int i, CxFloat cf) {
      cf.x = datax[i];
      cf.y = datay[i];
    }
    public void get2 (int i, CxFloat cf, CxFloat df) {
      cf.x = datax[i];
      cf.y = datay[i];
      df.x = datax[i+1];
      df.y = datay[i+1];
    }
    public void getrN (float ifps, int i, CxFloat cf) {
      int off=(int)ifps; if (i>0) i=(i-off);
      cf.x = datax[i];
      cf.y = datay[i];
    }
    public void set (int i, CxFloat cf) {
      datax[i] = cf.x;
      datay[i] = cf.y;
    }
    public void setxy (int i, float x, float y) {
      datax[i] = x;
      datay[i] = y;
    }
    public void print() {
      float sumx=0, sumy=0;
      System.out.print("Taps ");
      for (int i=0; i<n-IW; i++) { System.out.printf("(%7.3f,%6.3f) ",datax[i],datay[i]); sumx+=datax[i]; sumy+=datay[i]; }
      System.out.printf(" [%d] (%7.3f,%6.3f)\n",n,sumx,sumy);
    }
    public void getSum (CxFloat v) {
      float sumx=0,sumy=0;
      for (int i=0; i<n-IW; i++) { sumx+=datax[i]; sumy+=datay[i]; }
      v.x=sumx; v.y=sumy;
    }
    public String toString() { return "TapBufCF N="+N; }
    public final static String handleArgs = "noinit,i:4:stride,i:split,i:load,i:CxFptx:wbus,i:ren[#3],i:#1:radr[#3],o:CxFptx:rbus[#3]";
    public final static String handleAuto = "load=i,loadA=ii,get=io,get2=ioo,get_vs=ioi,getrN_NPO=iioi,set=ii,setxy=iii";
    public final static String handleMethod_load = "$load==T,$wbus==%1";
    public final static String handleMethod_loadA = "$load==T,$wbus==%1";
    public final static String handleMethod_get = "$stride==#2,$ren[IX]==T,$radr[IX]==%1,x=0,%2=$rbus[IX],o=1";
    public final static String handleMethod_get2 = "$stride==#2,$ren[IX]==T,$radr[IX]==%1,x=0,%2=$rbus[IX],%3=$rbus2[IX],o=1";
    public final static String handleMethod_get_vs = "$stride==%3,$ren[IX]==T,$radr[IX]==%1,x=0,%2=$rbus[IX],o=1";
    public final static String handleMethod_getrN_NPO = "$ren[%4][IX]==T,$radr[%4][IX]==%2,$fps[%4]==%1,x=0,%3=$rbus[%4][IX],o=1";
  }

  public static class CxTapBuf_NP extends CxTapBuf {
    public CxTapBuf_NP (int ord, int iw, int ow, int dly, int nps) { super(ord,iw,ow,dly); }
    public final static String handleArgs = "noinit,i:4:stride,i:split,i:load,i:CxFptx:wbus[#5],i:ren[#3],i:#1:radr[#3],o:CxFptx:rbus[#3]";
  }
  public static class CxTapBuf_NP2 extends CxTapBuf {
    public CxTapBuf_NP2 (int ord, int iw, int ow, int dly, int nps) { super(ord,iw,ow,dly); }
    public final static String handleArgs = "noinit,i:4:stride,i:load,i:CxFptx:wbus[#5],i:ren[#3],i:#1:radr[#3],o:CxFptx:rbus[#3],o:CxFptx:rbus2[#3]";
  }

  public static class CxTapBuf_NPO extends CxTapBuf {
    public CxTapBuf_NPO (int ord, int iw, int ow, int dly, int nps) { super(ord,iw,ow,dly); }
    public final static String handleArgs = "noinit,i:load,i:CxFptx:wbus,i:fptx:fps[#5],i:ren[#5][#3],i:#1:radr[#5][#3],o:CxFptx:rbus[#5][#3]";
  }

  // A Stat Buffer is an array of values shifted in from 0 to N with statistics measurements
  public static class StatBuf {
    int N,n,m;
    public float[] data;

    public StatBuf (int ord) {
      N = (1<<ord);
      n = N;
      data = new float[N];
    }
    public void setLength (int len) {
      n = imin(len,N);
      m = n>>1;
      if (len>N) System.out.println ("Vector limited from "+len+" to "+n+" taps");
    }
    public void load (float v) {
      for (int i=n-1; i>0; i--) data[i]=data[i-1];
      data[0] = v;
    }
    public float getMid() {
      return data[m];
    }
    public float getMin() {
      float v = data[0];
      for (int i=1; i<n; i++) if (data[i]<v) v=data[i];
      return v;
    }
    public float getMax() {
      float v = data[0];
      for (int i=1; i<n; i++) if (data[i]>v) v=data[i];
      return v;
    }
    public float getMinM (int mode) {
      if (mode>0) return getMinFilt(1,m);
      if (mode<0) return getMinFilt(m,n-1);
      return fmaxf( getMinFilt(1,m), getMinFilt(m,n-1) );
    }
    public float getMaxM (int mode) {
      if (mode>0) return getMaxFilt(1,m);
      if (mode<0) return getMaxFilt(m,n-1);
      return fminf( getMaxFilt(1,m), getMaxFilt(m,n-1) );
    }
    public float getMinFilt (int n1, int n2) {
      float v=1e32F,vx; 
      for (int i=n1; i<n2; i++) {
        vx = data[i];
        if (data[i-1]>vx) vx=data[i-1];
        if (data[i+1]>vx) vx=data[i+1];
        if (vx<v) v=vx;
      }
      return v;
    }
    public float getMaxFilt (int n1, int n2) {
      float v=-1e32F,vx; 
      for (int i=n1; i<n2; i++) {
        vx = data[i];
        if (data[i-1]<vx) vx=data[i-1];
        if (data[i+1]<vx) vx=data[i+1];
        if (vx>v) v=vx;
      }
      return v;
    }
    public int getMinI (int mode) {
      int i=0,j=m;
      float v,vmin=data[j]; 
      for (i=1; i<m-1; i++,j+=mode) {
	v = data[j];
	if (v>vmin) break;
	vmin=v;
      }
      return i;
    }
    public String toString() { return "StatBuf N="+N; }
    public final static String handleArgs = "sets,noinit,i:load,i:fptx:vbus,i:intx:mode,i:domid,o:fptx:omid,i:domin,i:dominm,o:fptx:omin,i:domax,i:domaxm,o:fptx:omax,i:domini,o:intx:omini";
    public final static String handleAuto = "load=i,getMid=,getMin=,getMax=,getMinM=i,getMaxM=i,getMinI=i";
    public final static String handleMethod_load   = "$load==T,$vbus==%1,o=1";
    public final static String handleMethod_getMid = "$domid==T,x=0,%0=$omid,o=1";
    public final static String handleMethod_getMin = "$domin==T,x=0,%0=$omin,o=4";
    public final static String handleMethod_getMax = "$domax==T,x=0,%0=$omax,o=4";
    public final static String handleMethod_getMinM = "$dominm==T,$mode=%1,x=0,%0=$omin,o=4";
    public final static String handleMethod_getMaxM = "$domaxm==T,$mode=%1,x=0,%0=$omax,o=4";
    public final static String handleMethod_getMinI = "$domini==T,$mode=%1,x=0,%0=$omini,o=4";
  }

  // A Ram Buffer is a dual port random access array of values
  public static class RamBuf {
    int N,n;
    boolean rdOnly;
    public float[] data;
    public RamBuf (int ord, int iw, int ow) {
      N = (1<<ord);
      data = new float[N];
    }
    public void setLength (int len) {
      n = imin(len,N);
      if (len>N) System.out.println ("RamBuffer limited from "+len+" to "+n+" taps");
    }
    public void setRdOnly (boolean ro) {
      rdOnly = ro;
    }
    public void set (int i, float x) {
      if (rdOnly) return;
      data[i] = x;
    }
    public float get (int i) {
      return data[i];
    }
    public String toString() { return "RamBuf N="+N; }
    public final static String handleArgs = "sets,noinit,i:wen[#3],i:#1:wadr[#3],i:fptx:wbus[#2],i:ren[#3],i:#1:radr[#3],o:fptx:rbus[#3],i:rdonly";
    public final static String handleAuto = "set=ii,get=i";
    public final static String handleMethod_set = "$wen[IX]==T,$wadr[IX]==%1,wbus[IX]==%2";
    public final static String handleMethod_get = "$ren[IX]==T,$radr[IX]==%1,x=0,%0=$rbus[IX],o=1";
    public final static String handleMethod_setRdOnly = "x=0,$rdonly=%1";
  }

  public static class CxRamBuf {
    int N,n;
    int rdOnly;
    DataFile df;
    public float[] datax,datay;
    public CxRamBuf (int ord, int iw, int ow) {
      N = (1<<ord);
      datax = new float[N];
      datay = new float[N];
    }
    public void setLength (int len) {
      n = imin(len,N);
      if (len>N) System.out.println ("RamBuffer limited from "+len+" to "+n+" taps");
    }
    public void setRdOnly (int ro) {
      rdOnly = ro;
    }
    public void get (int i, CxFloat cf) {
      cf.x = datax[i];
      cf.y = datay[i];
    }
    public void set (int i, CxFloat cf) {
      if ((rdOnly&1)==0) datax[i] = cf.x;
      if ((rdOnly&2)==0) datay[i] = cf.y;
    }
    public void setxy (int i, float x, float y) {
      if ((rdOnly&1)==0) datax[i] = x;
      if ((rdOnly&2)==0) datay[i] = y;
    }
    public void print () {
      float sumx=0, sumy=0;
      System.out.print("Taps ");
      for (int i=0; i<n; i++) { System.out.printf("(%7.3f,%6.3f) ",datax[i],datay[i]); sumx+=datax[i]; sumy+=datay[i]; }
      System.out.printf(" [%d] (%7.3f,%6.3f)\n",n,sumx,sumy);
    }
    public void getSum (CxFloat v) {
      float sumx=0,sumy=0;
      for (int i=0; i<n; i++) { sumx+=datax[i]; sumy+=datay[i]; }
      v.x=sumx; v.y=sumy;
    }
    public void dump (String fname) {
      if (df==null) {
        df = new DataFile(null,fname,"1000","CF",0,128.0,null);
	df.setFS(n); df.setXUnits(1); df.setYUnits(1);
	df.open(df.OUTPUT);
      }
      byte[] buffer = new byte[8];
      for (int i=0; i<n; i++) {
        Convert.packF(buffer,0,datax[i]);
        Convert.packF(buffer,4,datay[i]);
        df.write(buffer,0,8);
      }
      df.flush();
    }
    public String toString() { return "RamBufCF N="+N; }
    public final static String handleArgs = "sets,noinit,i:wen[#3],i:#1:wadr[#3],i:CxFptx:wbus[#2],i:ren[#3],i:#1:radr[#3],o:CxFptx:rbus[#3],i:rdonly";
    public final static String handleAuto = "set=ii,get=io,setxy=iii,setRdOnly=i,setLength=i";
    public final static String handleMethod_set = "$wen[IX]==T,$wadr[IX]==%1,$wbus[IX]==%2";
    public final static String handleMethod_get = "$ren[IX]==T,$radr[IX]==%1,x=0,%2=$rbus[IX],o=1";
    public final static String handleMethod_setRdOnly = "x=0,$rdonly==%1";
  }

  // dummy process methods to be overridden in actual Cores
  public int process (Stream si, Stream so) {
    return -1;
  }

  public static class DelayLine {
    double[] dar;
    float[] far,fai;
    int n;
    public DelayLine (int size) {
      n = size;
      dar = new double[n+1];
      far = new float[n+1];
      fai = new float[n+1];
      System.out.println("New DelayLine size="+n);
    }
    public double loadD (double d) {
      for (int i=0; i<n; i++) dar[i]=dar[i+1];
      dar[n]=d;
      return dar[0];
    }
    public float loadF (float f) {
      for (int i=0; i<n; i++) far[i]=far[i+1];
      far[n]=f;
      return far[0];
    }
  }

  public static class Viterbi {
    public Viterbi () {
    }
    public int process (int bi, int pflg, int vflg) {
      return (0x8|bi);
    }
    public final static String handleArgs = "noinit,i:load,i:uint8:bi,i:uint4:pflg,i:uint8:vflg,o:uint4:bo";
    public final static String handleAuto = "process=iii";
    public final static String handleMethod_process = "$load==T,$bi==%1,$pflg==%2,$vflg==%3,x=0,%0=$bo,o=1";
  }

}
