package nxm.ice.lib;

import nxm.sys.lib.Args;
import nxm.sys.lib.Data;
import nxm.sys.lib.Time;
import nxm.sys.lib.Shell;
import nxm.sys.lib.Table;
import nxm.sys.lib.Parser;
import nxm.sys.lib.Convert;
import nxm.sys.lib.DataFile;
import nxm.sys.lib.MidasException;

/**
  Structure for a Code-Once-Run-Everywhere library function with Stream|Buffer IO, state, and public class interface

  These functions support implementations in more than one language.

  Core.init	- instantiate, allocate resources
  Core.set	- set key=value named parameters
  Core.get	- get key=value named parameters
  Core.open	- ready for processing
  Core.process	- process a buffer of data
  Core.close	- finish up processing

  @author Jeff Schoen
  @version $Id: Core.java,v 1.2 2011/05/24 13:20:30 jgs Exp $
*/
public class Core extends CoreComponent {

  public class HALO {
    // Original HW register map
    public int sys,dec,gain,rate,ratio,frame,freq,flag;
    public int user0,user1,user2,user3,user4,user5,user6,user7;
    // internals
    public int func,fmts,flow,impl;
    public int flgs,psize,status,scid;
    public int mcid,ncvar,name1,name2;
    public int state,vbpr;
    public boolean prep;
    public int loadOff;
    // next are function handles for C routines
    public long config,setkey,getkey,init,work,free,set,get,open,process,close;
  }
  public HALO halo = new HALO();
  public String name;

  // Create an instance of the function specified by name and parsing of the Args class 
  public static Core forName (String name, Args args) {
    Core c = CoreFactory.getCore(name,args);
    return c;
  }

  public Core () {
    setFlow(ARRAY);
  }
  public void initSub (int sc, HALO halo) {
    this.halo.scid=sc;
    this.halo.vbpr=halo.vbpr;
    this.halo.impl=halo.impl;
    if (sc>0) init();
  }

  // Initialize this engine with given config 
  public void allocNative (String config) {
    System.out.println("Err: Should not be calling allocNative from this implementation");
  }

  public Buffer getBuffer (int tl, int ibw, int obw) {
    boolean useNative = (halo.impl>CoreFactory.JVM);
    return new Buffer (tl*2,ibw*8,obw*8,useNative);
  }

  public Buffer getBuffer (int tl, String ifmt, String ofmt) {
    int cvt  = 0;
    int ibpa = Data.getBPA(ifmt);
    int obpa = Data.getBPA(ofmt);
    if (ifmt.equals(ofmt));
    else if (ifmt.equals("NH") || ofmt.equals("NH"));
    else if (ifmt.charAt(1)=='B' && ofmt.charAt(1)=='I') cvt=1;
    else if (ifmt.equals("CJ") && ofmt.equals("CI")) cvt=2;
    else if (ibpa==obpa);
    else System.out.println("Buffer format convert "+ifmt+" to "+ofmt+" not handled yet");
    Buffer b = getBuffer(tl,ibpa,obpa);
    b.cvt = cvt;
    return b;
  }

  public void setFlow (int flow) {
    halo.flow = flow;
  }

  // Initialize plan for parameter discovery 
  public int init() {
    return 0;
  }

  // Ready this engine with current parameters 
  public int open() {
    this.halo.state=1;
    return 0;
  }

  // Process a buffer of data 
  public int process() {
    System.out.println("This method process() should have been overridden"); Time.sleep(1.0);
    return -1;
  }
  public int process (Buffer bi, Buffer bo) {
    if (halo.flow==STREAM) return process((Stream)bi,(Stream)bo);
    System.out.println("This flow=BUFFER method process(bi,bo) should have been overridden"); Time.sleep(1.0);
    return -1;
  }
  public int process (Buffer bi, Stream so) {
    System.out.println("This flow=BUFSTR method process(bi,so) should have been overridden"); Time.sleep(1.0);
    return -1;
  }
  public int process (Stream si, Stream so) {
    System.out.println("This flow=STREAM method process(si,so) should have been overridden"); Time.sleep(1.0);
    return -1;
  }
  public int process (Buffer bi, Stream sip, Stream so, Stream sop) {
    System.out.println("This flow=PACKET method process(bi,sip,so,sop) should have been overridden"); Time.sleep(1.0);
    return -1;
  }


  public int poll() {
    return 0;
  }
  
  // Finalize this engine 
  public int close() {
    this.halo.state=0;
    return 0;
  }

  public boolean isMatch (String key, String name) {
    if (key.endsWith("?")) key = key.substring(0,key.length()-1);
    boolean kht = (key.length()>2) && (key.charAt(1)==':');
    boolean nht = (name.length()>2) && (name.charAt(1)==':');
    // all values default to numerical and all numerical types match 
    if (kht && !(nht && key.startsWith("S:"))) key=key.substring(2);
    if (nht && !(kht && name.startsWith("S:"))) name=name.substring(2);
    return name.equalsIgnoreCase(key);
  }
  public boolean isSubMatch (String key, String name) {
    int ip = key.indexOf('_');
    if (ip<=0) return false;
    key = key.substring(0,ip);
    return isMatch(key,name);
  }
  public String subKey (String key) {
    int ip = key.indexOf('_');
    if (ip<=0) return "";
    return key.substring(ip+1);
  }

  // SetKey for generic parameter. Override this method and call super if not handled  
  public int set (String key, Value value) {
    if (halo.scid>0) return -1;
    if (key.startsWith("NAME")||key.startsWith("S:NAME"));
    else if (key.equals("ICE.CONFIG")) System.out.println("Setting the key="+key+" to "+value);
    else throw new MidasException ("Problem setting key="+key+" to value="+value);
    return 0;
  }

  // SetKey for generic parameter. Override this method and call super if not handled  
  public synchronized int set (String key, Data data) {
    Value value = new Value(data.buf,data.mode,data.type,0,data.size);
    return set (key,value);
  }
  // SetKey helper for String parameter 
  public int set (String key, String value) {
    return set (key, new Data(value));
  }
  // SetKey helper for double parameter 
  public int set (String key, double value) {
    return set (key, new Data(value));
  }
  // SetKey helper for float parameter 
  public int set (String key, float value) {
    return set (key, new Data(value));
  }
  // SetKey helper for int parameter 
  public int set (String key, int value) {
    return set (key, new Data(value));
  }
  // SetKey helper for state parameter 
  public int set (String key, boolean value) {
    return set (key, new Data(value?1:0));
  }
  // SetKey helper for float array parameter 
  public int set (String key, float[] value) {
    return set (key, new Data(value));
  }
  // SetKey helper for int array parameter 
  public int set (String key, int[] value) {
    return set (key, new Data(value));
  }
  // SetKey helper for int array parameter 
  public int set (String key, short[] value) {
    return set (key, new Data(value));
  }
  // SetKey helper for int array parameter 
  public int set (String key, byte[] value) {
    return set (key, new Data(value));
  }

  // SetKey helper for a file of parameters
  public int setKeyFile (String fn, boolean verbose) {
    Table keys = Convert.o2t(fn,Shell.getMidasContext());
    return setKeyTable(null,keys,verbose);
  }

  // SetKey helper for a table of parameters
  public int setKeyTable (String root, Table keys, boolean verbose) {
    int n=0; 
    keys.setMode("KV");
    String[] list = keys.getKeys();
    for (int i=0; i<list.length; i++) {
      String key = list[i];
      Object value = keys.get(key);
      if (value instanceof Table) setKeyTable (key,(Table)value,verbose);
      else if (value==null || value.equals("") || value.equals("NULL")); // status only key
      else {
	Data d = keys.getData(key);
	String type = d.getFormat().substring(1,2);
	if (type.equals("A")) type="S";
	if (root!=null) key = root+"_"+key;
	if ("SLFD".indexOf(type)>=0) key = type+":"+key;
	if (verbose) System.out.println("Set Key="+key+" Value="+value);
        set (key,d); n++;
      }
    }
    return n;
  }

  // GetKey for generic parameter. Override this method and call super if not handled  
  public int get (String key, Value value) {
    value = null; int stat=0;
    if (halo.scid>0) return -1;
    if (key.startsWith("BOGUS"));
    else if (key.equals("NAME")||key.equals("S:NAME")) value.fromS(name);
    else if (key.endsWith("?")) stat=-1; 	// not required
    else if (key.equals("TIME")||key.equals("D:TIME")) value.fromD(Time.current());
    else if (key.equals("SSM") ||key.equals("F:SSM") ) value.fromF((float)Time.currentTime().getSoD());
    //else if (key.equals("IMPL")||key.equals("L:IMPL")) value.fromL(halo.impl);
    else { System.out.println ("Core get of key="+key+" not handled"); stat=-1; }
    return stat;
  }
  // GetKey for generic parameter. Override this method and call super if not handled  
  public int get (String key, Data data) {
    Value value = new Value(data.buf,data.mode,data.type,0,data.size);
    return get(key,value);
  }
  public int get (String key, Data data, int off, int len) {
    Value value = new Value(data.buf,data.mode,data.type,off,len);
    return get(key,value);
  }
  // GetKey for generic parameter. Override by CoreNative 
  public Data get (String key, byte type) {
    Data data = new Data();
    data.setType(type);
    data.setSize(1);
    int stat = get(key,data);
    return (stat<0)? null : data;
  }
  // GetKey of type specified in key name
  public Data get (String key) {
    byte type = LONG;
    if (key.indexOf(':')==1) type=(byte)key.charAt(0);
    return get (key,type);
  }

  // GetKey helper for String parameter 
  public String getS (String key) {
    Data value = get(key,STRING);
    return value.toString();
  }
  // GetKey helper for double parameter 
  public double getD (String key) {
    Data value = get(key,DOUBLE);
    return value.toD();
  }
  // GetKey helper for float parameter 
  public float getF (String key) {
    Data value = get(key,FLOAT);
    return value.toF();
  }
  // GetKey helper for int parameter 
  public int getL (String key) {
    Data value = get(key,LONG);
    return value.toL();
  }
  // GetKey helper for state parameter 
  public boolean getState (String key) {
    Data value = get(key,LONG);
    return value.toL()>0;
  }

  // Helper to get List of keys in one data block
  public int getList (String keys, Data data) {
    int off=0,len=0,i=0,j=0,ls=keys.length();
    if (keys.charAt(0)=='(') { i++; ls--; }
    for (; i<ls; i=j+1) {
      j=keys.indexOf(',',i); if (j<0) j=ls;
      String key = keys.substring(i,j);
      char type = (key.charAt(1)==':')? key.charAt(0) : 'F';
      int status = get(key,data,off,1);
      if (status>=0) off += Data.getBPS(type);
    }
    return off;
  }


  // Helper for testing options bits 
  public boolean isSet (String list, String key, int mask) {
    int kmask = Parser.mask(list,key,0);
    return ((mask&kmask) != 0);
  }

  // get the natural data format type 
  public byte getDataType() { 
    return CoreFactory.getDataType(halo.impl);
  }

  public static int getFormats (String formats) {
    char fm=formats.charAt(0), ft=formats.charAt(1);
    int fmt = 0;
         if (ft=='I') fmt = 0x0; 
    else if (ft=='B') fmt = 0x1;
    else if (ft=='N') fmt = 0x2;
    else if (ft=='P') fmt = 0x3;
    else if (ft=='L') fmt = 0x4;
    else if (ft=='X') fmt = 0x5;
    else if (ft=='F') fmt = 0x6; 
    else if (ft=='D') fmt = 0x7; 
    if      (fm=='C') fmt |= 0x8;
    return fmt;
  }

  public static int getFormats (String fmt1, String fmt2) {
    return (getFormats(fmt1)<<0) | (getFormats(fmt2)<<8);
  }
  public static int getFormats (String fmt1, String fmt2, String fmt3) {
    return (getFormats(fmt1)<<0) | (getFormats(fmt2)<<8) | (getFormats(fmt3)<<16);
  }

  public int getFormatBits (int fmts, int n) {
    int fmt = (fmts>>((n-1)*8)) & 0xFF;
    int bps = 1;
    if ((fmt&0x8)!=0) { bps=2; fmt=fmt&0x7; } // complex
         if (fmt==0) bps *= 16; // int
    else if (fmt==1) bps *= 8;  // byte
    else if (fmt==2) bps *= 4;  // nibble
    else if (fmt==3) bps *= 1;  // bit
    else if (fmt==4) bps *= 32; // long
    else if (fmt==5) bps *= 64; // xlong
    else if (fmt==6) bps *= 32; // float
    else if (fmt==7) bps *= 64; // double
    return bps;
  }
  public int getFormatBytes (int fmts, int n) {
    int byps = getFormatBits(fmts,n)>>3;
    return (byps<=0)? 1:byps;
  }
  public int getFormatBytes (int n) {
    return getFormatBytes(halo.fmts,n);
  }
  public int getFormatBits (int n) {
    return getFormatBits(halo.fmts,n);
  }
  public int getFormatMode (int fmts, int n) {
    int fmt = (fmts>>((n-1)*8)) & 0xFF;
    int spa = ((fmt&0x8)!=0)? 2 : 1; 
    return spa;
  }
  public int getFormatMode (int n) {
    return getFormatMode(halo.fmts,n);
  }

  public int setFormats (String fmt1, String fmt2) {
    halo.fmts = getFormats(fmt1,fmt2);
    return halo.fmts;
  }
  public int setSCID (int index) {
    halo.scid = index;
    return halo.scid;
  }
  public int setMCID (int index) {
    halo.mcid = index;
    return halo.mcid;
  }
  public int setTraceMode (int mode) {
    halo.flgs |=  (mode<<24);
    return halo.flgs;
  }
  public int getTraceData (int addr) {
    return 0;
  }

  public void getVars (String list) { }
  public void putVars (String list) { }
  public void upDate (String list) { }

  public int load (Stream si, DataFile df) {
    if (halo.impl>CoreFactory.JVM) {
      long lbuf = CoreNative.loadStreamBuffer(si.handle);
      int bytes = CoreNative.loadStreamAvail(si.handle);
      int status = df.read(lbuf,0,bytes);
      if (df.getFormatType()=='F' && halo.impl>=CoreFactory.VHS) CoreNative.fp2fptx(lbuf,status);
      return CoreNative.loadStreamStatus(si.handle,status);
    }
    return si.load(df);
  }

  public int unload (Stream so, DataFile df) {
    if (halo.impl>CoreFactory.JVM) {
      long lbuf = CoreNative.unloadStreamBuffer(so.handle);
      int bytes = CoreNative.unloadStreamAvail(so.handle);
      if (df.getFormatType()=='F' && halo.impl>=CoreFactory.VHS) CoreNative.fptx2fp(lbuf,bytes);
      int status = df.write(lbuf,0,bytes);
      return CoreNative.unloadStreamStatus(so.handle,status);
    }
    return so.unload(df);
  }
}
