/**
  Common math function used if ICE software builds 

*/

#ifndef _icemath_h

/* Common Helper Functions */
real_8 log2 (real_8 value) {
  real_8 val;
  val = log(value) * 1.44269504088896;
  return val;
}
real_8 powi (int_4 base, int_4 power) {
  real_8 val;
  if (power==2) val = (real_8)base*(real_8)base;
  else val = pow( (real_8)base, (real_8)power );
  return val;
}
int_4 pow2ge (int_4 size) {
  int_4 p2; for (p2=0;(1<<p2)<size;p2++);
  return p2;
}
int_4 power2ge (int_4 size) {
  int_4 p2; for (p2=0;(1<<p2)<size;p2++);
  return (1<<p2);
}
int_4 imin (int_4 x, int_4 y) {
  return (y<x)? y : x;
}
int_4 imax (int_4 x, int_4 y) {
  return (y>x)? y : x;
}
int_4 ones (int_4 x, int_4 n) {
  int_4 i,ones=0;
  for (i=0; i<n; i++) { ones+=(x&1); x>>=1; }
  return ones;
}
int_4 iabs (int_4 x) { return (x<0)? -x:x; }

int_2 bswap2 (int_2 x) { return (((x>>8)&0xFF)|(x<<8)); }
int_4 bswap4 (int_4 x) { return ((x>>24)&0xFF)|((x>>8)&0xFF00)|((x<<8)&0xFF0000)|(x<<24); }
int_4 bswap2s (int_4 x) { return ((x>>8)&0xFF)|((x<<8)&0xFF00)|((x>>8)&0xFF0000)|((x<<8)&0xFF000000); }

int_4 brev4 (int_4 val) {
  int_4 i,rval=0;
  if (val!=0) for (i=0; i<32; i++) if (((val>>i)&1)!=0) rval |= (1<<(31-i));
  return rval;
}

int_4  s2i (int_2 s) { return (int_4)s; }
int_8  s2l (int_2 s) { return (int_8)s; }
int_2  i2s (int_4 i) { return (int_2)i; }
real_4 i2f (int_4 i) { return (real_4)i; }
real_4 d2f (real_8 f) { return (real_4)f; }
int_4  d2i (real_8 f) { return (int_4)f; }
int_8  d2l (real_8 f) { return (int_8)f; }
int_4  f2i (real_4 f) { return (int_4)f; }
int_4  l2i (int_8 l) { return (int_4)l; }
int_2  l2s (int_8 l) { return (int_2)l; }
real_8 f2d (real_4 f) { return (real_8)f; }
real_8 i2d (int_4 i) { return (real_8)i; }
real_8 l2d (int_8 l) { return (real_8)l; }
int_8 ii2l (int_4 iu, int_4 il) { return ((int_8)iu<<32) | (int_u4)il; }
int_4  f2x (real_4 f)  { return (int_4)(f*65536.0); }
int_8  d2x (real_8 f)  { return (int_8)(f*4294967296.0); }
real_4 x2f (int_4 f)  { return (real_4)(f/65536.0); }
real_8 x2d (int_8 f)  { return (real_8)(f/4294967296.0); }
int_4  d2fi (real_8 f) { return (int_4)(d2x(f)&0xFFFFFFFF); }
int_4  d2wi (real_8 f) { return (int_4)(d2x(f)>>32); }
real_8 fi2d (int_4 f)  { return (real_8)(f/4294967296.0); }

int_2 f2si (real_4 f)  { return (int_2)(f*32768.0F); }
real_4 si2f (int_2 s)  { return (s/32768.0F); }
real_4 i2fp (int_4 i)  { return (i/65536.0F); }
int_4 fp2i (real_4 f)  { return (int_4)(f*65536.0F); }

real_4 invf  (real_4 f) { return (real_4)(1.0F/f); }
real_4 invfi (int_4 i) { return (real_4)(1.0F/i); }
real_4 divf  (real_4 f, real_4 i) { return (real_4)(f/i); }
real_4 divfi (real_4 f, int_4 i) { return (real_4)(f/i); }
real_8 divii (int_4 x, int_4 y) { return ((real_8)x)/y; }
real_4 mulfi (real_4 f, int_4 i) { return (real_4)(f*i); }
real_8 muldi (real_8 f, int_4 i) { return (real_8)(f*i); }
real_4 divfp2 (real_4 f, int_4 i) { return (real_4)(f/(1<<i)); }
real_4 mulfp2 (real_4 f, int_4 i) { return (real_4)(f*(1<<i)); }
real_8 divp2 (real_8 f, int_4 i) { return (real_8)(f/(1<<i)); }
real_8 mulp2 (real_8 f, int_4 i) { return (real_8)(f*(1<<i)); }

real_8 log10 (real_8 f) { return 0.4342944819*log(f); }
real_4 log10f (real_4 f) { return 0.4342944819F*logf(f); }

real_8 frac (real_8 f) { return f-floor(f); }
real_4 fracf (real_4 f) { return f-floorf(f); }

real_8 wrap (real_8 f) { return frac(f+0.5)-0.5; }
real_4 wrapf (real_4 f) { return fracf(f+0.5)-0.5; }

real_8 clip (real_8 f, real_8 fmin, real_8 fmax) { return (f<fmin)? fmin : (f>fmax)? fmax : f; }
real_4 clipf (real_4 f, real_4 fmin, real_4 fmax) { return (f<fmin)? fmin : (f>fmax)? fmax : f; }
int_4 clipl (int_4 f, int_4 fmin, int_4 fmax) { return (f<fmin)? fmin : (f>fmax)? fmax : f; }

real_8 sclip (real_8 f, real_8 fmax) { return (f< -fmax)? -fmax : (f>fmax)? fmax : f; }
real_4 sclipf (real_4 f, real_4 fmax) { return (f< -fmax)? -fmax : (f>fmax)? fmax : f; }
int_4 sclipl (int_4 f, int_4 fmax) { return (f< -fmax)? -fmax : (f>fmax)? fmax : f; }

#define PI    3.141592653589793238462643
#define TWOPI 6.283185307179586476925287

real_8 sinc (real_8 x) { return (real_8)sin(x*TWOPI); }   // in cycles
real_8 cosc (real_8 x) { return (real_8)cos(x*TWOPI); }   // in cycles
real_8 tanc (real_8 x) { return (real_8)tan(x*TWOPI); }   // in cycles
real_8 atan2c (real_8 x, real_4 y) { return (real_8)(atan2(x,y)/TWOPI); }   // in cycles

real_4 sincf (real_4 x) { return (real_4)sinf(x*TWOPI); }   // in cycles
real_4 coscf (real_4 x) { return (real_4)cosf(x*TWOPI); }   // in cycles
real_4 tancf (real_4 x) { return (real_4)tanf(x*TWOPI); }   // in cycles
real_4 atan2cf (real_4 x, real_4 y) { return (real_4)(atan2f(x,y)/TWOPI); }   // in cycles

real_8 sinx (real_8 arg) {
  real_8 num = sin(arg);
  if (fabs(arg)<1e-20 && fabs(num)<1e-20) return 1.0;
  return num/arg;
}
real_8 bessel_io (real_8 x) {
  real_8 ds=1, d=0, s=1;
  for (;;) {
    d += 2;
    ds = (ds*x*x)/(d*d);
    s += ds;
    if (ds-(s*0.2e-8) < 0) break;
  }
  return s;
}
#define FIRKAIS_LOW 1
#define FIRKAIS_HIGH 2
#define FIRKAIS_BAND 3
int_4 firkais_generate (int_4 type, real_8 f1, real_8 f2, real_8 f3, real_8 attn, real_4* filt, int_4 ntap, int_4 dbg) {
  int_4 i;
  real_8 fcntr,fpass,ftnbw,fsamp,mtap,beta,alpha,fcut,wcut,den,strt,lo,pcos,psin,resp,factor,wind;

  fcntr = f1;
  fpass = (type==FIRKAIS_BAND)? f2*.5 : f1;
  ftnbw = f3;
  fsamp = 2.0;

  mtap = (attn-8)/(2.285*2*PI*ftnbw/fsamp);
  if (ntap <= 0) ntap = (int_4)(mtap+1.5);
  if (ntap <= 0) return 0;

  if (attn > 50.0) beta = 0.1102*(attn-8.7);
  else if (attn >= 21.0) beta = 0.58420*pow(attn-21.0,0.4) + 0.07886*(attn-21.0);
  else beta = 0.0;
  alpha = 0.5 * (ntap-1);
  fcut = (type==FIRKAIS_HIGH)? fpass - .5*ftnbw : fpass + .5*ftnbw;
  fcut /= fsamp;
  wcut = 2*PI*fcut;

  den = bessel_io(beta);
  if (type!=FIRKAIS_BAND) fcntr = 0;
  strt = -.5*(ntap-1);
  lo   = 2.0*PI*fcntr/fsamp;
  for (i=0; i<ntap; i++) {
    pcos = cos((strt+i)*lo);
    psin = sin((strt+i)*lo);
    resp = 2.0*fcut*sinx(wcut*(i-alpha));
    if (type==FIRKAIS_HIGH) resp = sinx(PI*(i-alpha)) - resp;
    factor = sqrt(1.0-pow((1.0*i-alpha)/alpha,2.0));
    wind = bessel_io(beta*factor) / den;
    filt[i+i+0] = (real_4)(wind*resp*pcos);
    filt[i+i+1] = (real_4)(wind*resp*psin);
  }
  return ntap;
}

int_4 firrrc_generate (int_4 type, real_8 ratio, real_8 attn, real_4* filt, int_4 ntap, int_4 dbg) {
  return -1;
}

int_4 firwind_generate (char *name, real_8 scale, real_4* buf, int_4 size) {
  int_4 i, sizeh=size/2; char wind[5];
  real_8 c0=0, c1=0, c2=0, c3=0, a,d;
  strncpy(wind,name,4);
  for (i=0;i<4;i++) if (wind[i]>='a'&&wind[i]<='z') wind[i]-=32;
       if (strcmp(wind,"NONE")==0) for (i=0; i<=sizeh; i++) buf[i]=1;
  else if (strcmp(wind,"BART")==0) for (i=0; i<=sizeh; i++) buf[i]=((float)i)/sizeh;
  else if (strcmp(wind,"HANN")==0) { c0=0.5; c1=-0.5; }
  else if (strcmp(wind,"HAMM")==0) { c0=0.54; c1=-0.46; }
  else if (strcmp(wind,"BLAC")==0) { c0=0.42; c1=-0.5; c2=0.08; }
  else if (strcmp(wind,"BH61")==0) { c0=0.44959; c1=-0.49364; c2=0.05677; }
  else if (strcmp(wind,"BH67")==0) { c0=0.42323; c1=-0.49755; c2=0.07922; }
  else if (strcmp(wind,"BH74")==0) { c0=0.40217; c1=-0.49703; c2=0.09897; c3=-.00183; }
  else if (strcmp(wind,"BH92")==0) { c0=0.35875; c1=-0.48829; c2=0.14128; c3=-.01168; }
  else { printf("Unrecognized window name: %s\n",wind); return 0; }
  if (c0!=0) for (i=0; i<=sizeh; i++) {
    a = TWOPI*i/size;
    d = c0 + c1*cos(a) + c2*cos(2*a) + c3*cos(3*a);
    buf[i] = d;
  }
  for (i=sizeh+1; i<size; i++) buf[i] = buf[size-i];
  if (scale!=1) for (i=0; i<size; i++) buf[i] *= scale;
  return size;
}
#define _icemath_h
#endif
