/**
  Common defines for processing COREs

  CORE_AREA is the java native package location
  CORE_NAME is the Library Function Name
  CORE_TYPE is the Processor Type Java,CPU,GPU,VHS,MLS,ICE

  @author Jeff Schoen
  @version $Id: CoreDefs.h,v 1.1 2011/04/27 13:09:47 jgs Exp $
*/

#ifndef _CoreLibs_
#define _CoreLibs_

#include "icecores.h"
#include "icemath.h"
#include <stdarg.h>

#if _UNIX
#include <sys/time.h>
#include <unistd.h>
#include <time.h>
double cpuTime() { 
  struct timeval tv;
  gettimeofday(&tv,0);
  return (double)tv.tv_sec + (double)tv.tv_usec*1.e-6;
}
char cpuDateStr[64];
char *cpuDate() { 
  time_t t; time(&t);
  struct tm *tmp = localtime(&t);
  strftime(cpuDateStr,64,"%Y:%m:%d::%H:%M:%S",tmp);
  return cpuDateStr;
}
double cpuWait (double sec) { usleep((int)(sec*1000000)); }
#else
double cpuTime() { return 0.0; }
char *cpuDate (double cpuTime) { return NULL; }
#endif

void sincoscf (real_4 a, CxReal_4 *sc) { sc->x=(real_4)cosf(a*TWOPI); sc->y=(real_4)sinf(a*TWOPI); }
void pol2recf (real_4 a, real_4 x, CxReal_4 *sc) { sc->x=a*(real_4)cosf(x*TWOPI); sc->y=a*(real_4)sinf(x*TWOPI); }
void dupcc (CxReal_4 *a, CxReal_4 *b) { b->x=a->x;  b->y=a->y; } 
void addcc (CxReal_4 *a, CxReal_4 *b, CxReal_4 *c) { c->x=a->x+b->x;  c->y=a->y+b->y; } 
void mulcc  (CxReal_4 *a, CxReal_4 *b, CxReal_4 *c) { c->x=a->x*b->x - a->y*b->y;  c->y=a->y*b->x + a->x*b->y; } 
void mulccj (CxReal_4 *a, CxReal_4 *b, CxReal_4 *c) { c->x=a->x*b->x + a->y*b->y;  c->y=a->y*b->x - a->x*b->y; } 
void mulfc (real_4 a, CxReal_4 *b, CxReal_4 *c) { c->x=a*b->x;  c->y=a*b->y; } 
void mulcf (CxReal_4 *a, real_4 b, CxReal_4 *c) { c->x=a->x*b;  c->y=a->y*b; } 
void mulccA (CxReal_4 *a, CxReal_4 **b, CxReal_4 **c, int_4 n) { int_4 i; for (i=0; i<n; i++) mulcc(a,b[i],c[i]); }
void mulcAc (CxReal_4 **a, CxReal_4 *b, CxReal_4 **c, int_4 n) { int_4 i; for (i=0; i<n; i++) mulcc(a[i],b,c[i]); }
void mulcAcA (CxReal_4 **a, CxReal_4 **b, CxReal_4 **c, int_4 n) { int_4 i; for (i=0; i<n; i++) mulcc(a[i],b[i],c[i]); }
void CI2CF (CxInt_2 *ci, CxReal_4 *cf) { cf->x=ci->x/32768.0F; cf->y=ci->y/32768.0F; }
real_4 CI2Fx (int_4 i) { i<<=16;        return i/2147483648.0F; }
real_4 CI2Fy (int_4 i) { i&=0xFFFF0000; return i/2147483648.0F; }
int_2 F2I (real_4 f)  { return (int_2)(f*32768.0F); }
real_4 I2F (int_2 s)  { return (s/32768.0F); }
real_4 L2F (int_4 i)  { return (i/2147483648.0F); }
real_4 K2F (int_4 i)  { return (i/65536.0F); }
int_4 F2K (real_4 f)  { return (int)(f*65536); }
int_4 CI2L (int_2 ix, int_2 iy) { return (iy<<16) | (ix&0xFFFF); }
int_4 CF2L (real_4 fx, real_4 fy)  { return CI2L(F2I(fx),F2I(fy)); }
int_1 IN2P (int_2 i)  { return ((i>>3 )&0x01) | ((i>>6 )&0x02) | ((i>>9 )&0x04) | ((i>>12)&0x08); } 
int_2 LN2P (int_4 i)  { return ((i>>3 )&0x01) | ((i>>6 )&0x02) | ((i>>9 )&0x04) | ((i>>12)&0x08) |
                               ((i>>15)&0x10) | ((i>>18)&0x20) | ((i>>21)&0x40) | ((i>>24)&0x80); }
int_1 IO2P (int_4 i)  { return ((i>>2 )&0x01) | ((i>>4 )&0x02) | ((i>>6 )&0x04) | ((i>>8)&0x08) | ((i>>10)&0x10); }
int_2 LO2P (int_4 i)  { return ((i>>2 )&0x01) | ((i>>4 )&0x02) | ((i>>6 )&0x04) | ((i>>8)&0x08) |
	((i>>10)&0x10) | ((i>>12)&0x20) | ((i>>14)&0x40) | ((i>>16)&0x80) | ((i>>18)&0x100) | ((i>>20)&0x200); }

real_8 V2d (Value *v)  { return *((real_8*)(v->b)); }
real_4 V2f (Value *v)  { return *((real_4*)(v->b)); }
int_4 V2i (Value *v)  { return *((int_4*)(v->b)); }
int_2 V2s (Value *v)  { return *((int_2*)(v->b)); }

real_4 phaseditherf () { return 0.0F; }
real_4 mag2f (CxReal_4 *a) { return a->x*a->x + a->y*a->y; } 
real_4 mulff (real_4 a, real_4 b) { return a*b; } 
real_4 mulffrnc (real_4 a, real_4 b) { real_4 c=a*b; if (c>1.0F) c=1.0F; else if (c<-1.0F) c=-1.0F; return c; }
void mulfcrnc (real_4 a, CxReal_4 *b, CxReal_4 *c) { c->x=mulffrnc(a,b->x);  c->y=mulffrnc(a,b->y); } 
void mulcfrnc (CxReal_4 *a, real_4 b, CxReal_4 *c) { c->x=mulffrnc(a->x,b);  c->y=mulffrnc(a->y,b); } 
void muxf (real_4 a, real_4 b, CxReal_4 *c) { c->x=a;  c->y=b; } 
int_4 iLRS (int_4 lrs) { return (lrs<<1) | ( ((lrs>>0)&1) ^ ((lrs>>1)&1) ^ ((lrs>>5)&1) ^ ((lrs>>25)&1) ); }
int_4 iLRSP (int_4 lrs, int_4 lrsp) { return (lrs<<1) | ((ones(lrs&lrsp,32)+1)&1); }

void qrotcc (CxReal_4 *a, int_4 quad, CxReal_4 *b) {
  switch (quad&3) {
  case 0: b->x =  a->x; b->y =  a->y; break;
  case 1: b->x = -a->y; b->y =  a->x; break;
  case 2: b->x = -a->x; b->y = -a->y; break;
  case 3: b->x =  a->y; b->y = -a->x; break;
  }
}
int_4 qwrap2z (CxReal_4 *a) {
  real_4 absx = (a->x<0)? -a->x : a->x;
  return (a->y>absx)? 1 : (a->y<-absx)? 3 : (a->x<0)? 2 : 0;
}

/* Common Helper Functions */
#define null NULL
#define zalloc(A) calloc(1,A)
#define ZBUFSIG 0x12345678

void *zallocBuf (int_4 size) {
  int_4 sigx = ZBUFSIG;
  void *p = zalloc(size+8);
  memcpy(p+0,&sigx,4);
  memcpy(p+4,&size,4);
  return p+8;
}

CxInt_2*  CxInt_2_new ()  { return (CxInt_2*)zalloc(4); }
CxReal_4* CxReal_4_new () { return (CxReal_4*)zalloc(8); }
CxReal_8* CxReal_8_new () { return (CxReal_8*)zalloc(16); }

VxInt_2*  VxInt_2_new ()  { return (VxInt_2*)zalloc(6); }
VxReal_4* VxReal_4_new () { return (VxReal_4*)zalloc(12); }
VxReal_8* VxReal_8_new () { return (VxReal_8*)zalloc(24); }

int_4 strNcmp (const char* str1, const char* str2, int_4 n) {
  int_4 i; char c1,c2;
  for (i=0; i<n; i++) {
    c1=str1[i]; if (c1>'_') c1-=32;
    c2=str2[i]; if (c2>'_') c2-=32;
    if (c1!=c2) return (c2>c1)?1:-1;
  }
  return 0;
}
int_4 strNcpy (char* str1, const char* str2, int_4 n) {
  int_4 i; char c2;
  for (i=0; i<n; i++) {
    c2=str2[i]; if (c2>'_') c2-=32;
    str1[i]=c2;
  }
  str1[i]=0;
  return 0;
}

int_4 getListBase (const char *sa) {
  if (strNcmp(sa,"NONE,",5)==0 || strNcmp(sa,"OFF,",4)==0) return 0;
  if (strNcmp(sa,"AUTO,",5)==0 || strNcmp(sa,"ALL,",4)==0) return -1;
  return 1;
}

char getListTemp[40];
char* getListItem (const char* sa, int_4 index, int offset) {
  int_4 i, j=0, k, la=strlen(sa);
  k=(offset==-2)? getListBase(sa) : offset;
  strcpy(getListTemp,"Undefined");
  for (i=0;i<=la;i++) {
    if (i==la || sa[i]==',' || sa[i]=='|') {
      if (k==index) { strNcpy(getListTemp,sa+j,i-j); break; }
      j=i+1; k++;
    }
  }
  return getListTemp;
}
int_4 findListItem (const char* sa, char* sb, int offset) {
  int_4 i, j=0, k=1, la=strlen(sa), lb=strlen(sb);
  k=(offset==-2)? getListBase(sa) : offset;
  for (i=0;i<=la;i++) {
    if (i==la || sa[i]==',') {
      if (strNcmp(sa+j,sb,lb)==0) return k;
      j=i+1; k++;
    }
  }
  return -2;
}

Value* Value_new (String* format, int_4 size) {
  int_4 sValue = sizeof(Value);
  int_1* ptr;
  Value val;
  val.mode=format[0];
  val.type=format[1];
  val.spa=(val.mode=='V')?3:(val.mode=='C')?2:1;
  val.bps=(val.type=='D')?8:(val.type=='F')?4:(val.type=='L')?4:(val.type=='I')?2:1;
  val.bpa=val.spa*val.bps;
  val.len=size*val.bpa;
  ptr = zalloc(sValue+val.len);
  val.b=ptr+sValue;
  *(Value*)ptr=val;
  return (Value*)ptr;
}
void Value_setSize (Value* val, int_4 len) {
  val->len = len*val->bpa;
}
void Value_setBPA (Value* val, int_4 len) {
  val->bpa = len;
}
void Value_free (Value* val) {
  free(val);
}
void Value_setS (Value* val, char *s) {
  int ls = strlen(s);
  if (ls>val->len-1) ls=val->len-1;
  val->type='S'; strncpy(val->b,s,ls);
}
real_8 Value_getxD (Value* val, int_4 off);
int_4 Value_getxL (Value* val, int_4 off);

int_2 Value_getB (Value* val, int_4 off) {
  int_1 *d = (int_1*)(val->b);
  return (val->type=='B')? d[off] : (int_1)Value_getxL(val,off);
}
int_2 Value_getI (Value* val, int_4 off) {
  int_2 *d = (int_2*)(val->b);
  return (val->type=='I')? d[off] : (int_2)Value_getxL(val,off);
}
int_4 Value_getL (Value* val, int_4 off) {
  int_4 *d = (int_4*)(val->b);
  return (val->type=='L')? d[off] : (int_4)Value_getxL(val,off);
}
real_4 Value_getF (Value* val, int_4 off) {
  real_4 *d = (real_4*)(val->b);
  return (val->type=='F')? d[off] : (real_4)Value_getxD(val,off);
}
void Value_getCF (Value* val, int_4 off, CxReal_4 *cf) {
  real_4 *d = (real_4*)(val->b);
  off+=off; cf->x=d[off]; cf->y=d[off+1];
}
real_8 Value_getD (Value* val, int_4 off) {
  real_8 *d = (real_8*)(val->b);
  return (val->type=='D')? d[off] : (real_8)Value_getxD(val,off);
}
int_4 Value_getxL (Value* val, int_4 off) {
  switch (val->type) {
  case 'B': return (int_4)Value_getB(val,off);
  case 'I': return (int_4)Value_getI(val,off);
  case 'L': return (int_4)Value_getL(val,off);
  case 'F': return (int_4)Value_getF(val,off);
  case 'D': return (int_4)Value_getD(val,off);
  }
  return 0.0;
}
real_8 Value_getxD (Value* val, int_4 off) {
  switch (val->type) {
  case 'B': return (real_8)Value_getB(val,off);
  case 'I': return (real_8)Value_getI(val,off);
  case 'L': return (real_8)Value_getL(val,off);
  case 'F': return (real_8)Value_getF(val,off);
  case 'D': return (real_8)Value_getD(val,off);
  }
  return 0.0;
}

int_4 Value_toItem (Value* val, const char *list) {
  int_4 item = findListItem(list,(char*)val->b,-2); 
  if (item<-1) printf("Item=%s not found in list=%s\n",(char*)val->b,list);
  return item;
}
int_4 Value_toMask (Value* val, const char *list) {
  int_4 i,bit,mask=0; char *s, *sitem=(char*)val->b;
  if (strNcmp(sitem,"None",4)==0) return 0;
  if (strNcmp(sitem,"All",3)==0) return -1;
  for (i=0; i<32; i++) {
    s = getListItem(sitem,i,0); 
    if (s[0]==0) continue;
    if (strNcmp(s,"Undefined",9)==0) break;
    bit = findListItem(list,s,0);
    if (bit<0) printf("Item=%s not found in list=%s\n",s,list);
    else mask |= (1<<bit);
  }
  return mask;
}
int_4 Value_toL (Value* val) {
  return Value_getL(val,0); 
}
real_4 Value_toF (Value* val) {
  return Value_getF(val,0); 
}
real_8 Value_toD (Value* val) {
  return Value_getD(val,0); 
}
char* Value_toS (Value* val) {
  return (char*)val->b;
}
real_4* Value_toFA (Value* val) {
  return (real_4*)val->b;
}

int_1 Value_setB (Value *val, int_4 off, int_1 value) {
  int_1 *d = (int_1*)(val->b);
  d[off]=value;
  return value;
}
int_2 Value_setI (Value *val, int_4 off, int_2 value) {
  int_2 *d = (int_2*)(val->b);
  d[off]=value;
  return value;
}
int_4 Value_setL (Value *val, int_4 off, int_4 value) {
  int_4 *d = (int_4*)(val->b);
  d[off]=value;
  return value;
}
real_4 Value_setF (Value *val, int_4 off, real_4 value) {
  real_4 *d = (real_4*)(val->b);
  d[off]=value;
  return value;
}
void Value_setCF (Value *val, int_4 off, CxReal_4 *cf) {
  real_4 *d = (real_4*)(val->b);
  off+=off; d[off]=cf->x; d[off+1]=cf->y;
}
void Value_setCFxy (Value *val, int_4 off, real_4 x, real_4 y) {
  real_4 *d = (real_4*)(val->b);
  off+=off; d[off]=x; d[off+1]=y;
}
void Value_setCFIS2 (Value *val, int_4 shf, CxReal_4 *cf1, CxReal_4 *cf2) {
  int_2 *d = (int_2*)(val->b); int_4 scl = (shf<0)? 0x8000>>-shf : 0x8000<<shf;
  d[0]=round(cf1->x*scl); d[1]=round(cf1->y*scl);
  d[2]=round(cf2->x*scl); d[3]=round(cf2->y*scl);
}
real_8 Value_setD (Value *val, int_4 off, real_8 value) {
  real_8 *d = (real_8*)(val->b);
  d[off]=value;
  return value;
}

char* Value_fromList (Value* val, const char *list, int_4 value) {
  return getListItem(list,value,-2);
}
int_4 Value_fromL (Value* val, int_4 value) {
  val->type='L'; val->len=4;
  return Value_setL(val,0,value); 
}
real_4 Value_fromF (Value* val, real_4 value) {
  val->type='F'; val->len=4;
  return Value_setF(val,0,value); 
}
real_8 Value_fromD (Value* val, real_8 value) {
  val->type='D'; val->len=8;
  return Value_setD(val,0,value); 
}
char *Value_fromS (Value* val, char *value) {
  Value_setS(val,value); 
  return value;
}

int_4 getBit (int_8 value, int_4 i) {
  int_8 mask = 1L<<i;
  return (value&mask)!=0;
}
int_4 getBits (int_4 value, int_4 i2, int_4 i1) {
  int_4 mask = ((2<<(i2-i1))-1);
  return (int_4)(value>>i1) & mask;
}
int_8 getBitsX (int_8 value, int_4 i2, int_4 i1) {
  int_8 mask = ((2L<<(i2-i1))-1);
  return (value>>i1) & mask;
}
int_8 getPBS (int_8 curr, int_8 value, int_4 i2, int_4 i1) {
  int_4 len  = i2-i1+1;
  int_8 mask = ((1L<<len)-1);
  return (curr<<len) | ((value>>i1)&mask);
}
int_8 getPABS (int_8 curr, int_8 value, int_4 len) {
  int_8 mask = ((1L<<len)-1);
  return (curr<<len) | (value&mask);
}
int_4 getPBBS (int_4 curr, int_1 value) {
  return (curr<<1) | value;
}
int_8 setBit (int_8 value, int_4 i, int_4 valueb) {
  int_8 mask = 1L<<i;
  return valueb? value|mask : value & ~mask;
}
int_8 setBits (int_8 value, int_4 i2, int_4 i1, int_8 valueb) {
  int_8 mask = ((2L<<(i2-i1))-1);
  return (value&~(mask<<i1)) | ((valueb&mask)<<i1);
}

Ram* Ram_new (int_4 OL, int_4 widthA, int_4 widthB) {
  int_4 sRam = sizeof(Ram);
  int_1* ptr = zalloc((1<<OL)+sRam);
  Ram* ram = (Ram*)ptr;
  ram->OL=OL;
  ram->len=1<<OL;
  ram->WA=widthA;
  ram->WB=widthB;
  ram->AL=1<<OL; ram->ML=ram->AL-1;
  ram->b=ptr+sRam;
  return ram;
}
void Ram_rd (Ram* ram, int_4 off, Value* val) {
  bcopy(ram->b+off, val->b, val->len);
}
void Ram_wr (Ram* ram, int_4 off, Value* val) {
  bcopy(val->b, ram->b+off, val->len);
}
int_1 Ram_rdB (Ram* ram, int_4 off) {
  off &= ram->ML; return ram->b[off];
}
void Ram_wrB (Ram* ram, int_4 off, int_4 val) {
  off &= ram->ML; ram->b[off]=(int_1)val;
}
int_2 Ram_rdI (Ram* ram, int_4 off) {
  off &= ram->ML; return *(int_2*)(&ram->b[off]);
}
void Ram_wrI (Ram* ram, int_4 off, int_2 val) {
  off &= ram->ML; *(int_2*)(&ram->b[off])=val;
}
int_4 Ram_rdL (Ram* ram, int_4 off) {
  off &= ram->ML; return *(int_4*)(&ram->b[off]);
}
void Ram_wrL (Ram* ram, int_4 off, int_4 val) {
  off &= ram->ML; *(int_4*)(&ram->b[off])=val;
}
real_4 Ram_rdF (Ram* ram, int_4 off) {
  off &= ram->ML; return *(real_4*)(&ram->b[off]);
}
void Ram_wrF (Ram* ram, int_4 off, real_4 val) {
  off &= ram->ML; *(real_4*)(&ram->b[off])=val;
}


RamRWC* RamRWC_new (int_4 OL, int_4 widthA, int_4 widthB, int_4 OC) {
  int_4 sRam = sizeof(RamRWC);
  int_1* ptr = zalloc((1<<(OC+OL))+sRam);
  RamRWC* ram = (RamRWC*)ptr;
  ram->OC=OC; 
  ram->OL=OL;
  ram->len=1<<(OL+OC);
  ram->WA=widthA;
  ram->WB=widthB;
  ram->AL=1<<OL; ram->ML=ram->AL-1;
  ram->AC=1<<OC; ram->MC=ram->AC-1;
  ram->chn=0;
  ram->b=ptr+sRam;
  return ram;
}
void RamRWC_setC (RamRWC* ram, int_4 chn) {
  ram->chn = chn;
}
void RamRWC_rd (RamRWC* ram, int_4 off, Value* val) {
  bcopy(ram->b+((ram->chn<<ram->OL)|off), val->b, val->len);
}
void RamRWC_wr (RamRWC* ram, int_4 off, Value* val) {
  bcopy(val->b, ram->b+((ram->chn<<ram->OL)|off), val->len);
}
int_1 RamRWC_rdB (RamRWC* ram, int_4 off) {
  off &= ram->ML; return ram->b[(ram->chn<<ram->OL)|off];
}
void RamRWC_wrB (RamRWC* ram, int_4 off, int_4 val) {
  off &= ram->ML; ram->b[(ram->chn<<ram->OL)|off]=(int_1)val;
}
real_4 RamRWC_rdF (RamRWC* ram, int_4 off) {
  off &= ram->ML; return *(real_4*)(&ram->b[(ram->chn<<ram->OL)|off]);
}
void RamRWC_wrF (RamRWC* ram, int_4 off, int_4 val) {
  off &= ram->ML; *(real_4*)(&ram->b[(ram->chn<<ram->OL)|off])=val;
}

#define StreamI Stream
#define StreamO Stream

Stream* Stream_new (int_4 len, int_4 IBW, int_4 OBW) {
  int_4 sStr = sizeof(Stream);
  int_1* ptr = zalloc(len+sStr);
  Stream* str = (Stream*)ptr;
  str->len=len;
  str->mtl=len/2;
  str->mask=len-1;
  str->wok=1;
  str->rper = str->rbpe = (IBW/8);
  str->wper = str->wbpe = (OBW/8);
  str->b=ptr+sStr;
  return str;
}

void Stream_update (Stream *str, int_4 xfer) {
  str->bib += xfer;
  if (xfer>0) {
    str->woff += xfer;
    if (str->woff >= str->len) str->woff -= str->len;
  } else {
    str->roff -= xfer;
    if (str->roff >= str->len) str->roff -= str->len;
  }
  str->rok = (str->bib >= str->rper);
  str->wok = (str->len-str->bib >= str->wper);
}

int_8 Stream_loadBuffer (Stream *str) {
  return (int_8)(int_x)(str->b + str->woff);
}
int_4 Stream_loadAvail (Stream *str) {
  int_4 n,avail = str->roff-str->woff;
  if (avail<0 || str->bib==0) avail = str->len-str->woff;
  if (avail>str->mtl) avail = str->mtl;
  return avail;
}
int_4 Stream_loadStatus (Stream *str, int_4 n) {
  if (n>0) Stream_update (str,n);
  return n;
}
int_8 Stream_unloadBuffer (Stream *str) {
  return (int_8)(int_x)(str->b + str->roff);
}
int_4 Stream_unloadAvail (Stream *str) {
  int_4 n, avail = str->woff-str->roff;
  if (avail<0 || str->bib==str->len) avail = str->len-str->roff;
  return avail;
}
int_4 Stream_unloadStatus (Stream *str, int_4 n) {
  if (n>0) Stream_update (str,-n);
  return n;
}

void Stream_rd (Stream* str, Value* val) {
  memcpy(val->b,str->b+str->roff,val->len);
  Stream_update(str,-val->len);
}

real_4 Stream_rdF (Stream* str) {
  real_4 val = *(real_4*)(str->b+str->roff);
  Stream_update(str,-4);
  return val;
}

int_4 Stream_rdL (Stream* str) {
  int_4 val = *(int_4*)(str->b+str->roff);
  Stream_update(str,-4);
  return val;
}

int_2 Stream_rdI (Stream* str) {
  int_2 val = str->b[str->roff]; 
  Stream_update(str,-2);
  return val;
}

int_1 Stream_rdB (Stream* str) {
  int_1 val = str->b[str->roff]; 
  Stream_update(str,-1);
  return val;
}

int_4 Stream_rdBits (Stream* str, int_4 n) {
  int_4 i,x=0;
  for (i=0; i<n; i++) {
    if ((str->b[str->roff]&(1<<str->rboff))!=0) x |= (1<<i);
    if (++str->rboff==8) { Stream_update(str,-1); str->rboff=0; }
  }
  return x;
}

int_8 Stream_rdX (Stream* str) {
  int_8 val = *(int_8*)(str->b+str->roff);
  Stream_update(str,-8);
  return val;
}

void Stream_rdCI (Stream* str, CxInt_2* val) {
  int_2 *fb = (int_2*)(str->b+str->roff);
  val->x = fb[0]; val->y = fb[1];
  Stream_update(str,-4);
}

void Stream_rdCF (Stream* str, CxReal_4* val) {
  real_4 *fb = (real_4*)(str->b+str->roff);
  val->x = fb[0]; val->y = fb[1];
  Stream_update(str,-8);
}

void Stream_rdCIF (Stream* str, CxReal_4* val) {
  int_2 *fb = (int_2*)(str->b+str->roff);
  val->x = (fb[0]/32768.0);
  val->y = (fb[1]/32768.0);
  Stream_update(str,-4);
}

void Stream_rdCIFA (Stream* str, CxReal_4** val, int_4 n) {
  int_4 i; for (i=0; i<n; i++) Stream_rdCIF(str,val[i]);
}

void Stream_rdCIFx (Stream* str, Value* val, int_4 n) {
  CxReal_4 cxt; int_4 i; for (i=0; i<n; i++) { Stream_rdCIF(str,&cxt); Value_setCF(val,i,&cxt); }
}

void Stream_wr (Stream* str, Value* val) {
  memcpy(str->b+str->woff,val->b,val->len);
  Stream_update(str,val->len);
}

void Stream_wrx (Stream* str, Value* val, int_4 n) {
  int_4 bpe = val->bpa*n;
  if (bpe>0) {
    memcpy(str->b+str->woff,val->b,bpe); 
    Stream_update(str,bpe);
  }
}

void Stream_wrX (Stream* str, int_8 val) {
  int_8 *fb = (int_8*)(str->b+str->woff);
  fb[0] = val;
  Stream_update(str,8);
}

void Stream_wrF (Stream* str, real_4 val) {
  real_4 *fb = (real_4*)(str->b+str->woff);
  fb[0] = val;
  Stream_update(str,4);
}

void Stream_wrL (Stream* str, int_4 val) {
  int_4 *fb = (int_4*)(str->b+str->woff);
  fb[0] = val;
  Stream_update(str,4);
}

void Stream_wrLCS (Stream* str, int_4 val, int_1 chn, int_1 swaps) {
  int_4 *fb = (int_4*)(str->b+str->woff);
  str->chn=chn;
       if (swaps==1) val = bswap2s(val);
  else if (swaps==2) val = bswap4(val);
  fb[0] = val;
  Stream_update(str,4);
}

void Stream_wrI (Stream* str, int_2 val) {
  int_2 *fb = (int_2*)(str->b+str->woff);
  fb[0] = val;
  Stream_update(str,4);
}

void Stream_wrB (Stream* str, int_1 val) {
  str->b[str->woff]=val;
  Stream_update(str,1);
}

void Stream_wrBits (Stream* str, int_4 n, int_4 x) {
   int_4 i;
   for (i=0; i<n; x>>=1,i++) {
     if ((x&1)==0) str->b[str->woff] &= ~(1<<str->wboff);
     else          str->b[str->woff] |=  (1<<str->wboff);
     if (++str->wboff==8) { Stream_update(str,1); str->wboff=0; }
   }
}

void Stream_wrCI (Stream* str, CxInt_2* val) {
  int_2 *fb = (int_2*)(str->b+str->woff);
  fb[0] = val->x; fb[1] = val->y;
  Stream_update(str,4);
}

void Stream_wrCF (Stream* str, CxReal_4* val) {
  real_4 *fb = (real_4*)(str->b+str->woff);
  fb[0] = val->x; fb[1] = val->y;
  Stream_update(str,8);
}

void Stream_wrCFx (Stream* str, CxReal_4* val, int_4 n) {
  int_4 i; for (i=0; i<n; i++) Stream_wrCF(str,val+i);
}

void Stream_wrCFB (Stream* str, CxReal_4* val) {
  int_1 *fb = (int_1*)(str->b+str->woff);
  fb[0] = (int_1)roundf(val->x*128);
  fb[1] = (int_1)roundf(val->y*128);
  Stream_update(str,2);
}
void Stream_wrCFBA (Stream* str, CxReal_4** val, int_4 n) {
  int_4 i; for (i=0; i<n; i++) Stream_wrCFB(str,val[i]);
}

void Stream_wrCFI (Stream* str, CxReal_4* val) {
  int_2 *fb = (int_2*)(str->b+str->woff);
  fb[0] = (int_2)roundf(val->x*32768);
  fb[1] = (int_2)roundf(val->y*32768);
  Stream_update(str,4);
}
void Stream_wrCFIA (Stream* str, CxReal_4** val, int_4 n) {
  int_4 i; for (i=0; i<n; i++) Stream_wrCFI(str,val[i]);
}
void Stream_wrCFIx (Stream* str, Value* val, int_4 n) {
  CxReal_4 cxt; int_4 i; for (i=0; i<n; i++) { Value_getCF(val,i,&cxt); Stream_wrCFI(str,&cxt); }
}
void Stream_wrCFxy (Stream* str, real_4 x, real_4 y) {
  real_4 *fb = (real_4*)(str->b+str->woff);
  fb[0] = x; fb[1] = y;
  Stream_update(str,8);
}
void Stream_wrCIxy (Stream* str, int_2 x, int_2 y) {
  int_2 *fb = (int_2*)(str->b+str->woff);
  fb[0] = x; fb[1] = y;
  Stream_update(str,4);
}

void Stream_setC (Stream* str, int_4 chn) {
  str->chn=chn;
}
int_4 Stream_getC (Stream* str) {
  return str->chn;
}


#define BufferI Buffer
#define BufferO Buffer

void Buffer_rd (Buffer* buf, int_4 ioff, Value* val) {
  int_4 bpe = val->len;
  memcpy(val->b,buf->b+((buf->roff+ioff)&buf->mask),bpe);
}
void Buffer_wr (Buffer* buf, int_4 ioff, Value* val) {
  int_4 bpe = val->len;
  memcpy(buf->b+((buf->woff+ioff)&buf->mask),val->b,bpe);
}
void Buffer_rokLen (Buffer* buf, int_4 n) { buf->rper = n; }
void Buffer_wokLen (Buffer* buf, int_4 n) { buf->wper = n; }
void Buffer_setBPE (Buffer* buf, int_4 n) { buf->rbpe = n; buf->wbpe = n; }

void Buffer_update (Buffer *buf, int_4 xfer) {
  buf->bib += xfer;
  if (xfer>0) {
    buf->woff += xfer;
    if (buf->woff >= buf->len) buf->woff -= buf->len;
  } else {
    buf->roff += -xfer;
    if (buf->roff >= buf->len) buf->roff -= buf->len;
  }
  buf->rok = (buf->bib >= buf->rper);
  buf->wok = (buf->len-buf->bib >= buf->wper);
}
void Buffer_rnxt (Buffer* buf, int_4 n) { Buffer_update(buf,-n); }
void Buffer_wnxt (Buffer* buf, int_4 n) { Buffer_update(buf,n); }
int_1 Buffer_getB (Buffer* buf, int_4 ioff) {
  int_1 val = buf->b[((buf->roff+ioff)&buf->mask)];
  return val;
}
int_2 Buffer_getI (Buffer* buf, int_4 ioff) {
  int_2 val = *(int_2*)(buf->b+((buf->roff+ioff)&buf->mask));
  return val;
}
int_4 Buffer_getL (Buffer* buf, int_4 ioff) {
  int_4 val = *(int_4*)(buf->b+((buf->roff+ioff)&buf->mask));
  return val;
}
real_4 Buffer_getF (Buffer* buf, int_4 ioff) {
  real_4 val = *(real_4*)(buf->b+((buf->roff+ioff)&buf->mask));
  return val;
}
int_8 Buffer_getX (Buffer* buf, int_4 ioff) {
  int_8 val = *(int_8*)(buf->b+((buf->roff+ioff)&buf->mask));
  return val;
}
void Buffer_setL (Buffer* buf, int_4 ioff, int_4 val) {
  memcpy(buf->b+((buf->woff+ioff)&buf->mask),&val,4);
}
void Buffer_setF(Buffer* buf, int_4 ioff, real_4 val) {
  memcpy(buf->b+((buf->woff+ioff)&buf->mask),&val,4);
}
void Buffer_setX (Buffer* buf, int_4 ioff, int_8 val) {
  memcpy(buf->b+((buf->woff+ioff)&buf->mask),&val,8);
}
void Buffer_setC (Buffer* buf, int_4 chn) {
  buf->chn=chn;
}

Queue* Queue_new (int_4 size, int_4 BW, int_4 AFULL, int_4 AEMPTY) {
  int_4 sStr = sizeof(Queue);
  int_4 bpe  = (BW+7)/8; 
  int_1* ptr = zalloc(size*bpe+sStr);
  Queue* str = (Queue*)ptr;
  str->bpe   = bpe;
  str->len   = size*bpe;
  str->mask  = str->len-1;
  for (str->shft=0; (1<<str->shft)<bpe; str->shft++);
  str->afull = AFULL;	str->wok=1;
  str->aempty = AEMPTY;	str->rok=0;
  str->roff = 0;
  str->woff = 0;
  str->b=ptr+sStr;
  return str;
}

void Queue_update (Queue* str, int_4 rw) {
  int nin;
  if (rw>0) str->woff = (str->woff+str->bpe)&str->mask;
  if (rw<0) str->roff = (str->roff+str->bpe)&str->mask;
  nin = ((str->woff-str->roff)&str->mask) >> str->shft;
  str->wok = nin <= str->afull;
  str->rok = nin > str->aempty;
}
int_4 Queue_wok (Queue* str) { return str->wok; }
int_4 Queue_rok (Queue* str) { return str->rok; }

void   Queue_rd   (Queue* str, Value* val) { memcpy(val->b,str->b+str->roff,str->bpe); Queue_update (str,-1); }
int_1  Queue_rdB  (Queue* str) { int_1 v; memcpy(&v,str->b+str->roff,str->bpe); Queue_update (str,-1); return v; }
int_2  Queue_rdI  (Queue* str) { int_2 v; memcpy(&v,str->b+str->roff,str->bpe); Queue_update (str,-1); return v; }
int_4  Queue_rdL  (Queue* str) { int_4 v; memcpy(&v,str->b+str->roff,str->bpe); Queue_update (str,-1); return v; }
real_4 Queue_rdF  (Queue* str) { real_4 v; memcpy(&v,str->b+str->roff,str->bpe); Queue_update (str,-1); return v; }
real_8 Queue_rdD  (Queue* str) { real_8 v; memcpy(&v,str->b+str->roff,str->bpe); Queue_update (str,-1); return v; }
void   Queue_rdCF (Queue* str, CxReal_4 *v) { memcpy(v,str->b+str->roff,str->bpe); Queue_update (str,-1); }

void Queue_wr   (Queue* str, Value* val) { memcpy(str->b+str->woff,val->b,str->bpe); Queue_update (str,1); }
void Queue_wrB  (Queue* str, int_1 v)    { memcpy(str->b+str->woff,&v,str->bpe); Queue_update (str,1); }
void Queue_wrI  (Queue* str, int_2 v)    { memcpy(str->b+str->woff,&v,str->bpe); Queue_update (str,1); }
void Queue_wrL  (Queue* str, int_4 v)    { memcpy(str->b+str->woff,&v,str->bpe); Queue_update (str,1); }
void Queue_wrF  (Queue* str, real_4 v)   { memcpy(str->b+str->woff,&v,str->bpe); Queue_update (str,1); }
void Queue_wrD  (Queue* str, real_8 v)   { memcpy(str->b+str->woff,&v,str->bpe); Queue_update (str,1); }
void Queue_wrCF (Queue* str, CxReal_4 *v) { memcpy(str->b+str->woff,v,str->bpe); Queue_update (str,1); }

#define DMAFLG_SWP2 0x1
#define DMAFLG_SWP4 0x2
#define DMAFLG_C12T16 0x4
#define DMAFLG_C16T12 0x8

Dma* Dma_new (int_4 AW, int_4 BW) {
  Dma* dma = (Dma*)zalloc(sizeof(Dma));
  dma->AW=AW;
  dma->BW=BW;
  dma->cnt=0;
  dma->flags=0;
  return dma;
}
void ba2ba (Dma* dma, int_1* bi, int offi, int leni,  int_1* bo, int offo, int leno, int len) {
  int maski=leni-1, masko=leno-1;
  while (len>0) {
    int ido = len;
    offi &= maski;
    offo &= masko;
    if (offi+ido>leni) ido = leni-offi;
    if (offo+ido>leno) ido = leno-offo;
    bcopy(bi+offi, bo+offo, ido);
    offi += ido;
    offo += ido;
    len  -= ido;
  }
}

void ba2bs (Dma* dma, int_1* bi, int offi, int leni,  int_1 *bo, int offo, int leno, int len, Stream* so) {
      int i, maski=leni-1, masko=leno-1; int_1 b0,b1,b2;
      offi &= maski;
      offo &= masko;
      if ((dma->flags&DMAFLG_SWP2)!=0) {
        for (i=0; i<len; i+=2) {
          bo[offo] = bi[offi+1];
          bo[offo+1] = bi[offi];
          offi += 2; offi &= maski;
          offo += 2; offo &= masko;
        }
      }
      else if ((dma->flags&DMAFLG_C16T12)!=0) {
        for (i=0; i<len; i+=4) {
          bo[offo++] = (int_1)(((bi[offi+1]<<4)&0xF0) | ((bi[offi+0]>>4)&0x0F)); offo &= masko;
          bo[offo++] = (int_1)(( bi[offi+2]    &0xF0) | ((bi[offi+1]>>4)&0x0F)); offo &= masko;
          bo[offo++] =  bi[offi+3];                                             offo &= masko;
          offi += 4; offi &= maski;
        }
        len = (len/4)*3;
      }
      else if ((dma->flags&DMAFLG_C12T16)!=0) {
        for (i=0; i<len; i+=3) {
          b0=bi[offi++]; offi &= maski;
          b1=bi[offi++]; offi &= maski;
          b2=bi[offi++]; offi &= maski;
          bo[offo++] = (int_1)(b0<<4);
          bo[offo++] = (int_1)(((b1<<4)&0xF0)|((b0>>4)&0x0F));
          bo[offo++] = (int_1)(b1&0xF0);
          bo[offo++] = b2;
          offo &= masko;
        }
        len = (len/3)*4;
      }
      else {
        for (i=0; i<len; i+=1) {
          bo[offo++] = bi[offi++];
          offi &= maski;
          offo &= masko;
        }
      }
      Stream_update(so,len);
}

void Dma_r2b (Dma* dma, RamRWC* ri,int_4 off,  Buffer* bo,int_4 boff, int_4 len) {
  off &= ri->ML; int_4 xlen = off+len-ri->AL; if (xlen>0) len-=xlen; else xlen=0;
  bcopy(ri->b+((ri->chn<<ri->OL)|off), bo->b+(bo->woff+boff), len);
  if (xlen>0) bcopy(ri->b+(ri->chn<<ri->OL), bo->b+(bo->woff+boff+len), xlen);
}
void Dma_b2r (Dma* dma, Buffer* bi,int_4 boff, RamRWC* ro,int_4 off, int_4 len) {
  off &= ro->ML; int_4 xlen = off+len-ro->AL; if (xlen>0) len-=xlen; else xlen=0;
  bcopy(bi->b+(bi->roff+boff), ro->b+((ro->chn<<ro->OL)|off), len);
  if (xlen>0) bcopy(bi->b+(bi->roff+boff+len), ro->b+(ro->chn<<ro->OL), xlen);
}
void Dma_r2r (Dma* dma, RamRWC* ri,int_4 offi,  RamRWC* ro,int_4 offo, int_4 len) {
  ba2ba (dma, ri->b,(ri->chn<<ri->OL)|offi,ri->len, ro->b,(ro->chn<<ro->OL)|offo,ro->len, len);
}
void Dma_b2b (Dma* dma, Buffer* bi,int_4 boffi, Buffer* bo,int_4 boffo, int_4 len) {
  ba2ba (dma, bi->b,bi->roff+boffi,bi->len, bo->b,bo->woff+boffo,bo->len, len);
}
void Dma_b2s (Dma* dma, Buffer* bi,int_4 boffi, Stream* so,int_4 soffo, int_4 len) {
  ba2bs (dma, bi->b,bi->roff+boffi,bi->len, so->b,so->woff,so->len, len, so);
}
void Dma_r2s (Dma* dma, RamRWC* ri,int_4 offi, Stream* so,int_4 soffo, int_4 len) {
  ba2ba (dma, ri->b,(ri->chn<<ri->OL)|offi,ri->len, so->b,so->woff,so->len, len);
}
void Dma_addReader (Dma* dma, void *obj) { }
void Dma_sync (Dma* dma) { }
void Dma_setFlags (Dma* dma, int_4 flags) { dma->flags = flags; }

RomF* RomF_new (int_4 oM, int_4 ow) {
  int_4 m = 1<<oM;
  int_4 sRam = sizeof(RomF);
  int_4 l = m+2;
  int_1* ptr = zalloc(l*4+sRam);
  RomF* ram = (RomF*)ptr;
  ram->N=4; 
  ram->M=m;
  ram->L=l;
  ram->b=(real_4*)(ptr+sRam);
  ram->bsigx=ZBUFSIG;
  ram->bsize=m*4;
  return ram;
}
void RomF_set (RomF* ram, int_4 m, real_4 d) {
  ram->b[m] = d;
}
real_4 RomF_get (RomF* ram, int_4 m) {
  return ram->b[m];
}
int_4 RomF_upload (RomF* ram, void *planp) {
  int len=3;	/* Only N,M,b in actual HW */
  HW_loadHdr(planp,len);
  HW_loadInt(planp,ram->N);
  HW_loadInt(planp,ram->M);
  HW_loadBuf(planp,ram->b,'f',1);
  return len;
}

RomF2D* RomF2D_new (int_4 oN, int_4 oM, int_4 ow) {
  int_4 n = 1<<oN;
  int_4 m = 1<<oM;
  int_4 l = n*m+2;
  int_4 sRam = sizeof(RomF2D);
  int_1* ptr = zalloc(l*4+sRam);
  RomF2D* ram = (RomF2D*)ptr;
  ram->N=n; 
  ram->M=m;
  ram->L=l;
  ram->b=(real_4*)(ptr+sRam);
  ram->bsigx=ZBUFSIG;
  ram->bsize=l*4;
  return ram;
}
void RomF2D_clear (RomF2D* ram) {
  int i; for (i=0; i<ram->L; i++) ram->b[i]=0;
}
void RomF2D_set (RomF2D* ram, int_4 n, int_4 m, real_4 d) {
  ram->b[m*ram->N+n] = d;
}
real_4 RomF2D_get (RomF2D* ram, int_4 n, int_4 m) {
  return ram->b[m*ram->N+n];
}
int_4 RomF2D_upload (RomF2D* ram, void *planp) {
  int len=3;	/* Only N,M,b in actual HW */
  HW_loadHdr(planp,len);
  HW_loadInt(planp,ram->N);
  HW_loadInt(planp,ram->M);
  HW_loadBuf(planp,ram->b,'f',1);
  return len;
}

RomF2Di* RomF2Di_new (int_4 oN, int_4 oM, int_4 ow) {
  int_4 n = 1<<oN;
  int_4 m = 1<<oM;
  int_4 l = n*m+1;
  int_4 sRam = sizeof(RomF2Di);
  int_1* ptr = zalloc(l*4+4+sRam);
  RomF2Di* ram = (RomF2Di*)ptr;
  ram->N=n; 
  ram->M=m;
  ram->L=l;
  ram->MC=ow/2;
  ram->b=(real_4*)(ptr+sRam);
  ram->bsigx=ZBUFSIG;
  ram->bsize=l*4;
  return ram;
}
void RomF2Di_clear (RomF2Di* ram) {
  int i; for (i=0; i<ram->L; i++) ram->b[i]=0;
}
void RomF2Di_set (RomF2Di* ram, int_4 n, int_4 m, real_4 d) {
  int_4 i = m*ram->N+n;
  ram->b[i] = d;
  if (++i==ram->L) ram->b[i] = 2*ram->b[i-1] - ram->b[i-2];
}
real_4 RomF2Di_get (RomF2Di* ram, int_4 n, int_4 m) {
  return ram->b[m*ram->N+n];
}
real_4 RomF2Di_getrN (RomF2Di* ram, real_4 n, int_4 m) {
  real_4 p,d0,d1;
  real_4 nN = n*ram->N;
  int_4  n0 = (int_4)nN;
  int_4  mn = n0+(m*ram->N);
  if (mn>=ram->L) return ram->b[ram->L];;
  p  = nN-n0;
  d0 = ram->b[mn];
  d1 = ram->b[mn+1];
  return d0 + (d1-d0)*p;
}
void RomF2Di_getrNc (RomF2Di* ram, real_4 n, int_4 m, int_1 cx, CxReal_4 *c) {
  c->x = RomF2Di_getrN (ram,n,m);
  c->y = cx? RomF2Di_getrN (ram,n,m+ram->MC) : c->x;
}
int_4 RomF2Di_upload (RomF2Di* ram, void *planp) {
  int len=3;	/* Only N,M,b in actual HW */
  HW_loadHdr(planp,len);
  HW_loadInt(planp,ram->N);
  HW_loadInt(planp,ram->M);
  HW_loadBuf(planp,ram->b,'f',1);
  return len;
}

Viterbi* Viterbi_new () {
  int_4 sStr = sizeof(Viterbi);
  int_1* ptr = zalloc(0+sStr);
  return (Viterbi*)ptr;
}
int_1 Viterbi_process (Viterbi *vit, int_1 bi, int_1 pflg, int_1 vflg) {
  return (0x8|bi);
}

TapBuf* TapBuf_new (int_4 ord, int_4 iw, int_4 ow, int_4 dly) {
  int_4 sStr = sizeof(TapBuf);
  int_4 N = (1<<ord);
  int_4 n = N+iw;
  int_4 len = n*4;
  int_1* ptr = zalloc(len+sStr);
  TapBuf* tb = (TapBuf*)ptr;
  tb->IW=iw;
  tb->N=N;
  tb->n=n;
  tb->b=(real_4*)(ptr+sStr);
  tb->bsigx=ZBUFSIG;
  tb->bsize=n*4;
  return tb;
}
void TapBuf_setLength (TapBuf *tb, int_4 len) {
  tb->n = len+tb->IW;
}
void TapBuf_load (TapBuf *tb, real_4 value) {
  int_4 i; for (i=tb->n-1; i>0; i--) tb->b[i]=tb->b[i-1];
  tb->b[0] = value;
}
real_4 TapBuf_get (TapBuf *tb, int_4 index) {
  return tb->b[index];
}
int_4 TapBuf_upload (TapBuf* tb, void *planp) {
  int len=2;
  HW_loadHdr(planp,len);
  HW_loadInt(planp,tb->n);
  HW_loadBuf(planp,tb->b,'f',1);
  return len;
}


CxTapBuf* CxTapBuf_new (int_4 ord, int_4 iw, int_4 ow, int_4 dly) {
  int_4 sStr = sizeof(CxTapBuf);
  int_4 N = (1<<ord);
  int_4 n = N+iw;
  int_4 len = n*8;
  int_1* ptr = zalloc(len+sStr);
  CxTapBuf* tb = (CxTapBuf*)ptr;
  tb->IW=iw;
  tb->N=N;
  tb->n=n;
  tb->b=(CxReal_4*)(ptr+sStr);
  tb->bsigx=ZBUFSIG;
  tb->bsize=n*8;
  return tb;
}
void CxTapBuf_setLength (CxTapBuf *tb, int_4 len) {
  tb->n = len+tb->IW;
}
void CxTapBuf_load (CxTapBuf *tb, CxReal_4 *value) {
  int_4 i; for (i=tb->n-1; i>0; i--) tb->b[i]=tb->b[i-1];
  tb->b[0] = *value;
}
void CxTapBuf_loadA (CxTapBuf *tb, CxReal_4 **value, int_4 nv) {
  int_4 i; for (i=tb->n-1; i>=nv; i--) tb->b[i]=tb->b[i-nv];
  int_4 j; for (i=nv-1,j=0; i>=0; i--,j++) tb->b[i] = *value[j];
}
void CxTapBuf_set (CxTapBuf *tb, int_4 index, CxReal_4 *value) {
  tb->b[index] = *value;
}
void CxTapBuf_setxy (CxTapBuf *tb, int_4 index, real_4 x, real_4 y) {
  tb->b[index].x = x;
  tb->b[index].y = y;
}
void CxTapBuf_get (CxTapBuf *tb, int_4 index, CxReal_4 *value) {
  *value = tb->b[index];
}
void CxTapBuf_get2 (CxTapBuf *tb, int_4 index, CxReal_4 *value, CxReal_4 *value2) {
  *value = tb->b[index];
  *value2 = tb->b[index+1];
}
void CxTapBuf_getrN (CxTapBuf *tb, real_4 ifps, int_4 index, CxReal_4 *value) {
  int_4 off=(int_4)ifps; if (index>0) index-=off;
  *value = tb->b[index];
}
int_4 CxTapBuf_upload (CxTapBuf* tb, void *planp) {
  int len=2;
  HW_loadHdr(planp,len);
  HW_loadInt(planp,tb->n);
  HW_loadBuf(planp,tb->b,'f',1);
  return len;
}


StatBuf* StatBuf_new (int_4 ord) {
  int_4 sStr = sizeof(StatBuf);
  int_4 N = (1<<ord);
  int_4 len = N*4;
  int_1* ptr = zalloc(len+sStr);
  StatBuf* sb = (StatBuf*)ptr;
  sb->N=N;
  sb->n=N;
  sb->b=(real_4*)(ptr+sStr);
  sb->bsigx=ZBUFSIG;
  sb->bsize=N*4;
  return sb;
}
void StatBuf_setLength (StatBuf *sb, int_4 len) {
  sb->n = len;
  sb->m = len>>1;
}
void StatBuf_load (StatBuf *sb, real_4 value) {
  int_4 i; for (i=sb->n-1; i>0; i--) sb->b[i]=sb->b[i-1];
  sb->b[0] = value;
}
real_4 StatBuf_getMid (StatBuf *sb) {
  return sb->b[sb->m];
}
real_4 StatBuf_getMin (StatBuf *sb) {
  real_4 v = sb->b[0];
  int_4 i; for (i=1; i<sb->n; i++) if (sb->b[i]<v) v=sb->b[i];
  return v;
}
real_4 StatBuf_getMax (StatBuf *sb) {
  real_4 v = sb->b[0];
  int_4 i; for (i=1; i<sb->n; i++) if (sb->b[i]>v) v=sb->b[i];
  return v;
}
real_4 StatBuf_getMinFilt (StatBuf *sb, int_4 n1, int_4 n2) {
  real_4 v = 1e32,vx;
  int_4 i; for (i=n1; i<n2; i++) {
    vx = sb->b[i];
    if (sb->b[i-1]>vx) sb->b[i-1]=vx;
    if (sb->b[i+1]>vx) sb->b[i+1]=vx;
    if (vx<v) v=vx;
  }
  return v;
}
real_4 StatBuf_getMaxFilt (StatBuf *sb, int_4 n1, int_4 n2) {
  real_4 v = -1e32,vx;
  int_4 i; for (i=n1; i<n2; i++) {
    vx = sb->b[i];
    if (sb->b[i-1]<vx) sb->b[i-1]=vx;
    if (sb->b[i+1]<vx) sb->b[i+1]=vx;
    if (vx>v) v=vx;
  }
  return v;
}
real_4 StatBuf_getMinM (StatBuf *sb, int_4 mode) {
  if (mode>0) return StatBuf_getMinFilt(sb,1,sb->m);
  if (mode<0) return StatBuf_getMinFilt(sb,sb->m,sb->n-1);
  return fmaxf( StatBuf_getMinFilt(sb,1,sb->m),StatBuf_getMinFilt(sb,sb->m,sb->n-1) );
}
real_4 StatBuf_getMaxM (StatBuf *sb, int_4 mode) {
  if (mode>0) return StatBuf_getMaxFilt(sb,1,sb->m);
  if (mode<0) return StatBuf_getMaxFilt(sb,sb->m,sb->n-1);
  return fminf( StatBuf_getMaxFilt(sb,1,sb->m),StatBuf_getMaxFilt(sb,sb->m,sb->n-1) );
}
int_4 StatBuf_getMinI (StatBuf *sb, int_4 inc) {
  int_4 i=0,j=sb->m,n=j-1;
  real_4 v,vmin=sb->b[j];
  for (i=1; i<n; i++,j+=inc) {
    v = sb->b[j];
    if (v>vmin) break;
    vmin=v;
  }
  return i;
}
int_4 StatBuf_upload (StatBuf* sb, void *planp) {
  int len=3;
  HW_loadHdr(planp,len);
  HW_loadInt(planp,sb->n);
  HW_loadInt(planp,sb->m);
  HW_loadBuf(planp,sb->b,'f',1);
  return len;
}


RamBuf* RamBuf_new (int_4 ord, int_4 iw, int_4 ow) {
  int_4 sStr = sizeof(RamBuf);
  int_4 N = (1<<ord);
  int_4 len = N*4;
  int_1* ptr = zalloc(len+sStr);
  RamBuf* tb = (RamBuf*)ptr;
  tb->N=N;
  tb->n=N;
  tb->b=(real_4*)(ptr+sStr);
  tb->bsigx=ZBUFSIG;
  tb->bsize=len;
  return tb;
}
void RamBuf_setLength (RamBuf *tb, int_4 len) {
  tb->n = len;
}
void RamBuf_setRdOnly (RamBuf *tb, int_4 ro) {
  tb->ro = ro;
}
void RamBuf_set (RamBuf *tb, int_4 index, real_4 value) {
  if (tb->ro)  return;
  tb->b[index] = value;
}
real_4 RamBuf_get (RamBuf *tb, int_4 index) {
  return tb->b[index];
}
int_4 RamBuf_upload (RamBuf* tb, void *planp) {
  int len=2;
  HW_loadHdr(planp,len);
  HW_loadInt(planp,tb->n);
  HW_loadBuf(planp,tb->b,'f',1);
  return len;
}


CxRamBuf* CxRamBuf_new (int_4 ord, int_4 iw, int_4 ow) {
  int_4 sStr = sizeof(CxRamBuf);
  int_4 N = (1<<ord);
  int_4 len = N*8;
  int_1* ptr = zalloc(len+sStr);
  CxRamBuf* tb = (CxRamBuf*)ptr;
  tb->N=N;
  tb->n=N;
  tb->ro=0;
  tb->b=(CxReal_4*)(ptr+sStr);
  tb->bsigx=ZBUFSIG;
  tb->bsize=len;
  return tb;
}
void CxRamBuf_setLength (CxRamBuf *tb, int_4 len) {
  tb->n = len;
}
void CxRamBuf_setRdOnly (CxRamBuf *tb, int_4 ro) {
  tb->ro = ro;
}
void CxRamBuf_set (CxRamBuf *tb, int_4 index, CxReal_4 *value) {
  int ro = tb->ro;
  if (ro==0) tb->b[index] = *value;
  else if (ro==1) tb->b[index].y = value->y;
  else if (ro==2) tb->b[index].x = value->x;
}
void CxRamBuf_setxy (CxRamBuf *tb, int_4 index, real_4 vx, real_4 vy) {
  int ro = tb->ro;
  if ((ro&1)==0) tb->b[index].x = vx;
  if ((ro&2)==0) tb->b[index].y = vy;
}
void CxRamBuf_get (CxRamBuf *tb, int_4 index, CxReal_4 *value) {
  *value = tb->b[index];
}
void CxRamBuf_print (CxRamBuf *tb) {
}
int_4 CxRamBuf_upload (CxRamBuf* tb, void *planp) {
  int len=2;
  HW_loadHdr(planp,len);
  HW_loadInt(planp,tb->n);
  HW_loadBuf(planp,tb->b,'f',1);
  return len;
}

#define SBUFSZ 1024
static char sbuf[SBUFSZ];
static int soff=0;

String* String_alloc (int len) {
  String* sa; len++;
  if (soff+len>SBUFSZ) soff=0;
  sa = sbuf+soff;
  soff += len;
  return sa;
}

String* String_toLowerCase (String* sa) {
  int i,len = strlen(sa); char c;
  String* sb = String_alloc(len);
  for (i=0; i<=len; i++) { c=sa[i]; sb[i] = (c>='A' && c<='Z')? c+32 : c; }
  return sb;
}

String* String_toUpperCase (String* sa) {
  int i,len = strlen(sa); char c;
  String* sb = String_alloc(len);
  for (i=0; i<=len; i++) { c=sa[i]; sb[i] = (c>='a' && c<='z')? c-32 : c; }
  return sb;
}

String* String_format (String* format, ...) {
  String* sb = String_alloc(128);
  va_list args;
  va_start (args,format);
  vsprintf(sb,format,args);
  va_end(args);
  return sb;
}

/*
int_4 core_setkey (void *handle, char* key, void* value, int len) {
  printf("Core set of key=%s not handled\n",key);
  return -1;
}
int_4 core_getkey (void *handle, char* key, void* value, int len) {
  printf("Core get of key=%s not handled\n",key);
  return -1;
}
*/

int_4 Core_set (void *handle, String* key, Value* value) {
  int stat=0;
  CorePlan *plan = (CorePlan*)handle;
  if (plan->halo.scid>0) return -1;
  printf("Core set of key=%s not handled\n",key); 
  stat=-1;
  return stat;
}
int_4 Core_get (void *handle, String* key, Value* value) {
  int stat=0, lk=strlen(key);
  CorePlan *plan = (CorePlan*)handle;
  if (plan->halo.scid>0) return -1;
  if (strcmp(key,"NAME")==0 || strcmp(key,"S:NAME")==0) Value_setS(value,plan->halo.name);
  else if (lk>1 && key[lk-1]=='?') stat=-1;	/* not required */
  else if (strcmp(key,"TIME")==0 || strcmp(key,"D:TIME")==0) Value_setD(value,0,cpuTime());
  else if (strcmp(key,"SSM")==0  || strcmp(key,"F:SSM")==0) Value_setF(value,0,(float)fmod(cpuTime(),86400.0));
  /* else if (strcmp(key,"IMPL")==0  || strcmp(key,"L:IMPL")==0) Value_setL(value,0,plan->halo.impl); */
  else { printf("Core get of key=%s not handled\n",key); stat=-1; }
  return stat;
}

void Core_set_value (void *handle, char type, String* prefix, String* key, void *data, int_4 len) {
  HALO *pt = (HALO*)handle;
  Value val; char pkey[40];
  sprintf(pkey,"%c:%s%s",type,prefix,key);
  val.spa=1; val.mode='S'; 
  val.type=type; val.bps=val.bpa=val.len=len;
  val.b=data;
  (*pt->set) (handle, pkey, &val);
}

int_4 Core_setKeyFile (void *handle, String* fn, int_4 verbose) {
  CorePlan *plan = (CorePlan*)handle;
  int i,stat=-1,lval; float fval; double dval;
  char line[80],prefix[40],key[40],value[80],*s;
  FILE *fd;
  strcpy(line,fn);
  if (strncmp(line,"nxm.",4)==0) {
    s=getenv("NMROOT");
    if (s==NULL) s=getenv("ICEROOT");
    if (s==NULL) sprintf(line,"/%s",fn);
    else sprintf(line,"%s/%s",s,fn);
    for (i=strlen(line)-5; i>0; i--) if (line[i]=='.') line[i]='/';
  }
  fd = fopen(line,"r");
  if(fd==NULL) { printf("Err opening setKeyFile=%s\n",line); return stat; }
  prefix[0]=0;
  for (stat=0; ;stat++) {
    if (fgets(line,80,fd)==NULL) break;
    s = strstr(line,"\n"); if (s!=NULL) s[0]=0;
    s = strstr(line,"!"); if (s!=NULL) s[0]=0;
    s = strstr(line,"="); if (s==NULL) continue; else s[0]=0;
    sscanf(line,"%s",key);
    sscanf(s+1,"%s",value);
         if (strstr(value,"{")==value) sprintf(prefix,"%s_",key);
    else if (strstr(key,"L:")==key) { lval=core_s2d(value); Core_set_value(handle,'L',prefix,key+2,&lval,4); }
    else if (strstr(key,"F:")==key) { fval=core_s2d(value); Core_set_value(handle,'F',prefix,key+2,&fval,4); }
    else if (strstr(key,"D:")==key) { dval=core_s2d(value); Core_set_value(handle,'D',prefix,key+2,&dval,8); }
    else if (strstr(key,"S:")==key) {                       Core_set_value(handle,'S',prefix,key+2,value,8); }
    else printf("Unhandled key=%s value=%s pair\n",key,value);
  }
  fclose(fd);
  return stat;
}
#endif
