//************
//  Description: Driver functions for the ICE-PIC DSP card
//
//  Author:	Jeff Schoen  3/13
//
//  $Revision:	$
//  $Modtime:	$
//
//************
#include <stdio.h>
#include <primitive.h>
#include <fintrinsics.h>
#define _XMIDAS
#include "icelib.h"
#include "sinkpic.h"
#include "qmessages.h"


char *s2c (string s) { return (char*)s.c_str(); }
char *l2c (int i) { static char cstr[8]; sprintf(cstr,"%d",i); return cstr; }

void mainroutine ()
{
  sinkpic().run();
}

sinkpic::sinkpic() { p=&ps; }

void sinkpic::run() 
{
  CPHEADER HIN,HOUT,HRAM,HWF;
  string cnum,label;
  int_1  *buf,*out;
  bool_4 spinning, wrap, setbreak, donereading, inmem, sss, xts, stats;
  int_4  throttle, mlost, trate, csize, lbuf, gain, xape;
  real_4 wait, pause, delay, fstat;
  real_8 freq, srate, onext, etime[2];

  // form device alias
  m_hwf_open (HWF);
  m_hwf_alias (HWF, m_apick(3), str);
  m_hwf_close (HWF);

  if (m_get_uswitch("PORT",label) > 0) {
    str += ",PORT="+label;
  }
  if (m_get_uswitch("FLAGS",tmpstr) > 0) {
    str = tmpstr+","+str;
  }
  i = m_get_uswitch("TC",tcmode);
  if (i>=0) {
    if (i==0) tcmode = "SDN4";
    tcoff = m_get_dswitch("TCOFF");
    if (tcoff==-1) {
      m_now (time1,time2);
      i = m_times2tod (time1,time2,label);
      label = label.substr(0,5)+":01:01::00:00:00";
      i = m_tod2times (label,tcoff,time2);
      if (tcmode.substr(0,3)=="CPU") tcoff = 0;
    }
    str += ",TC="+tcmode;
  } else {
    tcmode = "OFF";
  }

  flags = 0;
  i = m_get_switch("VERBOSE");
  if (i>=0) { str+=",VERBOSE="; str+=l2c(i); }
  str += ",";

  // open device
  status = pic_open (p,s2c(str),(int_4*)&Mc->break_,flags);
  if (status <= 0) m_error ("Opening port: "+str);

  status = pic_reset (p, 0);
  type  = p->ptype;
  port  = p->pindex;

  // open and check input file
  m_init (HIN, m_apick(1),"1000,2000","SP,SB,SI,SL,CB,CI",0);
  m_open (HIN, HCBF_INPUT);
  m_get_epoch (HIN, etime[1],etime[2]);

  // get reduction 
  ired   = m_lpick(4);
  if (ired<-1) m_error("Sinkpic no longer perform blocking. Use Fortran version.");
  if (ired==0) ired = 1;

  // get freq and gain
  freq   = m_dpick(5);
  gain   = m_lpick(6);

  // get the transfer length switch
  tl = m_get_switch("TL");
  throttle = m_get_switch("THROTTLE");
  setbreak = m_get_pswitch("BREAK");
  wrap = m_get_pswitch("WRAP");
  donereading = false;

  // fill out header and open ram file
  m_init (HRAM, m_apick(2),"1000,2000","SP,SB,SI,SL,CB,CI",0);
  m_open (HRAM, HCBF_INPUT+HCBF_APPEND);
  rate = 1.0 / max(1.e-10,HRAM.xdelta);
  bps = HRAM.bps;
  spa = HRAM.spa;
  ape = HRAM.ape;
  spe = ape * spa;
  epb = HRAM.size;
  dbpe = HRAM.dbpe;
  delta = HRAM.xdelta;
  lbuf = min(L_bbuf, NINT(epb*dbpe/4));
  buf = NULL;
  xape = HRAM.ape / HIN.ape;

  // fill out header and open output file
  m_propagate (HRAM,HOUT);
  if (m_get_uswitch("MON",HOUT.file_name)<0) HOUT.file_name="";
  m_open (HOUT, HCBF_OUTPUT+HCBF_OPTIONAL);

  // check for consistent file parameters
  if (HIN.format != HRAM.format) m_error("In/Out file formats do not match");

  // get startup modes
  mpp = 0;
  msync = 0;
  msyncid = m_get_switch("MASTER");
  if (msyncid>0) msync = 1;
  i = m_get_uswitch("SLAVE",cnum);
  if (i>=0) msync = 2;
  sss = (i==2 && cnum=="SS"); // same side slave
  xts = (i==2 && cnum=="XT"); // external trigger slave
  mgo = -1;			// non-piped default
  if (Mc->mode!=0) mgo = 2;	// piped default
  mgo = m_get_switch_def("REPLAY",mgo);
  if (msync>=2) mgo = 0;		// slave wait

  // get sync results label
  if (m_get_uswitch ("SYNC", synclab)<=0) synclab = "";
  if (m_get_uswitch ("LOST", lostlab)<=0) lostlab = "";
  if (m_get_uswitch ("PFULL",statlab)<=0) statlab = "";

  // start-up wait time 
  wait  = m_get_dswitch("WAIT");
  if (wait<0) wait  = m_get_dswitch("PAUSE");
  pause = m_get_dswitch_def("POLL",0.025);
  delay = m_get_dswitch_def("DELAY",0.20);
  stats = m_get_pswitch("STATS");

  // init the PIC device parameters
  dir   = 1;	// Write only primitive
  bits  = dbpe / ape / spa * 8;
  if (spa == 2) bits = -bits;
  bits  = m_get_switch_def ("BITS",bits);
  srate = m_get_dswitch_def ("SRATE",rate);
  if (ired<0) ired = srate/rate;
  trate = m_get_switch_def ("TRATE", max(10000,NINT(srate/8)) );
  fscale = (spa==2)? 1/srate : 2/srate;
  dmac  = -1;
  nblock = max(0,tl) * dbpe / ape;

  // init the PIC device for IO
  flags = 0;
  if (msync>1) flags = FLG_SGO;
  if (sss) flags = FLG_RGO;
  if (xts) flags = FLG_XTGO;
  if (throttle==1 || throttle==4) flags |= FLG_NCCLK;
  dmac = pic_ioport (p, type, port, dmac, dir, bits, NINT(srate), freq*fscale, ired, gain, flags);
  if (dmac<=0) m_error ("Bad I/O Port parameters");
  fmode = m_get_switch_def("FMODE",1);
  status = pic_mapfile (p, &map, &HRAM.hcb, fmode);
  if (status<=0) m_error ("Unacceptable DMA output file");
  nbytes = map.bytes;
  status = pic_dmasetup (p, dmac, dir, &map, nblock, 0);
  if (status<=0) m_error ("Unacceptable channel for DMA");

  // set up real time controls
  idm = m_mwinit("Replay Mode",mgo,"REPLAY;Play One,Continuous,Stop Top,Stop Now,Spin,Reload,ReStart,Abort");
  ids = m_mwinit("Sync Mode",msync,"Master,Slave Wait,Slave Run");
  idr = m_dwinit("Sample Rate", srate, 1.0, 100.e6, 1.e3);
  idd = m_lwinit("Decimation", ired, 1, 32768, 1);
  idc = m_lwinit("Curr Cycle", 0, 1, -1, 1);
  idlh = m_lwinit("Lost HostBuf", 0, 1, -1, 1);
  idlc = m_lwinit("Lost CardBuf", 0, 1, -1, 1);

  // report the DMA channel
  if (m_get_uswitch("DMAC",tmpstr)>0) m_lrslt(tmpstr,dmac);

  // initialize parameters
  nbuf = 0;
  nlost = 0;
  mlost = 0;
  npass = 0;
  ncycle = 1;
  lastout = 0;
  outoff = 0;
  ramsync (P_SYNCINIT);
  if (etime[1]>0.0) pic_setkey (p, dmac, KEY_TCOFF, etime, 16);
  csize = pic_getkeyl (p, dmac, KEY_CBUFSZ);

  // is data already in memory
  inmem = HRAM.file_name == HIN.file_name;
  if (inmem && HRAM.mode != 1) {
    out = (int_1*)map.vaddr;
    ngot = -1;
    m_grab(HIN,out,1.0,NINT(HIN.size),ngot);
  }

  // signal start of processing block
  m_sync();

  // initial slave sync up
  if (msyncid>0) m_mwput(msyncid-1,mgo);
  if (wait>0) m_pause(wait);

  // top of throttle loop
  TOP:
  while (mgo==0 || mgo==3 || mgo==4 || mgo==7) {
    update(0);
    isok = m_mwget(idm,mgo);
    if (mgo==7) { // restart ?
      mgo = 2; // already stopped - just start
      m_mwput(idm,mgo);
    } else if (mgo==3 || mgo==4) {
      mgo = 0;
      m_mwput(idm,mgo);
    }
    if (msyncid>0) m_mwput(msyncid-1,mgo);
    if (Mc->break_ || mgo==8) goto DONE;
    m_pause(pause);
  }
  dstart = min( 0.5*HRAM.size, delay*rate );
  dstart = min( HRAM.size, max(dstart, 2.0*csize/HRAM.dbpe) );
  if (throttle==1) dstart = HRAM.size+1;
  if (throttle==2) dstart = HRAM.size;
  if (throttle==4) dstart = 0;
  if (HIN.pipe==0 && wrap<=0) dstart = min(dstart,HIN.size*HIN.ape/ape);
  spinning = (mgo==5);

  // make sure the slaved PIC device is running
  if (msyncid>0) {	
    isok = m_mwget (msyncid,ssync);
    while (ssync==2 && !Mc->break_) {	// wait for slave
      m_pause (pause);
      isok = m_mwget (msyncid,ssync);
    }
  }

  // update the pass through RAM buffer count
  NEXT:
  npass = npass + 1;
  m_lwput (idc,npass);

  // calculate output frame skip 
  iskip = m_get_switch("SKIP");
  mdbpe = max( HIN.dbpe, HRAM.dbpe );
  iper = tl;
  if (iper<=0) iper = max(1, NINT(lbuf/mdbpe));
  ibuf = iper*mdbpe;
  if (ibuf>nbuf) {
    if (buf!=NULL) free (buf);
    nbuf = ibuf;
    buf = (int_1*)malloc (nbuf);
  }

  // set xfer parameters
  dnext = onext = 0;
  ntogo = HRAM.size;

  // debug statistics output
  if (stats || statlab!="") {
    status = pic_dmafuncx (p,dmac,DMA_STATUS);
    fstat = ((float)status)*100/nbytes;
    if (stats) printf ("Sinkpic top pass=%d cyc=%d favg=%f\n",npass,ncycle,fstat);
    if (statlab!="") m_frslt (statlab,fstat);
  }

  // warn if we are falling way behind
  if (mpp==0) {
    npass = 1;
  } else if (throttle==1) {

  } else if (ncycle>npass+nlost) {
    if (donereading) {
      mgo = 8;
      m_mwput(idm,mgo);
    } else {
      status = m_l2a (ncycle-(npass+nlost),cnum);
      if (!inmem) m_warning ("Fell behind "+cnum+" RAM buffers");
      nlost = (ncycle-npass);
      m_lrslt (lostlab,nlost);
    }
  } else {
    // wait for current data pointer
    while (npass+nlost>ncycle+1 && !Mc->break_) {
      ramsync (P_SYNCSTAT);
      m_pause (pause);
    }
  }

  // loop through buffers of data  (data buffer loop)
  while (ntogo>0 && !Mc->break_) {
    nget = min(iper,ntogo);
    ramsync (P_SYNCSTAT);

    // get new real time controls
    update(1);

    status = pic_dmafuncx (p,dmac,DMA_STATUS);

    // monitor output
    if (HOUT.open && status<(onext-iskip)*dbpe || status>=(onext+1)*dbpe) {
      out = (int_1*)(map.vaddr + (int_8)(onext*dbpe));
      m_filad(HOUT,out,1);
      onext += iskip; if (onext>=HRAM.size) onext = 0;
    }

    // dont test pointer if pre-filling the buffer
    if (mpp==0) {

    // check for falling behind
    } else if (npass+nlost<=ncycle && throttle!=1) {
      nnext = dnext*dbpe;
      if (status > nnext) {
        if (donereading) {
          mgo = 8;
          m_mwput(idm,mgo);
        } else {
          if (!inmem) m_warning ("Falling behind one RAM buffer");
          nlost = nlost + 1;
          m_lrslt (lostlab,nlost);
        }
      }

    // wait until space is available
    } else if (npass+nlost>=ncycle+1) {
      nnext = (dnext+nget)*dbpe;
      status = pic_dmafuncx (p,dmac,DMA_STATUS);
      while ( status<nnext && npass+nlost>=ncycle+1 && !Mc->break_) {
        update(1);
        m_pause(pause);
        ramsync (P_SYNCSTAT);
        if (throttle==1 && status==0) status = HRAM.size*dbpe;
        if (throttle==3) {
          pic_setkeyl (p, dmac, KEY_RATE, NINT(rate));
          throttle = 2;
        }
      }
    }

    // check for end of file handling
    if (HIN.pipe == 0) {
      if (HIN.offset >= HIN.size) {
        if (wrap) {
          HIN.offset = 0;
        } else {
          donereading = true;
          nget = 0;
        }
      }
      if (HIN.offset+nget*xape > HIN.size) nget = (HIN.size-HIN.offset)/xape;
    }
    if (donereading && throttle==3) {
      pic_setkeyl (p, dmac, KEY_RATE, NINT(rate));
      throttle = 2;
    }
    if (throttle==4 && nget>0) {
      m_hcbfunc(HIN,HCBF_AVAIL,&i);
      if (i<nget*xape) nget = i/xape;
    }

    // set the RAM buffer pointer
    out = (int_1*)(map.vaddr + (int_8)(dnext*dbpe));
    ngot = nget;
    ntogo = ntogo - nget;
    dnext = dnext + nget;

    // not ready yet
    if (nget == 0) {
      m_pause(pause);

    // spin mode
    } else if (spinning || inmem) {

    // large or no skip
    } else {	
      HIN.xfer_len = nget*xape;
      if (HIN.format == HRAM.format) {
        m_grabx (HIN,out,ngot);
      } else {
        m_grabx (HIN,buf,ngot);
        m_reformat (buf,HIN.format,out,HRAM.buf_type,ngot);
      }
    }

    if (throttle==4 && nget>0) i = pic_setkeyl (p,dmac,KEY_INBYTE,NINT(dnext*dbpe));

  // make sure the PIC device is running (except in throttle)
    if (mgo!=6 && mpp!=1 && dnext>=dstart) {
      mpp = 1;
      if (mgo<0) {
        status = pic_dmafunc (p,dmac,-mgo);
      } else if (throttle == 4) {
        status = pic_dmafunc (p,dmac,DMA_ONDEMAND);
      } else if (mgo==2 || mgo==5) {
        status = pic_dmafunc (p,dmac,DMA_CONTINUOUS);
      } else {
        status = pic_dmafunc (p,dmac,DMA_ONESHOT);
      }
      ramsync (P_SYNCSET);
      if (msync>1) m_mwput (ids,3);	// report slave running
    } 

  // check throttle control widget
    isok = m_mwget(idm,mgo);
    if (isok>0 && msyncid>0) {
      isok = m_mwget (msyncid+3,i);
      while (i<npass && !Mc->break_) {
        m_pause(pause);
        isok = m_mwget (msyncid+3,i);
      }
      m_mwput(msyncid-1,mgo);
    }
    if (mgo==4 || mgo==8) ntogo = 0;
    if (throttle==2 && mgo==2 && !donereading) {
      status = pic_dmafuncx (p,dmac,DMA_STATUS);
      // at 80% into buffer, check to see if we are keeping up
      if (status > nbytes*.8 && (status-dnext*HIN.dbpe)> nbytes*.1) {
        // fell back by > 10%, temporarily slow the clock
        pic_setkeyl (p, dmac, KEY_RATE, trate);
        throttle = 3;
      }
    }

  }

  // handle throttle modes
  if (throttle==1 && mgo==2) {
    if (mpp>0) {	// restart transfer
      status = pic_dmafunc (p,dmac,DMA_WAIT);
      status = pic_dmafunc (p,dmac,DMA_RESHOT);
    } else {			// start transfer
      status = pic_dmafunc (p,dmac,DMA_ONESHOT);
    }
    ramsync (P_SYNCSET);
    if (msync>1) m_mwput (ids,3);	// report slave running
    mpp = 1;
  }

  // update the status widgets
  m_lwput(idlh,nlost);
  i = pic_dmafunc(p,dmac,DMA_LOST);
  m_lwput(idlc,i);
  if (i>mlost) m_warning ("Lost one or more CardBuffers");
  mlost = i;

  // stop PIC if necessary
  if ((mgo==1 || (mgo>=3 && mgo!=5)) && mpp==1) {
    if (msyncid>0) {
      isok = m_mwget (msyncid+3,i);
      isok = m_mwget (msyncid,ssync);
      while (ssync==3 && !Mc->break_) {	// wait for stop
        m_pause (pause);
        isok = m_mwget (msyncid,ssync);
      }
      isok = m_mwget (msyncid+3,i);
    }
    while (ncycle <= npass+nlost && mgo <= 1) {
      m_pause(Mc->pause);
      ramsync (P_SYNCSTAT);
    }
    mpp = 0;
    // use delay on donereading abort to output extra samples 
    if (mgo==8 && donereading) m_pause(delay);
    if (mgo==1) status = pic_dmafunc (p,dmac,DMA_WAIT);
    status = pic_dmafunc (p,dmac,DMA_STOP);
    status = pic_dmafunc (p,dmac,DMA_RESET);
    npass = 0;
    nlost = 0;
    mlost = 0;
    ncycle = 1;
    outoff = lastout;
    if (msync>1) {
      if (mgo==7) {
        mgo = 0; // slave reports stopped
        m_mwput(idm,mgo);
      }
      m_mwput (ids,2); // report slave wait
    }
  }

  // finish RELOAD pass
  if (mgo==6) {
    npass = 0;
    nlost = 0;
    mlost = 0;
    ncycle = 1;
    mgo = 4;
  }

  // go back for replay
  if (mgo>=0) {
    if (mgo==1 || mgo==3 || mgo==4) {
      mgo = 0;
      m_mwput(idm,mgo);
    }
    if (mgo==2 && !Mc->break_) goto NEXT;
    if (mgo==5 && !Mc->break_) goto NEXT;
    if (mgo!=8 && !Mc->break_) goto TOP;
  } else {
    mgo = mgo + 1; // replicate mode
    m_mwput(idm,mgo);
    if (mgo<0 && !Mc->break_) goto TOP;
    status = pic_dmafunc (p,dmac,DMA_WAIT);
  }

  // close PIC DEVICE
  DONE:	
  status = pic_dmafunc (p,dmac,DMA_CANCEL);
  status = pic_mapfile  (p, &map, &HRAM.hcb, -fmode);
  status = pic_close (p);

  // close input/output files
  m_close(HIN);
  m_close(HOUT);
  m_close(HRAM);

  if (setbreak) Mc->break_ = true;
}


void sinkpic::ramsync (int_4 mode)
{
  real_4 time;

  if (mode==P_SYNCINIT) {		// to initialize
    statuslast = 0;
    return;
  }
  status = pic_dmafuncx (p,dmac,DMA_STATUS);
  if (status==-1) m_error ("ICE-PIC is down");
  status = max((int_8)0,min(nbytes,status));
  if (status<statuslast || mode==P_SYNCSET) { 	// next buffer
    m_secnds(0.0,time);
    time -= status/(rate*dbpe/ape);
    m_frslt (synclab, time);
    if (mode==P_SYNCSET) return;
    ncycle = pic_dmafunc (p,dmac,DMA_CYCLE)+1;	// 0->1 based
  }
  statuslast = status;
}


void sinkpic::update (int_4 mode)
{
  real_8 time, trigger, soy, fsoy, sample;
  int_4 ratex, iredx, iredorig, lostoff;

  iredorig = ired;
  // look for messages
  if (Mu->id <= 0) goto DONE;
  if (m_get_msg(MQH,MQD,0.0,0)<=0) goto DONE;
  if (MQH.name == "MGO") {
    if (MQH.ndata>0 && MQD.dbuf[0]>0) {
      trigger = MQD.dbuf[0];
    } else {
      trigger = 0;
    }
    NEXT:
    m_now (time1,time2);
    time = time1 + time2;
    if (trigger==0) {
    } else if (trigger-time<=0) {
      // take it now
    } else if (trigger-time>600.0) {
      m_warning ("Huge trigger offset in message");
    } else {
      m_pause (Mc->pause);
      goto NEXT;
    }
    m_send_msgl ("=MGO",0,MQH.sender,MQH.info,0,1,"D",(void*)&time,0);
    m_mwput (idm, MQH.info);
  } else if (MQH.name == "RATE") {
    m_dwput (idr, MQD.dbuf[0] );
  } else if (MQH.name == "DEC") {
    m_lwput (idd, NINT(MQD.dbuf[0]) );
  } else if (MQH.name == "TC") {
    trigger = MQD.dbuf[0];
    if (trigger==-1) trigger = lastout;
    lostoff = nlost * epb;
    sample = (trigger-outoff+lostoff) * ape;
    if (ired>1 && type!=IOPT_TUNER) sample = sample * ired;
    status = pic_tc (p, dmac, &sample, &delta, &soy, &fsoy, FLG_TCINTERP);
    MQD.dbuf[0] = trigger;
    MQD.dbuf[1] = tcoff + soy;
    MQD.dbuf[2] = fsoy;
    if (status<0) MQD.dbuf[1] = 0;
    m_send_msgl ("=TC",0,MQH.sender,status,0,3,"D",(void*)MQD.dbuf,0);
  } else if (MQH.name == "RFFREQ") {
    pic_setkeyd (p, dmac, KEY_RFFREQ, MQD.dbuf[0]);
  } else if (MQH.name == "RFGAIN") {
    pic_setkeyl (p, dmac, KEY_RFGAIN, MQD.ibuf[0]);
  } else if (MQH.name == "NFGAIN") {
    pic_setkeyl (p, dmac, KEY_NFGAIN, MQD.ibuf[0]);
  } else if (MQH.name == "FREQ") {
    pic_setkeyd (p, dmac, KEY_FREQ, fscale*MQD.dbuf[0]);
  } else if (MQH.name == "GAIN") {
    pic_setkeyl (p, dmac, KEY_GAIN, MQD.ibuf[0]);
  } else if (MQH.name == "REPLAY") {
    mgo = MQH.info;
  } else if (MQH.name == "EXIT") {
    m_mwput (idm,8);
  } else {
    m_warning ("Unexpected message: "+MQH.name);
  }

  DONE:	// look for widgets
  ratex = m_dwget(idr,rate);
  iredx = m_lwget(idd,ired);
  if (ratex>0) {
    pic_setkeyl (p, dmac, KEY_RATE, NINT(rate));
    delta = (ired/rate);
  }
}

