//************
//  Description: Driver functions for the ICE-PIC DSP card
//
//  Author:	Jeff Schoen  3/13
//
//  $Revision:	$
//  $Modtime:	$
//
//************
#include <stdio.h>
#include <primitive.h>
#include <fintrinsics.h>
#define _XMIDAS
#include "icelib.h"

char *s2c (string s) { return (char*)s.c_str(); }
char *l2c (int i) { static char cstr[8]; sprintf(cstr,"%d",i); return cstr; }

int_4 testit (int_4 *buf, int_4 size, string testfile)
{
  CPHEADER ht;
  string key; char type='L';
  int_4 *buft,ngot,npass,iper,i,j,k,n,skip,mask,md,mt,status=-3;

  m_init (ht,testfile,"1000","SL,SI,SB,SP,CI,CB",0);
  m_open (ht,HCBF_INPUT);
  key="SKIP"; if (m_get_keydata(ht,key,(void*)&skip,4,0,type) != 4) skip = 0;
  key="MASK"; if (m_get_keydata(ht,key,(void*)&mask,4,0,type) != 4) mask = 0;
  iper = ht.size * ht.dbpe / 4;
  npass = size * ht.dbpe / 4 / iper;
  skip = skip * ht.dbpe / 4;
  buft = (int_4*)malloc(iper*4);
  m_grab (ht,(void*)buft,1.0,size,ngot);
  m_close (ht);
	
  k = 0;
  mask = ~mask;
  for (n=0; n<npass; n++) {
    j = n*iper+skip;
    for (i=skip;i<iper;i++,j++) {
      mt = buft[i]&mask;
      md = buf[j]&mask;
      if (mt != md) {
        if (++k>16) return status;
        printf("i=%.8d T=%08x D=%08x xor=%08x\n",j,buft[i],buf[j],buft[i]^buf[j]);
      }
    }
  }
  if (k<=1) status = 1;
  return status;
}

void mainroutine ()
{
  CPHEADER hwf,hcb;
  PICSTRUCT ps, *p=&ps;
  DMAMAP map;

  int_4 i, j, n, status, status_, offset, value, ls, timeout;
  string verb, name, str, label, tmpstr, cnum;
  string fn_in, statlab, tcmode, trigger;
  int_4 nstart, nbytes, errors;
  int_4 port, bits, dec, dmac, dir, flags, dmamode, fmode, dmao;
  int_4 mask, tl, nget, ngot, idb, replay, gain, kick;
  int_4 ndata, block, maxsjump, maxbjump, ntotal;
  int_4 dmap, tcbp, channel, index, values[16];
  int_4 nchan, xfer, pktlen, test;
  real_4 time, timed, rates[2], wait, arate;
  real_8 rate, tune, tcoff, time1, time2, dummy, dfreq;
  real_8 etime[2], delta;
  bool_4 all, tuner;
  char type;
  int_4 *buf,*zbuf;

  // get the action
  verb = m_upick(1);

  // special detect function
  if (verb=="DETECT") {
    status = pic_detect(0);
    return;
  }

  // form device alias
  m_hwf_open (hwf);
  m_hwf_alias (hwf, m_apick(2), str);
  m_hwf_close (hwf);

  if (m_get_uswitch("PORT",label) > 0) {
    str += ",PORT="+label;
  }
  if (m_get_uswitch("FLAGS",tmpstr) > 0) {
    str = tmpstr+","+str;
  }
  i = m_get_uswitch("TC",tcmode);
  if (i>=0) {
    if (i==0) tcmode = "SDN4";
    tcoff = m_get_dswitch("TCOFF");
    if (tcoff==-1) {
      m_now (time1,time2);
      i = m_times2tod (time1,time2,label);
      label = label.substr(0,5)+":01:01::00:00:00";
      i = m_tod2times (label,tcoff,time2);
      if (tcmode.substr(0,3)=="CPU") tcoff = 0;
    }
    str += ",TC="+tcmode;
  } else {
    tcmode = "OFF";
  }

  flags = 0;
  i = m_get_switch("VERBOSE");
  if (i>=0) { str+=",VERBOSE="; str+=l2c(i); }
  str += ",";

  if (m_get_uswitch("STAT",statlab) < 0) statlab = "";
  all = m_get_sswitch ("ALL");

  // open device
  status = pic_open (p,s2c(str),(int_4*)&Mc->break_,flags);
  if (status <= 0) m_error ("Opening port: "+str);

  // show status
  if (verb=="SNIFF") {
    flags = 0;
    if (all) flags = -1;
    status = pic_sniff (p,flags);
  }
  // reset device
  else if (verb=="HALT") {
    status = pic_reset (p,FLG_DISABLE);
  }
  // reset device
  else if (verb=="RESET" || verb=="REBOOT") {
    status = pic_reset (p,0);
    if (status<0) m_warning ("Problem resetting card");
    else if (m_ppick(3)) {
      label = "*_"+m_apick(3);
      status = pic_loadfile (p, s2c(label), FLG_IOC);
    }
  }
  // write register
  else if (verb=="WRITE") {
    name = m_upick(3);
    offset = pic_name2offset(p,s2c(name),0);
    value = m_lpick(4);
    if (m_get_pswitch("ADDR")) value |= (1<<31);
    mask = m_get_switch ("MASK");
    if (mask==-1) status = pic_write (p, offset, &value);
    else          status = pic_writem (p, offset, &value, mask);
    if (Mu->verbose) printf("Wrote Value=%08x at Addr=%08x Status=%08x\n",value,offset,status);
  }
  // read register
  else if (verb=="READ") {
    name = m_upick(3);
    offset = pic_name2offset(p,s2c(name),0);
    label = m_upick(4);
    status = pic_read (p, offset, &value);
    if (Mu->verbose && label == "")
      printf("Read  Value=%08x at Addr=%08x\n",value,offset);
    else
      m_lrslt(label,value);
  }
  // perform tests
  else if (verb=="TEST") {
    value = m_lpick(3);
    label = m_upick(4);
    if (label=="") status = pic_test (p,value,1);
    else {
      status = pic_test (p,value,0);
      m_lrslt(label,status);
    }
  }
  // load SHARC code
  else if (verb=="LOAD" || verb=="LOADSHARC") {
    label  = m_apick(3);
    status = pic_loadfile (p, s2c(label), FLG_SHARC);
  }
  // load IOC program
  else if (verb=="LOADIOC") {
    label  = "*_"+m_apick(3);
    status = pic_loadfile (p, s2c(label), FLG_IOC);
  }
  // load MOD program
  else if (verb=="LOADMOD") {
    label  = m_apick(3);
    status = pic_loadfile (p, s2c(label), FLG_MOD+1);
  }
  // load PM program
  else if (verb=="LOADPM") {
    label  = m_apick(3);
    status = pic_loadfile (p, s2c(label), FLG_PM);
  }
  // load FLASH program
  else if (verb=="LOADFLASH") {
    printf("No longer supported from PICDRIVER in X-Midas. Use $ICEROOT/test/flashpic.\n");
  }
  // check FLASH program
  else if (verb=="CHECKFLASH") {
    label  = m_apick(3);
    status = pic_reset (p,FLG_BOOT);
    status = pic_setkeyl (p,0,KEY_VERBOSE,1);
    value  = pic_getkeyl (p,0,KEY_FLASH);
    if (label == "") printf("Flash CRC=0x%08x\n",value);
    else m_lrslt(label,value);
  }
  // get closest available tuner decimation ratio 
  else if (verb=="TDEC") {
    port = m_lpick(5);
    dec = m_lpick(6);
    status = pic_tuner_dec (p, dec, port, 0);
    m_lrslt (m_upick(4),status);
  }
  // get closest available tuner frequency 
  else if (verb=="TFREQ") {
    port = m_lpick(5);
    tune = m_dpick(7);
    tune = pic_tuner_freq (p, tune, 0);
    m_drslt (m_upick(4),tune);
  }
  // set tuner oversampling ratio 
  else if (verb=="OVSR") {
    value = m_lpick(4);
    port  = m_lpick(5);
    status = pic_tuner_ovsr (p, value, port, 0);
    if (status<0) m_error ("Bad Tuner OVSR parameters");
  }
  // load tuner coefficients
  else if (verb=="LOADFC") {
    fn_in = m_apick(3);
    port = m_lpick(5);
    m_init (hcb, fn_in, "1000","SI",0);
    m_open (hcb, HCBF_INPUT);
    nget = hcb.size; ngot = -1;
    buf = (int_4*)malloc(nget*hcb.bpe);
    m_grab (hcb, (void*)buf, 1.0, nget, ngot);
    m_close (hcb);
    flags = 0;
    if (m_get_pswitch("NCFIR")) flags |= FLG_NCFIR;
    status = pic_loadfc (p, port, (int_2*)buf, nget, flags);
    free (buf);
  }
  // set timer
  else if (verb=="TIMER") {
    status = pic_timer (p, m_lpick(5), m_lpick(3) );
  }
  // acquire/playback data file
  else if (verb=="ACQUIRE" || verb=="PLAY") {

    test = m_get_uswitch ("GOLDTEST",fn_in);
    dmac = m_get_switch_def ("DMAC",-1);
    m_init (hcb, m_apick(3), "1000","SP,SB,SI,SL,SF,CB,CI,CF",0);
    if (verb=="ACQUIRE") {
      m_open (hcb, HCBF_APPEND);
      dir  = -1;
    } else {
      m_open (hcb, HCBF_INPUT);
      dir  = 1;
    }
    tuner = (p->ptype==IOPT_TUNER) || (p->ptype==IOPT_TBANK);
    bits  = hcb.bps*8;
    if (hcb.bps<=0) bits = -hcb.bps;
    bits  = m_get_switch_def ("BITS", bits);
    if (hcb.spa == 2) bits = -bits;
    dec   = m_lpick(6);
    if (!tuner) dec = 1;
    rate  = dec / max(1.0e-9,hcb.xdelta);
    if (tuner) rate = rate*hcb.spa;
    rate  = m_get_dswitch_def ("SRATE",rate);
    tune  = m_dpick(7);
    if (tune==-1) tune = rate*0.25;
    gain  = m_lpick(8);
    replay = m_get_switch ("REPLAY");
    wait = m_get_dswitch ("WAIT");
    if (wait<0) wait = m_get_dswitch ("PAUSE");
    tl = m_get_switch ("TL");
    block = max(0,tl) * hcb.dbpe;
    kick = m_get_switch ("KICK");
    if (m_get_uswitch ("TRIGGER",trigger)<=0) trigger = "";
    if (m_get_uswitch ("ARATE",label)<=0) label = "";

         if (replay==1) dmamode = DMA_ONESHOT;
    else if (replay==2) dmamode = DMA_CONTINUOUS;
    else if (replay==4) dmamode = DMA_LOAD;
    else if (replay==5) dmamode = (dir<0)? DMA_SPIN : DMA_LOAD;
    else if (replay<=0) dmamode = -replay;
    else m_error ("Illegal replay mode");

    // reset the device
    status = pic_reset (p,0);
    if (status <= 0) m_error ("Resetting port: "+str);

    if (p->ptype == IOPT_TBANK) {
      nchan = pic_getkeyl (p,-1,KEY_CHNS);
      i     = pic_getkeyl (p,-1,KEY_PINDEX);
      if (i!=3) nchan = nchan/2;
      nchan = m_get_switch_def("NCHN",nchan);
      status = pic_setkeyl (p,-1,KEY_CHNS,nchan);
      xfer = m_get_switch_def("TL",4096);
      pktlen = m_get_switch_def("PKTLEN",xfer*hcb.bpa);
      status = pic_setkeyl (p,-1,KEY_PKTLEN,pktlen);
      dfreq = m_get_dswitch_def("DFREQ",(0.5/hcb.xdelta));
      status = pic_setkeyd (p,-1,KEY_DFREQ,2*dfreq/rate);
    }

    // set up data port
    dmac = pic_ioport (p, -1,-1, dmac, dir, bits, NINT(rate), 2*tune/rate, dec, gain, flags);
    if (dmac<=0) m_error ("Bad I/O Port parameters");

    // process the file for DMA access parameters
    fmode = m_get_switch_def("FMODE",1);
    status = pic_mapfile (p, &map, &hcb.hcb, fmode);
    if (status<=0) m_error ("Unacceptable input file");

    // set up host DMA channel
    status = pic_dmasetup (p, dmac, dir, &map, block, flags);
    if (status<=0) m_error ("Bad DMA channel parameters");

    m_sync();

    while (trigger!="") {
      m_pause(Mc->pause);
      if (m_rfind(trigger,tmpstr,dummy)<='A') m_error ("Trigger result not found or must be numeric");
      if (dummy>0.0) trigger = "";
    }

    if (wait>0) m_pause(wait);

    buf = (int_4*)map.vaddr;
    nbytes = map.bytes;
    if (hcb.mode>=2 && dir>0) m_fread (hcb,buf,hcb.first_byte,nbytes,nget);

    status = pic_dmafunc (p, dmac, dmamode);
    if (kick>0) status = pic_dmafunc (p, kick, DMA_SPIN);
    m_secnds(0.0,time);

    if (tcmode!="OFF" && dir<=0) {
      timeout = pic_getkeyl (p,dmac,KEY_TIMEOUT);
      TCAGAIN:
      dummy=0.0;
      status_ = pic_tc (p, dmac, &dummy, &hcb.xdelta, &etime[0], &etime[1], FLG_TCINTERP);
      if (status_>=0) {
        m_put_epoch (hcb, tcoff+etime[0], etime[1], false);
        m_update_header (hcb);
      } else if (pic_dmastat(p, dmac, &i, &j) == 1) {
        if (status_!=-1 && status_!=-3) printf("Bad TC status=%d\n",status_);
        m_pause (Mc->pause);
        m_secnds (time,timed);
        if (timeout<0 || timed<timeout) goto TCAGAIN;	// still running, try again
        status = pic_setkeyl (p,dmac,KEY_TIMEOUT,0);  // run out the timer
      } else {
        printf("Bad TC status=%d\n",status_);
      }
    }

    status = pic_dmafunc (p, dmac, DMA_WAIT);
    m_secnds(time,timed); if (time2<.001) time2=.001;

    if (test>0 && status>=0) {
      status = testit (buf,NINT(hcb.size),fn_in);
      if (status==1) dir = 0;	// its OK, bypass file write
    }

    if (hcb.mode>=2 && dir<0) m_fwrite (hcb,buf,hcb.first_byte,nbytes,nget);

    if (dmamode>0) {
      arate = 1.e-6*nbytes*dmamode/timed;
      if (Mu->verbose) printf("DMA stat=%d took=%f rate=%f Mby/sec\n",status,time,arate);
      arate = hcb.size*hcb.ape*dmamode/timed;
      m_frslt (label, arate);	// samples/second
    }
     if (replay==5) {
      status_ = pic_dmafunc (p, dmac, DMA_SPIN);	// spin
    } else if (replay!=4) {
      status_ =  pic_dmafunc (p, dmac, DMA_LOST);
      if (status_>0) printf("missed %d blocks\n",status);
      status_ = pic_dmafunc (p, dmac, DMA_STOP);	// stop
      status_ = pic_dmafunc (p, dmac, DMA_CANCEL);	// release
    }
    status_ = pic_mapfile (p, &map, &hcb.hcb, -fmode);
    if (replay==4 && status>=0) status = dmac;
    m_close (hcb);
    goto SKIPSYNC;
  }
  // start a spin transfer
  else if (verb=="START") {
    dir  = m_get_switch_def("DIR", 0);
    dmac = pic_ioport (p, -1,-1, -1, dir, 16, 0, 0.0, 64, 0, FLG_DISABLE);
    status = pic_dmafunc (p, dmac, DMA_SPIN);	// spin
  }
  // stop a spin transfer
  else if (verb=="STOP") {
    dir  = m_get_switch_def("DIR", 0);
    dmac = pic_ioport (p, -1,-1, -1, dir, 16, 0, 0.0, 64, 0, FLG_DISABLE);
    status = pic_dmafunc (p, dmac, DMA_STOP);	// stop
    status = pic_dmafunc (p, dmac, DMA_CANCEL);	// release
  }
  // loop active alternate input to named output port 
  else if (verb=="LOOP") {
    bits  = m_get_switch_def("BITS", 16);
    rate = m_get_dswitch_def("RATE",10.0e6);
    dmac = pic_ioport (p, -1,-1, 0, 1, bits, NINT(rate), 0.0, 64, 0, 0);
  }
  // loop input module-1 to output module-2 port 
  else if (verb=="LOOPIO") {
    bits  = m_get_switch_def("BITS", 16);
    rate = m_get_dswitch_def("RATE",10.0e6);
    dmac = pic_ioport (p, IOPT_MODULE,1, -1, -1, bits, NINT(rate), 0.0, 1, 0, 0);
    dmao = pic_ioport (p, IOPT_MODULE,2,  0, 1, bits, NINT(rate), 0.0, 1, 0, FLG_LOOP); // start output
    status = pic_dmafunc (p, dmac, DMA_LOOP);	// start input
    wait = m_get_dswitch ("POLL");
    delta = 1.0 / rate;
    if (tcmode!="OFF") {
      m_sync();
      LTCAGAIN:
      m_pause(0.1);
      dummy=0.0;
      status_ = pic_tc (p,dmac,&dummy,&delta,&etime[0],&etime[1],FLG_TCINTERP);
      if (status_ < 0 && !Mc->break_) goto LTCAGAIN;
      status_ = pic_setkey (p,dmao,KEY_TCOFF,(void*)etime,16);
      if (wait > 0.0 && !Mc->break_) {
        m_pause(wait);
        goto LTCAGAIN;
      }
      goto SKIPSYNC;
    }
  }
  // get a keyed value
  else if (verb.substr(0,3)=="GET" || verb=="QUERY") {
    name = m_upick(3);
    label = m_upick(4);
    index = pic_name2key (p,s2c(name));
    dmac = m_get_switch_def("DMAC",0);
    n    = pic_getkeysize(p,index);
    type = pic_getkeytype(p,index);
    ls = n*4;
    if (type=='D') ls = n*8;
    ls = m_get_switch_def("LEN",ls);
    if (index==KEY_NDEC) values[0] = m_lpick(4);
    if (index==KEY_NFREQ || index==KEY_NRATIO) {
      dummy = m_dpick(4);
      m_move((void*)&dummy,values,8);
    }
    status = pic_getkey (p,dmac,index,(void*)values,ls);
    if (status<0) {
      m_warning("Unrecognized key name: "+name);
    } else if (label!="") {
      if (m_get_pswitch("ASCII")) type='A';
      i = m_put_result(label,values,ls,0,type,0);
    } else if (m_get_pswitch("HEX")) {
      printf("Key %s = %08x\n",s2c(name),values[0]);
    } else if (type == 'D') {
      m_move((void*)values,(void*)&dummy,8);
      printf("Key %s = %lf\n",s2c(name),dummy);
    } else {
      printf("Key %s = %d\n",s2c(name),values[0]);
    }
  }
  // set a keyed value
  else if (verb.substr(0,3)=="SET") {
    name = m_upick(3);
    dmac = m_get_switch_def("DMAC",0);
    ls = m_get_switch_def("LEN",-1);
    index = pic_name2key (p,s2c(name));
    type = pic_getkeytype(p,index);
    if (type=='D') {
      dummy = m_dpick(4);
      status = pic_setkeyd (p,dmac,index,dummy);
    } else {
      value = m_lpick(4);
      status = pic_setkey (p,dmac,index,(void*)&value,ls);
    }
  }
  // join a multicast group
  else if (verb=="JOIN") {
    name = m_upick(3);
    value = pic_str2ip (p,s2c(name));
    status = pic_setkeyl (p,0,KEY_IPCONN,value);
  }
  // leave a multicast group
  else if (verb=="LEAVE") {
    name = m_upick(3);
    value = pic_str2ip (p,s2c(name));
    status = pic_setkeyl (p,0,KEY_IPDISC,value);
  }
  // test PCI bus speed
  else if (verb=="SPEC") {
    status = pic_test (p,7,1);
  }
  // test PCI bus speed
  else if (verb=="SPECS") {
    status = pic_reset (p,0);
    m_init (hcb, m_apick(3), "3000","SF",0);
    m_addsrec (hcb, "SRAT", "SF");
    m_addsrec (hcb, "BRAT", "SF");
    m_open (hcb, HCBF_OUTPUT);
    block = m_get_switch_def ("BLOCK",16384);
    idb = m_lwinit ("BlockSize",block,1,512*1024,1);
    ndata = m_get_switch_def ("SIZE",1024*1024);
    nbytes = ndata*4;
    if (pic_map(p,&buf,&nstart,nbytes,1)<=0) m_error ("Mapping DMA buffer");
    m_sync();
    m_pause (0.1);
    do {
      status = pic_spec (p, buf, nstart, ndata, block, &ntotal, &rates[0], &maxsjump, &rates[1], &maxbjump, 0);
      status = m_lwget(idb,block);
      m_filad (hcb, rates, 1);
    } while (Mc->mode>0 && !Mc->break_);
    pic_map(p,&buf,&nstart,nbytes,-1);
    m_close (hcb);
    goto SKIPSYNC;
  }
  // display SHARC DMA parameters 
  else if (verb=="DMAC") {
    dmac = m_lpick(3);
    pic_dmadump (p,dmac,0);
  }
  else if (verb=="MAP") {
    nbytes = m_lpick(4);
    printf("Mapping %d bytes\n",nbytes);
    if (pic_map(p,&buf,&nstart,nbytes,1)<0) m_error ("Mapping DMA buffer");
    zbuf = (int_4*)malloc(nbytes);
    n = max(1,256*1024*1024/nbytes);
    m_secnds(0.0,time);
    for (i=0; i<n; i++) {
      m_move(buf,zbuf+(nbytes/8),nbytes/2);
      m_move(zbuf+(nbytes/8),buf,nbytes/2);
    }
    m_secnds(time,timed);
    printf("Mapped Transfer rate %f Mby/s\n",(float)(n*nbytes/timed*1e-6));
    m_secnds(0.0,time);
    for (i=0; i<n; i++) {
      m_move(zbuf,zbuf+(nbytes/8),nbytes/2);
      m_move(zbuf+(nbytes/8),zbuf,nbytes/2);
    }
    m_secnds(time,timed);
    printf("UnMapped Transfer rate %f Mby/s\n",(float)(n*nbytes/timed*1e-6));
    free(zbuf);
    if (pic_map(p,&buf,&nstart,nbytes,-1)<0) m_error ("UnMapping DMA buffer");
  }
  // read NVRAM 
  else if (verb.substr(0,3)=="NVR") {
    pic_nvram (p,s2c(""),-1);
  }
  else {
    m_error ("Unsupported verb: "+verb);
  }

  // finish up
  m_sync();
  SKIPSYNC:
  m_lrslt (statlab,status);
  status = pic_close (p);

}

