/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/** 
  ICE-PIC I/O Module Controllers

  Type 0 H = 8b BiDir DDR HyperTransport Packetized at 400|200 MHz or 266|133 MHz
  Type 1 S = 1|4|8|16b Single-Ended Non-Packetized at up to 160MHz
  Type 2 D = 1|4|8|16b Differential DDR Non-Packetized at up to 160MHz
  Type 3 R = 4 lane BiDir RocketIO Transceiver Packetized at 1.5-5GHz each lane

  Mixer setup:
  
  outer32 = upper | lower | muxB | muxI
  muxer32 = input | output | test | alt
  mixer64 = individual byte lanes of muxer

	acq	ply		go	tc in
 
                                0
                                1
   0		woena		2
   1	voena	atOuter		3       vocnt
   2	atMuxer	atMuxer		4	voval
   3	atMixer	atMixer		5       votc
   4	wival	vival		6	tcena
				7	tcwr

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  12/15/2004

*/
`include "vdefs.h"
`include "ramdefs.h"
`include "hypdefs.h"

// handle connector differential inversions
`ifdef VIRTEX8
`define QIAINV 9'h008
`define QIBINV 9'h006
`define QOAINV 9'h1BF
`define QOBINV 9'h13C
`define QDAINV 16'h10FD
`define QDBINV 16'h603C
`define QIADLY 50
`define QIBDLY 50
`else
`define QIAINV 0
`define QIBINV 0
`define QOAINV 0
`define QOBINV 0
`define QDAINV 0
`define QDBINV 0
`define QIADLY 30
`define QIBDLY 30
`endif

`define IOC_IOM    0
`define IOC_OVSR   3
`define IOC_FRAME  4
`define IOC_PROG   6
`define IOC_TCMODE 7
`define IOC_TCOUNT 12
`define IOC_UREG   16
`define IOC_RAM    32

`ifdef VIRTEX7
`define VIO78 7
`elsif VIRTEX8
`define VIO78 8
`endif

module iomodule (
  sclk,rst, scs,saddr, swr,swrbus, srd,srdbus, sack,
  mclk, muxclk, muxhso, clocki, xsync, oppms,oppus,oppts,
  intclk, ymixo, ymixi, master, slave, 
  vclk,vclk2, vrst, vipc, vip, vopc, vop, vinc, vin, vonc, von, 
  ioclk, istat,iena,isel,ibus, ostat,oena,osel,obus, 
  rclk, rrst,ribus,robus, mcfg,tpi,test);			

  parameter PORT=1;
  parameter TYPE=0;
  parameter IBW=64;
  parameter CARD=5;

`ifdef IOM_QDR
  localparam PORTW=16;
  localparam F1IFLG=`BPAR;
  localparam F1OFLG=0;
  localparam QIINV = (PORT==2)? `QIBINV : `QIAINV;
  localparam QOINV = (PORT==2)? `QOBINV : `QOAINV;
  localparam QDINV = (PORT==2)? `QDBINV : `QDAINV;
  localparam QIDLY = (PORT==2)? `QIBDLY : `QIADLY;
  localparam QODLY = (PORT==1)? 1 : 30;
`ifdef DMA_PKT_4X
  localparam QDR=(TYPE==0)?2:0;
`else
  localparam QDR=(TYPE==0)?1:0;
`endif
  localparam RW=64;

`else
  localparam PORTW=(TYPE==0)?8:16;
  localparam F1IFLG=`BPAR;
  localparam F1OFLG=`OREG;
  localparam QDINV = 0;
  localparam QDR=0;
  localparam RW=32;
`endif

  localparam DPORTW=PORTW*2;
  localparam XPORTW=DPORTW/8;
  localparam FPORTW=DPORTW+XPORTW;
  localparam NR=(CARD==8)?8:2;
  localparam NS=(CARD==8)?4:3;
`ifdef VIO78
  localparam VIO=`VIO78;
`else
  localparam VIO=0;
`endif

  // System bus
  input sclk,rst,scs,swr,srd;
  input [31:0] saddr,swrbus;
  output [31:0] srdbus;
  output sack;

  // Global resources
  output master,clocki,muxhso;
  input  xsync,mclk,muxclk,slave,oppms,oppus,oppts;
  input intclk;
  output [32:0] ymixo;
  input [32:0] ymixi;

  // HyperTransport Interface
  input vclk, vclk2, vrst;
  inout vipc, vinc;
  inout [7:0] vip, vin;
  inout vopc, vonc;
  inout [7:0] vop, von;

  // DMA bus
  input ioclk,iena,oena;
  input [7:0] isel;
  output [7:0] osel;
  input [IBW-1:0] ibus;
  output [IBW-1:0] obus;
  output istat,ostat;

  // RocketIO Interfaces
  input rclk;
  output rrst;
  input [RW+1:0] ribus;
  output [RW+1:0] robus;

  // test ports
  output [35:0] mcfg;
  input  [7:0] tpi;
  output [7:0] test;
//  output [32:0] test;

  wire [7:0] ptest,qtest,ttest,testqc;

  // global clocks
  wire L=0, H=1;
  wire [2:0] L3 = {3{L}};
  assign sack=H;
  wire  pindex = PORT;

  // system registers
  reg [31:0] iom;
  reg [4:0] dclkcnt;
  reg ena,enb,fenb,genb,hold,dclkrst,dclkadj,dclkcyc,dclkud;
  wire swrreg = (scs && swr && saddr[7:6]==0);
  wire swriom = (swrreg && saddr[5:0]==0);
  wire swrion = (swrreg && saddr[5:0]==1);
  wire swriop = (swrreg && saddr[5:0]==6);
  wire swrtcm = (swrreg && saddr[5:0]==7);
  wire swriot = (swrreg && saddr[5:0]==8);
  wire swrflg = (swrreg && saddr[5:0]==16);
  wire swrdly = (swrreg && saddr[5:0]==17);
  wire swrtap = (swrreg && saddr[5:0]==18);
  wire swrinc = (swrreg && saddr[5:0]==19);
  wire swrseq = (swrreg && saddr[5:0]==20);
  wire swrflp = (swrreg && saddr[5:0]==21);
  wire swremt = (swrreg && saddr[5:0]==22);
  always @(posedge sclk) begin
    if (swriom) iom <= swrbus;
    hold <= swrion || (hold && !swriom);
    if (swriop) dclkcnt <= 0; else dclkcnt <= dclkcnt+1;
    dclkrst <= (swriop && swrbus[0]) || (dclkrst && dclkcnt!=15);
    dclkcyc <= swriop || (dclkcyc && dclkcnt!=31);
    dclkadj <= (swriop && swrbus[1]);
    dclkud  <= (swriop && swrbus[2]);
    ena <= iom[0];
  end
  wire len  = iom[1];
  wire ten  = iom[2];
  wire sen  = iom[3];

  wire ply  = iom[4];
  wire coe  = iom[5];
  wire mc1  = iom[6];
  wire mc2  = iom[7];

  wire [1:0] nbit = iom[9:8];
  wire [1:0] ibit = iom[11:10];

  wire lsbx = iom[12];	// insert X in LSB
  wire uopt = iom[13];	// enable user option
  wire inv  = iom[14];	// invert clock
  wire msbi = iom[15];	// invert MSB

  wire sgo  = iom[16];	// slave go
  wire tgo  = iom[17];	// triggered go
  wire ggo  = iom[18];	// gated go
  wire xgo  = iom[19];	// external go

  wire dual = iom[20];	// capture data on both clock edges
  wire muxy = iom[21];	// mux modules
  wire vopt = iom[22];	// very high speed ? vopt
  wire alt  = iom[23];	// alternate input

  wire lr1  = iom[24];	// link resource 1
  wire lr2  = iom[25];	// link resource 2
  wire flip = iom[26];	// spectral flip
  wire clkm = iom[27];	// muxed clock

  wire bmux = iom[31];  // block mux

  wire training = iom[20] && iom[18];		// special training combo bits
  wire packetized = (TYPE==0 || TYPE==3);

  wire tlen = (len||ten);

  reg ncclk,feedclk;
  always @(posedge sclk) begin
    ncclk   <= (ply && uopt && !packetized);	// non-continuous clock
    feedclk <= (coe && !ply);			// feed through clock for A2Ds
  end

`ifdef VIRTEX9
  wire bifi = isel[7];
`else
  reg bifi; always @(posedge ioclk) bifi <= isel[7];
`endif

  wire word = (nbit==0);
  wire byte = (nbit==1);
  wire bitp = (nbit==2);
  wire nibb = (nbit==3);

  // variables
  wire fclk, gclk;
  wire wrdyi, wrdyo, fbit, trigger;
  wire [31:0] dummy,srmem,cdat,gdat;
  wire [15:0] srsig;

  // TC variables
  reg twr, tval, tcyc, tdc, tnclk;
  reg [5:0]  tadr;
  reg [31:0] tdat;
  wire tcbito; 

  // error circuit
  reg [7:0] seqfill,seqerrs;
  reg err,tsync,primed,swait,sdiff,sfill,ftest;
  wire [7:0] demt;
  wire rstat = (saddr[7:2]==0);
  wire cstat = (saddr[7:2]==1);
  wire gstat = (saddr[7:2]==2);
  wire [31:0] stat = { srsig, demt, tval,tcyc,tdc,tnclk, tsync,err,fenb,ena };

  assign srdbus[31:0] = cstat? cdat : rstat? stat : gstat? gdat : srmem;

  dpram #(2,32,32) ramc (sclk,scs, {L,L,L,saddr[7:2]}, swr, swrbus, srmem,
                         intclk,H, {L,L,L,tadr},       twr, tdat, dummy);

/* verilator lint_off UNOPTFLAT */
  // operational mode flags
  reg [7:0] delay,taps,minc,mrate,flips;
  reg [2:0] mshift;
  reg [15:0] tcmode,cemt;
  reg [23:0] flags;
  wire decdelay = swait&oppms;
  always @(posedge sclk) begin
    if (swrflg) flags  <= swrbus;
    if (swrdly) delay  <= swrbus; else if (decdelay) delay <= delay-1;
    if (swrtcm) tcmode <= swrbus;
    if (swrtap) taps   <= swrbus;
    if (swrinc) minc   <= swrbus[7:0];
    if (swrinc) mshift <= swrbus[10:8];
    if (swrinc) mrate  <= swrbus[23:16]; // sample rate in (N/256)*100MHz
    if (swrflp) flips  <= swrbus;
    if (swremt) cemt   <= swrbus;
  end

  // flags
  wire flg_in    = flags[0];
  wire flg_out   = flags[1];
  wire flg_dcm   = flags[2];
  wire flg_sdds  = flags[3] && packetized;
  wire flg_ramp  = flags[4];
  wire flg_white = flags[5];
  wire flg_loop  = flags[6];
  wire flg_good  = flags[7];
  wire flg_isx   = flags[8];
  wire flg_reset = flags[9];
  wire flg_synf  = flags[10] && (CARD==7);// sync or IRIG-B filter
`ifdef IOM_QDR
  wire flg_qdr   = flags[11] && (TYPE==0 || TYPE==3); // dual input/output  port mode
`else
  wire flg_qdr   = L;
`endif
  wire flg_tout  = flags[12] && (QDR>=1); // test output waveform
  wire flg_ohos  = flags[13]; 	// output holdoff select clki:di[0]	
  wire flg_m20f  = flags[14] && (PORT==2);// m19 fixup mode
  wire flg_nofc  = flags[16];	// no flow control
  wire flg_cben  = flags[17]==0;// clock buffer enable
  wire flg_mggo  = flags[18];	// module gated go
  wire flg_zfill = flags[19];	// zero fill between packets
  wire flg_ssclk = flags[20];	// spread spectrum clocking
  wire flg_csel  = flags[21];	// PLL clock select
  wire flg_qdrm  = flags[22] && (QDR>0); // QDR match mode
  wire flg_seqf  = flags[23] && (QDR>0); // QDR match mode

  wire fx12 = flg_qdr && ibit[0];	// fix filler at end of 12b packets N/A
  wire up12 = flg_qdr && ibit[1];	// use 12b up converter module
  wire noseq = !flg_isx;

  // timecode handlers
  wire tctst = tcmode[0];
  wire tcrst = tcmode[1];
  wire tchit = tcmode[2];
  wire tcswp = tcmode[3];
  wire tcopp = tcmode[4];
  wire tcirb = tcmode[5];
  wire tcfil = tcmode[6];
  wire tcndc = tcmode[7];
  wire tcsdn = tcmode[8];
  wire tcsdd = tcmode[9];
/* verilator lint_on UNOPTFLAT */

  reg  fval,fovalp,fovaln;
  wire frst = !ena;
  wire good = flg_good;
  wire acq  = !ply;
  wire dir  = flg_out;
  wire xalt = ply && !flg_loop && vopt;
  wire muxx = muxy && !bmux;
  wire falt = (alt || flg_loop); // loop back implied alt
  wire ialt = (falt && acq) || xalt; // input alt
  wire oalt = (falt && ply); // output alt
  wire testdata = (flg_ramp || flg_white);
  wire usealt = (acq && falt) || (muxx && PORT==2) || xalt; // use alt's go signal
  wire xonoff = (flg_in ^ flg_out) && flg_good && !flg_loop && !testdata;
  wire sdncnt = flg_sdds && tcsdn;
  wire dcm = (taps==8'hFF);
  wire stx = !flg_isx; // stratix based module
  wire qmux = muxx && flg_qdr;

  // sync filter
  reg [10:0] yscnt;
  reg ysync,ysyncn,ysyncl;
  wire ysame = (ysyncn==ysyncl);
  wire ygood = yscnt[10];
  always @(posedge sclk) begin
    ysyncn <= xsync;
    ysyncl <= ysyncn;
    if (!ysame) yscnt <= 0; else if (!ygood && oppus) yscnt <= yscnt+1;
    ysync <= ygood? ysyncl : ysync;
  end
  wire xsyncf = flg_synf? ysync : xsync;

  // port startup
  reg cgo,tgoc0,tgoc1,fsync,triggerd,fsyncd;
  wire cgot = !ply || primed || flg_ramp || flg_white;
  wire usesgo = sgo || (muxy && PORT==2);
  wire eclk = packetized? sclk : fclk;
  always @(posedge eclk or posedge frst) begin
    if (frst) fsync<= 1; else fsync <= xsyncf;
    if (frst) fsyncd<=1; else fsyncd <= fsync;
    if (frst) triggerd<=1; else triggerd <= trigger;
    if (frst) tgoc0<= 0; else tgoc0 <= tgo && !xgo && trigger && !triggerd;
    if (frst) tgoc1<= 0; else tgoc1 <= tgo &&  xgo && fsync && !fsyncd;
    if (frst) cgo  <= 0; else cgo   <= !sgo && !tgo && cgot;
    if (frst) enb  <= 0; else enb   <= enb || cgo || tgoc0 || tgoc1;
    if (frst) fenb <= 0; else fenb  <= usesgo? slave : enb;
  end
  always @(posedge gclk or posedge frst) begin
    if (frst) genb <= 0; else genb <= usesgo? slave : enb;
  end
  assign master = enb;

  // data input/output clocks
  wire vic,usemux,outmux,iclk,jclk,kclk,oclk,lclk,vicd,lock;
  wire [1:0] gcsel = usemux? 3 : dcm? 2 : (taps!=0)? 1 : 0;
  assign jclk   = (TYPE==0)? lclk : iclk;
  assign outmux = (TYPE==0)? !flg_good : flg_out;
  assign rrst = !ena;

  // input/output data interface
  wire [PORTW-1:0] fibusp,fibusn,fjbusp,fjbusn;
  wire [PORTW-1:0] fobusp,fobusn,fqbusp,fqbusn;
  wire [DPORTW-1:0] fobus,fqbus;
  wire [FPORTW-1:0] fibus,fjbus;
  wire foe = ply && flg_out && !flg_loop;
  wire riena,roena,rqena,rirdy,rordy;

  // locally generated delay
  tapdelay dly (vicd, vic, taps);
  mux4xN gcm (iclk, vic,vicd,lclk,muxclk, gcsel);

`ifdef VIRTEX2
  BUFGMUX _clki (kclk,jclk,oclk,outmux); 
`elsif VIRTEX8
  BUFGCE _clkg (.O(kclk),.I(vic),.CE(1'b1));
`else
  wire voclk = packetized? oclk : vic;
  reg s1; always @(posedge sclk) s1 <= packetized? outmux : (gcsel==0);
  BUFGCTRL _clki (.O(kclk),.I0(jclk),.I1(voclk),.S0(!s1),.S1(s1),.IGNORE0(H),.IGNORE1(H),.CE0(flg_cben),.CE1(flg_cben)); 
`endif

`ifdef VIO78 
  assign lclk = vic;
`else
  // vic -> CLKIN = 1.3ns A&B sides of PIC6
  DCM i_dcm (.DSSEN(L),.PSCLK(sclk),.PSEN(dclkadj),.PSINCDEC(dclkud),.RST(dclkrst), .CLKIN(vic),.CLK0(lclk),.CLKFB(jclk));
  defparam i_dcm.CLK_FEEDBACK = "1X";
  defparam i_dcm.DLL_FREQUENCY_MODE = (TYPE==2 || TYPE==1)? "LOW" : "HIGH";
  defparam i_dcm.DUTY_CYCLE_CORRECTION = "TRUE";
  defparam i_dcm.CLKIN_PERIOD = ((TYPE==2)? 10 : (TYPE==1)? 10 : 3);
  defparam i_dcm.CLKOUT_PHASE_SHIFT = "VARIABLE";
`ifdef VIRTEX2
  defparam i_dcm.PHASE_SHIFT = (TYPE==1 || TYPE==2)? -100 : 0; 
`else
  defparam i_dcm.PHASE_SHIFT = (TYPE==1 || TYPE==2)? 0 : 0; 
`endif
  defparam i_dcm.STARTUP_WAIT = "FALSE";
  // phase shift accounts for input setup time
  // clean range is 40->160 @105 70->180 @120 on PIC5_SS
  // clean range is 40->160 @105 70->180 @120 on PIC5_DD
`endif

  // Branch on IO module transport type

  generate
  if (TYPE==0) begin:id0	// HyperTransport bidirectional differential DDR byte lanes (packetized)
`ifdef IOM_QDR
    wire [5:0] ss;
    wire prst,drst,pclk,qclk,_fclk;
    reg qrst; always @(posedge gclk) qrst <= flg_reset;
    wire [7:0] cadj = {flg_csel,good,fclk,dclkcyc,dclkud,dclkadj,dclkrst,sclk};
    assign drst = !ena;
    assign vic = testqc[0];
    wire tsa = !good || !(flg_out&flg_qdr);
    wire tsb = !good ||  (flg_in&flg_qdr);
    wire tsc = !good;
`ifdef VIRTEX8
    assign fclk = gclk;	// cant use the generated _fclk and gclk in the dioqdr 
    diqdrc #(`SYS_CLKU) _clkids (vipc,vinc, _fclk,drst,pclk,prst,ss, cadj,testqc);
    doqdr #(1,QOINV>>8,QODLY) _clkods (vopc,vonc,  gclk,qclk,qrst,ss,tsc, H,L,H,L); 
    // OSERDESE3 are very sensitive to CLK/CLKDIV skew - this is necessary to insure proper reset
    BUFGCE_DIV #(.BUFGCE_DIVIDE("2")) _clkg (.O(gclk),.I(vclk),.CLR(1'b0),.CE(flg_cben));
    BUFGCE_DIV #(.BUFGCE_DIVIDE("1")) _clkq (.O(qclk),.I(vclk),.CLR(1'b0),.CE(flg_cben));
`else
    diqdrc _clkids (vipc,vinc, fclk,drst,pclk,prst,ss, cadj,testqc);
    doddr _clkods (vopc,vonc, qclk,good, H,L); 
    assign gclk = vclk;
    assign qclk = vclk2;
`endif
    dioqdrN #(8,1,QIINV,QIDLY) _dati (vip,vin, fclk,pclk,prst,ss,  fibusp[7:0],fibusn[7:0],fibusp[15:8],fibusn[15:8],
                                               gclk,qclk,qrst,tsa, fqbusp[7:0],fqbusn[7:0],fqbusp[15:8],fqbusn[15:8]);
    dioqdrN #(8,1,QOINV,QIDLY) _dato (vop,von, fclk,pclk,prst,ss,  fjbusp[7:0],fjbusn[7:0],fjbusp[15:8],fjbusn[15:8],
                                               gclk,qclk,qrst,tsb, fobusp[7:0],fobusn[7:0],fobusp[15:8],fobusn[15:8]);
    assign oclk = vclk2;
`else
    dibuf _clkids (vipc,vinc,vic);
    diddrN #(8,1) _dati (vip,vin, fclk, fibusp,fibusn);
    doddr _clkods (vopc,vonc, gclk,good, H,L);
    doddrN #(8,1) _dato (vop,von, gclk,good, fobusp,fobusn);
    assign fclk = kclk;
    assign gclk = vclk2;
    assign oclk = vclk;
`endif
    assign usemux = flg_out;
    assign muxhso = H;
    assign clocki = L;
  end
  if (TYPE==1) begin:id1	// Single ended in|out Rising|Falling 16b word lane
    sibuf _clkids (vipc,vic);
    oddrt _clkods (vopc, gclk,good, fovalp|feedclk,fovaln);
    ioddrN #(16,1) iobs ({vip[0],vin[0],vip[1],vin[1],vip[2],vin[2],vip[3],vin[3],
			  vip[4],vin[4],vip[5],vin[5],vip[6],vin[6],vip[7],vin[7]},
    				fclk, fibusp,fibusn, gclk,foe, fobusp,fobusn);
    // module enables (vinc is en, vonc is oe)
    assign vonc = good? !mc1 : 'bZ;
    assign vinc = good? !mc2 : 'bZ;
    assign oclk = muxclk;
    assign fclk = kclk;
    assign gclk = (VIO==8)? kclk:mclk;
    assign usemux = flg_out || clkm;
    assign muxhso = L;
    assign srsig  = { vop[0],von[0],vop[1],von[1],vop[2],von[2],vop[3],von[3],
		      vop[4],von[4],vop[5],von[5],vop[6],von[6],vop[7],von[7] };
    assign clocki = (CARD<=6)? vic:L;
  end
  if (TYPE==2) begin:id2	// Differential in DDR 16b word
    dibuf _clkids (vipc,vinc, vic);
    doddr _clkods (vopc,vonc, sclk,good, H,L); // 100MHz ref
    wire [15:0] vipf = {vip[0],vip[1],vip[2],vip[3],vip[4],vip[5],vip[6],vip[7],vop[0],vop[1],vop[2],vop[3],vop[4],vop[5],vop[6],vop[7]};
    wire [15:0] vinf = {vin[0],vin[1],vin[2],vin[3],vin[4],vin[5],vin[6],vin[7],von[0],von[1],von[2],von[3],von[4],von[5],von[6],von[7]};
    diddrN #(16,1,0,QDINV) _dati (vipf,vinf, fclk, fibusp,fibusn);
    assign oclk = muxclk;
    assign fclk = kclk;
    assign gclk = mclk;
    assign usemux = flg_out || clkm;
    assign muxhso = L;
    assign srsig = 0;
    assign clocki = (CARD<=6)? vic:L;
  end
  if (TYPE==3) begin:id3	// 3GIO RocketIO port, 1-4 lanes (packetized)
    assign vic = rclk;
`ifdef IOM_QDR
    assign {rordy,riena,fjbusn,fjbusp,fibusn,fibusp} = ribus;
    assign robus = {rirdy,roena,fqbusn,fqbusp,fobusn,fobusp};
`else
    assign {rordy,riena,fibusn,fibusp} = ribus;
    assign robus = {rirdy,roena,fobusn,fobusp};
`endif
    assign oclk = muxclk;
    assign fclk = rclk;
    assign gclk = rclk;
    assign usemux = L;
    assign muxhso = H;
    assign clocki = L;
  end
  if (TYPE==4) begin:id4	// Single ended test port
    sibuf _clkids (vipc,vic);
    //assign vic = tpi[0];
    oddrt _clkods (vopc, gclk,good, fovalp|feedclk,fovaln);
    assign fibusp = {tpi,tpi};
    assign fibusn = {tpi,tpi};
    // module enables (vinc is en, vonc is oe)
    assign vonc = 'bZ;
    assign vinc = 'bZ;
    assign oclk = muxclk;
    assign fclk = kclk;
    assign gclk = mclk;
    assign usemux = flg_out || clkm;
    assign muxhso = L;
    assign srsig = { vop[0],von[0],vop[1],von[1],vop[2],von[2],vop[3],von[3],
		     vop[4],von[4],vop[5],von[5],vop[6],von[6],vop[7],von[7] };
    assign clocki = L;
  end
  if (TYPE==5) begin:id5	// High-Z internal only
    assign vic = mclk;
    assign oclk = mclk;
    assign fclk = mclk;
    assign gclk = mclk;
    assign usemux = clkm;
    assign muxhso = L;
    assign srsig = 0;
    assign clocki = L;
  end
  endgenerate

  assign trigger  = inv? fibusp[0] : fibusn[0];

  // SDDS-isms
  reg shdr,shdrd,snibb,sbyte,sword; 
  wire sdds  = flg_sdds && !ggo;
  wire burst = flg_sdds && packetized;
  wire afull,bfull;

  // the input/output throttle circuits
  wire throttle = clkm && (testdata || falt || burst || (ply && packetized)); 
  wire nflowctrl = testdata || (falt && acq);
  reg mixinc,mixclr,muxcok,sinc,iorst;
  reg [1:0]  mixq,muxq,sixq,mixdif,sixdif;
  reg [15:0] sixcnt; // sdds throttle sample count
  reg [15:0] mixcnt; // mixclk sample count
  wire [15:0] muxcnt; // muxclk sample count
  mcounter mcount (mclk,iorst,muxcnt); // sample counter for placement in specified clock region
  always @(posedge sclk) begin
    sinc <= frst || afull;
    if (sinc) sixcnt <= {~mixcnt[7],mixcnt[6:0],8'h0}; else sixcnt <= sixcnt + mrate;
  end
  always @(posedge intclk) begin
    iorst <= frst;
    if (iorst) mixcnt <= 0; else if (mixinc) mixcnt <= mixcnt + minc[6:0];
    muxq    <= muxcnt[15:8] >> mshift;
    mixq    <= mixcnt[15:8] >> mshift;
    mixdif  <= muxq-mixq;
    sixdif  <= sixcnt[15:14] - mixcnt[7:6];
    muxcok  <= (burst && acq)? sixdif[1] : mixdif[1];
  end

  // zeroth stage
  wire firdy_,fjrdy_,fordy,fqrdy,fival,fjval,foena,fqena;
  reg [7:0] foenas;
  wire ocinv = inv && ply;
  always @(posedge gclk or posedge frst) begin
    if (frst) foenas <= 0; else foenas <= {foenas[6:0],foena};
    if (frst) fovalp <= 0; else fovalp <= (foenas[1] && !feedclk && !inv);
  end
`ifdef VIO78
  always @(posedge gclk or posedge frst) begin
`else
  always @(negedge gclk or posedge frst) begin
`endif
    if (frst) fovaln <= 0; else fovaln <= (foenas[2] && !feedclk && inv);
  end
  reg firdy; always @(posedge fclk) firdy <= firdy_;
  reg firdyd; always @(posedge fclk) if (firdy==firdy_) firdyd <= firdy;
  reg iready; always @(posedge gclk) iready <= genb && (firdyd || flg_nofc);
  assign rirdy = firdy;

  wire voena,voenai,voenaj;
`ifdef IOM_QDR
generate
if (QDR>0) begin:qdr
  wire xrdy = flg_ohos? vic : fibusp[0];
  wire csmi,cabi,csmj,cabj;
  wire [3:0] oholdi,oholdj,ihold,fichn,fjchn,fochn;
  wire [7:0] foenad,calibi,calibj,vitest,vjtest,votest,vqtest;
  reg [2:0] qenb;
  always @(posedge gclk) qenb[0] <= flg_tout? genb : genb && ply && (!xonoff || xrdy) && !(stx && foenas[3]); // output holdoff
  always @(posedge gclk) qenb[1] <= flg_zfill && !training;	
  always @(posedge gclk) qenb[2] <= noseq;	
  reg ionoff;  always @(posedge gclk) ionoff <= flg_tout? L : !dir && xonoff && !ggo;
  assign ihold = {L,L,L,L}; // not using header holdoff mechanism
  assign fochn = 4'hF;
  assign afull = !xrdy;
  wire [31:0] vdati = {fibusn[15:8],fibusp[15:8],fibusn[7:0],fibusp[7:0]};
  wire [31:0] vdatj = {fjbusn[15:8],fjbusp[15:8],fjbusn[7:0],fjbusp[7:0]};
  wire [31:0] vdato; assign {fobusn[15:8],fobusp[15:8],fobusn[7:0],fobusp[7:0]} = ionoff? {4{7'h0,iready}} : vdato;
  wire [31:0] vdatq; assign {fqbusn[15:8],fqbusp[15:8],fqbusn[7:0],fqbusp[7:0]} = ionoff? {4{7'h0,iready}} : vdatq;
  hypinq #(32,FPORTW,4,`HYP_TC)	qhypi (fclk,fenb,vdati, fival,fichn,fibus, {flg_seqf,fsync,ggo},oholdi,csmi,cabi,calibi,vitest);
  hypinq #(32,FPORTW,4,`HYP_TC)	qhypj (fclk,fenb,vdatj, fjval,fjchn,fjbus, {flg_seqf,fsync,ggo},oholdj,csmj,cabj,calibj,vjtest);
  hypoutq #(32,DPORTW,4)	qhypo (gclk,qenb,vdato, fordy,foena,fochn,fobus, ihold,votest);
  hypoutq #(32,DPORTW,4)	qhypq (gclk,qenb,vdatq, fqrdy,fqena,fochn,fqbus, ihold,vqtest);
  fifoseqfill sfi (intclk,frst,flg_seqf,calibi, voena,voenai);
  fifoseqfill sfj (intclk,frst,flg_seqf,calibj, voena,voenaj);
//  assign test[31:0] = {calibj,calibi,vitest, voenaj,voenai,voena,ggo, fjval,fival,ftest,master};	// master is required now
//  assign test[32] = intclk;
end
if (QDR==0) begin:iop
  wire ionoff = !dir && xonoff && !ggo;
  reg foqrdy; always @(posedge gclk) foqrdy <= fordy && fqrdy;
  ioprep #(TYPE,PORTW) mpn (fclk,frst,fenb, fibusp,fibusn,fibus,firdy,fival, riena, 
                            gclk,frst,genb, fobusp,fobusn,fobus,foqrdy,foena, roena,
                                 dir,acq,ply,ggo,rordy,ncclk,xonoff,burst,delay, fsync,stx,afull,ptest);
  ioprep #(TYPE,PORTW) mpm (fclk,frst,fenb, fjbusp,fjbusn,fjbus,firdy,fjval, riena, 
                            gclk,frst,genb, fqbusp,fqbusn,fqbus,foqrdy,fqena, rqena,
                                 dir,acq,ply,ggo,rordy,ncclk,xonoff,burst,delay, fsync,stx,bfull,qtest);
  assign voenai = voena;
  assign voenaj = voena;
end
endgenerate
`else
  wire ionoff = !dir && xonoff && !ggo;
  ioprep #(TYPE,PORTW) mpn (fclk,frst,fenb, fibusp,fibusn,fibus,firdy,fival, riena, 
                            gclk,frst,genb, fobusp,fobusn,fobus,fordy,foena, roena,
                                 dir,acq,ply,ggo,rordy,ncclk,xonoff,burst,delay, fsync,stx,afull,ptest);
  assign voenai = voena;
`endif

  // first stage FIFOs - to/from Module
  wire vival,voenc,voenan, vivalx;
  wire virdy,vjrdy,vordy,vqrdy;
  wire [63:0] vidat;
  wire [71:0] vodat;
  fifoNxM #(FPORTW,36,1,3,0,F1IFLG) fifo1i (fclk,frst,  firdy_,fival|(flg_qdrm&fjval),fibus,  intclk,frst, vordy,voenai,vodat[35:0]);
  fifoNxM #(32,DPORTW,1,3,0,F1OFLG) fifo1o (intclk,frst, virdy,vivalx,vidat[31:0],   gclk,frst, fordy,foena,fobus);
`ifdef IOM_QDR
  fifoNxM #(FPORTW,36,1,3,0,F1IFLG) fifo1j (fclk,frst,  fjrdy_,fjval|(flg_qdrm&fival),fjbus,  intclk,frst, vqrdy,voenaj,vodat[71:36]);
  fifoNxM #(32,DPORTW,1,3,0,F1OFLG) fifo1q (intclk,frst, vjrdy,vivalx,vidat[63:32],  gclk,frst, fqrdy,fqena,fqbus);
`endif
  wire vordys = (flg_qdr && flg_seqf)? (vordy && vqrdy) : vordy;

  // second stage FIFO - to/from XBar
  wire wivaln,wivalx,wival,woenan,woenax,woena;
  wire wirdy,wordy,ostat_;
  wire [63:0] widat,wodat;
  reg ival; always @(posedge ioclk) ival <= iena;
  reg oval; always @(posedge ioclk) oval <= oena && !oval;
  reg woval; always @(posedge intclk) woval <= woena;
  wire oena_ = qmux? oval : oena;	// qmux only reads every other cycle
  bififoNxM #(IBW,64,NR,NS, `LOOSE,`CENA) fifo2 (
   ioclk,frst, istat,ival,ibus,   ostat_,oena_,obus, 
   intclk,frst, wirdy,wival,widat, wordy,woena,wodat, ply,bifi);

  // output data primed
  wire loop = (ply && falt);
  wire primer; BUF prime (primer,loop? wordy : fordy);
  always @(posedge eclk or posedge frst) begin
    if (frst) primed <= 0; else primed <= primed || primer;
  end

  // sequencer logic {mixsel[8][2], mixena[8], ulnena[2],swpsel,outsel, vival,woena,wival,voena}
  wire outsel,swpsel,vivalp,vocnt;
  reg gorst,go0,go1,go2,go3,go4,igo,ogo,tcena,iflip,tflip,goalt,votc,seqtop,vivald,spacer;
  reg [1:0] bcnt;
  reg [3:0] seqa,bflip,swapn,vcnt;
  wire [31:0] seqdata;
  wire [31:0] outer,muxer;
  wire go  = usealt? goalt : go0;
  wire gos = go && shdr;
  wire gof = go && !sfill;
  wire gon = go && (uopt || (!swait && !shdr));
  wire gov = ply? seqdata[3] : (seqdata[0] || shdr);
  wire goq = bitp? (vcnt[1:0]==3) : H; // for bit packed step sequence every 4th cycle
  wire gor = bitp? ena : (go && (uopt || !shdr));
  wire bpps = !bitp || (ply && (seqa==0)  && (vcnt[1:0]==0));
  wire bppe = !bitp || (acq && (seqa==15) && (vcnt[1:0]==3));
  wire vcdn = (vcnt==15); // last phase of go cycle
  wire g0rst = gorst || usealt;
  always @(posedge intclk) begin
    gorst <= ply? !ena : !fenb;
    if (!acq)  igo <= 0; else igo <= (!throttle || muxcok) && ((nflowctrl || vordys) && wirdy) && !spacer;
    if (!ply)  ogo <= 0; else ogo <= (!throttle || muxcok) && ((nflowctrl || wordy) && (!good || virdy));
    if (g0rst) go0 <= 0; else go0 <= igo || ogo || (go0 && !(gov && vcdn));
    if (!gor) seqa <= 0; else if (goq) seqa <= seqa+1;
    if (!go)  vcnt <= 0; else if (gov) vcnt <= vcnt+1;
    if (gorst) go1 <= 0; else go1 <= go; 
    if (gorst) go2 <= 0; else go2 <= go1; 
    if (gorst) go3 <= 0; else go3 <= go2; 
    if (gorst) go4 <= 0; else go4 <= go3; 
    goalt <= ymixi[32];
    tcena <= votc;
    if (!ena) iflip <= 0; else if (go4) iflip <= !iflip;
    tflip <= flips[7] || (snibb&flips[3]);
    bflip <= tflip? {4{iflip}} : sbyte? {flips[3],flips[1],flips[2],flips[0]} : flips[3:0];
    swapn <= (nibb&sword)? {~swpsel,~swpsel,swpsel,swpsel}: (nibb&sbyte)? {~swpsel,swpsel,~swpsel,swpsel}: {4{swpsel}};
    mixinc <= ply? woena : seqdata[1]&bppe&go&(!shdr); // (wival)
    if (!ena) bcnt <= flip?3:0; else if (vival) bcnt <= flip? bcnt-1 : bcnt+1;
    vivald <= vival;
  end

  // programmed delays on pipelined signals
  wire [1:0] ulnena;
  wire [7:0] mixena;
  wire [15:0] mixsel;
  wire [3:0] seqadr = {sdds?(shdr||!sword):seqa[3], (snibb||seqa[2])&&(!sbyte), seqa[1:0]};

  srl16xN   #(32) seq (sclk, swrseq,  seqadr, swrbus, seqdata);

  srl16xNp  #(1)  sqf (intclk, H, 4'h1,(seqdata[0]|shdr)&gof, voenan);
  srl16xNp  #(1)  sq0 (intclk, H, 4'h1,(seqdata[0]|shdr)&go,  voenc);
  srl16xNp  #(1)  sqc (intclk, H, 4'h1,                  gon, vocnt); // special counter for SDN in packets
  srl16xNp  #(1)  sqj (intclk, H, 4'h3, seqdata[0]&gon,       wivalx); // QDR mode - direct - no mixer
  srl16xNp  #(1)  sqi (intclk, H, 4'h4, seqdata[1]&bppe &gon, wivaln);
  srl16xNp  #(1)  sq2 (intclk, H, 4'h0, seqdata[2]&bpps &go,  woenan);
  srl16xNp  #(1)  sq2x(intclk, H, 4'h3, seqdata[3]&go,        woenax);
  srl16xNp  #(1)  sq3 (intclk, H, 4'h4, seqdata[3]&go,        vival);
  srl16xNp  #(1)  sq3p(intclk, H, 4'h1, seqdata[3]&go,        vivalp); // early vival for test output generator
  srl16xNp  #(1)  sq4 (intclk, H, 4'h1, seqdata[4],           outsel);
  srl16xNp  #(1)  sq5 (intclk, H, 4'h2, seqdata[5],           swpsel);
  srl16xNp  #(2)  sq6 (intclk, H, 4'h3,~seqdata[7:6],         ulnena);
  srl16xNp  #(8)  sq8 (intclk, H, 4'h3, seqdata[15:8],        mixena);
  srl16xNp  #(16) sqm (intclk, H, 4'h3, seqdata[31:16],       mixsel);

  // SDDS cycle where sdata[2:0]=DataMode & sdata[12:8]=BitsPerSample
  wire stena; 
  reg sdgo,schk,schkd,schke,stenac,stval,stcyc,sthold;
  srl16xNp #(1) sqt (intclk, H, 4'h3, gos, stena);
  reg [4:0] scnt;
  reg [3:0] stcnt;
  wire rscnt = iorst || (sdgo && scnt[4]);
  always @(posedge intclk) begin
    if (iorst) stcnt <= 0; else if (stena) stcnt <= stcnt+1;
    if (!sdds) shdr  <= 0; else            shdr  <= (scnt==0);
    if (!sdds) spacer<= 0; else           spacer <= ((shdr || scnt[4]) && go1);
    if (rscnt) scnt  <= sfill; else if (sdgo) scnt <= scnt+1;
    sdgo   <= sdds && go0 && gov && vcdn;		// end of cycle in SDDS mode
    if (!sdds) schk <= 0; else schk <= (stcnt==1);	// parse the format word (after 8by ICE hdr)
    schkd  <= schk;
    schke  <= schkd && !swait;
    shdrd  <= shdr;
    if (!ena) snibb <= 0; else if (schk) snibb <= (muxer[1:0]==0);
    if (!ena) sbyte <= 0; else if (schk) sbyte <= (muxer[1:0]==1);
    if (!ena) sword <= 0; else if (schk) sword <= (muxer[1:0]==2);
    if (!ena) stval <= 0; else if (schkd) stval <= (muxer[7:6]!=0);
    stenac <= tcsdd && stval && (stcnt==5) && !swait && !sdiff;
    mixclr <= iorst || (shdrd && !shdr); // clear LSBs at end of SDDS header feed through
    if (!sdds) swait <= 0; else if (schk) swait <= (delay!=0);
    if (iorst) stcyc <= 0; else if (stval&mixclr) stcyc <= !stcyc;
  end
  wire mixclrd; delaypipe #(2) mcdp (intclk,mixclr,mixclrd);

  // SDDS sequence checks stcnt=(0:ice 2:schk/form 3:schkd/tcstat 4:schke 6:stenac)
  reg [3:0] sfillc;
  reg [7:0] seqnxt,seqsub;
  reg seqprime,schk1,schk2,sfilx,sfily,seqerr,seqdif,seqfil,seqoor,scross;
  wire [7:0] seqlsb = muxer[31:24];
  wire sfilld; srl16xNp sfd (intclk, H, 4'h1, sfill, sfilld);	// local fill (delay to Muxer clear)
  wire sfillb; srl16xN  sfb (intclk, H, 4'h2, (muxer[3:0]==4'h4), sfillb); // filled from Bill (in ICE header)
  always @(posedge intclk) begin
    if (schk) seqnxt <= seqlsb; else if (schk1 || schk2) seqnxt <= seqnxt+1;
    if (schk) seqsub <= seqlsb - seqnxt;
    if (schk) scross <= seqlsb[5] ^ seqnxt[5]; // does gap span the parity packet
    if (iorst) sdiff <= 0; else if (schkd) sdiff <= (seqsub != 0);
    schk1  <= schk || (muxy && schke);
    schk2  <= schk1 && (seqnxt[4:0]==30);
    sfilx  <= (stcnt==11);
    sfily  <= seqdif && scross; // dec fill count if contains parity packet
    seqoor <= (seqsub[7:4]!=0) || vopt;
    if (iorst) sfillc <= 0; else if (seqdif) sfillc <= seqsub[3:0]; else if ((rscnt&sfill) || sfily) sfillc <= sfillc-1;
    if (iorst) sfill  <= 0; else sfill <= sdds && (sfillc!=0);
    if (iorst) seqprime <= 0; else seqprime <= seqprime || (sfilx && !swait);
    if (!seqprime) seqdif <= 0; else seqdif <= sfilx && sdiff && !(seqoor||uopt);
    if (!seqprime) seqfil <= 0; else seqfil <= (sfill && rscnt) || (schkd && sfillb);
    if (!seqprime) seqerr <= 0; else seqerr <= sfilx && sdiff &&  (seqoor||uopt);
    if (!seqprime) seqfill <= 0; else if (seqfil) seqfill <= seqfill+1;
    if (!seqprime) seqerrs <= 0; else if (seqerr) seqerrs <= seqerrs+1;
  end

//  assign test = {sbyte,gof,shdr,schk,go,seqerr,seqfil,sfill};

  // complex test waveform {LRS,RAMP}
  reg [15:0] countl,counth;
  reg [25:0] lrs;
  reg ucount,voval,vovalx,wvalx;
  always @(posedge intclk) begin
    voval  <= flg_tout? fenb && !voval : ply? vivalp : sdncnt? vocnt : (voenc && !shdr && !swait);	// dont count the header words
    ucount <= voval && (countl==16'hFFFE);
    if (iorst) lrs <= 1;    else if (voval)  lrs <= {lrs[24:0], lrs[0]^lrs[1]^lrs[5]^lrs[25] };
    if (iorst) countl <= 0; else if (voval)  countl <= countl+1;
    if (iorst) counth <= 0; else if (ucount) counth <= counth+1;
    votc   <= up12? vovalx : voval;
  end
  wire [31:0] waves = {lrs[25:10],countl[7:0],countl[15:8]};

  // output mux
  wire [31:0] muxCB = (PORT==2)? {wodat[63:56],wodat[47:40],wodat[31:24],wodat[15:8]}
                               : {wodat[55:48],wodat[39:32],wodat[23:16],wodat[07:0]};
  wire [31:0] muxCI = (PORT==2)? {wodat[63:48],wodat[31:16]} 
			       : {wodat[47:32],wodat[15:00]};
  mux4xNp #(32) outerM (outer, wodat[31:0],wodat[63:32],muxCB,muxCI, {muxx,outsel},intclk);

  // source mux
  wire lsb = vodat[33];
  wire [31:0] vodatm = {msbi^vodat[31],vodat[30:17],lsbx?lsb:vodat[16],
                        msbi^vodat[15],vodat[14:1], lsbx?lsb:vodat[0]};
  wire [1:0]  muxsel = ialt? 3 : testdata? 2 : ply? 1 : 0;
  mux4xNpec #(32) muxerM (muxer, vodatm,outer,waves,ymixi[31:0], muxsel,intclk,H,sfilld);

  // bit-to-nibble collector
  reg [3:0] bitsid; 
  wire [3:0] bitsi = flip? {bitsid[2:0],inv?muxer[15]:muxer[31]}:{inv?muxer[15]:muxer[31],bitsid[3:1]};
  always @(posedge intclk) bitsid <= bitsi;
  wire [31:28] muxeq = (bitp && acq)? bitsi : muxer[31:28];

  // nibble swapper (per byte)
  wire [31:0] muxed = { swapn[3]?{muxer[27:24],muxeq[31:28]}:{muxeq[31:28],muxer[27:24]}, 
                        swapn[2]?{muxer[19:16],muxer[23:20]}:{muxer[23:16]},
                        swapn[1]?{muxer[11:08],muxer[15:12]}:{muxer[15:08]}, 
                        swapn[0]?{muxer[03:00],muxer[07:04]}:{muxer[07:00]} };

  // spectral flipper (per byte)
  wire [31:0] mixin = { bflip[3]?~muxed[31:24]:muxed[31:24], bflip[2]?~muxed[23:16]:muxed[23:16],
                        bflip[1]?~muxed[15:08]:muxed[15:08], bflip[0]?~muxed[07:00]:muxed[07:00]};

  // mixer
  wire [63:0] mixer;
  genvar i;
  generate
  for (i=0; i<8; i=i+1) begin:mix
    mux4xNpec #(4) lnib (mixer[i*8+3:i*8+0], mixin[03:00],mixin[11:08],mixin[19:16],mixin[27:24],
					     mixsel[i*2+1:i*2],intclk,mixena[i]&ulnena[0],mixclrd);
    mux4xNpec #(4) unib (mixer[i*8+7:i*8+4], mixin[07:04],mixin[15:12],mixin[23:20],mixin[31:28],
					     mixsel[i*2+1:i*2],intclk,mixena[i]&ulnena[1],mixclrd);
  end
  endgenerate

  // nibble-to-bit collector
  wire [3:0] bitso = mixer[31:28];
  reg bito; always @(posedge intclk) bito <= bitso[bcnt];

  // output lines
  wire [31:0] moxer = {msbi^(bitp?bito:mixer[31]),bitp?L3:mixer[30:28],mixer[27:17],lsbx?tcbito:mixer[16],
                       msbi^(bitp?bito:mixer[15]),bitp?L3:mixer[14:12],mixer[11:01],lsbx?tcbito:mixer[00]};

`ifdef IOM_QDR
  wire [63:0] mixit = ply? wodat : {vodat[67:36],vodat[31:0]};
  wire [63:0] mixjt = {vodat[67:53],lsbx?lsb:vodat[52],vodat[51:37],lsbx?lsb:vodat[36], vodat[31:17],lsbx?lsb:vodat[16],vodat[15:1],lsbx?lsb:vodat[0]};
  wire [1:0]   xsel = (TYPE==3)? 2 : ggo? 1 : up12? 3 : word? 1 : 0;
  reg wggo; always @(posedge intclk) wggo <= !flg_mggo || (vodat[0] || vodat[16] || vodat[36] || vodat[52]);
`ifdef IOM_UP12
  wire voenax,wenax; assign voena = up12? voenax : voenan;
  wire [63:0] mixcv; convert_12to16i #(64) cvt12 (intclk,gorst, lsbx&lsb, go,voenax,mixit, go,wenax,mixcv);
  wire [63:0] mixex; mux4xNp #(64) xnib (mixex, mixit,mixjt,mixit,mixcv, xsel,intclk);
  always @(posedge intclk) begin vovalx <= voenax; wvalx <= wenax; end
  wire wivaly = wggo && (up12? wvalx : wivalx);
`else
  wire voenax=L,wenax=L;
  assign voena = voenan;
  wire [63:0] mixcv;  
  wire [63:0] mixex;  mux2xNp #(64) xnib (mixex, mixit,mixjt, xsel,intclk);
  wire wivaly = wggo && wivalx;
`endif
  assign woena = flg_qdr? (up12? voenax : woenax) : woenan;
  assign wival = flg_qdr? wivaly : wivaln;
  assign widat = flg_qdr? mixex  : mixer;
  ff #(1) ff1 (vivalx, flg_qdr? (up12? wenax : vival) : flg_tout? voval : vival, intclk); // bit pack has extra output stage
  ff #(64) ff2 (vidat, flg_qdr? (up12? mixcv : wodat) : flg_tout? waves : mixer, intclk);

`else
  assign voena = voenan;
  assign woena = woenan;
  assign wival = wivaln;
  assign widat = mixer;
  assign vivalx= flg_tout? voval : bitp? vivald:vival; // bit pack has extra output stage
  assign vidat = flg_tout? waves : moxer;
`endif

  // envelope measurement and track algorithm
  wire vemt;
  emtfilter emt (intclk,iorst, cemt[11:0],cemt[15:12], wival,{widat[31:24],widat[15:8]}, vemt,demt);

  // cross channel data output - go gets registered in alt module
  assign ymixo[32]   = ply? (bitp?go2:go1) : go0;
  assign ymixo[31:0] = ply? moxer : muxer[31:0];
  
  // tuner syncronization bit
  reg [31:0] oscnt;
  reg ostat,ot1,ot2,sen2;
  wire oenad; srl16xN oenadelay (ioclk,H, 4'h5, oena,oenad);
  always @(posedge ioclk) begin
    ot1 <= (oscnt[2:0]==7);
    ot2 <= ot1 && (oscnt[15:3]==13'h1FFF);
    if (iorst) oscnt[2:0] <= 0; else if (oena) oscnt[2:0] <= oscnt[2:0]+1;
    if (iorst) oscnt[15:3] <= 0; else if (ot1) oscnt[15:3] <= oscnt[15:3]+1;
    if (iorst) oscnt[31:16] <= 0; else if (ot2) oscnt[31:16] <= oscnt[31:16]+1;
    if (iorst) tsync <= 0; else tsync <= sen && fenb;
    if (iorst) ostat <= 0; else ostat <= ostat_ && !hold;
    if (iorst) sen2  <= 0; else sen2  <= sen || (sen2 && !oenad);
  end
  assign cdat  = oscnt;
  assign osel  = {6'd0,ten,sen2};

  // SDN timecode handler
  wire [3:0]  tadrs,tcbits; 
  wire [31:0] tdats;
  wire [31:0] countd;
  wire [1:0] tcbadr = {mixsel[1],sdds};
  wire tcwrs,tcwrcs,tvals,tcycs,tdcs,tnclks;
  mux4xN tcmux0 (tcbits[0], muxer[00+3],muxer[00+1],muxer[00+4],muxer[00+0], ibit);
  mux4xN tcmux1 (tcbits[1], muxer[08+3],muxer[08+1],muxer[08+4],muxer[08+0], ibit);
  mux4xN tcmux2 (tcbits[2], muxer[16+3],muxer[16+1],muxer[16+4],muxer[16+0], ibit);
  mux4xN tcmux3 (tcbits[3], muxer[24+3],muxer[24+1],muxer[24+4],muxer[24+0], ibit);
  reg tcbip,tcbjp,tcbop,tcbit;
  always @(posedge intclk) begin
    tcbit <= tcbits[tcbadr];
    tcbip <= tcswp? vodat[32] : inv? vodat[0] : vodat[16];
    tcbjp <= tcswp? vodat[33] : vodat[32];
    tcbop <= (TYPE==0)? tcbjp : tcbip;
  end
  tcreader #(TYPE) tcr (intclk,frst, tcmode, tcena,tcbit,tcbop, sclk,oppms,xsyncf,
			tcwrs,tcwrcs,tadrs,tdats, tvals,tcycs,tdcs,tnclks, ttest);
  srl16xN #(32) tcd (intclk,H,4'h2, {counth,countl},countd);

  // timecode fields
  wire tcwrcd = stenac || tcwrcs;
  wire tcsddx = tcsdd || stena;
  always @(posedge intclk) begin
    if (iorst) tval <= 0; else tval <= tval || (tcsdd? stenac : tvals);
    twr  <= stena || tcwrs;
    tdat <= tcwrcd? countd : tcsddx? muxer : tdats;
    tadr <= tcsddx? {1'h1,stcyc,stcnt[3:0]} : {2'h0,tadrs};
    tcyc <= tcsdd? stcyc : tcycs;
    tdc  <= tdcs;
    tnclk <= tnclks;
  end

  // time code generator
  tcgen tcg (intclk,iorst, sclk,swriot,swrbus, vival,tcbito);

  // data gap timer (use transition from SDDS hdr to data)
  reg ginit,gena,gset,gsetd,grset,gup,gdn,gzero;
  reg [15:0] gmin,gmax,gcur;
  always @(posedge sclk) begin
    gena  <= fenb && ( (gset && !gsetd) || gena);
    ginit <= !fenb || (srd && scs && gstat);
    gzero <= (gcur==0);
    gset  <= sdds? !shdr : L;
    gsetd <= gset;
    grset <= (gset && !gsetd) || !gena;
    gup   <= (gcur>gmax) && (gset && !gsetd && !gzero);
    gdn   <= (gcur<gmin) && (gset && !gsetd && !gzero);
    if (grset) gcur <= 0;        else if (oppus) gcur <= gcur+1;
    if (ginit) gmax <= 16'h0000; else if (gup) gmax <= gcur;
    if (ginit) gmin <= 16'hFFFF; else if (gdn) gmin <= gcur;
  end

  assign mcfg = {qmux,iready,master,flg_sdds,iom};
  assign gdat = flg_sdds? {gmax,gmin} : 0;

                always @(posedge fclk) ftest <= fival^fjval;
  reg testvclk; always @(posedge gclk) testvclk <= !testvclk;
  reg testfclk; always @(posedge fclk) testfclk <= !testfclk;
  reg testmclk; always @(posedge mclk) testmclk <= !testmclk;
  reg [15:0] z; always @(posedge fclk) z <= {fibusn[7:0],fibusp[7:0]};
  reg ctest;    always @(posedge fclk) ctest <= (z[15:8]^z[7:0]) != 8'hFF;
  reg dtest;    always @(posedge fclk) dtest <= (z[7:4]==z[3:0]);
  reg etest;    always @(posedge fclk) etest <= (z[7:4]==z[11:8]);
  assign test = {oena,iena,voena,ftest, fjval,fival,fsync,master};
//  assign test = {oena,iena,foena,fival, wivalx,wenax,voena,go};
//  assign test = {oena,iena,foena,fival, xsyncf,fsync,testfclk,master};	// master is required now
//  assign test = {iena,istat,foena,fival, fordy,wordy,virdy,master};	// master is required now
//  assign test = {oena,iena,foena,fival,muxq[1],mixq[1],mixdif[1],muxcok};
//  assign test = {s1,muxclk,jclk,iclk,usemux,clkm, iready,master};	// master is required now
//  assign test = {cgot,vic,foena,fordy, go,frst,eclk,master};	// master is required now
//  assign test = {oena,iena,fjval,fival, iready,dtest,ctest,master};	// master is required now
//  assign test = {oena,iena,foena,fival, testfclk,testvclk,fclk&good,gclk&good};

endmodule

/*

  Hyperlink protocol

    RE  0xA5  Sync
    FE  0x00  Flags (TC valid 0x01)
    RE  0xnn  Size in number of 64by packets
    FE  0xFF  Auxiliary data (TC offset)
    RE        Data Byte 0
    FE        Data Byte 1
    ...
    RE        Data Byte N-2
    FE        Data Byte N-1

    RE  train Training sequence NN where N is 4bit incrementing counter
    FE ~train Complement of train

    ibusd   icnt[4:0]      
    0x01A5  0
    0xFFnn  0	isync,jsync
    0xDATA  0	icycle,jsyncd
    0xDATA  31	icycle,tcval
    0xDATA  30	icycle
    0xDATA  29	icycle,itick
      ...
    0xDATA  1	icycle
    0xDATA  0	icycle
    0xDATA  31	icycle
    0xDATA  30	icycle
      ...
    0xDATA  2	icycle
    0xDATA  1	icycle,idone
    0xDATA  0	

    tst
    0x00
    0x11
    0x22  oready
    0x33  ostart
    0x44  osel
		
*/

// this module must be placed near the pins
module ioprep (fclk,frst,fenb, fibusp,fibusn,fibus,firdy,fival, riena,
               gclk,grst,genb, fobusp,fobusn,fobus,fordy,foval, roena,
                               dir,acq,ply,ggo,rordy,ncclk,xonoff,burst,delay,fsync,stx,afull,test);
  parameter TYPE=0;
  parameter PORTW=8;

  localparam DPORTW=PORTW*2;
  localparam XPORTW=DPORTW/8;
  localparam FPORTW=DPORTW+XPORTW;

  input  fclk,frst,fenb,firdy, gclk,grst,genb,fordy,riena, dir,acq,ply,ggo,rordy,ncclk,xonoff,burst;
  output fival,foval,roena;
  input  [PORTW-1:0] fibusp,fibusn;
  output [PORTW-1:0] fobusp,fobusn;
  output [FPORTW-1:0] fibus;
  input  [DPORTW-1:0] fobus;
  input  [7:0] delay;

  input  fsync,stx;
  output afull;
  output [7:0] test;

  wire L=0, H=1;
  wire [DPORTW-1:0] ibus;
  wire [FPORTW-1:0] ibuse;
  reg  [DPORTW-1:0] ibusd,obus,obusd;
  wire [XPORTW-1:0] xbus = {XPORTW{fsync}};
  reg  [XPORTW-1:0] fxbus;

  assign ibus   = {fibusn,fibusp};
  assign fobusn = obusd[DPORTW-1:PORTW];
  assign fobusp = obusd[PORTW-1:0];

  wire calib = (TYPE==0)? ggo : L;
  wire gated = (TYPE==0)? L : (ggo && ibusd[1]);

  // input startup
  reg ienb,rienb;
  always @(posedge fclk or posedge frst) begin
    if (frst) ienb <= 0; else ienb <= fenb && acq && !gated;
  end
  srl16xNp #(XPORTW+2) srlx (fclk,H,delay[7:4], {xbus,fibusn[0],fibusp[0]},  {ibuse[FPORTW-1:DPORTW],ibuse[PORTW],ibuse[0]});
  srl16xNp #(DPORTW-2) srld (fclk,H,delay[3:0], {fibusn[PORTW-1:1],fibusp[PORTW-1:1]}, {ibuse[DPORTW-1:PORTW+1],ibuse[PORTW-1:1]});

  //input
  reg [7:0] icnt;
  reg isync,jsyncd,jsync,ksync,afull,iready,icycle,idone,itick,ilow;
  always @(posedge fclk) begin
    iready <= ienb && (firdy || !burst);
    ibusd  <= ibus;
    rienb  <= fenb && riena;
    if (icycle) isync <= 0; else isync <= (ibusd[07:0]==8'hA5);
    if (icycle) jsync <= 0; else jsync <= (ibusd[11:8]==4'h1);  // TC valid
    if (icycle) ksync <= 0; else ksync <= (ibusd[15:12]>14);    // FIFO full flag
    icycle <= ienb && (isync || calib || (icycle && !idone));
    ilow   <= (icnt[7:4]==0);
    idone  <= ilow && (icnt[3:0]==2); // (icnt==2)
    if (!icnt[4]) itick <= 0; else itick <= (icnt[3:0]==14);  // (icnt[4:0]==30)
    if (!icycle) icnt[4:0] <= 0;          else            icnt[4:0] <= icnt[4:0]-1; // 32 2by words per packet
    if (!icycle) icnt[7:5] <= ibusd[2:0]; else if (itick) icnt[7:5] <= icnt[7:5]-1; // 64by packet count
    if (!ienb) afull <= 0; else if (isync) afull <= ksync;
  end

  // input opps trigger
  reg tcval,tcbit,tcbar,tcuse;
  reg [7:0] tcoff;
  always @(posedge fclk) begin
    jsyncd <= isync && jsync;
    if (isync) tcoff <= ibusd[15:8];
    tcval <= jsyncd || (tcval && (icnt[4:0]!=0));	// 32 clocks
    tcbit <= tcoff[ ~icnt[3:1] ];			// weave count into tcbit for a 32b interface
    if (!tcval) tcbar <= 0; else tcbar <= ~icnt[3];
    if (!tcval) tcuse <= 0; else tcuse <= ~icnt[4];
    fxbus <= {fsync,tcuse?tcbit:tcbar};
  end

  assign fival = (TYPE==0)? icycle : (TYPE==3)? rienb : ienb;
  assign fibus = (TYPE==0 || TYPE==3)? {fxbus,ibusd} : ibuse;
    
  // output startup
  reg oenb;
  wire ordy = (TYPE==0)? H : (TYPE==3)? (fordy && rordy) : (fordy || (oenb && !ncclk));
  always @(posedge gclk or posedge grst) begin
    if (grst) oenb <= 0; else oenb <= genb && ply && ordy;
  end

  // output
  reg oready,ostart,ostartd,ocyclep,ocycle,odone,otick,olow,irchk;
  reg roena,roenb,roenc;
  reg [7:0] ocnt,ocycled;
  reg [1:0] osel;
  reg [3:0] tst;
  wire ionoff = !dir && xonoff && !calib;
  wire dsel = (TYPE!=0) || ocycled[1];
  wire irline; BUF orb (irline,ibusd[0]);
  wire owait = stx && ocycled[3]; // delay for bill
  always @(posedge  gclk) begin
    oready  <= oenb && (!xonoff || irline) && fordy && (stx || tst[0]); // align on word boundary
    ostart  <= !ostart && !ostartd && !ocycle && oready && !owait;
    ostartd <= ostart;
    ocyclep <= ostart || (ocyclep && !odone);
    ocycle  <= ocyclep; 
    ocycled <= {ocycled[6:0],ocycle};
    olow    <= (ocnt[7:4]==0);
    odone   <= olow && (ocnt[3:0]==3); // (ocnt==3)
    if (!ocnt[4]) otick <= 0;    else otick <= (ocnt[3:0]==14); // (ocnt[4:0]==30)
    if (!ocycle) ocnt[4:0] <= 0; else            ocnt[4:0] <= ocnt[4:0]-1; // 32 2by words per packet
    if (!ocycle) ocnt[7:5] <= 4; else if (otick) ocnt[7:5] <= ocnt[7:5]-1; // 64by packet count = 4
    tst   <= tst+1;
    osel  <= ostart? 2 : ostartd? 3 : ionoff? 1 : 0;
    obus  <= (osel==0)? {~tst,~tst,tst,tst} : (osel==1)? {2{7'b0,iready}} : (osel==2)? 16'h00A5 : 16'hFF04;
    obusd <= dsel? fobus : obus;
    roenc <= oenb;
    roenb <= roenc;
    roena <= roenb;
    irchk <= irline;
  end
  assign foval = (TYPE==0)? ocycle : oenb;

  always @(posedge fclk) begin
    if (TYPE==0 && (ocyclep|ocycled[4])) $write("cycle=%d vblk=%x vcnt=%x vout=%x\n",ocycle,ocnt[7:5],ocnt[4:0],obusd);
    if (TYPE!=0 && foval) $write("Output=%x\n",obusd);
  end

//  assign test = {fxbus,irchk,isync,jsync,jsyncd,tcval,irchk};
  assign test = {oready,ocycle,ionoff,xonoff,irchk,ply,acq,dir};

endmodule

module mcounter (clk,rst,count);
  input clk,rst;
  output reg [15:0] count;
  reg ena;
  always @(posedge clk or posedge rst) begin
    if (rst) ena <= 0; else ena <= 1;
    if (rst) count <= 0; else if (ena) count <= count+1;
  end
endmodule

module iobuffer (ip,in,ipnb, clk, opn,opb,onb);
  parameter TYPE=0;
  parameter BW=8;
  localparam DBW=BW*2;
  input clk;
  input [BW-1:0] ip,in;
  input [DBW-1:0] opn;
  output reg [BW-1:0] opb,onb;
  output reg [DBW-1:0] ipnb;
  reg [BW-1:0] onb_,ipb_,inb_;
  always @ (posedge clk) begin
    ipb_ <= ip[BW-1:0];
    ipnb[BW-1:0]   <= ipb_;
    ipnb[DBW-1:BW] <= inb_;
    opb  <= opn[BW-1:0];
    onb_ <= opn[DBW-1:BW];
  end
  always @ (negedge clk) begin
    inb_ <= in[BW-1:0];
    onb  <= onb_;
  end
endmodule


/*
  fclk is actually the intclk internal clock after the input FIFO
  sclk is the 100MHz system clock
  oppms is a one pulse per millisecond enable at sclk

  IRIG-B  is one pulse every 10ms

  2ms = zero
  5ms = one
  8ms = sync

  double sync is 11ms past the nominal 1pps

*/


module tcreader (fclk,frst, tcmode, tena,tcbit,tcbop, sclk,oppms,xsync, 
		  tcwr,tcwrc,tadr,tdat, tval,tcyc,tdc,tnclk, test);
  parameter TYPE=0;
  input fclk,frst, tena,tcbit,tcbop;
  input sclk,oppms,xsync;
  input [15:0] tcmode;
  output reg tcwr, tval, tdc;
  output tcwrc, tcyc, tnclk;
  output [3:0]  tadr;
  output [31:0] tdat;
  output [7:0] test;

  wire tsdn = tcmode[8];
  wire nodc = tcmode[7];
  wire fill = tcmode[6];
  wire irig = tcmode[5];
  wire opps = tcmode[4];
  wire swap = tcmode[3];

  reg [7:0] tcount;
  reg [31:0] tcdata;
  reg thitsc, thitdc, thitop, thitir, tcycle, thold, tdone, tcload, tdcyc, tcbir, ghit, ihit;

  wire tcb = (irig && tcycle)? tcbir : (irig || opps)? tcbop : tcbit;

  wire tdload = tena && !thold;
  wire tcena  = tena && tcycle && !thold;
  wire filled = (tcdata[3:0]==4'b1111);

  // SMS, SDN, and Datolite handler
  always @(posedge fclk) begin
    if (frst) thitsc <= 0; else if (tena) thitsc <= (tcdata[10:4]==7'b0100111) && (filled || !fill) && tsdn;
    if (frst) thitdc <= 0; else if (tena) thitdc <= (tcdata[17:4]==14'b00110000111111) && (filled || !fill) && tsdn && !nodc;
    if (frst) thitop <= 0; else if (tena) thitop <= (tcdata[13:10]==4'b1100) && opps && (!irig || ghit);
    if (frst) thitir <= 0; else if (tena) thitir <= ihit && irig && !opps;
    if (frst) tcycle <= 0; else if (tena) tcycle <= tcycle? !tdone : (thitop || thitsc || thitdc || thitir);
    if (frst) tdc    <= 0; else if (tena) tdc    <= tcycle? tdc : (thitdc || (tdc && !thitsc));
    if (frst) tdcyc  <= 0; else if (tena) tdcyc  <= (tdc && tcount[6:0]==1) || (tdcyc && tcount[6:0]!=95);
    if (frst) thold  <= 0; else if (tena) thold  <= (tdcyc && !thold);
    if (frst) tdone  <= 0; else if (tena) tdone  <= (tcount[6:0]==126);
    if (frst) tcload <= 0; else if (tena) tcload <= (tcount[6:0]==97);
    if (frst) tcount <= 0; else if (tcena) tcount <= tcount+1;
    if (frst) tcdata <=~0; else if (tdload) tcdata <= {tcb,tcdata[31:1]};
    if (frst) tval   <= 0; else if (tena) tval   <= (tval || tdone);
    if (frst) tcwr   <= 0; else           tcwr   <= tena && (tcount[4:0]==1);
  end

  // IRIG-B handler  (oppms L=1-3 H=4-6 S=7-9)
  reg sre,sfe,sv,slv,sdat,sync,synd,syne;
  reg [3:0] scnt;
  wire sout;
  always @(posedge sclk) begin
    sv  <= xsync;
    slv <= sv;
    sre <= sv && !slv;
    sfe <= !sv && slv;
    if (sre) scnt <= 0; else if (oppms) scnt <= scnt+1;
    if (sfe) sdat <= (scnt>=4);	// data 100Hz
    if (sfe) sync <= (scnt>=7);	// sync 10Hz
    if (sfe) synd <= sync;
    if (sre) syne <= sync && synd;
  end
  reg fre,fv,flv;
  wire irst = frst || tcycle || !irig;
  always @(posedge fclk) begin
    fv  <= xsync;
    flv <= fv;
    fre <= fv && !flv;
    if (frst) ghit <= 0;  else if (fre) ghit <= (sync && synd) || ghit;
    if (irst) ihit <= 0;  else if (fre) ihit <= (sync && synd) && ghit;
    if (tena) tcbir <= sout;
  end
  wire sena = tcycle? tena : fre;
  srlMxN #(7) isr (fclk,sena,7'h64,sdat,sout);

  assign tadr = {1'b1,tcount[7:5]};
  assign tcyc = tcount[7];
  assign tnclk = 0;
  assign tdat = tcdata;
  assign tcwrc = tcload;

  assign test = {irig,oppms,sv,sre,sdat,sync,tcycle,thitir};

endmodule



module tcgen (clk,rst, sclk,scs,sdata, dena,dbit);

  parameter CBW=17;

  input clk,rst,sclk,scs;
  input [31:0] sdata;
  input dena;
  output dbit;

  reg [CBW-1:0] data, ticker, count;
  reg scsd,csd,cs;

  always @(posedge sclk) begin
    if (scs) data <= sdata;
    if (scs) scsd <= !scsd;
  end
  always @(posedge clk) begin
    cs <= (csd^scsd) && !cs;
    if (cs) csd <= !csd;
  end

  wire zero = (count==0);
  wire load = cs || (zero&dena);
  always @(posedge clk) begin
    if (load) count <= ticker; else if (dena) count <= count-1;
    if (cs) ticker <= data[CBW-1:0];
  end

  reg [1:0] ibit;
  reg [3:0] curp;
  reg cyc,inc,dbit,skip,trip;
  reg [15:0] tcdat; 
  wire [15:0] tcdati,tcdato;

  wire [3:0] cur  = tcdat[3:0];
  wire [3:0] max  = tcdat[7:4];
  wire [1:0] bits = tcdat[9:8];
  wire pred       = tcdat[12];	// number pre-trigger
  wire posd       = tcdat[13];	// number post-trigger
  wire trig       = tcdat[14];
  wire done       = tcdat[15];
  wire next       = (ibit==bits)&dena;
  wire nex1       = (ibit==1)&dena;
  wire over       = (cur==max);
  wire overp      = (curp==max);
  wire same       = !(pred || posd) || !inc;
  always @(posedge clk) begin
    if (rst) cyc <= 0; else if (dena) cyc <= zero || (cyc && !(next&done));
    if (rst|next) ibit <= 0; else if (cyc&dena) ibit <= ibit+1;
    if (dena) curp <= same? cur : over? 0 : cur+1;
    if (next) inc <= trig || (trip&!posd) || (over & inc);
    if (rst|dena) dbit <= cur[ibit];
    if (nex1) trip <= trig || (trip && (overp|!posd)) && !pred;
//    if (cyc&dena) $write("TcOut %04x ena %d over %d inc %d trig %d trip %d\n",tcdat,dena,over,inc,trig,trip);
  end

  wire tcena = cs | (next&dena);
  assign tcdati = cs? count[15:0] : {tcdat[15:4],curp};
  srl16xN #(16) tcpipe (clk,tcena, 4'hF,tcdati,tcdato);
  always @(posedge clk) if (tcena) tcdat <= tcdato;

endmodule

