/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  ICE HyperLink Input/Output Controllers

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  1/15/2003

  IO=SS

  Input: 210Mby each channel
   iclkp = input clock
   iclkn = A/B valid
   idatp = A/B  7:0
   idatn = A/B	15:8

  Output:	100Mby each channel
   oclkp = A valid
   oclkn = B valid
   odatp = A  7:0
   odatn = B  7:0

  IO=HH

  Input: 532Mby each channel
   iclkp = input clock
   iclkn = input clock
   idatp = A/B  7:0
   idatn = A/B	15:8

  Output: 532Mby each channel
   oclkp = output clock
   oclkn = output clock
   odatp = A/B  7:0
   odatn = B/B  15:8

  Packet Header: 0xccssmmA5
    A5 = start symbol
    mm = command flags 0=ohold
    ss = size in 64by chunks
    cc = channel number

   Test Pattern: 

     FF:00 EE:11 DD:22 CC:33   BB:44 AA:55 99:66 88:77   77:88 66:99 55:AA 44:BB   33:CC 22:DD 11:EE 00:FF

     Bit#0 0110011001100110 0110011001100110
     Bit#2 0101101001011010 0101101001011010
     Bit#3 0101010110101010 0101010110101010
     Bit#4 0101010101010101 1010101010101010

*/
`include "mdefs.h"
`include "dmadefs.h"
`include "ramdefs.h"
`include "hypdefs.h"

module hypinqx (iclk,iena,idat, hval,hsel,hdat, mode,hold,csm,cab,calib,test);
  parameter IBW=32;
  parameter OBW=32;
  parameter NPKT=1;
  parameter TEST=0;

  localparam OBV=OBW/8;
  localparam IBV=IBW/8;

  input iclk,iena;
  input [IBW-1:0] idat;
  output hval;
  output [OBV-1:0] hsel;
  output [OBW-1:0] hdat;
  input  mode;
  output [3:0] hold;
  output csm,cab;
  output [7:0] calib;
  output [7:0] test;

  wire L=0, H=1;

  wire foena;
  wire [OBW-1:0] fobus;

  // packet logic
  reg syt0,syt1,syt0d,syt1d,oop,active;
  reg isyna,isynb,isync,icycle,idone,ipha;
  reg [IBW-1:0] jdat,kdat;
  reg [7:0] icnt,ichan;
  reg [3:0] hold;
  reg sen,ten,chm,chb;
  wire [3:0] npkt = (NPKT>0)? NPKT : kdat[19:16];
  wire [7:0] vchan = kdat[31:24];
  wire vchb; vote3 v1 (vchb, vchan[4],vchan[2],vchan[0]);
  wire vsen; vote3 v2 (vsen, vchan[5],vchan[3],vchan[1]);
  wire vten; vote3 v3 (vten, vchan[7],vchan[6],1'b0);
  wire voteh; vote3 vh (voteh, vchan[7],vchan[6],vchan[5]);
  wire votel; vote3 vl (votel, vchan[2],vchan[1],vchan[0]);
  wire ichn = isync;
  always @(posedge iclk) begin
    syt0 <= (jdat[ 7: 0]==8'hAA); syt0d <= syt0 && !(isync || active);
    syt1 <= (jdat[23:16]==8'hAA); syt1d <= syt1 && !(isync || active);
    oop  <= L;
    jdat <= idat;
    kdat <= jdat;
    isyna  <= (idat[7:0]==8'hA5);
    isynb  <= (idat[23:16]==8'hA5);
    isync  <= !isync && !(icycle && !idone) && (oop? isynb : isyna);
    icycle <= (isync && kdat[18:16]!=0) || (icycle && !idone) || mode;
    idone  <= icycle && (icnt[5:0]==2);
    if (isync) icnt <= {npkt,4'h0}; else if (icycle) icnt <= icnt-1; // 16x4by pkt count for NPKT 
    if (isync) ipha <= 0; else ipha <= !ipha;
    if (mode) chb <= 0; else if (ichn) chb <= vchb;
    if (mode) chm <= 0; else if (ichn) chm <= voteh;
    if (mode) sen <= 0; else if (ichn) sen <= vsen;
    if (mode) ten <= 0; else if (ichn) ten <= vten;
    if (mode) hold <= 0; else if (ichn) hold <= kdat[11:8];
    if (mode) ichan <= 0; else if (ichn) ichan <= voteh? {H,L,L,vchan[4:0]} : {L,{6{L}},votel};
  end

`ifdef ICE_MB
  wire [OBV-1:0] ksel = ipha? ichan[7:4] : ichan[3:0];	// need full input channel address
  assign cab  = ichan[0];
`else
  wire [OBV-1:0] ksel = {ten,sen,ten,sen};		// only A/B channel number and startup synchronization signals are important
  assign cab  = chb;
`endif

  assign hval = icycle;
  assign hsel = ksel;
  assign hdat = kdat;
  assign csm  = chm;

  // calibration check
  reg err,estr,eend;
  reg [3:0] tst,errc;
  reg [7:0] tcnt,errs,terrs;
  wire [3:0] tst0 = {tst[3:1],L};
  wire [3:0] tst1 = {tst[3:1],H};
  always @(posedge iclk) begin
    if (isync|icycle) tcnt <= 0; else tcnt <= tcnt+1;
    estr <= (tcnt==5);
    eend <= (tcnt==255);
    if (estr) tst <= {jdat[19:17],L}; else tst <= tst+2;
    if (estr) errs <= 0; else if (err) errs <= errs+1;
    // err <= {~tst1,~tst1,tst1,tst1,~tst0,~tst0,tst0,tst0} != jdat;
    errc[0] <= kdat[ 7: 0] != { tst0, tst0};
    errc[1] <= kdat[15: 8] != {~tst0,~tst0};
    errc[2] <= kdat[23:16] != { tst1, tst1};
    errc[3] <= kdat[31:24] != {~tst1,~tst1};
    active  <= isync || icycle || (active && !tcnt[3]);
    if (active) err <= 0; else err <= (errc!=0);
    if (eend) terrs <= errs;
  end
  assign calib = terrs;

  reg fff; always @(posedge iclk) fff <= !fff;
  reg ccc; always @(posedge iclk) ccc <= (terrs!=0);
//  assign test = {fff,err,estr,oop,mode,chb,icycle,isync};
  assign test = {fff,err,estr,oop, ccc,mode,icycle,isync};
//  assign test = kdat[7:0];

endmodule

module hypoutiom (iclk,iena,idat, oclk,good,odat);
  parameter IBW=32;
  parameter OBW=16;
  parameter TEST=0;

  input iclk,oclk;
  input iena,good;
  input [IBW-1:0] idat;
  output [OBW-1:0] odat;

  wire irdy,ordy,oena;
  wire frst = !good;
  wire [15:0] fobus;

  fifoNxM #(IBW,OBW,0,3,0,1) fifo (iclk,frst, irdy,iena,idat,  oclk,frst, ordy,oena,fobus);

  // output
  reg ostart,ostartd,ocyclep,ocycle,odone,otick,olow;
  reg [7:0] ocnt,ocycled;
  reg [1:0] osel;
  reg [3:0] tst;
  reg [15:0] obus,obusd,obusc;
  wire dsel = ocycled[1];
  always @(posedge  oclk) begin
    ostart  <= !ostart && !ostartd && !ocycle && ordy;
    ostartd <= ostart;
    ocyclep <= ostart || (ocyclep && !odone);
    ocycle  <= ocyclep;
    ocycled <= {ocycled[6:0],ocycle};
    olow    <= (ocnt[7:4]==0);
    odone   <= olow && (ocnt[3:0]==3); // (ocnt==3)
    if (!ocnt[4])    otick <= 0; else otick <= (ocnt[3:0]==14); // (ocnt[4:0]==30)
    if (!ocycle) ocnt[4:0] <= 0; else            ocnt[4:0] <= ocnt[4:0]-1; // 32 2by words per packet
    if (!ocycle) ocnt[7:5] <= 4; else if (otick) ocnt[7:5] <= ocnt[7:5]-1; // 64by packet count = 4
    tst   <= tst+1;
    osel  <= ostart? 2 : ostartd? 3 : 0;
    obus  <= (osel==0)? {~tst,~tst,tst,tst} : (osel==2)? 16'h00A5 : 16'hFF04;
    obusd <= dsel? (TEST==1)? obusc:fobus : obus;
    obusc <= obusc+1;
  end
  assign oena = ocycle;
  assign odat = obusd;

endmodule

module hypoutiom2 (iclk,iena,idat, good, oclk,odat, test);
  parameter IBW=32;
  parameter OBW=16;
  parameter NFLGS=0;
  parameter MFLGS=0;
  parameter TEST=0;

  input iclk,oclk;
  input iena,good;
  input [IBW-1:0] idat;
  output [31:0] odat;
  output [7:0] test;

  wire irdy,ordy,oena;
  wire frst = !good;
  wire [OBW-1:0] fobus;
  wire L=0, H=1;

  wire hyptst;
`ifdef HYP_IOMTC
  localparam IBWB = IBW/9;
  localparam OBWB = OBW/9;

  wire pps_tick1 = (idat[IBW-1:IBW-IBWB] != 0);

`ifdef USEBPAR
  fifoNxM #(IBW,OBW,0,3,`BPAR,`BPAR) fifod (iclk,frst, irdy,iena,idat[IBW-1:0],  oclk,frst, ordy,oena,fobus[OBW-1:0]);
`else
  fifoNxM #(IBW-IBWB,OBW-OBWB,0,3) fifod (iclk,frst, irdy,iena,idat[IBW-IBWB-1:0],  oclk,frst, ordy,oena,fobus[OBW-OBWB-1:0]);
  fifoNxM #(IBWB,OBWB,0,3) fifop (iclk,frst, ,iena,idat[IBW-1:IBW-IBWB],  oclk,frst, ,oena,fobus[OBW-1:OBW-OBWB]);
`endif

  wire pps_tick2 = (fobus[OBW-1:OBW-OBWB] != 0);

  hypoutq #(OBW,32,4,`HYP_TC) hpo (oclk,H,odat, ordy,oena,8'h00,fobus, 4'b0,hyptst);
`else
  wire pps_tick1 = 0;
  fifoNxM #(IBW,OBW,0,3,`BPAR,`BPAR) fifo (iclk,frst, irdy,iena,idat,  oclk,frst, ordy,oena,fobus);
  wire pps_tick2 = 0; 

  hypoutq #(OBW,32,4) hpo (oclk,H,odat, ordy,oena,8'h00,fobus, 4'b0,hyptst);
`endif
 
  assign test = {hyptst,pps_tick2,pps_tick1};

endmodule

module hypoutiomfmt3 (iclk,iena,idat, good,oclk,odatb,odata);
  parameter IBW=32;
  parameter OBW=16;
  parameter NFLGS=0;
  parameter MFLGS=0;
  parameter TEST=0;

  input iclk,oclk;
  input iena,good;
  input [IBW-1:0] idat;
  output [31:0] odata;
  output [31:0] odatb;

  wire irdy,ordy,oena,oenaa,oenab;
  wire frst = !good;
  wire L=0, H=1;

  wire hyptsta;
  wire hyptstb;

`ifdef HYP_IOMTC
  localparam IBWB = IBW/9;
  localparam OBWB = OBW/9;
  wire [2*(OBW-OBWB)-1:0] fobus;
  wire [2*OBWB-1:0] fopps;

  assign oena = oenaa & oenab;
  fifoNxM #(IBW-IBWB,2*(OBW-OBWB),0,3,`BPAR,`BPAR) fifo (iclk,frst, irdy,iena,idat[IBW-IBWB-1:0],  oclk,frst, ordy,oena,fobus);
  fifoNxM #(IBWB,2*OBWB,0,3,`BPAR,`BPAR) fifop (iclk,frst, ,iena,idat[IBW-1:IBW-IBWB],  oclk,frst, ,oena,fopps);

  hypoutq #(OBW,32,4,`HYP_TC) hpoa (oclk,H,odata, ordy,oenaa,8'h00,{fopps[OBWB-1:0],fobus[OBW-OBWB-1:0]}, 4'b0,hyptsta);
  hypoutq #(OBW,32,4,`HYP_TC) hpob (oclk,H,odatb, ordy,oenab,8'h00,{fopps[2*OBWB-1:OBWB],fobus[2*(OBW-OBWB)-1:OBW-OBWB]}, 4'b0,hyptstb);

`else
  wire [2*OBW-1:0] fobus;

  assign oena = oenaa & oenab;
  fifoNxM #(IBW,2*OBW,0,3,NFLGS,MFLGS) fifo (iclk,frst, irdy,iena,idat,  oclk,frst, ordy,oena,fobus);
  
  hypoutq #(OBW,32,4) hpoa (oclk,H,odata, ordy,oenaa,8'h00,fobus[OBW-1:0], 4'b0,hyptsta);
  hypoutq #(OBW,32,4) hpob (oclk,H,odatb, ordy,oenab,8'h00,fobus[2*OBW-1:OBW], 4'b0,hyptstb);

`endif
 
endmodule

module hypoutiom3 (iclk,iena,idat, good,ifmt, oclk,odatb,odata);
  parameter IBW=32;
  parameter OBW=16;
  parameter NFLGS=0;
  parameter MFLGS=0;
  parameter TEST=0;

  input iclk,oclk;
  input iena,good,ifmt;
  input [IBW-1:0] idat;
  output [31:0] odata;
  output [31:0] odatb;

  wire irdy,ordy,oena,oenaa,oenab;
  wire frst = !good;
  wire L=0, H=1;

  wire hyptsta;
  wire hyptstb;

`ifdef HYP_IOMTC
  localparam IBWB = IBW/9;
  localparam OBWB = OBW/9;
  wire [2*(OBW-OBWB)-1:0] fobus,fmtbus;
  wire [2*OBWB-1:0] fopps,fmtpps;

  assign oena = oenaa & oenab;
  fifoNxM #(IBW-IBWB,2*(OBW-OBWB),0,3,`BPAR,`BPAR) fifo (iclk,frst, irdy,iena,idat[IBW-IBWB-1:0],  oclk,frst, ordy,oena,fobus);
  fifoNxM #(IBWB,2*OBWB,0,3,`BPAR,`BPAR) fifop (iclk,frst, ,iena,idat[IBW-1:IBW-IBWB],  oclk,frst, ,oena,fopps);

  assign fmtpps = (ifmt)? {fopps[7:6],fopps[3:2],fopps[5:4],fopps[1:0]} :
                          {fopps[7],fopps[5],fopps[3],fopps[1],fopps[6],fopps[4],fopps[2],fopps[0]};

  assign fmtbus = (ifmt)? {fobus[63:48],fobus[31:16],fobus[47:32],fobus[15:0]} :
                          {fobus[63:56],fobus[47:40],fobus[31:24],fobus[15:8],fobus[55:48],fobus[39:32],fobus[23:16],fobus[7:0]};

  hypoutq #(OBW,32,4,`HYP_TC) hpoa (oclk,H,odata, ordy,oenaa,8'h00,{fmtpps[OBWB-1:0],fmtbus[OBW-OBWB-1:0]}, 4'b0,hyptsta);
  hypoutq #(OBW,32,4,`HYP_TC) hpob (oclk,H,odatb, ordy,oenab,8'h00,{fmtpps[2*OBWB-1:OBWB],fmtbus[2*(OBW-OBWB)-1:OBW-OBWB]}, 4'b0,hyptstb);

`else
  wire [2*OBW-1:0] fobus,fmtbus;

  assign oena = oenaa & oenab;
  fifoNxM #(IBW,2*OBW,0,3,NFLGS,MFLGS) fifo (iclk,frst, irdy,iena,idat,  oclk,frst, ordy,oena,fobus);
  
  assign fmtbus = (ifmt)? {fobus[63:48],fobus[31:16],fobus[47:32],fobus[15:0]} :
                          {fobus[63:56],fobus[47:40],fobus[31:24],fobus[15:8],fobus[55:48],fobus[39:32],fobus[23:16],fobus[7:0]};

  hypoutq #(OBW,32,4) hpoa (oclk,H,odata, ordy,oenaa,8'h00,fmtbus[OBW-1:0], 4'b0,hyptsta);
  hypoutq #(OBW,32,4) hpob (oclk,H,odatb, ordy,oenab,8'h00,fmtbus[2*OBW-1:OBW], 4'b0,hyptstb);

`endif
 
endmodule

module hypoutiomfmt4 (iclk,iena,idat, good,oclk,odatb,odata,odatc,odatd);
  parameter IBW=32;
  parameter OBW=16;
  parameter NFLGS=0;
  parameter MFLGS=0;
  parameter TEST=0;

  input iclk,oclk;
  input iena,good;
  input [IBW-1:0] idat;
  output [31:0] odata;
  output [31:0] odatb;
  output [31:0] odatc;
  output [31:0] odatd;

  wire irdy,ordy,oena,oenaa,oenab,oenac,oenad;
  wire frst = !good;
  wire L=0, H=1;

  wire hyptsta;
  wire hyptstb;
  wire hyptstc;
  wire hyptstd;

`ifdef HYP_IOMTC
  localparam IBWB = IBW/9;
  localparam OBWB = OBW/9;
  wire [4*(OBW-OBWB)-1:0] fobus;
  wire [4*OBWB-1:0] fopps;

  assign oena = oenaa & oenab & oenac & oenad;
  fifoNxM #(IBW-IBWB,4*(OBW-OBWB),0,3,`BPAR,`BPAR) fifo (iclk,frst, irdy,iena,idat[IBW-IBWB-1:0],  oclk,frst, ordy,oena,fobus);
  fifoNxM #(IBWB,4*OBWB,0,3,`BPAR,`BPAR) fifop (iclk,frst, ,iena,idat[IBW-1:IBW-IBWB],  oclk,frst, ,oena,fopps);

  hypoutq #(OBW,32,4,`HYP_TC) hpoa (oclk,H,odata, ordy,oenaa,8'h00,{fopps[OBWB-1:0],fobus[OBW-OBWB-1:0]}, 4'b0,hyptsta);
  hypoutq #(OBW,32,4,`HYP_TC) hpob (oclk,H,odatb, ordy,oenab,8'h00,{fopps[2*OBWB-1:OBWB],fobus[2*(OBW-OBWB)-1:OBW-OBWB]}, 4'b0,hyptstb);
  hypoutq #(OBW,32,4,`HYP_TC) hpoc (oclk,H,odatc, ordy,oenac,8'h00,{fopps[3*OBWB-1:2*OBWB],fobus[3*(OBW-OBWB)-1:2*(OBW-OBWB)]}, 4'b0,hyptstc);
  hypoutq #(OBW,32,4,`HYP_TC) hpod (oclk,H,odatd, ordy,oenad,8'h00,{fopps[4*OBWB-1:3*OBWB],fobus[4*(OBW-OBWB)-1:3*(OBW-OBWB)]}, 4'b0,hyptstd);

`else
  wire [4*OBW-1:0] fobus;

  assign oena = oenaa & oenab & oenac & oenad;
  fifoNxM #(IBW,4*OBW,0,3,NFLGS,MFLGS) fifo (iclk,frst, irdy,iena,idat,  oclk,frst, ordy,oena,fobus);
  
  hypoutq #(OBW,32,4) hpoa (oclk,H,odata, ordy,oenaa,8'h00,fobus[OBW-1:0], 4'b0,hyptsta);
  hypoutq #(OBW,32,4) hpob (oclk,H,odatb, ordy,oenab,8'h00,fobus[2*OBW-1:OBW], 4'b0,hyptstb);
  hypoutq #(OBW,32,4) hpoc (oclk,H,odatc, ordy,oenac,8'h00,fobus[3*OBW-1:2*OBW], 4'b0,hyptstc);
  hypoutq #(OBW,32,4) hpod (oclk,H,odatd, ordy,oenad,8'h00,fobus[4*OBW-1:3*OBW], 4'b0,hyptstd);

`endif
 
endmodule

