/*
   Non-Midas routine for USB arduino to ICE interface functions

   Author: Jeff Schoen

*/

#define DEVN "ICEUIC,DEVNO=%d,%s,IOM=NONE,NOLOG,MUXCLK=N,"
#define DEVM "ICEPIC,DEVNO=%d,IOM%d=GPS,%s,IOM=NONE,NOLOG,MUXCLK=N,"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <math.h>
#include <termios.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <errno.h>

#include "icedefs.h"
#include "icelib.h"

#define FLAGS 0
#define DEBUG 0
#define STDOUT stdout
#define STDERR stderr
#define USELOCK 0

int_4 ipmice_lock (int_4 id, int_4 mode);

int getstrw (char *str, int len, int fno, double wait) {
  int i,stat;
  for (i=0; i<len && wait>0; ) {
    stat = read(fno,str+i,1);
    if (stat<=0) { usleep(10000); wait-=.010; continue; }
    i += stat;
    if (str[i-1]=='\n') { str[i]==0; return i; }
  }
  return -1;
}

int main (int argc, char *argv[]) {
  PICSTRUCT ps,*p=&ps;
  char c,devname[200],func[60],fname[40],flagstr[80],FLAGstr[80],str[160],*s,buf[512],sdev[8];
  int_4 flags=FLAGS;
  int_4 Mcbreak=0;
  int_4 status=0,io=0,blk=512,port=3,loop=0;
  int_4 test,idev,jdev,mrev,xlink=0,i,j,k;
  double fdev=0.0;
  FILE *fd=NULL,*fn=NULL;
  int_4 fno;

  strcpy(func,"status");
  strcpy(flagstr,"none");
  strcpy(FLAGstr,"NONE");

  xlink = access("/dev/icetty-mb",F_OK);
  if (argc > 1) strcpy(func,argv[1]);

  if (strcmp(func,"help")==0) {	
    printf(" IPMICE - Integrated Platform Management for devICEs.\n");
    printf("            \n");
    printf(" Syntax: ipmice <func> <devno> <file>\n");
    printf("            \n");
    printf("  <func>  = status - report power, temperature, fan status and uptime\n");
    printf("            power=on|off|cycle|softcycle - turn CPU && FPGA power On|Off\n");
    printf("            checkpwr - report power supply status\n");
    printf("            drive=on|off|rst - set FPGA to NVME drive enable\n");
    printf("            fan=speed - set fan speed 0-100 \n");
    printf("            checkfan - report fan wall status\n");
    printf("            pciena=T|F - enable PCIe host interface\n");
    printf("            oc2ena=N|L|U|B - enable occulink OC2 ports N=none,L=lower,U=upper,B=both\n");
    printf("            pex=n - PCI Express switch config on|off|oc2N|oc2U|oc2L|oc2B where On=oc2N and Off=oc2B\n");
    printf("            pmx=n - Processor Module Switch config N=None 1=PIC1, 2=PIC2 B=Both\n");
    printf("            extjtag=T|F - enable external JTAG cable interface\n");
    printf("            jmod=n - select which module to address with jtag commands 0=XBar 3=GPS\n");
    printf("            uloada - upload arduino code from local file\n");
    printf("            uloadv - upload virtex code from local file\n");
    printf("            uloadf - upload flash code from local file\n");
    printf("            uuid - report FPGA ID on this JTAG port \n");
    printf("            rev - report hardware and firmware revision\n");
    printf("         IOC Only:\n");
    printf("            power=on|off|onD|onP - turn power on Drives or PCIe only\n");
    printf("         GPS Only:\n");
    printf("            uloadg - upload and setup GPS code on IOMx site. (ipmice uloadg - GPS=IOM3) with GPSOPTS=(APWR)\n");
    printf("            gstatus - report current GPS status on IOMx site\n");
    printf("         Xcvr Only:\n");
    printf("            freq=rf - sets freq of all tuners in MHz\n");
    printf("            freqN=rf - sets freq of tuner N in MHz (N=1|2) \n");
    printf("            attn=atn - sets attenuation of all tuners in dB\n");
    printf("            attnN=atn - sets attenuation of tuner N in dB (N=1|2) \n");
    printf("            ref=0|1|2 - sets the receiver external reference port\n");
    printf("            pwr=0|1 - turn tuner power off|on\n");
    printf("         uPAC Only:\n");
    printf("            reset - reset the FPGA\n");
    printf("            sniff - sniff the FPGA PCIe registers\n");
    printf("            halt  - halt the FPGA\n");
    printf("            dir    - list files on uPAC SD card\n");
    printf("            put=fn - upload a local <file> to SD card file <fn>\n");
    printf("            get=fn - download to a local <file> the SD card file <fn>\n");
    printf("            load=fn - load the SD card file <fn> onto the FPGA\n");
    printf("            getregs - list the USB1/2/3 and DP crossbar registers on uPAC\n");
    printf("            setreg:a=d - sets a crossbar register at address <a>to <d>\n");
    printf("            getreg:a - gets a crossbar register at address <a>\n");
    printf("            \n");
    printf("  <devno> = 0|1|2[.#] - zero based index of each device in system (.#=chan optional)\n");
    printf("            alias[.#] - assumes a link /dev/icetty-<alias> points to the device (mb,ioc,ar,at,ax,...)\n");
    printf("            -[.#]     - assumes default device\n");
    printf("            \n");
    printf("  <file>  = local filename or flags string\n");
    printf("            loop - to run status or dir continually\n");
    exit(0);
  }

  if (argc > 2) strcpy(sdev,argv[2]); else strcpy(sdev,"-");
  if (strcmp(sdev,"-")==0) strcpy(sdev,(xlink==0)?"mb":"0");

  s = strstr(sdev,".");
  if (isdigit(sdev[0])) fdev = atof(sdev);
  else if (s!=NULL) { fdev = atof(s); s[0]=0; } 

  if (argc > 3) for (i=0,c=1; c!=0; i++) { c = argv[3][i]; flagstr[i]=c; FLAGstr[i]=toupper(c); } 

  s=strstr(FLAGstr,"LOOP="); if (s!=NULL) loop = atoi(s+5);

  idev=(int_4)fdev; jdev=(int_4)((fdev-idev)*10.1);
  if (isdigit(sdev[0])) sprintf(fname,"/dev/ttyACM%d",idev);
  else { sprintf(fname,"/dev/icetty-%s",sdev); i=readlink(fname,devname,40);
    if (i<=0) { printf("Could not resolve alias=%s\n",fname); goto BAIL; }
    idev = devname[i-1]-'0';
  }
  if (idev<0||idev>4||jdev<0||jdev>4) { fprintf(STDERR,"Illegal devno=%s Allowed=[0-4].[1-4]\n",argv[2]); goto BAIL; }

  if (strncmp(func,"put=",4)==0) {	/* prep for ftp put */
    fn = fopen(flagstr,"rb");
    if (fn==NULL) { fprintf (STDERR,"Error opening file=%s - put aborted\n",flagstr); goto DONE; }
    io = 1;
  }
  if (strncmp(func,"get=",4)==0) {	/* prep for ftp get */
    fn = fopen(flagstr,"wb");
    if (fn==NULL) { fprintf (STDERR,"Error opening file=%s - get aborted\n",flagstr); goto DONE; }
    io = -1;
  }

  LOOP:
  if (ipmice_lock(idev,1)<0) { printf("Could not aquire lock for Arduino id=%d\n",idev); goto BAIL; }

  if (strstr("reset|sniff|test|halt|tap",func)!=NULL) {	/* special exception for k8 reset,sniff,test */
    sprintf(devname,DEVN,idev,flagstr);
    if (pic_open (p,devname,&Mcbreak,flags)<0) goto PICBAIL;
    if (strcmp(func,"reset")==0) status = pic_reset (p,0);
    if (strcmp(func,"sniff")==0) status = pic_sniff (p,0);
    if (strcmp(func,"test")==0)  status = pic_test (p,0,0);
    if (strcmp(func,"halt")==0)  status = pic_reset (p,FLG_DISABLE);
    if (strcmp(func,"tap")==0)  status = pic_test_upac (p,0);
    pic_close(p);
  } 
  else if (strcmp(func,"uloada")==0) {	/* special exception for arduino load */
    s = getenv("USER"); if (s==NULL || strcmp(s,"root")!=0)  { printf("Arduino reload function must be run as root\n"); goto BAIL; }
    if (flagstr==NULL || strstr(flagstr,".bin")==NULL) { printf("Arduino executable %s should end in .bin\n",flagstr); goto BAIL; }
    fd = fopen(flagstr,"r"); if (fd==NULL) { printf("Unable to open %s for Arduino load\n",flagstr); goto BAIL; }
    static struct termios old, new;
    int tty_flag = TIOCM_DTR;
    int tty_fd = open(fname,O_RDWR|O_NOCTTY);
    if (tty_fd<=0) { printf("Unable to open port=%s for Arduino load\n",fname); goto BAIL; }
    printf("Reboot Arduino port=%s file=%s\n",fname,flagstr);
    speed_t speed=1200;
    tcgetattr(tty_fd, &old);
    new = old; 
    cfmakeraw (&new);
    cfsetspeed (&new, speed);
    tcsetattr(tty_fd, TCSANOW, &new);
    ioctl(tty_fd,TIOCMBIS,&tty_flag);
    ioctl(tty_fd,TIOCMBIC,&tty_flag);
    close(tty_fd);
    printf("Waiting 5 sec for /dev/ttyACM to rediscover ..\n");
    usleep(5000000);
    sprintf(str,"/usr/bin/bossac -i -d --port=%s -U true -i -e -w -v %s -R",fname,flagstr);
    printf("Exec: %s\n",str);
    system(str);
    printf("Done: %s\n",str);
  }
  else if (strcmp(func,"uuid")==0) {	/* special exception for virtex uuid */
    sprintf(devname,DEVN,idev,strcat(flagstr,",NOSIG"));
    if (pic_open (p,devname,&Mcbreak,flags)<0) goto PICBAIL;
    i = pic_getkeyl (p,0,KEY_IDCODE);
    j = pic_getkeyl (p,0,KEY_UIDCODE);
    printf ("Module ID=%08x UID=%08x\n",i,j);
    pic_close(p);
  }
  else if (strcmp(func,"uloadv")==0) {	/* special exception for virtex load */
    sprintf(devname,DEVN,idev,",NOSIG,VERBOSE");
    if (pic_open (p,devname,&Mcbreak,flags)<0) goto PICBAIL;
    status = pic_loadfile (p,flagstr,FLG_VIRTEX);
    pic_close(p);
  } 
  else if (strcmp(func,"uloadf")==0) {	/* special exception for flash load */
    sprintf(devname,DEVN,idev,",NOSIG,VERBOSE");
    if (pic_open (p,devname,&Mcbreak,flags)<0) goto PICBAIL;
    status = pic_loadfile (p,flagstr,FLG_FLASH);
    pic_close(p);
  } 
  else if (strcmp(func,"uloadg")==0 || strcmp(func,"gstatus")==0 
        || strcmp(func,"ustatg")==0 || strcmp(func,"udumpg")==0) {	/* gps functions */
    s = strstr(FLAGstr,"GPS=IOM");
    if (s!=NULL) port = s[7]-'0';
    strcpy(str,flagstr); strcat(str,",GPSOPTS=(APWR)");
    if (port==3) { port=9; strcat(str,",NOLOCK,NOSIG=0x83,IOM3=GPS"); }
        if (port==9) sprintf(devname,DEVN,idev,str);
    else if (port<3) sprintf(devname,DEVM,0,port,str);
    else {  port-=3; sprintf(devname,DEVM,1,port,str); }
    if (pic_open (p,devname,&Mcbreak,flags)<0) goto PICBAIL;
    if (strcmp(func,"uloadg")==0) {
      mrev=pic_getGPSrev(p,port);
      if (port==9) p->mrev[0]=mrev; else p->mrev[port-1]=mrev;
      if (port==9) status = pic_loadfile (p,(mrev==2)?"icegpsx_d":"icegps_d",FLG_VIRTEX|port);
      pic_setup_gps(p,port);
    } else {
      pic_dump_gps(p,port);
    }
    pic_close(p);
  } 
  else {				/* all other are normal text commands parsed by the Arduino code */
    if (strncmp(func,"freqN=",6)==0) func[4]='0'+jdev;
    if (strncmp(func,"attnN=",6)==0) func[4]='0'+jdev;
    if (fn!=NULL) fno = open(fname,O_RDWR);	/* use blocking IO for get and set file functions */
    else fno = open(fname,O_RDWR|O_NONBLOCK);	/* else non-blocking for normal Arduino failsafe reads and writes */
    if (fno<0) { fprintf(STDERR,"Err opening dev=%s for func=%s. Try ipmice help.\n",fname,func); goto BAIL; }
    if ( (i=read(fno,str,160)) > 0) fprintf(STDERR,"Purging %d bytes of old data\n",i);
    write (fno, func, strlen(func)+1);
    if (getstrw(str,32,fno,1.0)<=0) { fprintf(STDOUT,"Error getting command status from cmd: %s\n",func); status=-1; goto DONE; }
    if (strstr(str,"CMD=")==NULL) { fprintf(STDOUT,"Error getting command status=%s from cmd: %s\n",str,func); status=-1; goto DONE; }
    if (io<0) write (fno, (char*)(&blk), 4); 
    if (io!=0) fprintf(STDOUT,"Start %s ",func);
    if (io>0) for (i=0;;i++) {			/* file input */
      status = fread (buf, 1, 1024, fn);
      write (fno, (char*)(&status), 4);
      if (status<=0) break;
      write (fno, buf, status);
      if (i%10==0) fprintf(STDOUT,".");
    }
    if (io<0) for (i=0;;i++) {			/* file output */
      while (read (fno, (char*)(&status), 4)!=4);
      if (status<=0) break;
      for (j=0; j<status; j+=k) k=read (fno, buf+j, status-j);
      status = fwrite (buf, 1, status, fn);
      if (i%10==0) fprintf(STDOUT,".");
    }
    OUTPUT:
    for (;;) {					/* final text output */
      if (getstrw(str,128,fno,2.0)==0) continue;
      if (strstr(str,"endCMD")!=NULL) break;
      if ((s=strstr(str,"&\n"))!=NULL) { s[0]='.'; s[1]=0; }
      fprintf(STDOUT,"%s",str); fflush(STDOUT);
    }
    if (io!=0) fprintf(STDOUT," done\n");
  }
  if (--loop>0) {
    if (fno>0) close(fno);
    usleep(1000000); 
    goto LOOP;
  }

  DONE:
  if (fno>0) close(fno);
  if (fn!=NULL) fclose(fn);
  if (ipmice_lock(idev,-1)<0) printf("Could not release lock for Arduino id=%d\n",idev);
  BAIL:
  /* if (fn==NULL) printf("Try running: ipmice help\n",idev); */
  exit( (status<0)? EXIT_FAILURE:EXIT_SUCCESS);
  PICBAIL:
  fprintf (STDERR,"Error opening USB PIC device=%s - aborted\n",devname); 

  exit(EXIT_FAILURE);
}

#if !USELOCK
int_4 ipmice_lock (int_4 id, int_4 mode) { return 0; }

#elif _LINX
#include <sys/sem.h>
#include <sys/syscall.h>
#define MAXARDS 6
#ifndef SEMVMX
#define SEMVMX 32767
#endif

union semun {
  int val;                    /* value for SETVAL */
  struct semid_ds *buf;       /* buffer for IPC_STAT, IPC_SET */
  unsigned short int *array;  /* array for GETALL, SETALL */
  struct seminfo *__buf;      /* buffer for IPC_INFO */
};

int_4 ipmice_semid = -1;


int_4 ipmice_lock (int_4 id, int_4 mode)
{
  key_t semkey;                   /* Key used to create semaphore set */
  int   lockid;
  int   status=0;
  struct sembuf op[2];
  union semun semctl_arg;               /* For semctl functions */

  lockid = id;

  if (ipmice_semid<0) {
    semkey = 0x1CE1;
    if ((ipmice_semid = semget(semkey, MAXARDS, 0666 | IPC_CREAT)) < 0) perror("semget - IPC_CREAT");
  }

  if (mode>=0) {
    semctl_arg.val = 0;
    status = semctl(ipmice_semid, lockid, GETVAL, semctl_arg);
    if (status<0) { perror("semctl - IPC_GETVAL"); return -1; }
    else if (status>0) return 1;
    else return 0;
  }

  if (mode>0) {
    op[0].sem_num = op[1].sem_num = lockid;
    op[0].sem_op=0; op[1].sem_op  = SEMVMX;
    op[0].sem_flg = op[1].sem_flg = 0;
    while (semop(ipmice_semid, &op[0], 2) < 0) {
      if (errno == EINTR);  /* semop int by a signal (probably SIGCHLD) */
      else if (errno == EAGAIN) udelay(10000); /* try later */
      else {
        perror("semop - WRLCK failure"); return -1; }     /* shouldn't get here */ 
    }
  } 
  
  if (mode<0) {
    op[0].sem_num = lockid;
    op[0].sem_op  = -1*SEMVMX;
    op[0].sem_flg = 0;
    while (semop(ipmice_semid, &op[0], 1) < 0) {
      if (errno == EINTR);    /* semop int by a signal (probably SIGCHLD) */
      else if (errno == EINVAL) {  /* Lock already taken from us */
        printf("Lock cleared prior to exclusive lock release attempt");
        break;  /* out of while */
      } 
      else { perror("semop - WRUNLCK"); return -1; }
    }
  }

  if (mode==-9) {
    semctl_arg.val = 0;
    if ( semctl(ipmice_semid, lockid, SETVAL, semctl_arg) < 0) {
      perror("semctl - IPC_SETVAL");
    }
  }
  return (0);
}

#else
int_4 ipmice_lock (int_4 id, int_4 mode) { return 0; }
#endif
