// Build me!
//  gcc -L/opt/icexxx/core -l:libAll\$ICE_lin64.so -lpthread -ldl icefsTiming_kernelRamBuffer.c -o icefsTiming_kernelRamBuffer
//
// Run me!
//  ICEROOT=/opt/icexxx LD_LIBRARY_PATH=/usr/local/lib:/opt/icexxx/core ./icefsTiming_kernelRamBuffer

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include <time.h>

#include "icedefs.h"
#include "icelib.h"

//#undef USE_ICE_KERNEL_RAM_BUFFER
#define USE_ICE_KERNEL_RAM_BUFFER

#undef RESTART_FILE_ON_EACH_READ
//#define RESTART_FILE_ON_EACH_READ

#define MAXFILES 8
#define READ_FILE_X "/mnt/icefs/testfile%d.tmp"

#define ICE_PAGE_SIZE_IN_BYTES 4096 // 4KB
#define ICE_PAGE_ADDRESS_MASK  ~0xFFF

PICSTRUCT myPicStruct = {0};
DMAMAP myDmaMap = {0};


uint8_t * iceMalloc(uint32_t sizeInBytes)
{
  int err = 0;

  err = pic_open(&myPicStruct, "ICEPIC,DEVNO=0,IOM=A2Dr20,", NULL, 1);
  if (err < 0) {
    printf("pic_open(...) failed (returned %d)\n", err);
    return 0;
  }

  err = pic_mapmem(&myPicStruct, &myDmaMap, sizeInBytes, 1);
  if (err < 0) {
    printf("pic_mapmem(1) failed (returned %d)\n", err);
    return 0;
  }

  return (uint8_t *)(myDmaMap.vaddr);
}

int iceFree()
{
  int err = 0;

  err = pic_mapmem(&myPicStruct, &myDmaMap, 0, -1);
  if (err < 0) {
    printf("pic_mapmem(-1) failed (returned %d)\n", err);
    return -1;
  }

  err = pic_close(&myPicStruct);
  if (err < 0) {
    printf("pic_close(...) failed (returned %d)\n", err);
    return -1;
  }

  return 0;
}


int main (int argc, char **argv)
{
  ///
  // Initialization
  ///
  int inFiles[MAXFILES];
  char fname[80];
  int n;

  char mode = 'r';
  if (argc>1) mode = argv[1][0];
  int numFiles=1;
  if (argc>2) numFiles = atol(argv[2]);
  int xferCount=40;
  if (argc>3) xferCount = atol(argv[3]);
  int xferSize=4*1024*1024;
  if (argc>4) xferSize = atol(argv[4]);
  if (argc>4 && argv[4][strlen(argv[4])-1]=='K') xferSize *= 1024;
  if (argc>4 && argv[4][strlen(argv[4])-1]=='M') xferSize *= (1024*1024);
  int testCount=1;
  if (argc>5) testCount = atol(argv[5]);

  if (numFiles>MAXFILES) { printf("Limiting files from %d to %d\n",numFiles,MAXFILES); numFiles=MAXFILES; }

  printf("Running mode=%c nf=%d rc=%d rsz=%d tc=%d\n",mode,numFiles,xferCount,xferSize,testCount);

#ifndef USE_ICE_KERNEL_RAM_BUFFER
  char * buffer = (char *)malloc(xferSize + ICE_PAGE_SIZE_IN_BYTES);
#else
  char * buffer = (char *)iceMalloc(xferSize + ICE_PAGE_SIZE_IN_BYTES);
  if (buffer == 0)
  {
    printf("Failed to fetch ice kernel ram buffer :(\n");
    return -1;
  }
#endif

  // Page align our address
  //char * pageAlignedAddress = (char *)((uint64_t)(buffer + 0x1000) & ~0xFFF);
  char * pageAlignedAddress = (char *)((uint64_t)(buffer + ICE_PAGE_SIZE_IN_BYTES) & ICE_PAGE_ADDRESS_MASK);

  double totalTestTime = 0;
  double minXferTime = 1000;
  double maxXferTime = 0;
  uint32_t minXferTimeIteration = 0;
  uint32_t maxXferTimeIteration = 0;

  char * xferAddress = pageAlignedAddress;
  printf("Xfering at address 0x%08x%08x\n", (uint64_t)xferAddress >> 32, (uint64_t)xferAddress & 0x00000000FFFFFFFF);

  int ioflgs =  (mode=='w')? O_RDWR|O_CREAT : O_RDONLY;
  for (n=0; n<numFiles; n++) {
    sprintf(fname,READ_FILE_X,n+1);
    inFiles[n] = open(fname, ioflgs,0666);
    if (inFiles[n]<0) printf("Failed to open file=%s\n",fname);
    else printf("Opened file=%s\n",fname);
  }

  uint32_t testIteration = 0;
  for (testIteration = 0; testIteration < testCount; testIteration++)
  {
    double totalXferTime = 0;
    double totalRateInMb = 0;

    size_t nBytes = xferSize;
    size_t bytes_xfer = 0;

    for (n=0; n<numFiles; n++) {
      lseek(inFiles[n], 512+testIteration*xferCount*xferSize, SEEK_SET);
    }

    uint32_t xferIteration;
    for (xferIteration = 0; xferIteration < xferCount; xferIteration++)
    {
      // Start timestamp
      struct timespec startTime = {0};
      clock_gettime(CLOCK_REALTIME, &startTime);

      // xfer
      bytes_xfer = 0;

      if (mode=='w') {
        for (n=0; n<numFiles; n++) bytes_xfer += write(inFiles[n], xferAddress, nBytes);
      } else {
        for (n=0; n<numFiles; n++) bytes_xfer += read(inFiles[n], xferAddress, nBytes);
      }

      // Stop timestamp
      struct timespec stopTime = {0};
      clock_gettime(CLOCK_REALTIME, &stopTime);

      // Prepare xfer timing information
      double xferTime = (stopTime.tv_sec + stopTime.tv_nsec*1e-9) - (startTime.tv_sec + startTime.tv_nsec*1e-9);
      double xferRateInMb = bytes_xfer / xferTime / 1024 / 1024;

      ///
      // Log xfer information
      ///
      printf("  %u Xfer time for %u:%u bytes: %g seconds (%g Mbytes per second)\n",
        xferIteration, nBytes, bytes_xfer, xferTime, xferRateInMb);

      totalXferTime += xferTime;
      totalRateInMb += xferRateInMb;

      if (xferTime < minXferTime) { minXferTime = xferTime; minXferTimeIteration = xferIteration; }
      if (xferTime > maxXferTime) { maxXferTime = xferTime; maxXferTimeIteration = xferIteration; }

    }

    ///
    // Prepare summary information
    ///
    double averageXferRateInMb = totalRateInMb / xferCount;
    double minXferRateInMb = bytes_xfer / maxXferTime / 1024 / 1024;
    double maxXferRateInMb = bytes_xfer / minXferTime / 1024 / 1024;

    ///
    // Log the iteration summary
    ///
    printf("Total Xfer Time: %g seconds (avg=%g  min=%g  max=%g  MBytes per second)\n", totalXferTime, 
		averageXferRateInMb,minXferRateInMb,maxXferRateInMb);
    totalTestTime += totalXferTime;
  }

  for (n=0; n<numFiles; n++) {
    close(inFiles[n]);
  }

  ///
  // Log the test summary information
  printf("Total Test Time: %g seconds\n", totalTestTime);
  printf("minXferTime (iteration %u): %g\n", minXferTimeIteration, minXferTime);
  printf("maxXferTime (iteration %u): %g\n", maxXferTimeIteration, maxXferTime);

  ///
  // Time for cleanup
  ///
#ifndef USE_ICE_KERNEL_RAM_BUFFER
  free(buffer);
#else
  int retVal = iceFree();
  if (retVal < 0) {
    printf("Failed to free ice kernel ram buffer :(\n");
    return -1;
  }
#endif

  return 0;
}
