package nxm.ice.prim;

import nxm.ice.lib.FileIO;
import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.sys.libm.Waveform;
import nxm.sys.libm.Fft;
import nxm.sys.libm.Add;
import nxm.sys.libm.Multiply;
import nxm.dsp.lib.FilterFIR;
import nxm.dsp.prim.firkais;

/**
  Creates signal test waveforms of various shapes.

  @author Jeff Schoen
  @version $Id: icewave.java,v 1.50 2013/05/02 14:11:54 jgs Exp $
*/
public final class icewave extends Primitive {

  private int shape, throttle, cur=0, tcur=0, olut=0, nlut=0, hlut=0, mode,wmode, spb=1;
  private int lrs,lrsx,lrsy, seed=123456,seedx=123456,seedy=654321,lrsp;
  private int  ntap=0,nfft=0,fb=0,nbins=0,nrpt,nper,mper,ndone=0,ovsr=1,itag=0;
  private double size, amp, freq, dfreq, freq1, freq2, chirp, poff, var, sdev, delta, tstart, rate, gain, baud;
  private double phase, phase1, phase2, timer, sweep=0;
  private String form;
  private DataFile ho, hf;
  private float[] tbuf,rtap;
  private byte[] bbuf;
  private boolean wrap,filter,lut,dfs,stats,iqoff,fsk,fram,scrambler;
  private byte type;
  private float midx=1;		// modulation index or CW sweep rate
  private float stag;		// tag the start by scaling first block by stag

  static final byte _JINT = (byte)'J';	// until all are running nxm410+
  static final int LRSPM = 0x2000023;	// LRS Polynomial for Midas sequence

  /** The waveform shapes this primitive can generate {@value} */
  public  static final String shapeList = 
	"Sine,Cosine,Square,Triangle,SawTooth,Pulse,Constant,Zero,White,LRS,Ramp,TRamp,Notch,TwoTone,MixLS,MuxLS,MixNS,NFloor,"+
	"CW,FSK,4FSK,BPSK,QPSK,OQPSK,1P4QPSK,3P4QPSK,8PSK,16PSK,16APSK,32APSK,64APSK,8QAM,16QAM,32QAM,64QAM,128QAM,256QAM,512QAM,1024QAM";
  public static final int 
	SIN=1,COS=2,SQU=3,TRI=4,SAW=5,PUL=6,CON=7,ZER=8,WHI=9,LRS=10,RAMP=11,TRAM=12,NOTCH=13,TTONE=14,MIXLS=15,MUXLS=16,MIXNS=17,NFLR=18,
	CW=19,FSK=20,FSK4=21,BPSK=22,QPSK=23,OQPSK=24,QPSK14=25,QPSK34=26,PSK8=27,PSK16=28,APSK16=29,APSK32=30,APSK64=31,QAM8=32,QAM16=33,QAM32=34,QAM64=35,QAM128=36,QAM256=37,QAM512=38,QAM1024=39;

  /** The throttle mode {@value}. */
  public  static final String throttleList = "RealTime,Full,Block";
  private static final int REALTIME=1,FULL=2,BLOCK=3;

  @Override
  public int open () {

    // Parameters
    shape = MA.getChoice("SHAPE",shapeList,0);
    form  = MA.getFormat("FORM","S#,C#");
    size  = MA.getD("ELEM");
    rate  = MA.getD("RATE");
    gain  = MA.getD("GAIN"); 

    // Set amplitude and other shape modes
    if (shape==QPSK14||shape==QPSK34) shape = PSK8;		// fake 1/4 and 3/4 PSK signals
    if (shape==WHI) gain -= 6;					// prevent WHITENOISE clipping
    if (shape==LRS||shape==NOTCH||shape==MIXNS) gain -= 3;	// prevent LRS clipping
    if (shape==NOTCH||shape==MIXNS) { filter=true; gain -= 1; }	// prevent NOTCH filter clipping
    if (shape>=FSK) { lut=true; gain -= 3; }			// turn on LUT mode
    if (shape>=BPSK) { gain -= 3; }				// prevent clipping
    if (shape==TTONE) { dfs=true; }				// turn on dual frequency mode
    if (shape==OQPSK) { spb=2; iqoff=true; }			// samples per baud for Offset QPSK

    // Pick up Freq and Baud defaults
    freq  = MA.getD("FREQ", lut? 0 : rate/10);
    baud  = MA.getD("BAUD", lut? rate/4 : 0);

    type = (byte)form.charAt(1);
    fsk  = (shape==FSK||shape==FSK4);
    fram = (shape==RAMP && type==LONG);
    amp  = (type==LONG)? 2e9 : (type==INT||type==_JINT)? 30720 : (type==BYTE)? 120 : (type==NIBBLE)? 7 : 1.0;
    amp  = MA.getD("/AMP",amp);
    amp *= Math.pow(2.0,(gain/6));
    ntap = (shape>=QAM512)? 11 : 7;
    ntap = lut? ntap : filter? 512 : 0;

    // Modifiers
    poff  = MA.getD("/PHASE",0.0);
    chirp = MA.getD("/CHIRP",0.0);
    var   = MA.getD("/VARIANCE",Math.pow(amp/LNTEN,2));
    sdev  = MA.getD("/SDEV",Math.sqrt(var));
    dfreq = MA.getD("/DFREQ",(baud>0)?baud:rate/50);
    ntap  = MA.getL("/NTAP",ntap);
    nrpt  = MA.getL("/NRPT",16384);
    lrsp  = MA.getL("/LRSP",0);
    fb    = MA.getL("/FB",0);
    midx  = MA.getF("/MIDX",midx);	// modulation index
    ovsr  = MA.getL("/OVSR",ovsr);	// over sampling mode 
    stag  = MA.getF("/STAG");		// start tag
    itag  = MA.getL("/STAGOFF",0);		// start tag
    delta = 1.0/rate;
    freq1 = freq-dfreq;
    freq2 = freq+dfreq;
    if (lut) freq1 = baud;

    // Open filter file
    hf = MA.getDataFile ("/FIR","1000","SF",0);
    hf.open(hf.OPTIONAL);
    if (hf.isOpen) {
      ntap = (int)hf.getSize();
      Data db = hf.getDataBuffer(ntap,FLOAT);
      hf.read(db);
      hf.close();
      rtap = new float[ntap];
      db.toArray(rtap);
      if (lut) ntap /= nrpt;
    }
    else if (lut) {
      double fw = MA.getD("/FWIDTH",1.0);
      if (fsk) { ntap=256; nrpt=1; filter=true; fw *= (2*baud*midx/rate); if (shape==FSK4) fw*=2; }
      int n = ntap*nrpt;
      rtap = new float[n+1];
      float[] ctap = new float[n*2];
      if (fsk) firkais.generate(1, fw,0.0,fw/20, 80.0, ctap,n,false);
      else firkais.generate(1, fw/nrpt/ovsr,0.0,0.05/nrpt, 80.0, ctap,n,false);
      for (int i=0; i<n; i++) rtap[i]=ctap[i+i]*nrpt; // complex to real and scale
      rtap[n]=rtap[0];
    }
    
    // Open output file
    ho = MA.getDataFile("OUT","1000","SF",0);
    ho.setFormat(form);
    ho.setXDelta(delta);
    ho.setXUnits(Units.TIME);
    ho.setSize(size);
    ho.open(DataFile.OUTPUT|DataFile.NATIVE);
    
    // Setup throttling mode
    throttle = MA.getState("/RT")? REALTIME : FULL;
    throttle = MA.getChoice("/THROTTLE",throttleList,throttle); // Override any /RT input
    if (throttle==REALTIME) ho.setTimeAt(Time.current());

    // handle phase continuous wrap adjustments
    wrap = MA.getState("/WRAP");
    if (wrap) {
      double len = size*delta;
      freq  = Math.round(freq*len)/len;
      freq1 = Math.round(freq1*len)/len;
      freq2 = Math.round(freq2*len)/len;
    }

    // Setup processing
    mode = ho.spa;
    wmode = (shape>=FSK)? 2 : mode;
    xfer = (int)(8192/ho.dbpe);
    xfer = MA.getL("/TL",xfer);
    nbins = xfer;
    if (filter) { nfft=xfer; xfer-=ntap; filterPrep(fsk?rtap:getNotchFilter()); }

    int nbuf = 4*Math.max(xfer*wmode,nbins*2);
    bbuf = (fb>0)? FileIO.allocBuffer(nbuf) : new byte[nbuf];
    if (fb>1) { ho.seek(size); ho.seek(0); }	// allocate ?

    todo(size);

    seed = MA.getL("/SEED",seed);
    Waveform.setSeed(seed);
    lrs = ~seed;
    lrsx = ~seedx;
    lrsy = ~seedy;
    tbuf = new float[xfer*wmode];
    if (lut) preplut(); 

    stats = MA.getState("/STATS");
    timer = Time.current();

    return NORMAL;
  }

  @Override
  public int process () {
    double p,dp, p1=0,dp1=0, p2=0,dp2=0;

    // handle data throttling
    if (throttle>0) {
      if (throttle==REALTIME) {
        if (tstart==0) tstart = Time.current()-ho.seek()*delta;
        if (Time.current() < tstart+ho.seek()*delta) return NOOP;
      }
      else tstart=0;
      if (throttle==BLOCK) return NOOP;
    }

    int ndo = todo();
    if (ndo==0) return FINISH;

    if (fram) {
      int[] lbuf = Native.castL(bbuf);
      cur=fastramp(lbuf,(int)type,ndo,mode,cur); 
      Native.uncastL(lbuf);
      ho.write(bbuf,0,(int)(ndo*ho.dbpe));
      if (fb==2) fb=3;
      return NORMAL;
    }

    if (fb>2) {
      ho.write(bbuf,0,(int)(ndo*ho.dbpe));
      return NORMAL;
    }
    if (fb==2) fb=3;

    p  = phase+poff;		// offset phase
    dp = freq*delta;		// delta phase
    if (mode==1 && wmode==2) dp += 0.25;
    if (poff!=0) p -= Math.floor(p);    // modulo 1.0 if necessary
    if (dp<0 && shape!=SIN && shape!=COS && shape!=CW && !lut) dp = -dp;   // no negative freqs except for SINs
    if (shape==CW) { dp+=sweep*baud*delta; sweep+=ndo*midx*delta; if (sweep>0.5) sweep-=1; }

    if (dfs || lut) {	
      p1 = phase1+poff; dp1 = freq1*delta;
      p2 = phase2+poff; dp2 = freq2*delta;
    }

    float[] fbuf = Native.castF(bbuf);
    if (shape==MUXLS && mode==1) shape = MIXLS;

    switch (shape) {	// branch on wave shape
      case 0:   break;
      case CW:  
      case SIN: Waveform.sincos(fbuf,amp,p,dp,ndo,mode); break;
      case COS: Waveform.sincos(fbuf,amp,p+.25,dp,ndo,mode); break;
      case SQU: Waveform.square(fbuf,amp,p,dp,ndo,mode); break;
      case TRI: Waveform.triangle(fbuf,amp,p,dp,ndo,mode); break;
      case SAW: Waveform.sawtooth(fbuf,amp,p,dp,ndo,mode); break;
      case PUL: Waveform.pulse(fbuf,amp,p,dp,ndo,mode); break;
      case CON: Waveform.constant(fbuf,amp,ndo,mode); break;
      case ZER: Waveform.constant(fbuf,0.0,ndo,mode); break;
      case WHI: Waveform.whitenoise(fbuf,sdev,ndo,mode); break;
      case LRS: lrs=Waveform.lrs(fbuf,amp,ndo,mode,lrs); break;
      case RAMP: cur=Waveform.ramp(fbuf,amp,ndo,mode,cur); break;
      case TRAM: tcur=testramp(fbuf,(int)type,ndo,mode,tcur); break;
      case NFLR:  lrs(fbuf,amp,ndo,mode);
		  rotate(fbuf,ndo,mode); 
		  break;
      case NOTCH: lrs=Waveform.lrs(fbuf,amp,ndo,mode,lrs);
		  rotate(fbuf,ndo,mode);
		  filter(fbuf,ndo,mode); 
		  break;
      case TTONE: Waveform.sincos(fbuf,amp/2,p1,dp1,ndo,mode);
		  Waveform.sincos(tbuf,amp/2,p2,dp2,ndo,mode);
		  Add.SSS(fbuf,tbuf,fbuf,ndo*mode);
		  break;
      case MIXLS: lrs=Waveform.lrs(fbuf,amp/2,ndo,mode,lrs);
		  rotate(fbuf,ndo,mode);
		  Waveform.sincos(tbuf,amp/2,p,dp,ndo,mode);
		  Add.SSS(fbuf,tbuf,fbuf,ndo*mode);
		  break;
      case MUXLS: lrs=Waveform.lrs(fbuf,amp,ndo,1,lrs);
		  Waveform.sincos(tbuf,amp,p,dp,ndo,1);
		  mux(fbuf,tbuf,ndo);
		  rotate(fbuf,ndo,mode);
		  break;
      case MIXNS: lrs=Waveform.lrs(fbuf,amp*15/16,ndo,mode,lrs);
		  rotate(fbuf,ndo,mode);
		  filter(fbuf,ndo,mode);
		  Waveform.sincos(tbuf,amp*1/16,p,dp,ndo,mode);
		  Add.SSS(fbuf,tbuf,fbuf,ndo*mode);
		  break;
      default:    // perform in cx mode only 
		  if (fsk) resfsk (fbuf,p1,dp1,ndo);
		  else     reslut (fbuf,p1,dp1,ndo); 
		  if (fsk) filter(fbuf,ndo,wmode);
		  if (dp!=0) rotlut (fbuf,p,dp,ndo);
		  if (mode==1) cx2r(fbuf,ndo);
    }

    // add startup scaling tag ?
    if (stag >= 0.0) {
      for (int i=itag*mode; i<ndo*mode; i++) fbuf[i] = fbuf[i]*stag;
      stag = -1;
    }

    Native.uncastF(fbuf);
    if (type!=FLOAT) Convert.type( bbuf,0,FLOAT, bbuf,0,type, ndo*mode);
    ho.write(bbuf,0,(int)(ndo*ho.dbpe));

    phase += dp*ndo;		// increment phase
    ndone = (int)phase;		// num cycles complete
    phase -= ndone; 		// modulo 1.0
    if (chirp!=0) freq += chirp*delta*ndo;
    if (dfs || lut) {	
      phase1 += dp1*ndo; ndone = (int)phase1; phase1 -= ndone; if (lut) genlut (ndone,mper);
      phase2 += dp2*ndo; ndone = (int)phase2; phase2 -= ndone;
    }

    return NORMAL;
  }

  @Override
  public int close () {
    if (stats) {
      float elapse = (float)(Time.current()-timer);
      float size = (float)(ho.getDataSize()/(1024*1024));
      float rate = size/(float)Math.max(.001,elapse);
      M.info("Size="+size+" Mby, Time="+elapse+" sec, Rate="+rate+" Mby/s");
    }
    if (fb>0) FileIO.freeBuffer(bbuf);
    ho.close();
    return NORMAL;
  }

  @Override
  public int restart () {
    ho.close();
    ho = MA.getDataFile("OUT",ho,0);
    ho.setFormat (form);
    ho.setSize (todo);
    ho.open();
    tstart = 0;
    return NORMAL;
  }

  @Override
  public int processReady () {
    if (state!=PROCESS) return 0;
    return ho.processReady(xfer);
  }

  // set run-time properties
  public void setShape (String value) { shape = Parser.find(shapeList,value,shape,0,Parser.E_WARNING); }
  public void setGain (double value) { gain = value; }
  public void setFrequency (double value) { freq = value; }
  public void setChirp (double value) { chirp = value; }
  public void setPhase (double value) { phase = value; }
  public void setThrottle (String value) { throttle = Parser.find(throttleList,value,throttle,0,Parser.E_WARNING); }
  public void setFormat (String format) { if (form.equals(format)) return; form = format; setState(RESTART); }

  // get run-time properties
  public String getShape () { return Parser.get(shapeList,shape); }
  public double getGain () { return gain; }
  public double getAmplitude () { return amp; }
  public double getFrequency () { return freq; }
  public double getChirp () { return chirp; }
  public double getPhase () { return phase; }
  public String getThrottle () { return Parser.get(throttleList,throttle); }
  public String getFormat () { return form; }

  // Helper routines
  private Fft fft,ifft;
  private float[] filt,save,loop,work;
  private long iloop;

  private float[] getNotchFilter() {
    double fnyq = rate/2;
    double afreq = Math.max(Math.min(freq,fnyq-dfreq),dfreq);
    FilterFIR f = new FilterFIR(M, FilterFIR.FilterType.StopBand, ntap, afreq-dfreq, afreq+dfreq, fnyq, false, false, "BH92", 1.0);
    return f.makeFilter(ntap);
  }
  private void filterPrep(float[] ftap) {
    int flags = (wmode==2)? Fft.COMPLEX : Fft.REAL|Fft.UNPACKED;
    fft = new Fft(nfft,flags|Fft.FORWARD);
    ifft = new Fft(nfft,flags|Fft.INVERSE);
    nbins = (wmode==2)? nfft : nfft/2+1;
    int nbin2 = nbins*2;
    float scale = (float)(1.0/((wmode==2)?nfft:nfft*4));
    filt = new float[nbin2];
    for (int i=0; i<nbin2; i++) filt[i]=0;
    for (int i=0; i<ntap; i++) filt[(i+xfer)*wmode]=ftap[i]*scale;
    fft.work(filt);
    save = new float[ntap*wmode];
    lrs=Waveform.lrs(save,amp,ntap,wmode,lrs);
    loop = new float[ntap*wmode];
    System.arraycopy(save,0, loop,0, ntap*wmode);
    iloop = -(long)(size-ntap);
    work = new float[nbin2];
  }

  private void filter (float[] buf, int ndo, int mode) {
    System.arraycopy(save,0, work,0, ntap*mode);		// last save to front
    System.arraycopy(buf,0, work,ntap*mode, ndo*mode);		// next data to back
    if (wrap) { 
      iloop += xfer;
      if (iloop>0) System.arraycopy(loop,0, work,(nfft-(int)iloop)*mode, Math.min(ntap,(int)iloop)*mode);
    }
    System.arraycopy(buf,(xfer-ntap)*mode, save,0, ntap*mode);	// back of data to next save
    fft.work(work);
    Multiply.CCC(work,filt,buf,nbins);
    ifft.work(buf);
  }

  private long qloop;
  float[] ilut,qlut,idat,qdat,idat0,qdat0;
  final static int MLUT=1024; // max LUT depth
  int[] scram,lrsbuf;

  private void preplut() {
    ilut = new float[MLUT];
    qlut = new float[MLUT];
    float scl = (float)amp;
    int i,j,k=0,is=1,js=1;
    boolean map = MA.getState("/MAP");

    if (shape==BPSK) { olut=1; 
      float r1 = scl*1.414F/2;
      for (i=0;i<2;i++,is=-is) { ilut[k]=is*r1; qlut[k]=0; k++; }
    }
    else if (shape==FSK) { olut=1; 
      for (i=0;i<2;i++,is=-is) { ilut[k]=is*0.5F*midx; qlut[k]=0; k++; }
    }
    else if (shape==FSK4) { olut=2; 
      for (i=0;i<2;i++,is=-is) { ilut[k]=is*0.5F*midx; qlut[k]=0; k++; }
      for (i=2;i<4;i++,is=-is) { ilut[k]=is*1.5F*midx; qlut[k]=0; k++; }
    }
    else if (shape==PSK8) { olut=3; 
      float r1 = scl*1.414F/2;
      for (i=0; i<8; i++) { double p=TWOPI*(i+.5)/8; ilut[i]=r1*(float)Math.sin(p); qlut[i]=r1*(float)Math.cos(p); }
    }
    else if (shape==PSK16) { olut=4; 
      float r1 = scl*1.414F/2;
      for (i=0; i<16; i++) { double p=TWOPI*(i+.5)/16; ilut[i]=r1*(float)Math.sin(p); qlut[i]=r1*(float)Math.cos(p); }
    }
    else if (shape==APSK16) { olut=4; 
      float r1=scl*(1.0F/2.57F), r2=scl;	// 9/10 DVB-S2
      for (i=0; i< 4; i++) { double p=TWOPI*(i+0.5)/4;  ilut[i+0]=r1*(float)Math.sin(p); qlut[i+0]=r1*(float)Math.cos(p); }
      for (i=0; i<12; i++) { double p=TWOPI*(i+0.5)/12; ilut[i+4]=r2*(float)Math.sin(p); qlut[i+4]=r2*(float)Math.cos(p); }
    }
    else if (shape==APSK32) { olut=5; 
      float r1=scl*(1.0F/4.30F), r2=scl*(2.53F/4.30F), r3=scl;	// 9/10 DVB-S2
      for (i=0; i< 4; i++) { double p=TWOPI*(i+0.5)/4;  ilut[i+0]=r1*(float)Math.sin(p);  qlut[i+0]=r1*(float)Math.cos(p); }
      for (i=0; i<12; i++) { double p=TWOPI*(i+0.5)/12; ilut[i+4]=r2*(float)Math.sin(p);  qlut[i+4]=r2*(float)Math.cos(p); }
      for (i=0; i<16; i++) { double p=TWOPI*(i+0.0)/16; ilut[i+16]=r3*(float)Math.sin(p); qlut[i+16]=r3*(float)Math.cos(p); }
    }
    else if (shape==APSK64) { olut=6; 
      float r1=scl*(1.0F/5.30F), r2=scl*(2.53F/5.30F), r3=scl*(4.0F/5.30F), r4=scl;	// 9/10 DVB-S2
      for (i=0; i< 4; i++) { double p=TWOPI*(i+0.5)/4;  ilut[i+0]=r1*(float)Math.sin(p);  qlut[i+0]=r1*(float)Math.cos(p); }
      for (i=0; i<12; i++) { double p=TWOPI*(i+0.5)/12; ilut[i+4]=r2*(float)Math.sin(p);  qlut[i+4]=r2*(float)Math.cos(p); }
      for (i=0; i<20; i++) { double p=TWOPI*(i+0.5)/20; ilut[i+16]=r3*(float)Math.sin(p); qlut[i+16]=r3*(float)Math.cos(p); }
      for (i=0; i<28; i++) { double p=TWOPI*(i+0.5)/28; ilut[i+36]=r4*(float)Math.sin(p); qlut[i+36]=r4*(float)Math.cos(p); }
    }
    else if (shape==QAM8) { olut=3; 
      float r1=scl*(0.768F/1.848F), r2=scl;
      for (i=0; i<4; i++) { double p=TWOPI*(i+0.5)/4; ilut[i+0]=r1*(float)Math.sin(p); qlut[i+0]=r1*(float)Math.cos(p); }
      for (i=0; i<4; i++) { double p=TWOPI*(i+0.5)/4; ilut[i+4]=r2*(float)Math.sin(p); qlut[i+4]=r2*(float)Math.cos(p); }
    }
    else {
      switch (shape) {
        case QPSK    : olut=2; hlut=2; break;
        case OQPSK   : olut=2; hlut=2; break;
        case QAM16   : olut=4; hlut=4; break;
        case QAM32   : olut=5; hlut=6; break;
        case QAM64   : olut=6; hlut=8; break;
        case QAM128  : olut=7; hlut=12; break;
        case QAM256  : olut=8; hlut=16; break;
        case QAM512  : olut=9; hlut=24; break;
        case QAM1024 : olut=10; hlut=32; break;
      }
      if (map) scl=(float)hlut/(hlut-1);
      float ampd = 2*scl/hlut, amp0 = -scl + ampd/2;
      for (i=0;i<hlut;i++) {
	for (j=0;j<hlut;j++) { 
          // cut outs for the corners on non-pow4 grids
	  if (shape==QAM32 && (i<1||i>4) &&  (j<1||j>4)) continue; 
	  if (shape==QAM128 && (i<2||i>9) &&  (j<2||j>9)) continue; 
          if (shape==QAM512 && (i<4||i>19) &&  (j<4||j>19)) continue;
	  ilut[k]=amp0+i*ampd;
	  qlut[k]=amp0+j*ampd;
	  if (map) addRing(ilut[k],qlut[k]);
	  k++;
	}
      }
    }
    nlut = 1<<olut;
    // prefill resampler tap buffer
    nper = (int) (xfer*freq1*delta + 1);
    mper = ntap + nper;
    idat = new float[mper+1]; idat0 = new float[mper]; 
    qdat = new float[mper+1]; qdat0 = new float[mper]; 
    lrsbuf = new int[mper];
    qloop = mper;
    genscram();
    genlut(mper,mper);
    System.arraycopy(idat,0, idat0,0, mper);
    System.arraycopy(qdat,0, qdat0,0, mper);
    qloop = -Math.round(size*delta*freq1);
    if (map) printRings();
  }

  int nr=0,n;
  float sar=0;
  float[] rings;
  private void addRing (float x, float y) {
    float r = (float)Math.sqrt(x*x+y*y);
    sar += r;
    if (rings==null) rings = new float[MLUT];
    int i=0;
    for (; i<nr; i++) {
      if (Math.abs(r-rings[i])<.0001) return;
      if (r<rings[i]) break;
    }
    for (int j=nr; j>i; j--) rings[j]=rings[j-1];
    rings[i]=r;
    nr++;
  }
  private void printRings() {
    float scl=(float)(hlut-1)/hlut;
    for (int i=0; i<nr; i++) {
      System.out.println("Ring i="+i+" r="+rings[i]+" rs="+(scl*rings[i]));
    }
    float ar = scl*sar/nlut;
    System.out.println("Average = "+ar);
  }

  int iscram;
  final static int NSCRAM=1024,MSCRAM=NSCRAM-1; 
  private void genscram () {
    scram = new int[NSCRAM];
    int lrs = ~seed;
    for (int i=0; i<NSCRAM*8; i++) {
      int bit0 = (~(lrs^(lrs>>1)^(lrs>>5)^(lrs>>25))) & 0x1;
      lrs = (lrs<<1) | bit0;
      scram[i>>3] = lrs&0x3FF;	// every 8th lrs to be unique
    }
    iscram=0;
  }

  private int onesN (int mask) {
    int ones=1;
    for (int i=0; i<32; i++) { ones += (mask&1); mask>>=1; }
    return ones;
  }

  private int genlrs (int[] buf, int n, int lrs) {
    float factor = (float)(120.0/2/Constants.B1G);
    if (lrsp!=0) for (int i=0; i<n; i++) {
      lrs = (lrs<<1) | (onesN(lrs&lrsp)&0x1);
      buf[i] = lrs*0x3FF;
    }
    else for (int i=0; i<n; i++) {
      int bit0 = (~(lrs^(lrs>>1)^(lrs>>5)^(lrs>>25))) & 0x1;
      lrs = (lrs<<1) | bit0;
      int lrsb = (int)(factor*lrs);
      buf[i] = lrsb^scram[iscram++]; iscram &= MSCRAM;
    }
    return lrs;
  }

  float fskp=0;
  private void genlut (int ndone, int ndo) {
    int mlut = nlut-1;
    int ndif = ndo-ndone;
    for (int i=0; i<ndif; i++) {
      idat[i] = idat[i+ndone];
      qdat[i] = qdat[i+ndone];
    }
    lrs=genlrs(lrsbuf,ndone,lrs);
    if (shape==FSK||shape==FSK4) 
    for (int i=ndif,j=0; i<ndo; i++,j++) {
      int k = lrsbuf[j]&mlut;
      float fskdp = ilut[k];
      idat[i] = fskdp;
      qdat[i] = fskp;
      fskp += fskdp;
      fskp = fskp - (int)fskp;
    }
    else for (int i=ndif,j=0; i<ndo; i++,j++) {
      int k = lrsbuf[j]&mlut;
      idat[i] = ilut[k];
      qdat[i] = qlut[k];
    }
    if (wrap && qloop!=ndo) { 
      qloop += ndone;
      if (qloop+ndo>=0) {
	int ioff = -(int)qloop, i0off=0; 
	if (ioff<0) { i0off=-ioff; ioff=0; }
	int icpy = ndo-ioff;
	System.arraycopy(idat0,i0off, idat,ioff, icpy);
	System.arraycopy(qdat0,i0off, qdat,ioff, icpy);
      }
    }
  }

  private void resfsk (float[] buf, double p, double dp, int ndo) {
    int i,j,k;
    float scl = (float)amp;
    for (i=0,j=0; i<ndo; i++) {
      float di=0,dq=0;
      k = (int)p;
      double r = ((p-k)*idat[k]+qdat[k])*TWOPI;
      di = scl*(float)Math.sin(r); 
      dq = scl*(float)Math.cos(r); 
      buf[j++] = di;
      buf[j++] = dq;
      p += dp;
    }
  }
 
  private void reslut (float[] buf, double p, double dp, int ndo) {
    int i,j,k,k1,k2,k1q,k2q;
    for (i=0,j=0; i<ndo; i++) {
      float di=0,dq=0;
      k2 = (int)p;
      k1 = (int)((1.0-(p-k2))*nrpt);
      if (ntap==1) { 
	di=idat[k2];
	dq=qdat[k2];
      }
      else if (iqoff) {
        double pq = p+0.5;
        k2q = (int)pq;
        k1q = (int)((1.0-(pq-k2q))*nrpt);
        for (k=0; k<ntap; k++, k1+=nrpt,k2++, k1q+=nrpt,k2q++) { di += rtap[k1]*idat[k2]; dq += rtap[k1q]*qdat[k2q]; }
      }
      else {
	for (k=0; k<ntap; k++,k1+=nrpt,k2++) { di += rtap[k1]*idat[k2]; dq += rtap[k1]*qdat[k2]; }
      }
      buf[j++] = di;
      buf[j++] = dq;
      p += dp;
    }
  }

  private void rotlut (float[] buf, double p, double dp, int ndo) {
    float sa = (float)Math.sin(p*TWOPI);
    float ca = (float)Math.cos(p*TWOPI);
    float dsa = (float)Math.sin(dp*TWOPI);
    float dca = (float)Math.cos(dp*TWOPI);
    float cb;
    for (int i=0; i<ndo*2; i+=2) {
      float di=buf[i+0], dq=buf[i+1];
      buf[i+0] = di*ca - dq*sa;
      buf[i+1] = di*sa + dq*ca;
      cb = dca*ca - dsa*sa;
      sa = dca*sa + dsa*ca;
      ca = cb;
    }
  }

  private int fastramp (int[] buf, int type, int ndo, int mode, int cur) {
    if (type==INT && mode==2) {
      int inc = 0x00000001;
      for (int i=0; i<ndo; i++,cur+=inc) buf[i]=cur;
    }
    else M.error("FastRamp for type="+type+" and spa="+mode+" not supported");
    return cur;
  }

  private int testramp (float[] buf, int type, int ndo, int mode, int cur) {
    int range = (type==BYTE)? 0x100 : 0x10000;
    int max = range>>1;
    int inc = (type==LONG)? 0x03010402 : (type==BYTE)? 0x3 : 0x0100;
    for (int i=0,j=0; i<ndo; i++) {
      cur = (cur+inc);
      if (type!=LONG && cur>=max) cur -= range;
      if (type==INT && (cur&0xFF00)==0xFF00) cur++;
      buf[j++] = (float)cur;
      if (mode>1) buf[j++]=(float)cur;
    }
    return cur;
  }

  private void mux (float[] fbuf, float[] tbuf, int ndo) {
    for (int i=ndo-1,j=ndo*2; i>=0; i--) { fbuf[--j]=fbuf[i]; fbuf[--j]=tbuf[i]; }
  }

  private void rotate (float[] buf, int ndo, int spa) {
    if (spa==1) for (int i=0; i<ndo; i+=2) { buf[i+0]=-buf[i+0]; }
    else for (int i=0; i<ndo*2; i+=4) { buf[i+0]=-buf[i+0]; buf[i+1]=-buf[i+1]; }
  }

  private void cx2r (float[] buf, int ndo) {
    for (int i=0,j=0; i<ndo; i++,j+=2) buf[i]=buf[j]; 
  }

  public void lrs (float[] fbuf, double amp, int n, int spa) {
    int bit0; float factor = (float)(amp/2/Constants.B1G);
    for (int i=0; i<n*spa; i++) {
      fbuf[i] = factor * lrsx;
      bit0 = (~(lrsx ^ (lrsx>>1) ^  (lrsx>>5) ^ (lrsx>>25)))&0x1;
      lrsx <<= 1; lrsx |= bit0;
      if (spa==1) continue;
      fbuf[++i] = factor * lrsy;
      bit0 = (~(lrsy ^ (lrsy>>1) ^  (lrsy>>5) ^ (lrsy>>25)))&0x1;
      lrsy <<= 1; lrsy |= bit0;
    }
  }

}
