package nxm.ice.prim;

import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.ice.lib.TimeCode;
import nxm.ice.lib.ICEPacket;

/**
  Utilites for ICE time code processing

  @author Jeff Schoen
  @version $Id: icetc.java,v 1.3 2001/04/25 17:08:12 schoenj Exp $
 */
public class icetc extends Primitive {

  private final String funcList = "Top,Extract,Insert,Select";
  private final int TOP=1, EXTRACT=2, INSERT=3, SELECT=4;

  private DataFile hi,ho;
  private ICEPacket pkt,hpkt;
  private SDDSPacket spkt;
  private Data dbi,dbo;
  private int func,nchn,ichn;
  private TimeCode tc = new TimeCode();
  private Time time = new Time();
  private Time timeTop = new Time();
  private Time timeZero;
  private int ndo,hack,xper,tag,ltag,seq,nseq=-1,seqerr=0;
  private double delta,tolr;
  private boolean test,nopkt,gap,tic;
  private double exprate=1.0, drift=0, driftOffset=0;
  private static final double tag2time = 1.0/66.7e6;

  private final String selList = "MinMax,NearLE,NearGE";
  private final int MINMAX=1, NEARLE=2, NEARGE=3;	
  private int selMode;
  /** valid flag, column 11 */
  private final static int TCS = 11;
  /** time offset (in number of samples since TCWS), column 12 */
  private final static int TCO = 12;
  /** start time (j1950), column 13 */
  private final static int TCWS = 13;	

  public int open() {

    func = MA.getSelectionIndex("FUNC",funcList,0);
    nchn = MA.getL("/NCHN",1);
    ichn = MA.getL("/ICHN");
    test = MA.getState("/TEST");
    gap  = MA.getState("/GAP");
    tic  = MA.getState("/TIC");
    verbose = MA.getState("/VERBOSE");
    selMode = MA.getSelectionIndex("/SELMODE",selList,0);

    hi = MA.getDataFile("IN","1000,3000","SB,SI,CI,NH",0);
    hi.open(hi.INPUT);

    String mode = MA.getS("MODE");
    if (mode.equals("SDDS")) spkt = (SDDSPacket)hi.getPacketHandler();
    else if (!mode.equals("ICE")) tc.setMode(mode);	// embedded timecode data
    else if (hi.typeClass==3) hpkt = new ICEPacket();	// just ICEPacket headers
    else pkt = (ICEPacket)hi.getPacketHandler();	// ICEPacket data + headers
    nopkt = (pkt==null) && (hpkt==null) && (spkt==null);

    xfer = 8192;
    if (hpkt!=null) xfer = 1;
    if ( pkt!=null) xfer = pkt.getFixedSize();
    if (spkt!=null) xfer = 1024 / hi.getBPS();
    xfer = MA.getL("/TL",xfer);

    xper = xfer;		// input samples per output record
    delta = hi.getDelta();
    if (hpkt!=null) {
      String fs = hi.keywords.getMain("FS");
      if (fs!=null) xper = Convert.s2l(fs);
      delta /= xper;
    }
    delta = MA.getD("/DELTA",delta);

    dbi  = hi.getDataBuffer(xfer);
    hack = tc.getHack(delta);
    tolr = MA.getD("/TOLR",100e-9);

    if (func==INSERT) {
      ho = MA.getDataFile("OUT",hi,0);
      ho.open(ho.OUTPUT);
      time = MA.getTime("TIME",hi.getTime());
      tc.setTime (time);
      ho.setTime (time);
      ndo = tc.nextHack(delta);
    }
    else if (func==SELECT) {    	
      ho = MA.getDataFile("OUT",hi,0);
      ho.open(ho.OUTPUT);
      time = MA.getTime("TIME");
      tc.setTime(time);
      ho.setTime(time);
    }
    else {
      ho = MA.getDataFile("OUT","3000","NH",0);
      ho.setSubRecords("OFF|SD,WSEC|SD,FSEC|SD,DIFF|SD");
      if (gap) ho.setSubRecords("+GAP|SD");
      ho.setXDelta(delta*xper);
      ho.open(ho.OUTPUT|ho.OPTIONAL);
      MR.put (MA.getU("TIME"), time);
      dbo = ho.getDataBuffer(1);
      ndo = xfer;
    }

    return (NORMAL);
  }

  public int process() {
    int n=0;
    if (func==INSERT) {
      if (ndo==0) {
        n = hi.read(dbi,128);
        if (n<0) return FINISH;
        tc.setTimeCodeWords (dbi.buf,0,n);
        ndo = tc.nextHack(delta) - n;
        ho.write(dbi,n);
      } 
      n = hi.read(dbi,Math.min(ndo,xfer));
      if (n<0) return FINISH;
      tc.setFillWords (dbi.buf,0,n);
      ho.write(dbi,n);
      ndo -= n;
    }
    else if (func==SELECT) {
      double timeSearch = time.toJ1950();
      int numRows = (int) hi.getNumberOfRows();

      // determine min max
      int indBeg = determineFileMin(0,numRows-1);
      if(indBeg<0)
        throw new MidasException("Error! The input file has no valid times");
      int indEnd = determineFileMax(indBeg,numRows-1);

      // get the times in J1950 format
      double timeMin = getTimeFile(indBeg);
      double timeMax = getTimeFile(indEnd);
      boolean timeValid = (timeSearch>timeMin && timeSearch<timeMax);

      // now start working
      Table tabInsert;
      int currInd = -1;
      switch(selMode){
      case NEARLE:	// nearest index less than the given time
        if(!timeValid){
          M.warning("****************************************************************");
          M.warning("** Start time is not inside the file boundaries!");
          M.warning("** File range: " + timeString(timeMin) + " -> " + timeString(timeMax));
          M.warning("** Defaulting to the first valid index! ");
          M.warning("** Use /SELMODE=MINMAX  with icetc to determine valid times!");
          M.warning("****************************************************************");
          currInd = indBeg;
        }
        else
          currInd = searchRec(timeSearch,indBeg,indEnd);
        tabInsert = hi.getDataTable(currInd);
        ho.setData(0, tabInsert);
        break;
      case NEARGE:	// nearest index greater than the given time
        if(!timeValid){
          M.warning("****************************************************************");
          M.warning("** End time is not inside the file boundaries!");
          M.warning("** File range: " + timeString(timeMin) + " -> " + timeString(timeMax));
          M.warning("** Defaulting to the last valid index! ");
          M.warning("** Use /SELMODE=MINMAX  with icetc to determine valid times!");
          M.warning("****************************************************************");
          currInd = indEnd;
        }
        else
          currInd = searchRec(timeSearch,indBeg,indEnd)+1;
        tabInsert = hi.getDataTable(currInd);
        ho.setData(0, tabInsert);
        break;
      default:		// max and min
        // get the data tables and set the data
        tabInsert = hi.getDataTable(indBeg);
        ho.setData(0, tabInsert);
        tabInsert = hi.getDataTable(indEnd);
        ho.setData(1,tabInsert);
        break;
      }
      // we are done!
      return FINISH;    	
    }
    else {
      double offset = hi.getOffset();
      n = hi.read(dbi,xfer);
      if (n<0) return FINISH;
      if (nchn>0) offset /= nchn;
      if (hpkt!=null) {						// ICE Packets Only
        System.arraycopy(dbi.buf,0, hpkt.buf,0, 64);
        int tcstat = hpkt.getTC (tc.time,offset,0.0); // zero to bypass adjustment
        int count  = hpkt.getCount();
        int elem   = hpkt.getSize();
        int chan   = hpkt.getChannel();
        if (ichn>=0 && chan!=ichn) return NORMAL;
        if (verbose) MT.writeln(hpkt.listHeader());
        if (tcstat<=0) return NORMAL;
        if (xper==1) xper = elem;
        if (elem != xper) { M.info("Packet size="+xper+" change to "+elem); xper=elem; }
        if (count==0 && offset!=0) { 
          M.info("Packet count restart at "+offset);
          drift=0; driftOffset=ho.getOffset(); timeZero=null;
        }
        offset = hpkt.getTCO() + ((double)count)*((double)elem);
      }
      else if (pkt!=null) {					// ICE Packets + Data
        int tcstat = pkt.getTC (tc.time,offset,0.0); // zero to bypass adjustment
        int chan= pkt.getChannel();
        if (ichn>=0 && chan!=ichn) return NORMAL;
        if (verbose) MT.writeln(pkt.listHeader());
        if (tcstat<=0) return NORMAL;
        offset += pkt.getTCO();
      }
      else if (spkt!=null) {					// SDDS Packets + Data
        Time t = spkt.getTC();
        if (t!=null) tc.time=t; 
        else tc.time.addSec(xfer*delta); 
        if (verbose) MT.writeln(spkt.listHeader());
        seq = spkt.getCount();
        if (test && seq!=nseq && nseq>=0) { seqerr++;
        MT.writeln("Sequence Error expect="+nseq+"  got="+seq+"  at="+offset); }
        nseq = seq+1; if ((nseq&0x1F)==0x1F) nseq++; nseq &= 0xFFFF;
      }
      else {							// embedded timecode 
        ndo = xfer;
        int off = tc.findTimeCodeWords (dbi.buf,0,n);
        if (off<0) { ndo=xfer; return NORMAL; } // back to search mode
        offset += off;
      }
      timeTop.fromTime(tc.time);
      timeTop.addSec(-delta*offset);
      if (timeZero==null) {
        timeZero = new Time();
        timeZero.fromTime(timeTop);
      }
      double diff = timeTop.diff(timeZero);
      if (ho.isOpen) {
        dbo.setD(0,offset);
        dbo.setD(1,tc.time.getWSec());
        dbo.setD(2,tc.time.getFSec());
        dbo.setD(3,diff);
        if (gap) { 
          byte[] buf = new byte[64];
          spkt.getBuffer(buf,0);
          tag = Convert.unpackL(buf,4);
          double tgap = (ltag==0)? spkt.getSize()*delta : (tag-ltag)*tag2time;
          dbo.setD(4,tgap);
          ltag = tag;
        }
        ho.write(dbo,1);
      }
      double adiff = Math.abs(diff);
      double aerr  = adiff - drift;
      if (test && aerr>tolr) {
        M.warning("TC="+tc.time+" at index="+offset+" err="+diff+" tolr="+tolr);
      }
      drift = exprate*adiff + (1.0-exprate)*drift;
      if (func==TOP) {
        time.fromTime(timeTop);
        if (tic && offset<0);
        else return FINISH;
      }
      if (nopkt) hi.seek(offset+hack-4);
      ndo = 256;
      time.fromTime(tc.time);
    }
    return NORMAL;
  }

  public int close() {
    hi.close();
    ho.close();
    double driftrate = drift / ((ho.getOffset()-driftOffset)*delta*xper);
    if (test) M.info("TC drift total= "+drift+" sec  Rate="+driftrate+" sec/sec");
    if (test && seqerr>0)  M.info("SDDS seq err total= "+seqerr);
    if (func==TOP && !MA.isPresent("TIME")) MT.writeln("TopTime="+time.toString(9));
    return (NORMAL);
  }

  /** convert a J1950 double time to a pretty string

   @param currTime time since J1950 in seconds
   @return stringified time
   */
  private String timeString(double currTime){
    Time t = new Time();
    t.fromJ1950(currTime);
    return t.toString();
  }

  /** get the time for an index

   @param currInd desired index
   @return time since J1950 in seconds
   */
  private double getTimeFile(int currInd){
    return hi.getData(currInd,TCWS).getD(0) - hi.getData(currInd,TCO).getD(0)*delta;		
  }

  /** get the valid flag for a specified index

   @param currInd desired index
   @return true is valid, false if not
   */
  private boolean getValidFile(int currInd){
    return hi.getData(currInd,TCS).getP(0) > 0;
  }

  /** recursive search algorithm for min valid index.
	 if no valid indices are found, a linear
	 search function is called for thoroughness

	 @param currBeg minimum index for start
	 @param currEnd maximum index for start
	 @return first valid index, -1 for none
   */
  private int determineFileMin(int currBeg, int currEnd){
    boolean valBeg  = getValidFile(currBeg);
    boolean valEnd  = getValidFile(currEnd);

    // both indices at invalid (kick off linear search)
    if(!valBeg && !valEnd){ 
      return determineFileMinLinear(currEnd);
    }

    // if the first index is valid, we are done
    if(valBeg)
      return currBeg;

    // most likely exit case
    if((currEnd-currBeg)<=1){
      if(valEnd)
        return currEnd;
      // this shouldn't happen
      return -1;
    }

    // get the middle index and continue searching
    int currMid = currBeg + (currEnd-currBeg)/2;
    if(getValidFile(currMid))
      return determineFileMin(currBeg,currMid);
    else
      return determineFileMin(currMid,currEnd);
  }	

  /** linear search algorithm for min valid index.

	 @param currRows maximum index for search
	 @return first valid index, -1 for none
   */
  private int determineFileMinLinear(int currMax){
    for(int x=0;x<=currMax;x++){
      if(getValidFile(x))
        return x;
    }		
    return -1;
  }

  /** recursive search algorithm for max valid index.
	 if no valid index is found, linear search
	 algorithm is called for thoroughness

	 @param currBeg minimum index for search
	 @param currEnd maximum index for search
	 @return last valid index, -1 for none
   */
  private int determineFileMax(int currBeg, int currEnd){
    boolean valBeg  = getValidFile(currBeg);
    boolean valEnd  = getValidFile(currEnd);

    // both indices at invalid (kick off linear search)
    if(!valBeg && !valEnd){ 
      return determineFileMaxLinear(currBeg,currEnd);
    }

    // if the last index is valid, we are done
    if(valEnd)
      return currEnd;

    // less likely exit case
    if((currEnd-currBeg)<=1){
      if(valBeg)
        return currBeg;
      // this shouldn't happen
      return -1;
    }

    // get the middle index and continue searching
    int currMid = currBeg + (currEnd-currBeg)/2;
    if(getValidFile(currMid))
      return determineFileMax(currMid,currEnd);
    else
      return determineFileMax(currBeg,currMid);
  }

  /** linear search algorithm for max valid index.

	 @param currBeg minimum index for search
	 @param currEnd maximum index for search 
	 @return last valid index, -1 for none
   */
  private int determineFileMaxLinear(int currBeg, int currEnd){
    for(int x=currEnd;x>currBeg;x--){			
      if(getValidFile(x))
        return x;
    }		
    // this should be rare (only 1 valid index in the file)
    return currBeg;
  }

  /** O(log(n)) recursive search algorithm to find the 
	 closest time in a file to a user specified time

	 @param time desired search time
	 @param indBeg minimum index for search
	 @param indEnd maximum index for search
	 @return closest index less than the desired time
   */
  private int searchRec(double time, int indBeg, int indEnd){
    // exit case
    if(indEnd-indBeg<=1){
      return indBeg;
    }
    // get the mid index
    int indMid = indBeg + (indEnd-indBeg)/2;
    // get the time
    double currTime = getTimeFile(indMid);
    // compare and recursively call
    if(currTime < time)
      return searchRec(time,indMid,indEnd);
    else 
      return searchRec(time,indBeg,indMid);
  }

}
