package nxm.ice.prim;

import nxm.ice.lib.Core;
import nxm.ice.lib.CoreFactory;
import nxm.ice.lib.CoreComponent.Stream;
import nxm.ice.lib.CoreComponent.Buffer;
import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.sys.libg.GValue;

/**
  Implements a generic CORE library function 

  @author Jeff Schoen
  @version $Id: icecore.java,v 1.9 2011/05/24 13:21:12 jgs Exp $
*/
public class icecore extends Primitive implements Chainable, Keyable, DataTypes {

  private int dump,tmode,tl=4096,itl,otl,ilen,olen,iptl,optl,icvt=0,ocvt=0,flow,tagcnt=-1;
  private DataFile hi,ho, hip,hop, hcsf,ht;
  private Data dcsf,dt,d;
  private Core core;
  private Table keys;
  private String csfkeys;
  private double csfdt=1,csft=0,csfdi,csfi,startOffset,tcs;
  private Buffer bi,bo,bip,bop;
  private boolean isThreaded,isPacket,isICEX,mimicRT;
  private GValue gpw;

  public int open() {

    hi = MA.getDataFile ("IN","1000,2000","S#,C#",0);
    hi.open(hi.OPTIONAL);
    startOffset = hi.getOffset();

    String iform = hi.getFormat();
    String ifmt = MA.getS("/IFMT",iform);
    ilen = MA.getL("/ILEN");

    String ofmt = MA.getS("/OFMT",ifmt);
    olen = MA.getL("/OLEN");

    tmode = MA.getL("/TRACE",0x0);
    csfdt = MA.getD("/CPOLL",csfdt);

    Object gpwo = MA.getO("/GPWR");
    if (gpwo instanceof GValue) gpw = (GValue)gpwo;

    String func = MA.getCS("FUNC","Bogus");
    core = (Core)MA.getO("/CIN"); 
    if (core==null) core = Core.forName(func,MA);
    if (core==null) { M.error("Opening core name="+func); return FINISH; }
    core.setFormats(ifmt,ofmt);
    core.setTraceMode(tmode);

    flow = core.halo.flow;
    isThreaded = core.halo.impl > CoreFactory.CPU;

    keys = MA.getTable("/CFG"); 	// parameter table from command line
    if (keys==null) keys = new Table();
    keys.setMode("KV");

    for (int n=4; n<=MA.numberOf(); n++) {
      String key = MA.getKey(n);
      Data value = MA.getData(key);
      if (key.length()>2 && key.charAt(1)==':') key = key.substring(2);
      keys.put(key,value); 		// to show in Query
    }

    verbose = MA.getState("/VERBOSE");
    core.setKeyTable(null,keys,verbose);
    core.open();

    double oxd = hi.getXDelta();
    double oxs = hi.getXStart();

    d = core.get("XDIO?",DOUBLE);
    if (d!=null) oxd *= d.toD();
    d = core.get("XDO?",DOUBLE);
    if (d!=null) oxd = d.toD();
    d = core.get("XSIO?",DOUBLE);
    if (d!=null) oxs += d.toD();

    oxd = MA.getD("/OXD",oxd);
    oxs = MA.getD("/OXS",oxs);

    ho = MA.getDataFile ("OUT",hi,0);
    ho.setXDelta(oxd);
    ho.setXStart(oxs);
    ho.setFormat(ofmt);
    ho.open(ho.OPTIONAL);
    String oform = ho.getFormat();

    if (hi.typeClass==2) tl = hi.getSubsize();

    tl  = MA.getL("/TL",tl);
    itl = MA.getL("/ITL",tl);
    otl = MA.getL("/OTL",itl);

    hip = MA.getDataFile ("/PIN","1000","SL",0);	// packet in
    hip.open(hip.OPTIONAL);
    if (hip.isOpen) iptl = MA.getL("/IPTL", itl);
    if (flow==Core.PACKET) bip = core.getBuffer(128,4,4);

    hop = MA.getDataFile ("/POUT",hip,0);		// packet out
    hop.open(hop.OPTIONAL);
    if (hop.isOpen) optl = MA.getL("/OPTL", otl);
    if (flow==Core.PACKET) bop = core.getBuffer(1024,8,4);

    Table t = MA.getTable("/CSF");			// Core Status File Table
    if (t!=null) {
      hcsf = new DataFile(M,t.getS("FN"),"3000","NH",0);
      hcsf.setSubRecords(t.getS("FMT"));
      csfdt = t.getD("DT");
      hcsf.setXDelta(csfdt);
      csfkeys = t.getS("KEYS");
      hcsf.open(hcsf.OUTPUT);
      dcsf = hcsf.getDataBuffer(1);
    }

    if (hi.isOpen) {
      csfdi = hi.dbpe * csfdt / hi.getDelta();
    }
    mimicRT = MA.getState("/MRT");
    isICEX = MA.getS("/CORE").equals("ICEX");
    if (gpw!=null) gpw.setValue(0.0);

    tcs = MA.getD("/TCS");
    d = core.get("D:STARTUP?",DOUBLE);
    if (d!=null && tcs>0) core.set("D:STARTUP",tcs+Time.J1950TOJ1970);

    dump = MA.getL("/DUMP",0);				// debug parameter values
    if ((dump&1)!=0) dumpKeys();

    bi = core.getBuffer(itl,iform,ifmt);		// dble buffer by default
    bo = core.getBuffer(otl,ofmt,oform);

    return (NORMAL);
  }


  public int process() {
    int ni=0,no=0,nip=0,nop=0;
    boolean poll = mimicRT? csfi>csfdi : (Time.current()-csft)>csfdt;

    if (poll) {
      core.poll();
      if (hcsf!=null) {	// write state file with named parameters
        core.getList(csfkeys,dcsf);
        hcsf.write(dcsf);
      }
      csft = Time.current();
      csfi = 0;
    }
    if (isICEX) return NOOP;

    // handle input buffers
    if (hi.isOpen && ilen!=0) ni = core.load(bi,hi);
    if (hip.isOpen) nip = core.load(bip,hip);
    else if (flow==Core.PACKET) nip = gentags(bi,ni,bip);

    // handle processing pass
         if (flow==Core.STREAM) core.process((Stream)bi,(Stream)bo);
    else if (flow==Core.BUFFER) core.process(bi,bo);
    else if (flow==Core.BUFSTR) core.process(bi,(Stream)bo);
    else if (flow==Core.PACKET) core.process(bi,(Stream)bip,bo,(Stream)bop);
    else return (NOOP);

    // handle output buffers
    if (ho.isOpen) no = core.unload(bo,ho);
    if (hop.isOpen) nop = core.unload(bop,hop);
    else if (flow==Core.PACKET) nop = flush(bop);

    int ntot = ni+no+nip+nop;
    if (ntot == 0) return (NOOP);
    if (ntot < 0) return (FINISH);
    if (olen>0 && ho.getOffset()>=olen) return (FINISH);
    if (ilen>0 && hi.getOffset()>=ilen) return (FINISH);
    if (ilen==-2 && hi.getOffset()>=hi.getSize()) hi.seek(0);
    csfdi += ni;

    if (gpw!=null) gpw.setValue(getProgress());
    return (NORMAL);
  }

  public int close() {
    updateStatusKeys();
    if (gpw!=null) gpw.setValue(1.0);
    if (tmode!=0) dumpTrace();
    if ((dump&2)!=0) dumpKeys();
    String cout = MA.getU("/COUT");
    if (!StringUtil.isNullOrEmpty(cout)) MR.put(cout,core);
    else core.close();
    hi.close();
    ho.close();
    hip.close();
    hop.close();
    if (hcsf!=null) hcsf.close();
    return (NORMAL);
  }

  public double getProgress() {
    return hi.seek()/hi.getSize();
  }

  public int gentags (Buffer bi, int n, Buffer bip) {
    int stat=0;
    for (int i=-n; i<0; i+=4) { // look back over last load for start of packet 0xD55555555555 
      if (bi.xgetL(i)==0x55555555 && (bi.xgetL(i+4)&0xFFFF)==0xD555) {
	if (!isThreaded) {	// single thread - wait for entire packet
	  if (tagcnt>0) { bip.xwrL(0x10000); bip.xwrL(0x20000|tagcnt); stat+=8; }	// add EoP tag
	  tagcnt=0;
	} else {
	  if (tagcnt>0) { bip.xwrL(0x20000|tagcnt); stat+=4; }	// add EoP tag
	  tagcnt=0;       bip.xwrL(0x10000|tagcnt); stat+=4; 	// add SoP tag
	}
      }
      if (tagcnt>=0) tagcnt+=4;
    }
    if (n<0 && tagcnt>=0) {	// close out
      if (!isThreaded) bip.xwrL(0x10000);
      bip.xwrL(0x20000|tagcnt);
    }
    return stat;
  }

  public int flush (Buffer bop) {
    int stat = bop.bib;
    bop.rnxt(stat);
    return stat;
  }

  public void updateStatusKeys() {
    String[] list = keys.getKeys();
    for (int i=0; i<list.length; i++) {
      String key = list[i];
      Object value = keys.get(key);	// if value is blank - for status only
      if (value==null || value.equals("NULL") || value.equals("")) keys.put(key,core.get(key));
    }
  }

  public void dumpTrace() {
    int size = MA.getL("/TRACESZ",0x1000);
    ht = MA.getDataFile ("/TRACEFN","1000","SL",0); 	// trace out
    ht.setSize(size);
    ht.open(ht.OUTPUT);
    dt = ht.getDataBuffer(1);
    boolean compress = (tmode&0x3)!=3;
    for (int i=0; i<size; i++) {
      int data = core.getTraceData(i);
      dt.setL(0,data);
      int count = compress? (data>>28)&0xF : 1;
      for (;count>0; count--) ht.write(dt);
    }
    ht.close();
  }

  public void dumpKeys() {
    String[] list = keys.getKeys();
    for (int i=0; i<list.length; i++) {
      String key = list[i];
      byte type = (byte)'L';
      if (key.charAt(1)==':') type = (byte)key.charAt(0);
      Data value = core.get(key,type);
      MT.writeln("Key="+key+" Value="+value+" Hex="+Convert.l2x(Convert.o2l(value)));
    }
  }

  // Chainable interface
  public Object getNextLink() { return core; }
  public Object getPrevLink() { return null; }

  // use the keyable interface to expose all of the PIC library keys
  public String[] getKeys() {
    return keys.getKeys();
  }
  public Object setKey (String key, Object value) {
    Data data = (value instanceof Data)? (Data)value : new Data(value);
    if (key.equals("STATE")) setState((String)value);
    else if (key.equals("MCID")) core.setMCID(Convert.o2l(value));
    else core.set(key,data);
    return value;
  }
  public Object getKey (String key) {
    if (key.equals("STATE")) return getState();
    if (key.equals("PROGRESS")) return getProgress();
    Object value = core.get(key);
    if (verbose) MT.writeln("Get Key="+key+" Value="+value);
    return value;
  }

}
