package nxm.ice.prim;

import nxm.ice.lib.DevIce;
import nxm.ice.lib.MDevIce;
import nxm.ice.lib.Archiver;
import nxm.ice.lib.RamDiskResource;

import nxm.sys.lib.DataFile;
import nxm.sys.lib.Primitive;
import nxm.sys.lib.Native;
import nxm.sys.lib.Time;
import nxm.sys.libg.GProgress;

/** Copies data from input file. This primitives separates (detaches) the 
    header and the data into separate files. It also instantiates the native
    read and write functions in the IOResource.

  @author JGS

  TODO test packet handler

 */
public class icecopy extends Primitive {

  private static final int XFER = 4*1024*1024;	// default transfer length
  private static final long B2G = 0x80000000L;	// 2G

  private DataFile hi,ho;		// data files
  private long length,offset,olen,ooff;	// file length and offset
  private long ptr,ptrm,ptri,ptro;	// memory pointers
  private int flags=F_NATIVE; 		// operational modifiers
  private GProgress gpw;		// progress widget
  private double time,rate,osize;	// transfer rate reporting
  private MDevIce pic;			// handle to RAM driver
  private DevIce.DmaMap mapi,mapo;	// RAM driver buffers
  private Archiver ha;
  private int replay,pass,unpack,upm;
  private byte[] tbuf;
  private String arch;
  private boolean async;

  public static int F_ATTACH=1,F_NATIVE=2,F_NIN=4,F_NOUT=8,F_ASYNC=16;
  public static String flagList = "Attach,Native,nIn,nOut,Async";

  public int open () {    

    flags = MA.getOptionMask("/FLAGS",flagList,flags);
    boolean isn = is(F_NATIVE); // use native IO routines
    boolean att = is(F_ATTACH) || MA.getState("/ATTACH");
    int hnf = isn? DataFile.NATIVE : 0;
    unpack = MA.getL("/UNPACK");
    arch = MA.getS("/ARCH"); if (arch.length()<=0) arch=null;
    replay = MA.getL("/REPLAY", 1);    
    osize = MA.getD("/OSIZE", -1);
    async = is(F_ASYNC);

    // open ICE resource
    pic = new MDevIce(); 
    if (pic.open("ICERAM",0)<0) M.error("Problem opening ICERAM");

    // open the data files
    hi = MA.getDataFile("IN");
    hi.open(hi.INPUT|hnf);
    if (hi.io instanceof RamDiskResource) {
      mapi = pic.mapFile(hi); if (mapi==null) M.error("Problem mapping ICERAM DMA memory");
      mapi.setPaged(false);
      ptri = mapi.getVirtualAddress(0,mapi.bytes);	// map this space
    }
    rate = hi.getRate();

    ho = MA.getDataFile("OUT", hi, 0);
    ho.setDetached( att? 0 : 1);
    if (unpack==12) ho.setFormatType(INT);
    if (arch!=null) {
      ha = new Archiver();
      ha.init (this,ho.getURL()+arch,ho,0);
      ha.open(ha.NATIVE);
      osize = ha.getSize();
    } else {
      if (osize<0) osize = hi.getSize()*replay;
      ho.setSize(osize);
      ho.open(ho.OUTPUT|hnf);
    }
    if (ho.io instanceof RamDiskResource) {
      mapo = pic.mapFile(ho); if (mapo==null) M.error("Problem mapping ICERAM DMA memory");
      mapo.setPaged(false);
      ptro = mapo.getVirtualAddress(0,mapo.bytes);	// map this space
      ho.seek(osize);					// assume written
    }

    // get user supplied transfer length
    xfer = MA.getL("/TL", XFER);    
    pass = 0;
    offset = 0;
    length = (long) (hi.getSize()*hi.bpa);
    ooff = 0;
    olen = (long) (ho.getSize()*ho.bpa);

    // allocate a buffer    
    ptr = ptrm = Native.alloc(xfer);
    if (ptr<0) M.error("Unable to allocate buffer");

    // init/set read offset and size
    double top = MA.getD("/TOP");
    double dur = MA.getD("/DUR");
    if (unpack==12) {
      xfer = (xfer/768)*768;		// insure byte boundary
      upm = (rate>650e6)? -2 : -1;
      tbuf = new byte[(xfer/3)*4];
      if (top>0) offset = ((long)(top*(3.0/4.0)*rate)*hi.bpa) & 0xFFFFFFFFFFFFFF00L;
      if (dur>0) length = Math.min(length,offset+(long)(dur*(3.0/4.0)*rate)*hi.bpa);
    } else {
      if (top>0) offset = (long)(top*rate)*hi.bpa;
      if (dur>0) length = Math.min(length,offset+(long)(dur*rate)*hi.bpa);
    }
    if (offset!=0) hi.seek(offset/hi.bpa);

    // check for progress window
    if (MA.find("/AGPW")) {
      String text = "ICECOPY "+hi.getURL()+" to "+ho.getURL();
      gpw = new GProgress(this,"Transfer Progress","Run,Pause,Abort",text,0,0,0,this);
      gpw.setLabel("PROGRESS");
    }

    time = Time.current();
    return NORMAL;
  }

  public int process () {

    // determine the size and check for completion
    long avail = Math.min(length-offset,olen-ooff);
    int n = (int) Math.min(xfer,avail);
    if (n<=0) return FINISH;

    // read from input file
    int nr=n;
    if (is(F_NIN) || nr<=0);
    else if (mapi!=null) ptr = ptri+offset; 
    else if (mapo!=null) nr = hi.read(ptro+ooff,0,n);
    else nr = hi.read(ptr,0,n);

    // write to output file
    int nw=nr;
    if (is(F_NOUT) || nw<=0);
    else if (ha!=null) ha.write(ptr,0,nr,-1);
    else if (mapo!=null) { if (mapi!=null) Native.p2p(ptr,0,ptro+ooff,0, nw); }
    else if (unpack==12) { nw=nr*4/3; pic.cvt12(ptr,tbuf,nw,upm); ho.write(tbuf,0,nw); }
    else if (async && ho.io.avail()<nr);
    else { nw = ho.write(ptr,0,nr); }

    if (nw<nr) M.warning("Problem performing I/O");

    // update progress
    offset += nr; if (offset==length) { offset=0; hi.seek(offset); pass++; } 
    ooff += nw; if (ooff==olen) { ooff=0; ho.seek(ooff); } 
    if (gpw!=null) gpw.setProgress(getProgress());
    if (pass>=replay) return FINISH;

    return NORMAL;
  }

  public int close () {
    double total = pass*length + offset;
    double dtim = Time.current()-time;
    int rate = (int) (1e-6 * total / Math.max(0.0011,dtim));
    float ftot = (float) (total*1e-6), ftim = (float)(dtim - dtim%0.001);
    if (verbose) M.info("Transfer rate = "+ftot+" / "+ftim+" = "+rate+" Mby/sec");
    MR.put(MA.getU("/STAT"),rate);
    if (mapi!=null) mapi.close();
    if (mapo!=null) mapo.close();
    if (pic!=null) pic.close();
    if (ptrm>0) Native.free(ptrm);
    if (gpw!=null) gpw.close();
    hi.close();
    ho.close();
    if (ha!=null) { ha.flush(); ha.close(); }
    return FINISH;
  }

  // get the 0-1 progress of the transfer
  public double getProgress() {
    return (pass+((double)offset)/length)/replay;
  }

  private boolean is (int mask) {
    return (flags&mask) != 0;
  }
  
}
