package nxm.ice.libg;

import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.sys.libg.*;
import nxm.sys.libm.Window;
import nxm.sys.libm.Fft;
import nxm.sys.libm.Multiply;
import nxm.sys.libm.MagSquared;
import nxm.sys.lib.Native;
import nxm.ice.lib.ICEPacket;
import java.awt.*;

/**
  A class to plot multichannel data in the form of an ICE packet stream. 

  @author Jeff Schoen
  @version $Id: LayerMC.java,v 1.13 2002/10/22 16:09:57 schoenj Exp $

*/
public class LayerMC extends Layer {

  public float[] fbuf,wind,fbs;
  private ICEPacket pkth;
  private int nx, ny, nchan, ichan, nfft, navg, nout, nrot, spa, nfftx, navgx, nfs, iyrast, myrast, iexp;
  private double factor, delx, delf, dbRange=120, fixMin, fixMax, minRange, maxRange;
  private DataFile df;
  private Fft fft;
  private int skip=1;
  private int select=-1,selected=-1;
  private int single=-1;
  private int scaling=S_AUTOP;
  private String wname="HANN";
  private int type=T_LINE;
  private int proc=P_FFT;
  private int keep=0;
  private int[] ikeep;
  private boolean demux;
  private double a1s,a2s,asrat=0.9;
  public MImage mimage;
  private byte[] buf;
  private boolean yx=false,clear,first;
  private double[] ca1s,ca2s;
  private Color cgrid = new Color(0x505050);
  private Color cselect = null;
  private int[] iyrasts;
  private int mcs=0;

  public static String typeList = "Line,Raster,Point,XYPoint";
  public static String processList = "Raw,Magnitude,FFT,Histogram,Envelope";
  public static String windowList = nxm.sys.libm.Window.nameList;
  public static String scalingList = "FixMin,FixMax,AutoChan,AutoPage";

  private static int T_LINE=1, T_RASTER=2, T_POINT=3, T_XYPOINT=4;
  private static int P_RAW=1, P_MAG=2, P_FFT=3, P_HIST=4, P_ENV=5;
  private static int S_FIXMIN=0x1, S_FIXMAX=0x2, S_AUTOC=0x4, S_AUTOP=0x8;

  /* Constructors */
  public LayerMC () { 
    super(); 
    name = "MC";
    zaxis = true;
  }

  /** setup to attach a file */
  public boolean setFile (PlotFile pf, Table tab, Args MA) {
    setFile(pf);
    df = (DataFile)pf;
    pkth = (ICEPacket) df.getPacketHandler();
    if (pkth!=null) nfs=pkth.getFixedSize();		// its ICE packets
    else if (mcs>0) nfs=256;				// 1Kb frames with embedded channel #
    else if (df.typeClass==2) nfs=df.getFrameSize(); 	// 2D frames
    else throw new MidasException("LayerMC input must be framed as type2000 or ICE Packets");
    df.setDFS(0);
    cmap = MA.getSelectionIndex("CMAP",MColor.colorMapList,MColor.RAMP);
    cmap = MA.getSelectionIndex("/CMAP",MColor.colorMapList,cmap);
    ncolors = MA.getL("NC",32);
    colors = MColor.getColorMap(cmap,ncolors);
    spa = df.getSPA();
    setNChan(0);
    setNFft(256);
    ichan=0;
    first=true;
    return true;
  }

  public void setKeep (int ratio) {
    keep = ratio;
    ikeep = new int[nchan];
  }

  /** reset the Complex mode */
  public void update () {
    if (ndata>0) process();
  }

  public void setNChan (int n) {
    int nxy=0;
    for (int i=2; i<=6; i++) {	// check for perfect square
      if (i*i==n) nxy=nx=ny=i;
    }
    if (nxy==0) 
    for (nx=1; nx<64; nx*=2) {	// nice power of 2 numbers
      for (ny=1; ny<nx*2; ny++) {
        if (nx*ny>=n) break;
      }
      if (nx*ny>=n) break;
    }
    nchan = n;
    setXAxis(0.0,nx);
    setYAxis(0.0,ny);
  }
  public int getNChan() { return nchan; }

  private void redo() {
    findRange();
    MP.findRange();
    MP.useRange();
    MP.setup();
    refresh();
  }

  public void setNX (int n) {
    nx = Math.max(1,n);
    setXAxis(0.0,nx);
    if (nchan>0) setNY ((nchan-1)/nx + 1);
  }
  public int getNX() { return nx; }

  public void setNY (int n) {
    ny = Math.max(1,n);
    if (nchan<=0) nchan = nx*ny;
    setYAxis(0.0,ny);
  }
  public int getNY() { return ny; }

  public void setWindow (String wname) {
    this.wname = wname;
    wind = Window.get(wname,nfft,factor);
  }
  public String getWindow() { return wname; }

  public void setProcess (String process) {
    proc = Parser.find(processList,process,0);
    nfftx = nfft;
    clear = true;
  }
  public String getProcess() { return Parser.get(processList,proc); }

  public void setType (String typeName) {
    type = Parser.find(typeList,typeName,0);
    clear = true;
  }
  public String getType() { return Parser.get(typeList,type); }

  public void setMCS (int nc) { mcs=nc; if (nc>0) nfs=256; }

  public void setNFft (int nfft) { nfftx = nfft; }
  public int getNFft() { return nfft; }
  public void setNAvg (int navg) { navgx = navg; }
  public int getNAvg() { return navg; }
  public void setMin (double value) { fixMin = value; scaling|=S_FIXMIN; }
  public double getMin() { return fixMin; }
  public void setMax (double value) { fixMax = value; scaling|=S_FIXMAX; }
  public double getMax() { return fixMax; }
  public void setMinRange (double value) { minRange = value; }
  public double getMinRange() { return minRange; }
  public void setMaxRange (double value) { maxRange = value; }
  public double getMaxRange() { return maxRange; }
  public void setDbRange (double db) { dbRange = db; }
  public double getDbRange() { return dbRange; }
  public void setAutoScaleRatio (double value) { asrat = value; }
  public double getAutoScaleRatio() { return asrat; }
  public void setScaling (String value) { scaling = Parser.mask(scalingList,value,scaling); }
  public String getScaling() { return Parser.mask2s(scalingList,scaling); }

  private void setup() { 
    delx = df.getXDelta();
    delf = 1.0/(nfft*delx);
    int flags = Fft.FORWARD|Fft.PACKED;
    if (proc==P_FFT) {
      if (spa==2) {
        nout = nfft;
        nrot = nfft/2-1;
        flags |= Fft.COMPLEX;
        factor = 1.0/nfft;
      } else {
        nout = nfft/2;
        nrot = 0;
        flags |= Fft.REAL;
        factor = 1.0/(2*nfft);
      }
      fft = new Fft(nfft,flags);
      fbuf = new float[nout*2];
    } else {
      nout = nfft;
      fbuf = new float[nout*spa];
      if (type==T_POINT || type==T_XYPOINT) line.setType(line.POINT);
    }
    setFrame(nout);
    setSize(1);
    setData(df.getDataBuffer(nfft));
    setColor(1);
    setWindow(wname);
    ndata=0;
    iyrast=0;
    iyrasts = new int[nchan];
    if (type!=T_RASTER && nchan<=16) fbs = new float[nout*spa*nchan];
  }

  /** get the next available data from the input source */
  public int getData (int mode) {
    int i=0;
    if (df==null || !df.isOpen()) return 0;
    if (nfftx>0) { nfft=nfftx; setup(); nfftx=0; }
    int stat = 0;
    int ndo = (int)df.avail();
    if (ndo>0) {
      ndo = Math.min(nfft,nfs);
      myrast = (int)((MP.myy>0)?MP.myy:-MP.myy)-4;
      if (pkth==null) {
        df.read(data,ndo);
        if (nfs>nfft) df.skip(nfs-nfft);
	if (mcs>0) {
	  ichan = getMCSchan(data.buf);
	} else {
          ichan = (ichan%nchan) + 1;
          if (ichan==nchan && skip>1) df.skip(nfs*nchan*(skip-1));
	}
        if (ichan<=1 && ++iyrast>myrast) iyrast=0;
      } else {
        df.read(data,ndo);
        ndo = pkth.getSize();
        ichan = pkth.getChannel();
        if (ndo>nfft) df.skip(ndo-nfft);
	i = ichan-1; if (i<0 || i>=nchan) i=0;
        iyrast = iyrasts[i];
        if (++iyrast>myrast) iyrast=0;
        iyrasts[i] = iyrast;
      }
      process();
      stat = Drawable.DATAX;
    }
    else if (!realtime) stat = 0;
    else if (df.isStream() && !df.isStreaming()) stat = Drawable.DONE;
    return stat;
  }

  /** process the plot data for this frame */
  public void process() {
   int i,ix,iy;
    ndata = 0; if (data==null) return;
    navg = navgx;
    if (single>=0 && ichan!=single) return;
    i = ichan-1;
    if (yx) { ix = i/ny; iy = i-ix*ny; } 
    else { iy = i/nx; ix = i-iy*nx; }
if (MP.mxx>0) {
    if ((int)(MP.mxx*ix+MP.mxb) > MP.ix2) return;	// off screen R
    if ((int)(MP.mxx*(ix+1)+MP.mxb) < MP.ix1) return;	// off screen L
}
if (MP.myy>0) {
    if ((int)(MP.myy*iy+MP.myb) > MP.iy2) return; 	// off screen T
    if ((int)(MP.myy*(iy+1)+MP.myb) < MP.iy1) return; 	// off screen B
}
    boolean autoc = (scaling&S_AUTOC) != 0;
         if (proc==P_FFT) processFFT();
    else if (proc==P_RAW) processRaw();
    else if (proc==P_MAG) processMag();
    else if (proc==P_HIST) processHist();
    else if (proc==P_ENV) processEnv();
    if (autoc && (ca1s==null || ca1s.length<nchan)) {
      ca1s = new double[nchan];
      ca2s = new double[nchan]; 
      for (int j=0; j<nchan; j++) ca1s[j]=ca2s[j]=0; 
    }
    if (autoc) { a1s=ca1s[i]; a2s=ca2s[i]; }
    a1s = (a1*asrat) + (a1s*(1-asrat));
    a2s = (a2*asrat) + (a2s*(1-asrat));
    if (first) { a1s=a1; a2s=a2; first=false; }
    double ads = (a2s-a1s)*0.05;
    a1 = Math.min(a1,a1s-ads);
    a2 = Math.max(a2,a2s+ads);
    if ((scaling&S_FIXMIN)!=0) a1 = fixMin;
    if ((scaling&S_FIXMAX)!=0) a2 = fixMax;
    if (autoc) { ca1s[i]=a1s; ca2s[i]=a2s; }
  }

  public void processAvg (float[] fbuf, int n) {
    float scl0=1.0F/navg, scl1=1.0F-scl0;
    int j = (ichan-1)*nout*spa;
    for (int i=0; i<n; i++,j++) {
      fbs[j] = fbuf[i] = fbuf[i]*scl0 + fbs[j]*scl1;
    }
  }

  public void processFFT() {
    Convert.bb2ja (data.buf,0,data.type, fbuf,0,data.FLOAT, nfft*spa);
    if (nfs<nfft) for (int i=nfs*spa; i<nfft*spa; i++) fbuf[i]=0;
    if (spa==2) Multiply.CSC (fbuf,wind,fbuf,nfft);
    else        Multiply.SSS (fbuf,wind,fbuf,nfft);
    fft.work(fbuf);
    MagSquared.CS (fbuf,fbuf,nout);
    if (fbs!=null && navg>1) processAvg(fbuf,nout);
    if (data.type==DataTypes.BYTE || data.type==DataTypes.INT) Native.dbf (fbuf,nout,0);
    else nxm.sys.libm.LogarithmBounded.SS(fbuf,fbuf,nout,Math.log(10.0));
    if (spa==2) fft.rotateSF(fbuf);
    a2 = fbuf[0];
    for (int i=1; i<nout; i++) {
      if (fbuf[i]>a2) a2=fbuf[i];
    }
    a1 = a2-dbRange;
    ndata = nout;
  }

  public void processRaw() {
    if (spa==2 && type==T_RASTER) { processMag(); return; }
    Convert.bb2ja (data.buf,0,data.type, fbuf,0,data.FLOAT, nout*spa);
    a1 = a2 = fbuf[0];
    for (int i=1; i<nout*spa; i++) {
      if (fbuf[i]<a1) a1=fbuf[i];
      if (fbuf[i]>a2) a2=fbuf[i];
    }
    ndata = nout;
  }

  public void processMag() {
    if (spa==1) { processRaw(); return; }
    Convert.bb2ja (data.buf,0,data.type, fbuf,0,data.FLOAT, nout*spa);
    fbuf[0] = fbuf[0]*fbuf[0] + fbuf[1]*fbuf[1];
    a1 = a2 = fbuf[0];
    for (int i=1,j=2; i<nout; i++,j+=2) {
      float f = fbuf[j]*fbuf[j] + fbuf[j+1]*fbuf[j+1];
      if (f<a1) a1=f;
      if (f>a2) a2=f;
      fbuf[i] = f;
    }
    ndata = nout;
  }

  public void processHist() {
  }

  public void processEnv() {
  }

  /* find layer bounds */
  public void findRange() { 
    x1 = 0; x2 = nx;
    y1 = 0; y2 = ny;
  }

  /** dray the layer on the specified plot */
  public void draw (int flag) {
    super.draw(flag);
    if (!MP.is2D) return;
    int i,j,ix,iy,ix1,ix2,iy1,iy2;
    Color cfill = MP.getOption(MP.O_CONTRAST)? Color.black : MP.theme.cbg;
    // draw the channel grid
    if (flag != Drawable.DATAX || clear) { 
      MP.gc.setColor(cfill);
      MP.gc.fillRect(MP.ix1,MP.iy1,MP.ix21,MP.iy21);
      MP.setColor(cgrid);
      i = (MP.myy>0)? 0 : ny;
      iy1 = Math.max(MP.iy1,(int)(MP.myy*i+MP.myb));
      i = (MP.myy>0)? ny : 0;
      iy2 = Math.min(MP.iy2,(int)(MP.myy*i+MP.myb));
      for (i=0; i<=nx; i++) {
        ix = (int)(MP.mxx*i+MP.mxb);
	if (ix>MP.ix1 && ix<MP.ix2) MP.gc.drawLine(ix,iy1,ix,iy2);
      }
      i = (MP.mxx>0)? 0 : nx;
      ix1 = Math.max(MP.ix1,(int)(MP.mxx*i+MP.mxb));
      i = (MP.mxx>0)? nx : 0;
      ix2 = Math.min(MP.ix2,(int)(MP.mxx*i+MP.mxb));
      for (i=0; i<=ny; i++) {
        iy = (int)(MP.myy*i+MP.myb);
	if (iy>MP.iy1 && iy<MP.iy2) MP.gc.drawLine(ix1,iy,ix2,iy);
      }
      myrast = (int)((MP.myy>0)?MP.myy:-MP.myy)-4;
      clear=false; 
    }
    if (ndata<=0) return;
    // draw the channel data
    if (single>=0 && ichan!=single) return;
    i = ichan-1;
    if (yx) {
      ix = i/ny;
      iy = i-ix*ny;
    } else {
      iy = i/nx;
      ix = i-iy*nx;
    }
    i = (MP.mxx>0)? ix : ix+1;
    ix1 = (int)(MP.mxx*i+MP.mxb)+2;
    i = (MP.mxx>0)? ix+1 : ix;
    ix2 = (int)(MP.mxx*i+MP.mxb)-2;
    i = (MP.myy>0)? iy : iy+1;
    iy1 = (int)(MP.myy*i+MP.myb)+2;
    i = (MP.myy>0)? iy+1 : iy;
    iy2 = (int)(MP.myy*i+MP.myb)-2;
    if (ix1>MP.ix2 || ix2<MP.ix1) return;
    if (iy1>MP.iy2 || iy2<MP.iy1) return;
    double ry2 = a2;
    double ry1 = a1;
    if (minRange!=0 && ry2-ry1<minRange) ry2 = ry1+minRange;
    if (maxRange!=0 && ry2-ry1>maxRange) ry1 = ry2-maxRange;
    if (type==T_RASTER) {
      if (buf==null || buf.length<nout) {
	buf = new byte[nout];
	mimage = new MImage(MP,nout,1,buf,colors);
      }
      int b, nc=ncolors;
      float scl0 = (float)ry1;
      float scl1 = (float)(ncolors/(ry2-ry1));
      for (i=0; i<nout; i++) {
        b = (int)( (fbuf[i]-scl0)*scl1 );
        if (b<0) b=0;
        if (++b>nc) b=nc;
        buf[i]=(byte)b;
      }
      mimage.source.newPixels(0,0,nout,1);
      MP.gc.drawImage (mimage.image, ix1,iy1+iyrast,ix2,iy1+iyrast+1, 0,0,nout,1, mimage);
      //int iyn = iy1+iyrast+1, ix1n=Math.max(ix1,MP.ix1)+1, ix2n=Math.min(ix2,MP.ix2)-2;
      int iyn = iy1+iyrast+1, ix1n=Math.max(ix1,MP.ix1), ix2n=Math.min(ix2,MP.ix2);
      int ixd = (ix2n-ix1n)>>4; if (ixd<1) ixd=1;
      if (iyrast<myrast && iyn>MP.iy1 && iyn<MP.iy2) {
        MP.setColor(cgrid);
        MP.gc.drawLine (ix1n+1,iyn,ix1n+ixd,iyn);
        MP.gc.drawLine (ix2n-ixd,iyn,ix2n-1,iyn);
      }
      return;
    }
    boolean clearCell=true; i=ichan-1;
    if (keep>1 && ichan>0 && ichan<=nchan) { i=ichan-1; if (++ikeep[i]>=keep) ikeep[i]=0; else clearCell=false; }
    if (clearCell) { MP.gc.setColor(cfill); MP.gc.fillRect(ix1-1,iy1-1,ix2-ix1+2,iy2-iy1+2); }
    int[] pax=MP.pix.x, pay=MP.pix.y;
    double myy = (double)(iy1-iy2) / (ry2 - ry1);
    double myb = iy2 - ry1*myy;
    double mxx = (double)(ix2-ix1) / (ry2 - ry1);
    double mxb = ix1 - ry1*mxx;
    double x = ix1;
    double dx = (double)(ix2-ix1)/nout;
    int iymin = iy1+1;
    int iymax = iy2-1;
    int y;
    if (cselect!=null) {
      if (selected==ichan) setColor(cselect); else setColor(1);
    }
    if (type==T_XYPOINT && spa==2) {
      for (i=0,j=0; i<nout; i++) {
        ix = (int)( (fbuf[j++])*mxx + mxb);
        iy = (int)( (fbuf[j++])*myy + myb);
        if (ix<ix1) ix=1; else if (ix>ix2) ix=ix2; pax[i]=ix;
        if (iy<iy1) iy=1; else if (iy>iy2) iy=iy2; pay[i]=iy;
      }
      MP.drawPixels (pax,pay,nout,line);
    }
    else if (proc==P_RAW && spa==2) {
      for (i=0,j=0; i<nout; i++,j+=2,x+=dx) {
        pax[i] = (int)( x );
          y    = (int)( (fbuf[j])*myy + myb);
        if (y<iymin) y=iymin; else if (y>iymax) y=iymax; pay[i]=y;
      }
      MP.drawPixels (pax,pay,nout,line);
      for (i=0,j=1; i<nout; i++,j+=2) {
          y    = (int)( (fbuf[j])*myy + myb);
        if (y<iymin) y=iymin; else if (y>iymax) y=iymax; pay[i]=y;
      }
      MP.drawPixels (pax,pay,nout,line);
    }
    else {
      for (i=0; i<nout; i++,x+=dx) {
        pax[i] = (int)( x );
          y    = (int)( (fbuf[i])*myy + myb);
        if (y<iymin) y=iymin; else if (y>iymax) y=iymax; pay[i]=y;
      }
      MP.drawPixels (pax,pay,nout,line);
    }
  }

  private int getChan (Position pos) {
    int ix = (int)pos.x;
    int iy = (int)pos.y;
    if (ix<0 || ix>=nx || iy<0 || iy>=ny) select = 0;
    else if (yx) select = ix*ny + iy + 1;
    else         select = iy*nx + ix + 1;
    return select;
  }

  public void select (Position pos) {
    select = getChan(pos);
  }
  public int select() {
    return select;
  }
  public int getSelect() {
    return select;
  }
  public void setSelect (int chan) {
    selected = chan;
  }
  public void setSelectColor (String cstr) {
    cselect = MColor.getColor(cstr);
  }

  public void single (int channel) {
    single = channel;
  }
  public int single() {
    return single;
  }

  public void skip (int frames) {
    skip = frames;
  }
  public int skip() {
    return skip;
  }

  public String getReadOut (Position mp) {
    int iexp, jexp, chan=getChan(mp);
    double ax = mp.x % 1.0;
    double ay = mp.y % 1.0;
    int i = (int)(ax*nout);
    ay = a1 + (1.0-ay)*(a2-a1);
    jexp = getTicExp(0.0,(a2-a1));
    if (proc==P_FFT) {
      ax = (i-nrot)*delf;
      iexp = getTicExp(0.0,delf*nout);
    } else if (type==T_XYPOINT) {
      ax = a1 + ax*(a2-a1);
      iexp = jexp;
    } else {
      ax = i*delx;
      iexp = getTicExp(0.0,delx*nout);
    }
    String tmp = " MC (Chan="+chan+" x="+MP.formNumber(ax,iexp,0);
    if (type!=T_RASTER) tmp += " y="+MP.formNumber(ay,jexp,0);
    if (type!=T_RASTER && proc==P_FFT && chan>0 && ca2s!=null) {
      tmp += " dyPk="+MP.formNumber(ay-ca2s[chan-1],jexp,0);
    }
    tmp += ")";
    return tmp;
  }

  private int getTicExp (double dmin, double dmax) {
    double absmax; int kengr;
    absmax = Math.max( Math.abs(dmin),Math.abs(dmax) );
    if (absmax == 0.0) return 0;
    kengr = (int)( 0.1447648 * Math.log(absmax/2) );
    if (absmax<1.0) kengr--;
    return kengr*3;
  }

  public void setYX (boolean state) {
    yx = state;
  }

  public int getMCSchan (byte[] buf) {
    int chan = ((buf[0]&1)<<0) | ((buf[2]&1)<<1) |  ((buf[4]&1)<<2) |  ((buf[6]&1)<<3);
    if (++chan>=mcs) chan=mcs;
    return chan;
  }

}
