/*****************************************************************************/
/*                                                                           */
/* part_cmd.h                                                                */
/*                                                                           */
/* The command interface is 64-bits:                                         */
/*                                                                           */
/* [63:56] - Command Key associated with the current command                 */
/* [55:48] - Part ID of the target part (so C3 Part knows how to route it)   */
/* [47:44] - Function :                                                      */
/*                   0 = Pin Write                                           */
/*                   1 = Pin Read                                            */
/*                   2 = Register Write                                      */
/*                   3 = Register Read                                       */
/*                   4 = Command Part Pin Write                              */
/*                   5 = Command Part Pin Read                               */
/*                   6 = Command Part Register Write                         */
/*                   7 = Command Part Register Read                          */
/* [43:32] - Address                                                         */
/* [31:0]   - Data                                                           */
/*                                                                           */
/* Data will be routed to either a pin control register, or a part specific  */
/*   interface register depending on the Function field.                     */
/* Address 0x3FF is designated as a 32-bit mask for read or write data.      */
/*                                                                           */
/* To talk using the Command Part, you must first request and be granted     */
/*   a session lock and given a Command Key.  Then commands written to the   */
/*   Command Part address will be routed to the part given by the            */
/*   Part ID field.                                                          */
/*                                                                           */
/* Command Part Status Registers:                                            */
/*   Address 1 (R/W)  "Session Timeout"                                      */
/*                       Sets the duration the Command Part will wait after  */
/*                       the command fifo has been executed until releasing  */
/*                       the session lock                                    */
/*   Address 2 (R/W)  "Session Lock"                                         */
/*                       Read to check if the Command Part is currently      */
/*                       locked, write to request a session lock             */
/*   Address 3 (R/W)  "Executed Command Count"                               */
/*                       Holds the total number of commands the Command      */
/*                       Part has executed since reset.  Can write 0 to      */
/*                       clear the count.                                    */
/*   Address 4 (R)    "Command Count"                                        */
/*                       Holds the number of commands currently in the       */
/*                       Command Part FIFO                                   */
/*   Address 5 (R)    "Readback Count"                                       */
/*                       Holds the number of status readback words           */
/*                       currently in the Readback FIFO                      */
/*   Address 16 (R)   "Command FIFO Depth"                                   */
/*                       Holds the total depth of the Command Part FIFO      */
/*                                                                           */
/*                                                                           */
/*****************************************************************************/
#ifndef _PART_CMD_H
#define _PART_CMD_H

#include "part_common.h"

/* CMD unique part identifier  */
enum CMD_PART_ID {
  PTID_CMD = FPGA_PTID_CMD 
};

/*                                                                           */
/* CMDBUF - Command buffer data type                                         */
/*   Since the command,control,comms interace is currently 64-bit, we could  */
/*   use a UDB representation, but that would always assume a 64-bit         */
/*   interface. Best to define a specific implementation here.               */
/*                                                                           */
typedef union {

  /* Overlap data types in a single 64-bit word */
  uint8_t   u8[8];
  int8_t    i8[8];  
  uint16_t  u16[4];
  int16_t   i16[4];  
  uint32_t  u32[2];
  int32_t   i32[2];  
  float32_t f32[2];
  uint64_t  u64[1];
  int64_t   i64[1];  
  float64_t f64[1];
 
  /* CMD word register specific positions for fast access */
  struct {
    uint32_t data:32;   /* Bits [31:00] - register data       */
    uint32_t addr:12;   /* Bits [43:32] - register address    */
    uint32_t fcn:4;     /* Bits [47:44] - function to perform */
    uint32_t ptid:8;    /* Bits [55:48] - part identifier     */
    uint32_t cmdkey:8;  /* Bits [63:56] - command key         */
  } c3w;    

#ifdef _ISOC99_SOURCE
  /* Nibble (4-bit) positions within 64 bit word */
  struct {
    uint8_t p00:4;
    uint8_t p01:4;
    uint8_t p02:4;
    uint8_t p03:4;
    uint8_t p04:4;
    uint8_t p05:4;
    uint8_t p06:4;
    uint8_t p07:4;
    uint8_t p08:4;
    uint8_t p09:4;
    uint8_t p10:4;
    uint8_t p11:4;
    uint8_t p12:4;
    uint8_t p13:4;
    uint8_t p14:4;
    uint8_t p15:4;
  } n;

  /* Single bit positions within 64 bit word */
  struct {
    uint8_t p00:1;
    uint8_t p01:1;
    uint8_t p02:1;
    uint8_t p03:1;
    uint8_t p04:1;
    uint8_t p05:1;
    uint8_t p06:1;
    uint8_t p07:1;
    uint8_t p08:1;
    uint8_t p09:1;
    uint8_t p10:1;
    uint8_t p11:1;
    uint8_t p12:1;
    uint8_t p13:1;
    uint8_t p14:1;
    uint8_t p15:1;
    uint8_t p16:1;
    uint8_t p17:1;
    uint8_t p18:1;
    uint8_t p19:1;
    uint8_t p20:1;
    uint8_t p21:1;
    uint8_t p22:1;
    uint8_t p23:1;
    uint8_t p24:1;
    uint8_t p25:1;
    uint8_t p26:1;
    uint8_t p27:1;
    uint8_t p28:1;
    uint8_t p29:1;
    uint8_t p30:1;
    uint8_t p31:1;
    uint8_t p32:1;
    uint8_t p33:1;
    uint8_t p34:1;
    uint8_t p35:1;
    uint8_t p36:1;
    uint8_t p37:1;
    uint8_t p38:1;
    uint8_t p39:1;
    uint8_t p40:1;
    uint8_t p41:1;
    uint8_t p42:1;
    uint8_t p43:1;
    uint8_t p44:1;
    uint8_t p45:1;
    uint8_t p46:1;
    uint8_t p47:1;
    uint8_t p48:1;
    uint8_t p49:1;
    uint8_t p50:1;
    uint8_t p51:1;
    uint8_t p52:1;
    uint8_t p53:1;
    uint8_t p54:1;
    uint8_t p55:1;
    uint8_t p56:1;
    uint8_t p57:1;
    uint8_t p58:1;
    uint8_t p59:1;
    uint8_t p60:1;
    uint8_t p61:1;
    uint8_t p62:1;
    uint8_t p63:1;
  } b;
#endif

} CMDBUF;
  
static const CMDBUF cmdbuf_initializer = {
    {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00} 
}; 

/* CMDBUF 64-bit register Masks for fast access */
enum PT_CMDBUF_MASK {
  PT_CMDBUF_CMDKEY = ((int32_t)0xFF000000), /* Command key                  */
  PT_CMDBUF_PTID   = ((int32_t)0x00FF0000), /* Part ID                      */
  PT_CMDBUF_FCN    = ((int32_t)0x0000F000), /* Func (PinWrite,RGWrite,etc.) */
  PT_CMDBUF_ADDR   = ((int32_t)0x00000FFF), /* Register address             */
  PT_CMDBUF_DATA   = ((int32_t)0xFFFFFFFF)  /* Register data                */
};

/* Part Control Function Definitions */
enum PT_CMD_CTLFUNC {
  PFUNC_PWR = 0,   /* Part Pin Write */
  PFUNC_PRD = 1,   /* Part Pin Read */
  PFUNC_RWR = 2,   /* Part Register Write */
  PFUNC_RRD = 3,   /* Part Register Read */
  CFUNC_PWR = 4,   /* Command Part Pin Write */
  CFUNC_PRD = 5,   /* Command Part Pin Read */
  CFUNC_RWR = 6,   /* Command Part Register Write */
  CFUNC_RRD = 7    /* Command Part Register Read */
};

/* CMD Pin control Masks */
enum CMD_PIN_MASK {
  CMD_PIN_NULL = 0x00000001   /* Null function place holder */
};

/* CMD Register ID's */
enum CMD_RG_ID {
  CMD_RG_SESSN_TMOUT = 0,  /* Session Timeout register        */
  CMD_RG_SESSN_LOCK  = 1,  /* Session Lock register           */
  CMD_RG_EXEC_CMDCNT = 2,  /* Executed Command Count register */
  CMD_RG_FIFO_CMDCNT = 3,  /* Command Count register          */
  CMD_RG_FIFO_DEPTH  = 4,  /* Command FIFO Depth register     */
  CMD_RG_FIFO_RBCNT  = 5   /* Readback Count register         */
};


/* CMD Register Keys and Masks */
enum CMD_RG_MASK {
  CMD_RGM_00_CNT = 0x0000FFFF,  /* 16-bit counter of cmds to execute  */
  CMD_RGM_01_CNT = 0x0000FFFF   /* 16-bit counter of rtn values       */
};



/*---------------------------------------------------------------------------*/
/* Public function prototypes from part_cmd.c */


#endif  /* _PART_CMD_H */
