/****************************************************************************/
/*                                                                          */
/*   LB2D module interface routines                                         */
/*                                                                          */
/****************************************************************************/
#define LB2DR1_NUM_PRG_BITS 32
#define LB2DR1_SYSCFG_INDEX        0x00000000
#define LB2DR1_INITCFG_INDEX       0x10000000
#define LB2DR1_GENCFG_INDEX        0x20000000
#define LB2DR1_ATTENCFG_INDEX      0x30000000
#define LB2DR1_RCVRCFG_INDEX       0x40000000
#define LB2DR1_GAINCFG_INDEX       0x50000000
#define LB2DR1_SCLKCFG_INDEX       0x60000000
#define LB2DR1_POWER_INDEX         0x70000000
#define LB2DR1_A2DCFG_INDEX        0x80000000
#define LB2DR1_CXMCFGREG_INDEX     0x90000000
#define LB2DR1_CXMENABLE_INDEX     0xA0000000
#define LB2DR1_PPSCFG_INDEX        0xB0000000 

#define LB2DR1_RAMREG_SWITCH     0x40001000 
#define LB2DR1_RAMREG_FREQL      0x40006000 
#define LB2DR1_RAMREG_FREQU      0x40006400 
#define LB2DR1_RAMREG_FREFL      0x40006800 
#define LB2DR1_RAMREG_FREFU      0x40006C00 
#define LB2DR1_RAMREG_BW         0x40007000 
#define LB2DR1_RAMREG_OPTS       0x40007400 
#define LB2DR1_RAMREG_ATTN       0x40007800 
#define LB2DR1_RAMREG_GAIN       0x40007900 
#define LB2DR1_RAMREG_BBAMP      0x40007A00 

/* the read addresses are different for now */
#define LB2DR1_RAMREG_SWITCH_R   0x10000010 
#define LB2DR1_RAMREG_FREQL_R    0x10000060 
#define LB2DR1_RAMREG_FREQU_R    0x10000064 
#define LB2DR1_RAMREG_FREFL_R    0x10000068 
#define LB2DR1_RAMREG_FREFU_R    0x1000006C 
#define LB2DR1_RAMREG_BW_R       0x10000070 
#define LB2DR1_RAMREG_OPTS_R     0x10000074 
#define LB2DR1_RAMREG_ATTN_R     0x10000078 
#define LB2DR1_RAMREG_GAIN_R     0x10000079 
#define LB2DR1_RAMREG_BBAMP_R    0x1000007A 

#define LB2D_DBGPRINT 0

int_4 lb2d_wr (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 data, int_4 bytes) {
  int_4 i,mbval;
  if (LB2D_DBGPRINT >= 3) 
    printf("Write a=%08x d=%08x n=%d\n",addr,data,bytes);
  if (bytes==0) { 
    /* this case needed because value passed into wmodreg was more than 1 byte 
       meaning it did not follow the 1 byte data rule or included the lower address */
    pic_wmodreg (p,mport,0x0,addr|data,LB2DR1_NUM_PRG_BITS);
    udelay(1000);
  }
  else {
    for (i=0; i<bytes; i++) {
      mbval = addr | (data&0xFF); /* the data is in byte [0] */
      pic_wmodreg (p,mport,0x0,mbval,LB2DR1_NUM_PRG_BITS);
      data >>= 8;
      addr += 0x100; /* the write address is bytes [2:1] */
      udelay(1000);
    }
  }
  return 0;
}

int_4 lb2d_rd (PICSTRUCT *p, int_4 mport, int_4 addr, int_4 bytes) {
  int_4 i,mbval,data=0;
  if (bytes==0) { 
    /* this case needed because value passed into wmodreg was more than 1 byte 
       meaning it did not follow the 1 byte data rule or included the lower address */
    pic_rmodreg (p,mport,(int_u4)addr,(int_u4*)&mbval,LB2DR1_NUM_PRG_BITS);
    data = mbval;
    udelay(1000);
  }
  else for (i=0; i<bytes; i++) {
    pic_rmodreg (p,mport,(int_u4)addr,(int_u4*)&mbval,LB2DR1_NUM_PRG_BITS);
    data |= ((mbval&0xFF)<<(8*i));
    addr += 0x1; /* for now until we move the read address to the same byte as the output address */
    udelay(1000);
  }
  return data;
}

/*
-----------------------------------------------------------------------
  PRIVATE INTERFACE FUNCTIONS:
    int_4 lb2d_reset(PICSTRUCT *p, int_4 mport)
    int_4 lb2d_bmswitch_mrd(PICSTRUCT *p, int_4 mport, int_u4 *value)
    int_4 lb2d_bmswitch_mwr(PICSTRUCT *p, int_4 mport, int_u4 value)
    int_4 lb2d_max2112_mrd(PICSTRUCT *p, int_4 mport, int_u4 reg, int_u4 *value)
    int_4 lb2d_max2112_mwr(PICSTRUCT *p, int_4 mport,int_u4 value)
    int_4 lb2d_si5326_mwr(PICSTRUCT *p, int_4 mport,int_u4 value)
    int_4 lb2d_a2d_setup(PICSTRUCT *p, int_4 mport)
    int_4 lb2d_max2112_setup(PICSTRUCT *p, int_4 mport)
    int_4 lb2d_dcs_set(PICSTRUCT *p, int_4 mport, int_4 value)
    int_4 lb2d_dcs_get(PICSTRUCT *p, int_4 mport)

  REVISIONS:
  2008/11/01    wao     baseline
  2009/02/01    rdh     moved into library
-----------------------------------------------------------------------
*/

/*---------------------------------------------------------------------------*/
/* LB2DR1 Bit-mapped switch query */          
int_4 lb2d_bmswitch_mrd(PICSTRUCT *p, int_4 mport, int_u4 *value)
{
   /* read from MAX2112 mem-map-region */
   int_u4 mdata = lb2d_rd (p,mport,LB2DR1_RAMREG_SWITCH_R,4);

   *value = mdata;
   if (LB2D_DBGPRINT >= 3)
     printf("LB2D BMSwitch Read:  %8.8X\n",mdata);
   return (0);
}

/*---------------------------------------------------------------------------*/
/* LB2DR1 Bit-mapped switch configuration */
int_4 lb2d_bmswitch_mwr(PICSTRUCT *p, int_4 mport, int_u4 value)
{
   /* First, write to actual switch controls - only bottom 9 bits used */
   lb2d_wr(p,mport,LB2DR1_GENCFG_INDEX,value&0x1FF,0);

   /* Next, write to MAX2112 mem-map-region */
   lb2d_wr(p,mport,LB2DR1_RAMREG_SWITCH,value,4);

   if (LB2D_DBGPRINT >= 3)
     printf("LB2D BMSwitch Write: %8.8X\n",value);
   return (0);
}

/*---------------------------------------------------------------------------*/
/* LB2DR1 MAX2112 Status Register query function */
int_4 lb2d_max2112_mrd(PICSTRUCT *p, int_4 mport, int_u4 ireg, int_u4 *ival)
{
    int_u4 mdata = 0x00000000;

    if (ireg == 0x0000000C) {
      /* Read status register 0x0C on chip */
      lb2d_wr(p,mport,LB2DR1_RCVRCFG_INDEX|0x01C00C00,0xC1, 1);
      udelay(10000);
      mdata = lb2d_rd (p,mport,0,0);
      udelay(1000);
      *ival = mdata;
    }
    else if (ireg == 0x0000000D) {
      /* Read status register 0x0D on chip */
      lb2d_wr(p,mport,LB2DR1_RCVRCFG_INDEX|0x01C00D00, 0xC1, 1);
      udelay(10000);
      mdata = lb2d_rd (p,mport,0,0);
      udelay(1000);
      *ival = mdata;
    }
    else {
      /* Read memory backed register on Alterra */
      mdata = lb2d_rd (p,mport,0x10000000|ireg,0);
      *ival = mdata;
    }

    if (LB2D_DBGPRINT >= 3)
      printf("LB2D RMODREG_MAX2112: %8.8X  %8.8X\n",mdata,
        LB2DR1_RCVRCFG_INDEX|0x00C00000|ireg|mdata);

    return (0);
}

/*---------------------------------------------------------------------------*/
/* LB2DR1 MAX2112 Register set function */
int_4 lb2d_max2112_mwr(PICSTRUCT *p, int_4 mport, int_u4 value)
{
   udelay(1000);
   lb2d_wr(p,mport,LB2DR1_RCVRCFG_INDEX|0x00C00000,value, 0);
   udelay(1000);
   if (LB2D_DBGPRINT >= 3)
     printf("LB2D WMODREG_MAX2112: %8.8X  %8.8X\n",value,
       LB2DR1_RCVRCFG_INDEX|0x00C00000|value);
   return (0);
}

/*---------------------------------------------------------------------------*/
/* LB2DR1 Si5326 Register set function */
int_4 lb2d_si5326_mwr (PICSTRUCT *p, int_4 mport, int_u4 value)
{
#define LB2DR1_SETADR  0x00000000 /* Set Address To Be Written */
#define LB2DR1_WRDAT   0x00000040 /* Write Data                */
   if (LB2D_DBGPRINT > 6) printf("  Si5326 Setup: %8.8X\n",value);
   udelay(100);
   lb2d_wr(p, mport, LB2DR1_SCLKCFG_INDEX | (LB2DR1_SETADR<<16) , (value & 0x0000FF00), 0); /* Si5326 Addr  */
   udelay(100);
   lb2d_wr(p, mport, LB2DR1_SCLKCFG_INDEX | (LB2DR1_WRDAT<<16) , ((value<<8) & 0x0000FF00), 0);  /* Si5326 Value */
   if (value == 0x00008840) udelay(53000);  /* ICAL could take up to 53msec */
   udelay(100);

   return (0);
}


/* LB2DR1 Module reset */                 
int_4 lb2d_reset (PICSTRUCT *p, int_4 mport)
{
   /* Assert Config Logic Reset */ 
   lb2d_wr(p,mport,LB2DR1_INITCFG_INDEX,0,1);   
   udelay(1000);
   /* De-Assert Config Logic Reset */ 
   lb2d_wr(p,mport,LB2DR1_INITCFG_INDEX,0x01,1);  
   udelay(1000);
  /* Assert Standby   */  
   lb2d_wr(p,mport,LB2DR1_RCVRCFG_INDEX|0x00C00900,0x80,1);
   udelay(1000); 
   return (0);
}

/* LB2DR1 A2D setup */                    
int_4 lb2d_a2d_setup (PICSTRUCT *p, int_4 mport)
{
  /* LVDS, Invert, 2's Comp*/
  lb2d_wr(p,mport,LB2DR1_A2DCFG_INDEX|0x00001400,0xC5,1);
  udelay(1000);
  /* Clock Invert          */
  lb2d_wr(p,mport,LB2DR1_A2DCFG_INDEX|0x00001600,0,1);
  udelay(1000);        
  /* Checker Board         */
  /* lb2d_wr(p,mport,LB2DR1_A2DCFG_INDEX|0x00000D00,7,1);   */ 
  lb2d_wr(p,mport,LB2DR1_A2DCFG_INDEX|0x00000D00,0,1);
  udelay(1000);                                                                 
  /* Set input clock divider */
  lb2d_wr(p,mport,LB2DR1_A2DCFG_INDEX|0x00000B00,0,1);
  udelay(1000);        
  /* Turn Off Sport Clock  */
  lb2d_wr(p,mport,LB2DR1_A2DCFG_INDEX|0x00011100,0x06,1);
  udelay(1000);        
  /* Config Transfer       */
  lb2d_wr(p,mport,LB2DR1_A2DCFG_INDEX|0x0000FF00,0x01,1);
  udelay(1000);        

  return(0);
}

real_8 lb2d_get_ref_freq (PICSTRUCT *p, int_4 mport) 
{ 
  UnionBuf b;
  b.u4[0] = lb2d_rd(p,mport,LB2DR1_RAMREG_FREFL_R,4);
  b.u4[1] = lb2d_rd(p,mport,LB2DR1_RAMREG_FREFU_R,4);
  if (b.u4[0]==0 && b.u4[1]==0) return 0.0;
  return (b.r8[0]);
}

/*---------------------------------------------------------------------------*/
/* LB2DR1 - Set MAX2112 RF Frequency (input/output is MHz)                  */
real_8 lb2d_max2112_get_freq (PICSTRUCT *p, int_4 mport) 
{
  real_8 max2112_ref_freq = lb2d_get_ref_freq(p,mport);
  int_u4 status_pllld,status_vasa,status_vasadc;
  int_u4 iregval, ndiv, nfrac;
  real_8 rffreq = -1.0;

  /* Get N-Divider value */
  ndiv = 0x00000000;
  iregval = 0x00000000;
  lb2d_max2112_mrd(p,mport,0x00000000,&iregval);
  ndiv = ndiv | ((iregval << 8)&(0x00007F00));
  iregval = 0x00000000;
  lb2d_max2112_mrd(p,mport,0x00000001,&iregval);
  ndiv = ndiv | ((iregval)&(0x000000FF));

  /* Get F-Divider value */    
  nfrac = 0x00000000;
  iregval = 0x00000000;
  lb2d_max2112_mrd(p,mport,0x00000002,&iregval);
  nfrac = nfrac | ((iregval << 16)&(0x000F0000));
  iregval = 0x00000000;
  lb2d_max2112_mrd(p,mport,0x00000003,&iregval);
  nfrac = nfrac | ((iregval << 8)&(0x0000FF00));
  iregval = 0x00000000;
  lb2d_max2112_mrd(p,mport,0x00000004,&iregval);
  nfrac = nfrac | ((iregval)&(0x000000FF));

  /* Calculate exact frequency */
  rffreq = ((real_8) ndiv) + (((real_8) nfrac)/1048576.0);
  rffreq = rffreq*max2112_ref_freq;

  /* Check for VCO selection and PLL lock */
  iregval = 0x00000000;
  lb2d_max2112_mrd(p,mport,0x0000000D,&iregval);
  status_vasadc = (iregval & 0x00000007);
  iregval = 0x00000000;
  lb2d_max2112_mrd(p,mport,0x0000000C,&iregval);
  status_pllld = ((iregval & 0x00000010) == 0x00000010) ? (1) : (0);
  status_vasa  = ((iregval & 0x00000040) == 0x00000040) ? (1) : (0);

  if (LB2D_DBGPRINT >= 1)
    printf("LB2D(R) FREQ: %15.3f  PLL->%1.1d VASA->%1.1d VASADC->%2d\n",
      rffreq,status_pllld, status_vasa, status_vasadc);

  /* Return indicator of bad VCO autoselect and/or PLL lock status */
  if (status_pllld == 0)  return (-1.0);
  /* if (status_vasa == 0)   return (-2.0); */
  if (status_vasadc == 0) return (-3.0);
  if (status_vasadc == 7) return (-4.0);

  return (rffreq/1000000.0);
}

/* LB2DR1 MAX2112 initialize */           
int_4 lb2d_max2112_setup (PICSTRUCT *p, int_4 mport)
{
  int_4 i;
  int_u4 max2112_regval[] = 
     { 0x00000980, 0x00000A00, 0x00000501, 0x00000080, 0x0000013C, 
       0x000002B0, 0x00000300, 0x00000630, 0x00000B28, 0x000007CC, 
       0x0000087B, 0x00000400, 0x00000900, 0x00000400, 0xFFFFFFFF };
  i = 0;
  while (max2112_regval[i] != 0xFFFFFFFF) {
    lb2d_max2112_mwr(p,mport,max2112_regval[i]);
    i++;
  }
  return(0);
}

/* LB2D DCS (DC Suppression) Control */ 
int_4 lb2d_dcs_set(PICSTRUCT *p, int_4 mport, int_4 value)
{
  int_u4 iregval;
  lb2d_bmswitch_mrd(p,mport,&iregval);
  iregval = (value > 0) ? (iregval | (0x00000040)) : (iregval & (~0x00000040));
  lb2d_bmswitch_mwr(p,mport,iregval);
  return(0);
}

int_4 lb2d_dcs_get(PICSTRUCT *p, int_4 mport)
{
  int_4  sw_onoff;
  int_u4 iregval;
  lb2d_bmswitch_mrd(p,mport,&iregval);
  sw_onoff = (iregval & (0x00000040)) ? (1) : (0);
  return(sw_onoff);
}

/*
-----------------------------------------------------------------------
  PUBLIC INTERFACE FUNCTIONS:
    int_4 lb2d_init (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 rate, 
                                        int_4 gain, int_4 flags, int_4 bits) 
    int_4 lb2d_enable(PICSTRUCT *p, int_4 mport,int_u4 dis_enable)

    int_4 lb2d_set_freq (PICSTRUCT *p, int_4 mport, real_8 freqmhz)
    int_4 lb2d_set_bbgain (PICSTRUCT *p, int_4 mport, int_4 bbgain)
    int_4 lb2d_set_bw (PICSTRUCT *p, int_4 mport, int_4 bw)
    int_4 lb2d_set_attn (PICSTRUCT *p, int_4 mport, int_4 attn)
    int_4 lb2d_set_gain (PICSTRUCT *p, int_4 mport, int_4 gain)
    int_4 lb2d_set_ais (PICSTRUCT *p, int_4 mport, int_4 mode, real_8 *params)
    int_4 lb2d_set_opts (PICSTRUCT *p, int_4 mport, int_4 flags)

    int_4 lb2d_get_freq (PICSTRUCT *p, int_4 mport)
    int_4 lb2d_get_bbgain (PICSTRUCT *p, int_4 mport)
    int_4 lb2d_get_bw (PICSTRUCT *p, int_4 mport)
    int_4 lb2d_get_attn (PICSTRUCT *p, int_4 mport)
    int_4 lb2d_get_gain (PICSTRUCT *p, int_4 mport)
    int_4 lb2d_get_opts (PICSTRUCT *p, int_4 mport)
    int_4 lb2d_get_rfpwr(PICSTRUCT *p, int_4 mport)

  REVISIONS:
  2008/11/01    wao     baseline
  2009/02/01    rdh     moved into library
-----------------------------------------------------------------------
*/


/*---------------------------------------------------------------------------*/
/* LB2DR1 - Query A/D attached to AD5501 RMS Power Detector                 */
int_4 lb2d_get_rfpwr(PICSTRUCT *p, int_4 mport)
{
/* Power Bit Settings       */
#define LB2DR1_PWR_SLEEP       0x00000001
#define LB2DR1_PWR_WAKE        0x00000002
#define LB2DR1_PWR_ACTIVE      0x00000004
#define LB2DR1_RFPWR_CALTBL_SZ 40
  int_4 caltbl[LB2DR1_RFPWR_CALTBL_SZ][2] = {
        {-30,   40}, {-29,   42}, {-28,   44}, {-27,   47}, {-26,   50},
        {-25,   54}, {-24,   59}, {-23,   64}, {-22,   70}, {-21,   78},
        {-20,   86}, {-19,   96}, {-18,  107}, {-17,  119}, {-16,  133},
        {-15,  148}, {-14,  165}, {-13,  183}, {-12,  204}, {-11,  227},
        {-10,  252}, { -9,  279}, { -8,  308}, { -7,  341}, { -6,  376},
        { -5,  413}, { -4,  454}, { -3,  499}, { -2,  546}, { -1,  598},
        {  0,  653}, {  1,  712}, {  2,  775}, {  3,  845}, {  4,  915},
        {  5,  992}, {  6, 1074}, {  7, 1162}, {  8, 1254}, {  9, 1353} };

  int_4  ii,idBm,ipmin,iperr;
  int_u4 mdata = 0;
  int_u4 sumpwr = 0;

  /* Average 4 power measurements */
  for(ii=0;ii<1;ii++)
    {
      /* Turn Off A/D, A/D Sleep, A/D Wake, A/D Active, Turn Off A/D */
      lb2d_wr(p,mport,LB2DR1_POWER_INDEX, 0,1);
      udelay(3000);
      lb2d_wr(p,mport,LB2DR1_POWER_INDEX,LB2DR1_PWR_SLEEP,1);
      udelay(3000);
      lb2d_wr(p,mport,LB2DR1_POWER_INDEX,LB2DR1_PWR_WAKE,1);
      udelay(5000);
      lb2d_wr(p,mport,LB2DR1_POWER_INDEX,LB2DR1_PWR_ACTIVE,1);
      udelay(3000);
      lb2d_wr(p,mport,LB2DR1_POWER_INDEX,0,1);
      udelay(3000);
      /* Read A/D value from register */
      mdata = lb2d_rd (p,mport,0x20000000,0);
      sumpwr += mdata;           
    }
  /* Power A/D back to Sleep     */
  lb2d_wr(p,mport,LB2DR1_POWER_INDEX,LB2DR1_PWR_SLEEP,1);
  udelay(3000);

  /* Computer total input power in dBm as function of A/D reading */
  idBm  = caltbl[0][0];
  ipmin = 2147483647; 
  for (ii=0; ii<LB2DR1_RFPWR_CALTBL_SZ; ii++) {
    iperr = abs(sumpwr - caltbl[ii][1]);
    if (iperr < ipmin) {
      ipmin = iperr;
      idBm = caltbl[ii][0];
    }
  }
  if (LB2D_DBGPRINT >= 2) 
    printf("LB2D(R) RFPWR: %d (dBm)  %d (ADC)  \n",idBm,sumpwr);

  return(idBm);
}

/*---------------------------------------------------------------------------*/
/* LB2DR1 - MAX2112 RF Front End Gain via D/A attached to Vctl              */
int_4 lb2d_get_gain (PICSTRUCT *p, int_4 mport) 
{ 
  return lb2d_rd (p,mport,LB2DR1_RAMREG_GAIN_R,1);
}

int_4 lb2d_set_gain (PICSTRUCT *p, int_4 mport, int_4 gain)
{
  real_8 r8gain;
  int_u4 i4gain;

  if ((gain < 0) || (gain > 75)) {
    return(-1);
  }
  else {
    /* Convert gain to correct DAC output voltage level */
    r8gain = ((819.0-4095.0)/(75.0-0.0))*((real_8) gain) + 4095.0;
    i4gain = (int_4) (r8gain+0.5);
    i4gain = (i4gain >  500) ? (i4gain) : (500);
    i4gain = (i4gain < 4095) ? (i4gain) : (4095);
    /* Write to RF Gain DAC Register */
    lb2d_wr(p,mport,LB2DR1_GAINCFG_INDEX|0x00300000,(i4gain<<4), 0);
    lb2d_wr(p,mport,LB2DR1_RAMREG_GAIN,gain,1);
    if (LB2D_DBGPRINT >= 1) 
      printf("LB2D(W) GAIN: %6.3f (V)  %d (DAC)  \n",
        (float)((((float)i4gain)/4095.0)*2.5), i4gain);
  }
  return (0);
}


/*---------------------------------------------------------------------------*/
/* LB2DR1 - Set RF DAT-31 front-end attenuation                             */
int_4 lb2d_get_attn (PICSTRUCT *p, int_4 mport) 
{ 
  return lb2d_rd(p,mport,LB2DR1_RAMREG_ATTN_R,1);
}

int_4 lb2d_set_attn (PICSTRUCT *p, int_4 mport, int_4 attn)
{
  int_u4 lattn;

  if ((attn < 0 ) || (attn > 31)) {
    return(-1);
  }
  else {
    lattn = attn;
    lb2d_wr(p,mport,LB2DR1_ATTENCFG_INDEX,lattn,1);
    lb2d_wr(p,mport,LB2DR1_RAMREG_ATTN,lattn,1);
    if (LB2D_DBGPRINT >= 1) 
      printf("LB2D(W) ATTN: %d (dB) \n",lattn);
  }
  return (0);
}


/*---------------------------------------------------------------------------*/
/* LB2DR1 - MAX2112 Baseband Low Pass Filter (BW is in Hz)                  */
int_4 lb2d_get_bw (PICSTRUCT *p, int_4 mport) 
{ 
  return lb2d_rd(p,mport,LB2DR1_RAMREG_BW_R,4);
}

int_4 lb2d_ack_bw (PICSTRUCT *p, int_4 mport)
{
  int_u4 mdata = 0x00000000;
  int_4  i4bw; 
  real_8 r8bw; 

  /* Query memmapped register for LPF bandwidth and double */
  lb2d_max2112_mrd(p,mport,0x00000008,&mdata);
  r8bw = mdata;
  r8bw = 4000000.0 + ((r8bw - 12.0)*290000.0);
  r8bw = 2.0*r8bw; 
  r8bw = (r8bw > 0.0) ? (r8bw) : (0.0);
  r8bw = (r8bw < 150000000.0) ? (r8bw) : (150000000.0);
  i4bw = (int_4) (r8bw+0.5);
  mdata = 0x00000800|mdata;
  if (LB2D_DBGPRINT >= 1) 
      printf("LB2D(R) BW: %i Hz  0x%8.8X \n",i4bw,mdata);
  return (i4bw);
}

int_4 lb2d_set_bw (PICSTRUCT *p, int_4 mport, int_4 bw)
{
  int_u4 i4lpf;
  int_4  i4bw;
  real_8 r8lpf,r8err;

  /* MAX2112 REGISTER 08: LPF Bandwidth in Hz */
  r8lpf = 0.5*((real_8) bw);
  r8lpf = (r8lpf > 700000.0) ? (r8lpf) : (700000.0);
  r8lpf = (r8lpf < 74000000.0) ? (r8lpf) : (74000000.0);
  r8lpf = ((r8lpf - 4000000.0)/290000.0) + 12.0;
  i4lpf = (int_u4) (r8lpf+0.5);
  i4lpf = (i4lpf < 255) ? (i4lpf) : (255);
  i4lpf = 0x00000800|i4lpf;
  lb2d_max2112_mwr(p, mport, i4lpf);
  lb2d_wr(p,mport,LB2DR1_RAMREG_BW,bw,4);
  if (LB2D_DBGPRINT >= 1)
      printf("LB2D(W) BW: %i Hz  0x%8.8X \n",bw,i4lpf);
  i4bw = lb2d_ack_bw(p,mport);
  r8err = ((real_8) bw) - ((real_8) i4bw);
  if (fabs(r8err) > 800000.0) {
    printf("LB2D Warning: Unable to set requested RFBW %i\n",bw);
    return (-1);
  }
  return (0);
}

/*---------------------------------------------------------------------------*/
/* LB2DR1 - MAX2112 Baseband Gain (in dB)                                   */
int_4 lb2d_get_bbgain (PICSTRUCT *p, int_4 mport) 
{ 
  return lb2d_rd(p,mport,LB2DR1_RAMREG_BBAMP_R,1);
}

int_4 lb2d_ack_bbgain (PICSTRUCT *p, int_4 mport)
{
  int_u4 mdata = 0x00000000;
  int_4  i4gain;             

  /* Query memmapped register for current contents */
  lb2d_max2112_mrd(p,mport,0x00000009,&mdata);
  i4gain = (int_4) (mdata & 0x0000000F);
  if (LB2D_DBGPRINT >= 1)
      printf("LB2D(R) BBGAIN: %i dB  0x%8.8X \n",i4gain,0x00000900|mdata);
  return (i4gain);
}

int_4 lb2d_set_bbgain (PICSTRUCT *p, int_4 mport, int_4 bbgain)
{
  int_u4 mdata = 0x00000000;
  int_u4 u4gain;             
  int_4  i4gain;

  if ((bbgain < 0) || (bbgain > 15)) {
    return(-1);
  }
  else {
    /* MAX2112 REGISTER 09: Lower nibble is baseband gain in dB */
    lb2d_max2112_mrd(p,mport,0x00000009,&mdata);
    u4gain = (int_u4) bbgain;
    u4gain = (u4gain & 0x0000000F);
    lb2d_max2112_mwr(p, mport, 0x00000900|(mdata & 0x000000F0)|u4gain);
    if (LB2D_DBGPRINT >= 1)
      printf("LB2D(W) BBGAIN: %d dB  0x%8.8X \n",u4gain,
        0x00000900|(mdata & 0x000000F0)|u4gain);
    i4gain = lb2d_ack_bbgain(p,mport);
    if (i4gain != ((int_4) u4gain)) {
      printf("LB2D Warning: BBGAIN set failed %d \n",(u4gain & 0x0000000F));
      return (-1);
    }
    lb2d_wr(p,mport,LB2DR1_RAMREG_BBAMP,i4gain,1);
  }
  return (0);
}


/*---------------------------------------------------------------------------*/
/* LB2DR1 - MAX2112 Synthesizer Frequency (in MHz)                          */

real_8 lb2d_get_freq (PICSTRUCT *p, int_4 mport) 
{ 
  UnionBuf b;
  b.u4[0] = lb2d_rd(p,mport,LB2DR1_RAMREG_FREQL_R,4);
  b.u4[1] = lb2d_rd(p,mport,LB2DR1_RAMREG_FREQU_R,4);
  if (b.u4[0]==0 && b.u4[1]==0) return 0.0;
  if (LB2D_DBGPRINT >= 1) 
     printf("LB2D(W) RAMFREQ:  EXACT: %21.9f  \n", 1000000.0*b.r8[0]);
  return (b.r8[0]);
}

int_4 lb2d_set_freq (PICSTRUCT *p, int_4 mport, real_8 freqmhz)
{
  const real_8 max2112_maxfreq   = 2500000000.0;
  const real_8 max2112_minfreq   =  600000000.0;
  const real_8 max2112_divfour   = 1125000000.0;
  const real_8 max2112_preselect = 1200000000.0;

  int_4  ndiv, nfrac;
  int_u4 switchcfg, iregval;
  real_8 max2112_reffreq = 0;
  real_8 freqhz, fndiv, dfrac, freqhz_act, rffreq;

  UnionBuf b;

  /* Compute dividers for the nearest center frequency */
  max2112_reffreq = lb2d_get_ref_freq(p,mport);
  freqhz = freqmhz*1000000.0;
  freqhz = (freqhz > max2112_minfreq) ? (freqhz) : (max2112_minfreq);
  freqhz = (freqhz < max2112_maxfreq) ? (freqhz) : (max2112_maxfreq);
  fndiv  = freqhz/max2112_reffreq;
  fndiv  = (fndiv > 0.0) ? (fndiv) : (0.0);
  fndiv  = (fndiv < 2500.0) ? (fndiv) : (2500.0); 
  ndiv   = (int_4) fndiv;
  dfrac  = fndiv - ((real_8) ndiv);
  dfrac  = (dfrac > 0.0) ? (dfrac) : (0.0);
  dfrac  = (dfrac < 1.0) ? (dfrac) : (1.0);
  dfrac  = 1048575.0*dfrac;
  nfrac  = (int_4) (dfrac+0.5);
  nfrac  = (nfrac >= 0) ? (nfrac) : (0);
  nfrac  = (nfrac <= 1048575) ? (nfrac) : (1048575);

  freqhz_act = max2112_reffreq*(((real_8) ndiv) + (((real_8) nfrac)/1048576.0));
  if (LB2D_DBGPRINT >= 1)
    printf("LB2D(W) FREQ:  IN: %21.9f  EXACT: %21.9f \n    REF: %21.9f  N: %d   F: %d \n",
      freqhz, freqhz_act, max2112_reffreq, ndiv, nfrac);

  /* Set the preselector switches */
  switchcfg = (freqhz_act < max2112_preselect) ? (0x00000009) : (0x00000006);
  iregval = 0x00000000;  
  lb2d_bmswitch_mrd(p,mport,&iregval);
  iregval = (iregval & 0xFFFFFFF0) | switchcfg;
  lb2d_bmswitch_mwr(p,mport,iregval);

  /* MAX2112 - Wake up chip */
  iregval = 0x00000000;
  lb2d_max2112_mrd(p,mport,0x00000009,&iregval);
  iregval = 0x00000900 | (iregval & 0x0000000F);
  lb2d_max2112_mwr(p,mport,iregval);

  /* MAX2112 - set N register */
  iregval = 0x00000080|((ndiv >> 8)&(0x0000007F));
  lb2d_max2112_mwr(p,mport,iregval);
  iregval = 0x00000100|((ndiv)&(0x000000FF));
  lb2d_max2112_mwr(p,mport,iregval);
  
  /* MAX2112 - set F register (all but LSB) */
  iregval = 0x000002B0|((nfrac >> 16)&(0x0000000F));
  lb2d_max2112_mwr(p,mport,iregval);
  iregval = 0x00000300|((nfrac >> 8)&(0x000000FF));
  lb2d_max2112_mwr(p,mport,iregval);

  /* MAX2112 - R-divider setup */
  lb2d_max2112_mwr(p,mport,0x00000501);

  /* MAX2112 - set PLL divide-by-4/divide-by-2 register flag */
  iregval = (freqhz_act < max2112_divfour) ? (0x00000680) : (0x00000600);
  lb2d_max2112_mwr(p,mport,iregval);

  /* MAX2112 - PLL/CP setup */
  lb2d_max2112_mwr(p,mport,0x000007CC);
  lb2d_max2112_mwr(p,mport,0x00000A00);
  lb2d_max2112_mwr(p,mport,0x00000B28);

  /* MAX2112 - last thing set must be F divider LSB */
  iregval = 0x00000400|((nfrac)&(0x000000FF));
  lb2d_max2112_mwr(p,mport,iregval);

  /* MAX2112 - pause for VAS frequency lock status registers */
  udelay(100000);

  /* Query frequency and lock status */
  rffreq = lb2d_max2112_get_freq(p,mport);
  if (rffreq < 0.0) {
    printf("LB2D Warning: FREQ set failed %15.3f\n",freqhz_act);
    return (-1);
  }
  b.r8[0] = rffreq;
  lb2d_wr(p,mport,LB2DR1_RAMREG_FREQL,b.u4[0],4);
  lb2d_wr(p,mport,LB2DR1_RAMREG_FREQU,b.u4[1],4);

  return (0);
}

/*---------------------------------------------------------------------------*/
/* LB2DR1 AIS (Adaptive Image Suppression) configuration */
int_4 lb2d_set_ais (PICSTRUCT *p, int_4 mport, int_4 mode, real_8 *ais_parm)
{
  int_2  ais_i2;
  int_4  i;
  int_4  sw_ais = 0;
  int_u4 iregval = 0x00000000;
  real_8 r8val;
  real_8 ais_dflt[4] = { 1.0, 0.0, 0.0, 1.0 };

  /* Update AIS Coefficients */
  if ((mode == 2)||(mode == 3)) {  
    for (i=0; i<4; i++) {
      udelay(1000);
      if (ais_parm != NULL)
        r8val = ais_parm[i];
      else
        r8val = ais_dflt[i];
      r8val  = (r8val <  2.0) ? (r8val) :  (2.0);
      r8val  = (r8val > -2.0) ? (r8val) : (-2.0);
      ais_i2 = (int_2) (r8val*16384.0);
      lb2d_wr(p,mport,LB2DR1_CXMCFGREG_INDEX,(ais_i2&0x0000FFFF), 0);
    }
    sw_ais = 1;
  }
  /* Turn ON AIS */             
  if ((mode == 1)||(mode == 3)) {
    iregval = 0x00000020|0x0000000E;
    udelay(1000);
    lb2d_wr(p,mport,LB2DR1_CXMENABLE_INDEX,iregval,0);
    udelay(1000);
    iregval = 0x00000000;
    lb2d_bmswitch_mrd(p,mport,&iregval);
    iregval = iregval | 0x00000100;
    lb2d_bmswitch_mwr(p,mport,iregval);
    sw_ais = 1;
  }
  /* Turn OFF AIS */
  if (sw_ais == 0) {
    iregval = 0x00000000;
    lb2d_bmswitch_mrd(p,mport,&iregval);
    iregval = iregval & (~0x00000100);
    lb2d_bmswitch_mwr(p,mport,iregval);
    iregval = 0x00000000;
    udelay(1000);
    lb2d_wr(p,mport,LB2DR1_CXMENABLE_INDEX,iregval,0);
    udelay(1000);
  }
  return (0);
}

/*---------------------------------------------------------------------------*/
/* LB2DR1 - Module FLAGS                                                    */
int_4 lb2d_get_opts (PICSTRUCT *p, int_4 mport)
{ 
  return lb2d_rd (p,mport,LB2DR1_RAMREG_OPTS_R,4);
}

int_4 lb2d_set_opts (PICSTRUCT *p, int_4 mport, int_4 i4flags)
{
  int_u4 u4flags,iregval,switchcfg;
  u4flags = 0x00000000 | i4flags;
  if (LB2D_DBGPRINT) printf("LB2D(W) FLAGS: %8.8X \n",u4flags);

  /* Make certain options are enabled */
  if (!(u4flags & RFOPT_ENABLE))
    return (0);

  /* Check reference clock config */
  lb2d_bmswitch_mrd(p,mport,&iregval); 
  if (u4flags & RFOPT_XREF)
    iregval |= 0x20;
  else
    iregval &= ~0x20;
  lb2d_bmswitch_mwr(p,mport,iregval);

  /* Check for DCS (DC Suppression) option */
  if (u4flags & RFOPT_DCS) 
    lb2d_dcs_set(p,mport,1);
  else
    lb2d_dcs_set(p,mport,0);

  /* Check for AIS (Adaptive Image Suppression) option */
  if (u4flags & RFOPT_AIS)
    lb2d_set_ais(p,mport,1,NULL);
  else
    lb2d_set_ais(p,mport,0,NULL);

  /* Check for AGC (Automatic Gain Control) option */     
/*
  if (u4flags & RFOPT_AGC)
    lb2d_set_agc(p,mport,1,NULL); 
  else
    lb2d_set_agc(p,mport,0,NULL);
*/
  lb2d_wr (p,mport,LB2DR1_RAMREG_OPTS,i4flags,4);

  return (0);
}

  
/*---------------------------------------------------------------------------*/
/* LB2DR1 - Public Interface for Module Setup */
int_4 lb2d_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags) 
{   
  int_4  i,ii,opts; 
  int_u4 switchcfg = 0;
  int_u4 iregval   = 0;
  real_8 dfreq;
  Si5326_Reg_Obj Si5326_Reg;
  int_u4 ppscfg    = 0;
  UnionBuf b;

  opts = RFOPT_ENABLE|RFOPT_DCS; /* default options */
  if (findflag("RFOPT_XREF",p->config) >= 0) { opts |= RFOPT_XREF; printf("RFOPT_XREF deprecated, use RFOPTS=(a|XREF|c) syntax.\n"); }
  if (findflag("RFOPT_DCS",p->config) >= 0) { opts |= RFOPT_DCS; printf("RFOPT_DCS deprecated, use RFOPTS=(a|DCS|c) syntax.\n"); }
  if (findflag("RFOPT_AIS",p->config) >= 0) { opts |= RFOPT_AIS; printf("RFOPT_AIS deprecated, use RFOPTS=(a|AIS|c) syntax.\n"); }
  if (findflag("RFOPT_LNA",p->config) >= 0) { opts |= RFOPT_LNA; printf("RFOPT_LNA deprecated, use RFOPTS=(a|LNA|c) syntax.\n"); }
  if (findflag("PREFX",p->config) >= 0) opts |= RFOPT_XREF;      
  opts = findmaskflagdef("RFOPTS",p->config,RFOPTLIST,opts);
  if (opts!=0) opts |= RFOPT_ENABLE;

  /* Reset and initialize module */
  lb2d_reset(p,mport);

  /* BMS - Initialize Bit-mapped switches */
  lb2d_bmswitch_mwr(p,mport,0x00000000);

  /* &BILL--Added PPS Control Register Write */
  if (findflag("LSBP",p->config) >= 0)
   ppscfg = ppscfg | 0x0002;
  if ((findflag("SLAVEPPS",p->config) >= 0) && (mport == 2))
   ppscfg = ppscfg | 0x0001; 
  if (findflag("MTGO",p->config) >= 0) 
   ppscfg = ppscfg | 0x0004;  
  lb2d_wr(p,mport,LB2DR1_PPSCFG_INDEX, ppscfg, 1);
  udelay(1000);

  /* BMS - Input clock reference selection */
  if ((findflag("MUXCLK",p->config) >= 0) &&
      (findflag("MUXCLK=P",p->config) < 0)) { 
    printf("Illegal MUXCLK= flag - ignored\n");
    printf("Allowable Clock Options: MUXCLK=P or MUXCLK=PX \n");
  }      
  switchcfg = (opts&RFOPT_XREF)? 0x20 : 0x00;
  iregval = 0x00000000;  
  lb2d_bmswitch_mrd(p,mport,&iregval);
  lb2d_bmswitch_mwr(p,mport,(iregval | switchcfg));

  /* BMS - Complex Or Real 16 bit data format flag */
  switchcfg = (bits == 16) ? (0x00000080) : (0x00000000);
  iregval = 0x00000000;  
  lb2d_bmswitch_mrd(p,mport,&iregval);
  lb2d_bmswitch_mwr(p,mport,(iregval | switchcfg));
  if (abs(bits) != 16) {
    printf("ERR:LBAND Module Data Format Must Be 16 bits Real/Complex Only!\n");
    return(-1);
  }
     
  /* Si5326 Ref-Clock and A/D clock configuration    */   
  Si5326_Reg.debuglevel = -1;
  Si5326_Reg.fref   =  10000000.0;     
  Si5326_Reg.dclk1  =  30000000.0;     
  Si5326_Reg.dclk2  = ((real_8) rate);                                   
  Si5326_Reg.dwght1 = 0.1;                                 
  Si5326_Reg.dwght2 = 1.0;                                       
  ii = si5326_regcfg_init(&Si5326_Reg,SI5326_REGCFG_MT_LB2D);
  ii = si5326_regcfg_bwsel(&Si5326_Reg,7);
  ii = si5326_regcfg_clkin(&Si5326_Reg,SI5326_REGCFG_CLKIN_1);
  ii = si5326_regcfg_dividers(&Si5326_Reg);
  if (Si5326_Reg.fclk1 > 30000000.0) {
    Si5326_Reg.dclk1  =  28000000.0; /* try again with 28MHz target */
    ii = si5326_regcfg_dividers(&Si5326_Reg);
  }
  if (Si5326_Reg.fclk2 != Si5326_Reg.dclk2 || Si5326_Reg.fclk1 > 30000000.0) 
    printf("LB2D Warning: A/D sample rate=%f not possible. Using rate=%f ref=%f\n",Si5326_Reg.dclk2,Si5326_Reg.fclk2,Si5326_Reg.fclk1);
  si5326_regcfg_debug(&Si5326_Reg);
  for(i=0; i<Si5326_Reg.numreg; i++) 
    lb2d_si5326_mwr(p,mport,Si5326_Reg.regval[i]);

  b.r8[0] = Si5326_Reg.fclk1;
  lb2d_wr (p,mport,LB2DR1_RAMREG_FREFL,b.u4[0],4);
  lb2d_wr (p,mport,LB2DR1_RAMREG_FREFU,b.u4[1],4);

  /* A2D Setup */
  lb2d_a2d_setup (p,mport);

  /* Set RF front-end attenuation */
  ii = findintflag("RFATTN",p->config);         
  if (ii < 0) ii = lb2d_get_attn(p, mport);
  if (ii < 0) ii = 0;
  if (lb2d_set_attn(p,mport,ii) < 0)
    printf("LB2D Warning: Invalid RFATTN option value %i\n",ii);

  /* Query input power level to init A/D */
  lb2d_get_rfpwr(p,mport); 

  /* Set RF front-end gain on MAX2112 */
  ii = findintflag("RFGAIN",p->config); 
  if (ii < 0) ii = lb2d_get_gain(p, mport);
  if (ii < 0) ii = 0;
  if (lb2d_set_gain(p, mport, ii) < 0)
    printf("Warning: RFGAIN out of range (0dB-75dB): %d\n",ii);

  /* Initial MAX2112 Register Setup */
  lb2d_max2112_setup (p,mport);

  /* MAX2112 Base Band Low Pass Filter  */  
  ii = (int) (finddblflag("RFBW",p->config) * 1000000);
  if (ii <= 0) ii = lb2d_get_bw(p, mport);
  if (ii <= 0) ii = 80000000;
  if (lb2d_set_bw(p, mport, ii) < 0)
    printf("LB2D Warning: RFBW out of range: %d\n",ii);

  /* MAX2112 Base Band Gain */                            
  ii = findintflag("RFBBGAIN",p->config);
  if (ii < 0) ii = lb2d_get_bbgain(p, mport);
  if (ii < 0) ii = 0;
  if (lb2d_set_bbgain(p, mport, ii) < 0)
    printf("LB2D Warning: RFBBGAIN out of range: %d\n",ii);

  /* MAX2112 - Set center frequency in MHz (900MHz is default) */
  dfreq = finddblflag("RFFREQ",p->config);
  if (dfreq <= 0.0) dfreq = lb2d_get_freq(p, mport);
  if (dfreq <= 0.0) dfreq = 900.0;
  if (lb2d_set_freq(p,mport,dfreq) < 0)
    printf("LB2D Warning: RFFREQ out of range/lock failed %i\n",ii);

  /* Wait for A/D, Alterra to stabilize */
  udelay(50000);

  /* apply runtime options */
  lb2d_set_opts (p,mport,opts);

  return (0);  
}  


/*---------------------------------------------------------------------------*/
/* LB2D Module Enable Interface - Starts/stops data flow from module */
int_4 lb2d_enable (PICSTRUCT *p, int_4 mport,int_u4 dis_enable)
{
  /* Place Module in Reset (System Reg = 0) */
  udelay(1000);
  lb2d_wr(p,mport,LB2DR1_SYSCFG_INDEX,0,1);
  udelay(2000);
  if (dis_enable != 0) {
    /* Remove Reset to Enable Module */
    lb2d_wr(p,mport,LB2DR1_SYSCFG_INDEX,0x01,1);
    udelay(2000);
  }
  return (0);
}
