package nxm.ice.lib;

import nxm.sys.inc.*;
import nxm.sys.lib.*;

/**
  A class to add Midas specific functions to the DevIce class.

  @author Jeff Schoen
  @version %I%, %G%

*/
public class MDevIce extends DevIce implements Keyable {

  public Midas M;
  public String config;

  // get new device alias from implicit call to getConfig 
  public MDevIce () {
    config = "ICEPIC,DEVNO=0,AUTODETECT,";
    load();
  }
  public MDevIce (Args MA, String alias) {
    this (MA.M, getConfig (MA,alias) );
  }
  // get new device from supplied config string 
  public MDevIce (Midas M, String config) {
    this.M = M;
    this.config = config;
    load();
  }
  // load the native libraries 
  public static boolean load() {
    if (!useNative) try { useNative = Shell.loadLibrary ("ice","lib","DevIce"); }
    catch (Exception e) { throw new RuntimeException (e); }
    return useNative;
  }
  // detect any cards on the system 
  public static int detect() { return detect(0); }
  public static int detect (int flags) {
    load();
    return DevIce.detect(flags);
  }
  // open the device using the supplied config string and supplied flags 
  public int open (String config, int flags) {
    int stat = super.open(config,flags);
    long maxalloc = getIntFlagDef("MAXALLOC",-1);
    if (maxalloc>0) setMaxAlloc(maxalloc<<20);
    return stat;
  }
  // open the device using the current config string and supplied flags 
  public int open (int flags) {
    return open(config,flags);
  }
  // open the device using the current config string and no flags 
  public int open() {
    return open(config,0);
  }

  // generate a config string from the supplied alias using standard tuning switches 
  public static String getConfig (Args MA, String alias) {
    String hwfile = MA.M.results.getString("ENV.HWFILE");
    Data devnos = MA.M.results.getData("ENV.ICEPICS");
    String rconfig = null, config = alias;
    if (config==null || config.length()==0) return null;
    if (config.equals("DUMMY")) return "DEVNO=99,TEST,NOLOCK,";
    Table rt = MA.M.results;
    for ( ;rt!=null && rconfig==null; ) { Object o;
      o = rt.get("HWALIAS."+alias); if (o != null && o instanceof String) rconfig=(String)o;
      o = rt.get("HWALIAS_"+alias); if (o != null && o instanceof String) rconfig=(String)o;
      rt = rt.getPrevious();
    }
    if (rconfig != null) {
      config = rconfig;
    }
    else if (hwfile != null) {
      DataFile hwf = new DataFile();
      hwf.init(MA.M,hwfile);
      hwf.open();
      Object entry = hwf.keywords.get(alias);
      if (entry instanceof String) config = (String)entry;
      if (config.startsWith("ALIAS=")) {
        int i = config.indexOf(',');
	alias = config.substring(6,i);
        entry = hwf.keywords.get(alias);
        if (entry instanceof String) config = (String)entry+config.substring(i);
      }
      hwf.close();
    }
    config = config.trim();
    if (!config.endsWith(",")) config += ",";
    int i; 
    if (devnos != null && (i=config.indexOf("DEVNO="))>=0) {
      int devno = config.charAt(i+6) - '0';
      devno = devnos.getL(devno);
      config = config.substring(0,i+6) + devno + config.substring(i+7);
    }
    if (MA.find("/FLAGS")) config = MA.getS("/FLAGS")+","+config;
    if (config.indexOf('^')>=0) config = MA.evaluateCarets(config);
    if (MA.find("/VERBOSE")) config += "VERBOSE="+MA.getL("/VERBOSE")+",";
    if (MA.find("/VB")) config += "VERBOSE="+MA.getL("/VB")+",";
    if (MA.getState("/SKIPONCARD")) config += "SKIPONCARD,";
    if (MA.getState("/SKIPONCORE")) config += "SKIPONCORE,";
    if (MA.find("/PORT")) config += "PORT="+MA.getS("/PORT")+",";
    if (MA.find("/TC") && !MA.find("/NOTC")) {
      String tcmode = MA.getS("/TC");
      if (tcmode.length()==0) tcmode = "SDN4";
      config += "TC="+tcmode+",";
    }
    return config;
  }

  // generate a config string from the supplied alias using standard tuning switches 
  public static String getConfigEntry (String config, String name) {
    Table t = new Table("{"+config+"}");
    return t.getS(name);
  }

  public String getPortKey () {
    String port = getStrFlagDef("PORT","Unknown");
    if (port.indexOf("CORE")>=0) {
      String cname = getKeyS(0,KEY_CNAME,4);
      if (cname!=null) port += ":"+cname;
    }
    return port;
  }
  public int setKey (int dmac, int key, Data data) {
    data = data.convertTo((char)getKeyType(key));
    return setKey (dmac, key, data.buf, data.size*data.bpe);
  }
  public int getKey (int dmac, int key, Data data) {
    data.setType(getKeyType(key));
    data.setSize(getKeySize(key));
    return getKey (dmac, key, data.buf, data.size*data.bpe);
  }
  public Data getKey (int dmac, int key) {
    Data data = new Data(0);
    int status = getKey (dmac, key, data);
    if (status<0) data = null;
    return data;
  }

  public DmaMap mapFile (DataFile df) {
    DmaMap map;
    if (df.io instanceof IceBlockResource) {
      IceBlockResource ibr = (IceBlockResource)df.io;
      map = ibr.map;
    }
    else if (df.io instanceof RamDiskResource) {
      RamDiskResource rdr = (RamDiskResource)df.io;
      map = rdr.map;
    }
    else {
      map = new DmaMap();
      map.vbytes = map.bytes = (long)df.getDataSize();
      int psize = long2phys(map.bytes);
      map.mode = DmaMap.DISK;
      if (mapMem(map,psize,1)<0) return null;
    } 
    map.dev = (DevIce)this;
    return map;
  }

  public String getConfig() {
    return config;
  }
  public String getMcfg (int dmac) {
    int type = getKeyL(0,KEY_CTYPE);
    boolean isY = (type>=ICEPIC5) && (type<=ICEPIC8);
    String mcfgList = isY? mcfgYList : mcfgXList;
    return Parser.mask2s(mcfgList,getKeyL(dmac,KEY_MCFG));
  }
  public String getFlag (int dmac) {
    return Parser.mask2s(flagList,getKeyL(dmac,KEY_FLAG));
  }

  public String toString() {
    String ctype = Parser.get(cardList, getKeyL(0,KEY_CTYPE));
    String ptype = Parser.get(ioptList, getKeyL(0,KEY_PTYPE));
    int pindex = getKeyL(0,KEY_PINDEX);
    int i1 = getKeyL(0,KEY_MTYPE1), i2 = getKeyL(0,KEY_MTYPE1);
    String mtype1 = (i1<0)? Parser.get(imtList,-i1): Parser.get(omtList,i1);
    String mtype2 = (i2<0)? Parser.get(imtList,-i2): Parser.get(omtList,i2);
    return "ICE-DSP Card="+ctype+" Port="+ptype+pindex+" Mod1="+mtype1+" Mod2="+mtype2;
  }

  public void modifyAlgorithm (int arg1, int arg2) {
    int[] args = new int[2]; 
    args[0]=arg1; args[1]=arg2;
    modifyAlgorithm(args);
  }
  public void modifyAlgorithm (int arg1, int arg2, int arg3) {
    int[] args = new int[3]; 
    args[0]=arg1; args[1]=arg2; args[2]=arg3;
    modifyAlgorithm(args);
  }
  public void modifyAlgorithm (int arg1, int arg2, int arg3, int arg4) {
    int[] args = new int[4]; 
    args[0]=arg1; args[1]=arg2; args[2]=arg3; args[3]=arg4;
    modifyAlgorithm(args);
  }

  public void modifyAlgorithm (int[] args) {
    byte[] head = new byte[8];
    Data data = new Data(args);
    Convert.packI(head,0,(short)PKTF_MODIFY);
    send(0,head,data.buf,data.buf.length,PKT_ACMD);
  }

  public String[] getKeys() {
    return keys;
  }
  public Object getKey (String key) {
    Object value = null;
    int ikey = name2key(key);
    if (ikey<=0) return null;
    byte type = getKeyType(ikey);
    if (type==DataTypes.LONG) value = new Data( getKeyL(-1,ikey) );
    if (type==DataTypes.DOUBLE) value = new Data( getKeyD(-1,ikey) );
    return value;
  }
  public Object setKey (String key, Object value) {
    int ikey = name2key(key);
    if (ikey<=0) return null;
    byte type = getKeyType(ikey);
    if (type==DataTypes.LONG) setKeyL(-1,ikey,Convert.o2l(value));
    if (type==DataTypes.DOUBLE) setKeyD(-1,ikey,Convert.o2d(value));
    return value;
  }
  public int setKeyD (int dmac, int key, double data1, double data2) {
    byte[] bbuf = new byte[16];
    Convert.packD(bbuf,0,data1);
    Convert.packD(bbuf,8,data2);
    return setKey (dmac,key,bbuf,16);
  }
  public int setKeyD (int dmac, int key, double data1, double data2, double data3) {
    byte[] bbuf = new byte[24];
    Convert.packD(bbuf,0,data1);
    Convert.packD(bbuf,8,data2);
    Convert.packD(bbuf,16,data3);
    return setKey (dmac,key,bbuf,24);
  }
  public String setKeyTable (int dmac, String table) {
    byte[] tbuf = table.getBytes();
    int tlen = tbuf.length;
    int blen = Math.max(tlen,1024);
    byte[] bbuf = new byte[blen];
    System.arraycopy(tbuf,0,bbuf,0,tlen); bbuf[tlen]=0;
    int len = setKey (dmac,KEY_TABLE,bbuf,blen);
    return new String(bbuf,0,len);
  }
  public String getKeyTable (int dmac, String table) {
    byte[] tbuf = table.getBytes();
    int tlen = tbuf.length;
    int blen = Math.max(tlen,1024);
    byte[] bbuf = new byte[blen];
    System.arraycopy(tbuf,0,bbuf,0,tlen); bbuf[tlen]=0;
    int len = getKey (dmac,KEY_TABLE,bbuf,blen);
    return new String(bbuf,0,len);
  }
}
