package nxm.ice.lib;

import nxm.ice.lib.IceHW;

/* 
  This JVM responds to packets thrown to it by 1|4|8 PKT2DAT cores.
  Currently: ICMP, IGMP, ARP, IPKT, and OPKT.

  Status Bus:
  ocmd=0x8aaa0bbb   where aaa is offset, bbb is length in bytes

  Memory Map:
  0x004	Host Command
  0x020	IPKT x8
  0x040	OPKT x8
  0x060	Counts x8
  0x100 Host message x1 Buffer
  0x200	IGMP messages x8
  0x400	VCTL Messages x8
*/
public class IceNIO extends IceJVM {

  public final static int poff=0x00800000;					// offset to transfer cache RAM

  public final static int SEND_BASE=0x0100;					// Host packet area

  public final static int IGMP_BASE=0x0200,IGMP_HOFF=40,IGMP_POFF=48,IGMP_SIZE=IGMP_POFF+16;	// IGMP report packet storage

  public final static int VCTL_BASE=0x0400,VCTL_SIZE=40+8+76,VCTL_SEP=128;	// Vita Control packet storage

  public static void run() {
    int i,j,beat,bad=0,pass=0,status,arg,chn,op,size,ocmd;

    for (i=0; i<0x200; i+=4) {		// zero statistics counters
      awMem(i,0);
      awMem(HBT,i);			// heartbeat monitor
    }

    for (beat=i;;beat++) {
      status = arBusSync(0x4);
      while ((status&0x30000)!=0) status = rBus();
      if ((status&0x40)!=0) {		// FPGA has command
	chn = (status&0xF);		// find channel
	aBus((chn<<16)|8);		// address channel
	if ((status&(0x100<<chn))==0) continue; // check in case async read of status register - not really there
	arg  = rBus();			// read command
	op   = (arg>>16)&0xF;
	size = (arg&0x3FF);
	awMem(0x8,(pass++));
	awMem(0xC,arg);
	if ((arg>>20)!=0x1CE) {   	// bad format - resync - produce debug output
	  bad++; 
	  awMem(0x14,0x1CE00000|status); 
	  awMem(0x18,arg); 
	  awMem(0x1C,(chn<<24)|bad); 
	  continue;
	}
	arg  = rBus();			// read command data
	if (size>0) {
	  wBus(size);			// special register to read packet into memory
	}
	processResp(op,arg,size,chn);
      }
      arg = arMem(0x04);		// this IP is already swapped to BE 
      if ((arg>>20)==0x1CE) {		// valid host command
	op  = (arg>>16)&0xF;
	chn = arg&0x7;
	processHost(op,chn);
	awMem(0x04,0x20000000|arg);
      }
      awMem(HBT,beat);			// heartbeat monitor
    }
  }

  public static void processHost (int op, int chn) {
    int addr=0,size=0;
    switch (op) {
    case 1: 	// Join
    case 2: 	// Leave
      addr = IGMP_BASE + chn*IGMP_SIZE;
      processM2M(addr,poff,IGMP_SIZE);
      if (op==2) processM2M(addr+IGMP_POFF,poff+IGMP_HOFF,4);	// sub in the Leave command stored in 1st PAD word
      awMem(poff+IGMP_POFF,0);					// clear 1st PAD word
      addr = 0;							// signal already moved to buffer
      size = IGMP_SIZE;	
      break;
    case 3: 	// Send
      addr = SEND_BASE;
      size = (arMem(addr) & 0xFF);				// host buffer only supports 0x100 max size + add 2 prep bytes
      break;
    case 4: 	// Ctl
      addr = VCTL_BASE + chn*VCTL_SEP;
      size = VCTL_SIZE;
      break;
    }
    if (size>0) processHostOut(chn,addr,size);			// ethernet frame will be size-2 prep bytes
  }

  public static void processResp (int op, int arg, int size, int chn) {
    int myip=arg;
    switch (op) {			// process packet
    case 1: 	// IPKT
    case 2:	// 0PKT
      int adr = ((op==2)? 0x40 : 0x20) | (chn<<2);
      awMem(adr,arMem(adr)+1000);
      break;
    case 3:	// ARP
      int destip=arMem(poff|0x2C);
      if (destip != myip) return;
      int opcode=arMem(poff|0x18);
      if (opcode == 0x02000406) { 
	processM2M(poff+0x18,SEND_BASE,16);
      } else {
	int srcel=arMem(poff|0x1C);
	int srceu=arMem(poff|0x20);
	int srcip=arMem(poff|0x24);
	srcip = (srcip<<16)|(srceu>>>16);
	awMem(poff|0x04,(srcel<<16));
	awMem(poff|0x08,(srceu<<16)|(srcel>>>16));
	awMem(poff|0x0C,(myip<<16)|0xe01c);
	awMem(poff|0x10,0x06080000|(myip>>>16));
	awMem(poff|0x14,0x00080100);	// types
	awMem(poff|0x18,0x02000406);	// opcode
	awMem(poff|0x1C,(myip<<16)|0xe01c);
	awMem(poff|0x20,(myip<<16)|(myip>>>16));
	awMem(poff|0x24,(srcel<<16)|(myip>>>16));
	awMem(poff|0x28,(srceu<<16)|(srcel>>>16));
	awMem(poff|0x2C,srcip);
	processRespOut(chn,0x40);
      }
      break;
    case 4:	// ICMP
      int hdr = arMem(poff|0x28);
      if ((hdr&0xFF)==0) { 
	processM2M(poff+0x28,SEND_BASE,16);
	processCounts(chn);
      } else {
	swapETH(myip);
      	int ipsz = (arMem(poff|0x14)>>24)+16;
	awMem(poff|0x28,hdr-0xFFF80008);	// adjust checksum
	processRespOut(chn,ipsz);
      }
      break;
    case 5:	// IGMP
//      processJoin (chn);			// possibly find sub chan from message here
      break;
    case 6:	// Other
      processM2M(poff,SEND_BASE,56);
      processCounts(chn);
      break;
    }
  }

  public static void processCounts (int chn) {
    int adr = 0x60 + (chn<<2);
    awMem(adr,arMem(adr)+1);
  }
  public static void processM2M (int addri, int addro, int size) {
    for (int i=0; i<size; i+=4) awMem(addro+i,arMem(addri+i));
  }
  public static void swapETH (int myip) {
    int srcel=arMem(poff|0x0c);
    int srceu=arMem(poff|0x10);
    int srcip=arMem(poff|0x20);
    awMem(poff|0x04,(srcel<<16));
    awMem(poff|0x08,(srceu<<16)|(srcel>>>16));
    awMem(poff|0x0C,(myip<<16)|0xe01c);
    awMem(poff|0x10,(srceu&0xFFFF0000)|(myip>>>16));
    awMem(poff|0x20,myip);
    awMem(poff|0x24,srcip);
  }
  public static void processHostOut (int chn, int addr, int size) {
    aBus((chn<<16));
    if (addr!=0) processM2M(addr,poff,size);
    wBus(0x80000000|size);	// start at offset=0. total ethernet frame is size-2 prep bytes
    processCounts(chn);
  }
  public static void processRespOut (int chn, int size) {
    aBus((chn<<16));
    wBus(0x80040000|size);	// start at offset=4. total ethernet frame is size-2 prep bytes
    processCounts(chn);
  }

}
